/* $Header: /Nfs/radyr/usr11/rc136/Projet/GC/RCS/monitor.c,v 1.1 1992/09/03 21:15:02 rc136 Exp rc136 $ */
/* Auteur: Vincent Delacour (delacour@poly.polytechnique.fr) */
/* Sujet: temps d'execution et autres mesures du gc  */
/* Creation: Sat May 25 15:45:55 1991 */

#include "../Include/parametres.h"

#ifdef MONITOR_GC

int _gc_monitor_time = 1;
int _gc_monitor_locality = 0;
int _gc_monitor_others = 1;
int _gc_monitor_expansion = 1;

/**********************************************************************
 *                             MONITOR
 **********************************************************************
 * instrumentation des execution : enregistrement des temps
 * d'execution du gc,  de la localite des references dans les pages
 * d'objets,  etc.
 **********************************************************************/


#include <stdio.h>			/* rapports imprimes */

/*extern int fputs(char*, FILE*); */
/*extern int fprintf(FILE*, char*, ...); */

FILE *stdreport = stderr; 

/**********************************************************************
 * myputc
 **********************************************************************
 * Le caractere de fin de ligne ne force plus le vidage du tampon: c'est
 * le caractere '\a' (alerte) qui remplit ce role. Ceci pour alleger le
 * cout et l'effet perturbant des rapports de mesure sur le deroulement
 * du programme. 
 **********************************************************************/

#define myputc(x, p)	(--(p)->_cnt >= 0 ?\
	(int)(*(p)->_ptr++ = (unsigned char)(x)) :\
	(((p)->_flag & _IOLBF) && -(p)->_cnt < (p)->_bufsiz ?\
		((*(p)->_ptr = (unsigned char)(x)) != '\a' ?\
			(int)(*(p)->_ptr++) :\
			_flsbuf(*(unsigned char *)(p)->_ptr, p)) :\
		_flsbuf((unsigned char)(x), p)))

#define _REPORT_NEWLINE		/*my*/putc('\n', stdreport)



#include <sys/time.h>			/* appels systeme */

/*extern long time(void*);*/

#include <sys/resource.h>

/*extern void getrusage(int, struct rusage*);*/

/**********************************************************************
 * gestionnaire memoire
 **********************************************************************/

#include "../Include/public.h"                     
#include "private.h"                    
#include "gc.h"
#include "../Include/alloc.h"
#include "monitor.h"

extern npage_t InitialNPages;
extern npage_t DescOffset;

/**********************************************************************
 * Variables exportees
 **********************************************************************/

int _in_gc = 0;			/* = 1 pendant gc */

/**********************************************************************
 * forward
 **********************************************************************/

static struct rusage ru;
static void _report_time_and_faults(void);
static npage_t _current_minpool;

static int _count_fixedpages(void);

static void _verify_cons_reg(void); 

static int tot_ref;			/* nb total de references */
static int tot_pg;			/* nb ref. dans la page */
static int tot_seg;			/* nb ref. dans le segment */
static int cdr_ref;			/* nb total de ref dans CDR */
static int cdr_pg;			/* nb ref. CDR dans la page */
static int cdr_seg;			/* nb ref. CDR dans le segment */

static void _init_locality(void);

static void _record_other_locality(val_t, nwords_t);
static void _record_cons_locality(val_t);

static void _scan_copy_page(npage_t);
static void _scan_fix_page(npage_t);

					/* local scp */

static nwords_t _cp_words;
static nwords_t _local_scp;
static nwords_t _cp_cons_words;
static nwords_t _cons_local_scp;

static void _init_local_scp(void);
static void _report_local_scp(void);

/**********************************************************************
 * utilitaires locaux
 **********************************************************************/

static struct rusage ru;

static void _report_time_and_faults(void)
{
  /*  getrusage(RUSAGE_SELF, &ru);
    fprintf(stdreport, 
	    "\t#S(time utime %d.%-06d stime %d.%-06d minflt %d majflt %d)", 
	    ru.ru_utime.tv_sec, 
	    ru.ru_utime.tv_usec, 
	    ru.ru_stime.tv_sec, 
	    ru.ru_stime.tv_usec, 
	    ru.ru_minflt, 
	    ru.ru_majflt);
    _REPORT_NEWLINE; */ /* HP-UX n'a pas getrusage */
    return;
}


/**********************************************************************
 *			      EXECUTION
 **********************************************************************/


extern void _report_execution_begins(char *prog)
{
    long clockval = time(NULL);

    fprintf(stdreport, "#S(execution progname  %s ; date ", prog);
    fputs(ctime(&clockval), stdreport);
    if (_gc_monitor_locality)
	 fprintf(stdreport, "\tmonitor-locality t");
    else
	 fprintf(stdreport, "\tmonitor-locality ()");
    _REPORT_NEWLINE;
    fprintf(stdreport, "\tpagesize %d", PageSize);
    _REPORT_NEWLINE;
    fprintf(stdreport, "\tgmparams #S(gmparams x %g m %d th %d i %d)",
	    ExpansionFactor,  MinExpansion,
	    GCThreshold, InitialNPages);
    _REPORT_NEWLINE;
    return;
}

extern void _report_init_heap(void)
{
    fprintf(stdreport, "\tinit-heap ");
    _report_heap();
    fprintf(stdreport, "\tmoments (");	/* liste des moments */
    _REPORT_NEWLINE;
    return;
}


extern void _report_execution_ends(void)
{
    fprintf(stdreport, "\t)");		/* fin liste des moments */
    _REPORT_NEWLINE;
    fprintf(stdreport, "; Fin execution");
    _REPORT_NEWLINE;
    fprintf(stdreport, "tot-time ");
    _report_time_and_faults();
    fprintf(stdreport, ")");
    _REPORT_NEWLINE;
    return;
}

/**********************************************************************
 *			     ETAT DU TAS
 **********************************************************************/

extern void _report_maxpages(npage_t max)
{
    fprintf(stdreport,  "\tmaxpages %d", max);
    _REPORT_NEWLINE;
    return;
}

extern void _report_heap(void)
{
    fprintf(stdreport, "\t#S(heap heapsize %d poolsize %d freemem %d endpage %d)", 
	    HeapSize, 
	    PoolSize,
	    FreeMem, 
	    EndPage);
    _REPORT_NEWLINE;
    return;
}

/**********************************************************************
 *                            Expansion
 **********************************************************************
 * Une seule mesure de temps pour l'expansion : celle-ci est
 * suffisamment courte. Rapport a la fin de l'expansion.
 **********************************************************************/



extern void _report_expansion(void)
{
    fprintf(stdreport, "#S(expansion heap-after");
    _REPORT_NEWLINE;
    _report_heap();
    fprintf(stdreport, "\ttime ");
    _report_time_and_faults();
    fprintf(stdreport, ")");
    _REPORT_NEWLINE;
    return;
}


/**********************************************************************
 *				 pile
 **********************************************************************/

extern void _report_stack(nwords_t nw)
{
    fprintf(stdreport, "\tstack-size %d", nw);
    _REPORT_NEWLINE;
    return;
}


/**********************************************************************
 *				   GC
 **********************************************************************/

extern void _report_gc_begins(void)
{
    _current_minpool = PoolSize;

    fprintf(stdreport, "#S(gc number %d", _gc_count);
    _REPORT_NEWLINE;
    fprintf(stdreport, "\theap-before ");
    _REPORT_NEWLINE;
    _report_heap();
    fprintf(stdreport, "\ttime-before ");
    _report_time_and_faults();
    _init_locality();    
    return;
}


extern void _report_gc_ends(void)
{
    fprintf(stdreport, "; fin du gc");
    _REPORT_NEWLINE;
    fprintf(stdreport, "\theap-after ");
    _report_heap();
    fprintf(stdreport, "\ttime-after ");
    _report_time_and_faults();
    _report_minpool();
    fprintf(stdreport, ")");
    _REPORT_NEWLINE;
    return;
}


/**********************************************************************
 * Minimum POOL size
 **********************************************************************/

static npage_t _current_minpool;

extern void _record_minpool(void)
{
    if (PoolSize < _current_minpool)
         _current_minpool = PoolSize;
    return;
}

extern void _report_minpool(void)
{
    fprintf(stdreport, "\tminpool %d", _current_minpool);
    _REPORT_NEWLINE;
    return;
}


/**********************************************************************
 *			     PAGES FIXEES
 **********************************************************************/

extern void _report_fixedpages(void)
{
    fprintf(stdreport, "\tfixed-pages %d",
	    _count_fixedpages());
    _REPORT_NEWLINE;
    return;
}


static int _count_fixedpages(void)
{
    npage_t pn = DescOffset;
    npage_t fixed_pages = 0;
    while(pn != EndPage)
      {
          npage_t bsz = BlocSize(pn);
          pagetag_t pt = PageTag(pn);
          switch (pt.zone)
            {
           case HIDDEN_ZONE:
           case FREE_ZONE:
           case GC_ZONE:
           case FROM_FIX_ZONE:
           case FROM_FIX_OPAQUE_ZONE:
           case FROM_COPY_ZONE:
           case FROM_COPY_OPAQUE_ZONE:
           case FROM_BIG_ZONE:
           case FROM_BIG_OPAQUE_ZONE:
           case FOLLOW_ZONE:
           case TO_FIX_ZONE:
           case TO_FIX_OPAQUE_ZONE:
                break;
           case TO_FIXED_ZONE:
           case TO_FIXED_OPAQUE_ZONE:
                fixed_pages += bsz;
                break;
           case TO_COPY_ZONE:
           case TO_COPY_OPAQUE_ZONE:
                error("?");
                break;
           case TO_BIG_ZONE:
           case TO_BIG_OPAQUE_ZONE:
                break;
            }
	  pn += bsz;
      }
    return fixed_pages;
}

/**********************************************************************
 * Localite des references
 * **********************************************************************
 * Il y deux modes de mesure : 1- au cours du GC lui-meme, les objets
 * vivants sont examines individuellement lorsqu'ils sont promus. 2- a
 * la fin du gc, l'ensemble des pages d'objets est parcouru, page par
 * page.
 *
 * Les mesures portent sur (1) le nombre total de references,  (2) le
 * nombre total de references dans la meme page,  (3) le nombre total
 * de CDR referencant la meme page,  (4) le nombre total de CDR
 * pointant sur NIL,  (5) le nombre total de CONS,  etc,  etc. On
 * pourrait bien sur continuer a l'infini.
 **********************************************************************/

static int tot_ref;			/* nb total de references */
static int tot_pg;			/* nb ref. dans la page */
static int tot_seg;			/* nb ref. dans le segment */

static int cdr_ref;			/* nb total de ref dans CDR */
static int cdr_pg;			/* nb ref. CDR dans la page */
static int cdr_seg;			/* nb ref. CDR dans le segment */


#define SegmentSize (1 << 16)
#define SegmentMask (SegmentSize - 1)

#define SegmentStart(add) ((val_t)(((ulint)(add)) & ~SegmentMask))

#define _insamepage(add, pg_start) \
     ((((ulint)(add)) & ~PageMask) == (ulint)(pg_start))

#define _insamesegment(add, seg_start) \
     ((((ulint)(add)) & ~SegmentMask) == (ulint)(seg_start))

/**********************************************************************
 * CONS_REG
 **********************************************************************/

#define CONS_REG        9

/* extern int strcmp(char*, char*); */

static void _verify_cons_reg(void)
{
    if(strcmp(StaticRegions[CONS_REG].name, "CONS_REG") == 0)
         return;
    error("Editer le fichier monitor.c : CONS_REG pas conforme\n");
    return;
}


/**********************************************************************
 * Examen d'un objet
 **********************************************************************
 * On distingue le cas des listes.
 **********************************************************************/

extern void _record_obj_locality(val_t obj)
{
    npage_t pn = PageNum(obj);
    pagetag_t pt = PageTag(pn);
    nregion_t nreg = pt.nregion;

    if (nreg == CONS_REG)
      {
	  _record_cons_locality(obj);
	  return;
      }
    else
      {
	  nwords_t sz;

	  switch (pt.mode)
	    {
	   case NO_MODE:
		error("no mode!\n");
		break;
	   case W1_MODE:
	   case W2_MODE:
	   case W3_MODE:
	   case W4_MODE:
	   case W8_MODE:
	   case SMALLSIZE_MODE:
		sz = RegionSize(nreg);
		_record_other_locality(obj, sz);
		return;
	   case HEADER_MODE:
		error ("ajouter ce cas.\n");
		break;
	   case CSTRING_MODE:
	   case STRING_MODE:
	   case FOLLOW_MODE:
	   case BIG_MODE:
	   default:
		error("mauvais mode\n");
		break;
	    }
      }
    return;
}




/**********************************************************************
 * _record_other_locality : enregistre la localite dans les champs de
 * l'objet (autre qu'une cellule de liste).
 **********************************************************************/

static void _record_other_locality(val_t obj, nwords_t sz)
{
    val_t page_start = PageStart(obj);
    val_t page_end = page_start + OnePage;
    val_t seg_start = SegmentStart(obj);

    if ((obj + sz) > page_end)
	 error("objet chevauchant la frontiere de page\n");
 {
     nwords_t i;
	  
     for (i = 0; i < sz; i++)
       {
	   if (Ref_P(obj[i]))
	     {
		 tot_ref += 1;
		 if (_insamepage(obj[i], page_start))
		   {
		       tot_pg += 1;
		       tot_seg += 1;
			}
		 else
		      if (_insamesegment(obj[i], seg_start))
			   tot_seg += 1;

	     }
       }
 }
    return;
}

static void _record_cons_locality(val_t cons)
{
    val_t page_start = PageStart(cons);
    val_t seg_start = SegmentStart(cons);

    if (Ref_P(cons[0]))			/* examen CAR */
      {
	  tot_ref += 1;
	  if (_insamepage(cons[0], page_start))
	    {
		tot_pg += 1;	
		tot_seg += 1;
	    }
	  else if (_insamesegment(cons[0], seg_start))
	    {
		tot_seg += 1;
	    }
      }
    if (Ref_P(cons[1]))			/* examen CDR */
      {
	  tot_ref += 1;
	  cdr_ref += 1;
	  if (_insamepage(cons[1], page_start))
	    {
		tot_pg += 1;
		cdr_pg += 1;
		tot_seg += 1;
		cdr_seg += 1;
	    }
	  else if (_insamesegment(cons[1], seg_start))
	    {
		tot_seg += 1;
		cdr_seg += 1;
	    }
      }
    return;
}


/**********************************************************************
 * Rapport sur la localite des references
 **********************************************************************/

extern void _report_locality_before(void)
{
    fprintf(stdreport, "\tlocality-before ");
    _report_locality_of_references();
    return;
}


extern void _report_locality_of_references(void)
{
    fprintf(stdreport, "#S(locality tot-ref %d tot-pg %d tot-seg %d", 
	    tot_ref, tot_pg, tot_seg);
    _REPORT_NEWLINE;
    fprintf(stdreport, "\t\t\tcdr-ref %d cdr-pg %d cdr-seg %d)", 
	    cdr_ref, cdr_pg, cdr_seg);
    _REPORT_NEWLINE;
    return;
}

/**********************************************************************
 * Reinitialisation des compteurs.
 **********************************************************************/

static void _init_locality(void)
{
    tot_ref = 0;
    tot_pg = 0;
    tot_seg = 0;
    cdr_ref = 0;
    cdr_pg = 0;
    cdr_seg = 0;
    return;
}

/**********************************************************************
 * Localite dans les pages d'objets
 **********************************************************************/

static nwords_t _cp_words = 0;
static nwords_t _local_scp = 0;

static nwords_t _cp_cons_words = 0;
static nwords_t _cons_local_scp = 0;


static void _init_local_scp(void)
{
    _cp_words = 0;
    _local_scp = 0;
    _cp_cons_words = 0;
    _cons_local_scp = 0;
    return;
}


static void _report_local_scp(void)
{
    fprintf(stdreport, " #S(localscav cp-words %d lscp %d", 
	    _cp_words, _local_scp);
    _REPORT_NEWLINE;
    fprintf(stdreport, "\t\t\tcons-cp-words %d cons-lscp %d)", 
	    _cp_cons_words, 
	    _cons_local_scp);
    _REPORT_NEWLINE;
    return;
}


extern void _report_heap_locality(void)
{
    npage_t pn = 0;

    _verify_cons_reg();
    _init_locality();
    _init_local_scp();

    while(pn !=  EndPage)
      {
	  npage_t bsz = BlocSize(pn);
	  pagetag_t pt = PageTag(pn);
	  switch(pt.zone)
	    {
           case HIDDEN_ZONE:
           case FREE_ZONE:
           case GC_ZONE:
           case FROM_FIX_ZONE:
           case FROM_FIX_OPAQUE_ZONE:
           case FROM_COPY_ZONE:
           case FROM_COPY_OPAQUE_ZONE:
           case FROM_BIG_ZONE:
           case FROM_BIG_OPAQUE_ZONE:
           case FOLLOW_ZONE:
           case TO_FIX_OPAQUE_ZONE:
           case TO_FIXED_OPAQUE_ZONE:
           case TO_COPY_OPAQUE_ZONE:
                break;

           case TO_FIXED_ZONE:
           case TO_FIX_ZONE:
		if (bsz != 1)
		     error("bloc non-unitaire\n");
		_scan_fix_page(pn);
                break;		 
           case TO_COPY_ZONE:
		if (bsz != 1)
		     error("bloc de recopie non-unitaire\n");
		_scan_copy_page(pn);
                break;
           case TO_BIG_ZONE:
           case TO_BIG_OPAQUE_ZONE:
                break;
	    }
	  pn += bsz;
      }
    fprintf(stdreport, "; apres exploration");
    _REPORT_NEWLINE;
    fprintf(stdreport, "\t local-scavenging ");
    _report_local_scp();
    fprintf(stdreport, "\tlocality-after ");
    _report_locality_of_references();
    _REPORT_NEWLINE;
    return;
}

static void _scan_copy_page(npage_t pn)
{
    pagetag_t pt = PageTag(pn);
    nregion_t nreg = pt.nregion;
    val_t loc = PageAddress(pn);
    val_t top;
    val_t page_scp = PageSCP(pn);
    nwords_t local_scp_words;

    if (RegionAPG(nreg) == pn)
      {
	  top = RegionAP(nreg);
	  local_scp_words = RegionSCP(nreg) - loc;
      }
    else
      {
	  top = PageTop(pn);
	  local_scp_words = page_scp - loc;
      }
#ifdef DEBUG
    if ((int)local_scp_words < 0)
	 error("mauvais scp\n");
#endif    
    _cp_words += top - loc;
    _local_scp += local_scp_words;
    
    if (nreg ==  CONS_REG)
      {
	  _cp_cons_words +=  top - loc;
	  _cons_local_scp += local_scp_words;
	  
	  while (loc != top)
	    {
		_record_cons_locality(loc);
		loc += 2;
	    }
	  return;
      }
    else
      {
	  nwords_t sz;
	  
	  switch (pt.mode)
	    {
	   case NO_MODE:
		error("no mode!\n");
		break;
	   case W1_MODE:
	   case W2_MODE:
	   case W3_MODE:
	   case W4_MODE:
	   case W8_MODE:
	   case SMALLSIZE_MODE:
		sz = RegionSize(nreg);
		while(loc != top)
		  {
		      _record_other_locality(loc, sz);
		      loc += sz;
		  }
		break;
	   case HEADER_MODE:
		error ("ajouter ce cas.\n");
		break;
	   case CSTRING_MODE:
	   case STRING_MODE:
	   case FOLLOW_MODE:
	   case BIG_MODE:
		default:
		error("mauvais mode\n");
		break;
	    }
	  return;
      }
}

static void _scan_fix_page(npage_t pn)
{
    pagetag_t pt = PageTag(pn);
    nregion_t nreg = pt.nregion;
    val_t loc = PageAddress(pn);
    val_t top;
    bitmap_t bm = PageBitMap(pn);

    if (RegionAPG(nreg) == pn)
	 top = RegionAP(nreg);
    else
	 top = PageTop(pn);

    if (nreg ==  CONS_REG)
      {
	  while(loc != top)
	    {
		ulint mword =  MWord(loc, bm);
		ulint mask = 1 << MShift(loc);

		if (mask & mword)
		     _record_cons_locality(loc);

		loc += 2;
	    }
	  return;
      }
    else
      {
	  nwords_t sz;
	  
	  switch (pt.mode)
	    {
	   case NO_MODE:
		error("no mode!\n");
		break;
	   case W1_MODE:
	   case W2_MODE:
	   case W3_MODE:
	   case W4_MODE:
	   case W8_MODE:
	   case SMALLSIZE_MODE:
		sz = RegionSize(nreg);
		
		while(loc != top)
		  {
		      ulint mword = MWord(loc, bm);
		      ulint mask = 1 << MShift(loc);

		      if (mask & mword)
			   _record_other_locality(loc, sz);

		      loc += sz;
		  }
		break;
	   case HEADER_MODE:
		error ("ajouter ce cas.\n");
		break;
	   case CSTRING_MODE:
	   case STRING_MODE:
	   case FOLLOW_MODE:
	   case BIG_MODE:
	   default:
		error("mauvais mode\n");
		break;
	    }
	  return;
      }
}


extern void _report_no_locality(void)
{
    fprintf(stdreport, 
	    "\tlocality-before () local-scavenging () locality-after ()");
    _REPORT_NEWLINE;
}


    
    
#endif

