/* $Header: /Nfs/radyr/usr11/rc136/Projet/GC/RCS/gm.c,v 1.1 1992/09/03 21:15:02 rc136 Exp rc136 $ */
/* Auteur: Vincent Delacour (delacour@poly.polytechnique.fr) */

#include "../Include/parametres.h"
#include "../Include/public.h"
#include "private.h"

#include <sys/time.h>
#include <sys/resource.h>

#include "monitor.h"

extern char *etext, *edata, *end;
extern int getrlimit(int, struct rlimit*);

					/* divers unix */

/* extern int write(int, char*, int); */
/* extern int strlen(char*); */
extern void exit(int);
/* extern char *memcpy(char*, char*, int); */


/**********************************************************************
 *                          Modele memoire
 **********************************************************************
 * Le modele memoire regroupe les fonctionnalites de bas niveau
 * attachees a l'espace pagine, et a sa description par des attributs
 * de pages et de regions. Il comporte egalement l'allocateur de pages
 * (POOL) et l'expansion du tas. Tous ces mecanismes sont independants
 * du gc,  du compte de la memoire libre, de la strategie de gestion
 * de la memoire. Cependant, les structures de donnees comportent des
 * champs utilises par les couches clientes,  ceci dans le but de ne
 * pas multiplier les tableaux de descripteurs (ce qui pourrait
 * 1- compliquer le schema d'expansion,  2- conduire a un dispersement
 * de ces informations). 
 *
 * Un bon nombre des fonctionnalites offertes par le modele memoire
 * sont en fait de simples acces aux structures de donnees : voir le
 * fichier d'entete correspondant. 
 **********************************************************************/

val_t EndMem;                 /* dernier 'break' connu */
npage_t         EndPage;                /* page de ce break */
npage_t         MaxPages;               /* voir man limit(1) */

npage_t         HeapSize;		/* nb pages propres au GM */
/**********************************************************************
 * Tableau des descripteurs de pages (TDP)
 **********************************************************************/

npage_t 	DescBloc;		/* no de page du TDP */
npage_t 	DescSize;		/* nb pages TDP  */
int     	DescCapacity;		/* nb virtuel de descr. dans TDP */
pagedesc_t     *PageDescs;		/* adresse virtuelle du TDP */
npage_t         DescOffset;             /* offset */

/**********************************************************************
 * Erreurs,  avertissements,  points d'arret
 **********************************************************************/

extern void message(char *s)
{
    write(2, s, strlen(s));
    return;
}

/* stop: fonction ou mettre un break pour le debug */

extern void stop()
{
    return;
}

extern void error(char *s)
{
    message(s);
    exit(1);
    return;                             /* pfj */
}

/**********************************************************************
 * Initialisation du modele memoire
 **********************************************************************
 * InitMM(npages): initialise le plan memoire de telle sorte qu'il y
 * ait au moins npages pages libres dans le POOL.
 *
 * InitProcess(): calcule les parametres du processus,  renseigne la
 * structure 'process'. La valeur MaxPages est le nombre maxi
 * de pages de donnees allouables au processus. La valeur recueillie
 * respecte les limitations volontaires par limit(1).
 *
 * InitPages(npages): initialise le systeme de description de pages,
 * de telle sorte que npages pages soient disponibles dans le POOL. La
 * structure 'process' est supposee a jour a l'entree de cette
 * fonction. Attention: pas d'appel a sbrk entre l'initialisation de
 * 'process' et l'appel a InitPages!
 **********************************************************************/

/**********************************************************************
 * Fonctions locales
 **********************************************************************/

static void     InitProcess(void);
static void     InitPages(npage_t);


extern void InitMM(npage_t npages)
{
    InitProcess();
    InitPages(npages);
    return;
}

static void InitProcess(void)
{
 {   DATAMAX /* defines char *datamax */
  {
      char* datastart = sbrk(0);
      val_t rounded_datastart = (val_t)RoundUpToPageSize(datastart);

      int status = brk((void*)rounded_datastart);

      if (status != 0)
        {
            error("InitProcess error\n");
        }
      EndMem = rounded_datastart;
      EndPage = PageNum(rounded_datastart);
      DescOffset = EndPage;
      MaxPages = PageNum(datamax);
#ifdef MONITOR_GC
      _report_maxpages(MaxPages-DescOffset);
#endif
  }
 }
}

static npage_t    EstimatePages(npage_t net_pg);

static void InitPages(npage_t npages)
{
    npage_t stat_size = EndPage;
    npage_t end_page = EstimatePages(stat_size + npages);

    if(end_page > MaxPages)    /* init size too big */
      {
          message("*** Warning : heap size requested too big!\n");
          message("Trying with maximum value : see limit(1))...\n");
          end_page = MaxPages;
      }
 {     
     npage_t desc_bloc = stat_size;     /* pfj */
     npage_t desc_size = NPDesc(end_page);            
     npage_t free_bloc = desc_bloc + desc_size;
     npage_t free_size = end_page - free_bloc;
     
     
     if (free_bloc ==  end_page)
          message("tas vide\n");
     else if (free_bloc > end_page)    /* ? */
       {
           error("erreur interne");
       }
     else ;
  {
      val_t new_break = PageAddress(end_page);
      int status = brk((char*)new_break);

      if(status != 0)
        {
            error("Initial heap expansion failed\n");
        }
					/* Initialization succeeded */
					/* describe memory */

      EndMem = new_break;		/* break */
      EndPage = end_page;		/* page of break */

      DescBloc = desc_bloc;		/* description bloc */
      DescSize = desc_size;
      PageDescs = (pagedesc_t*)PageAddress(desc_bloc) - DescOffset;
      DescCapacity = DescCapacity(desc_size);

                                        /* there are 3 blocks */
/*      BlocSize(0)         = stat_size;	/* hidden block: static area */
      BlocSize(desc_bloc) = desc_size;	/* descriptor block */
      BlocSize(free_bloc) = free_size;	/* and free block */

                                        /* tags : zone, mode, region */

/*      PageTag(0)         = StaticRegionTag(STD_HIDDEN_REG); */
      PageTag(desc_bloc) = StaticRegionTag(STD_GM_REG);
      PageTag(free_bloc) = StaticRegionTag(STD_FREE_REG); 
        
                                        /* pool */
      PoolFirst = free_bloc; 
      PoolSize = free_size;
      NextBloc(PoolFirst) = PoolFirst;          
      PreviousBloc(PoolFirst) = PoolFirst;
					/* heap : all but hidden */
      HeapSize = free_size + desc_size;	/* (=  EndPage - free_bloc) */

      return;				/* that's all */
  }
 }
}

/**********************************************************************
 * EstimatePages
 **********************************************************************
 * EstimatePages : rend le nombre de pages necessaires pour avoir
 * net_pg pages utiles plus les descripteurs (et les descripteurs des
 * descripteurs); remarque: ca converge tres vite. Domaine de validite
 * : voir la macro NPDesc : approximativement
 * MAXINT/sizeof(pagedesc_t), c'est a dire 2^32/16 = 2^28 pages = 256
 * Megapages. C'est large. (plus large que le type npage_t).
 **********************************************************************/

static npage_t EstimatePages(npage_t net_pg)
{
     ulint tot_pg_needed = net_pg;
       
     ulint tot_pg_est;
     ulint desc_pg_est;
     ulint desc_pg_needed;

     do
       {
              
            desc_pg_est = NPDesc(tot_pg_needed);
            tot_pg_est =  net_pg + desc_pg_est;
            desc_pg_needed = NPDesc(tot_pg_est);


            tot_pg_needed = net_pg + desc_pg_needed;
       }
     while(desc_pg_needed > desc_pg_est);

     
     if (tot_pg_est-DescOffset > MAXPAGES)
	  error("Debordement du type npage_t (nombre de pages)\n");
	
     return (npage_t)tot_pg_est;
}


/**********************************************************************
 * POOL
 **********************************************************************
 * Le POOL est la structure de donnees contenant l'ensemble des blocs
 * libres. Seuls les entetes de blocs sont maintenus (taille). C'est
 * lors du GC que les pointeurs arriere sont etablis.
 *
 * Le Pool de blocs est implante a l'aide d'une liste doublement
 * chainee : ceci facilite l'operation d'insertion. La strategie
 * d'allocation des blocs est 'first-fit circulaire', avec fusion a` la 
 * demande des blocs adjacents...
 **********************************************************************
 * OPERATIONS
 *
 * PoolRemove(b)        : extrait le bloc b du POOL.
 * PoolMerge(b1, b2)    : b2 adjacent a b1 a droite. fusionne b1 et b2.
 *
 * PoolGetBloc(sz)      : rend un bloc libre de taille sz, ou NullPage.
 * PoolPutFirst(b)      : remet le bloc en premier choix dans le POOL.
 * PoolPutLast(b)       : remet le bloc en dernier choix dans le POOL.
 **********************************************************************/


#define PoolRemove(b) \
{ \
     npage_t p = PreviousBloc((b)); \
     npage_t n = NextBloc((b)); \
 \
     NextBloc(p) = n; \
     PreviousBloc(n) = p; \
} \


#define PoolMerge(b1, b2)\
{ \
     BlocSize((b1)) += BlocSize((b2)); \
     PoolRemove((b2)); \
} \

/**********************************************************************
 * Variables du POOL
 **********************************************************************/

npage_t PoolFirst;                      /* point d'entree sur le POOL */
npage_t PoolSize;                       /* nb pages libres dans le POOL */


/**********************************************************************
 * PoolGetBloc : imple'mente la strategie de recherche first-fit
 * (premiere offre). Lors de la recherche d'un bloc de plusieurs
 * pages,  les blocs libres adjacents sont fusionnes a la demande.
 **********************************************************************/

extern npage_t PoolGetBloc(npage_t sz)
{
    npage_t start = PoolFirst;
    npage_t current = start;

    if (start == NullPage)
         return NullPage;               /* POOL vide  */

    do
      {
          slint curr_size = BlocSize(current);
          slint diff_size = curr_size - sz;

     RETRY:
          if (diff_size ==  0)          /* just fits */
            {
                npage_t next = NextBloc(current);

                if (next ==  current)
                  {                     /* last block in POOL */
                      PoolFirst = NullPage;
                      PoolSize -= sz;

                      if (PoolSize != 0)
                           error("Pool empty and PoolSize != 0 !\n");

#ifdef MONITOR_GC
		      if(_in_gc)
			   _record_minpool();
#endif		      
                      return current;
                  }
                PoolRemove(current);
                PoolFirst = next;
                PoolSize -= sz;
#ifdef MONITOR_GC
		if(_in_gc)
		     _record_minpool();
#endif		      
                return current;
            }
          else if (diff_size > 0)       /* fits large */
            {                           /* take beginning */
                npage_t rem = current + sz;
					/* remain is PoolFirst */
                BlocSize(rem) = diff_size;
	     {
		 npage_t next = NextBloc(current);
		 npage_t previous = PreviousBloc(current);

		 if (next == current)	/* sole block in pool? */
		   {			/* (previous ==  current too) */
		       NextBloc(rem) = rem;
		       PreviousBloc(rem) = rem;
		   }
		 else			/* re-link future free block */
		   {
		       PreviousBloc(next) = rem;
		       NextBloc(rem) = next;
		       PreviousBloc(rem) = previous;
		       NextBloc(previous) = rem;
		   }
	     }				/* make it free */
		PageTag(rem) = StaticRegionTag(STD_FREE_REG);
		PoolFirst = rem;	/* get it as POOL head */
                PoolSize -= sz;

                BlocSize(current) = sz;
		
                return current;
            }
          else                          /* diff_size < 0, does not fit */
            {                           /* try and merge following block */
                npage_t after = current + curr_size;

                while (after < EndPage && FreeBloc_P(after))
                  {                     /* merge after-block */
                      npage_t after_size = BlocSize(after);
                                        /* care for start-block */
                      if (after == start)
                           start = current;

                      BlocSize(after) = 0; /* (erase after) pfj */
                      PoolRemove(after);
                      curr_size += after_size;
                      BlocSize(current) = curr_size;
                      diff_size += after_size;
                                        /* does it fit now ? */
                      if (diff_size >= 0)
                           goto RETRY;  /* just or large fit ? */
                                        /* else consider next */
                      after = current + curr_size;
                  } 
            }                           /* here merging didn't give */
          current = NextBloc(current);  /* enough. advance. */
      }
    while (current != start); 

#ifdef MONITOR_GC
    if(_in_gc)
	 _record_minpool();
#endif		      
                                        /* POOL completely scanned */
    return NullPage;                    /* fail */
}


/* PooPutFirst: remet le bloc en premier choix dans la liste des 
 * blocs libres. Le bloc ne doit pas etre deja libre. 
 */

extern void PoolPutFirst(npage_t bn)
{
    npage_t poolhead = PoolFirst;
    if (poolhead == NullPage)           /* pool initialement vide */
      {
          NextBloc(bn) = bn;
          PreviousBloc(bn) = bn;
          PoolFirst = bn;
      }
    else
      {
          npage_t last = PreviousBloc(poolhead);
        
          NextBloc(last) = bn;
          PreviousBloc(bn) = last;
          NextBloc(bn) = poolhead;
          PreviousBloc(poolhead) = bn;
        
          PoolFirst = bn;               /* place bn en te^te */
      }
    PoolSize += BlocSize(bn);           /* maintient PoolSize */
}     
   
/* PooPutLast: remet le bloc dans la liste des blocs libres, en situation  
 * de dernier choix.
 */ 

extern void PoolPutLast(npage_t bn)
{
    npage_t poolhead = PoolFirst;
    if (poolhead == NullPage)           /* pool initialement vide */
      {
          NextBloc(bn) = bn;
          PreviousBloc(bn) = bn;
          PoolFirst = bn;
      }
    else
      {
          npage_t last = PreviousBloc(poolhead);
        
          NextBloc(last) = bn;
          PreviousBloc(bn) = last;
          NextBloc(bn) = poolhead;
          PreviousBloc(poolhead) = bn;
        
       /*   PoolFirst = poolhead;         /* pfj : pas de changement ! */
      }
    PoolSize += BlocSize(bn);           /* maintient PoolSize */
}     

/* InitBig: affecte les champs taille et tag du bloc. Les eventuels 
 * chainages du bloc restent a completer.
 */

extern void InitBig(npage_t pagenumber, npage_t nb, nregion_t nreg)
{
    BlocSize(pagenumber) = nb;
    PageTag(pagenumber) = RegionTag(nreg);
}


/**********************************************************************
 * ExpandHeap
 **********************************************************************
 * ExpandHeap(wished, needed): tente d'expanser le tas de maniere a
 * ajouter au moins needed pages contigues au POOL, et si possible
 * wished pages.
 *
 * La variete des strategies possibles (notamment pour le placement
 * d'un nouveau tableau des pages) rend la realisation de cette
 * expansion un peu poilue. La solution adoptee permet un ajout aise
 * d'autres strategies.
 **********************************************************************
 * Si un increment de wished pages ne peut etre alloue,  la plus
 * grande valeur superieure ou egale a needed est adoptee. Les
 * differentes strategies d'expansion "proposent" des expansions
 * maximales par le biais d'un descripteur d'expansion (ce qui permet
 * la comparaison et le choix). Le descripteur d'expansion choisi est
 * lui-meme passe a la fonction DoExpandHeap qui realise l'expansion:
 * appel au systeme,  puis mise a jour des parametres de l'executif.
 **********************************************************************/


typedef struct                          /* - descripteur d'expansion - */
{
    int     how;
    npage_t desc_bloc;                  /* nouveau TDP     */
    npage_t desc_size;                  /* taille en Pages */
    npage_t hidd_bloc;                  /* bloc cache      */
    npage_t hidd_size;                  /* taille en pages */
    npage_t incr_bloc;                  /* bloc alloue     */
    npage_t incr_size;                  /* taille en pages */
    npage_t free_bloc;                  /* nouv. bloc libre */
    npage_t free_size;                   /* sa taille */
} expandesc_t;

/**********************************************************************
 * Prototypes pour l'expansion
 **********************************************************************/

extern ulint    ExpandHeap(npage_t*, npage_t,  npage_t);

static void     FindHiddenBlock(expandesc_t*);
static ulint    DescribeExpansion(npage_t, npage_t, expandesc_t*);
static ulint    DescribeExp_pool(npage_t, npage_t, expandesc_t*);
static ulint    DescribeExp_increment(npage_t, npage_t, expandesc_t*);
static npage_t  NetWeight(expandesc_t *psol);
static ulint    DoHeapExpansion(expandesc_t*);

/* static npage_t    EstimatePages(npage_t net_pg); */



/**********************************************************************
 * ExpandHeap : tente d'expanser le tas (acquisition de pages memoire
 * par requete au systeme). Les deuxiemes et troisiemes arguments
 * specifient l'expansion desire : si possible 'wished' pages,  au
 * moins 'needed' pour faire un succes. Le premier argument recueille
 * la quantite de memoire effectivement acquise par l'expansion (hors
 * l'eventuel nouveau tableau des pages,  et hors l'eventuel bloc
 * cache).
 **********************************************************************/

/* ExpandHeap : fait appel a DescribeExpansion pour calculer une solution 
 * acceptable. Si l'expansion demandee echoue (appel a
 * DoHeapExpansion),  essaie une expansion minimum (npg_needed). La
 * fonction ExpandHeap retourne l'une des valeurs suivantes :
 * EXP_SUCCESS (success: expansion d'au moins npg_needed pages),
 * EXP_EMAX (echec,  limite des droits ou des possibilites),  EXP_EBRK
 * (echec de l'expansion: probablement par manque d'espace virtuel du
 * aux limites de la machines et aux circonstances de l'execution. 
 */

extern ulint ExpandHeap(npage_t *free_pages,
                        npage_t npg_wished,
                        npage_t npg_needed)
{
    expandesc_t desc = {0, 0, 0, 0, 0, 0, 0, 0, 0};
    int status;
    
    FindHiddenBlock(&desc);             /* check hidden block */
 {                                      /* describe it in desc */
     npage_t incr_bloc =  desc.incr_bloc;

     if ((incr_bloc + npg_needed) > MaxPages) 
       {
           return EXP_EMAX;             /* can not satisfy */
       }

     if((incr_bloc + npg_wished) > MaxPages) 
       {                                /* no more than max */
           npg_wished =  MaxPages - incr_bloc;
       }

 RETRY:
     status = DescribeExpansion(npg_wished,  npg_needed,  &desc);

     switch (status)
       {
      case EXP_SUCCESS:                 /* might succeed,  try and expand */
      case EXP_PROPOSAL:                /* idem */
           break;
      case EXP_EMAX:                    /* too much asking */
           return EXP_EMAX;
      default:
           error ("DescribeExpansion: bad status\n");
           break;                       /* pfj */
       }
     
     status =  DoHeapExpansion(&desc);  /* try and expand really */
     
     switch (status)                    /* see what happened */
       {

      case EXP_SUCCESS:                 /* SUCCESS (only point of) */
	   HeapSize += desc.incr_size;	/* update heap description */
           *free_pages = desc.free_size;/* transmit gain to caller */
#ifdef MONITOR_GC
	   _report_expansion();
#endif
           return status;		/* and return */

      case EXP_EBRK:                    /* sbrk failed, try less */
           if (npg_wished > npg_needed) /* only ask for what is needed */
             {
                 npg_wished = npg_needed; 
                 goto RETRY;
             }
           else 
                return EXP_EBRK;        /* was min. needed : fail */
      default:
           error("DoHeapExpansion: bad status\n");
           break;                       /* pfj */
       }
     return status;                     /* pfj */
 }
}


/* FindHiddenBlock : detecte l'eventuel bloc cache (entre le dernier 
 * break connu et le break courant),  et renseigne les champs
 * hidd_bloc,  hidd_size et incr_bloc du descripteur d'expansion.
 */

static void FindHiddenBlock(expandesc_t *psol)
{
    val_t cur_break = (val_t)sbrk(0);
    npage_t hidd_bloc = EndPage;

    npage_t hidd_size;
    npage_t incr_bloc;
    

    if(cur_break < EndMem)
      {
          error("Someone de-allocated my memory!\n");
          return;                       /* pfj */
      }
    else if(cur_break == EndMem)
      {
          incr_bloc = hidd_bloc;
          hidd_size = 0;
      }
    else                                /* cur_break > EndMem */
      {
          incr_bloc = NPages(cur_break);
          hidd_size = incr_bloc - hidd_bloc;
      }

    psol->hidd_bloc = hidd_bloc;
    psol->hidd_size = hidd_size;
    psol->incr_bloc = incr_bloc;
    
    return;
}



/* DescribeExpansion: essaie successivement les deux strategies d'expansion 
 * donnees par DescribeExp_pool et DescribeExp_increment. Si les deux
 * echouent (npg_wished),  rend la meilleure des deux suggestions ( >=
 * npg_needed). 
 */

static ulint DescribeExpansion(npage_t wished_pg,
                               npage_t needed_pg,
                               expandesc_t *psol)
{
    expandesc_t solution_pool =  *psol;
    expandesc_t solution_increment = *psol;
    ulint status_pool;
    ulint status_increment;

    status_pool = DescribeExp_pool(wished_pg, needed_pg, &solution_pool); 
    if (status_pool ==  EXP_SUCCESS)
      {
          *psol = solution_pool;        /* first choice */
          return EXP_SUCCESS;
      }

    status_increment = DescribeExp_increment(wished_pg, needed_pg,
                                           &solution_increment);
    if(status_increment == EXP_SUCCESS)
      {
          *psol =  solution_increment;  /* second choice */
          return EXP_SUCCESS;
      }

    switch(status_pool)                 /* default: better proposal */
      {
     case EXP_PROPOSAL:         
       {                                
           if(status_increment ==  EXP_PROPOSAL)
             {                          /* is other proposal better? */
                 npage_t possible_in_pool = NetWeight(&solution_pool);
                 npage_t possible_in_increment = NetWeight(&solution_increment);
                                        /* choose maximum */
                 if (possible_in_pool >= possible_in_increment)
                      *psol = solution_pool;
                 else
                      *psol = solution_increment;

                 return EXP_PROPOSAL;
             }
           else
             {
                 *psol = solution_pool;
                 return EXP_PROPOSAL;
             }
       }
     case EXP_EBLOC:
     case EXP_EMAX:
          break;
     default: 
          error("DescribeExpansion : mauvais code de retour\n");
      }

    if (status_increment == EXP_PROPOSAL)
      {
          *psol = solution_increment;
          return EXP_PROPOSAL;
      }

    return EXP_EMAX;                    /* fail */
}


/* NetWeight: accepte une description d'expansion,  rend le nombre de 
 * pages utiles de cette solution.
 */

static npage_t NetWeight(expandesc_t *psol)
{
    switch (psol->how)
      {
     case IN_SAME:
     case IN_POOL:
          return psol->incr_size;
     case IN_BEGI:
          return psol->incr_size - psol->desc_size;
     case IN_VOID:
          return 0;
     default:
          error("error\n");
          return 666;                   /* pfj*/
      }
}


/* DescribeExp_pool: cherche a realiser l'expansion en gardant le 
 * tableau des pages dans le POOL actuel. DescribeExp_pool ne fait
 * pas de proposition,  et de ce fait ignore son second argument
 * 'needed'. La fonction DescribeExp_pool retourne l'une des valeurs
 * suivantes : EXP_SUCCESS (succes),  EXP_EBLOC (pas possible).
 */

static ulint DescribeExp_pool(npage_t wished,  
                            npage_t needed,
                            expandesc_t *psol)
{
    npage_t new_pagebreak = psol->incr_bloc + wished;
    npage_t new_desc_size = NPDesc(new_pagebreak);

    if (DescSize >=  new_desc_size)     /* succes */
      {
          psol->free_bloc = psol->incr_bloc;
          psol->free_size = wished;
          psol->incr_size = wished;
          psol->how = IN_SAME;
          return EXP_SUCCESS;
      }
    else                                /* try and get a new */
      {                                 /* page table in POOL */
          npage_t new_desc_bloc = PoolGetBloc(new_desc_size);

          if (new_desc_bloc != NullPage)
            {                           /* put bloc in first choice */
                PoolPutFirst(new_desc_bloc);
                psol->free_bloc = psol->incr_bloc;
                psol->free_size = wished;
                psol->desc_bloc = new_desc_bloc;
                psol->desc_size = new_desc_size;
                psol->incr_size = wished;
                psol->how = IN_POOL;
                return EXP_SUCCESS;
            }
          else
            {
                psol->how = IN_VOID;
                return EXP_EBLOC;       /* fail */
            }                           /* (no other proposal) */
      }
}


/* DescribeExp_increment: tente de realiser l'expansion en placant un
 * nouveau tableau des pages dans l'increment (au debut). Une solution
 * maximum est determinee en cas d'echec, et proposee si elle est
 * acceptable. La fonction DescribeExp_increment retourne l'une des
 * valeurs suivantes: EXP_SUCCES (succes), EXP_EMAX (limite des
 * droits), EXP_PROPOSAL (proposition < wished).
 */

static ulint DescribeExp_increment(npage_t wished,  
                                 npage_t needed,  
                                 expandesc_t *psol)
{
    npage_t incr_bloc = psol->incr_bloc;
    ulint new_end_page = EstimatePages(incr_bloc + wished);

    if (new_end_page <= EndPage) /* success */
      {                                 /* place descs at beginning */
          npage_t free_size = wished;
          npage_t free_bloc = new_end_page - free_size;
          npage_t incr_size = new_end_page - incr_bloc;
          npage_t desc_bloc = incr_bloc;
          npage_t desc_size = free_bloc - incr_bloc;
                                        /* report in descriptor */
          psol->free_bloc = free_bloc;
          psol->free_size = free_size;
          psol->incr_size = incr_size;
          psol->desc_bloc = desc_bloc;
          psol->desc_size = desc_size;

          psol->how = IN_BEGI;
          return EXP_SUCCESS;
      }
    else                                /* searching max. solution */
      {                                 /* must at least describe */
          npage_t minimum = EstimatePages(incr_bloc);
          /* what exists */
          if (minimum > MaxPages) 
            {                           /* could not add a new */
                psol->how = IN_VOID;    /* description array at end */
                return EXP_EMAX;        /* of memory,  so fail */
            }
          else                          /* compute max. expansion */
            {
                npage_t new_desc_size = minimum - incr_bloc;
                npage_t desc_capacity = DescCapacity(new_desc_size);

                psol->desc_bloc = incr_bloc;
                psol->incr_size = MaxPages - incr_bloc;

                if (desc_capacity >= MaxPages) /* all is described */
                     ;                  /* nothing to do */
                else                    /* else must describe */
                  {                     /* remaining pages */ 
                      npage_t rem_size = MaxPages - desc_capacity;
                      npage_t rem_desc_size = NPDesc(rem_size);
                      new_desc_size = new_desc_size + rem_desc_size;
                  }
             {
                 npage_t useful_bloc = incr_bloc + new_desc_size;
                 npage_t useful_size = MaxPages - useful_bloc;

                 if(useful_size >= needed)
                   {
                       psol->free_bloc = useful_bloc;
                       psol->free_size = useful_size;
                       psol->desc_size = new_desc_size;
                       psol->how = IN_BEGI; /* propose */
                       return EXP_PROPOSAL;
                   }
                 else
                   {
                       psol->how = IN_VOID; /* fail */
                       return EXP_EMAX;
                   }
             }
                
            }
      }
}
          
/* (1) : ce cas de figure correspond a un increment utile nul : on ne
 * fait que deplacer le tableau des pages a la fin du tas dans un
 * increment qui lui est consacre. Si ca marche, (2) le tableau des pages
 * peut decrire soit (i) tout jusqu'a MaxPages,  soit (ii) une
 * partie seulement. Le reste se decompose en (a) les pages
 * necessaires a la description de ce reste,  et (b) le reste 'utile'.
 * Noter que NPDescs(x) <=  x pour tout x. 
 */


/**********************************************************************
 * DoHeapExpansion
 **********************************************************************
 * DoHeapExpansion(psol) : etend le tas selon la solution calcule'e.
 * Cette fonction interprete le descripteur d'expansion dont l'adresse
 * lui est donnee. Le break est d'abord affecte, puis le nouveau tas
 * est decrit dans son (eventuellement nouveau) tableau des pages.
 **********************************************************************/


static ulint DoHeapExpansion(expandesc_t *psol) 
{
    if (psol->how == IN_VOID)
         return EXP_EBLOC;
 {
     npage_t new_end_page = psol->incr_bloc + psol->incr_size;
     val_t new_break = PageAddress(new_end_page);
     npage_t new_desc_bloc_diff = 0;
     
     int     status = brk((caddr_t)new_break);
     if (status != 0)
          return EXP_EBRK;

     EndPage = new_end_page; /* (1) nouveau break */
     EndMem = new_break;

     switch(psol->how)                  /* (2) nouveau TDP */
       {
      case IN_SAME:
           break;
      case IN_POOL:                     /* really take the desc_bloc */
        {
            npage_t new_desc_bloc = PoolGetBloc(psol->desc_size);
            if (new_desc_bloc != psol->desc_bloc)
              {
                  error("DoExpandHeap : bad new_desc_bloc\n");
                  break;                /* pfj */
              }
            new_desc_bloc_diff = psol->desc_size - DescSize;
        }                               /* fall through */
      case IN_BEGI:
        {
            void *from = PageDescs + DescOffset;
            void *to = PageAddress(psol->desc_bloc);
					/* npage_t DescSize (a changer!)  */
            nbytes_t nbytes = DescSize * PageSize; 
                                        /* ancien TDP */
            ChangeRegion(DescBloc, STD_FREE_REG);
            PoolPutLast(DescBloc);

            memcpy(to, from, nbytes);        /* initialise nouveau TDP */
            DescBloc = psol->desc_bloc;      /* change de TDP */
            PageDescs = (pagedesc_t*)PageAddress(DescBloc) - DescOffset; 
            DescSize = psol->desc_size;
            DescCapacity = DescCapacity(psol->desc_size);

            InitBig(psol->desc_bloc, psol->desc_size, STD_GM_REG);


            break;
        }
      default:
        {
            error("DoExpandHeap : bad psol->how\n");
            break;                      /* pfj */
        }
       }
  {                                     /* describe new free bloc */
      npage_t free_bloc;
      npage_t free_size;
      
      switch(psol->how)
        {
       case IN_POOL:
       case IN_SAME:
            free_bloc = psol->incr_bloc;
            free_size = psol->incr_size;
            break;
       case IN_BEGI:
            free_bloc = psol->desc_bloc + psol->desc_size;
            free_size = psol->incr_size - psol->desc_size;
            break;
       default:
            error("DoHeapExpansion : error\n");
            return 666;
        }
      InitBig(free_bloc, free_size, STD_FREE_REG);
      PoolPutFirst(free_bloc);
  }                                     /* bloc cache */
     if (psol->hidd_size != 0)          
          InitBig(psol->hidd_bloc, psol->hidd_size, STD_HIDDEN_REG);

     
     return EXP_SUCCESS;
 }
}

