/* $Header: /Nfs/radyr/usr11/rc136/Projet/GC/RCS/gc.c,v 1.1 1992/09/03 21:15:02 rc136 Exp rc136 $ */
/* Auteur: Vincent Delacour (delacour@poly.polytechnique.fr) */
/* gc.c: tout le gc de K2. */

/**********************************************************************
 *                                gc
 **********************************************************************
 * Le gc est client du modele memoire et de ses services. Il fournit
 * un service aux fonctions d'allocation de haut niveau.
 * Le gc est configure' en accord avec l'implementation choisie par
 * son client : K2. L'utilisation qui est faite des eventuels tags
 * doit bien sur etre coherente entre ces deux parties.
 **********************************************************************/

#include "../Include/parametres.h"     
#include "../Include/public.h"                    
#include "private.h"                    
#include "gc.h"

typedef int obj_t; /* coherent avec z2k2.h */
#define KREF(d)         ((val_t)((char*)(d)+1))
#define CREF(x)         ((val_t)((char*)(x)-1))

#include <sys/time.h>
#include <sys/resource.h>

#include "monitor.h"

/**********************************************************************
 * unix...
 **********************************************************************/


/* extern char *sbrk(int);
   extern int   brk(void *);
   extern int   write(int, char*, int);
   extern void  exit(int);
   extern void  printf(char*, ...); */
/* extern void *memset(char*, char, int); */
/* extern void *memcpy(char*,char*,int); */
/* extern int   strlen(char *); */

/* extern void *stderr, *stdout, *stdinput; */

/**********************************************************************/


extern nwords_t    initial_nglobals;	/* A fournir */
extern val_t      *initial_pglobals;	/* A fournir */
extern npage_t     DescOffset;

/**********************************************************************
 *                       Section declarative
 **********************************************************************/


/**********************************************************************
 * Regions utilisees par le gc
 *
 * Les 3 regions d'allocation utilisees par le gc sont 'BadRegions'
 * car elles donnent lieu a arrondi dans la tenue du compte de la
 * memoire libre.
 **********************************************************************/


#define UNTRACED_REG    3
#define AMBROOTS_REG    4
#define BITMAP_REG      5

#define NGCBADREGIONS   3 

int NGCBadRegions = NGCBADREGIONS;

/**********************************************************************
 * Etat du GC : variables.
 **********************************************************************/

int GCVerbose = 1;		/* verbose par defaut */

/*
 * Racines
 */ 

extern int NRoots;              /* nombre de racines */
extern val_t *PRoots;          /* tableau des adresses des racines */
                                /* (les racines sont de type void*) */

extern int NGcVars;             /* nb variables a visiter */
extern obj_t *GcVars[];         /* tableau initial des racines */

/* 
 * Acces aux regions (champs dedies au gc)
 */

#define NextRegion(nreg)        (PRegions[(nreg)]->next)


/* .Subsection PAGE PROMOTION */

#define COPYZONE_PUT(bn) \
                if (to_head == NullPage) \
                  { \
                      to_head = bn; \
                      to_tail = bn; \
                      NextPage(bn) = NullPage; \
                  } \
                else \
                  { \
                      NextPage(to_tail) = bn; \
                      NextPage(bn) = NullPage; \
                      to_tail = bn; \
                  } \


/**********************************************************************
 * Prototypes des fonctions locales
 **********************************************************************/

extern void    InitGC(void);            /* utilise' par InitMemoryManager */

static val_t CopyAllocateSmall(nwords_t,  nregion_t);
static val_t GCAllocateSmall(nwords_t, nregion_t);

static void    Prologue(void);
static void    ReleaseAllocationPages(void);
static void    NormalizePages(void);
static void    FlipPages(void);
static void    CleanBitMap(npage_t, nregion_t);
static void    CleanBitMap_Opaque(npage_t, nregion_t);

static void    ScanStack(void);
static void    Scavenge(void);
static void    Epilogue(void);

static void    CheckStackReference(val_t);
static void    ScanStack(void);
static void    do_scanstack(void);

static ulint   ScavengeLoc(val_t);



/* - Stackregs - */

void    PushUntraced(val_t);
void    PushAmbRoot(val_t);
val_t PopUntraced(void);
val_t PopAmbRoot(void);

void    Push_Static(val_t , nregion_t);
val_t Pop_Static(nregion_t);


/**********************************************************************
 * Bitmaps (tables de bits)
 **********************************************************************
 * L'implementation des bitmaps est cablee pour des mots de 32 bits:
 * 
 * - entiers longs non signes de 32 bits : la table de bits est vue
 * comme un tableau de mots de marquage; 
 * - pointeurs de 32 bits.
 * 
 * Une solution plus portable serait beaucoup plus lourde.
 **********************************************************************
 * MShift(add): donne le decalage a appliquer au mot de marquage pour
 * placer le bit de marquage a droite.
 *
 * MWordNumber(add): donne le numero de mot de marquage dans la table
 * de bits.
 *
 * MWord(add, bitmap): raccourci; donne le mot de de marquage dans la
 * table de bits.
 *
 * ClearBitMap(bm): efface tous les bits de marquage. 
 **********************************************************************/


/*
 * Allocation des bitmaps.
 */

#define ClearBitMap(bm) (void)memset((char*)(bm), 0, BitMapSize)


extern bitmap_t AllocateBitMap(void)
{
    bitmap_t bm;
    val_t ap, new_ap;

     ap = StaticRegionAP(BITMAP_REG);
     if((new_ap = ap + OneBitMap) > StaticRegionLimit(BITMAP_REG))
          bm = (bitmap_t)GCAllocateSmall(OneBitMap, BITMAP_REG);
     else
       {
           StaticRegionAP(BITMAP_REG) = new_ap;
           bm = (bitmap_t)ap;
       }
    ClearBitMap(bm);
    return bm;
}


/**********************************************************************
 * Recopie des objets
 **********************************************************************/

#define CopyWords(from, to, nwords) \
{ \
    int i; \
    for(i = 0; i < nwords; i++) \
         (to)[i] = (from)[i]; \
} \


#define M_CopyAllocateSmall(sz, nreg, res) \
{ \
     val_t ap, new_ap; \
 \
     ap = RegionAP(nreg); \
     if((new_ap = ap + sz) > RegionLimit(nreg)) \
          res = CopyAllocateSmall(sz, nreg); \
     else \
       { \
            RegionAP(nreg) = new_ap; \
            res = ap; \
       } \
} \


/**********************************************************************
 *		      Pointeurs de renvoi,  tags
 **********************************************************************
 * Les pointeurs de renvoi enrichissent un peu l'usage qui peut etre
 * fait des tags (un cas de plus). L'utilisation des tags doit etre en
 * accord avec celle de l'implementation cliente.
 **********************************************************************/

#if TAGSHIFT != 1
/*        ** Erreur: Le format des forwarding pointers veut que
        tagshift ==  1 */
#endif
#if REFTAG != 1
/*     ** erreur : editer les definitions ci dessous */
#endif


#define FP2KREF(fp)     ((void*)((ulint)(fp) ^ 2))
#define CREF2FP(cref)   ((void*)((ulint)(cref) | 3))

#define Forward_P(ref)  (((ulint)(ref) & 1) != 0) && (((ulint)(ref) & 2) != 0)


#define HalfWordStackAlignment  (0) /* faux sur mips R2000 */

/**********************************************************************
 * Mesures diverses
 *
 * Temps d'execution du gc,  nombre de gc,  temps cumules des gc,
 * mesures sur le graphe d'objets,  etc.
 **********************************************************************/

int _gc_count = 0;



/**********************************************************************
 * FreeMem
 *
 * FreeMem est defini dans alloc.h.
 **********************************************************************/

extern nwords_t FreeMem;                /* quantite de memoire disponible */
extern npage_t RoundUpPages;            /* pages d'arrondi (reg. inhom.) */
extern int      NBadRegions;            /* nb regions (utilisateur) a */
                                        /* residu variable */  

/**********************************************************************
 * Pile d'execution
 *
 * L'origine de la pile est determinee d'une maniere dependant du systeme
 **********************************************************************/

void* usrstack = NULL;

/**********************************************************************
 *                               GC
 **********************************************************************/


int gc(npage_t needed)
{
    nwords_t old_free_mem = FreeMem;

    _gc_count += 1;

#ifdef MONITOR_GC
    _report_gc_begins();
#endif
    
    if (GCVerbose)
	 message("GC : Prologue...");
    Prologue();
    if (GCVerbose)
	 message("OK Scanning stack...");
    ScanStack();
    if (GCVerbose)
	 message("OK Scavenging...");
    Scavenge();
    if (GCVerbose)
	 message("OK Epilogue...");
#ifdef _GC_MONITOR_LOCALITY
    if (_gc_monitor_locality)
      {
	  _report_locality_before(); /* ancien graphe */
	  _report_heap_locality();	/* nouveau graphe */
      }
    else
      {
	  _report_no_locality();
      }
#endif
    Epilogue();
    if (GCVerbose) message("OK\n");
#ifdef MONITOR_GC
    _report_gc_ends();
#endif

 {
    nwords_t new_free_mem = FreeMem;

     if (new_free_mem >=  old_free_mem)
       {
           nwords_t gain = new_free_mem - old_free_mem;
           if (Words2Pages(gain) >= needed)
	     {
		 return GC_SUCCESS;
	     }
           else
                return GC_FAIL;
       }
     else
          error("Backwards gc ??\n");
     return GC_FAIL;                    /* pfj */
 }
}

/**********************************************************************
 * Racines globales
 *
 * Les racines globales sont communiquees au gc par le biais d'un
 * tableau d'adresses. 
 **********************************************************************/


int NRoots;                     /* nombre de racines */
val_t *PRoots;                 /* tableau des adresses des racines */
                                /* (les racines sont de type void*) */

static npage_t to_head;         /* bloc contenant le scav. pointer      */
static npage_t to_tail;         /* derniere page chainee en zone to     */



/**********************************************************************
 *                             Prologue
 **********************************************************************/

static void Prologue(void)
{
    ReleaseAllocationPages();
    NormalizePages();
    return;                             /* return */
}

/* ReleaseAllocationPages : relache les pages d'allocation des
 * regions. Pour chaque region, deux possibilites : (1) ap =  0;
 * (2) ap > debut de la page d'allocation. Dans ce dernier cas, ap -1
 * pointe dans la page d'allocation (noter que ap peut pointer juste
 * apres la page d'allocation.). Les deux attributs de region AP et
 * Limit sont reinitialises.  L'attribut SCP (scavenging pointer) est
 * initialise.
 *
 * ReleaseAllocationPages a une deuxieme mission: reinitialiser les
 * champs freehead et freetail des regions: les free listes sont
 * recalculees lors de l'epilogue. 
 */

static void ReleaseAllocationPages(void)
{
    nregion_t nreg;
    for(nreg = 0; nreg < NRegions; nreg++)
      {
          val_t ap = RegionAP(nreg);  /* allocation pointer */

          if (ap != NULL)
            {
                npage_t alloc_page = RegionAPG(nreg);
#ifdef DEBUG		
		if(alloc_page != PageNum(ap -1))
		  {
		      printf("%x %x %x\n", alloc_page, ap, PageNum(ap-1));
		      error("discordance entre ap et apg\n");
		  }
#endif
                PageTop(alloc_page) = ap;
            }
          RegionAPG(nreg) = 0;
          RegionAP(nreg) = NULL;
          RegionLimit(nreg) = NULL;
          RegionSCP(nreg) = NULL;
					/* free listes */
	  RegionFreeHead(nreg) = NULL;
	  RegionFreeTail(nreg) = NULL;
	  
      }
    return;                             /* pfj */
}
            
         
/* NormalizePages: les blocs sont supposes bien formes (BlocSize). 
 * Les pages 'suite' des blocs sont traitees : etablissement
 * du pointeur arriere vers la page de tete du bloc. Incorporation
 * a la zone du bloc (superficialite d'acces, 1ere decision).
 * Pseudo-mode 'Follow'. Sommet du bloc. 
 */

static void NormalizePages(void)
{
    npage_t pn = DescOffset;                     /* (et non NullPage) */
    do
      {
          npage_t bsz = BlocSize(pn);
          npage_t tail_pg = bsz - 1;
          if(tail_pg == 0)
            {
                ++pn;
                continue;
            }
          else
            {
                npage_t head = pn;
                val_t top = PageTop(pn);
                pagetag_t pt = PageTag(pn);
                pt.mode = FOLLOW_MODE;

                do
                  {                     /* sur les pages suite */
                      ++pn;
                      PageTag(pn) = pt;
                      PageTop(pn) = top;
                      BlocHead(pn) = head;
                  }
                while(--tail_pg != 0);

                ++pn;
            }
      } while(pn != EndPage);           /* cf Note(Risque) */
}


/**********************************************************************
 *                              Epilogue
 **********************************************************************/

static void Epilogue(void)
{
    FlipPages();
    return;
}


/* FlipPages : reconstitution des invariants du gc : les objets actifs
 * se trouvent dans la zone FROM, la variable FreeMem reflete la
 * consommation de memoire en accord avec le mode des objets.
 *
 * Les residus variables sont traites comme dans la fonction
 * d'allocation de petits objets. nb: les pages d'allocation "en
 * cours" payent d'avance le residu maximum, car leur sommet (PageTop)
 * indique une page fictivement pleine : voir le changement de page
 * dans GCAllocateSmall. Cette maniere de faire,  un tout petit peu
 * plus couteuse que le minimum possible,  evite une passe sur les
 * regions d'allocation apres le balayage des pages par FlipPages (ou
 * un test dans FlipPages : pour chaque page,  avant de decompter le
 * residu variable : ``s'agit-il de la page d'allocation de la
 * region?''...).
 * 
 */

static void FlipPages(void)
{
    npage_t pn = DescOffset;            /* (et non NullPage) */
    nwords_t words = 0;

    PoolFirst = NullPage;               /* reconstruction du POOL */
    PoolSize = 0;
    do
      {
          pagetag_t pt = PageTag(pn);
          npage_t bsz = BlocSize(pn);
          switch (pt.zone)
            {
           case HIDDEN_ZONE:            /* coutent leur taille */
                words += bsz * OnePage;
                break;

           case FREE_ZONE:              /* les blocs libres le restent */
           case GC_ZONE:                /* bit-maps */
           case FROM_FIX_ZONE:          /* pages non-atteintes */
           case FROM_FIX_OPAQUE_ZONE:   /* idem */
           case FROM_COPY_ZONE:         /* objets recopies */
           case FROM_COPY_OPAQUE_ZONE:  /* idem */
           case FROM_BIG_ZONE:          /* pages non-atteintes */
           case FROM_BIG_OPAQUE_ZONE:   /* idem */
                PageTag(pn) = STD_FREE_PTAG;
                PoolPutLast(pn);        /* accumule dans le POOL */
                break;                  /* dans l'ordre physique */

                                        /* petits objets fixes */
           case TO_FIX_ZONE:
                PageTag(pn).zone = FROM_FIX_ZONE; /* BUG 2 */
                CleanBitMap(pn,pt.nregion);        
                words += bsz * RegionCost(pt.nregion);
                break;

           case TO_FIX_OPAQUE_ZONE:     /* decompte de FreeMem */
                                        /* pas de residu variable */
                PageTag(pn).zone = FROM_FIX_OPAQUE_ZONE; /* BUG 2 */
					/* make free list */
		CleanBitMap_Opaque(pn,pt.nregion);
                words += bsz * RegionCost(pt.nregion);
                break;

                                        /* PETITS OBJETS COPIABLES */
					/* traiter le residu variable */
           case TO_FIXED_OPAQUE_ZONE:   
					/* make free list */
		CleanBitMap_Opaque(pn,pt.nregion);
					/* and fall through */
           case TO_COPY_OPAQUE_ZONE:
                PageTag(pn).zone = FROM_COPY_OPAQUE_ZONE; /* BUG 2 */
                goto FREEMEM_COST;

           case TO_FIXED_ZONE:		
		CleanBitMap(pn,pt.nregion);
					/* and fall through */
           case TO_COPY_ZONE:
                PageTag(pn).zone = FROM_COPY_ZONE; /* BUG 2 */

           FREEMEM_COST:
             {
                 nregion_t nreg = pt.nregion;

                 PageTag(pn) = RegionTag(nreg);
              {
                  nwords_t cost = RegionCost(nreg);
                  nwords_t waste = RegionWaste(nreg);

                  if(waste != 0)        /* region inhomogene */
                    {
                        val_t top = PageTop(pn);
                        nwords_t actual_waste =
                             (val_t)(RoundUpToPageSize(top)) - top;
                        nwords_t remainder = RegionRemainder(nreg);
                        nwords_t diff = waste - actual_waste;
                        nwords_t max_remainder = OnePage - waste;
                        nwords_t new_remainder = remainder + diff;

                        if (new_remainder > max_remainder) 
                          {             /* residu plein */
                              new_remainder = new_remainder - (max_remainder);
                              cost += OnePage;
                          }

                        RegionRemainder(nreg) = new_remainder;
                        words += cost;
                    }
                  else
                    {
                        words += bsz * RegionCost(pt.nregion);
                    }
              }
             }
                
                break;                          
        
           case TO_BIG_ZONE:
                PageTag(pn).zone = FROM_BIG_ZONE; /* BUG 2 */
                words += bsz * OnePage;
                break;

           case TO_BIG_OPAQUE_ZONE:
                PageTag(pn).zone = FROM_BIG_OPAQUE_ZONE; /* BUG 2 */
                words += bsz * OnePage;
                break;
#ifdef DEBUG
           default:
                error("flip pages : mauvaise zone");
#endif
            }
          pn += bsz;
      }
    while(pn != EndPage);  
    FreeMem = ((EndPage - RoundUpPages - DescOffset) * OnePage) - words;

#ifdef DEBUG
    if (PoolFirst == NullPage)
        error("Pool vide a l'issue du gc!");
#endif

    PoolFirst = NextBloc(PoolFirst);    /* positionne au premier bloc */
           
    return;
}




/**********************************************************************
 * CleanBitMap
 *
 * Nettoyage des objets libres. La page et la bitmap sont parcourues
 * en parallele. Une liste des objets libres est constituee, et
 * ajoutee a celle de la region (si besoin est).
 * 
 **********************************************************************
 * 
 * Bis Repetita : le code pour les bitmaps est cable pour des pointeur
 * de 32 bits et des long ints de 32 bits.
 **********************************************************************/


#define ALLMARKED_2     0x55555555
#define CLEANWORD       0

static void CleanBitMap(npage_t pn, nregion_t nreg)
{
    bitmap_t bm = PageBitMap(pn);       /* attributs de page */
    pagetag_t pt = PageTag(pn);
    val_t top = PageTop(pn);

    val_t obj = PageAddress(pn);	/* premier objet dans page */
    ulint mword = bm[0];                /* premier mot de marquage */
    ulint mask = 1;                     /* premier bit de marquage */

    val_t dummy[2];			/* dummy object. Start of free list */
    val_t head = (void*)&dummy; 	
    val_t tail = (void*)&dummy;
    
    switch(pt.mode)
      {
     case W2_MODE:
          do
            {
                if(mword == ALLMARKED_2)/* tous objets marques */
                  {
                      bm += OneWord;	/* mot marquage suivant */
                      mword = bm[0];
                      obj += 32;        /* objet correspondant */
                      continue;
                  }
                do
                  {
                      if ((mword & mask) == 0) /* objet non atteint */
                        {		/* a inclure dans la free liste */
			    tail[0] = DISGUISE(obj);
			    tail = obj;
			    obj[1] = CLEANWORD; /* '' */
			   
                        }
                      obj += 2 * OneWord; /* objet suivant */
                      mask <<= 2;       /* bit suivant */
                  }
                while(mask != 0);       /* mot de marquage fini */

                bm += OneWord;		/* mot de marquage suivant */
		mword = bm[0];
                mask = 1;               /* premier bit */
                                        /* ici, obj a avance' de 32 mots */
            }
           while(obj < top);
	  tail[0] = DISGUISE(NULL);	/* finit liste libre */
          break;
                 
     case SMALLSIZE_MODE:               /* taille dans la region */
     case W1_MODE:                      /* provisoire */
     case W3_MODE:                      /* idem */
     case W4_MODE:                      /* idem */
     case W8_MODE:                      /* idem */
       
       {                                /* taille en mots */
           nwords_t sz = RegionSize(pt.nregion);
           while(1)
             {
                 if((mword & mask) == 0) /* objet non atteint */
                   {
                       int i;           /* le nettoyer */
                       for(i = 1; i < sz; i++)
                            obj[i] = CLEANWORD;
					/* inclure dans la free liste */
		       tail[0] = DISGUISE(obj);
		       tail = obj;
                   }
                 obj += sz;             /* objet suivant */
                 if (obj >= top)        /* arret */
                      break;
                 mword = MWord(obj, bm); /* sc */
                 mask = 1 << MShift(obj);
             }
	   tail[0] = DISGUISE(NULL);	/* finir liste libre */
           break;
       }
       
              
     case HEADER_MODE:                  /* tailles dans les objets */
       {
           nwords_t sz = CINT(obj[0]); /* taille du premier objet */
           while(1)
             {
                 if((mword & mask) == 0) /* objet non-atteint */
                   {                    /* a` nettoyer */
                       int i;
					/* nettoyer champs [2..sz)  */
                       for(i = 2; i < sz; i++)
                            obj[i] = CLEANWORD;
					/* inclure dans free-list: le lien */
		       tail[1] = DISGUISE(obj);
		       tail = obj;
                   }
                 obj += sz;             /* suivant */
                 if (obj >= top)
                      break;
                 sz = CINT(obj[0]);	/* taille suivant */
                 mword = MWord(obj, bm); /* mot marquage */
                 mask = 1 << MShift(obj); /* et bit */
             }
       }
	  tail[1] = DISGUISE(NULL);	/* finir liste libre */
          break;

     case FOLLOW_MODE:
     case BIG_MODE:
     case NO_MODE:
#ifdef DEBUG
     default:
#endif
          error("CleanBitMap : bad mode\n");
      }

    
    /* Ici, on a la liste des objets libres dans la page. Si cette */
    /* liste est non-vide, l'ajouter a la liste libre de la region. */
    
 {
     val_t free_list = UNDISGUISE(head[0]);

     if (free_list != NULL)
       {
	   region_t *preg = Region(nreg);

	   if (preg->freehead == NULL)
	     {				/* free liste de region vide */
		 preg->freehead = free_list;
		 preg->freetail = tail;
	     }
	   else		        /* ajouter a la free liste de region */
	     {
		 if (pt.mode == HEADER_MODE)
		      preg->freetail[1] = DISGUISE(free_list);
		 else
		      preg->freetail[0] = DISGUISE(free_list);
		 preg->freetail = tail;
	     }
       }
    return;
 }
}

/* Pour les regions opaques. Les objets ne sont pas "nettoyes". Cette */
 /* fonction se deduit trivialement de la precedente, par suppression */
 /* notamment des lignes contenant le mot CLEANWORD. */

static void CleanBitMap_Opaque(npage_t pn, nregion_t nreg)
{
    bitmap_t bm = PageBitMap(pn);       /* attributs de page */
    pagetag_t pt = PageTag(pn);
    val_t top = PageTop(pn);

    val_t obj = PageAddress(pn);	/* premier objet dans page */
    ulint mword = bm[0];                /* premier mot de marquage */
    ulint mask = 1;                     /* premier bit de marquage */

    val_t dummy[2];			/* dummy object. Start of free list */
    val_t head = (void*)&dummy; 	
    val_t tail = (void*)&dummy;
    
    switch(pt.mode)
      {
     case W2_MODE:
          do
            {
                if(mword == ALLMARKED_2)/* tous objets marques */
                  {
                      bm += OneWord;	/* mot marquage suivant */
                      mword = bm[0];
                      obj += 32;        /* objet correspondant */
                      continue;
                  }
                do
                  {
                      if ((mword & mask) == 0) /* objet non atteint */
                        {		/* a inclure dans la free liste */
			    tail[0] = DISGUISE(obj);
			    tail = obj;
                        }
                      obj += 2 * OneWord; /* objet suivant */
                      mask <<= 2;       /* bit suivant */
                  }
                while(mask != 0);       /* mot de marquage fini */

                bm += OneWord;		/* mot de marquage suivant */
		mword = bm[0];
                mask = 1;               /* premier bit */
                                        /* ici, obj a avance' de 32 mots */
            }
           while(obj < top);
	  tail[0] = DISGUISE(NULL);	/* finit liste libre */
          break;
                 
     case SMALLSIZE_MODE:               /* taille dans la region */
     case W1_MODE:                      /* provisoire */
     case W3_MODE:                      /* idem */
     case W4_MODE:                      /* idem */
     case W8_MODE:                      /* idem */
       
       {                                /* taille en mots */
           nwords_t sz = RegionSize(pt.nregion);
           while(1)
             {
                 if((mword & mask) == 0) /* objet non atteint */
                   {
					/* inclure dans la free liste */
		       tail[0] = DISGUISE(obj);
		       tail = obj;
                   }
                 obj += sz;             /* objet suivant */
                 if (obj >= top)        /* arret */
                      break;
                 mword = MWord(obj, bm); /* sc */
                 mask = 1 << MShift(obj);
             }
	   tail[0] = DISGUISE(NULL);	/* finir liste libre */
           break;
       }
       
              
     case HEADER_MODE:                  /* tailles dans les objets */
       {
           nwords_t sz = CINT(obj[0]);  /* taille du premier objet */
           while(1)
             {
                 if((mword & mask) == 0) /* objet non-atteint */
                   {                   
				        /* inclure dans free-list: le lien */
		       tail[1] = DISGUISE(obj);
		       tail = obj;
                   }
                 obj += sz;             /* suivant */
                 if (obj >= top)
                      break;
                 sz = CINT(obj[0]);	/* taille suivante */
                 mword = MWord(obj, bm); /* mot marquage */
                 mask = 1 << MShift(obj); /* et bit */
             }
       }
	  tail[1] = DISGUISE(NULL);	/* finir liste libre */
          break;

     case CSTRING_MODE:                 /* chaines terminees par '\0' */
       {
           nwords_t sz = strlen((char*)obj)/sizeof(val_t)+1;
           while(1)          /* taille du premier objet */
             {
                 if((mword & mask) == 0) /* objet non-atteint */
                   {                   
				        /* inclure dans free-list: le lien */
		       tail[0] = DISGUISE(obj);
		       tail = obj;
                   }
                 obj += sz;             /* suivant */
                 if (obj >= top)
                      break;
                 sz = strlen((char*)obj)/sizeof(val_t)+1;/* taille suivante */
                 mword = MWord(obj, bm); /* mot marquage */
                 mask = 1 << MShift(obj); /* et bit */
             }
       }
	  tail[0] = DISGUISE(NULL);	/* finir liste libre */
          break;

     case STRING_MODE:    /* chaines, taille en octets dans premier mot */
       {                  /* on compte un octet nul en plus a la fin */
           nwords_t sz = CINT(obj[0])/sizeof(val_t)+2;
           while(1)          /* taille du premier objet */
             {
                 if((mword & mask) == 0) /* objet non-atteint */
                   {                   
				        /* inclure dans free-list: le lien */
		       tail[0] = DISGUISE(obj);
		       tail = obj;
                   }
                 obj += sz;             /* suivant */
                 if (obj >= top)
                      break;
                 sz = CINT(obj[0])/sizeof(val_t)+2;
		    /* taille suivante */
                 mword = MWord(obj, bm); /* mot marquage */
                 mask = 1 << MShift(obj); /* et bit */
             }
       }
	  tail[0] = DISGUISE(NULL);	/* finir liste libre */
          break;

     case FOLLOW_MODE:
     case BIG_MODE:
     case NO_MODE:
#ifdef DEBUG
     default:
#endif
          error("CleanBitMap : bad mode\n");
      }

    
    /* Ici, on a la liste des objets libres dans la page. Si cette */
    /* liste est non-vide, l'ajouter a la liste libre de la region. */
    
 {
     val_t free_list = UNDISGUISE(head[0]);

     if (free_list != NULL)
       {
	   region_t *preg = Region(nreg);

	   if (preg->freehead == NULL)
	     {				/* free liste de region vide */
		 preg->freehead = free_list;
		 preg->freetail = tail;
	     }
	   else				/* ajouter a la free liste de */
					/* region */
	     {
		 if (pt.mode == HEADER_MODE)
		      preg->freetail[1] = DISGUISE(free_list);
		 else
		      preg->freetail[0] = DISGUISE(free_list);
		 preg->freetail = tail;
	     }
       }
    return;
 }
}


/**********************************************************************
 *                            ScanStack
 **********************************************************************/

#include <setjmp.h>

static void ScanStack(void)
{
    jmp_buf env;
    setjmp(env);

    do_scanstack();

#ifdef MONITOR_GC
    if (_gc_monitor_others)
	 _report_fixedpages();
#endif    
}


static void do_scanstack(void)
{
    val_t  stack_top = (val_t)&stack_top;
    val_t  stack_bottom = usrstack;

#ifdef MONITOR_GC
    if (_gc_monitor_others)
	 _report_stack(stack_bottom-stack_top);
#endif

#ifdef STACKGROWSUP    
    if(stack_top < stack_bottom)
#else
    if(stack_top > stack_bottom)
#endif
      {
          error("Unusual stack growth direction : fix!\n");
      }
 {
     val_t  p;
#ifdef STACKGROWSUP    
     for(p = stack_top; p >= stack_bottom; p--)
#else
     for(p = stack_top; p <= stack_bottom; p++)
#endif
       {
           CheckStackReference((void *)p);
       }

/*     if(HalfWordStackAlignment)
       {
           stack_top =  (val_t)((nbytes_t)stack_top + 2);
           for(p = stack_top ; p <= stack_bottom; p++)
             {
                 CheckStackReference((void *)p);
             }
       } */
 }
    return;
}


static void CheckStackReference(val_t loc)
{
    val_t ref = *loc;
   
    if (ref >= EndMem)           /* segmentation fault ? */
         return;

 {
     npage_t pn = PageNum(ref);
     pagetag_t pt;

     if ( pn < DescOffset || ref >= PageTop(pn) )
                /* also valid for FOLLOW pages (see function NormalizePages) */
          return;

     pt = PageTag(pn);                  /* now let's identify the object */
     
 SELECTION:
     switch(pt.zone)
       {
      case HIDDEN_ZONE:                 /* Not an object reference */
      case FREE_ZONE:                   /* Not an object reference */
      case GC_ZONE:                     /* Not an object reference */
      case TO_BIG_ZONE:                 /* Big object already promoted */
           /* and pushed */ 
      case TO_BIG_OPAQUE_ZONE:          /* Big object already promoted */
	   return;

      case TO_COPY_ZONE:                /* No copied object yet ! */
      case TO_COPY_OPAQUE_ZONE:         /* No copied object yet ! */
           error("scanstack : bad zone\n");
           return;                      /* pfj */
     
      case FROM_BIG_ZONE:
           if (pt.mode == FOLLOW_MODE)
             {
                 pn = BlocHead(pn);
                 pt = PageTag(pn);
                 goto SELECTION;        /* The head may have been marked */
             }
           pt.zone = TO_BIG_ZONE;       /* do promotion and push */
           PageTag(pn) = pt;
#ifdef _GC_MONITOR_LOCALITY
	   if (_gc_monitor_locality)
		_record_obj_locality(PageAddress(pn));
#endif
           PushUntraced(PageAddress(pn));
           return;
   
      case FROM_BIG_OPAQUE_ZONE:
           if (pt.mode == FOLLOW_MODE)
             {
                 pn = BlocHead(pn);
                 pt = PageTag(pn);
                 goto SELECTION;        /* The head may have been marked */
             }
           pt.zone = TO_BIG_OPAQUE_ZONE; /* do promotion */
           PageTag(pn) = pt;
           return;
     
      case FROM_FIX_OPAQUE_ZONE:              /* first mark the page */
           pt.zone = TO_FIX_OPAQUE_ZONE;
           PageTag(pn) = pt;
           PageBitMap(pn) = AllocateBitMap(); /* then get a bitmap */
	   goto MARKOBJ;                      /* 31/3/93 */

      case FROM_COPY_OPAQUE_ZONE:             /* first mark the page */
               
           pt.zone = TO_FIXED_OPAQUE_ZONE;
           PageTag(pn) = pt;
           PageBitMap(pn) = AllocateBitMap(); /* then get a bitmap */
	   goto MARKOBJ;

      case FROM_FIX_ZONE:               /* first mark the page */
           pt.zone = TO_FIX_ZONE;
           PageTag(pn) = pt;
           goto HAVEBITMAP;             /* then get a bitmap */
     
      case FROM_COPY_ZONE:              /* first mark the page */
           pt.zone = TO_FIXED_ZONE;
           PageTag(pn) = pt;            /* (fall through) */

      HAVEBITMAP:                       /* have a bitmap */
 
           PageBitMap(pn) = AllocateBitMap(); /* and fall through */

      case TO_FIX_ZONE:                 /* must identify the object */
      case TO_FIXED_ZONE:               /* idem */

      case TO_FIX_OPAQUE_ZONE:          /* Page already marked */
      case TO_FIXED_OPAQUE_ZONE:        /* Page already marked */

      MARKOBJ:
        {
            val_t obj;
        
            switch (pt.mode)            /* depends on mode */
              {
             case W2_MODE:
                  obj = (val_t)((ulint)ref & ~W2_MASK);
                  break;
             case SMALLSIZE_MODE:
             case W1_MODE:              /* provisoire */
             case W3_MODE:              /* idem */
             case W4_MODE:              /* idem */
             case W8_MODE:              /* idem */
	       {
		   ulint add = (ulint)ref;
		   ulint start = (ulint)PageStart(ref);
		   nbytes_t offset = add - start;
		   nbytes_t size = RegionSize(pt.nregion) * WordSize;
		   ulint trueadd = add - (offset % size);

		   obj = (val_t)trueadd;
	       }
		  break;

             case HEADER_MODE:          /* wish headers are OK */
               {
                   val_t curr_obj = PageStart(ref);
                   val_t prev_obj;
                   nwords_t step;
                   do
                     {
                         prev_obj = curr_obj;
                         step = CINT(curr_obj[0]);
                     /* step > 0 */
                         curr_obj = curr_obj + step;
                     } 
                   while(curr_obj <= ref);
                         
                   obj = prev_obj;
                   break;
               }
	     case STRING_MODE:
               {
                   val_t curr_obj = PageStart(ref);
                   val_t prev_obj;
                   nwords_t step;
                   do
                     {
                         prev_obj = curr_obj;
                         step = CINT(curr_obj[0])/sizeof(val_t)+2;
                     /* step contient la longueur vraie de la chaine */
                         curr_obj = curr_obj + step;
                     } 
                   while(curr_obj <= ref);
                         
                   obj = prev_obj;
                   break;
               }
             case CSTRING_MODE:
             case BIG_MODE:
             case FOLLOW_MODE:
#ifdef DEBUG
             default:
#endif
                  error("scanstack : bad mode\n");
                  return;               /* pfj */
		}

         {                              /* now object is identified */
             bitmap_t bm = PageBitMap(pn);
             ulint *mwordp = &(MWord(obj, bm));
             ulint mword = *mwordp;
             ulint mask = 1 << MShift(obj);
                   
             if (mword & mask)     /* already marked */
                  return;
             mword |= mask;             /* mark and push amb. root */
             *mwordp = mword;
#ifdef _GC_MONITOR_LOCALITY
	     if (_gc_monitor_locality)
		  _record_obj_locality(obj);
#endif
             switch(pt.zone)
	       {
	       case FROM_FIX_ZONE:
	       case FROM_COPY_ZONE:
	       case TO_FIX_ZONE:
	       case TO_FIXED_ZONE:
		   PushAmbRoot(obj);
	       }
         }
        }
       }
 }
}  


/**********************************************************************
 *                             Scavenge
 **********************************************************************
 * scavenge() : fonction de tracage. Choisit une location a tracer, lui 
 * applique la fonction de tracage ScavengeLoc. Pour implementer les
 * niveaux de priorite de l'algo de tracage, scavengeloc rend le
 * numero de la region s'il y a recopie,  ou l'un des codes definis
 * ci-dessous,  avec la signification suivante: CONTINUE_CONT: pas
 * d'ajout de location non-tracee; COPYZONE_CONT: ajout de locations
 * non-tracees dans les pages pleines (promotion d'un gros objet);
 * UNTRACED_CONT: ajout d'un objet isole a explorer. Aucune region de
 * recopie n'a pour numero 0, 1 ou 2. 
 **********************************************************************/

#define UNTRACED_CONT   2
#define COPYZONE_CONT   1
#define CONTINUE_CONT   0



static void Scavenge(void)
{
    nregion_t nreg;
    val_t u_scp = NULL;                /* scp in current untraced object */
    val_t u_end = NULL;                /* end of current untraced object */
    val_t a_scp = NULL;                /* scp in current amb. root object */
    val_t a_end = NULL;                /* end of current amb. root object */
    val_t c_scp = NULL;                /* scp in current bloc */
    val_t c_end = NULL;                /* end of current bloc */
    int       rootindex = 0;            /* examine first global root first */
    goto UNTRACEDSTACK;

 COPYPAGES:                             /* tracage,  niveau 1 */
    while(1)
      {
          val_t  scp = RegionSCP(nreg); /* prochaine location a tracer */
          nregion_t new_reg;            /* compte-rendu de ScavengeLoc */

          if (scp >= RegionAP(nreg))   /* rien a tracer dans cette region */
            {
                nreg =  NextRegion(nreg); /* essayer la region de recopie */
                                        /* suivante */
                if(nreg == (nregion_t)0) /* plus de regions de recopie */
                     break;             /* : epuisement niveau 1 */
                else
                     continue;          /* continuer avec next reg. */
            }
          else                          /* scp < ap : OK */
            {
                RegionSCP(nreg) = scp + OneWord; /* avancer dans la page */
                new_reg = ScavengeLoc(scp); /* tracer scp */
                if (new_reg >= nreg)    /* region plus prioritaire? */
                  {
                      nreg = new_reg;
                      continue;         /* continuer dans. new_reg */
                  }
                else
                     continue;          /* (meme region) */
            }                           /* (cf: note) */
      }

 UNTRACEDSTACK:
    while(1)
      {
          while(u_scp != u_end)
            {
                nreg = ScavengeLoc(u_scp);
                u_scp += OneWord;
                if (nreg > UNTRACED_CONT)
                     goto COPYPAGES;
                else 
                     continue;
            }
          u_scp = PopUntraced();
          if (u_scp == NULL)
            {
                u_end = NULL;
                break;                  /* return-from untraced_stack   */
            }   
          else                          /* else compute u_end */
            {
                pagetag_t pt = PageTag(PageNum(u_scp));
                switch (pt.mode)
                  {
                 case W2_MODE:
                      u_end = u_scp + (2 * OneWord);
                      break;
                 case SMALLSIZE_MODE:
                 case W1_MODE:          /* provisoire */
                 case W3_MODE:          /* idem */
                 case W4_MODE:          /* idem */
                 case W8_MODE:          /* idem */
                      u_end = u_scp + RegionSize(pt.nregion);
                      break;
                 case FOLLOW_MODE:      /* infixe */
                      error("scavenge error\n");
                 case BIG_MODE:
                   {
                       npage_t pn = PageNum(u_scp);
                       npage_t bsz = BlocSize(pn);
                       npage_t last_page = pn + bsz -1;
                       u_end = PageTop(last_page);
                   }
                      break;
                 case HEADER_MODE:
                      u_end = u_scp + CINT(u_scp[0]); 
                      break;
#ifdef DEBUG
		 default:
		      error("scavenge : mauvais mode\n");
		      break;		/* pfj */
#endif
                  }
            }
          continue;                     /* pfj */
      }             

 COPYZONE:
    while(1)
      {
          while(c_scp != c_end)         /* bloc courant */
            {
                nreg = ScavengeLoc(c_scp);
                c_scp += OneWord;
                if (nreg > UNTRACED_CONT)
                     goto COPYPAGES;
                else if (nreg == UNTRACED_CONT)
                     goto UNTRACEDSTACK;
                else 
                     continue;
            }
                                        /* choisir un nouveau bloc */
       {
           npage_t next_bloc = to_head;
           if (next_bloc == NullPage)
                break;                  /* return_from copyzone */
           to_head = NextBloc(next_bloc);
              
           c_scp = PageSCP(next_bloc);
           c_end = PageTop(next_bloc);
       }
          continue;                     /* pfj */
      }
        
    /* AMBROOTS: */
    
    while(1)
      {
          while(a_scp != a_end)
            {
                nreg = ScavengeLoc(a_scp);
                a_scp += OneWord;
                if (nreg > UNTRACED_CONT)
                     goto COPYPAGES;
                else if (nreg == UNTRACED_CONT)
                     goto UNTRACEDSTACK;
                else if (nreg == COPYZONE_CONT)
                     goto COPYZONE;
                else                    /* nreg == CONTINUE_CONT */
                     continue;          /* pfj */
            }
          a_scp = PopAmbRoot();
          if (a_scp == NULL)
            {
                a_end = NULL;
                break;                  /* return-from amb_roots */
            }   
          else                          /* else compute a_end */
            {
                pagetag_t pt = PageTag(PageNum(a_scp));
                switch (pt.mode)
                  {
                 case W2_MODE:
                      a_end = a_scp + (2 * OneWord);
                      break;
                 case SMALLSIZE_MODE:
                 case W1_MODE:          /* provisoire */
                 case W3_MODE:          /* idem */
                 case W4_MODE:          /* idem */
                 case W8_MODE:          /* idem */
                      a_end = a_scp + RegionSize(pt.nregion);
                      break;
                 case FOLLOW_MODE:      /* infixe */
                      error("scavenge error\n");
                 case BIG_MODE:
                   {
                       npage_t pn = PageNum(a_scp);
                       npage_t bsz = BlocSize(pn);
                       npage_t last_page = pn + bsz -1;
                       a_end = PageTop(last_page);
                   }
                      break;
                 case HEADER_MODE:
                      a_end = a_scp + CINT(a_scp[0]); 
		      a_scp++;
                      break;
                  }
            }
          continue;                     /* pfj */
      }


    /* TRUEROOTS: */

    while(rootindex != NRoots)
      {
          void *loc = PRoots[rootindex];

          rootindex +=1;
          nreg = ScavengeLoc(loc);
             
          if (nreg > UNTRACED_CONT)
               goto COPYPAGES;
          else if (nreg == UNTRACED_CONT)
               goto UNTRACEDSTACK;
          else if (nreg == COPYZONE_CONT)
               goto COPYZONE;
          else                          /* nreg == CONTINUE_CONT */
               continue;                /* pfj */
      }
    return;                             /* pfj */
}

/**********************************************************************
 *                           ScavengeLoc
 **********************************************************************
 * ScavengeLoc(loc) : la fonction de tracage proprement dite. Choisit
 * une action en fonction de la zone ou pointe loc...
 **********************************************************************/

#define TO_HELL         666

extern void dummy(val_t); 

/* ScavengeLoc(loc) : loc est l'adresse d'un mot a tracer (un champ 
 * d'objet par exemple.
 */

static ulint ScavengeLoc(val_t loc)
{ 
    void *v = *loc;                     /* valeur du champ d'objet */
    
    if (!(Ref_P(v)))                    /* imm ou ref. */
         return CONTINUE_CONT;
 {
     npage_t pn = PageNum(v);
     if ( pn < DescOffset )             /* rajoute ad hoc pour constants[] */
          return CONTINUE_CONT;
   {
     pagetag_t pt = PageTag(pn);
     
     /*  SELECTION:  */
     switch (pt.zone)
       {
      case GC_ZONE:
           error("ScavengeLoc : GC_ZONE\n"); 
      case FREE_ZONE:
           error("ScavengeLoc : FREE_ZONE\n"); 
      case HIDDEN_ZONE:
           error("ScavengeLoc : HIDDEN_ZONE\n");
      case TO_COPY_ZONE:
           error("ScavengeLoc : TO_COPY_ZONE\n"); 
      case TO_COPY_OPAQUE_ZONE:
           error("ScavengeLoc : TO_COPY_OPAQUE_ZONE\n"); 

           return TO_HELL;              /* pfj */
        
      case TO_BIG_ZONE:
      case TO_BIG_OPAQUE_ZONE:
           return CONTINUE_CONT;

      case FROM_BIG_ZONE:               /* les infixes ne sont pas permis */
           pt.zone = TO_BIG_ZONE;
           PageTag(pn) = pt;
           PageSCP(pn) = PageAddress(pn);
#ifdef _GC_MONITOR_LOCALITY
	   if (_gc_monitor_locality)
		_record_obj_locality(CREF(v));
#endif
           COPYZONE_PUT(pn);
           return COPYZONE_CONT;

      case FROM_BIG_OPAQUE_ZONE:    
           pt.zone = TO_BIG_OPAQUE_ZONE;
           PageTag(pn) = pt;
           return CONTINUE_CONT;        
      
      case TO_FIX_OPAQUE_ZONE:
      case TO_FIXED_OPAQUE_ZONE:
        {
            bitmap_t bm = PageBitMap(pn);
            ulint   *mwordp = &MWord(v,bm);
            ulint    mword = *mwordp;
            ulint    mask = 1 << MShift(v);
              
            mword |= mask;
            *mwordp = mword;

            return CONTINUE_CONT;
        }
      case TO_FIX_ZONE:
      case TO_FIXED_ZONE:
        {
            bitmap_t bm = PageBitMap(pn);
            ulint   *mwordp = &MWord(v,bm);
            ulint    mword = *mwordp;
            ulint    mask = 1 << MShift(v);
              
            if (mword & mask)
                 return CONTINUE_CONT;
               
            mword |= mask;
            *mwordp = mword;

#ifdef _GC_MONITOR_LOCALITY
	    if (_gc_monitor_locality)
		 _record_obj_locality(CREF(v));
#endif              
            PushUntraced(CREF(v));
            return UNTRACED_CONT;
        }
      case FROM_FIX_ZONE:               /* FIRST HIT IN THE PAGE */
           pt.zone = TO_FIX_ZONE;
           PageTag(pn) = pt;
        {
            bitmap_t bm = AllocateBitMap();
            PageBitMap(pn) = bm;
            MWord(v,bm) = 1 << MShift(v);
        }
#ifdef _GC_MONITOR_LOCALITY
	   if (_gc_monitor_locality)
		_record_obj_locality(CREF(v));
#endif              
           PushUntraced(CREF(v));
           return UNTRACED_CONT;
           
      case FROM_FIX_OPAQUE_ZONE:
           pt.zone = TO_FIX_OPAQUE_ZONE;
           PageTag(pn) = pt;
        {
            bitmap_t bm = AllocateBitMap();
            PageBitMap(pn) = bm;
            MWord(v,bm) = 1 << MShift(v);
        }
           return CONTINUE_CONT;
           
      case FROM_COPY_ZONE:
        {
            val_t old_obj = CREF(v);   /* adresse objet pointe' */
            ulint fp = (ulint)(old_obj[0]); /* premier mot de cet objet */
              
            
            if (Forward_P(fp))		/* deja recopie' ? */
              {
                  *loc = FP2KREF(fp);
                  return CONTINUE_CONT;
              }				/* sinon promotion (recopie) */

#ifdef _GC_MONITOR_LOCALITY
	    if (_gc_monitor_locality)
		 _record_obj_locality(CREF(v));
#endif              
         {                              /* l'objet doit e^tre copie' */
             nregion_t nreg = pt.nregion;
             val_t new_obj;
             nwords_t sz;
                
             switch (pt.mode)
               {
              case W2_MODE:             /* objets de 2 mots, any */
                {
                    M_CopyAllocateSmall(2 * OneWord, nreg, new_obj);
                    new_obj[0] = old_obj[0];
                    new_obj[1] = old_obj[1];
                    old_obj[0] = CREF2FP(new_obj); /* fp en tete  */
                    *loc = KREF(new_obj); /* corrige la reference */
                }
                   return nreg;

              case SMALLSIZE_MODE:
              case W1_MODE:             /* provisoire */
              case W3_MODE:             /* idem */
              case W4_MODE:             /* idem */
              case W8_MODE:             /* idem */
                   sz = RegionSize(pt.nregion);
                   break;
              case STRING_MODE:
                   sz = CINT(old_obj[0])/sizeof(val_t)+2;
                   break;
              case HEADER_MODE:
                   sz = CINT(old_obj[0]); 
                   break;
              case CSTRING_MODE:
                   error("ScavengeLoc : bad mode CSTRING\n");
                   return TO_HELL;      /* pfj */
              case FOLLOW_MODE:
                   error("ScavengeLoc : bad mode FOLLOW\n");
                   return TO_HELL;      /* pfj */
              case BIG_MODE:
                   error("ScavengeLoc : bad mode BIG\n");
                   return TO_HELL;      /* pfj */
#ifdef DEBUG
              default:
                   error("ScavengeLoc : bad mode1\n");
                   return TO_HELL;      /* pfj */
#endif
               }
             M_CopyAllocateSmall(sz, nreg, new_obj);
             CopyWords(old_obj, new_obj, sz);

             old_obj[0] = CREF2FP(new_obj);
             *loc = KREF(new_obj);
             return nreg;
         }
        }
      case FROM_COPY_OPAQUE_ZONE:
        {
            val_t old_obj = CREF(v);
            ulint fp = (ulint)old_obj[0];
              
            if (Forward_P(fp))
              {
                  *loc = FP2KREF(fp);
                  return CONTINUE_CONT;
              }
              
         {                              /* l'objet doit e^tre copie' */
             nregion_t nreg = pt.nregion;
             val_t new_obj;
             nwords_t sz;
                
             switch (pt.mode)
               {
              case W2_MODE:             /* objets de 2 mots, any */
                {
                    M_CopyAllocateSmall(2 * OneWord, nreg, new_obj);
                    new_obj[0] = old_obj[0];
                    new_obj[1] = old_obj[1];
                    old_obj[0] = CREF2FP(new_obj);

                    *loc = KREF(new_obj);
                    return CONTINUE_CONT;
                }
              case SMALLSIZE_MODE:
              case W1_MODE:             /* provisoire */
              case W3_MODE:             /* idem */
              case W4_MODE:             /* idem */
              case W8_MODE:             /* idem */
                   sz = RegionSize(pt.nregion);
                   break;
              case STRING_MODE:
                   sz = CINT(old_obj[0])/sizeof(val_t)+2;
                   break;
              case HEADER_MODE:
                   sz = CINT(old_obj[0]);
                   break;
              case CSTRING_MODE:
                   error("ScavengeLoc : bad mode CSTRING\n");
                   return TO_HELL;      /* pfj */
              case FOLLOW_MODE:
                   error("ScavengeLoc : bad mode FOLLOW\n");
                   return TO_HELL;      /* pfj */
              case BIG_MODE:
                   error("ScavengeLoc : bad mode BIG\n");
                   return TO_HELL;      /* pfj */
#ifdef DEBUG
              default:
                   error("ScavengeLoc : bad mode2\n");
                   return TO_HELL;      /* pfj */
#endif
               }
             M_CopyAllocateSmall(sz, nreg, new_obj);
             CopyWords(old_obj, new_obj, sz);
             old_obj[0] = CREF2FP(new_obj);
             *loc = KREF(new_obj);
             return CONTINUE_CONT;
         }
        }
#ifdef DEBUG
      default:
           error("ScavengeLoc : error\n");
           return TO_HELL;
#endif
       }
   }
 }
    return TO_HELL;
}



/**********************************************************************
 *                            Stackregs
 **********************************************************************
 * Piles d'exploration utilisees par le GC. Ces piles sont implantees
 * dans des pages prises dans le POOL, et chainees. Chaque pile occupe
 * une region (statique). Les operations sont :
 * 
 * PushStatic(val, nreg)
 * PopStatic(nreg)
 **********************************************************************/


#define MPush_Static(val,nreg) \
{ \
        val_t ap = StaticRegionAP(nreg); \
        val_t new_ap = ap + OneWord; \
        \
        if (new_ap > StaticRegionLimit(nreg)) \
          { \
              Push_Static(val, nreg); \
          } \
        else \
          { \
              StaticRegionAP(nreg) = new_ap; \
              ap[0] = val; \
          } \
} \


void Push_Static(val_t val, nregion_t nreg)
{
    val_t ap = StaticRegionAP(nreg);
    val_t new_ap = ap + OneWord;
        
    if (new_ap > StaticRegionLimit(nreg))
      {
          npage_t new_page = PoolGetBloc(1);

       {
           npage_t old_page = (ap == NULL) ? 0 : PageNum(ap - OneWord);

           PreviousPage(new_page) = old_page;
       }
          PageTag(new_page) = RegionTag(nreg);
          ap = PageAddress(new_page);
          new_ap = ap + OneWord;
                        
          StaticPopBreak(nreg) = ap;
          StaticRegionLimit(nreg) = ap + OnePage;
      }
    StaticRegionAP(nreg) = new_ap;
    ap[0] = val;
         
    return;
}

/* 
 * Pop
 */

#define MPop_Static(nreg, res) \
{ \
        val_t ap = StaticRegionAP(nreg); \
        \
        if (ap == StaticPopBreak(nreg)) \
          { \
             res = Pop_Static(nreg); \
          } \
        else \
          { \
                ap -= OneWord; \
                StaticRegionAP(nreg) = ap; \
                res = ap[0]; \
          } \
} \


val_t Pop_Static(nregion_t nreg)
{
        val_t ap = StaticRegionAP(nreg);
        
        if (ap == StaticPopBreak(nreg))
          {
                npage_t curr_page, prev_page;

                if (ap == NULL)
                     return NULL;

                curr_page = PageNum(ap); /* curr_page != 0 */
                prev_page = PreviousPage(curr_page);
                                        /* release page */
                PageTag(curr_page) = StaticRegionTag(STD_FREE_REG); 
                PoolPutFirst(curr_page);
                
                if (prev_page == NullPage)
                  {
                      StaticRegionAP(nreg) = NULL;
                      StaticPopBreak(nreg) = NULL;
                      StaticRegionLimit(nreg) = NULL;
                      return NULL;
                  }
            {
               val_t new_popbreak = PageAddress(prev_page);
               
               StaticPopBreak(nreg) = new_popbreak;
               ap = new_popbreak + OnePage;
               StaticRegionLimit(nreg) = ap;
            }
          }
        ap -= OneWord;
        StaticRegionAP(nreg) = ap;
        return ap[0];
}

/**********************************************************************
 * Piles d'exploration
 **********************************************************************/

void PushUntraced(val_t loc)
{
    MPush_Static(loc, UNTRACED_REG);
    return;
}

void PushAmbRoot(val_t loc)
{
    Push_Static(loc, AMBROOTS_REG);
    return;
}

val_t PopUntraced(void)
{
        void *res;
        MPop_Static(UNTRACED_REG, res);
        return res;
}

val_t PopAmbRoot(void)
{
        void *res;
        MPop_Static(AMBROOTS_REG, res);
        return res;
}

/**********************************************************************
 *                            GCAllocate
 **********************************************************************
 * L'allocation d'objets par le gc n'utilise pas les fonctions
 * definies dans le module alloc. Les quelques raisons a ceci sont :
 * 1- alloc est un client de gc, et non le contraire; 2- l'allocation
 * pour recopie ne s'inquiete pas pas des couts des pages, et alloue
 * dans la zone de recopie; 2- l'allocation d'objets temporaires
 * propres au gc ne s'occupe pas non plus du cout des pages...
 **********************************************************************/


/* GCAllocateSmall : allocation d'objets (opaques) propres au GC. 
 * La recopie des objets ``utilisateur'' utilise la fonction
 * CopyAllocateSmall. Pas de gestion du compte de la me'moire libre.
 */

static val_t GCAllocateSmall(nwords_t sz, nregion_t nreg)
{
    val_t ap, new_ap;

    ap = RegionAP(nreg);

    if((new_ap = ap + sz) > RegionLimit(nreg))
      {
          npage_t new_page;

          if (ap != NULL)           /* release old allocation page */
            {
                npage_t old_page = RegionAPG(nreg);
                PageTop(old_page) = ap;
            }
          
          new_page = PoolGetBloc(1);    /* this _should_ succeed */
#ifdef DEBUG
          if (new_page == NullPage)
            {
                error("Memory exhaustion during GC!!\n");
            }
#endif          
          PageTag(new_page) = RegionTag(nreg); /* incorporate page in region */
          RegionAPG(nreg) = new_page;   /* make it allocation page */

          ap = PageAddress(new_page);

          RegionSCP(nreg) = ap;
          PageTop(new_page) = ap + OnePage; /* cf: note */
          RegionAP(nreg) = ap + sz;
          RegionLimit(nreg) = ap + OnePage;
      }
    else
      {
          RegionAP(nreg) = new_ap;
      }

    return ap;
}

/* note: ceci permet de maintenir l'illusion que la page de recopie 
 * est pleine. Lors du flip (epilogue du gc),  cette page payera une
 * residu variable maximum, tout en devenant une page d'allocation.
 * Lorsqu'elle sera regulierement relachee,  le residu effectif sera
 * decompte selon le procede ordinaire. Une autre solution
 * compliquerait inutilement le flip. Le surcout (sur une operation de
 * page!) n'a aucune importance. 
 */

/**********************************************************************
 *                         Copie des objets
 **********************************************************************
 * CopyAllocateSmall : recopie des petits objets. Cette fonction
 * effectue si necessaire le changement de page d'allocation,  sans
 * gestion du compte de la memore libre,  mais avec incorporation de
 * la page pleine au chainage de la zone TO, si les objets contenus ne
 * sont pas opaques.
 **********************************************************************/

static val_t CopyAllocateSmall(nwords_t sz, nregion_t nreg)
{
    val_t ap = RegionAP(nreg);
    val_t new_ap = ap + sz;
    val_t limit = RegionLimit(nreg);

    if (new_ap > limit)                 /* must change AP */
      {
          npage_t new_page;
          
          if (ap != NULL)           /* release old allocation page */
            {
                npage_t old_page = RegionAPG(nreg);
            
                PageTop(old_page) = ap;
                if (RegionTag(nreg).zone == FROM_COPY_ZONE)
                  {
                      COPYZONE_PUT(old_page);
                      PageSCP(old_page) = RegionSCP(nreg);
                  }
            }
          new_page = PoolGetBloc(1);    
#ifdef DEBUG
          if (new_page ==  NullPage)
               error("CopyAllocateSmall : no page left!\n");
          if (PageTag(new_page).nregion != STD_FREE_REG)
               error("Reallocation d'une page?\n");
#endif
					/* incorporate page */
          PageTag(new_page) = RegionCopyTag(nreg);
          RegionAPG(nreg) = new_page;   /* and make it alloc. page */

          ap = PageAddress(new_page); /* start allocating in page */
          RegionSCP(nreg) = ap;         /* start local scav. also */
          RegionLimit(nreg) = ap + OnePage;
          PageTop(new_page) = ap + OnePage; /* BUG 1 de Vincent */
	  RegionAP(nreg) = ap + sz;
      }
    
    else
         RegionAP(nreg) = new_ap;       /* simple case: just allocate */

    return ap;
}


/**********************************************************************
 * Bit-maps
 **********************************************************************/


/* .Section FreeMem */


nwords_t FreeMem;			/* quantite de memoire disponible */
nwords_t safepages;			/* pages d'arrondi */
int      NBadRegions;			/* nb regions (utilisateur) a */
					/* residu variable */  


                       /*----------------------
                       | Declarations locales |
                       ----------------------*/

extern void     init_roots(void);
extern void     init_regions(void);
extern void     init_cost(void);



/**********************************************************************
 *                       Initialisation du GC
 **********************************************************************
 * L'initialisation des variables explorables est realisee "par acquis
 * de conscience",  et pour couper court a toute tentative de
 * l'utilisateur d'initialiser une variable de maniere statique.
 *
 * Les regions d'objets copiables sont chainees,  de maniere a
 * realiser simplement le changement de region au niveau 1 de
 * l'algorithme 'scavenge'.
 **********************************************************************/

static void InitRoots(void);
static void LinkCopyRegions(void);

extern void InitGC(void)
{
    InitRoots();
    LinkCopyRegions();
    return;
}

/* InitRoots: Initialisation des variables explorables. Les racines
 * globales sont communiquees au gc par le biais d'un tableau
 * d'adresses GcVars. 
 */

static void InitRoots(void)
{
    NRoots = NGcVars;
    PRoots = (val_t *)GcVars;

 {
     int i;

     for(i = 0; i < NRoots; i++)
          *((PRoots)[i]) = NULL;
 }
    return;
}

/* LinkCopyRegions: chainage des regions d'objets copiables: voir
 * la fonction de conduite de l'exploration scavenge.
 */

static void LinkCopyRegions(void)
{
    nregion_t next = 0;
    nregion_t curr;
       
    for(curr = 0; curr < NRegions; curr++)
      {
          pagetag_t pt = RegionTag(curr);

          switch (pt.zone)
            {
           case FROM_COPY_ZONE:
                NextRegion(curr) = next;
                next = curr;
                pt.zone =  TO_COPY_ZONE;
                RegionCopyTag(curr) = pt;
                break;
           case FROM_COPY_OPAQUE_ZONE:
                pt.zone =  TO_COPY_OPAQUE_ZONE;
                RegionCopyTag(curr) = pt;
                break;
           default:
                break;
            }
      }
    return;
}



/**********************************************************************
 * Debug...
 **********************************************************************/

#ifdef DEBUG
extern void dummy(val_t loc)
{
    return;
}


extern char * ismarked(val_t obj)
{
    npage_t pn = PageNum(obj);
    bitmap_t bm = PageBitMap(pn);

    if(bm == NULL)
        return "Pas de bitmap"; 
 {
     ulint mword = MWord(obj, bm);
     ulint mask = 1 << MShift(obj);

     if (mword & mask)
          return "objet marque'";
     else
          return "objet pas marque'";
 }
}

#endif

