/* $Header: /Nfs/radyr/usr11/rc136/Projet/GC/RCS/alloc.c,v 1.1 1992/09/03 21:15:02 rc136 Exp rc136 $ */
/* Auteur: Vincent Delacour (delacour@poly.polytechnique.fr) */

#include "../Include/parametres.h"
#include "../Include/public.h"
#include "private.h"
#include "gc.h"
#include "../Include/alloc.h"


/**********************************************************************
 * Allocation
 **********************************************************************
 * Fonctions exportees:
 *
 *      Allocate(sz, nreg)
 *      AllocateSmall(sz, nreg)
 *      AllocateBig(sz, nreg)
 *
 * Macros:
 *
 *      M_AllocateSmall(sz, nreg)
 *      F_AllocateSmall(sz, nreg)
 **********************************************************************
 * ChangePage + GetFreeMemory
 **********************************************************************
 * Changement de page d'allocation: tient le compte de la memoire
 * libre. En cas de saturation,  appelle GetFreeMemory,  qui
 * implemente la strategie de gestion de la memoire.
 **********************************************************************/


/**********************************************************************
 * Variables : FreeMem, RoundUpPages,  PagesSinceLastGC
 *
 * FreeMem est le compte de la memoire libre. RoundUpPages est le
 * nombre de pages d'arrondi dues aux regions inhomogenes d'objets
 * copiables. La variable PagesSinceLastGC controle la politique
 * d'appel au gc (par paliers). 
 **********************************************************************/

nwords_t FreeMem;                       /* compte de la memoire libre */
npage_t  RoundUpPages;			/* pages d'arrondi */
npage_t  PagesSinceLastGC;		/*  nb pages allouees avant gc */

/**********************************************************************
 * Declarations locales
 **********************************************************************/

static npage_t  ChangePage(nregion_t nreg);
static void     GetFreeMemory(npage_t needed);

static npage_t  ComputeExpansionWish(npage_t npg_needed);

static void     GetFreeBloc(npage_t needed);
static npage_t  GetBlocNoBloc(npage_t needed);
static npage_t  GetBlocNoMemory(npage_t needed);
static void     ReleasePage(npage_t pn);

/**********************************************************************/


/* Allocate: fonction generale d'allocation. 
 */

extern val_t Allocate(nwords_t sz, nregion_t nreg)
{
#ifdef DEBUG
    if (sz < OneWord)				/*  */
	 error("Allocate : mauvaise taille\n");
#endif
    if(sz > SmallSizeLimit)
         return AllocateBig(sz, nreg);
    else
         return AllocateSmall(sz, nreg);
}


/* AllocateBig: Le fait que les gros objets ne sont pas deplaces est
 * "cable": ils coutent exactement le nombre de pages qu'ils occupent,
 * quels que soient les attributs de la region d'allocation.  
 */


extern val_t AllocateBig(nwords_t sz, nregion_t nreg)
{
    npage_t npg = Words2Pages(sz);             /* indivisible pages */
    nwords_t words_cost = npg * OnePage;
    nwords_t mem = FreeMem;             

    npage_t start_pg;

    if(words_cost > mem)                
      {                                 /* memory consumption */
          start_pg = GetBlocNoMemory(npg);
          mem =  FreeMem;               /* read FreeMem */
          FreeMem = mem - words_cost;   /* pay for memory */
      }
    else if ((start_pg = PoolGetBloc(npg)) == NoBloc) /* try in POOL */

      {                                 /* Fragmentation */
          start_pg = GetBlocNoBloc(npg); 
          mem = FreeMem;                /* read FreeMem */
          FreeMem = mem - words_cost;   /* pay for memory */
      }
    else
      {
          FreeMem = mem - words_cost;   /* pay for memory */
      }
                                        /* now got free bloc */
                                        /* describe it */
    PageTag(start_pg) = RegionTag_Big(nreg);
 {
     val_t  res = PageAddress(start_pg);
     BlocTOP(start_pg) = res + sz;
     return res;
 }
}


/* AllocateSmall: allocation des petits objets. Pour diverses raisons,
 * les mecanismes d'allocation 'utilisateur' ne sont pas exactement
 * ceux qui permettent dans le gc 1- de recopier les objets, 2-
 * d'allouer des objets temporaires propres au gc.  (Raisons : 1-
 * alloc est client du gc et non le contraire, 2- la strategie de
 * changement de page est differente dans les deux (trois) cas.  */



val_t AllocateSmall(nwords_t sz, nregion_t nreg)
{
    val_t ap, new_ap;

    ap =  RegionAP(nreg);
    if((new_ap = ap + sz) > RegionLimit(nreg))
      {
          npage_t new_page = ChangePage(nreg);

	  RegionAPG(nreg) = new_page;
          ap = PageAddress(new_page);
          RegionAP(nreg) = ap + sz;
          RegionLimit(nreg) = ap + OnePage;
      }
    else
      {
          RegionAP(nreg) = new_ap;
      }
    return ap;
}

val_t Futur_AllocateSmall(nwords_t sz, nregion_t nreg)
{
    val_t ap, new_ap;

    ap =  RegionAP(nreg);
    if((new_ap = ap + sz) > RegionLimit(nreg))
      {
	  ChangePage(nreg);		/* peut faire un gc */
	  ap = RegionAP(nreg);

	  if((new_ap = ap + sz) > RegionLimit(nreg))
	    {
		npage_t new_page = PoolGetBloc(1);
		npage_t old_page = RegionAPG(nreg);
		val_t new_page_start = PageAddress(new_page);

		PageTop(old_page) = ap;	/* relache page courante */
		RegionAPG(nreg) = new_page;
		RegionAP(nreg) = new_page_start + sz;

		return new_page_start;
	    }
	  else
	    {
		RegionAP(nreg) = new_ap;
		return ap;
	    }
      }
    else
      {
          RegionAP(nreg) = new_ap;
	  return ap;
      }
}



/**********************************************************************
 * ChangePage
 **********************************************************************
 * ChangePage(nreg): Changement de la page d'allocation de la region,
 * rend la nouvelle page. Tient le compte de la memoire libre
 * (FreeMem) et celui des pages d'objets allouees depuis le dernier
 * GC (PagesSinceLastGC).
 **********************************************************************
 * L'arithmetique est ici plus delicate qu'il n'y parait : le
 * changement d'une page incomplete se traduit par une diminution du
 * residu, ce qui peut amener des valeurs negatives (comprendre
 * pourquoi un tel residu negatif est valide).
 * (Voir dans la fonction FlipPages une section de code
 * presqu'identique.)
 **********************************************************************/


static npage_t ChangePage(nregion_t nreg)
{
    nwords_t  cost = RegionCost(nreg);
    nwords_t  waste = RegionWaste(nreg);
    val_t top = RegionAP(nreg);
    npage_t   apg =  RegionAPG(nreg); 

    if (apg != 0) 
      PageTop(apg) = top;               /* release alloc. page */
    if (waste != 0)                     /* inhomogeneous region */
      {
          nwords_t actual_waste = ((val_t)(RoundUpToPageSize(top))) - top;
          nwords_t remainder = RegionRemainder(nreg);
          nwords_t new_remainder = remainder - actual_waste + waste;
          nwords_t max_remainder = OnePage - waste;

          if (new_remainder > max_remainder) /* remainder is full */
            {
                new_remainder = new_remainder - max_remainder;
                cost += OnePage;
            }

          RegionRemainder(nreg) = new_remainder;    /* anticipated */
      }
					/* now cost is known  */
 {
     npage_t new_pg;                    /* now just get the page */
          
     if (cost > FreeMem)
       {
           npage_t npg_needed = Words2Pages(cost - FreeMem);
           GetFreeMemory(npg_needed);   /* HERE: GC or EXPANSION */
#ifdef DEBUG
           if(cost > FreeMem)           /* unnecessary check */
                error("FreeMem still too low ??\n");
#endif
       }

  {
      val_t ap = RegionAP(nreg);

      if (ap !=  NULL)
	{
	    npage_t apg = RegionAPG(nreg);
	    PageTop(apg) = ap;
	}
  }
     
     new_pg = PoolGetBloc(1);		/* here cost <=  freemem */

#ifdef DEBUG
     if(new_pg == NullPage)             /* unnecessary check */
          error("ChangePage : POOL empty ?\n");
#endif
					/* the page is there */

     PageTag(new_pg) = RegionTag(nreg);	/* incorporate in region */
     RegionAPG(nreg) = new_pg;		/* make it allocation page of reg. */
     FreeMem -= cost;			/* consume memory */
     PagesSinceLastGC += 1;		/* maintain threshold count */

     return new_pg;
 }
}



/**********************************************************************
 * GetFreeMemory
 **********************************************************************
 * GetFreeMemory(needed): tente d'obtenir au moins needed pages de
 * memoire libre,  soit par une expansion du tas,  soit par un GC.
 * Cette fonction est appelee lorsque le compte de la memoire libre
 * ne permet pas de prendre une page dans le POOL. Cette fonction
 * implemente la strategie principale de gestion de la memoire.  
 *
 * GetFreeMemory s'applique lors des changements de pages d'allocation
 * (memoire libre saturee). Pour l'allocation de gros objets,  la
 * strategie est implementee par les fonctions GetBlocNoBloc et
 * GetBlocNoMemory (cf: ci-dessous). 
 **********************************************************************
 * Un echec de GetFreeMemory est normalement un echec d'execution.
 * Dans des circonstances limites on peut cependant envisager
 * d'outrepasser le compte de la memoire libre et de consommer les
 * pages du pool. Cette possibilite n'est pas implementee.
 **********************************************************************
 * Fonction annexe: ComputeExpansionWish(needed) determine le nombre
 * de pages a demander au systeme: il faut expanser le tas assez vite
 * au debut,  puis moins vite (on ne veut pas doubler a chaque
 * fois...)
 **********************************************************************/


/* GetFreeMemory : obtient au moins 'needed' pages de memoire libre 
 * pour le programme mutateur. L'echec est un echec d'execution. Le GC
 * est declenche par paliers : si une quantite de pages GCThreshold
 * n'a pas ete allouee depuis le dernier gc,  une expansion du tas est
 * preferee. 
 */

static void GetFreeMemory(npage_t needed)
{
    if (PagesSinceLastGC < GCThreshold) /* if not, prefer an expansion */
      {
          npage_t wished = ComputeExpansionWish(needed);
          npage_t gotten = 0;
          ulint status = ExpandHeap(&gotten, wished, needed);

          switch(status)                /* expansion succeeded ? */
            {
           case EXP_SUCCESS:            /* yes */
#ifdef DEBUG
                if(gotten < needed)
                     error("Didn't get enough ??\n");
#endif
					/* record gain */
                FreeMem += gotten * OnePage; 
                return;                 /* and return */
           case EXP_EMAX:
           case EXP_EBRK:
                break;                  /* else try a gc */
#ifdef DEBUG
           default:
                error("ExpandHeap: mauvais code de retour\n");
                return;
#endif
            }
       {
           int status =  gc(needed);
	   PagesSinceLastGC = 0;	/* start a new threshold trip */

           switch(status)
             {
            case GC_SUCCESS:
                 return;
            case GC_FAIL:
                 error("Memory full!\n");
		 return;		/* pfj */
#ifdef DEBUG
            default:
                 error("GC : bad status!\n");
                 return;                /* pfj */
#endif
             }
       }
      }
    else                                /* if GCThreshold reached */
      {                                 /* prefer gc */
          int status = gc(needed);
	  PagesSinceLastGC = 0;		/* start a new threshold trip */

          switch (status)               /* did gc get enough ? */
            {
           case GC_SUCCESS:             /* yes, ok return */
                return;
           case GC_FAIL:                /* no, try an expansion */
                break;
#ifdef DEBUG
           default:
                error("GC : bad status!\n");
#endif
            }
       {
           npage_t wished = ComputeExpansionWish(needed);
           npage_t gotten = 0;
           ulint status = ExpandHeap(&gotten, wished, needed);

           switch (status)              /* expansion succeeded ? */
             {
            case EXP_SUCCESS:           /* yes, record gain */
                 FreeMem += gotten * OnePage;
                 return;                /* and return */
            case EXP_EMAX:
                 error("Insufficient amount of memory : see limit (1)\n");
                 break;                 /* pfj */
            case EXP_EBRK:
                 error("Virtual memory exhausted!\n");
                 break;
             }
           return;                      /* pfj */
       }
      }
}

/* ComputeExpansionWish: */

#define max(x, y)       (((x) > (y)) ? (x) : (y))

static npage_t ComputeExpansionWish(npage_t npg_needed)
{
    int gc_step = 0;
    int factor_step = 0;
    int min_step = (int)MinExpansion;
    int heap_size = (int)HeapSize;
    int res;
    
    if (GCThreshold > PagesSinceLastGC)
         gc_step = GCThreshold - PagesSinceLastGC;

    factor_step = (int)((double)heap_size * (ExpansionFactor - (double)1.0));

    res = max(npg_needed, min_step);
    res = max(res, gc_step);
    res = max(res, factor_step);
    return (npage_t)res;
}

/**********************************************************************
 * GetBlocNoBloc,  GetBlocNoMemory
 **********************************************************************
 * Cette fonction s'applique lorsque,  malgre un compte de la memoire
 * libre suffisant,  le bloc demande ne peut etre obteu du POOL
 * (fragmentation). On preferera alors l'allouer par expansion du tas,
 * ne demandant un gc qu'en cas d'echec de cette methode (le gc ne
 * garantit pas d'obtenir un bloc libre d'une taille suffisante).
 *
 * GetBlocNoBloc ne tient pas le compte de la memoire libre
 * (c'est AllocateBig).
 **********************************************************************
 * GetBlocNoMemory s'applique lorsque le compte de la memoire
 * libre ne permet pas l'allocation du gros objet. La meme strategie
 * est appliquee: celle de GetFreeBloc. 
 **********************************************************************
 * GetFreeBloc
 *
 * La fonction GetFreBloc implemente la strategie d'acquisition d'un
 * bloc de pages contigues. La strategie implementee consiste a
 * preferer l'expansion du tas au gc.
 *
 * L'echec de GetFreeBloc se traduit par un echec d'execution.
 **********************************************************************/


static npage_t GetBlocNoBloc(npage_t needed)
{
    GetFreeBloc(needed);
 {                                      /* now block should be there */
     npage_t res =  PoolGetBloc(needed);
#ifdef DEBUG
     if(res == NullPage)
          error("Missing contiguous block!\n");
#endif
     return res;
 }
}

static npage_t GetBlocNoMemory(npage_t needed)
{
    GetFreeBloc(needed);
 {                                      /* now block should be there */
     npage_t res =  PoolGetBloc(needed);
#ifdef DEBUG
     if(res == NullPage)
          error("Missing contiguous block!\n");
#endif
     return res;
 }
}


/*
 * GetFreeBloc : assure la disponibilite de needed pages contigues.
 */

static void GetFreeBloc(npage_t needed)
{
    npage_t wished = ComputeExpansionWish(needed);
    npage_t gotten = 0;
    ulint status = ExpandHeap(&gotten, wished, needed);

    switch (status)
      {
     case EXP_SUCCESS:
#ifdef DEBUG
	  if (gotten < needed) 
	       error("ExpandHeap didn't get enough!\n");
#endif
	  FreeMem += gotten * OnePage;
          return;
     case EXP_EMAX:
     case EXP_EBRK:
       {
	   int status = gc(needed);
	   PagesSinceLastGC = 0;	/* new threshold trip */

	   switch(status)
	     {
	    case GC_SUCCESS:
	      {
		  npage_t res = PoolGetBloc(needed);
		  if (res == NullPage)
		       error("Too much fragmentation and memory full!\n");
		  PoolPutFirst(res);
		  return;
	     case GC_FAIL:
		  error("Memory full!\n");
#ifdef DEBUG
	     default:
		  error("GC : bad status!\n");
#endif
		  return;		/* pfj */
	      }
	     }
       }
#ifdef DEBUG	  
     default:
          error("GetFreeBloc: bad status\n");
          return;                       /* pfj */
#endif
      }
    return;
}





/**********************************************************************
 * Desallocation
 **********************************************************************
 * ReleasePage: rend au POOL une page de petits objets. Le cout de la
 * page est rendu au compte de la memoire libre. Le residu eventuel
 * est perdu (en attendant mieux).
 **********************************************************************/

static void ReleasePage(npage_t pn)
{
    nwords_t cost = RegionCost((PageTag(pn)).nregion);

    PoolPutFirst(pn);
    FreeMem += cost;

    return;
}

/* ReleaseBloc: rend un gros objet. Les pages occupees par l'objet 
 * sont rendues au compte de la memoire libre. 
 */

extern void ReleaseBloc(npage_t bn) 
{
    npage_t bsz = BlocSize(bn);

    PoolPutFirst(bn);
    FreeMem += bsz * OnePage;

    return;
}


/**********************************************************************
 *                          Initialisation
 *
 * La fonction d'initialisation InitMemoryManager est appelee par le
 * point d'entree de l'application. Elle demande au gestionnaire de
 * bas niveau suffisamment de pages pour avoir 'npages' pages d'objets
 * allouables,  ou plutot un credit de npages d'objets : si on alloue
 * des objets copiables,  ca fait moins en realite.
 **********************************************************************/

extern int NBadRegions,  NGCBadRegions;

extern void InitMemoryManager(npage_t npages)
{
    npage_t object_pages = npages;
    npage_t total_pages = object_pages;
    
    RoundUpPages = NBadRegions + NGCBadRegions;
    total_pages += RoundUpPages;

    PagesSinceLastGC = 0;

    InitMM(total_pages);

#ifdef DEBUG     
     if (PoolSize != total_pages)
          message("PoolSize !=  total_pages\n");
#endif

    object_pages = PoolSize - RoundUpPages; /* ce qu'on a eu ! */
    FreeMem = object_pages * OnePage;
    InitGC();
}

