(* streams.ml: translation of streams *)
(* Regis Cridlig 1993 *)

#open "config";;
#open "misc";;
#open "constants";;
#open "syntax";;
#open "k2";;
#open "match";;
#open "tr_env";;
#open "hashtable";;

(* The following constants must be kept in sync with the definition
   of type stream in file ../Lib/stream.ml *)

let sempty_tag = ConstrRegular(0,5)
and scons_tag  = ConstrRegular(1,5)
and sapp_tag   = ConstrRegular(2,5)
and sfunc_tag  = ConstrRegular(3,5)
;;

(* Translation of stream expressions *)

let translate_stream translate_expr env lenv decls defs stream_comp_list =
  let rec transl_stream env = function 
    [] -> (* Kconst(SCblock(sempty_tag,[])) *)
      Kconst(SCatom(ACtag 0))  (* tag de sempty_tag *)
  | component :: rest ->
      let tag =
        match component with Zterm _ -> scons_tag | Znonterm _ -> sapp_tag in
      let e = match component with Zterm e -> e | Znonterm e -> e 
      and nom = new_id (!module_name ^ "stream") in
      Kprim(Pmakeblock sfunc_tag,
        [Kflet([nom,["tr*strm"],
                 [(Kprim(Pmakeblock tag,
                         [translate_expr env lenv decls defs e;
                          transl_stream env rest]))]],
               [Kfunction nom]);
         Kconst(const_unit) ]) in
  transl_stream env stream_comp_list
;;

(* Translation of stream parsers *)

let stream_oper name decls args =
  let id = "stream:" ^ name ^ "*" ^ (string_of_int (list_length args))
  in  insere (Kexternfun(id,list_length args)) decls;
      Kapply(id, args)
;;

let parse_error_tag =
  ConstrExtensible ({qual="stream"; id="Parse_error"}, 1)
and parse_failure_tag =
  ConstrExtensible ({qual="stream"; id="Parse_failure"}, 2)
;;

let raise_parse_failure = 
    Kprim(Praise, [Kconst(SCblock(parse_failure_tag,[]))])
and raise_parse_error =     
    Kprim(Praise, [Kconst(SCblock(parse_error_tag,[]))])
;;

let catch_parse_failure l =
  let cont = new_id "conti"
  and res  = new_id "resu"
  and id   = new_id (!module_name ^ "catch")
  and tag  = tag_of_string("stream:Parse_failure")
  in Klet([cont,Kvar "*handle*"],
       [Kflet([id,[],[Ksetq("*handle*",Kthecont); l]],
          [Klet([res,Kapply(id,[])],
             [Ksetq("*handle*",Kvar cont);
               Kif(Kprim(Ptest (Pint_test PTeq),
                           [Kvar "*try*"; Kconst(SCatom(ACtag 0))]),
                   Kvar res,
                   Kprogn[Ksetq("*try*",Kconst(SCatom(ACtag 0)));
                          Kif(Kprim(Ptest (Pint_test PTeq),
                                      [Kvar res; Kconst(SCatom(ACtag tag))]),
                              Kreturn("parse_failure",Kvoid),
                              Kprim(Praise, [Kvar res]))])])])])
;;

let rec divide_term_parsing = function
    (Ztermpat pat :: spatl, act) :: rest ->
      let (pat_case_list, parsing) = divide_term_parsing rest in
        (pat, (spatl, act)) :: pat_case_list, parsing
  | parsing ->
        ([], parsing)
;;

let access_stream (* env *) =
  translate_access "%stream" (* env *)
;;

let translate_parser var translate_expr lenv decls defs loc init_env case_list=
  let rec transl_inner env (patl, act) var =
    match patl with
      [] ->
        translate_expr env lenv decls defs act
    | Ztermpat pat :: rest ->
        let (new_env, add_lets) = add_pat_to_env var env pat in
          Klet(["term*",stream_oper "stream_require" decls[access_stream env]],
               [translate_matching
                 (fun tsb -> raise_parse_error) loc
                 [[pat], (* un seul pattern *)
                  add_lets(Kprogn [stream_oper "stream_junk" decls
                                               [access_stream new_env];
                                   transl_inner new_env (rest,act) var])]
                 ["term*"]])
    | Znontermpat(parsexpr, pat) :: rest ->
        let id = new_id "nonterm" in
        let (new_env, add_lets) = add_pat_to_env id env pat in
          Klet([id,
                stream_oper "parser_require" decls
                         [translate_expr env lenv decls defs parsexpr; 
                          access_stream env]],
               [translate_matching
                 (fun tsb -> raise_parse_error) loc
                 [[pat], add_lets(transl_inner new_env (rest,act) var)]
                 [id]])
    | Zstreampat id :: rest ->
        Klet(["stream*", access_stream env],
             [transl_inner (Tenv([id,Path_root "stream*"],env))(rest,act) var])
 
    in let rec transl_top env parsing var =
    match parsing with
      [] ->
        raise_parse_failure
    | ([], act) :: _ ->
        translate_expr env lenv decls defs act
    | (Ztermpat _ :: _, _) :: _ ->
        let translate_line id (pat, case) =
          let (new_env, add_lets) = add_pat_to_env id env pat in
            ([pat],
             add_lets(Kprogn [stream_oper "stream_junk" decls
                                          [access_stream new_env];
                              transl_inner new_env case id])) in
        begin match divide_term_parsing parsing with
          (pat_case_list, []) -> (* que des patterns terminaux *)
            let id = new_id "tnil" in
            Klet([id,stream_oper "stream_peek" decls [access_stream env]],
                 [translate_matching
                    (fun tsb -> raise_parse_failure) loc
                    (map (translate_line id) pat_case_list)
                    [id]])
        | (pat_case_list, rest) -> (* cas general *)
            Kblock("success",
              [Kblock("parse_failure",
                 let id = new_id "trest" in
                 [Klet([id,
                         catch_parse_failure(
                          stream_oper "stream_peek" decls[access_stream env])],
                       [Kreturn("success",
                         translate_matching
                          (fun tsb -> Kreturn("parse_failure",Kvoid)) loc
                          (map (translate_line id) pat_case_list)
                          [id])])]);
               transl_top env rest var])
        end
    | (Znontermpat(parsexpr, pat) :: spatl, act) :: [] ->
(*        print_string "nontermpat1\n"; *)
        let id = new_id "ntnil" in
        let (new_env, add_lets) = add_pat_to_env id env pat in
          Klet([id,
                Kfuncall(translate_expr env lenv decls defs parsexpr, 
                         [access_stream env])],
               [translate_matching
                  (fun tsb -> raise_parse_failure) loc
                  [[pat], add_lets (transl_inner new_env (spatl,act) var)]
                  [id]])
    | (Znontermpat(parsexpr, pat) :: spatl, act) :: rest ->
(*        print_string "nontermpat2\n"; *)
        let id = new_id "ntrest" in
        let (new_env, add_lets) = add_pat_to_env id env pat in
          Kblock("success",
            [Kblock("parse_failure",
               [Klet([id,
                      catch_parse_failure(
                        Kfuncall(translate_expr env lenv decls defs parsexpr,
                                 [access_stream env]))],
                     [Kreturn("success",
                       translate_matching
                        (fun tsb -> Kreturn("staticfail",Kvoid)) loc
                        [[pat], 
                          add_lets(transl_inner new_env (spatl,act) var)]
                        [id])])]);
             transl_top env rest var])
    | (Zstreampat id :: spatl, act) :: _ ->
        Klet(["stream*pat",access_stream env],
             [transl_inner (Tenv([id,Path_root "stream*pat"],env)) (spatl,act)
                           var])

  in (*let nom = new_id (!module_name ^ "top")
     and var = new_id "top*arg"
     in Kflet([nom,[var],
                [*) (*print_string(var^"\n");*)
     transl_top (Tenv(["%stream", Path_root var],init_env)) case_list var  
(*]],
              [Kfunction nom]) *)
;;
