(* tr_env.ml: handling of the translation environment. *)

#open "misc";;
#open "constants";;
#open "syntax";;
#open "k2";;
#open "globals";;
#open "type_errors";;

let module_name = ref "";;

(* Chaque case de l'environnement correspond a un motif;
   on lui associe la liste des variables de ce motif avec pour chaque
   le chemin d'acces *)

type access_path =
    Path_root of k2ident (* variable liee par Klet *)
  | Path_son of int * access_path
;; (* ajuste' suivant la presence d'un tag pour les constructeurs *)

type adjust = Direct | Indirect of int (* decalage *)
;; (* on pourrait certainement s'en passer grace a la primitive shift_tag *)

type transl_env =
    Tnullenv
  | Tenv of (k2ident * access_path) list * transl_env
;; 

let rec translate_path = function (* tient compte de la ?presence? du tag *)
    Path_root id       -> Kvar id
  | Path_son(n, p)     -> Kprim(Pfield n, [translate_path p])
;;

let translate_access s env =
  let rec search_access = function
    Tnullenv         -> fatal_error ("search_access: " ^ s)
  | Tenv(L,env)      -> try assoc s L
                        with Not_found -> search_access env
  in let path = search_access env
     in translate_path path
;;

let translate_update s env newval =
  let rec search_update = function
    Tnullenv         -> fatal_error "search_update"
  | Tenv(L,env)      -> try 
                          match assoc s L with
                            Path_root _ -> raise Not_found
                          | Path_son(start,rest) -> 
                              Kprim(Psetfield start, 
                                [translate_path rest; newval])
                        with Not_found -> search_update env
  in search_update env
;;

let rec paths_of_pat adjust path (Pat(desc,loc)) =
  match desc with
    Zvarpat s ->
      [s, match adjust with Indirect inc -> Path_son(inc,path) 
                          | Direct       -> path]
  | Zaliaspat(pat,s) ->
      (s, match adjust with Indirect inc -> Path_son(inc,path) 
                          | Direct -> path) 
      :: paths_of_pat adjust path pat
  | Ztuplepat(patlist) ->
      let rec paths_of_patlist i = function
        [] -> []
      | p::pl ->
          paths_of_pat Direct (Path_son(i,path)) p 
          @ paths_of_patlist (i+1) pl
      in (match adjust with Indirect inc -> paths_of_patlist inc patlist
                          | Direct -> paths_of_patlist 0 patlist)
  | Zconstruct0pat(cstr) -> []  (* constant constructor *)
  | Zconstruct1pat(cstr, p) ->
      begin match cstr.info.cs_kind with
        Constr_superfluous -> paths_of_pat adjust path p
      | Constr_tagless _ -> 
          let path1 = match adjust with Indirect inc -> Path_son(inc,path)
                                      | Direct -> path
          in paths_of_pat (Indirect 0) path1 p
      | Constr_regular _ -> 
          let path1 = match adjust with Indirect inc -> Path_son(inc,path)
                                      | Direct -> path
          in paths_of_pat (Indirect 1) path1 p
      | Constr_constant _ -> failwith "paths_of_pat" 
      end
  | Zconstraintpat(pat,_) -> paths_of_pat adjust path pat
  | Zrecordpat lbl_pat_list -> (* tjs alloues separement. Pourquoi ?? *)
      let rec paths_of_lbl_pat_list = function
        [] -> []
      | (lbl,p)::pl ->
             paths_of_pat Direct 
                          (match adjust with 
                             Indirect inc -> 
                               Path_son(lbl.info.lbl_pos,Path_son(inc,path) )
                           | Direct -> Path_son(lbl.info.lbl_pos,path)) 
                          p
             @ paths_of_lbl_pat_list pl in
      paths_of_lbl_pat_list lbl_pat_list
  | _ -> []
;;

(*let rec mutable_vars_of_pat (Pat(desc,loc)) =
  match desc with
    Zaliaspat(pat,v) -> mutable_vars_of_pat pat
  | Zconstraintpat(pat, _) -> mutable_vars_of_pat pat
  | Ztuplepat patl -> flat_map mutable_vars_of_pat patl
  | Zconstruct1pat(cstr,pat) ->
      begin match cstr.info.cs_mut with
        Mutable -> free_vars_of_pat pat
      | Notmutable -> mutable_vars_of_pat pat
      end
  | Zrecordpat lbl_pat_list ->
      flat_map
        (fun (lbl,pat) ->
          match lbl.info.lbl_mut with
            Mutable -> free_vars_of_pat pat
          | Notmutable -> mutable_vars_of_pat pat)
        lbl_pat_list
  | _ -> []
;;*)

let rec add_lets_to_env varlist env =
  match varlist with
    [] -> env
  | var::rest -> add_lets_to_env rest (Tenv([var,Path_root var], env))
;; 

let rec addl_lets_to_env varlistl env =
  match varlistl with
    [] -> env
  | varlist::rest -> 
      addl_lets_to_env rest (add_lets_to_env varlist env)
;;

let add_lets_to_expr varlist env expr =
  let rec add = function
      [] -> []
    | var::rest -> (var,(translate_access var env)) :: add rest in
  Klet(add varlist,[expr])
;;

let add_pat_to_env id env pat =
  let env' = Tenv(paths_of_pat Direct (Path_root id) pat,env) in
  let mut_vars = free_vars_of_pat pat in (* was mutable_vars_of_pat *)
    (add_lets_to_env mut_vars env', add_lets_to_expr mut_vars env')
;;

let add_pat_list_to_env ids env patl = 
  let env' = Tenv(pat_list_to_env ids patl,env)
  where rec pat_list_to_env = fun
       []         []      -> []
  | (id::ids) (pat::patl) -> 
      (paths_of_pat Direct (Path_root id) pat) @ (pat_list_to_env ids patl)
  |    _          _       -> fatal_error "add_pat_list_to_env"
  in let mut_varsl = map free_vars_of_pat patl (* was mutable_vars_of_pat *)
     and mut_vars = flat_map free_vars_of_pat patl (* bof ! *)
  in (addl_lets_to_env mut_varsl env', add_lets_to_expr mut_vars env')
;;

let add_let_rec_to_env env pat_expr_list =
  let add env (Pat(p,loc), expr) =
    match p with
      Zvarpat v -> Tenv([v, Path_root v], env)
    | _ -> illegal_letrec_pat loc in
  it_list add env pat_expr_list
;;
    
(*let env_for_toplevel_let patl =
  it_list (fun env pat -> 
    Tenv(paths_of_pat false Path_root pat, env)) Tnullenv patl
;;*)

