(* hashtable.ml : des tables d'association chaine -> infos efficaces  *)

#open "hashtbl";;
#open "misc";;

type 'a hashtable =
  { mutable hash_max: int;                   (* Taille maxi d'un bucket *)
    mutable hash_tbl: 'a list vect }         (* Les buckets *)
;;

(* On fait du hachage dynamique avec doublement de la taille de la table
   quand les buckets deviennent trop longs, mais sans hacher de nouveau
   les elements de la table. Ceci assure des temps d'acces logarithmiques
   en le nombre d'elements de la table. *)

let new_hashtable initial_size =
  { hash_max=3; hash_tbl=make_vect initial_size [] }
;;

let reset_hashtable h =
  h.hash_max <- 3;
  for i = 0 to vect_length h.hash_tbl - 1 do
    h.hash_tbl.(i) <- []
  done
;;

let duplicate_vector v =
  let n = vect_length v in
  let newv = make_vect (n+n) v.(0) in
  for i = 0 to pred n do
    newv.(i) <- v.(i);
    newv.(i+n) <- v.(i)
  done;
  newv
;;

let resize_hashtable h =
  h.hash_tbl <- duplicate_vector h.hash_tbl;
  h.hash_max <- 2 * h.hash_max;
  ()
;;

let add_to_hashtable select h data =
  let i = (hash_param 10 100 (select data)) mod (vect_length h.hash_tbl)
  in h.hash_tbl.(i) <- data :: h.hash_tbl.(i); ()
;;

let add_unique_to_hashtable select h data =
  let i = (hash_param 10 100 (select data)) mod (vect_length h.hash_tbl)
  in if not (mem data h.hash_tbl.(i))
     then begin h.hash_tbl.(i) <- data :: h.hash_tbl.(i); () end
;;

let remove_from_hashtable select h data =
  let i = (hash_param 10 100 (select data)) mod (vect_length h.hash_tbl)
  in h.hash_tbl.(i) <- exceptq data h.hash_tbl.(i); ()
;;

let find_in_hashtable select h key =
  match h.hash_tbl.((hash_param 10 100 key) mod (vect_length h.hash_tbl))
  with
    [] ->
      raise Not_found
  | [data1] ->
      if key = select data1 then data1 else raise Not_found
  | [data1; data2] ->
      if key = select data1 then data1 else
      if key = select data2 then data2 else raise Not_found
  | [data1; data2; data3] ->
      if key = select data1 then data1 else
      if key = select data2 then data2 else
      if key = select data3 then data3 else raise Not_found
  | L ->
      let depth = ref 0 in
      let rec find = function
        [] ->
          if !depth >= h.hash_max then resize_hashtable h;
          raise Not_found
      | data::rest ->
          if key = select data then
            begin if !depth >= h.hash_max then resize_hashtable h; data end
          else
            begin incr depth; find rest end
      in find L
;;

(* Un iterateur sur tous les elements d'une table hachee *)

let do_hashtable f h =
  do_vect (do_list f) h.hash_tbl
;;

(* Specialisation aux paires *)

let add_to_assoctable = add_to_hashtable fst
and add_unique_to_assoctable = add_unique_to_hashtable fst
and remove_from_assoctable = remove_from_hashtable fst
and find_in_assoctable tbl key = snd(find_in_hashtable fst tbl key)
;;

(* Specialisation aux tables d'identification *)

let identity x = x;;
  
let add_to_idtable = add_to_hashtable identity
and add_unique_to_idtable = add_unique_to_hashtable identity
and remove_from_idtable = remove_from_hashtable identity
and is_in_idtable tbl key = try find_in_hashtable identity tbl key; true
                            with Not_found -> false
;;

let insere decl table = add_unique_to_hashtable identity table decl
;;
