#open "hashtable";;
#open "k2";;
#open "queue";;
#open "misc";;
#open "constants";;
#open "convert";;

(* globalise.ml : transforme du bivaleur avec fermeture en du bivaleur sans
                  sans fermetures (c'est-a-dire du Sqil)

                  On effectue une alpha-conversion au niveau de chaque
                  fonction; de plus, on suppose que les variables
                  ne sont pas mutables (ML).
*)

let rec map2 f = fun
     []       []    -> []
| (x1::l1) (x2::l2) -> (f x1 x2)::(map2 f l1 l2)
|    _        _     -> fatal_error "map2"
;;

let cons_ref elt reflst = begin reflst := elt :: !reflst; () end;;

let queue2list q = 
  let l = ref []
  in iter (function x -> l := x :: !l) q;
     rev !l
;;  

(* types utilises *)
type k2var = {Id:k2ident; Level:int (*; mutable Closed:bool*)};;

type k2fun = {Oldid:k2ident; 
              mutable Newid:k2ident; 
              Args:k2var list; 
              Body:k2exp list; 
              Niveau:int;
              Static:bool; (* static or extern *)
              Wrap:k2fun list; (* wrapping functions *)
              mutable Global:bool; (* a globaliser ? *)
              mutable Anonym:bool; (* de facon anonyme ? *)
              mutable Integ:k2ident; (* fonction a integrer *)
              mutable Closed_list:k2var list; (* variables closes *)
              mutable Free_vars:k2var list; 
              mutable Free_funs: k2fun list;
              mutable Loci: k2fun list};;

(* variables globales *)
let functions = ref ((new_hashtable 0):k2fun hashtable)
and global_functions = ref ([]:k2fun list)
and variables = ref ((new_hashtable 0):k2var hashtable)
and global_closed = ref ([]:k2var list)
and global_anonym = ref false
and symb_env = new_id "ENV"
and symb_val = new_id "VAL"
and symb_cenv = new_id "CENV"
and symb_fn = new_id "FN"
;;

let (*in_main = ref false (* si true, on compile dans main *) and *)
    const_list = ref ([]:(k2ident * k2exp) list)
and fun_extern = ref ([]:(k2ident * int) list)
and fun_static = ref ([]:(k2ident * int) list)
and extern_anonym = ref ((new_hashtable 0):k2ident hashtable)
(* fcts globales referencees de maniere anonyme *)
;;

let special id = let len=string_length id in "*env"=sub_string id (len-4) 4;;

(* creation de fonctions et variables *)
let select_var var = var.Id
and select_fun func = func.Oldid
;;

let the_var id = find_in_hashtable select_var !variables id
     
and the_fun id = find_in_hashtable select_fun !functions id
;;

let new_vars ids level =
  map (fun id -> (*print_string (id ^ " est nouvelle\n");*)
       let def_var = {Id=id; Level=level (*; Closed=false*)}
          in add_to_hashtable select_var !variables def_var; def_var)
      ids
;;

(* prise en compte d'une variable ou fonction libre *)

let rec filtre f x =
  let rec iter fx = function
     []  -> false
  | a::d -> if (f a)=fx then true else iter fx d
  in iter (f x)
;;

exception Fini;;

let is_free_var var wfuns =
  let l = var.Level
  in (*print_string (var.Id ^ " est libre\n");*)
     try do_list (fun wfun -> 
                    if wfun.Niveau <= l then raise Fini
                    else if not(filtre select_var var wfun.Free_vars)
                         then begin wfun.Free_vars <- var::wfun.Free_vars;
(*            print_string (var.Id ^ " var libre dans " ^ wfun.Oldid ^ "\n");*)
                                    () end)
	          wfuns
     with Fini -> ()
;;

let is_free_fun f wfuns =
  let l = f.Niveau
  in try do_list (fun wfun -> 
                    if wfun.Niveau < l then raise Fini
                    else if not(filtre select_fun f wfun.Free_funs)
                         then begin wfun.Free_funs <- f::wfun.Free_funs;
         (*print_string (f.Oldid ^ " fct libre dans " ^ wfun.Oldid ^ "\n");*)
                                    () end)
	          wfuns
     with Fini -> ()
;;

(* collecte des informations sur les variables et fonctions *)
let rec collects2 forms wfuns =
  let rec new_funs defs recp =
        let level = 1 + (hd wfuns).Niveau in
        let funs = map (function (id,argl,body) -> 
		           let args = new_vars argl level
                           in {Oldid=id;
			       Newid=id;
			       Args=args;
			       Body=body;
			       Niveau=level;
			       Global=false;
                               Anonym=false;
                               Integ="";
                               Static=true;
			       Closed_list=[];
			       Free_vars=[];
			       Free_funs=[];
                               Loci=[];
                               Wrap=wfuns})
  	               defs
        in if recp then 
             begin
               do_list 
                 (function f -> add_to_hashtable select_fun !functions f)
                 funs;
               do_list (function f -> collects2 f.Body (f::wfuns)) funs
             end
           else 
             begin
               do_list (function f -> collects2 f.Body (f::wfuns)) funs;
               do_list 
                 (function f -> add_to_hashtable select_fun !functions f)
                 funs
             end

  and collect = function
    Kvar id -> (try is_free_var (the_var id) wfuns
                with Not_found -> ())
  | Ksetq(id,exp) -> collect exp (* id: variable globale definie par defvar *)
  | Kfunction id -> (try let func = the_fun id
                         in func.Global <- true;
                            func.Anonym <- true;
                            if func.Niveau = 1 
                              then add_unique_to_idtable !extern_anonym id;
                            is_free_fun func wfuns;
                            if not(filtre select_fun (hd wfuns) func.Loci)
                              then begin func.Loci <- (hd wfuns)::func.Loci;
                                         () end
                     with Not_found ->
                       if not(special id) 
                         then add_unique_to_idtable !extern_anonym id)
  | Klet(binds,body) -> collects (map snd binds);
                        new_vars (map fst binds) (hd wfuns).Niveau;
                        collects body
  | Klabels(binds,body) -> new_funs binds true; collects body
  | Kflet(binds,body) -> new_funs binds false; collects body 
  | Kthecont -> (hd wfuns).Global <- true; ()
  | Kconst _ -> ()
  | Kintern _ -> failwith "globalise: Kintern"
  | Kif(test,th,el) -> collect test; collect th; collect el
  | Kcont(exp1,exp2) -> collect exp1; collect exp2
  | Kprogn expl -> collects expl
  | Kfuncall(func,expl) -> collect func; collects expl
  | Kblock(_,expl) -> collects expl
  | Kreturn(_,exp) -> collect exp
  | Kapply(id,expl) -> (try let f = the_fun id
                            in is_free_fun f wfuns;
                               if not(filtre select_fun (hd wfuns) f.Loci)
                                 then begin f.Loci <- (hd wfuns)::f.Loci;() end
                        with Not_found -> ());
                       collects expl
  | Kcase(exp,cond_list,other) -> 
      collect exp; do_list (function (_,ex) -> collect ex) cond_list;
      collect other
  | Kswitch(exp,switch_list,other) -> 
      collect exp; do_list (function (_,ex) -> collect ex) switch_list;
      collect other
  | Kprim(_,expl) -> collects expl
  | Kvoid -> ()

  and collects forml = do_list collect forml

in collects forms
;;

(* analyse des fermetures *)

let rec find_global_integ_functions () =
  let redo = ref false in
  let rec check_loci f id = function
    [] -> true
  | g::gs -> if (select_fun f)=(select_fun g) then check_loci f id gs
             else if g.Oldid = id then check_loci f id gs
                  else if g.Oldid != g.Newid or g.Niveau = 1 then false
                       else if mem id (map select_fun g.Wrap) 
                              then check_loci f id gs
                            else false  (* non optimal *)
  and is_global f =
    if f.Newid != f.Oldid or f.Niveau=1 then ()  (* deja globale *)
    else begin
         (*  print_string ((f.Oldid)^" globalisee\n"); *)
           redo := true;
           f.Newid <- new_id (f.Oldid); (* on renomme *)
           f.Global <- true; (* la fonction est globalisee *)
           global_functions := f :: !global_functions;
           do_list (fun g -> if (select_fun g)=(select_fun f) then ()
                             else integre g f.Oldid)
                   f.Free_funs
         end
  and integre f id =
(*    print_string ((f.Oldid)^" a integrer dans " ^ id ^ "\n"); *)
    if f.Global then is_global f
    else if check_loci f id f.Loci then 
           begin f.Integ <- id;
(*    print_string ((f.Oldid)^" est integree dans " ^ id ^ "\n"); *)
                 do_list (fun g -> if (select_fun g)=(select_fun f) then () 
                                   else integre g id)
                         f.Free_funs
           end
         else is_global f

  in do_hashtable (function f -> if f.Global then is_global f 
                                 else ())
                  !functions;
     if !redo then find_global_integ_functions ()
;;

let rec iterate () =
  let redo = ref false
  in do_hashtable 
       (fun f -> do_list (function f2 ->
			    if (select_fun f)=(select_fun f2) then ()
			    else do_list (function v -> 
				            let closed = f.Closed_list in
					    if not(filtre select_var v closed)
					      then begin
						   f.Closed_list <- v::closed;
					       	   redo := true; ()
					           end)
					 f2.Closed_list)
			 f.Free_funs)
       !functions;
     if !redo then iterate () else ()
;;

let find_closed_vars () =
  do_hashtable (fun f -> f.Closed_list <- f.Free_vars; ()) !functions;
  iterate ()
(*;do_list(fun f -> do_list (function var -> var.Closed <- true) f.Closed_list)
	  !global_functions *)
;;

let analyze_closures () =
  find_global_integ_functions ();
  find_closed_vars ()
;;  

(* reconstruction d'une forme *)
(* une cloture est un pointeur sur une fonction suivi d'un environnement *)
let prognify = function
  []     -> fatal_error "prognify"
| [form] -> form
| forms  -> Kprogn forms
;;

let rec build_function_bindings = function
  [] -> []
| (id,args_ids,body)::binds -> 
    let func = the_fun id 
    in match func with
         {Global=true} -> build_function_bindings binds
       | {Integ=st} -> 
          if st=""
            then (id,args_ids,builds body) :: (build_function_bindings binds)
          else build_function_bindings binds

and build = function
  (Kvar id) as form -> form
| Ksetq(id,exp) -> Ksetq(id,build exp) (* id est une variable globale *)
| (Kfunction id) as form -> 
    (try let func = the_fun id
         in match func with
              {Niveau=1} -> Kvar(id ^ "*closure") (* fct toplevel *)
            | {Closed_list=closed; Newid=st} ->
                 Kprim(Pbuildclosure, (Kfunction st)::
                                    (map (fun {Id=var} -> Kvar var) closed))
     with Not_found -> if special id then form
                                     else Kvar(id ^ "*closure"))
| Klet([],body) -> prognify (builds body)
| Klet(binds,body) -> 
    Klet(map (fun (id,exp) -> (id,build exp)) binds,builds body)
| Klabels(binds,body) -> let bindings = build_function_bindings binds
                         and b_body = builds body
                         in if bindings=[] then prognify b_body
                            else Klabels(bindings,b_body)
| Kflet(binds,body) ->  let bindings = build_function_bindings binds
                         and b_body = builds body
                         in if bindings=[] then prognify b_body
                            else Kflet(bindings,b_body)
| Kthecont as form -> form
| (Kconst (SCatom (ACstring _))) as form -> (* Pb: chaines non constantes *)
(*    if !in_main then form
    else*) let id = new_id "consts"
         in const_list := (id,form) :: !const_list; Kvar id
| (Kconst (SCatom _)) as form -> form
| (Kconst ((SCblock(ConstrExtensible _,[])) as block)) as form ->
(*    if !in_main then form
    else*) let id = new_id "constb"
         in const_list := (id,Kintern block) :: !const_list; Kvar id
| (Kconst (SCblock(_,[]))) as form -> form
| (Kconst ((SCblock(_,_)) as block)) as form ->
    let id = new_id "constb"
    in const_list := (id,Kintern block) :: !const_list; Kvar id
| (Kconst (SCtuple [])) as form -> form
| (Kconst ((SCtuple _) as tuple)) as form ->
    let id = new_id "constt"
    in const_list := (id,Kintern tuple) :: !const_list; Kvar id
| Kintern _ -> failwith "globalise: Kintern"
| Kif(test,th,el) -> Kif(build test,build th,build el)
| Kcont(exp1,exp2) -> Kcont(build exp1,build exp2)
| Kprogn [exp] -> build exp
| Kprogn expl -> Kprogn(builds expl)
| Kfuncall(func,expl) -> 
    let vars = new_ids "ARG" (list_length expl) in
    let binds = map2 (fun var val -> (var,build val)) vars expl
    and k2vars = map (fun var -> Kvar var) vars
    in Klet((symb_fn,build func)::binds,
         [Kfuncall(Kprim(Pclosurefun,[Kvar symb_fn]),(Kvar symb_fn)::k2vars)])
| Kblock(id,expl) -> Kblock(id,builds expl)
| Kreturn(id,exp) -> Kreturn(id,build exp)
| Kapply(id,expl) -> 
    let args = builds expl in
    (try let funct = the_fun id
         in match funct with
              {Global=false} -> Kapply(id,args)
            | {Niveau=1} -> Kapply(id,args)
            | {Newid=st; Closed_list=closed; Anonym=false} -> 
                 Kapply(st,(map (fun {Id=v} -> Kvar v) closed)@args)
            | {Newid=st; Closed_list=closed} ->
                if closed = !global_closed & !global_anonym then 
                  Kapply(st,(Kvar symb_env)::args)
	        else Kprim(Pstackenv(symb_cenv,
                                     Kapply(st,(Kvar symb_cenv)::args)),
                           map (fun {Id=var} -> Kvar var) closed)
     with Not_found -> Kapply(id,args))
| Kcase(exp,cond_list,other) -> 
    Kcase(build exp,map (function (kind,ex) -> (kind,build ex)) cond_list,
          build other)
| Kswitch(exp,switch_list,other) -> 
    Kswitch(build exp,map (function(const,ex) -> (const,build ex)) switch_list,
            build other)
| Kprim(Pccall(name,_) as prim ,expl) -> Kprim(prim,builds expl)
| Kprim(prim,expl) -> Kprim(prim,builds expl)
| Kvoid -> Kvoid

and builds = function
  [Kprogn forms] -> map build forms
| forms -> map build forms
;;

let fetch_closed_variables vars b_body =
  let rec init = fun
            []      _ -> []
  | ({Id=st}::lvar) n -> 
      (st,Kprim(Pfield n,[Kvar symb_env]))::init lvar (n+1)
  in match init vars 1 with (* was 0 *)
       [] -> prognify b_body
     | il -> Klet(il,b_body)
;;

let build_defun {Newid=id;
                 Oldid=oid;
	  	 Args=args;
		 Body=body;
                 Anonym=anonym;
                 Static=static;
		 Closed_list=closed} =
  global_closed := closed;
  global_anonym := anonym;
  let b_body = builds body
  and args_ids = map (fun {Id=st} -> st) args
  and integl = ref []
  in do_hashtable (fun f -> if f.Integ=oid & not f.Global then
                         begin integl:=(f.Newid,map (fun {Id=st} -> st) f.Args,
                                        builds f.Body)::!integl;
                               ()
                         end)
                  !functions;
     let b_body = if !integl=[] then b_body else [Klabels(!integl,b_body)] in
     if id != oid (* fonction globalisee *) then
       if anonym then 
         begin
           cons_ref (id,1 + list_length args) fun_static;
           Kdefun(id,symb_env::args_ids,[fetch_closed_variables closed b_body])
         end
       else begin
              cons_ref (id,(list_length closed)+(list_length args)) fun_static;
              Kdefun(id,(map (fun {Id=v} -> v) closed)@args_ids,b_body)
            end
     else begin if static then cons_ref (id,list_length args) fun_static
                          else cons_ref (id,list_length args) fun_extern;
                Kdefun(id,args_ids,b_body)
          end
;;
			
(* reconstruction d'une meta-forme *)
let handle_meta_form new_defs = function
  Kdefun(id,argl,body) -> 
    functions := new_hashtable 11;
    variables := new_hashtable 23;
    let func = {Oldid=id;
                Newid=id; 
                Args=new_vars argl 1; 
                Body=body; 
                Niveau=1;
		Static=false;
		Anonym=false;
                Global=true;
                Integ="";
                Closed_list=[];
                Free_vars=[];
                Wrap=[];
                Loci=[];
                Free_funs=[]}
    in add_to_hashtable select_fun !functions func;
       collects2 body [func];
       global_functions := [func];
       analyze_closures();
       do_list (function f -> add (build_defun f) new_defs)!global_functions
| (Kdefvar _)  as metaform -> add metaform new_defs
;;

(* la fonction d'appel *)
let globalise decls defs main module_name = 
  let new_defs = new ()
  and new_main = new ()
  and init = new ()
  in (*in_main := false; *)
     const_list := [];
     fun_extern := [];
     fun_static := [];
     extern_anonym := new_hashtable 37;
     begin
       try while true do
             handle_meta_form new_defs (convert (take defs))
           done
       with Empty -> ()
     end;
  (*   in_main := true; *)
     let func = Kdefun("main_of_" ^ module_name,[],queue2list main)
     in handle_meta_form new_defs (convert func);
        do_hashtable (function id -> 
          add (Kdefun(id ^ "*anonym",[symb_env;"arg"],
                           [Kapply(id,[Kvar "arg"])]))
                 new_defs;
          cons_ref (id ^ "*anonym",2) fun_static;
          cons_ref (id ^ "*closure",
                  Kprim(Pbuildclosure,[Kfunction(id ^ "*anonym")])) 
                 const_list)
          !extern_anonym;
        do_list (function (id,val) ->
                   add (Ksetq(id,val)) init;
                   add_unique_to_idtable decls (Kstaticvar id))
                !const_list;
        do_list (function (id,arite) -> 
                   add_unique_to_idtable decls (Kexternfun(id,arite)))
                !fun_extern;
        do_list (function (id,arite) -> 
                   add_unique_to_idtable decls (Kstaticfun(id,arite)))
                !fun_static;
        (decls,new_defs,init)
;;
