(* convert.ml *)
(* version 0.5 *)
(* effectue une alpha-conversion : *)
(* les noms de variables et de fonctions deviennent uniques a l'interieur *)
(* d'une fonction. *)
(* Regis Cridlig 1992 *)

#open "k2";;
#open "misc";;

type k2var = {Oldvar:k2ident; mutable Newvar:k2ident; mutable Good:bool};;

type k2fun = {Oldfun:k2ident; Newfun:k2ident};;

type k2env == (k2var list) * (k2fun list)
;;

let the_varname (env_var,_) id = 
  let rec loop = function
    [] -> id
  | {Oldvar=old;Newvar=new}::l -> if old=id then new else loop l
  in loop env_var

and the_var (env_var,_) id = 
  let rec loop = function
    [] -> failwith "find_var"
  | ({Oldvar=old} as v)::l -> if old=id then v else loop l
  in loop env_var

and the_funname (_,env_fun) id = 
  let rec loop = function
    [] -> failwith "find_fun"
  | {Oldfun=old;Newfun=new}::l -> if old=id then new else loop l
  in loop env_fun

and the_fun (_,env_fun) id = 
  let rec loop = function
    [] -> failwith "find_fun"
  | ({Oldfun=old} as f)::l -> if old=id then f else loop l
  in loop env_fun
;;

let the_varnames env ids = map (fun id -> the_varname env id) ids
and the_funnames env ids = map (fun id -> the_funname env id) ids
;;

let rec new_env = fun
  [] [] -> ([],[])
| [] (f::lf) -> let (_,fenv)=new_env [] lf in ([],{Oldfun=f;Newfun=f}::fenv)
| (v::lv) lf -> let (venv,fenv)=new_env lv lf 
                in ({Oldvar=v;Newvar=v;Good=false}::venv,fenv)
;;

let new_var ((venv,fenv) as env) = function
  (id,Kvar nom) -> ({Oldvar=id;
                     Newvar=
                     (try let v = the_var env nom
                          in if v.Good
                               then v.Newvar
                             else let newid=try the_var env id; new_id id
                                            with Failure "find_var" -> id
                                  in v.Newvar<-newid; v.Good<-true; newid
                       with Failure "find_var" -> (* variable globale *)
                              let newid=try the_var env id; new_id id
                                        with Failure "find_var" -> id
                              in newid);
                     Good=true}::venv,fenv)
| (id,_) -> let nom = try the_var env id; new_id id 
                      with Failure "find_var" -> id
            in ({Oldvar=id; Newvar=nom; Good=false}::venv,fenv)
;;

let rec new_vars env = function
  [] -> env
| bind::l -> let newenv = new_var env bind in new_vars newenv l
;;

let new_fun ((venv,fenv) as env) id =
  let nom = try the_fun env id; new_id id with Failure "find_fun" -> id
  in (venv,{Oldfun=id; Newfun=nom}::fenv)
;;

let rec new_funs env = function
  [] -> env
| f::l -> let newenv = new_fun env f in new_funs newenv l
;;

let rec build_function_bindings fenv env = function
  [] -> []
| (id,args_ids,body)::binds ->
    let func = the_funname fenv id
    and newfenv = new_vars env (map (fun id -> (id,Kvoid)) args_ids) (* truc *)
    in let newbody = builds newfenv body
    in (func,the_varnames newfenv args_ids,newbody) ::
          (build_function_bindings fenv env binds)

and build env = function
  Kvar id -> Kvar(try let v=the_var env id
                      in v.Good<-true; (* le nom ne doit plus changer ! *)
                         v.Newvar
                  with Failure "find_var" -> id)
| Kapply(id,expl) -> 
    Kapply((try the_funname env id with Failure "find_fun" -> id),
           builds env expl)
| Kfuncall(func,expl) -> Kfuncall(build env func,builds env expl)
| Ksetq(id,exp) -> Ksetq(id,build env exp) (* id est une variable globale *)
| Kif(test,th,el) -> Kif(build env test,build env th,build env el)
| Kblock(id,expl) -> Kblock(id,builds env expl)
| Kreturn(id,exp) -> Kreturn(id,build env exp)
| Kthecont as form -> form
| Kcont(exp1,exp2) -> Kcont(build env exp1,build env exp2)
| Klabels(binds,body) -> 
    let newenv = new_funs env (map (fun (f,_,_) -> f) binds) in
    let bindings = build_function_bindings newenv newenv binds in
    let b_body = builds newenv body
    in Klabels(bindings,b_body)
| Klet(binds,body) -> 
    let newenv  = new_vars env binds in
    let rec loop = function 
        [] -> []
      | (n,Kvar v)::bnds -> 
          (try the_var env v; loop bnds (* enleve les synonymes *)
           with Failure "find_var" -> 
             (the_varname newenv n,Kvar v)::(loop bnds))
      | (n,b)::bnds -> (the_varname newenv n,build env b)::(loop bnds)
          (* on garde meme les variables jamais referencees *)
    in let newbody = builds newenv body
    in Klet(loop binds,newbody)
| Kflet(binds,body) ->  
    let newenv = new_funs env (map (fun (f,_,_) -> f) binds) in
    let bindings = build_function_bindings newenv env binds in
    let b_body = builds newenv body
    in Kflet(bindings,b_body)
| Kfunction id -> Kfunction(try the_funname env id 
                            with Failure "find_fun" -> id)
| Kprogn expl -> Kprogn(builds env expl)
| Kcase((Kvar _) as exp,cond_list,other) ->
    Kcase(build env exp,
          map (function (kind,ex) -> (kind,build env ex)) cond_list,
          build env other)
| Kcase(exp,cond_list,other) ->
    let id = new_id "Kcase"
    in Klet([id,build env exp],
         [Kcase(Kvar id,
            map (function (kind,ex) -> (kind,build env ex)) cond_list,
                 build env other)])
| Kswitch(exp,switch_list,other) ->
    Kswitch(build env exp,
          map (function (const,ex) -> (const,build env ex)) switch_list,
          build env other)
| (Kconst _) as form -> form
| Kintern _ -> failwith "convert: Kintern"
| Kprim(prim,expl) -> Kprim(prim,builds env expl)
| Kvoid as form -> form

and builds env forms = map (fun form -> build env form) forms
;;

let convert = function
  (Kdefvar _) as metaform -> metaform
| Kdefun(id,argl,body) ->
    let env = new_env argl [id]
    in let newbody = builds env body
    in Kdefun((the_funname env id),(the_varnames env argl),newbody)
;;
