(* mycompiler.ml *)
(* The compiler entry points *)

#open "obj";;
#open "misc";;
#open "constants";;
#open "lexer";;
#open "lexing";;
#open "parser";;
#open "parsing";;
#open "locations";;
#open "syntax";;
#open "builtins";;
#open "hashtable";;
#open "globals";;
#open "modules";;
#open "types";;
#open "type_errors";;
#open "typing";;
#open "typing_decl";;
#open "print_decl";;
#open "typing_intf";;
#open "front";;
#open "globalise";;
#open "ecrit";;
#open "k2";;
#open "queue";;
#open "config";;
#open "filename";;
#open "sys";;
(*#open "unix";;*)

(* Fonctions de parsing *)

let parse_phrase parsing_fun lexing_fun lexbuf =
  let rec skip () =
    try
      match lexing_fun lexbuf with
        EOF -> ()
      | SEMISEMI -> ()
      | _ -> skip()
    with Lexical_error(_,_,_) ->
      skip() in
  try
    parsing_fun lexing_fun lexbuf
  with Parse_error f ->
         let pos1 = get_lexeme_start lexbuf in
         let pos2 = get_lexeme_end lexbuf in
         if f (repr EOF) or f (repr SEMISEMI) then () else skip();
         prerr_location (Loc(pos1, pos2));
         prerr_begline ">> Syntax error.";
         prerr_endline "";
         raise Toplevel
     | Lexical_error(msg, pos1, pos2) ->
         if pos1 >= 0 & pos2 >= 0 then prerr_location (Loc(pos1, pos2));
         prerr_begline ">> Lexical error: ";
         prerr_string msg;
         prerr_endline ".";
         skip();
         raise Toplevel
     | Toplevel ->
         skip ();
         raise Toplevel
;;

let parse_impl_phrase = parse_phrase Implementation Main
and parse_intf_phrase = parse_phrase Interface Main
;;

(* Execution des directives *)

let do_directive = function
    Zdir("open", name) ->
      used_modules := find_module name :: !used_modules; ()
  | Zdir("close", name) ->
      used_modules := exceptq (find_module name) !used_modules; ()
  | Zdir("infix", name) -> add_infix name; ()
  | Zdir("uninfix", name) -> remove_infix name; ()
  | Zdir(d, name) ->
      prerr_begline " Warning: Unknown directive \"";
          prerr_string d;
      prerr_endline2 "\", ignored."
;;

(* Compilation d'une interface *)

let compile_intf_phrase (Intf(desc,loc)) =
  begin match desc with
    Zvaluedecl decl ->
      type_valuedecl loc decl; ()
  | Ztypedecl decl ->
      type_typedecl loc decl; ()
  | Zexcdecl decl ->
      type_excdecl loc decl; ()
  | Zintfdirective dir ->
      do_directive dir
  end;
  flush std_out
;;

let compile_intf modname filename =
  if is_absolute filename
    then load_path := (dirname filename) :: !load_path;
  let intf_name = filename ^ ".ki" 
  and source_name = filename ^ ".mli" in
  let ic = open_in source_name
  and oc = open_out intf_name in
    new_globalenv modname;
    let lexbuf = create_lexer_channel ic in
    input_name := source_name;
    input_chan := ic;
    input_lexbuf := lexbuf;
    try
      while true do
        compile_intf_phrase(parse_intf_phrase lexbuf)
      done
    with End_of_file ->
      close_in ic;
      write_compiled_interface oc;
      close_out oc
    | x ->
      close_in ic;
      close_out oc;
      remove intf_name;
      raise x
;;

(* Compilation d'une implementation *)

let verbose = ref false;;

let compile_impl_phrase decls defs main (Impl(desc,loc)) name =
  reset_type_expression_vars [];
  begin match desc with
    Zexpr expr ->
      let ty = type_expression loc expr
      in add (translate_expression decls defs expr name) main;
         if !verbose then print_expr ty
  | Zletdef(rec_flag, pat_expr_list) ->
      let env = type_letdef loc rec_flag pat_expr_list
      in translate_letdef rec_flag decls defs main loc pat_expr_list name;
         if !verbose then print_valdef env
  | Ztypedef decl ->
      let ty_decl = type_typedecl loc decl
      in if !verbose then print_typedecl ty_decl
  | Zexcdef decl ->
      let exc_decl = type_excdecl loc decl
      in if !verbose then print_excdecl exc_decl
  | Zimpldirective dir ->
      do_directive dir
  end;
  flush std_out
;;

let compile_impl modname filename =
  let source_name = filename ^ ".ml" in
  let ic = open_in source_name in
  let lexbuf = lexing__create_lexer_channel ic in
    type_stamp := 0x40000000;
 (* This is a Dirty Hack to avoid giving the same stamp to a type defined or
    declared in the interface and to a type defined in the implementation.*)
    input_name := source_name;
    input_chan := ic;
    input_lexbuf := lexbuf;
    defined := ((new_hashtable 13) : string hashtable);
    let decls = new_hashtable 101
    and defs = new ()
    and main = new ()
    in try
         while true do
           compile_impl_phrase decls defs main (parse_impl_phrase lexbuf) 
                               modname
         done
       with End_of_file -> 
         begin
           close_in ic;
           let extstream = open_out (filename ^ ".k2")
           in k2_output (globalise decls defs main modname) extstream modname;
              close_out extstream
         end
       | x -> close_in ic; raise x
;;

let compile_implementation modname filename =
  new_globalenv modname;
  if is_absolute filename
    then load_path := (dirname filename) :: !load_path;
  if file_exists (filename ^ ".mli") then
    begin
      let intf = load_module modname in
        enter_interface_definitions intf;
        compile_impl modname filename;
        check_interface intf;
        kill_module modname intf
    end
  else
    begin
      let intf_filename = filename ^ ".ki" in
      let oc = open_out intf_filename in
        try
          compile_impl modname filename;
          write_compiled_interface oc;
          close_out oc
        with x ->
          close_out oc;
          remove_file intf_filename;
          raise x
    end
;;
