(* builtins.ml : the pre-defined global identifiers *)

#open "misc";;
#open "constants";;
#open "globals";;
#open "hashtbl";;
#open "modules";;

let builtin n d = {qualid={qual="builtin"; id=n}; info=d}
;;

(* Some types that must be known to the type checker *)

let constr_type_unit = builtin "unit"     {ty_stamp=2; ty_dang=false}
and constr_type_exn = builtin "exn"       {ty_stamp=3; ty_dang=false}
and constr_type_bool = builtin "bool"     {ty_stamp=4; ty_dang=false}
and constr_type_int = builtin "int"       {ty_stamp=5; ty_dang=false}
and constr_type_float = builtin "float"   {ty_stamp=6; ty_dang=false}
and constr_type_string = builtin "string" {ty_stamp=7; ty_dang=false}
and constr_type_char = builtin "char"     {ty_stamp=8; ty_dang=false}
and constr_type_list = builtin "list"     {ty_stamp=9; ty_dang=false}
and constr_type_vect = builtin "vect"     {ty_stamp=10; ty_dang=true}
and constr_type_stream = {qualid = {qual="stream"; id="stream"};
                          info   = {ty_stamp=1; ty_dang=false}}
    (* This assumes that "stream" is the first type defined in
       the module "stream". *)
;;

let type_arrow (t1,t2) =
  {typ_desc=Tarrow(t1, t2); typ_level=notgeneric}
and type_product tlist =
  {typ_desc=Tproduct(tlist); typ_level=notgeneric}
and type_unit =
  {typ_desc=Tconstr(constr_type_unit, []); typ_level=notgeneric}
and type_exn =
  {typ_desc=Tconstr(constr_type_exn, []); typ_level=notgeneric}
and type_bool =
  {typ_desc=Tconstr(constr_type_bool, []); typ_level=notgeneric}
and type_int =
  {typ_desc=Tconstr(constr_type_int, []); typ_level=notgeneric}
and type_float =
  {typ_desc=Tconstr(constr_type_float, []); typ_level=notgeneric}
and type_string =
  {typ_desc=Tconstr(constr_type_string, []); typ_level=notgeneric}
and type_char =
  {typ_desc=Tconstr(constr_type_char, []); typ_level=notgeneric}
and type_vect t =
  {typ_desc=Tconstr(constr_type_vect, [t]); typ_level=notgeneric}
and type_stream t =
  {typ_desc=Tconstr(constr_type_stream, [t]); typ_level=notgeneric}
;;

(* Some constructors that must be known to the parser *)

let constr_void =
  builtin "()"
    { cs_res = {typ_desc=Tconstr(constr_type_unit,[]); typ_level=notgeneric};
      cs_arg = type_unit;
      cs_mut = Notmutable;
      cs_kind= Constr_constant (ConstrRegular(0,1))}
;;

let constr_nil =
  let arg = {typ_desc=Tvar(Tnolink); typ_level=generic} in
  builtin "[]"
    { cs_res = {typ_desc=Tconstr(constr_type_list, [arg]); typ_level=generic};
      cs_arg = type_unit;
      cs_mut = Notmutable;
      cs_kind= Constr_constant (ConstrRegular(0,2))}

and constr_cons =
  let arg1 = {typ_desc=Tvar(Tnolink); typ_level=generic} in
  let arg2 = {typ_desc=Tconstr(constr_type_list, [arg1]); typ_level=generic} in
  builtin "::"
    { cs_res = arg2;
      cs_arg = {typ_desc=Tproduct[arg1; arg2]; typ_level=generic};
      cs_mut = Notmutable;
      cs_kind= Constr_tagless(2,2)}
;;

let constr_false =
  builtin "false"
    { cs_res = {typ_desc=Tconstr(constr_type_bool,[]); typ_level=notgeneric};
      cs_arg = type_unit;
      cs_mut = Notmutable;
      cs_kind= Constr_constant (ConstrRegular(0,2))}

and constr_true =
  builtin "true"
    { cs_res = {typ_desc=Tconstr(constr_type_bool,[]); typ_level=notgeneric};
      cs_arg = type_unit;
      cs_mut = Notmutable;
      cs_kind= Constr_constant (ConstrRegular(1,2))}
;;

(* Some exceptions that must be known to the compiler *)

let match_failure_tag =
  ConstrExtensible ({qual="builtin"; id="Match_failure"}, 1)
;;

let constr_match_failure =
  builtin "Match_failure"
    { cs_res = {typ_desc=Tconstr(constr_type_exn,[]); typ_level=notgeneric};
      cs_arg = type_product [type_string; type_int; type_int];
      cs_mut = Notmutable;
      cs_kind = Constr_regular(match_failure_tag,false,3)}
;;

(* Construction of the "builtin" module *)

let module_builtin = new_module "builtin";;

do_list
  (fun (name,desc) ->
      add module_builtin.mod_types name (builtin name desc))
  ["unit",
   {ty_constr=constr_type_unit; ty_arity=0; ty_desc=Variant_type[constr_void]};
   "exn",
    {ty_constr=constr_type_exn; ty_arity=0; ty_desc=Variant_type []};
   "bool",
    {ty_constr=constr_type_bool; ty_arity=0;
     ty_desc=Variant_type [constr_false; constr_true]};
   "int",
    {ty_constr=constr_type_int; ty_arity=0; ty_desc=Abstract_type};
   "float",
    {ty_constr=constr_type_float; ty_arity=0; ty_desc=Abstract_type};
   "string",
    {ty_constr=constr_type_string; ty_arity=0; ty_desc=Abstract_type};
   "char",
    {ty_constr=constr_type_char; ty_arity=0; ty_desc=Abstract_type};
   "list",
    {ty_constr=constr_type_list; ty_arity=1;
     ty_desc=Variant_type [constr_nil; constr_cons]};
   "vect",
    {ty_constr=constr_type_vect; ty_arity=1; ty_desc=Abstract_type}
   ]
;;
(* The type "stream" is defined in the "stream" module *)

do_list
  (fun desc -> add module_builtin.mod_constrs desc.qualid.id desc)
  [constr_void; constr_nil; constr_cons; constr_true; constr_false;
   constr_match_failure ]
;;

add module_table "builtin" module_builtin
;;
