;;;-*- Mode: Lisp; Syntax: Common-Lisp; Base: 10; Package: code-identifier -*-
#|
-----------------------------------------------------------------------------------
TITLE: Mapping of Lisp-Identifiers to C-Identifiers
-----------------------------------------------------------------------------------
File:    code-identifier.em
Version: 1.16 (last modification on Fri Feb  4 13:38:55 1994)
State:   published

DESCRIPTION:

DOCUMENTATION:
The following prefixes are used to avoid unnecessary number suffixes:
(This would be the case because
 1. there are different name spaces (symbols, dynamics and lexicals)
 2. deleting '<' and '>' in class names may result in already existing
    identifiers, for example <cons> -> cons
 3. we need identifiers for C-things and for the corresponding Lisp-Objects 
    (especially in the case of functions: C-function and Lisp-function-object)

c_<class-name> : class object
t_<class-name> : type which corresponds to class c_<class-name>
i_<class-name> : an instance of class c_<class-name>
s_<symbol-name>: symbol
f_<function-id>: function object for the function <function-id>

NOTES:

REQUIRES:

PROBLEMS:

AUTHOR:
Ingo Mohr

CONTACT: 
ingo.mohr@isst.fhg.de

HISTORY: 
Log for /export/home/saturn/ukriegel/Dist/Apply/code-identifier.em[1.16]:
  central functions for generating identifiers for C and asm
[1.1] Fri Aug  6 15:55:18 1993 imohr@isst proposed
  [Fri Aug  6 15:18:15 1993] Intention for change:
  identifiers for <tempvar>'s for better debugging
[1.2] Mon Aug  9 09:06:25 1993 imohr@isst saved
  handling of tempvar's now in c-code.em
[1.3] Tue Aug 10 16:11:58 1993 imohr@isst proposed
  minimal changes
[1.4] Tue Aug 17 14:23:16 1993 imohr@isst proposed
  capitalized C-identifiers
[1.5] Thu Aug 26 09:53:56 1993 imohr@isst proposed
  c-code for data
[1.6] Thu Aug 26 10:36:55 1993 imohr@isst proposed
  small letters for local identifiers
[1.7] Mon Aug 30 13:52:14 1993 imohr@isst published
  collection of GC root addresses simplified
  collection of literals ok
[1.8] Fri Sep 17 16:02:21 1993 imohr@isst proposed
  better naming for symbols, generic functions, and others
[1.9] Tue Sep 21 14:54:28 1993 imohr@isst proposed
  identifiers as lists
[1.10] Thu Sep 23 13:55:23 1993 imohr@isst proposed
  improved naming
[1.11] Fri Oct  1 18:47:53 1993 imohr@isst proposed
  [Fri Oct  1 16:50:08 1993] Intention for change:
  + %jmpbuf
[1.12] Wed Oct 13 15:45:23 1993 imohr@isst published
  [Mon Oct 11 09:30:14 1993] Intention for change:
  different naming of generic function objects and function objects
[1.13] Mon Dec 13 11:56:03 1993 imohr@isst proposed
  [Tue Dec  7 13:09:29 1993] Intention for change:
  generation of identifiers concerning h-files
  generation of h-file for modules with export interface
[1.14] Sat Dec 18 15:46:39 1993 imohr@isst proposed
  [Thu Dec 16 16:17:51 1993] Intention for change:
[1.15] Tue Feb  1 16:19:34 1994 imohr@isst proposed
  [Tue Feb  1 16:18:25 1994] Intention for change:
  --- no intent expressed ---
[1.16] Mon Feb  7 08:25:57 1994 imohr@isst published
  [Fri Feb  4 13:37:23 1994] Intention for change:
  imported objects
  new slot access and imported classes ok

-----------------------------------------------------------------------------------
|#

#module code-identifier
(import (level-0-eulisp
         lzs
         accessors
         binding
         lzs-mop
         expand-literal
         tail-module whc-classes
         (only (position-if substitute-if delete-if-not find find-if-not assoc copy-tree
                format substitute
                string-downcase string string-capitalize 
                char alpha-char-p alphanumericp lower-case-p upper-case-p
                mapcar mapc
                copy-seq subseq)
           common-lisp))

 syntax (level-0-eulisp 
         dynamic
         (only (when unless unwind-protect)
           common-lisp))

 export (c-identifier 
         asm-identifier 
         type-identifier 
         local-c-identifier
         reset-code-identifier
         name-global-object
         name-exported-object)
 )

;(export-syntax with-local-identifiers)

;;; -----------------------------------------------------------------------------------
;;; c-identifier, asm-identifier and type-identifier
;;; -----------------------------------------------------------------------------------

(defun c-identifier (object)
  (string
   (or (?code-identifier object)
       (setf (?code-identifier object)
             (make-unique-identifier object 
                                     (make-c-identifier (?identifier object)))))))

(defun asm-identifier (object)
  (format nil "_~A" (c-identifier object)))

(defun type-identifier (class)
  (or (?type-identifier class)
      (progn (c-identifier class)       ; sets type-identifier
             (?type-identifier class))))

(defun local-c-identifier (object)
  (dynamic-let ((*identifier-table* (dynamic *local-identifier-table*))
                (*capitalize* nil))
     (unwind-protect 
       (c-identifier object)
       (dynamic-setq *local-identifier-table*
                     (dynamic *identifier-table*)))))

(defun reset-code-identifier ()
  (dynamic-setq *identifier-table* 
      (mapcar (lambda (id)
                (cons (make-symbol id) 0))
              $reserved-identifiers)))

(defmacro with-local-identifiers forms
  `(dynamic-let ((*local-identifier-table* 
                  (copy-tree (dynamic *identifier-table*))))
      ,@forms))

;;; -----------------------------------------------------------------------------------
;;; 
;;; -----------------------------------------------------------------------------------

(defconstant $generated-identifier-prefix "G")

(defconstant $chars-substituted-by-underscore "_-+$.=*/!?% ")

(defconstant $basic-class-type-ids
  `((,%unsigned-byte-integer . "unsigned char")
    (,%signed-byte-integer . "signed char")
    (,%unsigned-half-integer . "unsigned short")
    (,%signed-half-integer . "signed short")
    (,%unsigned-word-integer . "unsigned long")
    (,%signed-word-integer . "signed long")
    (,%single-float . "float")
    (,%double-float . "double")
    (,%extended-float . "long double")
    (,%string . "char*")
    (,%function . "function") ; explicitely defined for Lisp
    (,%void . "void") ;*8*
    (,%jmpbuf . "jmp_buf")
    (,%pjmpbuf . "jmp_buf*")
    ))

(defconstant $reserved-identifiers
    '(;--- ANSI C
      "auto"
      "break"
      "case"
      "char"
      "const"
      "continue"
      "default"
      "do"
      "double"
      "else"
      "enum"
      "extern"
      "float"
      "for"
      "goto"
      "if"
      "int"
      "long"
      "main"
      "register"
      "return"
      "short"
      "signed"
      "sizeof"
      "static"
      "struct"
      "switch"
      "typedef"
      "union"
      "unsigned"
      "void"
      "volatile"
      "while"
      ; --- standard libraries
      "jmp_buf"
      ; --- GNU C
      "asm"
      "typeof"
      "inline"
      ; --- X ---
      "Object"
      ; --- C-code generation
      "function"
      "G"        ;to avoid 'G' as the first generated identifier
      "NIL"
      "AUX"
      "V_LITERAL"
      "S_LITERAL"
      "LITERAL"
      "SMALL_INT_SKIP"
      "STRUCTURE_ROOTS"
      "VECTOR_ROOTS"
      "VARIABLE_ROOTS"
      "add_to_root_set"
      ))

;;; -----------------------------------------------------------------------------------
;;; Specal Mappings (works only for symbols as identifiers)
;;; -----------------------------------------------------------------------------------
;;; Special mappings for some symbols are needed to avoid generation of
;;; C-identifiers like G17 for functions like + or *. Using special mappings it
;;; is possible

(deflocal *special-identifier-mappings* 
  (mapcar (lambda (pair)
            (cons (make-eulisp-symbol (car pair))
                  (make-eulisp-symbol (cdr pair))))
          '(
            (+ . o-plus)
            (- . o-minus)
            (* . o-times)
            (/ . o-divide)
            (= . o-equal)
            (== . o-identity)
            (!= . o-not-equal)
            (< . o-less)
            (<= . o-less-or-equal)
            (> . o-greater)
            (>= . o-greater-or-equal)
            (% . o-percent)
            (& . o-and)
            ($ . o-dollar)
            (! . o-not)
            (~ . o-negate)
            )))

(defun special-mapping (symbol)
  (let ((entry (assoc symbol *special-identifier-mappings*)))
    (and entry
         (cdr entry))))

;;; -----------------------------------------------------------------------------------
;;; global variables
;;; -----------------------------------------------------------------------------------

(defvar *identifier-table* nil)

(defvar *local-identifier-table* nil)

(defvar *capitalize* t)

(deflocal *max-identifier-size* 40) ; take into account, that a two char prefix
                                    ; or a number suffix are added to identifiers

;;; -----------------------------------------------------------------------------------
;;; Prefixing Identifiers
;;; -----------------------------------------------------------------------------------

(defun replace-prefix (new-prefix-char name)
  (setq name (copy-seq name))
  (setf (char name 0) new-prefix-char)
  name)

(defun add-prefix (prefix-char name)
  (format nil "~A_~A" prefix-char name))

;;; -----------------------------------------------------------------------------------
;;; identifier generation for exported objects
;;; -----------------------------------------------------------------------------------

(defvar *identifier-for-export* nil)

(defun export-identifier (object)
  (dynamic-let ((*identifier-for-export* t))
    (string
     (or (?code-identifier object)
         #|(and (?code-identifier object)
              (check-x-identifier (?code-identifier object)))|#
         (setf (?code-identifier object)
               (make-unique-identifier 
                object 
                (make-x-identifier (?identifier object))))))))

(defun check-x-identifier (id)
  (when (symbolp id) (setq id (symbol-name id)))
  (check-c-unique (check-for-non-c-id-chars id))
  id)

(defgeneric make-x-identifier (identifier))

(defmethod make-x-identifier ((identifier <null>))
  nil)

(defmethod make-x-identifier ((identifier <symbol>))
  (make-x-identifier (symbol-name identifier)))

(defmethod make-x-identifier ((constituent-list <pair>))
  (make-x-identifier (format nil "~{~A~^_~}" constituent-list)))

(defmethod make-x-identifier ((identifier <string>))
  (check-c-unique
   (check-for-non-c-id-chars
    (string-capitalize
     (x-substitute-with-underscore
      (x-delete-leading-and-terminating-nonalpha 
       identifier))))))

(defun x-delete-leading-and-terminating-nonalpha (string)
  (cond ((eql #\% (char string 0))
         (subseq string 1 (length string)))
        ((and (eql #\< (char string 0))
              (eql #\> (char string (- (length string) 1))))
         (subseq string 1 (- (length string) 1)))
        (t string)))

(defun x-substitute-with-underscore (string)
  (substitute #\_ #\- string))

(defun check-for-non-c-id-chars (string)
  (when (find-if-not (lambda (char)
                       (or (alphanumericp char) (eql char #\_)))
                     string)
    (format t "~%!!!!exportierter Id nicht fuer C geeignet: ~A" string))
  string)

(defun check-c-unique (string)
  ;
  ; This function must check a given name if it
  ; 1. it cannot conflict with usual identifiers of C (functions, variables, and
  ;    also macros) 
  ; 2. any prefixing with a one-lowercase-letter&underscore-prefix (e.g. 'c_') 
  ;    doesn't result in name conflicts
  ;see also make-c-unique
  (when (and 
         (> (length string) 2)
         (lower-case-p (char string 0))
         (eql (char string 1) #\_)
         (or
          (upper-case-p (char string 2)) ; all global id's begin with
          ; upper-case
          (eql (char string 2) #\_)))
    (format t "~%!!!!exportierter Id nicht fuer C geeignet: ~A" string))
  string)

;;; -----------------------------------------------------------------------------------
;;; 
;;; -----------------------------------------------------------------------------------

(defgeneric make-c-identifier (identifier))

(defmethod make-c-identifier ((identifier <null>))
  nil)    ;old: $generated-identifier-prefix)

(defmethod make-c-identifier ((identifier <symbol>))
  (make-c-identifier 
   (string-downcase 
    (symbol-name 
     (or (special-mapping identifier) identifier)))))

(defmethod make-c-identifier ((constituent-list <pair>))
  (make-c-identifier (format nil "~{~A~^_~}" constituent-list)))

(defmethod make-c-identifier ((identifier <string>))
  (reduce-length
   (make-c-unique
    (delete-non-c-id-chars
     (substitute-with-underscore
      (delete-leading-and-terminating-nonalpha 
       identifier))))))

(defun reduce-length (string)
  (if (> (length string) *max-identifier-size*)
    (subseq string 0 *max-identifier-size*)
    string))

(defun delete-leading-and-terminating-nonalpha (string)
  (let ((start (position-if #'alpha-char-p string))
        (end (position-if #'alphanumericp string :from-end t)))
    (cond ((null start) $generated-identifier-prefix)
          ((and (= start 0)
                (= (+ end 1) (length string))) 
           string)
          (t (subseq string start (+ end 1))))))

(defun substitute-with-underscore (string)
  (substitute-if #\_ 
                 (lambda (char) (find char $chars-substituted-by-underscore))
                 string))

(defun delete-non-c-id-chars (string)
  (delete-if-not (lambda (char)
                   (or (alphanumericp char) (eql char #\_)))
                 string))

(defun make-c-unique (string)
  ;
  ; This function must transform a given name such that
  ; 1. it cannot conflict with usual identifiers of C (functions, variables, and
  ;    also macros) 
  ; 2. any prefixing with a one-lowercase-letter&underscore-prefix (e.g. 'c_') 
  ;    doesn't result in name conflicts
  ;
  ; The following solution assumes that C-identifiers are consisting of
  ; lowercase letters only (functions, variables) or of uppercase letters only
  ; (macros). Because all Lisp identifiers are capitalized, they cannot start
  ; with a lowercase-letter&underscore-prefix. So, a name conflict can't occur
  ; when prefixing generated identifiers. Only local identifiers, which are not
  ; capitalized, must be checked in a special way.
  ;
  (cond ((dynamic *capitalize*)
         (string-capitalize string))
        ((and 
          (> (length string) 2)
          (lower-case-p (char string 0))
          (eql (char string 1) #\_)
          (or
           (upper-case-p (char string 2)) ; all global id's begin with
                                          ; upper-case
           (eql (char string 2) #\_)))
         (format nil "~A__~A" (char string 0) (subseq string 2)))
        (t string)))

;;; -----------------------------------------------------------------------------------
;;; 
;;; -----------------------------------------------------------------------------------
(defgeneric make-unique-identifier (object name))

(defmethod make-unique-identifier (object name)
  ; the default case: no prefixing or other special handling necessary
  (make-unique-id name))

(defmethod make-unique-identifier (object (name <null>))
  (make-unique-identifier object $generated-identifier-prefix))

(defmethod make-unique-identifier ((inst <literal-instance>) (name <null>))
  (make-unique-id (replace-prefix #\i (c-identifier (?class inst)))))
  
(defmethod make-unique-identifier ((class <defined-class>) name)
  (make-unique-id (add-prefix #\c name)))

(defmethod make-unique-identifier ((class <defined-sym>) name)
  (make-unique-id (add-prefix #\s name)))

(defmethod make-unique-identifier ((method <method-def>) name)
  (make-unique-id 
   (add-prefix #\m 
               (c-identifier (~method-generic-function method)))))

(defmethod make-unique-identifier ((binding <binding>) name)
  (c-identifier (finally-refered-object binding)))

(defun make-unique-id (name)
  (let* ((sym (make-symbol name))
         (entry (assoc sym (dynamic *identifier-table*))))
    (if entry
      (progn 
        (when (dynamic *identifier-for-export*)
          (format t "~%!!!!Namenskonflikt bei export fuer h-File: ~A" name))
        (setf (cdr entry) (+ 1 (cdr entry)))
        (make-unique-id (format nil "~A_~D" name (cdr entry))))
      (progn 
        (dynamic-setq *identifier-table*
                      `((,sym . 0) ,@(dynamic *identifier-table*)))
        name))))

;;; -----------------------------------------------------------------------------------
;;; naming global objects
;;; -----------------------------------------------------------------------------------

(defgeneric name-global-object (obj))

(defmethod name-global-object (obj)
  ; do nothing
  nil)

(defmethod name-global-object ((class <defined-class>))
  (setf (?code-identifier (expand-literal class))
        (c-identifier class))
  (setf (?type-identifier class)
        (replace-prefix #\t (c-identifier class)))
  )

(defmethod name-global-object ((class <imported-class>))
  (setf (?type-identifier class)
        (replace-prefix #\t (c-identifier class)))
  )

(defmethod name-global-object ((class <basic-class-def>))
  (setf (?code-identifier (expand-literal class))
        (c-identifier class))
  (let ((type-id (cdr (assoc class $basic-class-type-ids))))
    (when type-id
      (setf (?type-identifier class) type-id))))

(defmethod name-global-object ((fun <defined-fun>))
  (let ((id (c-identifier fun)))
    (when (?expanded-literal fun)
      (setf (?code-identifier (?expanded-literal fun))
            (add-prefix #\f id)))))

(defmethod name-global-object ((gf <defined-generic-fun>))
  (let ((gf-id (c-identifier gf)))
    (setf (?code-identifier 
           (~generic-function-discriminating-function gf))
          gf-id)
    (when (?expanded-literal gf)
      (setf (?code-identifier (?expanded-literal gf))
            (add-prefix #\f gf-id)))))

(defmethod name-global-object ((sym <defined-sym>))
  (setf (?code-identifier (expand-literal sym))
        (c-identifier sym)))

(defmethod name-global-object ((var <var>))
  (c-identifier var))

(defmethod name-global-object ((const <named-const>))
  (if (fun-p (?value const)) 
    (name-global-object (?value const))
    (c-identifier const)))

(defmethod name-global-object ((object <null>))
  (let ((lit (expand-literal () )))
    (setf (?identifier lit) "empty-list")
    (c-identifier lit)))

;;; -----------------------------------------------------------------------------------
;;; naming exported objects
;;; -----------------------------------------------------------------------------------

(defgeneric name-exported-object (object))

(defmethod name-exported-object (obj)
  ; do nothing
  nil)

(defmethod name-exported-object ((binding <binding>))
  (export-identifier binding) ; sets code-identifier as a side effect
  (setf (?code-identifier (finally-refered-object binding))
        (c-identifier (finally-refered-object binding))))

(defmethod name-exported-object ((module <module>))
  (let ((module-id (?identifier module)))
    (setf (?identifier module) (list ^initialize ^module module-id))
    (export-identifier module)
    (setf (?identifier module) module-id)))

(defmethod name-exported-object ((class <defined-class>))
  (setf (?code-identifier (expand-literal class))
        (export-identifier class))
  (setf (?type-identifier class)
        (replace-prefix #\t (c-identifier class)))
  )

(defmethod name-exported-object ((class <basic-class-def>))
  (setf (?code-identifier (expand-literal class))
        (export-identifier class))
  (let ((type-id (cdr (assoc class $basic-class-type-ids))))
    (when type-id
      (setf (?type-identifier class) type-id))))

(defmethod name-exported-object ((fun <defined-fun>))
  (let ((id (export-identifier fun)))
    (when (?expanded-literal fun)
      (setf (?code-identifier (?expanded-literal fun))
            (add-prefix #\f id)))))

(defmethod name-exported-object ((gf <defined-generic-fun>))
  (let ((gf-id (export-identifier gf)))
    (setf (?code-identifier 
           (~generic-function-discriminating-function gf))
          gf-id)
    (when (?expanded-literal gf)
      (setf (?code-identifier (?expanded-literal gf))
            (add-prefix #\f gf-id)))))

(defmethod name-exported-object ((sym <defined-sym>))
  (setf (?code-identifier (expand-literal sym))
        (export-identifier sym)))

(defmethod name-exported-object ((var <var>))
  (export-identifier var))

(defmethod name-exported-object ((const <named-const>))
  (if (fun-p (?value const))
    (name-exported-object (?value const))
    (export-identifier const)))

#module-end

