/*

   number.c

   Copyright, 1993, Brent Benson.  All Rights Reserved.
   0.4 Revisions Copyright 1994, Joseph N. Wilson.  All Rights Reserved.
   
   Permission to use, copy, and modify this software and its
   documentation is hereby granted only under the following terms and
   conditions.  Both the above copyright notice and this permission
   notice must appear in all copies of the software, derivative works
   or modified version, and both notices must appear in supporting
   documentation.  Users of this software agree to the terms and
   conditions set forth in this notice.

*/

#include <math.h>
#include "number.h"
#include "prim.h"

/* primitives */

static Object odd_p (Object n);
static Object even_p (Object n);
static Object int_zero_p (Object n);
static Object double_zero_p (Object n);
static Object int_positive_p (Object n);
static Object double_positive_p (Object n);
static Object int_negative_p (Object n);
static Object double_negative_p (Object n);
static Object integral_p (Object n);
static Object int_to_double (Object n);
static Object double_to_int (Object n);
static Object int_negative (Object n);
static Object double_negative (Object n);
static Object int_inverse (Object n);
static Object double_inverse (Object n);
static Object binary_int_plus (Object n1, Object n2);
static Object binary_int_minus (Object n1, Object n2);
static Object binary_int_times (Object n1, Object n2);
static Object binary_int_divide (Object n1, Object n2);
static Object binary_double_plus (Object n1, Object n2);
static Object binary_double_minus (Object n1, Object n2);
static Object binary_double_times (Object n1, Object n2);
static Object binary_double_divide (Object n1, Object n2);
static Object binary_less_than (Object n1, Object n2); 
static Object int_sqrt (Object n);
static Object double_sqrt (Object n);
static Object int_abs (Object n);
static Object double_abs (Object n);
static Object int_quotient (Object n1, Object n2);
static Object ash (Object n, Object count);
static Object binary_logand (Object n1, Object n2);
static Object binary_logior (Object n1, Object n2);
static Object double_sin (Object n1);
static Object double_cos (Object n1);
static Object double_atan2 (Object n1, Object n2);



static struct primitive number_prims[] =
{
  {"%odd?", prim_1, odd_p},
  {"%even?", prim_1, even_p},
  {"%int-zero?", prim_1, int_zero_p},
  {"%double-zero?", prim_1, double_zero_p},
  {"%int-positive?", prim_1, int_positive_p},
  {"%double-positive?", prim_1, double_positive_p},
  {"%int-negative?", prim_1, int_negative_p},
  {"%double-negative?", prim_1, double_negative_p},
  {"%integral?", prim_1, integral_p},
  {"%int-to-double", prim_1, int_to_double},
  {"%double-to-int", prim_1, double_to_int},
  {"%int-negative", prim_1, int_negative},
  {"%double-negative", prim_1, double_negative},
  {"%int-inverse", prim_1, int_inverse},
  {"%double-inverse", prim_1, double_inverse},
  {"%binary-int+", prim_2, binary_int_plus},
  {"%binary-int-", prim_2, binary_int_minus},
  {"%binary-int*", prim_2, binary_int_times},
  {"%binary-int/", prim_2, binary_int_divide},
  {"%binary-double+", prim_2, binary_double_plus},
  {"%binary-double-", prim_2, binary_double_minus},
  {"%binary-double*", prim_2, binary_double_times},
  {"%binary-double/", prim_2, binary_double_divide},
  {"%binary-less-than", prim_2, binary_less_than},
  {"%int-sqrt", prim_1, int_sqrt},
  {"%double-sqrt", prim_1, double_sqrt},
  {"%int-abs", prim_1, int_abs},
  {"%double-abs", prim_1, double_abs},
  {"%quotient", prim_2, int_quotient},
  {"%ash", prim_2, ash},
  {"%sin", prim_1, double_sin},
  {"%cos", prim_1, double_cos},
  {"%atan2", prim_2, double_atan2},
  {"%binary-logand", prim_2, binary_logand},
  {"%binary-logior", prim_2, binary_logior},
};

/* function definitions */

void 
init_number_prims (void)
{
  int num;

  num = sizeof (number_prims) / sizeof (struct primitive);
  init_prims (num, number_prims);
}

Object 
make_integer (int i)
{
#ifndef SMALL_OBJECTS
  Object obj;

  obj = allocate_object (sizeof (struct object));
  TYPE (obj) = Integer;
  INTVAL (obj) = i;
  return (obj);
#else
  return (MAKE_INT (i));
#endif
}

Object
make_dfloat (double d)
{
  Object obj;

  obj = allocate_object (sizeof (struct double_float));
  DFLOATTYPE (obj) = DoubleFloat;
  DFLOATVAL (obj) = d;
  return (obj);
}

/* primitives */

static Object 
odd_p (Object n)
{
  if ((INTVAL(n) % 2) == 1)
    {
      return (true_object);
    }
  else
    {
      return (false_object);
    }
}

static Object 
even_p (Object n)
{
  if ((INTVAL(n) % 2) == 0)
    {
      return (true_object);
    }
  else
    {
      return (false_object);
    }
}

static Object 
int_zero_p (Object n)
{
  if (INTVAL(n) == 0)
    {
      return (true_object);
    }
  else
    {
      return (false_object);
    }
}

static Object
double_zero_p (Object n)
{
  if (DFLOATVAL(n) == 0.0)
    {
      return (true_object);
    }
  else
    {
      return (false_object);
    }
}

static Object 
int_positive_p (Object n)
{
  if (INTVAL(n) > 0)
    {
      return (true_object);
    }
  else
    {
      return (false_object);
    }
}

static Object 
double_positive_p (Object n)
{
  if (DFLOATVAL(n) > 0.0)
    {
      return (true_object);
    }
  else
    {
      return (false_object);
    }
}

static Object 
int_negative_p (Object n)
{
  if (INTVAL(n) < 0)
    {
      return (true_object);
    }
  else
    {
      return (false_object);
    }
}

static Object 
double_negative_p (Object n)
{
  if (DFLOATVAL(n) < 0.0)
    {
      return (true_object);
    }
  else
    {
      return (false_object);
    }
}

static Object 
integral_p (Object n)
{
  if (INTEGERP(n))
    {
      return (true_object);
    }
  else
    {
      return (false_object);
    }
}

static Object 
int_to_double (Object n)
{
  return (make_dfloat (INTVAL(n)));
}

static Object 
double_to_int (Object n)
{
  return (make_integer (DFLOATVAL(n)));
}

static Object 
int_negative (Object n)
{
  return (make_integer (-INTVAL(n)));
}

static Object 
double_negative (Object n)
{
  return (make_dfloat (-DFLOATVAL(n)));
}

static Object 
int_inverse (Object n)
{
  return (make_dfloat (1.0/INTVAL(n)));
}

static Object 
double_inverse (Object n)
{
  return (make_dfloat (1/DFLOATVAL(n)));
}

static Object
binary_int_plus (Object n1, Object n2)
{
  return (make_integer (INTVAL(n1) + INTVAL(n2)));
}

static Object 
binary_int_minus (Object n1, Object n2)
{
  return (make_integer (INTVAL(n1) - INTVAL(n2)));
}

static Object 
binary_int_times (Object n1, Object n2)
{
  return (make_integer (INTVAL(n1) * INTVAL(n2)));
}

static Object 
binary_int_divide (Object n1, Object n2)
{
  if ((INTVAL(n1) % INTVAL(n2)) == 0)
    {
      return (make_integer (INTVAL(n1) / INTVAL(n2)));
    }
  else
    {
      return (make_dfloat ((double)INTVAL(n1) / (double)INTVAL(n2)));
    }
}

static Object 
binary_double_plus (Object n1, Object n2)
{
  return (make_dfloat (DFLOATVAL(n1) + DFLOATVAL(n2)));
}

static Object 
binary_double_minus (Object n1, Object n2)
{
  return (make_dfloat (DFLOATVAL(n1) - DFLOATVAL(n2)));
}

static Object 
binary_double_times (Object n1, Object n2)
{
  return (make_dfloat (DFLOATVAL(n1) * DFLOATVAL(n2)));
}

static Object 
binary_double_divide (Object n1, Object n2)
{
  return (make_dfloat (DFLOATVAL(n1) / DFLOATVAL(n2)));
}

static Object 
binary_less_than (Object n1, Object n2)
{
  if (INTEGERP (n1)) {
    if (INTEGERP (n2)) {
      if (INTVAL (n1) < INTVAL (n2)) {
	return (true_object);
      } else {
	return (false_object);
      }
    } else {
      if (INTVAL (n1) < DFLOATVAL (n2)) {
	return (true_object);
      } else {
	return (false_object);
      }
    }
  } else {
    if (INTEGERP (n2)) {
      if (DFLOATVAL (n1) < INTVAL (n2)) {
	return (true_object);
      } else {
	return (false_object);
      }
    } else {
      if (DFLOATVAL (n1) < DFLOATVAL (n2)) {
	return (true_object);
      } else {
	return (false_object);
      }
    }
  }
}

static Object 
int_sqrt (Object n)
{
  double ans;

  ans = sqrt (INTVAL (n));
  if ((ans - floor (ans)) == 0)
    {
      return (make_integer (ans));
    }
  else
    {
      return (make_dfloat (ans));
    }
}

static Object 
double_sqrt (Object n)
{
  return (make_dfloat (sqrt (DFLOATVAL (n))));
}

static Object 
int_abs (Object n)
{
  int val;

  val = INTVAL (n);
  if (val < 0)
    {
      return (make_integer (-val));
    } 
  else
    {
      return (n);
    }
}

static Object 
double_abs (Object n)
{
  return (make_dfloat (fabs (DFLOATVAL (n))));
}

static Object 
int_quotient (Object n1, Object n2)
{
  return (make_integer (INTVAL(n1)/INTVAL(n2)));
}

static Object 
ash (Object n, Object count)
{
    int num;
    num = INTVAL (count);
    return (make_integer ((num > 0) ? (INTVAL (n) << num)
			  : (INTVAL (n) >> -num)));
}

static Object
double_sin (Object n1)
{
    return (make_dfloat (sin (DFLOATVAL (n1))));
}

static Object
double_cos (Object n1)
{
    return (make_dfloat (cos (DFLOATVAL (n1))));
}

static Object
double_atan2 (Object n1, Object n2)
{
    return (make_dfloat (atan2 (DFLOATVAL (n1), DFLOATVAL (n2))));
}

static Object 
binary_logand (Object n1, Object n2)
{
  return (make_integer (INTVAL(n1) & INTVAL(n2)));
}

static Object 
binary_logior (Object n1, Object n2)
{
  return (make_integer (INTVAL(n1) | INTVAL(n2))); 
}
