/*

   env.c

   Copyright, 1993, Brent Benson.  All Rights Reserved.
   0.4 Revisions Copyright 1994, Joseph N. Wilson.  All Rights Reserved.
   
   Permission to use, copy, and modify this software and its
   documentation is hereby granted only under the following terms and
   conditions.  Both the above copyright notice and this permission
   notice must appear in all copies of the software, derivative works
   or modified version, and both notices must appear in supporting
   documentation.  Users of this software agree to the terms and
   conditions set forth in this notice.

*/

#include "env.h"
#include "error.h"
#include "eval.h"
#include "alloc.h"
#include "class.h"

/* the environment */
struct frame *the_env;

/* the top level environment */
#define BIND_ALLOC_CHUNK 4

/* If TOP_LEVEL_SIZE is not a power of two, see change required below */
#define TOP_LEVEL_SIZE 1024
struct binding *top_level_env[TOP_LEVEL_SIZE];

/* local function prototypes */
static struct binding *symbol_binding (Object sym);

/* function definitions */

void 
add_top_level_binding (Object sym, Object val)
{
  struct binding *binding, *old_binding;
  int i;
  unsigned h;
  char *str;
  binding = (struct binding *) allocate_binding ();
  if (PAIRP (sym)) {
      binding->sym = CAR (sym);
      binding->type = eval (SECOND (sym));
  } else {
      binding->sym = sym;
      binding->type = object_class;
  }
  old_binding = symbol_binding_top_level (binding->sym);
  if (old_binding!= NULL) {
      error("Symbol already defined", sym, old_binding->val, val, NULL);
  }
  binding->val = val;

  i = h = 0;
  str = SYMBOLNAME (binding->sym);
  while ( str[i] )
    {
      h += str[i++];
    }
/*
  h = h % TOP_LEVEL_SIZE;
 */

  /* Works only if TOP_LEVEL_SIZE is a power of 2 */
  h &= (TOP_LEVEL_SIZE - 1);

  binding->next = top_level_env[h];
  top_level_env[h] = binding;
}

void
push_scope (void)
{
  struct frame *frame;

  /* push a new frame */
  frame = (struct frame *) allocate_frame ();
  frame->size = 0;
  frame->bindings = NULL;
  frame->next = the_env;
  the_env = frame;
}

void
pop_scope (void)
{
  the_env = the_env->next;
}

void 
add_bindings (Object syms, Object vals)
{
  struct frame *frame;
  struct binding **bindings;
  struct binding *binding;
  int num_bindings, i;
  Object sym_list;

  sym_list = syms;
  num_bindings = 0;
  while (! NULLP (sym_list))
    {
      num_bindings++;
      sym_list = CDR (sym_list);
    }

  frame = the_env;
  frame->bindings = (struct binding **)
    checking_realloc (frame->bindings, 
		      (frame->size + num_bindings) * sizeof (struct binding *));

  for ( i=0 ; i < num_bindings ; ++i )
    {
      if ((!syms) || (!vals))
	{
	  error ("mismatched number of symbols and values", NULL);
	}

      binding = (struct binding *) allocate_binding ();
      binding->sym = CAR (syms);
/* ??? */
      binding->type = object_class;
      binding->val = CAR (vals);
      frame->bindings[i+frame->size] = binding;

      syms = CDR (syms);
      vals = CDR (vals);
    }
  frame->size += num_bindings;
}

void
add_binding (Object sym, Object val)
{
  struct frame *frame;
  struct binding *binding;

  binding = (struct binding *) allocate_binding ();
  if (PAIRP (sym)) {
      binding->sym = CAR (sym);
      binding->type = eval (SECOND (sym));
  } else {
      binding->sym = sym;
      binding->type = object_class;
  }
  binding->val = val;

  frame = the_env;

  if ((frame->size % BIND_ALLOC_CHUNK) == 0)
    {
      frame->bindings = (struct binding **)
	checking_realloc (frame->bindings,
			  (frame->size + BIND_ALLOC_CHUNK) * sizeof (struct binding *));
    }
  frame->bindings[frame->size] = binding;
  frame->size++;
}

/* Change the binding of the symbol in top-most frame.
   Return 1 on success.  If there is no such binding,
   return 0.

   This isn't correct.  It uses symbol_binding() which
   checks *all* bindings of the symbol, not just the
   top level.
*/
int 
change_binding (Object sym, Object val)
{
  struct binding *binding;

  binding = symbol_binding (sym);
  if (! binding )
    {
      return (0);
    }
  else
    {
/*
  if ( ! instance (val, binding->type)) {
 */
	    binding->val = val;
	    return 1;
/*
  } else {
  error("attempt to assign binding of wrong type", sym, val,NULL);
  return 0;
  }
  */
	}
}

Object 
symbol_value (Object sym)
{
    struct binding *binding;
    
    binding = symbol_binding (sym);
    if ( ! binding ) {
	return (NULL);
    }
    return (binding->val);
}

void 
modify_value (Object sym, Object new_val)
{
    struct binding *binding;

    binding = symbol_binding (sym);
    if ( ! binding ) {
	error ("attempt to modify value of unbound symbol", sym, NULL);
    } else if (instance (new_val, binding->type)){
	binding->val = new_val;
    } else {
	error("attempt to assign variable an incompatible object",
	      sym, new_val, NULL);
    }
}

struct frame *
current_env (void)
{
  return (the_env);
}

/* local functions */

static struct binding *
symbol_binding (Object sym)
{
  struct frame *frame;
  struct binding *binding;
  int i;

  frame = the_env;
  while ( frame )
    {
      for ( i=0 ; i < frame->size ; ++i )
	{
	  binding = frame->bindings[i];
	  if (binding->sym == sym)
	    {
	      return (binding);
	    }
	}
      frame = frame->next;
    }
  /* can't find binding in frames, look at top_level */
  return (symbol_binding_top_level (sym));
}

struct binding *
symbol_binding_top_level (Object sym)
{
  struct binding *binding;
  int h, i;
  char *str;

  i = h = 0;
  str = SYMBOLNAME (sym);
  while ( str[i] )
    {
      h += str[i++];
    }
  h = h % TOP_LEVEL_SIZE;

  binding = top_level_env[h];
  while ( binding )
    {
      if (binding->sym == sym)
	{
	  return (binding);
	}
      binding = binding->next;
    }
  return (NULL);
}

/* Unwind the stack of frames until we reach a frame
   with exit_sym as its only binding.  Perform unwind-protect
   cleanups when we find them. */
void 
unwind_to_exit (Object exit_sym)
{
  struct frame *frame;
  Object body, ret;
  int i;

  frame = the_env;
  while ( frame )
    {
	if (frame->bindings) {
	    if (frame->bindings[0]->sym == exit_sym) {
		the_env = frame->next;
		return;
	    }
	    if (frame->bindings[0]->sym == unwind_symbol) {
		body = UNWINDBODY (frame->bindings[0]->val);
		while (! NULLP (body))
		    {
			ret = eval (CAR (body));
			body = CDR (body);
		    }
	    }
	}
	frame = frame->next;
    }
  error ("unwound to end of stack and can't find exit procedure binding", exit_sym, NULL);

}
