/*  (C) Copyright 1990-1992 by Wade L. Hennessey. All rights reserved. */

#include "lisp.h"
#include "closure.h"

#ifdef MIPS
#define NUMBER_OF_REGS_TO_SCAN  9
#endif

#ifdef SPARC
#define NUMBER_OF_REGS_TO_SCAN  1 /* 0 size causes compiler grief */
#endif

#define PAGE_POWER 9

#define BYTES_PER_PAGE (1 << PAGE_POWER)
#define WORDS_PER_PAGE (BYTES_PER_PAGE >> 2)
#define PAGE_OFFSET_MASK (BYTES_PER_PAGE - 1)
#define PAGE_BYTE_OFFSET(ptr) ((int) ptr & PAGE_OFFSET_MASK)
/* #define PAGE_INDEX_MASK (0xFFFFFFFF << PAGE_POWER) */
#define PAGE_TO_ADDRESS(page) ((LP) (first_ptr + (page << PAGE_POWER)))
#define ADDRESS_TO_PAGE(ptr) ((ptr - first_ptr) >> PAGE_POWER)
#define NEXT_PAGE(page) ((page == last_page) ? 0 : (page + 1))
#define ROUND_TO_WORD(x) ((((int) x % 4) == 0) ? x : \
                                    x + (4 - ((int) x & 3)))
#define ROUND_TO_PAGE(x) ((((int) x % BYTES_PER_PAGE) == 0) ? x : \
                          x + (BYTES_PER_PAGE - ((int) x & PAGE_OFFSET_MASK)))

/* All objects must have at least 4 bytes of storage to hold a forwarding
   pointer, even if the length field contains 0. */
#define CORRECT_ZERO_OBJECT_SIZE(x) ((x == 0) ? 4 : x)

#define IN_HEAP(ptr) ((((LP) ptr) >= first_ptr) && \
                      (((LP) ptr) <= frontier_limit_ptr))

#define END_LOCK_LIST -1
#define UNSCANNED     -2
#define CONTIG_LOCK   -3

#define PAGE_LOCKED(page) \
  ((pageinfo[page].generation == next_generation) && \
   (pageinfo[page].next_lock != UNSCANNED))


#define PAGE_UNLOCKED(page) \
  ((pageinfo[page].generation == next_generation) && \
   (pageinfo[page].next_lock == UNSCANNED))

extern LP least_positive_bignum;
extern LP least_negative_bignum;
extern unsigned long w_lib_symbols[],w_app_symbols[];
extern unsigned long key_lib_symbols[],key_app_symbols[];

typedef struct page_info {
  int generation;
  int contig_flag;
  int next_lock;
} PAGE_INFO;

typedef struct symbol_record {
  char *package_name;
  unsigned long *symbols;
  struct symbol_record *next;
} SYMBOL_RECORD;

/* Init these so the debugger can find them. Bletch... */

PAGE_INFO *pageinfo = 0;
LP first_ptr = 0;
LP frontier_ptr = 0;
LP frontier_limit_ptr = 0;

int current_generation = 0;
int next_generation = 0;
int last_page = 0;
int total_pages = 0;
int allocated_pages = 0;
int frontier_page = 0;
int lock_list = 0;

int remaining_page_bytes = 0;
unsigned long *stack_bottom = 0;
unsigned long *stack_top = 0;
int debug_gc = 0;

LP dynamic_frontier_ptr = 0;
LP dynamic_frontier_limit_ptr = 0;
int dynamic_remaining_page_bytes = 0;

int static_total_pages = 0;
LP static_first_ptr = 0;
LP static_frontier_ptr = 0;
LP static_frontier_limit_ptr = 0;
int static_remaining_page_bytes = 0;

/* #define DEBUG_GC (debug_gc != 0) */
#define DEBUG_GC 0

SYMBOL_RECORD *registered_symbols = 0;

int delay_symbol_interns = 1;

int inside_gc_flag = 0;
int print_gc_messages_flag = 0;
int gc_count = 0;

switch_to_static_space()
{
  dynamic_frontier_ptr = frontier_ptr;
  dynamic_frontier_limit_ptr = frontier_limit_ptr;
  dynamic_remaining_page_bytes = remaining_page_bytes;

  frontier_ptr = static_frontier_ptr;
  frontier_limit_ptr = static_frontier_limit_ptr;
  remaining_page_bytes = static_remaining_page_bytes;
}

switch_to_dynamic_space()
{
  static_frontier_ptr = frontier_ptr;
  static_frontier_limit_ptr = frontier_limit_ptr;
  static_remaining_page_bytes = remaining_page_bytes;

  frontier_ptr = dynamic_frontier_ptr;
  frontier_limit_ptr = dynamic_frontier_limit_ptr;
  remaining_page_bytes = dynamic_remaining_page_bytes;
}

int set_gc_messages(n)
     int n;
{
  print_gc_messages_flag = n;
  return(n);
}

int heap_start()
{
  return((int) first_ptr);
}

int heap_frontier()
{
  return((int) frontier_ptr);
}

int heap_frontier_limit()
{
  return((int) frontier_limit_ptr);
}

int heap_page_size()
{
  return(BYTES_PER_PAGE);
}

int total_heap_pages()
{
  return(total_pages);
}

int free_heap_pages()
{
  int count,page,gen;

  count = 0;
  for (page = 0; page <= last_page; page = page + 1) {
    gen = pageinfo[page].generation;
    if ((gen != current_generation) && (gen != next_generation)) {
      count = count + 1;
    }
  }
  return(count);
}

int total_static_pages()
{
  return(static_total_pages);
}

int free_static_bytes()
{
  return(static_frontier_limit_ptr - static_frontier_ptr);
}

terminate_page()
{
  if (DEBUG_GC) {
    printf("terminate_page, frontier: %x, remaining: %d\n",
	   frontier_ptr,remaining_page_bytes);
  }
  if (remaining_page_bytes != 0) {
    *((LPL) frontier_ptr) = TYPE_END_OF_PAGE;
    remaining_page_bytes = 0;
  }
}

allocate_pages(n)
     int n;
{
  int limit_page,start_page,contig_count,i,generation;

  if (frontier_limit_ptr == static_frontier_limit_ptr) {
    printf("\nError: out of static space!\n");
    lisp_debug();
  }
  if ((n + allocated_pages) > (total_pages / 2)) {
    full_gc();
  }
  terminate_page();
  limit_page = frontier_page;
  contig_count = n;
  while (contig_count > 0) {
    generation = pageinfo[frontier_page].generation;
    if ((generation != current_generation) &&
	(generation != next_generation)) {
      if (contig_count == n) {
	start_page = frontier_page;
      }
      contig_count = contig_count - 1;
    } else {
      contig_count = n;
    }
    if (frontier_page == last_page) {
      frontier_page = 0;
      if (contig_count != 0) {
	contig_count = n;
      }
    } else {
      frontier_page = frontier_page + 1;
    }
    if (frontier_page == limit_page) {
      printf("Error: cannot find %d contiguous free pages. Must be out of memory\n",n);
      lisp_debug();
    }
  }
  if (DEBUG_GC) {
    printf("Alloc page, start: %d, len: %d, gen: %d\n",
	   start_page,n,next_generation);
  }
  allocated_pages = allocated_pages + n;
  frontier_ptr = PAGE_TO_ADDRESS(start_page);
  remaining_page_bytes = n * BYTES_PER_PAGE;
  bzero(frontier_ptr,remaining_page_bytes);
  pageinfo[start_page].generation = next_generation;
  pageinfo[start_page].contig_flag = 0;
  pageinfo[start_page].next_lock = UNSCANNED;
  n = n - 1;
  while (n > 0) {
    start_page = start_page + 1;
    pageinfo[start_page].generation  = next_generation;
    pageinfo[start_page].contig_flag = 1;
    pageinfo[start_page].next_lock = UNSCANNED;
    n = n - 1;
  }
}      

/* All memory allocation goes through this function and c_cons.
   However, these are not safe wrt to interrupts.
   */
LP alloc_words_1(num_words, tag, len_field)
     int num_words; int tag; int len_field;
{
  int total_num_bytes;
  LP ptr;

  total_num_bytes = (num_words << 2) + 4;
  if (total_num_bytes > remaining_page_bytes) {
    if (total_num_bytes <= BYTES_PER_PAGE) {
      allocate_pages(1);
    } else {
      allocate_pages((total_num_bytes + BYTES_PER_PAGE - 1) / BYTES_PER_PAGE);
    }
  }
  /* Install header BEFORE incrementing frontier so we don't have
     a pointer to bogus memory  */
  ptr = frontier_ptr + 5;
  HEADER(ptr) = (len_field << 8) + tag;
  frontier_ptr =  frontier_ptr + total_num_bytes;
  remaining_page_bytes = remaining_page_bytes - total_num_bytes;
  return(ptr);
}

/* Save a few instructions consing. Not sure if this is worth
   duplicating the allocation code.  */
LP c_cons(x,y)
  LP x; LP y;  			
{
  LP ptr;

  if (sizeof(struct cons) > remaining_page_bytes) {
    allocate_pages(1);
  }
  /* Install header BEFORE incrementing frontier so we don't have
     a pointer to bogus memory  */
  ptr = frontier_ptr + 5;
  HEADER(ptr) = (2 << 8) + TYPE_CONS;
  LDREF(ptr,CONS,car) = x;
  LDREF(ptr,CONS,cdr) = y;
  frontier_ptr =  frontier_ptr + sizeof(struct cons);
  remaining_page_bytes = remaining_page_bytes - sizeof(struct cons);
  return(ptr);
}
     
LP alloc_words(num_words, tag)
     int num_words; int tag;
{
  return(alloc_words_1(num_words,tag,num_words));
}

LP alloc_memory(num_units, unit_size, tag)
     int num_units; int unit_size; int tag;
{
  LP ptr;
  int num_bits,num_words;

  num_bits = num_units * unit_size;
  num_words = (num_bits >> 5)  + (((num_bits & 31) == 0) ? 0 : 1);
  return(alloc_words_1(num_words,tag,num_units));
}

LP alloc_bits(num_bits, tag)
     int num_bits; int tag;
{
  int num_words;
  num_words = (num_bits >> 5)  + (((num_bits & 31) == 0) ? 0 : 1);
  return(alloc_words_1(num_words,tag,num_bits));
}


LP alloc_bytes(num_bytes, tag)
     int num_bytes; int tag;
{
  int num_words;
  num_words = (num_bytes >> 2) + (((num_bytes & 3) == 0) ? 0 : 1);
  return(alloc_words_1(num_words,tag,num_bytes));
}

LP alloc_shorts(num_shorts, tag)
     int num_shorts; int tag;
{
  int num_words;
  num_words = (num_shorts >> 1) + (((num_shorts & 1) == 0) ? 0 : 1);
  return(alloc_words_1(num_words,tag,num_shorts));
}

LP alloc_doubles(num_doubles, tag)
     int num_doubles; int tag;
{
  return(alloc_words_1(num_doubles << 1,tag,num_doubles));
}

/* Return object size in bytes (NOT including header bytes) */
object_size(ptr)
     LP ptr;
{
  int i;

  switch ((TAG(ptr) & TAG_MASK)) {
  case TYPE_END_OF_PAGE : return(0);
  case TYPE_CONS: return(sizeof(struct cons) - 4);
  case TYPE_PROCEDURE:
    if (HEADER(ptr) == FUNCALLABLE_INSTANCE_HEADER) {
      return(sizeof(struct funcallable_instance) - 4);
    } else {
      return(sizeof(struct procedure) - 4);
    }
  case TYPE_SYMBOL: return(sizeof(struct symbol) - 4);
  case TYPE_LINE_SYMBOL: return(sizeof(struct line_symbol) - 4);
  case TYPE_CHARACTER: return(sizeof(struct character) - 4);
  case TYPE_COMPLEX: return(sizeof(struct complex) - 4);
  case TYPE_RATIO: return(sizeof(struct ratio) - 4);
  case TYPE_FLOAT: return(sizeof(struct double_float) - 4);
  case TYPE_OE: return(LEN_FIELD(ptr) * 4);
  case TYPE_STRUCTURE: return(LEN_FIELD(ptr) * 4);
  case TYPE_BIGNUM: return(ROUND_TO_WORD(LEN_FIELD(ptr)));
  case TYPE_CLOSURE: return(sizeof(CLOSURE));
  case TYPE_FORWARDING_PTR: return(LEN_FIELD(ptr));

  case TYPE_SIMPLE_BIT_VECTOR:
    i = LEN_FIELD(ptr);
    i = (i >> 5) + (((i & 31) == 0) ? 0 : 1); /* words */
    i = i << 2;			              /* bytes */
    return(CORRECT_ZERO_OBJECT_SIZE(i));

  case TYPE_SIMPLE_STRING:
    i = LEN_FIELD(ptr) + 1;	/* account for terminating #\Null */
    i = ROUND_TO_WORD(i);
    return(i);

  case TYPE_SIMPLE_SIGNED_8BIT_VECTOR:
  case TYPE_SIMPLE_UNSIGNED_8BIT_VECTOR:
    i = LEN_FIELD(ptr);
    i = ROUND_TO_WORD(i);
    return(CORRECT_ZERO_OBJECT_SIZE(i));    

  case TYPE_SIMPLE_SIGNED_16BIT_VECTOR:
  case TYPE_SIMPLE_UNSIGNED_16BIT_VECTOR:
    i = LEN_FIELD(ptr) * 2;
    i = ROUND_TO_WORD(i);
    return(CORRECT_ZERO_OBJECT_SIZE(i));

  case TYPE_SIMPLE_SIGNED_32BIT_VECTOR:
  case TYPE_SIMPLE_UNSIGNED_32BIT_VECTOR:
  case TYPE_SIMPLE_VECTOR:
    i = LEN_FIELD(ptr) * 4;
    return(CORRECT_ZERO_OBJECT_SIZE(i));

  case TYPE_SIMPLE_FLOAT_VECTOR:
    i = LEN_FIELD(ptr) * 8;
    return(CORRECT_ZERO_OBJECT_SIZE(i));

  case TYPE_SIMPLE_BIT_MULTI_ARRAY:
  case TYPE_SIMPLE_SIGNED_8BIT_MULTI_ARRAY:
  case TYPE_SIMPLE_UNSIGNED_8BIT_MULTI_ARRAY:
  case TYPE_SIMPLE_CHAR_MULTI_ARRAY:
  case TYPE_SIMPLE_SIGNED_16BIT_MULTI_ARRAY:
  case TYPE_SIMPLE_UNSIGNED_16BIT_MULTI_ARRAY:
  case TYPE_SIMPLE_SIGNED_32BIT_MULTI_ARRAY:
  case TYPE_SIMPLE_UNSIGNED_32BIT_MULTI_ARRAY:
  case TYPE_SIMPLE_PTR_MULTI_ARRAY:
  case TYPE_SIMPLE_FLOAT_MULTI_ARRAY:
    return(sizeof(struct simple_multi_array) - 4);

  case TYPE_COMPLEX_BIT_VECTOR:
  case TYPE_COMPLEX_SIGNED_8BIT_VECTOR:
  case TYPE_COMPLEX_UNSIGNED_8BIT_VECTOR:
  case TYPE_COMPLEX_CHAR_VECTOR:
  case TYPE_COMPLEX_SIGNED_16BIT_VECTOR:
  case TYPE_COMPLEX_UNSIGNED_16BIT_VECTOR:
  case TYPE_COMPLEX_SIGNED_32BIT_VECTOR:
  case TYPE_COMPLEX_UNSIGNED_32BIT_VECTOR:
  case TYPE_COMPLEX_PTR_VECTOR:
  case TYPE_COMPLEX_FLOAT_VECTOR:
    return(sizeof(struct complex_vector) - 4);

  case TYPE_COMPLEX_BIT_MULTI_ARRAY:
  case TYPE_COMPLEX_SIGNED_8BIT_MULTI_ARRAY:
  case TYPE_COMPLEX_UNSIGNED_8BIT_MULTI_ARRAY:
  case TYPE_COMPLEX_CHAR_MULTI_ARRAY:
  case TYPE_COMPLEX_SIGNED_16BIT_MULTI_ARRAY:
  case TYPE_COMPLEX_UNSIGNED_16BIT_MULTI_ARRAY:
  case TYPE_COMPLEX_SIGNED_32BIT_MULTI_ARRAY:
  case TYPE_COMPLEX_UNSIGNED_32BIT_MULTI_ARRAY:
  case TYPE_COMPLEX_PTR_MULTI_ARRAY:
  case TYPE_COMPLEX_FLOAT_MULTI_ARRAY:
    return(sizeof(struct complex_multi_array) - 4);

  case TYPE_FOREIGN_PTR:
    return(sizeof(struct foreign_pointer) - 4);
    
  case TYPE_VOID:
  default: printf("object_size error: unknown tag: %x\n",TAG(ptr));
    lisp_debug();
  }
}

gc_call_count()
{
  return(gc_count);
}

int kbytes_to_pages(kbytes)
     int kbytes;
{
  return((PAGE_POWER > 10) ?
	 (kbytes >> (PAGE_POWER - 10)) :
	 (kbytes << (10 - PAGE_POWER)));
}


init_memory_allocator(dynamic_kbytes,static_kbytes)
     int dynamic_kbytes,static_kbytes;
{
  int i;

  inside_gc_flag = 0;
  print_gc_messages_flag = 0;
  gc_count = 0;
  /* Not sure we should rely on this as a way to get a stack ptr, but... */
  stack_bottom = (unsigned long *) &i;	
  total_pages = kbytes_to_pages(dynamic_kbytes);
  static_total_pages = kbytes_to_pages(static_kbytes);
  current_generation = 1;
  next_generation = 1;
  pageinfo = (struct page_info *)
             malloc(sizeof(struct page_info) * total_pages);
  for (i = 0; i <= total_pages; i++) {
    pageinfo[i].generation = 0;
    pageinfo[i].contig_flag = 0;
    pageinfo[i].next_lock = UNSCANNED;
  }
  first_ptr = (LP) malloc(BYTES_PER_PAGE * (total_pages + 1));
  if ((first_ptr == 0) || (pageinfo == 0)) {
    printf("Cannot allocate enough memory\n");
    /* HEY! Should try to allocate less memory rather than exiting
       Put this in a common safe_malloc routine and use it consistently. */
    exit(1);
  }
  /* Round up to the nearest page */
  first_ptr = ROUND_TO_PAGE(first_ptr);
  last_page = total_pages - 1;
  frontier_limit_ptr = PAGE_TO_ADDRESS(total_pages);
  allocated_pages = 0;
  frontier_page = 0;
  frontier_ptr = 0;
  remaining_page_bytes = 0;

  static_first_ptr = (LP) malloc(BYTES_PER_PAGE * (static_total_pages + 1));
  static_first_ptr = ROUND_TO_PAGE(static_first_ptr);
  static_frontier_limit_ptr = static_first_ptr +
                              (static_total_pages * BYTES_PER_PAGE);
  static_frontier_ptr = static_first_ptr;
  static_remaining_page_bytes = static_total_pages * BYTES_PER_PAGE;
}

char* unknown_tag_name = "unknown tag";

char* tag_name(tag)
     int tag;
{
  switch (tag & TAG_MASK) {
  case TYPE_SIMPLE_BIT_VECTOR:
    return("TYPE_SIMPLE_BIT_VECTOR");
  case TYPE_SIMPLE_SIGNED_8BIT_VECTOR:
    return("TYPE_SIMPLE_SIGNED_8BIT_VECTOR");
  case TYPE_SIMPLE_UNSIGNED_8BIT_VECTOR:
    return("TYPE_SIMPLE_UNSIGNED_8BIT_VECTOR");
  case TYPE_SIMPLE_STRING:
    return("TYPE_SIMPLE_STRING");
  case TYPE_SIMPLE_SIGNED_16BIT_VECTOR:
    return("TYPE_SIMPLE_SIGNED_16BIT_VECTOR");
  case TYPE_SIMPLE_UNSIGNED_16BIT_VECTOR:
    return("TYPE_SIMPLE_UNSIGNED_16BIT_VECTOR");
  case TYPE_SIMPLE_SIGNED_32BIT_VECTOR:
    return("TYPE_SIMPLE_SIGNED_32BIT_VECTOR");
  case TYPE_SIMPLE_UNSIGNED_32BIT_VECTOR:
    return("TYPE_SIMPLE_UNSIGNED_32BIT_VECTOR");
  case TYPE_SIMPLE_VECTOR:
    return("TYPE_SIMPLE_VECTOR");
  case TYPE_SIMPLE_FLOAT_VECTOR:
    return("TYPE_SIMPLE_FLOAT_VECTOR");
  case TYPE_SIMPLE_BIT_MULTI_ARRAY:
    return("TYPE_SIMPLE_BIT_MULTI_ARRAY");
  case TYPE_SIMPLE_SIGNED_8BIT_MULTI_ARRAY:
    return("TYPE_SIMPLE_SIGNED_8BIT_MULTI_ARRAY");
  case TYPE_SIMPLE_UNSIGNED_8BIT_MULTI_ARRAY:
    return("TYPE_SIMPLE_UNSIGNED_8BIT_MULTI_ARRAY");
  case TYPE_SIMPLE_CHAR_MULTI_ARRAY:
    return("TYPE_SIMPLE_CHAR_MULTI_ARRAY");
  case TYPE_SIMPLE_SIGNED_16BIT_MULTI_ARRAY:
    return("TYPE_SIMPLE_SIGNED_16BIT_MULTI_ARRAY");
  case TYPE_SIMPLE_UNSIGNED_16BIT_MULTI_ARRAY:
    return("TYPE_SIMPLE_UNSIGNED_16BIT_MULTI_ARRAY");
  case TYPE_SIMPLE_SIGNED_32BIT_MULTI_ARRAY:
    return("TYPE_SIMPLE_SIGNED_32BIT_MULTI_ARRAY");
  case TYPE_SIMPLE_UNSIGNED_32BIT_MULTI_ARRAY:
    return("TYPE_SIMPLE_UNSIGNED_32BIT_MULTI_ARRAY");
  case TYPE_SIMPLE_PTR_MULTI_ARRAY:
    return("TYPE_SIMPLE_PTR_MULTI_ARRAY");
  case TYPE_SIMPLE_FLOAT_MULTI_ARRAY:
    return("TYPE_SIMPLE_FLOAT_MULTI_ARRAY");
  case TYPE_COMPLEX_BIT_VECTOR:
    return("TYPE_COMPLEX_BIT_VECTOR");
  case TYPE_COMPLEX_SIGNED_8BIT_VECTOR:
    return("TYPE_COMPLEX_SIGNED_8BIT_VECTOR");
  case TYPE_COMPLEX_UNSIGNED_8BIT_VECTOR:
    return("TYPE_COMPLEX_UNSIGNED_8BIT_VECTOR");
  case TYPE_COMPLEX_CHAR_VECTOR:
    return("TYPE_COMPLEX_CHAR_VECTOR");
  case TYPE_COMPLEX_SIGNED_16BIT_VECTOR:
    return("TYPE_COMPLEX_SIGNED_16BIT_VECTOR");
  case TYPE_COMPLEX_UNSIGNED_16BIT_VECTOR:
    return("TYPE_COMPLEX_UNSIGNED_16BIT_VECTOR");
  case TYPE_COMPLEX_SIGNED_32BIT_VECTOR:
    return("TYPE_COMPLEX_SIGNED_32BIT_VECTOR");
  case TYPE_COMPLEX_UNSIGNED_32BIT_VECTOR:
    return("TYPE_COMPLEX_UNSIGNED_32BIT_VECTOR");
  case TYPE_COMPLEX_PTR_VECTOR:
    return("TYPE_COMPLEX_PTR_VECTOR");
  case TYPE_COMPLEX_FLOAT_VECTOR:
    return("TYPE_COMPLEX_FLOAT_VECTOR");
  case TYPE_COMPLEX_BIT_MULTI_ARRAY:
    return("TYPE_COMPLEX_BIT_MULTI_ARRAY");
  case TYPE_COMPLEX_SIGNED_8BIT_MULTI_ARRAY:
    return("TYPE_COMPLEX_SIGNED_8BIT_MULTI_ARRAY");
  case TYPE_COMPLEX_UNSIGNED_8BIT_MULTI_ARRAY:
    return("TYPE_COMPLEX_UNSIGNED_8BIT_MULTI_ARRAY");
  case TYPE_COMPLEX_CHAR_MULTI_ARRAY:
    return("TYPE_COMPLEX_CHAR_MULTI_ARRAY");
  case TYPE_COMPLEX_SIGNED_16BIT_MULTI_ARRAY:
    return("TYPE_COMPLEX_SIGNED_16BIT_MULTI_ARRAY");
  case TYPE_COMPLEX_UNSIGNED_16BIT_MULTI_ARRAY:
    return("TYPE_COMPLEX_UNSIGNED_16BIT_MULTI_ARRAY");
  case TYPE_COMPLEX_SIGNED_32BIT_MULTI_ARRAY:
    return("TYPE_COMPLEX_SIGNED_32BIT_MULTI_ARRAY");
  case TYPE_COMPLEX_UNSIGNED_32BIT_MULTI_ARRAY:
    return("TYPE_COMPLEX_UNSIGNED_32BIT_MULTI_ARRAY");
  case TYPE_COMPLEX_PTR_MULTI_ARRAY:
    return("TYPE_COMPLEX_PTR_MULTI_ARRAY");
  case TYPE_COMPLEX_FLOAT_MULTI_ARRAY:
    return("TYPE_COMPLEX_FLOAT_MULTI_ARRAY");
  case TYPE_BIGNUM:
    return("TYPE_BIGNUM");
  case TYPE_RATIO:
    return("TYPE_RATIO");
  case TYPE_FLOAT:
    return("TYPE_FLOAT");
  case TYPE_COMPLEX:
    return("TYPE_COMPLEX");
  case TYPE_SYMBOL:
    return("TYPE_SYMBOL");
  case TYPE_LINE_SYMBOL:
    return("TYPE_LINE_SYMBOL");
  case TYPE_CHARACTER:
    return("TYPE_CHARACTER");
  case TYPE_PROCEDURE:
    return("TYPE_PROCEDURE");
  case TYPE_CONS:
    return("TYPE_CONS");
  case TYPE_STRUCTURE:
    return("TYPE_STRUCTURE");
  case TYPE_OE:
    return("TYPE_OE");
  case TYPE_VOID:
    return("TYPE_VOID");
  case TYPE_FOREIGN_PTR:
    return("TYPE_FOREIGN_PTR");
  case TYPE_CLOSURE:
    return("TYPE_CLOSURE");
  case TYPE_FORWARDING_PTR:
    return("TYPE_FORWARDING_PTR");
  default: return(unknown_tag_name);
  }
}

void check_memory(start_page, end_page, verbose)
     int start_page;
     int end_page;
     int verbose;
{
  int page,current_page,len,tag;
  char* name;
  LP ptr;
  LP next;
  LP tmp;

  if (pageinfo[start_page].contig_flag != 0) {
    printf("Cannot start memory check on a contiguous page, searching back\n");
    while (pageinfo[start_page].contig_flag != 0) {
      start_page = start_page - 1;
    }
  }
  page = start_page;  
  ptr = PAGE_TO_ADDRESS(page) + 5;
  while ((page <= end_page) &&
	 (verbose || (page != ADDRESS_TO_PAGE(frontier_ptr)))) {
    if (verbose) {
      printf("***** page: %d, gen: %d, contig: %d, next: %d *****\n",
	     page,
	     pageinfo[page].generation,
	     pageinfo[page].contig_flag,
	     pageinfo[page].next_lock);
    }
    if ((pageinfo[page].generation != current_generation) &&
	(pageinfo[page].generation != next_generation)) {
      if (verbose) {
	printf("skipping old generation\n");
      }
      page = NEXT_PAGE(page);
      ptr = PAGE_TO_ADDRESS(page) + 5;
    } else {
      current_page = page;
      while (current_page == page) {
	if (HEADER(ptr) == TYPE_END_OF_PAGE) {
	  if (verbose) {
	    printf("ptr %x, END_OF_PAGE header\n",ptr);
	  }
	  tmp = ptr - 5;
	  ptr = ROUND_TO_PAGE(tmp) + 5;
	  page = page + 1;
	} else {
	  len = object_size(ptr);
	  tag = TAG(ptr);
	  name = tag_name(tag);
	  if (name == unknown_tag_name) {
	    printf("unknown tag in heap\n");
	  }
	  if (verbose) {
	    printf("ptr %x, tag: %s (%x) rest: (%d), len: %d\n",
		   ptr,name,tag,LEN_FIELD(ptr),len);
	  }
	  ptr = ptr + len + 4;
	  tmp = ptr - 5;
	  page = ADDRESS_TO_PAGE(tmp);
	}
      }
    }
  }
}

unsigned long *get_stack_pointer()
{
  int i;
  
  return((unsigned long *) &i);
}

intern_symbols(package_name, symbols)
     LP package_name;
     unsigned long *symbols; 
{
  unsigned long* syms;
  LP package;

  /* HEY! should just use c_to_lisp_string and let MAKE-PACKAGE coerce
     coerce to simple-string */
  package = p_lsp_FIND_2DOR_2DMAKE_2DPACKAGE(1,copy_c_to_lisp_string(package_name));
  for (syms = symbols; *syms != 0; syms = syms + 1) {
    p_lsp_ADD_2DSYMBOL(2,*syms,package);
  }
}

register_symbols(package_name, symbols)
     char *package_name;
     unsigned long *symbols;
{
  SYMBOL_RECORD *ptr;

  ptr = (SYMBOL_RECORD *) malloc(sizeof(SYMBOL_RECORD));
  ptr->package_name = package_name;
  ptr->symbols = symbols;
  if (!delay_symbol_interns) {
   intern_symbols(symbols, package_name);
  }
  ptr->next = registered_symbols;
  registered_symbols = ptr;
}


intern_static_symbols()
{
  SYMBOL_RECORD *ptr;

  for (ptr = registered_symbols; ptr != 0; ptr = ptr->next) {
    intern_symbols(ptr->package_name, ptr->symbols);
  }
  delay_symbol_interns = 0;
}

scan_static_symbols()
{
  SYMBOL_RECORD *ptr;
  unsigned long *syms;

  for (ptr = registered_symbols; ptr != 0; ptr = ptr->next) {
    for (syms = ptr->symbols; *syms != 0; syms = syms + 1) {
      move_sub_objects(*syms);
    }
  }
}

int scan_memory_segment(low,high,segment_name)
     unsigned long *low,*high;
     char* segment_name;
{
  unsigned long *ptr,*heap_ptr;
  int page,page_lock_count;

  page_lock_count = 0;
  ptr = low;
  while (ptr <= high) {
    heap_ptr = (unsigned long *) *ptr;
    if IN_HEAP(heap_ptr) {
      page = ADDRESS_TO_PAGE((LP) heap_ptr);
      if (pageinfo[page].generation == current_generation) {
	/* promote page(s) */
	if ((pageinfo[page].contig_flag != 0) && DEBUG_GC) {
	  printf("Counting locked page back to contig start\n");
	}
	while (pageinfo[page].contig_flag != 0) {
	  page = page - 1;
	  if (pageinfo[page].generation != current_generation) {
	    printf("Internal error - contig gen mismatch");
	  }
	}
	if (DEBUG_GC) {
	  printf("%d ",page);
	}
	pageinfo[page].generation = next_generation;
	page_lock_count = page_lock_count + 1;
	pageinfo[page].next_lock = lock_list;
	lock_list = page;
	/* promote all contig pages, but don't add them to the lock list */
	page = NEXT_PAGE(page);
	while ((pageinfo[page].contig_flag != 0) &&
	       (pageinfo[page].generation == current_generation)) {
	  pageinfo[page].generation = next_generation;
	  pageinfo[page].next_lock = CONTIG_LOCK;
	  page_lock_count = page_lock_count + 1;
	  page = NEXT_PAGE(page);
	}
      }
    }
    ptr = ptr + 1;
  }
  if (print_gc_messages_flag) {
    printf("%d Page locks from %s segment\n",page_lock_count,segment_name);
  }
  allocated_pages = allocated_pages + page_lock_count;
  return(page_lock_count);
}

scan_stack()
{
  int regs[NUMBER_OF_REGS_TO_SCAN];
  unsigned long *low;

  copy_regs_to_stack(regs);
  low = get_stack_pointer();
  scan_memory_segment(low,stack_bottom,"Stack");
}

LP move_object(ptr)
     LP ptr;
{
  int page = ADDRESS_TO_PAGE(ptr);
  if (FIXNUMP(ptr) ||
      !(IN_HEAP(ptr)) ||
      pageinfo[page].generation != current_generation) {
    return(ptr);
  } else {
    if ((TAG(ptr)) == TYPE_FORWARDING_PTR) {
      return((LP) DEREF(ptr));
    } else {
      int size = object_size(ptr);
      LP new_ptr = alloc_bytes(size,TAG(ptr));
      int i;

      for (i = -4; i < size; i = i + 4) {
	DEREF(new_ptr + i) = DEREF(ptr + i);
      }
      HEADER(ptr) = (size << 8) + TYPE_FORWARDING_PTR;
      DEREF(ptr) = (LD) new_ptr;
      return(new_ptr);
    }
  }
}

LP move_object_unless_locked(ptr)
     LP ptr;
{
  int page = ADDRESS_TO_PAGE(ptr);
  
  if (PAGE_LOCKED(page)) {
    return(ptr);	
  } else {
    return(move_object(ptr));
  }
}

scan_misc_static_pointers()
{
  OE = move_object_unless_locked(OE);
  least_positive_bignum =
    REMOVE_TAG(move_object_unless_locked(ADD_TAG(least_positive_bignum)));
  least_negative_bignum =
    REMOVE_TAG(move_object_unless_locked(ADD_TAG(least_negative_bignum)));
}

scan_static_space()
{
  LP ptr;
  ptr = static_first_ptr + 5;
  while (ptr < static_frontier_ptr) {
    move_sub_objects(ptr);
    ptr = ptr + object_size(ptr) + 4;
  }
}

scan_root_set()
{
  lock_list = END_LOCK_LIST;
  scan_stack();
  scan_static_symbols();
  scan_static_space();
  scan_misc_static_pointers();
}

move_sub_objects(ptr)
     LP ptr;
{
  switch (TAG(ptr) & TAG_MASK) {
  case TYPE_FORWARDING_PTR:
    printf("Error to move sub_objects of fwd ptr! \n");
    lisp_debug();

  case TYPE_CONS:
    LDREF(ptr,CONS,car) = move_object(LDREF(ptr,CONS,car));
    LDREF(ptr,CONS,cdr) = move_object(LDREF(ptr,CONS,cdr));
    break;

  case TYPE_PROCEDURE:
    switch (HEADER(ptr)) {
    case FUNCALLABLE_INSTANCE_HEADER:
      LDREF(ptr,FUNCALLABLE_INSTANCE,code_pointer) =
	REMOVE_TAG(move_object(ADD_TAG(LDREF(ptr,FUNCALLABLE_INSTANCE,
					     code_pointer))));
      LDREF(ptr,FUNCALLABLE_INSTANCE,wrapper) =
	move_object(LDREF(ptr,FUNCALLABLE_INSTANCE,wrapper));
      LDREF(ptr,FUNCALLABLE_INSTANCE,slots) =
	move_object(LDREF(ptr,FUNCALLABLE_INSTANCE,slots));
      break;
    case CLOSED_PROCEDURE_HEADER:
      LDREF(ptr,PROCEDURE,code_pointer) =
	REMOVE_TAG(move_object(ADD_TAG(LDREF(ptr,PROCEDURE,code_pointer))));
      break;
    }
    break;  

  case TYPE_CLOSURE:
#ifdef SPARC
    {
        unsigned long hi22,low10,old_oe,new_oe;

	hi22 = ((CLOSURE *) (ptr - 1))->sethi_oe;
	low10 = ((CLOSURE *) (ptr - 1))->add_oe;
	old_oe = (hi22 << 10) + low10;
	new_oe = (unsigned long) move_object(old_oe);
	
	((CLOSURE *) (ptr - 1))->sethi_oe = HI22(new_oe);
	((CLOSURE *) (ptr - 1))->add_oe = LOW10(new_oe);
     }
#else
    fixme;
#endif  
    break;

  case TYPE_COMPLEX:
    LDREF(ptr,COMPLEX,real) = move_object(LDREF(ptr,COMPLEX,real));
    LDREF(ptr,COMPLEX,imaginary) = move_object(LDREF(ptr,COMPLEX,imaginary));
    break;

  case TYPE_RATIO:
    LDREF(ptr,RATIO,numerator) = move_object(LDREF(ptr,RATIO,numerator));
    LDREF(ptr,RATIO,denominator) = move_object(LDREF(ptr,RATIO,denominator));
    break;

  case TYPE_SYMBOL:
    LDREF(ptr,SYMBOL,self_link) = ptr;
    LDREF(ptr,SYMBOL,value) = move_object(LDREF(ptr,SYMBOL,value));
    LDREF(ptr,SYMBOL,package) = move_object(LDREF(ptr,SYMBOL,package));
    LDREF(ptr,SYMBOL,name) = move_object(LDREF(ptr,SYMBOL,name));
    LDREF(ptr,SYMBOL,plist) = move_object(LDREF(ptr,SYMBOL,plist));
    LDREF(ptr,SYMBOL,function) = move_object(LDREF(ptr,SYMBOL,function));
    break;

  case TYPE_LINE_SYMBOL:
    LDREF(ptr,LINE_SYMBOL,self_link) =
      move_object(LDREF(ptr,LINE_SYMBOL,self_link));
    break;

  case TYPE_OE:
  case TYPE_STRUCTURE:
  case TYPE_SIMPLE_VECTOR:
    {
      int i;
      int len = LEN_FIELD(ptr) * 4;
      for (i = 0; i < len; i = i + 4) {
	DEREF(ptr + i) = (LD) move_object(DEREF(ptr + i));
      }
    }
    break;

  case TYPE_CHARACTER:
  case TYPE_FLOAT:
  case TYPE_BIGNUM:
  case TYPE_SIMPLE_BIT_VECTOR:
  case TYPE_SIMPLE_SIGNED_8BIT_VECTOR:
  case TYPE_SIMPLE_UNSIGNED_8BIT_VECTOR:
  case TYPE_SIMPLE_STRING:
  case TYPE_SIMPLE_SIGNED_16BIT_VECTOR:
  case TYPE_SIMPLE_UNSIGNED_16BIT_VECTOR:
  case TYPE_SIMPLE_SIGNED_32BIT_VECTOR:
  case TYPE_SIMPLE_UNSIGNED_32BIT_VECTOR:
  case TYPE_SIMPLE_FLOAT_VECTOR:
    break;


  case TYPE_SIMPLE_BIT_MULTI_ARRAY:
  case TYPE_SIMPLE_SIGNED_8BIT_MULTI_ARRAY:
  case TYPE_SIMPLE_UNSIGNED_8BIT_MULTI_ARRAY:
  case TYPE_SIMPLE_CHAR_MULTI_ARRAY:
  case TYPE_SIMPLE_SIGNED_16BIT_MULTI_ARRAY:
  case TYPE_SIMPLE_UNSIGNED_16BIT_MULTI_ARRAY:
  case TYPE_SIMPLE_SIGNED_32BIT_MULTI_ARRAY:
  case TYPE_SIMPLE_UNSIGNED_32BIT_MULTI_ARRAY:
  case TYPE_SIMPLE_PTR_MULTI_ARRAY:
  case TYPE_SIMPLE_FLOAT_MULTI_ARRAY:
    LDREF(ptr,SIMPLE_MULTI_ARRAY,underlying_vector) =
      move_object(LDREF(ptr,SIMPLE_MULTI_ARRAY,underlying_vector));
    LDREF(ptr,SIMPLE_MULTI_ARRAY,dims_vector) =
      move_object(LDREF(ptr,SIMPLE_MULTI_ARRAY,dims_vector));
    LDREF(ptr,SIMPLE_MULTI_ARRAY,multiplier_vector) =
      move_object(LDREF(ptr,SIMPLE_MULTI_ARRAY,multiplier_vector));
    break;

  case TYPE_COMPLEX_BIT_VECTOR:
  case TYPE_COMPLEX_SIGNED_8BIT_VECTOR:
  case TYPE_COMPLEX_UNSIGNED_8BIT_VECTOR:
  case TYPE_COMPLEX_CHAR_VECTOR:
  case TYPE_COMPLEX_SIGNED_16BIT_VECTOR:
  case TYPE_COMPLEX_UNSIGNED_16BIT_VECTOR:
  case TYPE_COMPLEX_SIGNED_32BIT_VECTOR:
  case TYPE_COMPLEX_UNSIGNED_32BIT_VECTOR:
  case TYPE_COMPLEX_PTR_VECTOR:
  case TYPE_COMPLEX_FLOAT_VECTOR:
    LDREF(ptr,COMPLEX_VECTOR,underlying_vector) =
      move_object(LDREF(ptr,COMPLEX_VECTOR,underlying_vector));
    /* Ignore fill pointer and displaced-index offset */
    break;

  case TYPE_COMPLEX_BIT_MULTI_ARRAY:
  case TYPE_COMPLEX_SIGNED_8BIT_MULTI_ARRAY:
  case TYPE_COMPLEX_UNSIGNED_8BIT_MULTI_ARRAY:
  case TYPE_COMPLEX_CHAR_MULTI_ARRAY:
  case TYPE_COMPLEX_SIGNED_16BIT_MULTI_ARRAY:
  case TYPE_COMPLEX_UNSIGNED_16BIT_MULTI_ARRAY:
  case TYPE_COMPLEX_SIGNED_32BIT_MULTI_ARRAY:
  case TYPE_COMPLEX_UNSIGNED_32BIT_MULTI_ARRAY:
  case TYPE_COMPLEX_PTR_MULTI_ARRAY:
  case TYPE_COMPLEX_FLOAT_MULTI_ARRAY:
    LDREF(ptr,COMPLEX_MULTI_ARRAY,underlying_vector) =
      move_object(LDREF(ptr,COMPLEX_MULTI_ARRAY,underlying_vector));
    LDREF(ptr,COMPLEX_MULTI_ARRAY,dims_vector) =
      move_object(LDREF(ptr,COMPLEX_MULTI_ARRAY,dims_vector));
    LDREF(ptr,COMPLEX_MULTI_ARRAY,multiplier_vector) =
      move_object(LDREF(ptr,COMPLEX_MULTI_ARRAY,multiplier_vector));
    /* Ignore displaced-index offset */
    break;

  case TYPE_FOREIGN_PTR:
    /* DEREF(ptr) = (LD) move_object(DEREF(ptr)); */
    break;

  case TYPE_VOID:
  default:
    printf("move_sub_object error: unknown tag: %x\n",TAG(ptr));
    lisp_debug();
  }
}

/* Return next page, or -1 if done */
int scan_page(page)
     int page;
{
  LP ptr;
  LP end_ptr;
  int next_page;

  if (pageinfo[page].contig_flag == 0) {
    ptr = PAGE_TO_ADDRESS(page) + 5;
  } else {
    int start_page = page;
    while (pageinfo[start_page].contig_flag != 0) {
      start_page = start_page - 1;
    }
    ptr = PAGE_TO_ADDRESS(start_page) + 5;
    ptr = ptr + object_size(ptr) + 4;
  }
  end_ptr = ROUND_TO_PAGE(ptr + object_size(ptr) - 1);
  
  if (DEBUG_GC) {
    printf("scan page %d, ",page);
  }
  /* Scan page and any contig pages */
  while (1) {
    if (ptr == frontier_ptr + 5) {
      return(-1);
    }
    if ((HEADER(ptr) == TYPE_END_OF_PAGE) || (ptr >= end_ptr)) {
      next_page = (NEXT_PAGE(ADDRESS_TO_PAGE(end_ptr - 1)));
      if (next_page > last_page) {
	printf("Internal page scanning error\n");
	lisp_debug();
      } else {
	if (DEBUG_GC) {
	  printf("next page: %d\n",next_page);
	}
	return(next_page);
      }
    }
    move_sub_objects(ptr);
    ptr = ptr + object_size(ptr) + 4;
  }
}

scan_pages(start_page)
     int start_page;
{
  int page;
  LP ptr;

  page = start_page;
  while (lock_list != END_LOCK_LIST) {
    scan_page(lock_list);
    lock_list = pageinfo[lock_list].next_lock;
  }
  while (page != -1) {
    if (PAGE_UNLOCKED(page)) {
      page = scan_page(page);
    } else {
      page = NEXT_PAGE(page);
    }
  }
}

full_gc()
{
  int start_page; 

  if (inside_gc_flag != 0) {
    printf("\nGC called recursively, must be out of memory.\n");
    lisp_debug();
  } 

  inside_gc_flag = 1;    
  if (print_gc_messages_flag) {
    printf("GC called. Frontier page = %d\n",frontier_page);
  }

  terminate_page();
  next_generation = current_generation + 1;
  allocated_pages = 0;
  start_page = frontier_page;

  scan_root_set();
  scan_pages(start_page);
  current_generation = next_generation;
  inside_gc_flag = 0;
  gc_count = gc_count + 1;

  /*
  printf("GC done, checking memory...");
  check_memory(0,last_page,0);
  printf("done\n\n");
  */

}





