/*  (C) Copyright 1990-1992 by Wade L. Hennessey. All rights reserved. */

#include "lisp.h"
#include <math.h>
#include "bignum/BigZ.h"


#define swap(x,y) {LP tmp = x; x = y; y = tmp;}

#ifdef ansi_fixed_args
extern int integer_add(int i, int j);
extern int integer_subtract(int i, int j);
extern int integer_multiply(int i, int j);
#endif

BigZ least_positive_bignum;
BigZ least_negative_bignum;

/* lots of code duplication here because the wretched C macro
   facility is so weak... */

LP new_float(f)
     double f;
{
  LP  result;

  /* Account for header size when allocating float */
  result = alloc_bytes((sizeof(struct double_float) - 4),TYPE_FLOAT);
  RAW_FLOAT(result) = f;
  return(result);
}

/* HEY! passing floats as unknowns (ala printf) is a general problem...*/
int print_double_float(f, stream)
     double f; FILE *stream;
{
  int size,i;
  double frac;
  
  /* Hack to get a single 0 after the decimal point sometimes.
     Isn't there some way to get printf to do this by itself???? */
  frac = modf(f,&i);
  if (frac == 0.0) {
    fprintf(stream,"%#.1f%n",f,&size);
  } else {
    fprintf(stream,"%.16G%n",f,&size);
  }
  /* %n seems to be broken under SunOS, so size is probably incorrect */
  return(size);
}

int double_float_to_string(f, buffer)
     double f; char *buffer;
{
  int size,i;
  double frac;
  
  /* Hack to get a single 0 after the decimal point sometimes.
     Isn't there some way to get printf to do this by itself???? */
  frac = modf(f,&i);
  if (frac == 0.0) {
    sprintf(buffer,"%#.1f%n",f,&size);
  } else {
    sprintf(buffer,"%.16G%n",f,&size);
  }
  return(size);
}

double double_round(x, y)
     double x; double y;
{
  double result;

  result = rint(x / y);
}

int double_truncate (x, y)
     double x; double y;
{
  int i;
  
  i = (x / y);
  return(i);
}

LP bignum_err(x)
     LP x;
{
  printf("Finish bignum arith ops!\n");
}

LP ratio_err(x)
     LP x;
{
  printf("Finish ratio arith ops!\n");
}


LP complex_err(x)
     LP x;
{
  printf("Finish complex arith ops!\n");
}


LP normalize_bignum(z)
     BigZ z;
{
  int i;

  if ((BzNumDigits(z) <= 1) &&
      (BzCompare(z,least_positive_bignum) == BZ_LT) &&
      (BzCompare(z,least_negative_bignum) == BZ_GT)) {
    i = BzToInteger(z);
    return((LP) INT_TO_FX(i));
  } else {
    return(ADD_TAG(z));
  }
}

LP int32_to_bignum(i)
     int i;
{
    BigZ z;
    z = BzFromInteger(i);
    return(ADD_TAG(z));
}

LP uint32_to_bignum(l)
     unsigned long l;
{
    BigZ z;
    z = BzFromUnsignedLong(l);
    return(ADD_TAG(z));
}
     
int bignum_to_int32(b)
     LP b;
{
  int i;
  i = BzToInteger(REMOVE_TAG(b));
  return(i);
}

unsigned long bignum_to_uint32(b)
     LP b;
{
  unsigned long l;
  l = BzToUnsignedLong(REMOVE_TAG(b));
  return(l);
}


LP bignum_add(x, y)
     BigZ x; BigZ y;
{
  BigZ z;

  z = BzAdd(x,y);
  return(normalize_bignum(z));
}

LP bignum_subtract(x, y)
     BigZ x; BigZ y;
{
  BigZ z;

  z = BzSubtract(x,y);
  return(normalize_bignum(z));
}

LP bignum_multiply(x, y)
     BigZ x; BigZ y;
{
  BigZ z;

  z = BzMultiply(x,y);
  return(normalize_bignum(z));
}

LP bignum_divide(x, y)
     BigZ x; BigZ y;
{
  BigZ z;

  z = BzDiv(x,y);
  return(normalize_bignum(z));
}


LP addition_overflow_handler(x, y)
     LP x; LP y;
{
  BigZ bx,by,bz;

  bx = BzFromInteger(FX_TO_INT(x));
  by = BzFromInteger(FX_TO_INT(y));
  bz = BzAdd(bx,by);
  return(normalize_bignum(bz));
}

LP subtraction_overflow_handler(x, y)
     LP x; LP y;
{
  BigZ bx,by,bz;

  bx = BzFromInteger(FX_TO_INT(x));
  by = BzFromInteger(FX_TO_INT(y));
  bz = BzSubtract(bx,by);
  return(normalize_bignum(bz));
}


LP multiply_overflow_handler(x, y)
     int x; LP y;
{
  BigZ bx,by,bz;

  bx = BzFromInteger(x);
  by = BzFromInteger(FX_TO_INT(y));
  bz = BzMultiply(bx,by);
  return(normalize_bignum(bz));
}


LP add(x, y)
     LP x;  LP y;
{
  BigZ bx, by, bz;

  if FIXNUMP(x) { 
    if FIXNUMP(y) {
      return((LP) integer_add((int) x, (int) y));	
    } else {
    fx_first:
      switch (TAG(y)) {
      case TYPE_FLOAT: return(NEW_FLOAT(FX_TO_INT(x) + RAW_FLOAT(y))); 
      case TYPE_BIGNUM:
	bx = BzFromInteger(FX_TO_INT(x));
	return(bignum_add(bx,(BigZ) REMOVE_TAG(y)));
      case TYPE_RATIO: return(ratio_err(y));
      case TYPE_COMPLEX: return(complex_err(y));
      } 
    } 
  } else {
    if FIXNUMP(y) {
      swap(x,y); goto fx_first;
    }  else { 
      switch (TAG(x)) {
      case TYPE_FLOAT:
	switch (TAG(y)) {
	case TYPE_FLOAT: return(NEW_FLOAT(RAW_FLOAT(x) + RAW_FLOAT(y)));
	case TYPE_BIGNUM: float_bignum:
	  return(NEW_FLOAT(RAW_FLOAT(x) + bignum_to_double(y)));
	case TYPE_RATIO: float_ratio: return(ratio_err(y));
	case TYPE_COMPLEX: float_complex: return(complex_err(y));
	}
      case TYPE_BIGNUM:
	switch (TAG(y)) {
	case TYPE_FLOAT: swap(x,y); goto float_bignum;
	case TYPE_BIGNUM: 
	  return(bignum_add((BigZ) REMOVE_TAG(x), (BigZ) REMOVE_TAG(y)));;
	case TYPE_RATIO: bignum_ratio: return(ratio_err(y));
	case TYPE_COMPLEX: bignum_complex: return(complex_err(y));
	}
      case TYPE_RATIO:
	switch (TAG(y)) {
	case TYPE_FLOAT: swap(x,y); goto float_ratio;
	case TYPE_BIGNUM: swap(x,y); goto bignum_ratio;
	case TYPE_RATIO: return(ratio_err(y));
	case TYPE_COMPLEX: ratio_complex: return(complex_err(y));
	}
      case TYPE_COMPLEX:
	swap(x,y);
	switch (TAG(x)) {
	case TYPE_FLOAT: goto float_complex;
	case TYPE_BIGNUM: goto bignum_complex;
	case TYPE_RATIO: goto ratio_complex;
	case TYPE_COMPLEX: return(complex_err(x));
	}
      }
    } 
  } 
  p_lsp_ARITH_2DERROR(2,x,y);
}

LP subtract(x, y)
     LP x;  LP y;
{ 
  BigZ bx, by, bz;

  if FIXNUMP(x) { 
    if FIXNUMP(y) {
      return((LP) integer_subtract((int) x,(int) y));	
    } else {
      switch (TAG(y)) {
      case TYPE_FLOAT: return(NEW_FLOAT(FX_TO_INT(x) - RAW_FLOAT(y))); 
      case TYPE_BIGNUM:
	bx = BzFromInteger(FX_TO_INT(x));
	return(bignum_subtract(bx,(BigZ) REMOVE_TAG(y)));
      case TYPE_RATIO: return(ratio_err(y));
      case TYPE_COMPLEX: return(complex_err(y));
      } 
    } 
  } else {
    if FIXNUMP(y) {
      switch (TAG(x)) {
      case TYPE_FLOAT: return(NEW_FLOAT(RAW_FLOAT(x) - FX_TO_INT(y))); 
      case TYPE_BIGNUM:
	by = BzFromInteger(FX_TO_INT(y));
	return(bignum_subtract((BigZ) REMOVE_TAG(x),by));
      case TYPE_RATIO: return(ratio_err(x));
      case TYPE_COMPLEX: return(complex_err(x));
      } 
    }  else { 
      switch (TAG(x)) {
      case TYPE_FLOAT:
	switch (TAG(y)) {
	case TYPE_FLOAT: return(NEW_FLOAT(RAW_FLOAT(x) - RAW_FLOAT(y)));
	case TYPE_BIGNUM: 
	  return(NEW_FLOAT(RAW_FLOAT(x) - bignum_to_double(y)));
	case TYPE_RATIO: float_ratio: return(ratio_err(y));
	case TYPE_COMPLEX: float_complex: return(complex_err(y));
	}
      case TYPE_BIGNUM:
	switch (TAG(y)) {
	case TYPE_FLOAT: 
	  return(NEW_FLOAT(bignum_to_double(x) - RAW_FLOAT(y)));
	case TYPE_BIGNUM:
	  return(bignum_subtract((BigZ) REMOVE_TAG(x), (BigZ) REMOVE_TAG(y)));
	case TYPE_RATIO: bignum_ratio: return(ratio_err(y));
	case TYPE_COMPLEX: bignum_complex: return(complex_err(y));
	}
      case TYPE_RATIO:
	switch (TAG(y)) {
	case TYPE_FLOAT: return(ratio_err(x));
	case TYPE_BIGNUM: return(ratio_err(x));
	case TYPE_RATIO: return(ratio_err(y));
	case TYPE_COMPLEX: ratio_complex: return(complex_err(y));
	}
      case TYPE_COMPLEX:
	swap(x,y);
	switch (TAG(x)) {
	case TYPE_FLOAT: goto float_complex;
	case TYPE_BIGNUM: goto bignum_complex;
	case TYPE_RATIO: goto ratio_complex;
	case TYPE_COMPLEX: return(complex_err(x));
	}
      }
    } 
  } 
  p_lsp_ARITH_2DERROR(2,x,y);
}

LP multiply(x, y)
     LP x;  LP y;
{ 
  BigZ bx, by, bz;

  if FIXNUMP(x) { 
    if FIXNUMP(y) {
      return((LP) integer_multiply((int) x, (int) y));
    } else {
    fx_first:
      switch (TAG(y)) {
      case TYPE_FLOAT: return(NEW_FLOAT(FX_TO_INT(x) * RAW_FLOAT(y))); 
      case TYPE_BIGNUM:
	bx = BzFromInteger(FX_TO_INT(x));
	return(bignum_multiply(bx,(BigZ) REMOVE_TAG(y)));
      case TYPE_RATIO: return(ratio_err(y));
      case TYPE_COMPLEX: return(complex_err(y));
      } 
    } 
  } else {
    if FIXNUMP(y) {
      swap(x,y); goto fx_first;
    }  else { 
      switch (TAG(x)) {
      case TYPE_FLOAT:
	switch (TAG(y)) {
	case TYPE_FLOAT: return(NEW_FLOAT(RAW_FLOAT(x) * RAW_FLOAT(y)));
	case TYPE_BIGNUM: float_bignum:
	  return(NEW_FLOAT(RAW_FLOAT(x) * bignum_to_double(y)));
	case TYPE_RATIO: float_ratio: return(ratio_err(y));
	case TYPE_COMPLEX: float_complex: return(complex_err(y));
	}
      case TYPE_BIGNUM:
	switch (TAG(y)) {
	case TYPE_FLOAT: swap(x,y); goto float_bignum;
	case TYPE_BIGNUM:
	  return(bignum_multiply((BigZ) REMOVE_TAG(x), (BigZ) REMOVE_TAG(y)));
	case TYPE_RATIO: bignum_ratio: return(ratio_err(y));
	case TYPE_COMPLEX: bignum_complex: return(complex_err(y));
	}
      case TYPE_RATIO:
	switch (TAG(y)) {
	case TYPE_FLOAT: swap(x,y); goto float_ratio;
	case TYPE_BIGNUM: swap(x,y); goto bignum_ratio;
	case TYPE_RATIO: return(ratio_err(y));
	case TYPE_COMPLEX: ratio_complex: return(complex_err(y));
	}
      case TYPE_COMPLEX:
	swap(x,y);
	switch (TAG(x)) {
	case TYPE_FLOAT: goto float_complex;
	case TYPE_BIGNUM: goto bignum_complex;
	case TYPE_RATIO: goto ratio_complex;
	case TYPE_COMPLEX: return(complex_err(x));
	}
      }
    } 
  } 
  p_lsp_ARITH_2DERROR(2,x,y);
}

LP divide(x, y)
     LP x;  LP y;
{ 
  BigZ bx, by, bz;
  int q,r;
  double fx,fy;

  if FIXNUMP(x) { 
    if FIXNUMP(y) {
      r = (int) x % (int) y;
      q = (int) x / (int) y;
      if (r == 0) {
	return((LP) INT_TO_FX(q)); /* post-adjust */
      } else {
	fx = (int) x;
	fy = (int) y;
	return(NEW_FLOAT(fx / fy));
      }
    } else {
    fx_first:
      switch (TAG(y)) {
      case TYPE_FLOAT: return(NEW_FLOAT(FX_TO_INT(x) / RAW_FLOAT(y))); 
      case TYPE_BIGNUM:
	bx = BzFromInteger(FX_TO_INT(x));
	return(bignum_divide(bx,(BigZ) REMOVE_TAG(y)));
      case TYPE_RATIO: return(ratio_err(y));
      case TYPE_COMPLEX: return(complex_err(y));
      } 
    } 
  } else {
    if FIXNUMP(y) {
      switch (TAG(x)) {
      case TYPE_FLOAT: return(NEW_FLOAT(RAW_FLOAT(x) / FX_TO_INT(y))); 
      case TYPE_BIGNUM:
	by = BzFromInteger(FX_TO_INT(y));
	return(bignum_divide((BigZ) REMOVE_TAG(x),by));
      case TYPE_RATIO: return(ratio_err(x));
      case TYPE_COMPLEX: return(complex_err(x));
      } 
    }  else { 
      switch (TAG(x)) {
      case TYPE_FLOAT:
	switch (TAG(y)) {
	case TYPE_FLOAT: return(NEW_FLOAT(RAW_FLOAT(x) / RAW_FLOAT(y)));
	case TYPE_BIGNUM: 
	  return(NEW_FLOAT(RAW_FLOAT(x) / bignum_to_double(y)));
	case TYPE_RATIO: float_ratio: return(ratio_err(y));
	case TYPE_COMPLEX: float_complex: return(complex_err(y));
	}
      case TYPE_BIGNUM:
	switch (TAG(y)) {
	case TYPE_FLOAT:
	  return(NEW_FLOAT(bignum_to_double(x) / RAW_FLOAT(y)));
	case TYPE_BIGNUM: 
	  return(bignum_divide((BigZ) REMOVE_TAG(x), (BigZ) REMOVE_TAG(y)));
	case TYPE_RATIO: bignum_ratio: return(ratio_err(y));
	case TYPE_COMPLEX: bignum_complex: return(complex_err(y));
	}
      case TYPE_RATIO:
	switch (TAG(y)) {
	case TYPE_FLOAT: ratio_err(x);
	case TYPE_BIGNUM: ratio_err(x);
	case TYPE_RATIO: return(ratio_err(y));
	case TYPE_COMPLEX: ratio_complex: return(complex_err(y));
	}
      case TYPE_COMPLEX:
	swap(x,y);
	switch (TAG(x)) {
	case TYPE_FLOAT: goto float_complex;
	case TYPE_BIGNUM: goto bignum_complex;
	case TYPE_RATIO: goto ratio_complex;
	case TYPE_COMPLEX: return(complex_err(x));
	}
      }
    } 
  } 
  p_lsp_ARITH_2DERROR(2,x,y);
}

LP num_equal_p(x, y)  
     LP x; LP y;
{ 
  BigZ bx;
  BigNumCmp flag;

  if FIXNUMP(x) { 
    if FIXNUMP(y) {
      return(((int) x == (int) y) ? T : NIL); 
    } else {
    fx_first:
      switch (TAG(y)) {
      case TYPE_FLOAT: return((FX_TO_INT(x) == RAW_FLOAT(y)) ? T : NIL); 
      case TYPE_BIGNUM:
	bx = BzFromInteger(FX_TO_INT(x));
	flag = BzCompare(bx,(BigZ) REMOVE_TAG(y));
	return((flag == BZ_EQ) ? T : NIL);
      case TYPE_RATIO: return(ratio_err(y));
      case TYPE_COMPLEX: return(complex_err(y));
      } 
    } 
  } else {
    if FIXNUMP(y) {
      swap(x,y); goto fx_first;
    }  else { 
      switch (TAG(x)) {
      case TYPE_FLOAT:
	switch (TAG(y)) {
	case TYPE_FLOAT: return((RAW_FLOAT(x) == RAW_FLOAT(y)) ? T : NIL); 
	case TYPE_BIGNUM: float_bignum:
	  return((RAW_FLOAT(x) == bignum_to_double(y)) ? T : NIL);
	case TYPE_RATIO: float_ratio: return(ratio_err(y));
	case TYPE_COMPLEX: float_complex: return(complex_err(y));
	}
      case TYPE_BIGNUM:
	switch (TAG(y)) {
	case TYPE_FLOAT: swap(x,y); goto float_bignum;
	case TYPE_BIGNUM:
	  flag = BzCompare((BigZ) REMOVE_TAG(x),(BigZ) REMOVE_TAG(y));
	  return((flag == BZ_EQ) ? T : NIL);
	case TYPE_RATIO: bignum_ratio: return(ratio_err(y));
	case TYPE_COMPLEX: bignum_complex: return(complex_err(y));
	}
      case TYPE_RATIO:
	switch (TAG(y)) {
	case TYPE_FLOAT: swap(x,y); goto float_ratio;
	case TYPE_BIGNUM: swap(x,y); goto bignum_ratio;
	case TYPE_RATIO: return(ratio_err(y));
	case TYPE_COMPLEX: ratio_complex: return(complex_err(y));
	}
      case TYPE_COMPLEX:
	swap(x,y);
	switch (TAG(x)) {
	case TYPE_FLOAT: goto float_complex;
	case TYPE_BIGNUM: goto bignum_complex;
	case TYPE_RATIO: goto ratio_complex;
	case TYPE_COMPLEX: return(complex_err(x));
	}
      }
    } 
  } 
  p_lsp_ARITH_2DERROR(2,x,y);
}


LP greaterp(x, y)  
     LP x;  LP y;
{
  BigZ bignum;
  BigNumCmp flag;
 
  if FIXNUMP(x) { 
    if FIXNUMP(y) {
      return(((int) x > (int) y) ? T : NIL); 
    } else {
    fx_first:
      switch (TAG(y)) {
      case TYPE_FLOAT: return((FX_TO_INT(x) > RAW_FLOAT(y)) ? T : NIL); 
      case TYPE_BIGNUM:
	bignum = BzFromInteger(FX_TO_INT(x));
	flag = BzCompare(bignum,(BigZ) REMOVE_TAG(y));
	return((flag == BZ_GT) ? T : NIL);
      case TYPE_RATIO: return(ratio_err(y));
      case TYPE_COMPLEX: return(complex_err(y));
      } 
    } 
  } else {
    if FIXNUMP(y) {
      switch (TAG(x)) {
      case TYPE_FLOAT: return((RAW_FLOAT(x) > FX_TO_INT(y)) ? T : NIL); 
      case TYPE_BIGNUM:
	bignum = BzFromInteger(FX_TO_INT(y));
	flag = BzCompare((BigZ) REMOVE_TAG(x),bignum);
	return((flag == BZ_GT) ? T : NIL);
      case TYPE_RATIO: return(ratio_err(x));
      case TYPE_COMPLEX: return(complex_err(x));
      } 
    }  else { 
      switch (TAG(x)) {
      case TYPE_FLOAT:
	switch (TAG(y)) {
	case TYPE_FLOAT: return((RAW_FLOAT(x) > RAW_FLOAT(y)) ? T : NIL); 
	case TYPE_BIGNUM: 
	  return((RAW_FLOAT(x) > bignum_to_double(y)) ? T : NIL);
	case TYPE_RATIO: float_ratio: return(ratio_err(y));
	case TYPE_COMPLEX: float_complex: return(complex_err(y));
	}
      case TYPE_BIGNUM:
	switch (TAG(y)) {
	case TYPE_FLOAT:
	  return((bignum_to_double(x) > RAW_FLOAT(y)) ? T : NIL);
	case TYPE_BIGNUM: 
	  flag = BzCompare((BigZ) REMOVE_TAG(x),(BigZ) REMOVE_TAG(y));
	  return((flag == BZ_GT) ? T : NIL);
	case TYPE_RATIO: bignum_ratio: return(ratio_err(y));
	case TYPE_COMPLEX: bignum_complex: return(complex_err(y));
	}
      case TYPE_RATIO:
	switch (TAG(y)) {
	case TYPE_FLOAT: swap(x,y); goto float_ratio;
	case TYPE_BIGNUM: swap(x,y); goto bignum_ratio;
	case TYPE_RATIO: return(ratio_err(y));
	case TYPE_COMPLEX: ratio_complex: return(complex_err(y));
	}
      case TYPE_COMPLEX:
	swap(x,y);
	switch (TAG(x)) {
	case TYPE_FLOAT: goto float_complex;
	case TYPE_BIGNUM: goto bignum_complex;
	case TYPE_RATIO: goto ratio_complex;
	case TYPE_COMPLEX: return(complex_err(x));
	}
      }
    } 
  } 
  p_lsp_ARITH_2DERROR(2,x,y);
}

LP geq_p(x, y)  
     LP x;  LP y;
{
  BigZ bignum;
  BigNumCmp flag;
 
  if FIXNUMP(x) { 
    if FIXNUMP(y) {
      return(((int) x >= (int) y) ? T : NIL); 
    } else {
    fx_first:
      switch (TAG(y)) {
      case TYPE_FLOAT: return((FX_TO_INT(x) >= RAW_FLOAT(y)) ? T : NIL); 
      case TYPE_BIGNUM:
	bignum = BzFromInteger(FX_TO_INT(x));
	flag = BzCompare(bignum,(BigZ) REMOVE_TAG(y));
	return(((flag == BZ_GT) || (flag == BZ_EQ)) ? T : NIL);
      case TYPE_RATIO: return(ratio_err(y));
      case TYPE_COMPLEX: return(complex_err(y));
      } 
    } 
  } else {
    if FIXNUMP(y) {
      switch (TAG(x)) {
      case TYPE_FLOAT: return((RAW_FLOAT(x) >= FX_TO_INT(y)) ? T : NIL); 
      case TYPE_BIGNUM:
	bignum = BzFromInteger(FX_TO_INT(y));
	flag = BzCompare((BigZ) REMOVE_TAG(x),bignum);
	return(((flag == BZ_GT) || (flag == BZ_EQ)) ? T : NIL);
      case TYPE_RATIO: return(ratio_err(x));
      case TYPE_COMPLEX: return(complex_err(x));
      } 
    }  else { 
      switch (TAG(x)) {
      case TYPE_FLOAT:
	switch (TAG(y)) {
	case TYPE_FLOAT: return((RAW_FLOAT(x) >= RAW_FLOAT(y)) ? T : NIL); 
	case TYPE_BIGNUM: 
	  return((RAW_FLOAT(x) >= bignum_to_double(y)) ? T : NIL);
	case TYPE_RATIO: float_ratio: return(ratio_err(y));
	case TYPE_COMPLEX: float_complex: return(complex_err(y));
	}
      case TYPE_BIGNUM:
	switch (TAG(y)) {
	case TYPE_FLOAT:
	  return((bignum_to_double(x) >= RAW_FLOAT(y)) ? T : NIL);
	case TYPE_BIGNUM: 
	  flag = BzCompare((BigZ) REMOVE_TAG(x),(BigZ) REMOVE_TAG(y));
	  return(((flag == BZ_GT) || (flag == BZ_EQ)) ? T : NIL);
	case TYPE_RATIO: bignum_ratio: return(ratio_err(y));
	case TYPE_COMPLEX: bignum_complex: return(complex_err(y));
	}
      case TYPE_RATIO:
	switch (TAG(y)) {
	case TYPE_FLOAT: swap(x,y); goto float_ratio;
	case TYPE_BIGNUM: swap(x,y); goto bignum_ratio;
	case TYPE_RATIO: return(ratio_err(y));
	case TYPE_COMPLEX: ratio_complex: return(complex_err(y));
	}
      case TYPE_COMPLEX:
	swap(x,y);
	switch (TAG(x)) {
	case TYPE_FLOAT: goto float_complex;
	case TYPE_BIGNUM: goto bignum_complex;
	case TYPE_RATIO: goto ratio_complex;
	case TYPE_COMPLEX: return(complex_err(x));
	}
      }
    } 
  } 
  p_lsp_ARITH_2DERROR(2,x,y);
}

LP lessp(x, y)  
     LP x;  LP y;
{ 
  BigZ bignum;
  BigNumCmp flag;

  if FIXNUMP(x) { 
    if FIXNUMP(y) {
      return(((int) x < (int) y) ? T : NIL); 
    } else {
    fx_first:
      switch (TAG(y)) {
      case TYPE_FLOAT: return((FX_TO_INT(x) < RAW_FLOAT(y)) ? T : NIL); 
      case TYPE_BIGNUM:	
	bignum = BzFromInteger(FX_TO_INT(x));
	flag = BzCompare(bignum,(BigZ) REMOVE_TAG(y));
	return((flag == BZ_LT) ? T : NIL);
      case TYPE_RATIO: return(ratio_err(y));
      case TYPE_COMPLEX: return(complex_err(y));
      } 
    } 
  } else {
    if FIXNUMP(y) {
      switch (TAG(x)) {
      case TYPE_FLOAT: return((RAW_FLOAT(x) < FX_TO_INT(y)) ? T : NIL); 
      case TYPE_BIGNUM:
	bignum = BzFromInteger(FX_TO_INT(y));
	flag = BzCompare((BigZ) REMOVE_TAG(x),bignum);
	return((flag == BZ_LT) ? T : NIL);
      case TYPE_RATIO: return(ratio_err(x));
      case TYPE_COMPLEX: return(complex_err(x));
      } 
    }  else { 
      switch (TAG(x)) {
      case TYPE_FLOAT:
	switch (TAG(y)) {
	case TYPE_FLOAT: return((RAW_FLOAT(x) < RAW_FLOAT(y)) ? T : NIL); 
	case TYPE_BIGNUM: 
	  return((RAW_FLOAT(x) < bignum_to_double(y)) ? T : NIL);
	case TYPE_RATIO: float_ratio: return(ratio_err(y));
	case TYPE_COMPLEX: float_complex: return(complex_err(y));
	}
      case TYPE_BIGNUM:
	switch (TAG(y)) {
	case TYPE_FLOAT: 
	  return((bignum_to_double(x) < RAW_FLOAT(y)) ? T : NIL);
	case TYPE_BIGNUM: 
	  flag = BzCompare((BigZ) REMOVE_TAG(x),(BigZ) REMOVE_TAG(y));
	  return((flag == BZ_LT) ? T : NIL);
	case TYPE_RATIO: bignum_ratio: return(ratio_err(y));
	case TYPE_COMPLEX: bignum_complex: return(complex_err(y));
	}
      case TYPE_RATIO:
	switch (TAG(y)) {
	case TYPE_FLOAT: swap(x,y); goto float_ratio;
	case TYPE_BIGNUM: swap(x,y); goto bignum_ratio;
	case TYPE_RATIO: return(ratio_err(y));
	case TYPE_COMPLEX: ratio_complex: return(complex_err(y));
	}
      case TYPE_COMPLEX:
	swap(x,y);
	switch (TAG(x)) {
	case TYPE_FLOAT: goto float_complex;
	case TYPE_BIGNUM: goto bignum_complex;
	case TYPE_RATIO: goto ratio_complex;
	case TYPE_COMPLEX: return(complex_err(x));
	}
      }
    } 
  } 
  p_lsp_ARITH_2DERROR(2,x,y);
}

LP leq_p(x, y)  
     LP x;  LP y;
{
  BigZ bignum;
  BigNumCmp flag;
 
  if FIXNUMP(x) { 
    if FIXNUMP(y) {
      return(((int) x <= (int) y) ? T : NIL); 
    } else {
    fx_first:
      switch (TAG(y)) {
      case TYPE_FLOAT: return((FX_TO_INT(x) <= RAW_FLOAT(y)) ? T : NIL); 
      case TYPE_BIGNUM:	
	bignum = BzFromInteger(FX_TO_INT(x));
	flag = BzCompare(bignum,(BigZ) REMOVE_TAG(y));
	return(((flag == BZ_LT) || (flag == BZ_EQ)) ? T : NIL);
      case TYPE_RATIO: return(ratio_err(y));
      case TYPE_COMPLEX: return(complex_err(y));
      } 
    } 
  } else {
    if FIXNUMP(y) {
      switch (TAG(x)) {
      case TYPE_FLOAT: return((RAW_FLOAT(x) <= FX_TO_INT(y)) ? T : NIL); 
      case TYPE_BIGNUM:
	bignum = BzFromInteger(FX_TO_INT(y));
	flag = BzCompare((BigZ) REMOVE_TAG(x),bignum);
	return(((flag == BZ_LT) || (flag == BZ_EQ)) ? T : NIL);
      case TYPE_RATIO: return(ratio_err(x));
      case TYPE_COMPLEX: return(complex_err(x));
      } 
    }  else { 
      switch (TAG(x)) {
      case TYPE_FLOAT:
	switch (TAG(y)) {
	case TYPE_FLOAT: return((RAW_FLOAT(x) <= RAW_FLOAT(y)) ? T : NIL); 
	case TYPE_BIGNUM:
	  return((RAW_FLOAT(x) <= bignum_to_double(y)) ? T : NIL);
	case TYPE_RATIO: float_ratio: return(ratio_err(y));
	case TYPE_COMPLEX: float_complex: return(complex_err(y));
	}
      case TYPE_BIGNUM:
	switch (TAG(y)) {
	case TYPE_FLOAT: 
	  return((bignum_to_double(x) <= RAW_FLOAT(y)) ? T : NIL);
	case TYPE_BIGNUM:
	  flag = BzCompare((BigZ) REMOVE_TAG(x),(BigZ) REMOVE_TAG(y));
	  return(((flag == BZ_LT) || (flag == BZ_EQ)) ? T : NIL);
	case TYPE_RATIO: bignum_ratio: return(ratio_err(y));
	case TYPE_COMPLEX: bignum_complex: return(complex_err(y));
	}
      case TYPE_RATIO:
	switch (TAG(y)) {
	case TYPE_FLOAT: swap(x,y); goto float_ratio;
	case TYPE_BIGNUM: swap(x,y); goto bignum_ratio;
	case TYPE_RATIO: return(ratio_err(y));
	case TYPE_COMPLEX: ratio_complex: return(complex_err(y));
	}
      case TYPE_COMPLEX:
	swap(x,y);
	switch (TAG(x)) {
	case TYPE_FLOAT: goto float_complex;
	case TYPE_BIGNUM: goto bignum_complex;
	case TYPE_RATIO: goto ratio_complex;
	case TYPE_COMPLEX: return(complex_err(x));
	}
      }
    } 
  } 
  p_lsp_ARITH_2DERROR(2,x,y);
}

LP c_eql(x, y)
     LP x; LP y;
{
  if (x == y) {			/* works for chars  + fixnums */
    return(T);
  } else {
    if (OTHER_PTRP(x) && OTHER_PTRP(y) &&
	((TAG(x) & 3) == 1) && ((TAG(y) & 3) == 1)) {
      return(num_equal_p(x,y));
    }
  }
  return(NIL);
}

double float_significand(f)
     double f;
{
  int exponent;
  
  return(frexp(f,&exponent));
}

int float_exponent(f)
     double f;
{
  int exponent;
  
  frexp(f,&exponent);
  return(exponent);
}

int_length(n)
     int n;
{
  int len; 
  for (len = 0; n > 0; len = len + 1) {
    n = n >> 1;
  }
  return(len);
}


init_arith()
{
  least_positive_bignum = BzFromInteger(MOST_POSITIVE_FIXNUM + 1);
  least_negative_bignum = BzFromInteger(MOST_NEGATIVE_FIXNUM - 1);
}

