/* 
Copyright (C) 1993 by Roger Sheldon

This file is part of the Lily C++ Library.  This library is free
software; you can redistribute it and/or modify it under the terms of
the GNU Library General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.  This library is distributed in the hope
that it will be useful, but WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU Library General Public License for more details.
You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the Free Software
Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "function.h"
#include "integer.h"
#include "real.h"
#include "cons.h"
#include <string.h>
#include <stdio.h>
#include <strstream.h>

// IMPORTANT:  Currently, pipe symbols (eg |foo bar|) and strings are
// treated as Symbols.  In the future more data types may be added to
// support pipe symbols and strings.  If so, maybe the easiest way
// to do it would be to derive from the Symbol class and simply
// override the Print method so it wraps the symbol with
// |'s for pipe symbols or "'s for strings.  Here's how
// a string class for Lily might look:

// class String : public Symbol {
// public:
//  ostream& Print(ostream& s) { return s << '\"' << value << '\"'; }
//  LObject_type Type() { return type_String; }
// };

// BNF handled by yyparse():
//
// ... TBD (for now, the file modis.vax illustrates most possibilities)

////////////////////////////////////////////////////////////////////// gnu hole

#ifdef __GNUG__

static inline int strcmpi_aux(char a, char b) {
	if (a == b)
		return 1;
	else if ('A' <= a && a <= 'Z' && a+32 == b)
		return 1;
	else if ('a' <= a && a <= 'z' && a-32 == b)
		return 1;
	else
		return 0;
}

static int strcmpi(char *s, char *t) {
	for (; strcmpi_aux(*s, *t); s++, t++)
		if (*s == '\0')
			return 0;
	return *s - *t;
}

#endif	// __GNUG__

/////////////////////////////////// local data structures and static variables

static enum token { L_PAREN,
                    R_PAREN,
                    NIL_ATOM,
                    SYMBOL,
                    INTEGER,
                    REAL,
                    EOF_TOK,    // just EOF would probably clash with stdio
                    UNKNOWN };

static const int    BUF_SIZE = 64;      // Limits # chars in a SYMBOL or INTEGER
static char         yytext[BUF_SIZE];   // yylex() copies chars into yytext
static int          yypos;              // where next char will go in yytext
static int          inPipeSymbol = 0;   // true if parsing a symbol
                                        // enclosed in pipes (eg: |foo bar|)
static int          inString = 0;       // true if parsing a string


//////////////////////////////////// Inline Character Classification Functions

static inline int is_digit(char c) { return '0'<=c && c<='9'; }

// The scanner is extremely forgiving since is_char() is simply defined
// as anything besides a digit.

static inline int is_char(char c) { return ! is_digit(c); }

static inline int is_white_space(char c) {
    return c == ' ' || c == '\t' || c == '\0' || c == '\n';
}

///////////////////////////////////////////////// Utility macros and functions

// Define AddChar(c), a convenience macro since adding a character
// to yytext[] is so common.  (It has to be a macro since it
// conditionally returns.)

#define AddChar(c)                  \
    {   if (yypos == BUF_SIZE) {    \
            DEretLObject(tok);       \
        }                           \
        else                        \
            yytext[yypos++] = c;    \
    }

// IMPORTANT: Immediately after calling next_black_char(s), you should
// check if (s.eof()).  (A better way would be to return a special char
// which signifies eof, but I don't know what that special char should be.)

static char next_black_char(istream &s) {   // return next non-whitespace char
    char c;
    while (!s.eof()) {
        s.get(c);
        if (!is_white_space(c))
            return c;
    }
    return c;                               // caller will find s.eof() true
}

////////////////////////////////////////////////////////////////////// yylex()

// yylex() scans the input for the next token.  After the next token is
// found, yylex() returns the token id -- the calling function can find
// the corresponding string in yytext, a global char array.
// yylex() is independent of
// Lily and could be reused elsewhere.  Note that yylex() always terminates
// the string in yytext with '\0' (unless yytext won't be used
// by yyparse as is the case for L_PAREN and R_PAREN).


static token yylex(istream &s) {        // Lexical analyzer for Lily
    token   tok = UNKNOWN;
    char    c;                          // next char read from input stream

    yypos = 0;                          // Start filling yytext from beginning
    s >> ws;                            // Ignore white space
    while (1) {
        if (s.eof()) {                  // EOF? It's ok unless you're in the
            if (inPipeSymbol) {         // middle of a pipe symbol or string
                cerr << "INPUT ERROR: unmatched '|'\n";
                return UNKNOWN;
            }
            else if (inString) {
                cerr << "INPUT ERROR: unmatched '\"'\n";
                return UNKNOWN;
            }
            else if (tok == UNKNOWN) {
                return EOF_TOK;
            }
            else {
                AddChar('\0');
                return tok;
            }
        }
        s.get(c);                       // Read char from input into c

        // Now analyze the current character with respect to what we
        // think the token is going to be.  For example, given "12.34", you
        // think it's going to be an INTEGER until you hit the '.', and
        // then it becomes a REAL.

            // Anything is OK inside a pipe symbol or string.

        if ((c != '|' && inPipeSymbol) || (c != '\"' && inString)) {
            AddChar(c);
        }

            // White space terminates current token (we know we're not in
            // the midst of a pipe symbol or string).

        else if (is_white_space(c)) {
            AddChar('\0');
            if (tok == SYMBOL && !strcmpi(yytext, "nil"))   // nil or NIL ?
                return NIL_ATOM;
            else
                return tok;
        }

            // Start a list

        else if (c == '(') {
            if (tok!=UNKNOWN) {         // Put c back for next call to yylex()
                s.putback(c);
                AddChar('\0');
                if (tok == SYMBOL && !strcmpi(yytext, "nil"))   // nil or NIL ?
                    return NIL_ATOM;
                else
                    return tok;
            }
            else {              // Is next char ')' ?, If so, return NIL_ATOM
                c = next_black_char(s);
                if (s.eof()) {
                    cerr << "INPUT ERROR: unmatched '('\n";
                    return UNKNOWN;
                }
                else if (c == ')')
                    return NIL_ATOM;
                else {
                    s.putback(c);
                    return L_PAREN;
                }
            }
        }

            // End a list

        else if (c == ')') {
            if (tok!=UNKNOWN) {         // Put c back for next call to yylex()
                s.putback(c);
                AddChar('\0');
                if (tok==SYMBOL && !strcmpi(yytext, "nil")) // nil or NIL ?
                    return NIL_ATOM;
                else
                    return tok;
            }
            else
                return R_PAREN;
        }

            // Start/End a pipe symbol

        else if (c == '|') {
            if (inPipeSymbol) {
                inPipeSymbol = 0;
                AddChar(c);
                AddChar('\0');
                return SYMBOL;
            }
            else if (tok != UNKNOWN) {
                cerr << "INPUT ERROR: unexpected '|'\n";
                return UNKNOWN;
            }
            else {
                AddChar(c);
                inPipeSymbol = 1;
                tok = SYMBOL;
            }
        }

            // Start/End a string

        else if (c == '\"') {
            if (inString) {
                inString = 0;
                AddChar(c);
                AddChar('\0');
                return SYMBOL;
            }
            else if (tok != UNKNOWN) {
                cerr << "INPUT ERROR: unexpected '\"'\n";
                return UNKNOWN;
            }
            else {
                AddChar(c);
                inString = 1;
                tok = SYMBOL;
            }
        }

            // Build a number (eg 1, -1, 2.3, .04, -.04, -0.33e21) or
            // a symbol (eg heat34).

        else if (is_digit(c)) {
            // If a '-' was the last char, then tok needs to be changed
            // from SYMBOL to INTEGER
            if (yypos==0 || (yypos==1 && yytext[0]=='-')) {
                tok = INTEGER;
            }
            // Check for REAL with no whole number digits (eg -.3, -.04, .2)
            else if ((yypos==1 && yytext[0]=='.')
                        || (yypos==2 && yytext[0]=='-' && yytext[1]=='.')) {
                tok = REAL;
            }
            AddChar(c);
        }

            // Treat something like "2.3.4" as a symbol

        else if (tok==REAL && c=='.') {
            tok = SYMBOL;
            AddChar(c);
        }

            // Check for decimal point to trigger REAL from INTEGER

        else if (tok==INTEGER && c=='.') {
            tok = REAL;
            AddChar(c);
        }

            // Check for exponent notation

        else if (tok==REAL && c=='e') {
            AddChar(c);
        }

            // Add char to current token

        else if (is_char(c)) {
            if (tok!=SYMBOL && tok!=UNKNOWN)
                return UNKNOWN;
            else {
                tok = SYMBOL;
                AddChar(c);
            }
        }

			// Interpret ~ as EOF.  I added this because I'm using Turbo C++
			// for Windows which doesn't seem to take ctrl-d or ctrl-z for EOF.

		else if (c=='~')
			return EOF_TOK;

            // Don't know how to handle current character, return UNKNOWN

        else 
            return UNKNOWN;             // Parsing error
    }   // end of while (1)
    DE; // shouldn't reach here
    return UNKNOWN;
}

//////////////////////////////////////////////////////////////////// yyparse()
// yyparse() can parse list structures of arbitrary nestedness (is that a
// word?)  It's not trivial.  Even with my liberal commenting, I suppose
// it's pretty unreadable.  You've really got to know the guts of
// Lily inside-out to make sense of this.  I wrote it and I even have trouble.



static LObject yyparse(istream &s) {
    Base        *T=nil.value,       // Stack of last cells for each sublist
                *L=nil.value,       // Top-level list returned (if not atom)
                *temp=nil.value;
    int         Parens = 0;         // Used to balance parentheses
    int         NextCar = 1;        // True if next object will become
                                    // the car of the current cell; false
                                    // if the car of the current cell has
                                    // already been filled.

        // The trick to ref maintenance here is to make sure each
        // object gets 1 ref except the 1st cell of a list which gets 0.
        // (The LObject returned will ref the 1st cell.)
        // Note that references are ignored for T cells.

    while (1) {
        token tok = yylex(s);           // Fetch token

        switch(tok) {                   // Parse tokens returned from yylex()

        case L_PAREN:
            if (T==nil.value) {         // 1st '(', initialize T
                temp = new Cons;        // will be ref'ed by T = new Cons...
                L = temp;               // L will ultimately be returned
                T = new Cons(*temp);    // Ignore refs for T cells
                NextCar = 1;
            }
            else if (NextCar) {         // Down
                T->Car().Rplaca(*new Cons); // Rplaca refs new cell
                T = new Cons(T->Car().Car(), *T);// Ignore refs for new T cell
                T->Car().Deref();       // Remove unwanted ref from push
            }
            else {                      // Right & down
                // This is only case where 2 cells are added to T at one time
#ifdef __TURBOC__
                // Given ((a) (b)) as input, this is what happens
                // when       ^
                // the third LPAREN is encountered (only in Turbo C++):
                //
                // before Rplacd, T = (((a)))
                // after Rplacd, T = (((a) NIL)
                // I think it should be T = (((a) (NIL))
                //
                // cerr << "before Rplacd, T = ";
                // T->Print();
                // cerr << "after Rplacd, T = ";
                // T->Print();
                // cerr << "\n";
                //
                // Apparently I was correct -- I don't know why Turbo C++
                // doesn't work in this isolated situation -- for example,
                // the following program:
                //      LObject a = *new Cons(*new Cons);
                //      cout << a << "\n";
                // correctly prints:
                //      (NIL)
                //
                // (I'm not positive it's the fault of Turbo C++ -- I hope
                // it's not some unfindable bug of mine.)
                //
                Base *a1 = new Cons;        // a1 & a2 are hacks for Turbo C++
                Base *a2 = new Cons(*a1);
                T->Car().Rplacd(*a2);
#else
                T->Car().Rplacd(*new Cons(*new Cons));
#endif


                temp = T;
                T = new Cons(T->Car().Cdr(), T->Cdr()); // Pop & Push
                T->Car().Deref();       // Remove unwanted ref from push
                delete temp;            // Delete popped cell
                T = new Cons(T->Car().Car(), *T);   // Push 2nd
                T->Car().Deref();       // Remove unwanted ref from push
                NextCar = 1;
            }
            Parens++;
            break;
        case R_PAREN:
            if (!--Parens) {            // Outermost closing paren found
                DET(& T->Cdr() != nil.value);// Internal error if cdr(T) != nil
                delete T;               // Delete final T cell
                L->refs--;              // Decr 1st cell's refs.  The object
                return *L;              // created when returning will incr refs
            }
            temp = T;
            T = & T->Cdr();             // Up a level
            delete temp;                // Delete popped cell
            break;
        case SYMBOL:                    // Handle atoms
        case INTEGER:
        case REAL:
        case NIL_ATOM:
            switch(tok) {
                case SYMBOL:
                    if (!strcmpi("t", yytext))  // is yytext 't' or 'T' ?
                        temp = t.value;
                    else
                        temp = assign_sym_or_func(yytext);
                    break;
                case INTEGER:
                    temp = new Integer(atol(yytext));
                    break;
                case REAL:
                    temp = new Real(atof(yytext));
                    break;
                case NIL_ATOM:
                    temp = nil.value;
                    break;
                default:
                    DE;                 // internal error
                    break;
            }

            if (T == nil.value)         // Atom
                return *temp;           // Implicit constructor will inc refs
            if (NextCar) {              // Down
                T->Car().Rplaca(*temp); // Rplaca refs the atom
                NextCar = 0;
            }
            else {                      // Right
#ifdef __TURBOC__
                Base *a1 = new Cons(*temp);
                T->Car().Rplacd(*a1);
#else
                T->Car().Rplacd(*new Cons(*temp));  // atom ref'ed by 'new Cons'
#endif
                temp = T;
                T = new Cons(T->Car().Cdr(), T->Cdr()); // Pop & Push
                T->Car().Deref();       // Remove unwanted ref from push
                delete temp;            // Delete popped cell
            }
            break;
        case EOF_TOK:
            return LilyEOF;
            break;
        case UNKNOWN:                   // Parse error
            cerr << "Cannot parse input, returning nil\n";
            return nil;
            break;
        default:                        // yylex() goofed
            DE;
            return nil;
        }   // end of switch(tok)

    }   // end of while(1)
    DE; // shouldn't reach here
    return nil;
}

///////////////////////////////////////////////////////////// public Functions

istream & operator >> (istream &s, LObject &a) { a = yyparse(s); return s; }

LObject      read() { LObject a; cin >> a; return a; }
LObject      read(istream &s) { LObject a; s >> a; return a; }

LObject      read_from_string(char *s)   
            { istrstream is(s, strlen(s)); LObject a; is >> a; return a; }
