/* 
Copyright (C) 1993 by Roger Sheldon

This file is part of the Lily C++ Library.  This library is free
software; you can redistribute it and/or modify it under the terms of
the GNU Library General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.  This library is distributed in the hope
that it will be useful, but WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU Library General Public License for more details.
You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the Free Software
Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#ifndef object_h
#define object_h

// Lily conditional-compilation options:
//
// WARN_NTH         if set, nth(n, list) prints warning when n > length(list)
//
// COUNTOBJS        for debugging; obj counts printed to FILE tracefp;
//                  final count should be 0; execution is much slower when on
//
// INLINE           control whether friend functions are compiled inline
//                  It's difficult to get this to work.  It seemed that
//					you had to compile everything inline to get
//					anything to compile inline.  I suggest you
//					don't try it unless performance is critical.
// 					This option hasn't been tested and probably doesn't work.

#define WARN_NTH

#define COUNTOBJS

// To enable inlining do:
//#define INLINE inline
// otherwise do:
#define INLINE
#define NOINLINE

#include "dbg.h"        // includes <stream.h>

#include <stdio.h>

extern FILE *tracefp;	// defined in global.h, used to trace reference counting

typedef enum {      // New classes must register their type here
    type_null,		// This will go away if object-typing is added to C++
    type_Symbol,
    type_Integer,
	type_Real,
    type_Function,
    type_Cons
} LObject_type;

// Forward declarations for some classes

class Base;
class LObject;
extern LObject LastArg;

///////////////////////////////////////////////////////////////////// VAR_ARGS

// Simulate LISP's &rest variable arguments.  Also used for user Functions.
// User Functions look like: 
//      LObject foo(VAR_ARGS) { LObject *arg[]={ARG_LIST}; ... }
const int MAX_ARGS = 5;

#define VAR_ARGS \
            LObject &x0=LastArg, \
            LObject &x1=LastArg, \
            LObject &x2=LastArg, \
            LObject &x3=LastArg, \
            LObject &x4=LastArg

    // Stuff for user Functions

#define ARG_LIST &x0,&x1,&x2,&x3,&x4
#define CALL_ARGS x0,x1,x2,x3,x4

    // Stuff for optimized Functions:

#define ARG_LIST2 x0.value,x1.value,x2.value,x3.value,x4.value
#define CALL_ARGS2 *x0.value,*x1.value,*x2.value,*x3.value,*x4.value
#define ARGSM1 *arg[0],*arg[1],*arg[2],*arg[3]

typedef LObject (*FctPtr)(VAR_ARGS); // typedef for a function pointer
typedef char *CharPtr;		// Used in function map code (function.h, etc.)

typedef LObject (*LilyFunction)(LObject &);

////////////////////////////////////////////////////////////////////// LObject

class LObject {
#define FRIEND friend	// see friend.h to see what this does
#include "friend.h"
#undef VAR_ARGS
#define VAR_ARGS \
            LObject &x0, \
            LObject &x1, \
            LObject &x2, \
            LObject &x3, \
            LObject &x4
    Base *value;
		// All Lisp object classes must be friend in order to access 
		// an object's 'value' data member.
    friend class Base;
    friend class Cons;
    friend class Null;
    friend class Symbol;
    friend class Function;
    friend class Integer;
    friend class Long;
    friend class Real;
    friend LObject yyparse(istream &);	// in read.c, this is the Lily parser
public:
	// Constructors
    LObject();
    LObject(LObject &);
    LObject(Base &);
    LObject(char *);     // defined in Symbol.c
    LObject(int);        // conversion constructor used in logical expressions
                        // defined in Integer.c
    LObject(float);      // conversion constructor defined in real.c
    ~LObject();
	// Assignment
    void operator=(LObject &);
	// Stream I/O
    friend ostream& operator << (ostream& s, LObject &a);
    friend istream& operator >> (istream& s, LObject &a); // defined in read.c
	// Logical operations
    operator == (LObject &a);
    operator != (LObject &a);
    operator int();     // conversion operator for logical operations
	// Math operations
    friend LObject operator + (LObject &, LObject &);
    friend LObject operator - (LObject &, LObject &);
    friend LObject operator * (LObject &, LObject &);
    friend LObject operator / (LObject &, LObject &);
    void operator +=(LObject &);
    void operator -=(LObject &);
};

extern LObject nil, t, LilyEOF;

//////////////////////////////////////////////////////////////////////// Base

class Base {            // new Lisp classes must be derived from Base
    friend class LObject;
    friend class Cons;
    friend class Function;  // temporary
    friend LObject yyparse(istream &);
protected:
    short   refs;       // number of references to this object
public:
#ifdef COUNTOBJS
    static int count;   // for debugging; should be 0 objects when all done
#endif
    Base()									{ refs = 0; }
//  void    pr_refs();  // for debugging
    virtual ~Base()							{ }
    virtual operator    int();  // conversion for logical expressions

    virtual Base &      Assoc(Base &a);//      { DE; }
    virtual Base &      Atom()              { return *t.value; }
    virtual Base &      Car()               { DE; return *nil.value; }
    virtual Base &      Cdr()               { DE; return *nil.value; }
    virtual Base &      Copy()              { DE; return *nil.value; }
    virtual Base &      Copy_list()         { DE; return *nil.value; }
    virtual Base &      Copy_tree()         { return *this; }
    virtual void        Decr(Base &a)       { DE; }
    virtual void        Deref();
    virtual Base &      Equal(Base &a)      { DE; return *nil.value; }
    virtual LObject     Funcall(VAR_ARGS)   { DE; return nil; }
    virtual void        Incr(Base &a)       { DE; }
    virtual int         Integer_value()     { DE; return 1; }
    virtual Base &      Last()              { DE; return *nil.value; }
    virtual Base &      Length()            { DE; return *nil.value; }
    virtual Base &      Mapcar(LilyFunction f) { DE; return *nil.value; }
    virtual Base &      Mapcar(FctPtr)
                                            { DE; return *nil.value; }
    virtual Base &      Mapcar(FctPtr , Base &)
                                            { DE; return *nil.value; }
    virtual Base &      Member(Base &a)     { DE; return *nil.value; }
    virtual Base &      Nth(Base &n)        { DE; return *nil.value; }
    virtual Base &      Null_()             { return *nil.value; }
    virtual Base &      Numberp()           { return *nil.value; }
    virtual ostream &   Print(ostream& s=cout)  { DE; return s; }
    virtual Base &      Push(Base &a)       { DE; return *nil.value; }
    virtual float       Real_value()        { DE; return 1.0; }
    virtual Base *      Ref();
    virtual Base &      Reverse()           { DE; return *nil.value; }
    virtual Base &      Reverse1()          { DE; return *nil.value; }
    virtual Base &      Rplaca(Base &a)     { DE; return *nil.value; }
    virtual Base &      Rplacd(Base &a)     { DE; return *nil.value; }
    virtual Base &      Set_difference(Base &l2) { DE; return *nil.value; }
    virtual Base &      Setq(Base &a)       { DE; return *nil.value; }
    virtual FctPtr     Symbol_Function()   { DE; return 0; }
    virtual char *      Symbol_name()       { DE; return 0; }
    virtual LObject_type Type()              { DE; return 0; }
    virtual Base &      Typep(LObject_type a){ DE; return *nil.value; }
    // Now swap VAR_ARGS with VAR_ARGS_DECL to avoid compiler error for
    // redefining default arguments.
#undef VAR_ARGS
#define VAR_ARGS \
          LObject &x0=LastArg, \
          LObject &x1=LastArg, \
          LObject &x2=LastArg, \
          LObject &x3=LastArg, \
          LObject &x4=LastArg
#define VAR_ARGS_DECL \
          LObject &x0, \
          LObject &x1, \
          LObject &x2, \
          LObject &x3, \
          LObject &x4
};

/////////////////////////////////////////////////////////// Lisp-Style Macros

// The dolist macro has the limitation that you cannot have more than
// one instance of dolist for the same variable within a given scope.  This
// is because dolist creates an internal variable.

#define dolist(elt, list) 								\
	for (LObject list_##elt=list, elt=car(list_##elt); 		\
		list_##elt != nil; 								\
		list_##elt=cdr(list_##elt), elt=car(list_##elt))


//////////////////////////////////////////////////////////////////////// or()

// You can't use functions for or() because functions unconditionally evaluate 
// all arguments.  This implementation of or() isn't pretty but it does the job.
// Nested or()'s won't work.

extern LObject ORA;  	// final answer to or(), defined in global.c
extern LObject ORSTOP;	// controls evaluation of arguments to or()
void OR_RESET();    	// defined in object.c
LObject OR(LObject &a);	// defined in object.c

// The following or() macro uses the comma operator and conditional expressions

#define or(x0,x1) (OR_RESET(), OR(x0), (ORSTOP)?nil:OR(x1), ORA)

#define or3(x0,x1,x2) \
    (OR_RESET(),OR(x0),(ORSTOP)?0:OR(x1),(ORSTOP)?0:OR(x2),ORA)

#endif  // #ifndef object_h
