/* 
Copyright (C) 1993 by Roger Sheldon

This file is part of the Lily C++ Library.  This library is free
software; you can redistribute it and/or modify it under the terms of
the GNU Library General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.  This library is distributed in the hope
that it will be useful, but WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU Library General Public License for more details.
You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the Free Software
Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

// Lily Interpreter

#include "interp.h"

#define static static

static LObject 
    micro_bind_and_assign (LObject &key_list, LObject &value_list, LObject &alist),
    micro_get_value       (LObject &key, LObject &alist),
    micro_assign_value    (LObject &variable, LObject &value, LObject &alist),
    micro_eval_cond       (LObject &clauses, LObject &env),
    micro_eval            (LObject &form, LObject &env),
    micro_apply           (LObject &proc, LObject &args, LObject &env);

static LObject
    M_COND  ("cond"),
    M_CLOSE ("close"),
    M_DEFUN ("defun"),
    M_EXIT  ("exit"),
    M_QUOTE ("quote"),
    M_SETQ  ("setq"),

    M_ATOM  ("atom"),
    M_CAR   ("car"),
    M_CDR   ("cdr"),
    M_CONS  ("cons"),
    M_EQUAL ("equal"),
    M_LAMBDA("lambda"),
    M_TIMES ("*"),
    M_NULL  ("null");

static LObject micro_eval_lambda1(VAR_ARGS) { return micro_eval(x0, x1); }
static LObject func_micro_eval_lambda1 = 
        defun("micro_eval_lambda1", micro_eval_lambda1);

static LObject micro_bind_and_assign(LObject &key_list, LObject &value_list, LObject &alist) {
    if (!key_list || !value_list)
        return alist;
    else
        return cons(list(car(key_list), car(value_list)),
                    micro_bind_and_assign(cdr(key_list),cdr(value_list),alist));
}

static LObject micro_get_value(LObject &key, LObject &alist) {
    return cadr(assoc(key, alist));
}

static LObject micro_assign_value(LObject &variable, LObject &value, LObject &alist) {
    LObject entry = assoc(variable, alist);
    if (entry)
        rplaca(cdr(entry), value);
    else
        rplacd(last(alist), list(list(variable, value)));
    return value;
}

static LObject micro_eval_cond(LObject &clauses, LObject &env) {
    if (!clauses)
        return nil;
    else if (micro_eval(caar(clauses), env))
        return micro_eval(cadar(clauses), env);
    else
        return micro_eval_cond(cdr(clauses), env);
}

static LObject micro_eval(LObject &form, LObject &env) {
    if (atom(form))
        if (numberp(form))
            return form;
        else
            return micro_get_value(form, env);
    else if (car(form) == M_QUOTE)
        return cadr(form);
    else if (car(form) == M_COND)
        return micro_eval_cond(cdr(form), env);
    else if (car(form) == M_CLOSE)
        return list(M_CLOSE, cadr(cadr(form)), caddr(cadr(form)), env);
    else if (car(form) == M_DEFUN)
        return micro_assign_value(cadr(form), cons(M_LAMBDA, cddr(form)), env);
    else if (car(form) == M_SETQ)
        return micro_assign_value(cadr(form),
                        micro_eval(caddr(form), env), env);
    else
        return micro_apply(car(form), 
            mapcar_lambda(func_micro_eval_lambda1, cdr(form), env), env);
}

static LObject micro_apply(LObject &proc, LObject &args, LObject &env) {
    if (atom(proc))
        if (proc == M_ATOM) return atom(car(args));
        else if (proc == M_CAR) return caar(args);
        else if (proc == M_CDR) return cdar(args);
        else if (proc == M_CONS) return cons(car(args), cadr(args));
        else if (proc == M_EQUAL) return equal(car(args), cadr(args));
        else if (proc == M_TIMES) return product(car(args),cadr(args));
        else if (proc == M_NULL) return null(car(args));
        else
            return micro_apply(micro_eval(proc, env), args, env);
    else if (car(proc) == M_LAMBDA)
        return micro_eval(caddr(proc),
                micro_bind_and_assign(cadr(proc), args, env));
    else if (car(proc) == M_CLOSE)
        return micro_eval(caddr(proc),
                          micro_bind_and_assign(cadr(proc),args,cadddr(proc)));
    else
        return nil;
}

void micro_read_eval_print() {
    LObject env = list(list(t, t), list(nil, nil));
    while (t) {
        printf("Lily> ");
        fflush(stdout);
        LObject input = read();
        if (input == list(M_EXIT))
            break;
        print(micro_eval(input, env));
        printf("\n");
    }
}
