/* 
Copyright (C) 1993 by Roger Sheldon

This file is part of the Lily C++ Library.  This library is free
software; you can redistribute it and/or modify it under the terms of
the GNU Library General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.  This library is distributed in the hope
that it will be useful, but WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU Library General Public License for more details.
You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the Free Software
Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdarg.h>
#include "cons.h"
#include "integer.h"    // Needed for Cons::Length(), I really don't want this
                        // dependency.

// This file includes the Null and Cons classes for representing Lisp lists.
// See Steele's "Common Lisp" for details on how the functions behave.

INLINE LObject   cons(LObject &a, LObject &b) 
    { return *new Cons(*a.value, *b.value); }

/////////////////////////////////////////////////////////////////////////// Null

INLINE Null::Null() {
    _car = _cdr = this;
#ifdef COUNTOBJS
    fprintf(tracefp,
        "count++ = %d, new Null is %d\n", ++count, (int)this);
#endif
    DETexit(count_calls++ > 0);
}

INLINE Null::~Null() {
#ifdef COUNTOBJS
    fprintf(tracefp,"count-- = %d, Null\n", --count);
#endif
}

INLINE Base &   Null::Atom()                { return *t.value; }
INLINE Base &   Null::Copy()                { return *nil.value; }
INLINE Base &   Null::Copy_list()           { return *nil.value; }
INLINE Base &   Null::Copy_tree()           { return *nil.value; }
INLINE void     Null::Deref()               { }
INLINE Base &   Null::Equal(Base &a)        
                { return (&a == this) ? *t.value : *nil.value; }
INLINE Base &   Null::Mapcar(LilyFunction)  { return *nil.value; }
INLINE Base &   Null::Mapcar(FctPtr)       { return *nil.value; }
INLINE Base &   Null::Mapcar(FctPtr , Base &) { return *nil.value; }
INLINE Base &   Null::Null_()               { return *t.value; }
INLINE ostream& Null::Print(ostream &s)     { return s << "NIL"; }
INLINE Base &   Null::Push(Base &a)         { return *new Cons(a); }//correct?
INLINE Base *   Null::Ref()                 { return this; }
INLINE Base &   Null::Reverse()             { return *nil.value; }
INLINE Base &   Null::Reverse1()            { return *nil.value; }
INLINE Base &   Null::Rplaca(Base &a)       { DE; return *nil.value; }
INLINE Base &   Null::Rplacd(Base &a)       { DE; return *nil.value; }
INLINE Base &   Null::Setq(Base &a)         { DE; return *nil.value; }
INLINE LObject_type  Null::Type()            { return type_null; }
INLINE Base &   Null::Typep(LObject_type type)   
    { return (type == type_null || type == type_Cons) ? *t.value : *nil.value;}

////////////////////////////////////////////////////////////////// Constructors

INLINE Cons::Cons() {
    _car = nil.value;
    _cdr = nil.value;
#ifdef COUNTOBJS
    fprintf(tracefp,
        "Cons::Cons(), count++ = %d, new Cons is %d\n", 
        ++count, (int)this);
#endif
}

INLINE Cons::Cons(Base &a) {
    _car = a.Ref();
    _cdr = nil.value;
#ifdef COUNTOBJS
    fprintf(tracefp,
        "Cons::Cons(LObject&), count++ = %d, new Cons is %d\n", 
            ++count, (int)this);
#endif
}

INLINE Cons::Cons(Base &a, Base&b) {
    _car = a.Ref();
    _cdr = b.Ref();
#ifdef COUNTOBJS
    fprintf(tracefp,
        "count++ = %d, new Cons is %d\n", ++count, (int)this);
#endif
}

INLINE Cons::~Cons() {
#ifdef COUNTOBJS
    fprintf(tracefp,
        "count-- = %d, Cons is %d\n", --count, (int)this);
#endif
}

//////////////////////////////////////////////////////////////////////// Methods

INLINE Base &Cons::Assoc(Base &x) {
    for (Base *a=this; a!=nil.value; a=&a->Cdr())
        if (&x.Equal(a->Car().Car()) != nil.value)
            return a->Car();
    return *nil.value;
}

INLINE Base & Cons::Atom() { return *nil.value; }

INLINE Base & Cons::Copy() { return Copy_list(); }

INLINE Base & Cons::Copy_list() {      // Copy only the first level of the list.
    Base *first = new Cons(*_car);
    Base *prev = first;
    Base *l = _cdr;
    while (l != nil.value) {
        Base *temp = new Cons(l->Car());
        prev->Rplacd(*temp);
        prev = temp;
        l = &l->Cdr();
    }
    return *first;
}

INLINE Base & Cons::Copy_tree() {        // Copy tree recursively
    Base *first = new Cons(*_car);
    Base *prev = first;
    Base *l = _cdr;
    while (l != nil.value) {
        Base *temp = new Cons(l->Car().Copy_tree());
        prev->Rplacd(*temp);
        prev = temp;
        l = &l->Cdr();
    }
    return *first;
}

INLINE void Cons::Deref() {
    Base *a = this;
    while (a!=nil.value && --a->refs==0) {  // need to add check for dotted pair
        a->Car().Deref();
        Base *save = &a->Cdr();
        delete a;   // May be deleting *this*, so don't use *this*.
        a = save;
    }
}

INLINE Base & Cons::Equal(Base &a) {
    if (a.Type() != type_Cons)
        return *nil.value;
    for (Base *l=this,*b=&a; l!=nil.value && b!=nil.value ; 
                                l=&l->Cdr(),b=&b->Cdr())
        if (&(l->Car().Equal(b->Car())) == nil.value)
            return *nil.value;
    return (l==nil.value && b==nil.value) ? *t.value : *nil.value;
}

INLINE Base & Cons::Last() {
    for (Base *a=this; &a->Cdr() != nil.value && 
            &a->Cdr().Typep(type_Cons) != nil.value; a=&a->Cdr())
        ;
    return *a;
}

INLINE Base & Cons::Length() {
    int length = 0;
    for (Base *a=this; a!=nil.value && a->Typep(type_Cons); a=&a->Cdr())
        length++;
    return *new Integer(length);
}

INLINE Base & Cons::Mapcar(LilyFunction f) {    // see also Null::Mapcar()
    LObject func_ans = f(Car());
    Base *first = new Cons(*func_ans.value);
    Base *prev = first;
    for (Base *l=& Cdr(); l != nil.value; l=&l->Cdr()) {
        func_ans = f(l->Car());
        prev = & prev->Rplacd(*new Cons(*func_ans.value));
    }
    return *first;
}

INLINE Base & Cons::Mapcar(FctPtr f) { // see also Null::Mapcar()
    LObject func_ans = f(Car());
    Base *first = new Cons(*func_ans.value);
    Base *prev = first;
    for (Base *l=& Cdr(); l != nil.value; l=&l->Cdr()) {
        func_ans = f(l->Car());
        prev = & prev->Rplacd(*new Cons(*func_ans.value));
    }
    return *first;
}

INLINE Base & Cons::Mapcar(FctPtr f, Base &a2) {   // see also Null::Mapcar()
    LObject func_ans = f(Car(), a2.Car());
    Base *first = new Cons(*func_ans.value);
    Base *prev = first;
    Base *l1, *l2;
    for (   l1=& Cdr(), l2=&a2.Cdr();
            l1 != nil.value && l2 != nil.value;
            l1=&l1->Cdr(), l2=&l2->Cdr())
            {
        func_ans = f(l1->Car(), l2->Car());
        prev = & prev->Rplacd(*new Cons(*func_ans.value));
    }
    return *first;
}

INLINE Base & Cons::Member(Base &a) {       // pg. 275 Steele
    for (Base *l=this; l!=nil.value; l=&l->Cdr())
        if (&a.Equal(l->Car()) != nil.value)
            return *l;
    return *nil.value;
}

INLINE Base & Cons::Nth(Base &_n) {         // pg. 265 Steele
    int     n = _n.Integer_value();
    Base *  prev = this;
    Base *  l = & Cdr();
    int     i = 0;
    for (; l!=nil.value && i<n; prev=l, l=&l->Cdr(), i++)
        ;

#ifdef WARN_NTH // control whether (nth n '()) prints warning or not
    if (!prev)
        cerr << "WARNING: nth(" << n << ", " << (LObject)*this
             << ") returning nil since n < length of list\n";
#endif
    return prev->Car();
}

// PPrint is a pretty printer

INLINE ostream& Cons::PPrint(ostream& s, int level) {
    if (level > 0) {
        s << "\n";
        for (int i=0; i<level; i++)
            s << " ";
    }
    if (&Car() == nil.value && &Cdr() == nil.value)
        return s << "NIL";
    if (this == nil.value) {
        s << "NIL";
        return s;
    }
    s << "(";
    for (Base *a=this; a!=nil.value && a->Typep(type_Cons); a=&a->Cdr()) {
        if (a->Car().Typep(type_Cons)) {
            Cons *temp = (Cons *) & a->Car();
            temp->PPrint(s, level+1);
        }
        else
            a->Car().Print(s);
        if (&a->Cdr() != nil.value)
            if (&a->Cdr().Typep(type_Cons) == nil.value) {// dotted pair ?
                s << " . "; 
                a->Cdr().Print(s);
            }
        else
            s << " ";
    }
//  return s << ')';    // I don't know why this screws things up.
    s << ")";
    return s;
}

INLINE ostream& Cons::Print(ostream& s) {
#ifdef PPRINT
    return PPrint(s, 0);
#else
    s << "(";
    for (Base *a=this; a!=nil.value && a->Typep(type_Cons); a=&a->Cdr()) {
        a->Car().Print(s);
        if (&a->Cdr() != nil.value)
            if (&a->Cdr().Typep(type_Cons) == nil.value) {// dotted pair ?
                s << " . "; 
                a->Cdr().Print(s);
            }
        else
            s << " ";
    }
    if (a != nil.value) {
        s << " . ";
        a->Print(s);
    }
//  return s << ')';    // I don't know why this screws things up.
    s << ")";
    return s;
#endif  // PPRINT
}

INLINE Base & Cons::Push(Base &a) {
    Base *b = new Cons(Car(), Cdr());
    Rplaca(a);
    Rplacd(*b);
    return *this;
}

INLINE Base & Cons::Reverse1() { // in-place reverse; reverses all sublists too
    Base *l = new Cons(*_car, *_cdr);
    l->refs++;
    _car->refs--; 
    _cdr->refs--;
    Base *p = nil.value;
    while (l!=nil.value) {
        if (&l->Car().Typep(type_Cons) != nil.value)
            l->Car().Reverse1();
        Base *n=&l->Cdr();
        ((Cons*)l)->_cdr = p;
        p = l;
        l = n;
    }
    _car = &p->Car();
    _cdr = &p->Cdr();
    delete p;
    return *this;
}

INLINE Base & Cons::Reverse() { // in-place reverse; doesn't reverse sublists
    Base *l = new Cons(*_car, *_cdr);
    l->refs++;
    _car->refs--;
    _cdr->refs--;
    Base *p = nil.value;            // p is previous cell
    while (l != nil.value) {
        Base *n = &l->Cdr();        // save Cdr() before it's clobbered
        ((Cons*)l)->_cdr = p;   	// change pointer direction
        p = l;                      // p winds up as first cons of list
        l = n;
    }
    _car = &p->Car();
    _cdr = &p->Cdr();
    delete p;
    return *this;
}

INLINE Base & Cons::Rplaca(Base &a) { 
    _car->Deref(); 
    _car = a.Ref(); 
    return *_car; 
}

// According to Steele, rplacd is supposed to return 'this'.  I changed
// it so it returns the new cdr -- this often makes list-building
// loops more efficient.

INLINE Base & Cons::Rplacd(Base &a) { 
    _cdr->Deref(); 
    _cdr = a.Ref(); 
    return *_cdr; 
}

INLINE Base & Cons::Set_difference(Base &l2) {  // no Null:: method needed
    Base *first_dummy = new Cons();
    Base *last = first_dummy;
    for (Base *a=this; a != nil.value; a=&a->Cdr())
            // Is the car() a member of l2?  If not, then
            // it becomes part of the set difference.
        if (& l2.Member(a->Car()) == nil.value &&
            & first_dummy->Cdr().Member(a->Car()) == nil.value)
            last = & last->Rplacd(*new Cons(a->Car()));
    Base *second = & first_dummy->Cdr();
    delete first_dummy;
    second->refs--;
    return *second;
}


INLINE LObject_type Cons::Type() { return type_Cons; }

INLINE Base & Cons::Typep(LObject_type a) 
    { return (a==type_Cons) ? *t.value : *nil.value; }
