/*
    num_log.c  -- Logical operations on numbers.
*/
/*
    Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/

#include "config.h"
#include "num_include.h"

/*
	x : fixnum or bignum (may be not normalized)
	y : integer
   returns
	fixnum or bignum ( not normalized )
*/

object
log_op(int narg, int (*op)(), object *nums)
{
	object x, numi;
	int	i, j; bool jset;
	struct bignum *big_log_op();

	if (narg < 2) FEtoo_few_arguments(&narg);
	x = *nums++;
	if (type_of(x) == t_bignum) goto BIG_OP;
	j = fix(x);
	jset = TRUE;
	for (i = 1; i < narg; i++) {
	  numi = *nums++;
	  if (type_of(numi) == t_bignum) goto BIG_OP;
	  j = (*op)(j, fix(numi));
	}
	return(MAKE_FIXNUM(j));

BIG_OP:
	if (jset) {
	  x = (object)copy_to_big(MAKE_FIXNUM(j));
	  x = (object)big_log_op(x, numi, op);
	  i++;
	}
	else
	  x = (object)copy_to_big(x);

	for (; i < narg; i++) {
	  numi = *nums++;
	  x = (object)big_log_op(x, numi, op);
	}
	x = normalize_big_to_object(x);
	return(x);
}
/*
	big_log_op(x, y, op) performs the logical operation op onto
	x and y, and return the result in x destructively.
*/
struct bignum *
big_log_op(struct bignum *x, object y, int (*op)())
{
	struct bignum *r;
	int	sign_x, sign_y;
	int	ext_x, ext_y;
	int	end_x, end_y;
	int	i, j;

	r = x;		/* remember start of x */
	if (type_of(x) != t_bignum)
		FEwrong_type_argument(Sbignum, (object)x);
	else if (big_sign(x) < 0) {
		sign_x = ~MASK;
		ext_x = MASK;
	     } else
		sign_x = ext_x = 0;
	if (FIXNUMP(y))
		if (fix(y) < 0) {
			sign_y = ~MASK;
			ext_y = MASK;
		} else
			sign_y = ext_y = 0;
	else if (type_of(y) == t_bignum)
		if (big_sign(y) < 0) {
			sign_y = ~MASK;
			ext_y = MASK;
		} else
			sign_y = ext_y = 0;
	else
		FEwrong_type_argument(Sinteger, y);

	end_x = end_y = 0;
	while ((end_x == 0) || (end_y == 0)) {
		if (end_x == 0)
			i = (x->big_car) & MASK;
		else
			i = ext_x;
		if (end_y == 0)
			if (FIXNUMP(y))
				j = (fix(y)) & MASK;
			else
				j = (y->big.big_car) & MASK;
		else
			j = ext_y;
		i = (*op)(i, j);
		if (end_x == 0)
			x->big_car = i & MASK;
		else
			x = stretch_big(x, i & MASK);
		if (x->big_cdr != NULL)
			x = x->big_cdr;
		else
			end_x = 1;
		if (FIXNUMP(y))
			end_y = 1;
		else if (y->big.big_cdr != 0)
			y = (object)y->big.big_cdr;
		else
			end_y = 1;
	}
	/* Now x points ths last sell of bignum.
	   We must set the sign bit according to operation.
	   Sign bit of x is already masked out in previous
	   while-iteration */
	x->big_car |= ((*op)(sign_x, sign_y) & ~MASK);

	return(r);
}

int
ior_op(int i, int j)
{
	return(i | j);
}

int
xor_op(int i, int j)
{
	return(i ^ j);
}

int
and_op(int i, int j)
{
	return(i & j);
}

int
eqv_op(int i, int j)
{
	return(~(i ^ j));
}

int
nand_op(int i, int j)
{
	return(~(i & j));
}

int
nor_op(int i, int j)
{
	return(~(i | j));
}

int
andc1_op(int i, int j)
{
	return((~i) & j);
}

int
andc2_op(int i, int j)
{
	return(i & (~j));
}

int
orc1_op(int i, int j)
{
	return((~i) | j);
}

int
orc2_op(int i, int j)
{
	return(i | (~j));
}

b_clr_op(int i, int j)
{
	return(0);
}

b_set_op(int i, int j)
{
	return(-1);
}

b_1_op(int i, int j)
{
	return(i);
}

b_2_op(int i, int j)
{
	return(j);
}

b_c1_op(int i, int j)
{
	return(~i);
}

b_c2_op(int i, int j)
{
	return(~j);
}

int
big_bitp(object	x, int p)
{
	int	sign, cell, bit, i;

	if (p >= 0) {
		cell = p / 31;
		bit = p % 31;
		while (cell-- > 0) {
			if (x->big.big_cdr != NULL)
				x = (object)x->big.big_cdr;
			else if (x->big.big_car < 0)
				return(1);
			else
				return(0);
		}
		return((x->big.big_car >> bit) & 1);
	} else
		return(0);
}

int
fix_bitp(object	x, int p)
{
/* No need to distinguish sign. Beppe
	if (p > 30)		/* fix = sign + bit0-30 *//*
		if (fix(x) < 0)
			return(1);
		else
			return(0);
 */
	return((fix(x) >> p) & 1);
}	

int
count_int_bits(int x)
{
	int	i, count;

	count = 0;
	for (i=0; i < 31; i++) count += ((x >> i) & 1);
	return(count);
}

int
count_bits(object x)
{
	int	i, count, sign;

	if (FIXNUMP(x)) {
		i = fix(x);
		if (i < 0) i = ~i;
		count = count_int_bits(i);
	} else if (type_of(x) == t_bignum) {
		count = 0;
		sign = big_sign(x);
		for (;;) {
			i = x->big.big_car;
			if (sign < 0) i = ~i & MASK;
			count += count_int_bits(i);
			if (x->big.big_cdr == NULL) break;
			x = (object)x->big.big_cdr;
		}
	} else
		FEwrong_type_argument(Sinteger, x);
	return(count);
}

/*
	double_shift(h, l, w, hp, lp) shifts the int h & l ( 31 bits)
	w bits to left ( w > 0) or to right ( w < 0).
	result is returned in *hp and *lp.
*/
double_shift(int h, int l, int w, int *hp, int *lp)
{

	if (w >= 0) {
		*lp = (l << w) & MASK;
		*hp = ((h << w) & MASK) | ((l & MASK) >> (31 - w));
	} else {
		w = -w;
		*hp = (h & MASK) >> w;
		*lp = ((h << (31 - w)) & MASK) | ((l & MASK) >> w);
	}
}

object
shift_integer(object x, int w)
{
	struct bignum *y, *y0;
	object	r;
	int	cell, bits, sign, i;
	int	ext, h, l, nh, nl, end_x;
	
	cell = w / 31;
	bits = w % 31;
	if (FIXNUMP(x)) {
		i = fix(x);
		if (cell == 0) {
			if (w < 0) {
				if (i >= 0)
					return(MAKE_FIXNUM(i >> -w));
				else
					return(MAKE_FIXNUM(~((~i) >> -w)));
			} if (i >= 0) {
				if (((-1<<(31-w)) & i) == 0)
				/* if (((~MASK >> w) & i) == 0) */
					return(MAKE_FIXNUM(i << w));
			} else {
				if (w < 32 && ((-1<<(31-w)) & ~i) == 0)
				/* if (w < 32 && ((~MASK >> w) & ~i) == 0) */
					return(MAKE_FIXNUM(i << w));
			}
		}
		x = alloc_object(t_bignum);
		x->big.big_car = i;
		x->big.big_cdr = NULL;
	}

	if ((sign = big_sign(x)) < 0)
		ext = MASK;
	else
		ext = 0;

	y = y0 = (struct bignum *)alloc_object(t_bignum);
	y->big_car = 0;
	y->big_cdr = NULL;

	if (w < 0) goto RIGHT;
LEFT:
	while (cell-- > 0)
		y = stretch_big(y, 0);
	l = 0;
	h = x->big.big_car;
	end_x = 0;
	goto COMMON;

RIGHT:
	end_x = 0;
	h = x->big.big_car;
	while (cell++ <= 0) {
		l = h;
		if (end_x == 1) break;
		if (x->big.big_cdr != NULL) {
			x = (object)x->big.big_cdr;
			h = x->big.big_car;
		} else {
			end_x = 1;
			h = ext;
		}
	}

COMMON:
	for (;;) {
		double_shift(h, l, bits, &nh, &nl);
		if (w < 0)
			y->big_car = nl;
		else
			y->big_car = nh;
		if (end_x == 1) break;
		l = h;
		if (x->big.big_cdr != NULL) {
			x = (object)x->big.big_cdr;
			h = x->big.big_car;
		} else {
			h = ext;
			end_x = 1;
		}
		y = stretch_big(y, 0);
	}
	/* set sign bit */
	if (sign < 0) y->big_car |= ~MASK;
	r = normalize_big_to_object(y0);
	return(r);
}

int
int_bit_length(int i)
{
	int	count, j;

	count = 0;
	for (j = 0; j < 31 ; j++)
		if (((i >> j) & 1) == 1) count = j + 1;
	return(count);
}

Llogior(int narg, ...)
{
	object  x;
	int	i;
	int	ior_op();
	va_list nums;

	if (narg == 0) {
		 VALUES(0) = MAKE_FIXNUM(0);
		 RETURN(1);
	}
	va_start(nums, narg);
	for (i = 0; i < narg; i++) {
	  x = va_arg(nums, object);
	  check_type_integer(&x);
	}
	va_start(nums, narg);
	if (narg == 1) {
		 VALUES(0) = va_arg(nums, object);
		 RETURN(1);
	}
	VALUES(0) = log_op(narg, ior_op, (object *)nums);
	RETURN(1);
}

Llogxor(int narg, ...)
{
	object  x;
	int	i;
	int	xor_op();
	va_list nums;

	if (narg == 0) {
		 VALUES(0) = MAKE_FIXNUM(0);
		 RETURN(1);
	}
	va_start(nums, narg);
	for (i = 0; i < narg; i++) {
	  x = va_arg(nums, object);
	  check_type_integer(&x);
	}
	va_start(nums, narg);
	if (narg == 1) {
		 VALUES(0) = va_arg(nums, object);
		 RETURN(1);
	}
	VALUES(0) = log_op(narg, xor_op, (object *)nums);
	RETURN(1);
}

Llogand(int narg, ...)
{
	object  x;
	int	i;
	int	and_op();
	va_list nums;

	if (narg == 0) {
		 VALUES(0) = MAKE_FIXNUM(-1);
		 RETURN(1);
	}
	va_start(nums, narg);
	for (i = 0; i < narg; i++) {
	  x = va_arg(nums, object);
	  check_type_integer(&x);
	}
	va_start(nums, narg);
	if (narg == 1) {
		 VALUES(0) = va_arg(nums, object);
		 RETURN(1);
	}
	VALUES(0) = log_op(narg, and_op, (object *)nums);
	RETURN(1);
}

Llogeqv(int narg, ...)
{
	object  x;
	int	i;
	int	eqv_op();
	va_list nums;

	if (narg == 0) {
		 VALUES(0) = MAKE_FIXNUM(-1);
		 RETURN(1);
	}
	va_start(nums, narg);
	for (i = 0; i < narg; i++) {
	  x = va_arg(nums, object);
	  check_type_integer(&x);
	}
	va_start(nums, narg);
	if (narg == 1) {
		 VALUES(0) = va_arg(nums, object);
		 RETURN(1);
	}
	VALUES(0) = log_op(narg, eqv_op, (object *)nums);
	RETURN(1);
}

Lboole(int narg, object o, ...)
{
	int	(*op)();
	va_list nums;
	object x;

	check_arg(3);
	va_start(nums, o);
	check_type_integer(&o);
	x = va_arg(nums, object);
	check_type_integer(&x);
	x = va_arg(nums, object);
	check_type_integer(&x);
	switch(fixint(o)) {
		case BOOLCLR:	op = b_clr_op;	break;
		case BOOLSET:	op = b_set_op;	break;
		case BOOL1:	op = b_1_op;	break;
		case BOOL2:	op = b_2_op;	break;
		case BOOLC1:	op = b_c1_op;	break;
		case BOOLC2:	op = b_c2_op;	break;
		case BOOLAND:	op = and_op;	break;
		case BOOLIOR:	op = ior_op;	break;
		case BOOLXOR:	op = xor_op;	break;
		case BOOLEQV:	op = eqv_op;	break;
		case BOOLNAND:	op = nand_op;	break;
		case BOOLNOR:	op = nor_op;	break;
		case BOOLANDC1:	op = andc1_op;	break;
		case BOOLANDC2:	op = andc2_op;	break;
		case BOOLORC1:	op = orc1_op;	break;
		case BOOLORC2:	op = orc2_op;	break;
		default:
			FEerror("~S is an invalid logical operator.",
				1, o);
	}
	va_start(nums, o);
	VALUES(0) = log_op(2, op, (object *)nums);
	RETURN(1);
}

Llogbitp(int narg, object p, object x)
{
	int	i;

	check_arg(2);
	check_type_integer(&p);
	check_type_integer(&x);
	if (FIXNUMP(p))
		if (FIXNUMP(x))
			i = fix_bitp(x, fix(p));
		else
			i = big_bitp(x, fix(p));
	else if (big_sign(p) < 0)
			i = 0;
		/*
		   bit position represented by bignum is out of
		   our address space. So, result is returned
		   according to sign of integer.
		*/

	else if (FIXNUMP(x))
		if (fix(x) < 0)
			i = 1;
		else
			i = 0;
	else if (big_sign(x) < 0)
			i = 1;
		else
			i = 0;

	VALUES(0) = (i == 1) ? Ct : Cnil;
	RETURN(1);
}

Lash(int narg, object x, object y)
{
	object	r;
	int	w, sign_x;

	check_arg(2);
        check_type_integer(&x);
	check_type_integer(&y);
	if (FIXNUMP(y)) {
		w = fix(y);
		r = shift_integer(x, w);
	} else if (type_of(y) == t_bignum)
		goto LARGE_SHIFT;
	else
		;
	goto BYE;

	/*
	bit position represented by bignum is probably
	out of our address space. So, result is returned
	according to sign of integer.
	*/
LARGE_SHIFT:
	if (FIXNUMP(x))
		if (fix(x) > 0)
			sign_x = 1;
		else if (fix(x) == 0)
			sign_x = 0;
		else
			sign_x = -1;
	else
		sign_x = big_sign(x);
	if (big_sign(y) < 0)
		if (sign_x < 0)
			r = MAKE_FIXNUM(-1);
		else
			r = MAKE_FIXNUM(0);
	else if (sign_x == 0)
		r = MAKE_FIXNUM(0);
	else
		FEerror("Insufficient memory.", 0);

BYE:
	VALUES(0) = r;
	RETURN(1);
}

Llogcount(int narg, object x)
{
	int	i;

	check_arg(1);
	check_type_integer(&x);
	i = count_bits(x);
	VALUES(0) = MAKE_FIXNUM(i);
	RETURN(1);
}

Linteger_length(int narg, object x)
{
	int	count, cell, i;

	check_arg(1);
	check_type_integer(&x);
	if (FIXNUMP(x)) {
		i = fix(x);
		if (i < 0) i = ~i;
		count = int_bit_length(i);
	} else if (type_of(x) == t_bignum) {
		cell = 0;
		while (x->big.big_cdr != NULL) {
			cell++;
			x = (object)x->big.big_cdr;
		}
		i = x->big.big_car;
		if (i < 0) i = ~i;
		count = cell * 31 + int_bit_length(i);
	}
	VALUES(0) = MAKE_FIXNUM(count);
	RETURN(1);
}


object Sbit;

init_num_log()
{
	int siLbit_array_op();

	make_constant("BOOLE-CLR", MAKE_FIXNUM(BOOLCLR));
	make_constant("BOOLE-SET", MAKE_FIXNUM(BOOLSET));
	make_constant("BOOLE-1", MAKE_FIXNUM(BOOL1));
	make_constant("BOOLE-2", MAKE_FIXNUM(BOOL2));
	make_constant("BOOLE-C1", MAKE_FIXNUM(BOOLC1));
	make_constant("BOOLE-C2", MAKE_FIXNUM(BOOLC2));
	make_constant("BOOLE-AND", MAKE_FIXNUM(BOOLAND));
	make_constant("BOOLE-IOR", MAKE_FIXNUM(BOOLIOR));
	make_constant("BOOLE-XOR", MAKE_FIXNUM(BOOLXOR));
	make_constant("BOOLE-EQV", MAKE_FIXNUM(BOOLEQV));
	make_constant("BOOLE-NAND", MAKE_FIXNUM(BOOLNAND));
	make_constant("BOOLE-NOR", MAKE_FIXNUM(BOOLNOR));
	make_constant("BOOLE-ANDC1", MAKE_FIXNUM(BOOLANDC1));
	make_constant("BOOLE-ANDC2", MAKE_FIXNUM(BOOLANDC2));
	make_constant("BOOLE-ORC1", MAKE_FIXNUM(BOOLORC1));
	make_constant("BOOLE-ORC2", MAKE_FIXNUM(BOOLORC2));

	make_function("LOGIOR", Llogior);
	make_function("LOGXOR", Llogxor);
	make_function("LOGAND", Llogand);
	make_function("LOGEQV", Llogeqv);
	make_function("BOOLE", Lboole);
	make_function("LOGBITP", Llogbitp);
	make_function("ASH", Lash);
	make_function("LOGCOUNT", Llogcount);
	make_function("INTEGER-LENGTH", Linteger_length);

	Sbit = make_ordinary("BIT");
	make_si_function("BIT-ARRAY-OP", siLbit_array_op);
}


siLbit_array_op(int narg, object o, object x, object y, object r)
{
	int i, j, n, d;
	object r0;
	int (*op)();
	bool replace = FALSE;
	int xi, yi, ri;
	char *xp, *yp, *rp;
	int xo, yo, ro;

	check_arg(4);
	if (type_of(x) == t_bitvector) {
		d = x->bv.bv_dim;
		xp = x->bv.bv_self;
		xo = x->bv.bv_offset;
		if (type_of(y) != t_bitvector)
			goto ERROR;
		if (d != y->bv.bv_dim)
			goto ERROR;
		yp = y->bv.bv_self;
		yo = y->bv.bv_offset;
		if (r == Ct)
			r = x;
		if (r != Cnil) {
			if (type_of(r) != t_bitvector)
				goto ERROR;
			if (r->bv.bv_dim != d)
				goto ERROR;
			i = (r->bv.bv_self - xp)*8 + (r->bv.bv_offset - xo);
			if (i > 0 && i < d || i < 0 && -i < d) {
				r0 = r;
				r = Cnil;
				replace = TRUE;
				goto L1;
			}
			i = (r->bv.bv_self - yp)*8 + (r->bv.bv_offset - yo);
			if (i > 0 && i < d || i < 0 && -i < d) {
				r0 = r;
				r = Cnil;
				replace = TRUE;
			}
		}
	L1:
		if (Null(r)) {
			siLmake_vector(7, Sbit, MAKE_FIXNUM(d), Cnil, Cnil, Cnil, Cnil, Cnil);
			r = VALUES(0);
		}
	} else {
		if (type_of(x) != t_array)
			goto ERROR;
		if ((enum aelttype)x->a.a_elttype != aet_bit)
			goto ERROR;
		d = x->a.a_dim;
		xp = x->bv.bv_self;
		xo = x->bv.bv_offset;
		if (type_of(y) != t_array)
			goto ERROR;
		if ((enum aelttype)y->a.a_elttype != aet_bit)
			goto ERROR;
		if (x->a.a_rank != y->a.a_rank)
			goto ERROR;
		yp = y->bv.bv_self;
		yo = y->bv.bv_offset;
		for (i = 0;  i < x->a.a_rank;  i++)
			if (x->a.a_dims[i] != y->a.a_dims[i])
				goto ERROR;
		if (r == Ct)
			r = x;
		if (r != Cnil) {
			if (type_of(r) != t_array)
				goto ERROR;
			if ((enum aelttype)r->a.a_elttype != aet_bit)
				goto ERROR;
			if (r->a.a_rank != x->a.a_rank)
				goto ERROR;
			for (i = 0;  i < x->a.a_rank;  i++)
				if (r->a.a_dims[i] != x->a.a_dims[i])
					goto ERROR;
			i = (r->bv.bv_self - xp)*8 + (r->bv.bv_offset - xo);
			if (i > 0 && i < d || i < 0 && -i < d) {
				r0 = r;
				r = Cnil;
				replace = TRUE;
				goto L2;
			} 
			i = (r->bv.bv_self - yp)*8 + (r->bv.bv_offset - yo);
			if (i > 0 && i < d || i < 0 && -i < d) {
				r0 = r;
				r = Cnil;
				replace = TRUE;
			}
		}
	L2:
		if (Null(r)) {
		  r = alloc_object(t_array);
		  r->a.a_self = NULL;
		  r->a.a_displaced = Cnil;
		  r->a.a_rank = 1;
		  r->a.a_dims = NULL;
		  r->a.a_elttype = (short)get_aelttype(Sbit);
		  r->a.a_dims = (int *)alloc_relblock(sizeof(int),
						      sizeof(int));
		  r->a.a_dim = x->a.a_dim;
		  r->a.a_adjustable = FALSE;
		  array_allocself(r, FALSE);
		}
	}
	rp = r->bv.bv_self;
	ro = r->bv.bv_offset;
	switch(fixint(o)) {
		case BOOLCLR:	op = b_clr_op;	break;
		case BOOLSET:	op = b_set_op;	break;
		case BOOL1:	op = b_1_op;	break;
		case BOOL2:	op = b_2_op;	break;
		case BOOLC1:	op = b_c1_op;	break;
		case BOOLC2:	op = b_c2_op;	break;
		case BOOLAND:	op = and_op;	break;
		case BOOLIOR:	op = ior_op;	break;
		case BOOLXOR:	op = xor_op;	break;
		case BOOLEQV:	op = eqv_op;	break;
		case BOOLNAND:	op = nand_op;	break;
		case BOOLNOR:	op = nor_op;	break;
		case BOOLANDC1:	op = andc1_op;	break;
		case BOOLANDC2:	op = andc2_op;	break;
		case BOOLORC1:	op = orc1_op;	break;
		case BOOLORC2:	op = orc2_op;	break;
		default:
			FEerror("~S is an invalid logical operator.", 1, o);
	}

#define	set_high(place, nbits, value) \
	((place)=((place)&~(-0400>>(nbits))|(value)&(-0400>>(nbits))))

#define	set_low(place, nbits, value) \
	((place)=((place)&(-0400>>(8-(nbits)))|(value)&~(-0400>>(8-(nbits)))))

#define	extract_byte(integer, pointer, index, offset) \
	(integer) = (pointer)[(index)+1] & 0377; \
	(integer) = ((pointer)[index]<<(offset))|((integer)>>(8-(offset)))

#define	store_byte(pointer, index, offset, value) \
	set_low((pointer)[index], 8-(offset), (value)>>(offset)); \
	set_high((pointer)[(index)+1], offset, (value)<<(8-(offset)))

	if (xo == 0 && yo == 0 && ro == 0) {
		for (n = d/8, i = 0;  i < n;  i++)
			rp[i] = (*op)(xp[i], yp[i]);
		if ((j = d%8) > 0)
			set_high(rp[n], j, (*op)(xp[n], yp[n]));
		if (!replace) {
			VALUES(0) = r;
			RETURN(1);
		}
	} else {
		for (n = d/8, i = 0;  i <= n;  i++) {
			extract_byte(xi, xp, i, xo);
			extract_byte(yi, yp, i, yo);
			if (i == n) {
				if ((j = d%8) == 0)
					break;
				extract_byte(ri, rp, n, ro);
				set_high(ri, j, (*op)(xi, yi));
			} else
				ri = (*op)(xi, yi);
			store_byte(rp, i, ro, ri);
		}
		if (!replace) {
			VALUES(0) = r;
			RETURN(1);
		}
	}
	rp = r0->bv.bv_self;
	ro = r0->bv.bv_offset;
	for (n = d/8, i = 0;  i <= n;  i++) {
		if (i == n) {
			if ((j = d%8) == 0)
				break;
			extract_byte(ri, rp, n, ro);
			set_high(ri, j, r->bv.bv_self[n]);
		} else
			ri = r->bv.bv_self[i];
		store_byte(rp, i, ro, ri);
	}
	VALUES(0) = r0;
	RETURN(1);

ERROR:
	FEerror("Illegal arguments for bit-array operation.", 0);
}
