/*------------------------------------------------------------------------------
 * Copyright (C) 1993 Christian-Albrechts-Universitaet zu Kiel
 *-----------------------------------------------------------------------------
 * Projekt  : APPLY - A Practicable And Portable Lisp Implementation
 *            ------------------------------------------------------
 * Funktion : Laufzeitsystem
 *            - SYMBOL-VALUE
 *            - BOUNDP
 *            - SET
 *            - MAKUNBOUND
 *            - SYMBOL-PLIST
 *            - set-symbol-plist
 *            - SYMBOL-NAME
 *            - MAKE-SYMBOL
 *            - SYMBOL-PACKAGE
 *            - set-symbol-package
 *            - set-constant-flag
 *            - setup-symbols-iterator
 *
 * $Revision: 1.11 $
 * $Log: symbols.c,v $
 * Revision 1.11  1993/07/06  12:32:50  sma
 * OFFSET-Makro eingefhrt.
 *
 * Revision 1.10  1993/06/16  14:43:22  hk
 * Copyright Notiz eingefuegt.
 *
 * Revision 1.9  1993/05/12  11:36:38  hk
 * symbol_package_index definiert, wie Fsymbol_package, aber mit anderem
 * Resultattyp.
 *
 * Revision 1.8  1993/05/08  18:16:23  hk
 * set_symbol_plist -> Fset_symbol_plist, Argumentreihenfolge geaendert.
 *
 * Revision 1.7  1993/04/22  10:23:04  hk
 * fun_decl.h -> sys.h, Symbole NIL + T in Ssys definiert,
 * Funktionen fuer den Zugriff auf Komponenten von Symbolen umgestellt,
 * so dass sie die Komponenten von NIL kennen, auch wenn der Wert einer
 * CL_FORM, die NIL darstellt, keinen Zeiger auf das Symbol NIL enthaelt.
 *
 * Revision 1.6  1993/02/17  15:48:19  hk
 * CLICC -> APPLY, Revison Keyword.
 *
 * Revision 1.5  1993/01/08  09:44:13  hk
 * Namen C_ nach c_.
 *
 * Revision 1.4  1992/09/30  17:25:21  hk
 * unbound_value_p neu.
 *
 * Revision 1.3  1992/09/28  17:20:28  hk
 * Lerror -> Labort, neues Lerror mit Lisp-Parameter
 *
 * Revision 1.2  1992/07/21  14:57:11  hk
 * Fset_symbol_plist --> set_symbol_plist.
 *
 * Revision 1.1  1992/03/24  17:03:37  hk
 * Initial revision
 *----------------------------------------------------------------------------*/

#include "c_decl.h"
#include "sys.h"

static char SYM_EXPECTED[] = "~a is not a symbol";
static char TRY_CHANGE_CONST[] =
   "can not change value of ~a, which is a constant";
static char STRING_EXPECTED[] = "~a is not a string";

/*------------------------------------------------------------------------------
 * Die Symbole NIL Und T
 *----------------------------------------------------------------------------*/
CL_INIT Ssys[2*SYM_SIZE+1]=
{
{ CL_FIXNUM,   (long)3 },
{ RT_CHAR_PTR, (long)"NIL" },
{ CL_NIL,      (long)NIL_VALUE },  /* Property Liste */
{ CL_NIL,      (long)NIL_VALUE },  /* Wert: Wird nicht benutzt */
{ CL_NIL,      (long)NIL_VALUE },  /* Das Package wird vom Lisp Modul gesetzt */
{ CL_SYMBOL,   (long)T_VALUE },    /* Constant-Flag: Wird nicht benutzt */

{ CL_FIXNUM,   (long)1 },
{ RT_CHAR_PTR, (long)"T" },
{ CL_NIL,      (long)NIL_VALUE },
{ CL_SYMBOL,   (long)T_VALUE },
{ CL_NIL,      (long)NIL_VALUE },
{ CL_SYMBOL,   (long)T_VALUE },

  0
};

#define NIL_PLIST           (CL_FORM *)&Ssys[OFF_SYM_PLIST]
#define LOAD_NIL_PNAME(loc) LOAD_SMSTR((CL_FORM *)&Ssys[OFF_SYM_PNAME], loc)
#define NIL_PACKAGE         (CL_FORM *)&Ssys[OFF_SYM_PACKAGE]

/*------------------------------------------------------------------------------
 * symbol-value symbol
 *----------------------------------------------------------------------------*/
void Fsymbol_value(base)
CL_FORM *base;
{
   switch(TYPE_OF(STACK(base, 0)))
   {
   case CL_SYMBOL:
      COPY(SYM_VALUE(STACK(base, 0)), STACK(base, 0));
      break;
   case CL_NIL:
      break; 
   default:
      Lerror(STACK (base, 0), SYM_EXPECTED);
   }
}

/*------------------------------------------------------------------------------
 * boundp symbol
 *----------------------------------------------------------------------------*/
void Fboundp(base)
CL_FORM *base;
{
   switch(TYPE_OF(STACK(base, 0)))
   {
   case CL_SYMBOL:
      if(CL_UNBOUNDP(SYM_VALUE(STACK(base, 0))))
         LOAD_NIL(STACK(base, 0));
      break;
   case CL_NIL:
      LOAD_T(STACK(base, 0));
      break;
   default:
      Lerror(STACK (base, 0), SYM_EXPECTED);
   }
}

/*------------------------------------------------------------------------------
 * Prueft, ob das Argument der spezielle Wert UNBOUND ist.
 * Der kann nur entstehen, wenn auf ein ungebundenes Symbol zugeriffen wird,
 * ohne das Programm abzubrechen.
 * Wird in print benutzt.
 *----------------------------------------------------------------------------*/
void unbound_value_p(base)
CL_FORM *base;
{
   if(CL_UNBOUNDP(STACK(base, 0)))
      LOAD_T(STACK(base, 0));
   else
      LOAD_NIL(STACK(base, 0));
}

/*------------------------------------------------------------------------------
 * set symbol value
 *----------------------------------------------------------------------------*/
void Fset(base)
CL_FORM *base;
{
   switch(TYPE_OF(STACK(base, 0)))
   {
   case CL_SYMBOL:
      if(!CL_NILP(SYM_CONSTFLAG(STACK(base, 0))))
         Lerror(STACK (base, 0), TRY_CHANGE_CONST);
      COPY(STACK(base, 1), SYM_VALUE(STACK(base, 0)));
      break;
   case CL_NIL:
      Lerror(STACK (base, 0), TRY_CHANGE_CONST);
   default:
      Lerror(STACK (base, 0), SYM_EXPECTED);
   }
}

/*------------------------------------------------------------------------------
 * makunbound symbol
 *----------------------------------------------------------------------------*/
void Fmakunbound (base)
CL_FORM *base;
{
   switch(TYPE_OF(STACK(base, 0)))
   {
   case CL_SYMBOL:
      if(!CL_NILP(SYM_CONSTFLAG(STACK(base, 0))))
         Lerror(STACK (base, 0), TRY_CHANGE_CONST);
      LOAD_UNBOUND(SYM_VALUE(STACK(base, 0)));
      break;
   case CL_NIL:
      Lerror(STACK (base, 0), TRY_CHANGE_CONST);
   default:
      Lerror(STACK (base, 0), SYM_EXPECTED);
   }
}

/*------------------------------------------------------------------------------
 * symbol-plist symbol
 *----------------------------------------------------------------------------*/
void Fsymbol_plist (base)
CL_FORM *base;
{
   switch(TYPE_OF(STACK(base, 0)))
   {
   case CL_SYMBOL:
      COPY(SYM_PLIST(STACK(base, 0)), STACK(base, 0));
      break;
   case CL_NIL:
      COPY(NIL_PLIST, STACK(base, 0));
      break;
   default:
      Lerror(STACK (base, 0), SYM_EXPECTED);
   }
}

/*------------------------------------------------------------------------------
 * (setf symbol-plist) value symbol
 * Resultat: value
 *----------------------------------------------------------------------------*/
void Fset_symbol_plist(base)
CL_FORM *base;
{
   switch(TYPE_OF(STACK(base, 1)))
   {
   case CL_SYMBOL:
      COPY(STACK(base, 0), SYM_PLIST(STACK(base, 1)));
      break;
   case CL_NIL: 
      COPY(STACK(base, 0), NIL_PLIST);
      break;
   default:
      Lerror(STACK (base, 1), SYM_EXPECTED);
   }
}

/*------------------------------------------------------------------------------
 * symbol-name symbol
 *----------------------------------------------------------------------------*/
void Fsymbol_name (base)
CL_FORM *base;
{
   switch(TYPE_OF(STACK(base, 0)))
   {
   case CL_SYMBOL:
      LOAD_SYM_PNAME(STACK(base, 0), STACK(base, 0));
      break;
   case CL_NIL: 
      LOAD_NIL_PNAME(STACK(base, 0));
      break;
   default:
      Lerror(STACK (base, 0), SYM_EXPECTED);
   }
}

/*------------------------------------------------------------------------------
 * make-symbol print-name
 *----------------------------------------------------------------------------*/
void Fmake_symbol(base)
CL_FORM *base;
{
   CL_FORM *sym;
   
   switch(TYPE_OF(STACK(base, 0)))
   {
   case CL_SMSTR:
      break; 
   case CL_STRING:
      /* mittels copy-seq eine Kopie vom Typ CL_SMSTR erzeugen */
      /* ----------------------------------------------------- */
      Fcopy_seq(STACK(base, 0));
      break; 
   default:
      Lerror(STACK (base, 0), STRING_EXPECTED);
   }
   sym = form_alloc(STACK(base, 1), SYM_SIZE);
   LOAD_FIXNUM(AR_SIZE(GET_FORM(STACK(base, 0))), OFFSET(sym,OFF_SYM_LEN));
   LOAD_CHAR_PTR(sm_get_c_string(STACK(base, 0)), OFFSET(sym,OFF_SYM_C_STRING));
   LOAD_NIL(OFFSET(sym,OFF_SYM_PLIST));
   LOAD_UNBOUND(OFFSET(sym,OFF_SYM_VALUE));
   LOAD_NIL(OFFSET(sym,OFF_SYM_PACKAGE));
   LOAD_NIL(OFFSET(sym,OFF_SYM_CONSTFLAG));
   LOAD_SYMBOL(sym, STACK(base, 0));
}

/*------------------------------------------------------------------------------
 * symbol-package sym
 *----------------------------------------------------------------------------*/
void Fsymbol_package (base)
CL_FORM *base;
{
   switch(TYPE_OF(STACK(base, 0)))
   {
   case CL_SYMBOL:
      COPY(SYM_PACKAGE(STACK(base, 0)), STACK(base, 0));
      break;
   case CL_NIL:
      COPY(NIL_PACKAGE, STACK(base, 0));
      break;
   default:
      Lerror(STACK (base, 0), SYM_EXPECTED);
   }
}

/*------------------------------------------------------------------------------
 * symbol-package-index symbol
 *----------------------------------------------------------------------------*/
void symbol_package_index (base)
CL_FORM *base;
{
   COPY(SYM_PACKAGE(STACK(base, 0)), STACK(base, 0));
}

/*------------------------------------------------------------------------------
 * set-symbol-package value sym
 * wird nur intern verwendet, keine Typueberpruefung notwendig
 *----------------------------------------------------------------------------*/
void set_symbol_package (base)
CL_FORM *base;
{
   switch(TYPE_OF(STACK(base, 1)))
   {
   case CL_SYMBOL:
      COPY(STACK(base, 0), SYM_PACKAGE(STACK(base, 1)));
      break;
   case CL_NIL:
      COPY(STACK(base, 0), NIL_PACKAGE);
      break;
   default:
      Lerror(STACK (base, 0), SYM_EXPECTED);
   }
}

/*------------------------------------------------------------------------------
 * set-constant-flag sym                                                  
 * wird nur intern verwendet, keine Typueberpruefung notwendig
 *----------------------------------------------------------------------------*/
void set_constant_flag (base)
CL_FORM *base;
{
   switch(TYPE_OF(STACK(base, 0)))
   {
   case CL_SYMBOL:
      LOAD_T(SYM_CONSTFLAG(STACK(base, 0)));
      break;
   case CL_NIL:
      break;
   default:
      Lerror(STACK (base, 0), SYM_EXPECTED);
   }
}

/*------------------------------------------------------------------------------
 * setup-symbols-iterator first-sym package-vector
 * Wendet setup-symbol auf alle zur Uebersetzungszeit definierten
 * Symbole eines Moduls an.
 *----------------------------------------------------------------------------*/
void setup_symbols_iterator (base)
CL_FORM *base;
{
   CL_FORM *sym = GET_SYMBOL(STACK(base, 0));
   
   /* das Array ist mit einem NULL-Eintrag abgeschlossen */
   /* -------------------------------------------------- */
   while(TYPE_OF(sym) != 0)
   {
      LOAD_SYMBOL(sym, STACK(base, 2));
      COPY(STACK(base, 1), STACK(base, 3));
      setup_symbol(STACK(base, 2));
      sym += SYM_SIZE; 
   }
}
