;;;-----------------------------------------------------------------------------
;;; Copyright (C) 1993 Christian-Albrechts-Universitaet zu Kiel, Germany
;;;-----------------------------------------------------------------------------
;;; Projekt  : APPLY - A Practicable And Portable Lisp Implementation
;;;            ------------------------------------------------------
;;; Inhalt   : Die generische Funktion simplify-1form nimmt einfache
;;;            Verbesserungen an einem Zwischensprachknoten vor. Dabei werden
;;;            insbesondere Optimierungen fuer einige Funktionen vorgenommen.
;;;
;;; $Revision: 1.1 $
;;; $Log: simplifier.lisp,v $
;;; Revision 1.1  1993/06/30  15:23:15  jh
;;; Initial revision
;;;
;;;-----------------------------------------------------------------------------

(in-package "CLICC")

;;------------------------------------------------------------------------------
;; Funktionen, die in diesem Pass (Pass2) optimiert werden.
;;------------------------------------------------------------------------------

(p0-special-funs
 (?pass2 "P2")
 clicc-lisp::eq
 clicc-lisp::eql
 clicc-lisp::null
 clicc-lisp::=
 clicc-lisp::>
 clicc-lisp::<
 clicc-lisp::+
 clicc-lisp::-
 clicc-lisp::funcall
 clicc-lisp::set
 clicc-lisp::symbol-value)

;;------------------------------------------------------------------------------

(defmethod simplify-1form ((a-form form))
  a-form)

(defmethod simplify-1form ((a-progn-form progn-form))
  (let ((form-list (mapcan #'(lambda (form)
                               ;; (g1 .. gi (PROGN f1 .. fn) gi+2 .. gm) -->
                               ;; (g1 .. gi f1 .. fn gi+2 .. gm)
                               (if (progn-form-p form)
                                   (?form-list form)
                                   (list form)))
                           (?form-list a-progn-form))))
    (cond
      ((null form-list) empty-list)     ; (PROGN) -> ()
      ((null (rest form-list))          ; (PROGN form) -> form
       (first form-list))
      (t (setf (?form-list a-progn-form) form-list) a-progn-form))))

(defmethod simplify-1form ((an-if-form if-form))
  (let ((pred (?pred an-if-form))
        (then (?then an-if-form))
        (else (?else an-if-form)))
    (loop
     (if (and (app-p pred)
              (let ((form (?form pred)))
                (and (special-sys-fun-p form)
                     (eq (?symbol form) 'clicc-lisp::not))))
         (psetf pred (first (?arg-list pred))
                then else
                else then)
         (return)))
    (setf (?pred an-if-form) pred
          (?then an-if-form) then
          (?else an-if-form) else)
    an-if-form))
    
(defmethod simplify-1form ((a-switch-form switch-form))
  (let ((key (?form a-switch-form)))
    (if (copy-is-eq-p key)
        (dolist (a-labeled-form (?case-list a-switch-form)
                 (?otherwise a-switch-form))
          (when (eq key (?value a-labeled-form))
            (return (?form a-labeled-form))))
        a-switch-form)))

(defmethod simplify-1form ((a-let*-form let*-form))
  (let ((body (?body a-let*-form)))
    (if (?var-list a-let*-form)
        (progn
          ;; Geschachtelte let*-forms werden zu einer zusammengefasst.
          ;;----------------------------------------------------------
          (when (let*-form-p body)
            (setf (?var-list a-let*-form) (append (?var-list a-let*-form)
                                                  (?var-list body))
                  (?init-list a-let*-form) (append (?init-list a-let*-form)
                                                   (?init-list body))
                  (?body a-let*-form) (?body body)))
          a-let*-form)
        ;; let*-forms mit leerer var-list werden entfernt.
        ;;------------------------------------------------
        (?body a-let*-form))))

(defmethod simplify-1form ((a-labels-form labels-form))
  (if (?fun-list a-labels-form)
      a-labels-form
      (?body a-labels-form)))

(defmethod simplify-1form ((a-let/cc-form let/cc-form))
  (if (zerop (?read (?cont a-let/cc-form)))
      (?body a-let/cc-form)
      a-let/cc-form))

(defmethod simplify-1form ((a-tagbody-form tagbody-form))
  ;; Entfernen von nicht benoetigten Sprungmarken und ggf. Zusammenfassen
  ;; von aufeinanderfolgenden tagged-forms.
  ;;---------------------------------------------------------------------
  (let ((tagged-form-queue (empty-queue)))
    (dolist (a-tagged-form (?tagged-form-list a-tagbody-form))
      (cond
        ((plusp (?used a-tagged-form))
         (add-q a-tagged-form tagged-form-queue))
        ;; Die erste tagged-form wird in first-form integriert.
        ;;-----------------------------------------------------
        ((empty-queue-p tagged-form-queue)
         (setf (?first-form a-tagbody-form)
               (optimize-1form
                (make-instance 'progn-form
                               :form-list (list (?first-form a-tagbody-form)
                                                (?form a-tagged-form))
                               :type (?type (?form a-tagged-form))))))
        ;; Die tagged-form wird in die zuletzt bearbeitete tagged-form
        ;; integriert.
        ;;------------------------------------------------------------
        (t
         (let ((last-tagged-form (last-q tagged-form-queue)))
           (setf (?form last-tagged-form)
                 (optimize-1form
                  (make-instance 'progn-form
                                 :form-list (list (?form last-tagged-form)
                                                  (?form a-tagged-form))
                                 :type (?type (?form a-tagged-form)))))))))
    (if (empty-queue-p tagged-form-queue)
        (?first-form a-tagbody-form)
        (progn
          (setf (?tagged-form-list a-tagbody-form)
                (queue2list tagged-form-queue))
          a-tagbody-form))))
            
(defmethod simplify-1form ((a-mv-lambda mv-lambda))
  (if (eq (?body a-mv-lambda) empty-list)
      (?arg a-mv-lambda)
      a-mv-lambda))

(defmethod simplify-1form ((an-app app))
  (let ((fun (?form an-app)))
    (if (and (special-sys-fun-p fun) (?pass2 fun))
        (funcall (?pass2 fun) an-app)
        an-app)))

;;------------------------------------------------------------------------------
;; Funcall kann in der Zwischensprache direkt ausgedrueckt werden.
;;------------------------------------------------------------------------------
(defun p2-funcall (app)
  (let ((arg-list (?arg-list app)))
    (setf (?form app) (first arg-list))
    (setf (?arg-list app) (rest arg-list)))
  app)

;;------------------------------------------------------------------------------
(defun p2-eq (app)
  (let* ((arg-list (?arg-list app))
         (arg1 (first arg-list))
         (arg2 (second arg-list)))
    (cond

      ;; (eq <self-evaluating> <self-evaluating>)
      ;; wird ersetzt duch NIL oder T.
      ;;-----------------------------------------
      ((and (self-evaluating-p arg1)
            (self-evaluating-p arg2))
       (if (typecase arg1
             (null-form (null-form-p arg2))
             (character-form (and (character-form-p arg2)
                                  (eql (?value arg1) (?value arg2))))
             (int (and (int-p arg2)
                       (eql (?value arg1) (?value arg2))))
             (t (eq arg1 arg2)))
           
           (get-symbol-bind 'clicc-lisp::T)
           empty-list))

      ;; (eq nil x) -> (not x)
      ;;-----------------------
      ((null-form-p arg1)
       (setf (?form app) (get-global-fun 'clicc-lisp::not))
       (setf (?arg-list app) (list arg2))
       app)

      ;; (eq x nil) -> (not x)
      ;;-----------------------
      ((null-form-p arg2)
       (setf (?form app) (get-global-fun 'clicc-lisp::not))
       (setf (?arg-list app) (list arg1))
       app)
      
      (t app))))

;;------------------------------------------------------------------------------
(defun p2-eql (app)
  (let ((arg-list (?arg-list app)))
    (labels ((eql=eq (arg)
               (or (copy-is-eq-p arg)
                   (structured-literal-p arg))))

      ;; weiter mit EQ, wenn eines der Argumente eine Konstante
      ;; aber kein Float ist.
      ;;---------------------------------------------------------------
      (when (or (eql=eq (first arg-list)) (eql=eq (second arg-list)))
        (setf (?form app) (get-global-fun 'clicc-lisp::eq))
        (setq app (p2-eq app)))))
  app)

;;------------------------------------------------------------------------------
;; (null x) -> (not x)
;;
;; Das ist nur notwendig, da fur not weitreichendere Optimierungen vorgenommen
;; werden als fuer null.
;;------------------------------------------------------------------------------
(defun p2-null (app)
  (setf (?form app) (get-global-fun 'clicc-lisp::not))
  app)

;;------------------------------------------------------------------------------
;; = number &REST more-numbers 
;;------------------------------------------------------------------------------
(defun p2-= (app)
  (let ((arg-list (?arg-list app)))
    (cond
      ((= (length arg-list) 2)

       ;; (= form {0 | 0.0}), (= {0 | 0.0} form) -> (ZEROP form)
       ;;-------------------------------------------------------
       (let ((arg1 (first  arg-list))
             (arg2 (second arg-list)))
         (cond
           ((and (num-p arg1)
                 (zerop (?value arg1)))
            (setf (?form app) (get-global-fun 'clicc-lisp::ZEROP))
            (setf (?arg-list app) (list arg2))
            app)
           ((and (num-p arg2)
                 (zerop (?value arg2)))
            (setf (?form app) (get-global-fun 'clicc-lisp::ZEROP))
            (setf (?arg-list app) (list arg1))
            app)
           (t app))))
      (t app))))

;;------------------------------------------------------------------------------
;; < number &REST more-numbers 
;;------------------------------------------------------------------------------
(defun p2-< (app)
  (let ((arg-list (?arg-list app)))
    (cond
      ((= (length arg-list) 2)

       ;; (< {0 | 0.0} form) -> (PLUSP  form),
       ;; (< form {0 | 0.0}) -> (MINUSP form)
       ;;-------------------------------------
       (let ((arg1 (first  arg-list))
             (arg2 (second arg-list)))
         (cond
           ((and (num-p arg1)
                 (zerop (?value arg1)))
            (setf (?form app) (get-global-fun 'clicc-lisp::PLUSP))
            (setf (?arg-list app) (list arg2))
            app)
           ((and (num-p arg2)
                 (zerop (?value arg2)))
            (setf (?form app) (get-global-fun 'clicc-lisp::MINUSP))
            (setf (?arg-list app) (list arg1))
            app)
           (t app))))
      (t app))))

;;------------------------------------------------------------------------------
;; > number &REST more-numbers  
;;------------------------------------------------------------------------------
(defun p2-> (app)
  (let ((arg-list (?arg-list app)))
    (cond
      ((= (length arg-list) 2)

       ;; (> {0 | 0.0} form) -> (MINUSP form),
       ;; (> form {0 | 0.0}) -> (PLUSP  form)
       ;;-------------------------------------
       (let ((arg1 (first  arg-list))
             (arg2 (second arg-list)))
         (cond
           ((and (num-p arg1)
                 (zerop (?value arg1)))
            (setf (?form app) (get-global-fun 'clicc-lisp::MINUSP))
            (setf (?arg-list app) (list arg2))
            app)
           ((and (num-p arg2)
                 (zerop (?value arg2)))
            (setf (?form app) (get-global-fun 'clicc-lisp::PLUSP))
            (setf (?arg-list app) (list arg1))
            app)
           (t app))))
      (t app))))

;;------------------------------------------------------------------------------
;; + &REST numbers
;;------------------------------------------------------------------------------
(defun p2-+ (app)
  (let ((arg-list (?arg-list app)))
    (cond
      ((= (length arg-list) 2)

       ;; (+ form 1), (+ 1 form) -> (1+ form)
       ;;------------------------------------
       (let ((arg1 (first  arg-list))
             (arg2 (second arg-list)))
         (cond
           ((and (int-p arg1) (eql (?value arg1) 1))
            (setf (?form app) (get-global-fun 'clicc-lisp::1+))
            (setf (?arg-list app) (list arg2))
            app)
           ((and (int-p arg2) (eql (?value arg2) 1)) 
            (setf (?form app) (get-global-fun 'clicc-lisp::1+))
            (setf (?arg-list app) (list arg1))
            app)
           (t app))))
      (t app))))

;;------------------------------------------------------------------------------
;; - number &REST numbers   
;;------------------------------------------------------------------------------
(defun p2-- (app)
  (let ((arg-list (?arg-list app)))
    (cond
      ((= (length arg-list) 2)

       ;; (- form 1) -> (1- form)
       ;;------------------------
       (let ((arg1 (first  arg-list))
             (arg2 (second arg-list)))
         (cond
           ((and (int-p arg2) (eql (?value arg2) 1))
            (setf (?form app) (get-global-fun 'clicc-lisp::1-))
            (setf (?arg-list app) (list arg1))
            app)
           (t app))))
      (t app))))

;;------------------------------------------------------------------------------
;; Ueberfuehre (SET symbol form) in (SETQ zugeh. dynamic form).
;;------------------------------------------------------------------------------
(defun p2-set (app)
  (let ((first-arg (first (?arg-list app))))
    (if (sym-p first-arg)
        (make-instance 'setq-form 
                       :location (make-instance 'var-ref 
                                                :var (get-global-dynamic 
                                                      (?symbol first-arg)))
                       :form (second (?arg-list app)))
        app)))

;;------------------------------------------------------------------------------
;; Ueberfuehre (SYMBOL-VALUE symbol) in eine Referenz auf die zugehoergige
;; dynamische Variable.
;;------------------------------------------------------------------------------
(defun p2-symbol-value (app)
  (let ((first-arg (first (?arg-list app))))
    (if (sym-p first-arg)
        (make-instance 'var-ref
                       :var (get-global-dynamic (?symbol first-arg)))
        app)))

;;------------------------------------------------------------------------------
(provide "simplifier")
