;;;-----------------------------------------------------------------------------
;;; Copyright (C) 1993 Christian-Albrechts-Universitaet zu Kiel, Germany
;;;-----------------------------------------------------------------------------
;;; Projekt  : APPLY - A Practicable And Portable Lisp Implementation
;;;            ------------------------------------------------------
;;; Funktion : Optimierungen der Let*-Ausdruecke.
;;;
;;; $Revision: 1.14 $
;;; $Log: optimize.lisp,v $
;;; Revision 1.14  1993/07/14  10:47:16  atr
;;; Keliner Fehler bei update-let korrigiert.
;;;
;;; Revision 1.13  1993/07/13  16:28:24  atr
;;; Fehler bei optimize-let von Switch-form korrigiert.
;;;
;;; Revision 1.12  1993/07/06  14:26:46  atr
;;; Supplied Parameter entfernt aus der Parameterliste
;;; von Effect-of-form.
;;; *vars-bound-but-not-referenced* ist jetzt mit defvar anstatt mit
;;; declare (special ...) deklariert.
;;;
;;; Revision 1.11  1993/07/06  13:20:23  atr
;;; Tippfehler subst var --> subst-var korrigiert und
;;; Fehler bei subst-var in defined-fun korrigiert.
;;;
;;; Revision 1.10  1993/07/02  10:22:48  atr
;;; Fehler bei subst-with-app-permissible korrigiert.
;;;
;;; Revision 1.9  1993/07/01  16:03:18  atr
;;; Fehler bei effect-of-form von der switch-form korrigiert.
;;;
;;; Revision 1.8  1993/06/30  16:26:19  atr
;;; Unnoetige Ausgabe entfernt.
;;;
;;; Revision 1.7  1993/06/30  16:00:06  atr
;;; (defvar *let-effect* (empty-effect)) --> (defvar *let-effect*).
;;;
;;; Revision 1.6  1993/06/30  15:45:30  atr
;;; (require static-effect) entfernt.
;;;
;;; Revision 1.5  1993/06/29  23:25:23  atr
;;; Fehler bei der Substitution in den INIT-FORMEN der analysierten
;;; let-form korrigiert.
;;;
;;; Revision 1.4  1993/06/27  15:28:31  atr
;;; Erste ablauffaehige Version der Optimierungen
;;; der Let-ausdruecke.
;;;
;;; Revision 1.3  1993/06/17  08:18:10  hk
;;; Revision und Log eingefuegt.
;;;
;;; revision 1.2 1993/06/17 08:00:09 hk
;;; Copright Notiz eingefuegt
;;; 
;;; revision 1.1 1993/05/10 15:18:50 atr
;;; Initial revision
;;;-----------------------------------------------------------------------------

(in-package "CLICC")

(defvar *eliminated-vars* 0)
(defvar *eliminated-lets* 0)
(defvar *subst-number*    0)
(defvar *let-effect* )
(defvar *vars-bound-but-not-used* 0)

(defun let-optimizing ()
  (let ((*eliminated-vars* 0)
        (*eliminated-lets*  0)
        (*vars-bound-but-not-used* 0)
        (*subst-number*  0  ))
    (clicc-message "Optimizing the let-forms ...")
    (clicc-message "----------------------------")
    (setf (?fun-list *module*) 
          (mapcar #'optimize-a-fun (?fun-list *module*)))
    (setf (?toplevel-forms *module*) 
          (optimize-a-fun (?toplevel-forms *module*)))
    (clicc-message "~s variables are bound by let but not referenced"
                   *vars-bound-but-not-used*)
    (clicc-message "~s let-forms are eliminated" *eliminated-lets*)
    (clicc-message "~s let-variables are eliminated " *eliminated-vars*)
    (clicc-message "~s substitutions are done " *subst-number*)))
  
(defun optimize-a-fun (fun)
  (setf (?body fun) (optimize-let (?body fun)))
  fun)
        
;;;---------------------------------------------------------------------
;;; Optimize-let Methoden traversiert die Zwischnesprache, und 
;;; optimiert die Let-ausdruecke.
;;;---------------------------------------------------------------------

(defmethod optimize-let ((a-if-form if-form))
  (setf (?pred a-if-form) (optimize-let (?pred a-if-form)))
  (setf (?then a-if-form) (optimize-let (?then a-if-form)))
  (setf (?else a-if-form) (optimize-let (?else a-if-form)))
  a-if-form)

(defmethod optimize-let  ((a-progn progn-form))
  (setf (?form-list a-progn)
        (mapcar #'optimize-let (?form-list a-progn)))
  a-progn)

(defmethod optimize-let  ((a-let/cc let/cc-form))
  (setf (?body a-let/cc) (optimize-let (?body a-let/cc)))
  a-let/cc)

(defmethod optimize-let ((a-switch-form switch-form))
  (setf (?form a-switch-form) (optimize-let a-switch-form))
  (setf (?case-list  a-switch-form)
        (mapcar #'optimize-let (?case-list a-switch-form)))
  (setf (?otherwise a-switch-form)
        (optimize-let (?otherwise a-switch-form)))
  a-switch-form)

(defmethod optimize-let ((a-labeled-form labeled-form))
  (setf (?value a-labeled-form) (optimize-let (?value a-labeled-form)))
  (setf (?form  a-labeled-form) (optimize-let (?form  a-labeled-form)))
  a-labeled-form)

(defmethod optimize-let ((a-tagbody tagbody-form))
  (setf (?first-form a-tagbody) (optimize-let (?first-form a-tagbody)))
  (dolist (one-tagged-form (?tagged-form-list a-tagbody))
    (setf (?form one-tagged-form)  (optimize-let (?form one-tagged-form))))
  a-tagbody)

(defmethod optimize-let ((a-class-def class-def))
  (dolist (one-slot-desc (?slot-descr-list a-class-def))
    (setf (?initform one-slot-desc)
          (optimize-let (?initform one-slot-desc))))
  a-class-def)


(defmethod optimize-let ((a-slot-descr slot-desc))
  (setf (?initform a-slot-descr) (optimize-let (?initform a-slot-descr)))
  a-slot-descr)

(defmethod optimize-let ((a-app app))
  (setf (?arg-list a-app) 
        (mapcar #'optimize-let (?arg-list a-app)))
  a-app)

(defmethod optimize-let ((a-mv-lambda mv-lambda))
  (setf (?params a-mv-lambda) (optimize-let (?params a-mv-lambda)))
  (setf (?body   a-mv-lambda) (optimize-let (?body   a-mv-lambda)))
  (setf (?arg    a-mv-lambda) (optimize-let (?arg    a-mv-lambda)))
  a-mv-lambda)

(defmethod optimize-let ((a-labels labels-form))
  (setf (?fun-list a-labels) (mapcar #'optimize-a-fun (?fun-list a-labels)))
  (setf (?body     a-labels) (optimize-let (?body     a-labels)))
  a-labels)

(defmethod optimize-let ((a-defined-fun defined-fun))
  
  a-defined-fun)

(defmethod optimize-let ((a-let-form let*-form))
  (let ((init-list    (?init-list a-let-form))
        (var-list     (?var-list  a-let-form))
        (*let-effect* (empty-effect)))
    
    ;; Jetzt werden zunaechst die Init-formen untersucht,
    ;; und eventuell let-formen optimiert.
    ;;---------------------------------------------------
    (setf (?init-list a-let-form) 
          (mapcar #'optimize-let (?init-list a-let-form)))
    
    ;;Nun werden die Let-formen in dem  Rumpf der Let-form optimiert.
    ;;---------------------------------------------------------------
    (setf (?body a-let-form) (optimize-let (?body a-let-form)))
    
    ;; Hier passiert die Optimierung dieses Let-konstrukts.
    ;;-----------------------------------------------------
    (effect-of-form a-let-form *let-effect*)
    (if (and (null var-list)
             (null init-list))
        
        ;; (Let*  ()() (form1 ... formN)) --> (progn form1... formN)
        ;;----------------------------------------------------------
        (progn (setq *eliminated-lets* (1+ *eliminated-lets*))
               (setq a-let-form (optimize-0 a-let-form)))
        
        ;; Eventuell wird VAR durch INIT-FORM substituiert.
        ;;-------------------------------------------------
        (let (var init-form)
          (dotimes (i  (length (?var-list a-let-form)))
            (setq var        (nth  i (?var-list a-let-form)))
            (setq init-form  (nth  i (?init-list a-let-form)))
            (unless (dynamic-p var)
              (setq a-let-form 
                    (optimize-1 var init-form  a-let-form))))
          
          ;; Wenn eine Variable nicht mehr referenziert ist,
          ;; wird sie und die dazugehoerige Init-form aus 
          ;; var-list bzw aus init-form-list entfernt.
          ;;------------------------------------------------
          (setq a-let-form (delete-unref-var a-let-form))
          (if  (and (null (?var-list  a-let-form))
                    (null (?init-list a-let-form)))
               
               ;; (Let*  ()() (form1 ... formN)) --> (progn form1... formN)
               ;;----------------------------------------------------------
               (progn 
                 (setq *eliminated-lets* (1+ *eliminated-lets*))
                 (setq a-let-form (optimize-0 a-let-form)))
               a-let-form)))
    a-let-form))

(defmethod optimize-let ((any-thing-else t))
  any-thing-else)


(defun optimize-0 (let-form)
  (make-instance 'progn-form :form-list (list  (?body let-form))))

;;;-----------------------------------------------------------------------
;;; Optimize-1 untersucht die Init-form einer Variablen, und entscheidet
;;; in Abhaengigkeit der moeglichen Effekte des Rumpfes des Let-
;;; konstrukts, ob eine Substitution aller Vorkommen der Variablen 
;;; durch die Init-form moeglich ist.
;;; Falls nicht alle Vorkommen der Variablen substituiert werden koennen,
;;; wird keine Substitution gemacht.
;;;-----------------------------------------------------------------------

(defun optimize-1 (var init-form let-form)
  (when (and (eql 0 (?read var))
             (eql 1 (?write var)))
    (incf *vars-bound-but-not-used*))
  (let ((init-form-effect (empty-effect)))
    (cond 
      ((var-ref-p init-form)
       (if (subst-with-var-is-pemissilble var init-form)
           (update-let var init-form let-form)
           let-form))
      ((and (equal 1  (?write var))
            (equal 1 (?read var)))
       (effect-of-form init-form init-form-effect)
       (if (only-alloc init-form-effect)
           (update-let var init-form let-form)
           (if (simple-app init-form)
               (if (subst-with-app-permissible var init-form)
                   (update-let var init-form let-form)
                   let-form)
               (if (equal (empty-effect) init-form-effect)
                   (update-let var init-form let-form)
                   let-form))))
      (t  let-form))))

;;;------------------------------------------------------------------------
;;; 
;;;------------------------------------------------------------------------
(defun subst-with-var-is-pemissilble (var init-var)
  
  (and (not (or (and (listp  (?write-list *let-effect*))
                     (member var (?write-list *let-effect*)))
                (equal :unknown (?write-list *let-effect*))))
       (not (or (and 
                 (listp (?write-list *let-effect*))
                 (member (?var init-var) 
                         (?write-list *let-effect*)))
                (equal :unknown (?write-list *let-effect*))))
       (equal  1 (?write (?var init-var)))))


;;;------------------------------------------------------------------------
;;; 
;;;------------------------------------------------------------------------
(defun subst-with-app-permissible (var init-app)
  (let* ((functional (?form    init-app))
         (arg-list   (?arg-list init-app))
         (fun-effect (get-effect functional))
         (read-vars  (mapcar #'?var (remove-if-not #'var-ref-p arg-list))))
    
    (and (not (dynamic-p var))
         (not (or (and (listp  (?write-list *let-effect*))
                       (member var (?write-list *let-effect*)))
                  (equal :unknown (?write-list *let-effect*))))
         (fun-p (?form init-app))
         (only-alloc fun-effect)
         (not-destructive *let-effect*)
         (let ((write-vars (?write-list *let-effect*)))
           (not (or (equal :unknown write-vars)
                    (and (listp write-vars)
                         (intersection read-vars  write-vars))))))))

(defun simple-app (form)
  (if (app-p form)
      (and 
       (all-are-side-effect-free (?arg-list form))
       (if (fun-p (?form form))
           (only-alloc (get-effect (?form form)))
           nil))))



(defun all-are-side-effect-free (liste)
  (if (endp liste)
      T
      (if (not (has-no-side-effect (car liste)))
          nil
          (all-are-side-effect-free (cdr liste)))))


(defun only-alloc (effect)
  (and (null (?read-list  effect))
       (null (?write-list effect))
       (not-destructive   effect)))

;;;------------------------------------------------------------------------
;;;
;;;------------------------------------------------------------------------

(defun update-let (var init-form let-form)
  
  ;; Update der globalen Variablen zur Statistics.
  ;;----------------------------------------------

  ;; Die Substitution wird zunaechst bei den 
  ;; Initformen der restlichen Bindungen der
  ;; let-form vorgenommen.
  ;;-------------------------------------------
  (dotimes (i    (length (?init-list let-form)))
    (setf (nth i (?init-list let-form))
          (subst-var var  init-form (nth i (?init-list let-form)))))
  
  ;; Nun wird in dem Rumf substituiert.
  ;;-----------------------------------
  (setf (?body let-form) (subst-var  var init-form  (?body let-form)))
  let-form)

;;;-------------------------------------------------------------------------
;;; DO-NOT-INTERFERE bestimmt ob es Konflikte zwischen zwei Effekten
;;; gibt, also ein READ/WRITE Konflikt.
;;;-------------------------------------------------------------------------

(defun do-not-interfere (effect1 effect2)
  (if (and (listp (?read-list  effect1))
           (listp (?write-list effect2)))
      (null (intersection (?read-list  effect1)
                          (?write-list effect2)))
      nil))

;;;-------------------------------------------------------------------------
;;; Diese  Funktion fragen ab, ob eine Form auf dem Heap etwas alloziiert,
;;; also eventuell die "EQ-SEMANTIK" beeiflusst .
;;;- -----------------------------------------------------------------------
(defun no-alloc-effect (effect)
  (not (or (eq :alloc      (?data-effects effect))
           (eq :alloc-dest (?data-effects effect)))))

;;;-------------------------------------------------------------------------
;;; Diese Funktion fragt ab, ob eine Form dessen Seiteneffekt 'Effect' ist
;;; destruktiv Daten veraendert.
;;;-------------------------------------------------------------------------
(defun not-destructive (effect)
  (not (or (equal (?data-effects effect) :dest)
           (equal (?data-effects effect) :alloc-dest))))

(defun delete-unref-var (let-form)
  (let ((var-list nil)
        (init-list nil)
        var init-form)
    (dotimes (i (length (?var-list let-form)))
      (setq var (nth i (?var-list let-form)))
      (setq init-form (nth i (?init-list let-form)))
      (if  (or (< 0 (?read var)) (< 1 (?write var)))
           
           ;; Die Variable wurde nicht durch Init-form substituiert.
           ;;-------------------------------------------------------
           (progn 
             (setq var-list  (append var-list (list var)))
             (setq init-list (append init-list (list init-form))))
           
           ;; Die Variable wurde Substituiert.
           ;;---------------------------------
           (incf *eliminated-vars*)))
    
    (setf (?var-list let-form) var-list)
    (setf (?init-list let-form) init-list))
  let-form)


;;;-------------------------------------------------------------------------
;;; Hier wird jedes  Vorkommen von "var" in "body" durch "form" ersetzt.
;;;-------------------------------------------------------------------------

(defun subst-var (var form body)
  (typecase body
    (var-ref 
     (if  (equal  (?var body) var)
          (progn 
            (decf (?read (?var body )))
            (incf *subst-number*)
            (cond ((var-ref-p form) (incf (?read (?var form))))
                  ((app-p form) (dolist (one-arg (?arg-list form))
                                  (when (var-ref-p one-arg)
                                    (incf (?read (?var one-arg))))))
                  (t))
            form)
          body))
    
    (if-form
     (setf (?pred body ) (subst-var var form (?pred body)))
     (setf (?then body ) (subst-var var form (?then body)))
     (setf (?else body ) (subst-var var form (?else body)))
     body)
    
    (setq-form
     (setf (?form body) (subst-var var form (?form body)))
     body)
    (progn-form
     
     (let ((form-list nil))
       (dolist (one-form (?form-list body))
         (setq one-form (subst-var var form one-form))
         (setq form-list (append form-list (list one-form))))
       (setf (?form-list body) form-list))
     body)
    
    (tagbody-form 
     (setf (?first-form  body ) (subst-var var form (?first-form body)))
     (let ((tagged-list nil))
       (dolist (one-tagged-form (?tagged-form-list body))
         (setf (?form one-tagged-form )
               (subst-var var form (?form one-tagged-form)))
         (setq tagged-list (append tagged-list (list one-tagged-form))))
       (setf (?tagged-form-list body) tagged-list))
     body)
    
    (switch-form 
     (setf (?form body) (subst-var var form (?form body)))
     (let ((case-list nil))
       (dolist (one-labeled-form (?case-list body))
         (setf one-labeled-form (subst-var var form one-labeled-form))
         (setq case-list (append case-list (list one-labeled-form))))
       (setf (?case-list body) case-list))
     (setf (?otherwise body) (subst-var var form (?otherwise body)))
     body)
    
    (labeled-form 
     (setf (?form body) (subst-var var form  (?form body)))
     body)
    
    (let/cc-form
     (setf (?body body) (subst-var var form (?body body)))
     body)
    
    (let*-form
     (let ((init-list nil))
       (dolist (one-init-form (?init-list body))
         (setq one-init-form (subst-var var form one-init-form))
         (setq init-list (append init-list 
                                 (list  one-init-form))))
       (setf (?init-list body) init-list))
     (setf (?body body) (subst-var var form (?body body)))
     body)
    
    
    (labels-form 
     (dolist (one-fun (?fun-list body))
       (setf (?params one-fun)  (subst-var var form (?params one-fun)))
       (setf (?body one-fun)    (subst-var var form (?body one-fun))))
     (setf (?body body) (subst-var var form (?body body)))
     body)
    
    (defined-fun 
        body)
     
    (params 
     (when (?opt-list body)
       (let ((opt-list nil))
         (dolist (one-opt-param (?opt-list body))
           (setq one-opt-param (subst-var var form one-opt-param))
           (setq opt-list (append opt-list (list one-opt-param))))
         (setf (?opt-list body) opt-list)))
     (when (?key-list body)
       (let ((key-list nil))
         (dolist (one-key-param (?key-list body))
           (setq one-key-param (subst-var var form  one-key-param))
           (setq key-list (append key-list (list one-key-param))))
         (setf (?key-list body) key-list)))
     body)
    
    (opt
     (setf (?init body) (subst-var var form (?init body)))
     body)
    
    (app 
     (unless (fun-p (?form body))
       (setf (?form body) (subst-var var form (?form body))))
     (let ((arg-list nil))
       (dolist (one-arg (?arg-list body))
         (setq arg-list (append arg-list 
                                (list (subst-var  var form one-arg)))))
       (setf (?arg-list body)  arg-list))
     body)
    
    (mv-lambda 
     (setf (?params body) (subst-var var form (?params body)))
     (setf (?body   body) (subst-var var form (?body   body)))
     (setf (?arg    body) (subst-var var form (?arg    body)))
     body)
    
    (class-def 
     (dolist (one-slot-descr (?slot-descr-list body))
       (setf (?initform one-slot-descr)
             (subst-var var form (?initform one-slot-descr))))
     body)
    
    (slot-desc
     (setf (?initform body) (subst-var var form (?initform body)))
     body)
    (t body)))


(defun side-effect-free (form)
  (or (simple-constant-p form)
      (fun-p form)
      (and (app-p form)
           (or (and (fun-p form)
                    (equal (empty-effect) (get-effect form)))
               (and (labels-form-p form)
                    (fun-p (?body form))
                    (equal (empty-effect) (get-effect (?body form)))))
           (all-are-side-effect-free (?arg-list form)))))



(defun effect-of-form (form effect &optional (but-form nil))
  (typecase form 
    (if-form
     (unless (equal but-form form)
       (effect-of-form (?pred form) effect but-form)
       (effect-of-form (?then form) effect but-form)
       (effect-of-form (?else form) effect but-form)))
    (progn-form
     (unless (equal but-form form)
       (dolist (one-form (?form-list form))
         (effect-of-form one-form effect but-form))))


    (let*-form
     (unless (equal but-form form)
       (effect-of-form (?body form) effect but-form)
       (dolist (one-init-form (?init-list form))
         (effect-of-form one-init-form effect but-form))))

    (var-ref 
     (unless (equal but-form form)
       (when (listp (?read-list effect))
         (pushnew (?var form) (?read-list effect)))))

    (tagbody-form 
     (unless (equal but-form form)
       (effect-of-form (?first-form  form) effect but-form)
       (dolist (tagged-form (?tagged-form-list form))
         (effect-of-form (?form tagged-form) effect but-form))))

    (switch-form 
     (unless (equal but-form form)
       (effect-of-form (?form form) effect but-form)
       (dolist (one-labeled-form (?case-list form))
         (effect-of-form one-labeled-form effect but-form))
       (effect-of-form (?otherwise form) effect but-form)))
    
    (labeled-form
     (unless (equal but-form form)
       (effect-of-form (?value form) effect but-form)
       (effect-of-form (?form form) effect  but-form)))

    (let/cc-form 
     (unless (equal but-form form)
       (effect-of-form (?body form) effect but-form)))
    
    (app
     (unless (equal but-form form)
       (union-all-effects  effect effect (get-effect form))))

    (labels-form 
     (unless (equal but-form form)
       (dolist (one-local-fun (?fun-list form))
         (unless (equal one-local-fun but-form)
           (union-all-effects effect effect (get-effect one-local-fun))))
       (effect-of-form (?body form) effect but-form)))

    (mv-lambda 
     (unless (equal but-form form)
       (effect-of-form (?params form) effect but-form)
       (effect-of-form (?body form)   effect but-form)
       (effect-of-form (?arg form)    effect but-form)))
    (setq-form
     (unless (equal but-form form)
       (effect-of-form (?form form) effect but-form)
       (let ((loc (?location form)))
         (when (listp (?write-list effect))
           (pushnew (?var loc) (?write-list effect))))))
    
    (class-def
     (dolist (one-slot-descr (?slot-descr-list form))
       (effect-of-form (?initform one-slot-descr)  effect but-form)))

    (fun 
     (unless (equal form but-form)
       (union-all-effects effect effect (get-effect form))))
    
    (t (empty-effect))))

(defun unify-effect (effect1 effect2 effect3)
  (setf (?read-list effect1)
        (unify-lists (?read-list effect2 ) (?read-list effect3)))
  (setf (?write-list effect1)
        (unify-lists (?write-list effect2) (?write-list effect3))))

;;;--------------------------------------------------------------------------
(provide "optimize")
