;;;-----------------------------------------------------------------------------
;;; Projekt  : APPLY - A Practicable And Portable Lisp Implementation
;;;            ------------------------------------------------------
;;; Funktion : Funktionen zur Steuerung der Optimierungsdurchlaeufe.
;;;            Optimierungen eines einzelnen Ausdrucks oder rekursiv eines
;;;            Ausdrucks und aller seiner Komponenten.
;;;
;;; $Revision: 1.4 $
;;; $Log: optimain.lisp,v $
;;; Revision 1.4  1993/07/23  09:39:07  hk
;;; *optimize-verbosity* auf 1 (wg. tomain)
;;;
;;; Revision 1.3  1993/07/08  10:48:30  jh
;;; Einfache Optimierungen, die auf der Seiteneffektanalyse beruhen (seomain)
;;; eingebaut.
;;;
;;; Revision 1.2  1993/06/30  16:41:15  hk
;;; Schreibfehler in (optimize-parts mv-lambda) behoben.
;;;
;;; Revision 1.1  1993/06/30  15:22:53  jh
;;; Initial revision
;;;
;;;-----------------------------------------------------------------------------

(in-package "CLICC")
(require "subst")
(require "tomain")
(require "simplifier")
(require "seomain")

;;------------------------------------------------------------------------------
;; Globale Variablen zum Abschalten einzelner Optimierungen.
;;------------------------------------------------------------------------------

(defvar *no-subst* nil)
(defvar *no-to* nil)
(defvar *no-seo* nil)
(defvar *no-simp* nil)

;;------------------------------------------------------------------------------
;; Globale Variablen, Konstanten und Funktionen fuer die Statistik.
;;------------------------------------------------------------------------------

(defvar *optimize-verbosity* 1)
(defvar *optimize-statistics* nil)

(defconstant statistics-output
  '((then-optis . "~D times then case eliminated.")
    (else-optis . "~D times else case eliminated.")))

(defun write-optimize-statistics ()
  (when (> *optimize-verbosity* 0)
    (setq *optimize-statistics* (sort *optimize-statistics* #'> :key #'cdr))
    (dolist (counter *optimize-statistics*)
      (clicc-message (cdr (assoc (car counter) statistics-output))
                     (cdr counter)))))

(defun inc-stats (stats-id)
  (let ((assoc (assoc stats-id *optimize-statistics*)))
    (if assoc
        (incf (cdr assoc))
        (push (cons stats-id 1) *optimize-statistics*))))

;;------------------------------------------------------------------------------
;; optimize-field optimiert einen Ausdruck und schreibt das Ergebnis an dieselbe
;; Stelle zurueck.
;;------------------------------------------------------------------------------

(defmacro optimize-field (field)
  `(setf ,field (optimize-form ,field)))

;;------------------------------------------------------------------------------
;; optimize-module optimiert die im Modul definierten Funktionen sowie die
;; toplevel-forms.
;;------------------------------------------------------------------------------

(defun optimize-module (a-module)
  (optimize-fun-def-list (?all-global-funs a-module)))

(defun do-optimization ()
  (let (*optimize-statistics*)
    (clicc-message "Optimization")
    (optimize-module *module*)
    (write-optimize-statistics)))

;;------------------------------------------------------------------------------

(defmethod optimize-fun-def ((a-simple-fun simple-fun))
  (let ((*current-fun* (?symbol a-simple-fun))
        (*result-used* t))
    (optimize-params (?params a-simple-fun))
    (optimize-field (?body a-simple-fun))))

(defun optimize-fun-def-list (fun-def-list)
  (mapc #'optimize-fun-def fun-def-list))

(defun optimize-form-list (form-list)
  (unless (endp form-list)
    (optimize-field (first form-list))
    (optimize-form-list (rest form-list))))

(defun optimize-params (params)
  (mapc #'optimize-opt/key (?opt-list params))
  (mapc #'optimize-opt/key (?key-list params)))

(defun optimize-opt/key (opt/key)
  (optimize-field (?init opt/key)))

;;------------------------------------------------------------------------------
;; optimize-form optimiert einen Zwischensprachausdruck sowie rekursiv seine
;; Teilausdruecke.
;;------------------------------------------------------------------------------

(defmethod optimize-form ((a-form form))
  ;; Wegen der Ersetzungen muessen die Bestandteile eines Zwischensprachknotens
  ;; vor dem Zwischensprachknoten selbst analysiert werden. (Da noch Ersetzungen
  ;; innerhalb der form einer setq-form mit den alten Bindungen moeglich sind,
  ;; waehrend dies nachher fuer die betroffene Variable nicht mehr erlaubt ist.)
  (optimize-1form (optimize-parts a-form)))

(defmethod optimize-form ((a-cont cont))
  a-cont)

;;------------------------------------------------------------------------------
;; optimize-1form optimiert einen Zwischensprachausdruck, ohne seine Teil-
;; ausdruecke zu optimieren. optimize-parts optimiert die Teilausdruecke eines
;; Zwischensprachausdrucks. (Aendert sich durch eine der Teiloptimierungen die
;; Identitaet des Zwischensprachobjekts, so ist diese Teiloptimierung dafuer
;; verantwortlich, dass ein vollstaendig optimierter Zwischensprachausdruck
;; zurueckgeliefert wird.
;;------------------------------------------------------------------------------

(defmethod optimize-1form ((a-form form))
  (let ((new-form a-form))
    (unless *no-subst*
      (setq new-form (subst-1form a-form))
      ;; Hat sich die Identitaet geaendert, so wird der erhaltene
      ;; Zwischensprachausdruck zurueckgeliefert.
      (unless (eq new-form a-form)
        (return-from optimize-1form new-form)))
    (unless *no-to*
      (setq new-form (to-1form a-form))
      (unless (eq new-form a-form)
        (return-from optimize-1form new-form)))
    (unless *no-seo*
      (setq new-form (seo-1form a-form))
      (unless (eq new-form a-form)
        (return-from optimize-1form new-form)))
    (unless *no-simp*
      (setq new-form (simplify-1form a-form)))
    new-form))

(defmethod optimize-parts ((a-form form))
  a-form)

(defmethod optimize-parts ((a-setq-form setq-form))
  (let ((*result-used* t))
    (optimize-field (?form a-setq-form)))
  a-setq-form)

(defmethod optimize-parts ((a-progn-form progn-form))
  (let ((forms (?form-list a-progn-form)))
    (let ((*result-used* nil))
      (optimize-form-list (butlast forms)))
    (when forms
      (optimize-field (first (last forms)))))
  a-progn-form)

(defmethod optimize-parts ((an-if-form if-form))
  (let ((*result-used* t))
    (optimize-field (?pred an-if-form)))
  (let ((subst-pred *SUBSTITUTION*)
        subst-then
        subst-else)
    (optimize-field (?then an-if-form))
    (setq subst-then *SUBSTITUTION*
          *SUBSTITUTION* subst-pred)
    (optimize-field (?else an-if-form))
    (setq subst-else *SUBSTITUTION*)
    (setq *SUBSTITUTION*
          (restrict-map-to subst-pred
                           (intersection subst-then subst-else :key #'car))))
  an-if-form)

(defmethod optimize-parts ((an-app app))
  (let ((*result-used* t))
    (optimize-field (?form an-app))
    (optimize-form-list (?arg-list an-app)))
  an-app)

(defmethod optimize-parts ((a-switch-form switch-form))
  (let ((*result-used* t))
    (optimize-field (?form a-switch-form)))
  (optimize-form-list (?case-list a-switch-form))
  (optimize-field (?otherwise a-switch-form))
  a-switch-form)

(defmethod optimize-parts ((a-let*-form let*-form))
  
  ;; Hier wird ermittelt, welche der in der let*-form gebundenen Variablen
  ;; durch ihren Wert ersetzt werden sollen. Dies muss leider zwischen den
  ;; Optimierungen der init-forms stattfinden und kann daher nicht in
  ;; subst.lisp stehen.
  ;;----------------------------------------------------------------------
  (let (actual-subst)
    (let ((*SUBSTITUTION* *SUBSTITUTION*))
      (let ((*result-used* t))
        (setf (?init-list a-let*-form)
              (mapcar #'(lambda (var init-form)
                          (setq init-form (optimize-form init-form))
                          (when (and (local-static-p var)
                                     (copy-is-eq-p init-form))
                            (push (cons var init-form) *SUBSTITUTION*))
                          init-form)
                      (?var-list a-let*-form) (?init-list a-let*-form))))
      (optimize-field (?body a-let*-form))
      (setq actual-subst *SUBSTITUTION*))

    ;; Nach dem Optimieren der let*-form ist wieder die alte Substitution
    ;; gueltig, vermindert um die inzwischen geloeschten Bindungen.
    ;;-------------------------------------------------------------------
    (setq *SUBSTITUTION* (restrict-map-to *SUBSTITUTION* actual-subst)))
  a-let*-form)

(defmethod optimize-parts ((a-labels-form labels-form))
  ;; In und nach einer labels-form duerfen keine Ersetzungen vorgenommen werden.
  ;; (Bei Beachtung der Seiteneffekte kann hier genauer gearbeitet werden.)
  (clear-substitution)
  (optimize-fun-def-list (?fun-list a-labels-form))
  (optimize-field (?body a-labels-form))
  a-labels-form)


(defmethod optimize-parts ((a-let/cc-form let/cc-form))
  (optimize-field (?body a-let/cc-form))
  a-let/cc-form)

(defmethod optimize-parts ((a-tagbody-form tagbody-form))
  (let ((tagged-forms (?tagged-form-list a-tagbody-form)))
    (when (?first-form a-tagbody-form)
      (let ((*result-used* (and (null tagged-forms) *result-used*)))
        (optimize-field (?first-form a-tagbody-form))))
  
  ;; Bis zur ersten Sprungmarke durfte optimiert werden.
  ;;----------------------------------------------------
  (clear-substitution)
  (let ((*result-used* nil))
    (mapc #'(lambda (a-tagged-form) (optimize-field (?form a-tagged-form)))
          (butlast tagged-forms)))
  (when tagged-forms
    (optimize-field (?form (first (last tagged-forms))))))
  a-tagbody-form)

(defmethod optimize-parts ((a-tagged-form tagged-form))
  a-tagged-form)

(defmethod optimize-parts ((a-mv-lambda mv-lambda))
  (let ((*result-used* t))
    (optimize-field (?arg a-mv-lambda))
    (optimize-params (?params a-mv-lambda)))
  (optimize-field (?body a-mv-lambda))
  a-mv-lambda)

;;------------------------------------------------------------------------------
(provide "optimain")
