;;;-----------------------------------------------------------------------------
;;; Copyright (C) 1993 Christian-Albrechts-Universitaet zu Kiel, Germany
;;;-----------------------------------------------------------------------------
;;; Projekt  : APPLY - A Practible And Portable Lisp Implementation
;;;            ----------------------------------------------------
;;; Funktion : search-fun-calls ermittelt, zu welchen Funktionen die form einer
;;;            app-form ausgewertet werden kann. Ausserdem wird zu jeder
;;;            Funktion bestimmt, in welchen Funktionen sie aufgerufen werden
;;;            und ob alle Aufrufstellen bekannt sind.
;;;
;;; $Revision: 1.11 $
;;; $Log: appfuns.lisp,v $
;;; Revision 1.11  1993/06/17  08:00:09  hk
;;; Copright Notiz eingefuegt
;;;
;;; Revision 1.10  1993/05/19  14:43:57  uho
;;; Aenderungen fuer CLISP, wegen seines seltsamen IN-PACKAGE Verahltens,
;;; eingebaut.
;;;
;;; Revision 1.9  1993/05/14  10:53:32  jh
;;; Fehler bei der Initialisierung des unknown-caller-Slots behoben.
;;;
;;; Revision 1.8  1993/04/22  11:11:37  hk
;;; Bearbeitung von (?toplevel-forms *module*) eingebaut.
;;;
;;; Revision 1.7  1993/04/22  08:56:59  jh
;;; Fehler beseitigt.
;;;
;;; Revision 1.6  1993/04/20  14:34:26  jh
;;; Auf die Annotation special-caller von special-sys-fun angepasst.
;;;
;;; Revision 1.5  1993/04/20  12:27:54  jh
;;; search-fun-calls setzt jetzt auch die called-by-slots von Funktionen.
;;; Fehler im Zusammenhang mit *other-funs* behoben.
;;;
;;; Revision 1.4  1993/04/14  07:16:15  kl
;;; Schreibfehler behoben.
;;;
;;; Revision 1.3  1993/04/07  15:42:42  hk
;;; ?form in (sfc-form tagbody-form) war vergessen worden.
;;;
;;; Revision 1.2  1993/04/02  09:47:38  kl
;;; provide eingefuegt.
;;;
;;; Revision 1.1  1993/03/31  14:11:14  jh
;;; Initial revision
;;;
;;;-----------------------------------------------------------------------------

(in-package "CLICC") 
#+CLISP (use-package "PCL") 

;;------------------------------------------------------------------------------
(defun search-fun-calls (&key ((:module *module*) *module*))
  (mapc #'(lambda (a-fun) (setf (?called-by a-fun) nil)) (?fun-list *module*))
  (sfc-fun-list (?fun-list *module*))
  (setf (?called-by (?toplevel-forms *module*)) nil)
  (sfc-fun (?toplevel-forms *module*))
  (mapc #'sfc-class (?class-def-list *module*))
  (dolist (a-named-const (?named-const-list *module*))
    (when (?exported a-named-const)
      (sfc-topform a-named-const)))
  ;; Da Symbole jederzeit auch ueber 'read' angewendet werden koennen, sind
  ;; von eventuell in ihnen enthaltenen Funktionen nicht alle Aufrufstellen
  ;; bekannt.
  (dolist (a-sym (?sym-list *module*))
    (sfc-topform (?constant-value a-sym))))

(defvar *current-context* nil)
(defvar *other-funs* nil)

(defun sfc-fun-list (fun-list)
  (mapc #'initialize-unknown-caller fun-list)
  (mapc #'sfc-fun fun-list))

(defun sfc-fun (a-fun)
  (let ((*current-context* a-fun))
    (sfc-params (?params a-fun))
    (sfc-topform (?body a-fun))))

(defmethod initialize-unknown-caller ((a-fun fun))
  (setf (?unknown-caller a-fun) nil))

(defmethod initialize-unknown-caller ((a-global-fun global-fun))
  (setf (?unknown-caller a-global-fun) (?exported a-global-fun)))

(defun sfc-class (a-class-def)
  (let ((*current-context* a-class-def))
    (dolist (a-slot-desc (?slot-descr-list a-class-def))
      (sfc-topform (?initform a-slot-desc)))))

(defun sfc-params (params)
  (dolist (a-opt (?opt-list params))
    (sfc-topform (?init a-opt)))
  (dolist (a-key (?key-list params))
    (sfc-topform (?init a-key))))

(defun sfc-topform (a-form)
  (let ((*other-funs* nil))
    (sfc-form a-form)))

(defmethod sfc-form ((anything T) &optional known-caller)
  (declare (ignore known-caller))
  (setq *other-funs* T)
  nil)

(defmethod sfc-form ((an-app app) &optional known-caller)
  (declare (ignore known-caller))
  (setf (?called-funs an-app)
        (let ((*other-funs* nil)
              (form (?form an-app)))
          (if (and (special-sys-fun-p form) (?special-caller form))
              ;; Bei Iterationsfunktionen sowie apply und funcall interessiert
              ;; nicht die form, sondern der erste Parameter, d.h. die von
              ;; dieser Funktion applizierten Funktionen. Der called-funs-Slot
              ;; aendert in diesem Fall seine Bedeutung entsprechend.
              (prog1
                  (sfc-form (first (?arg-list an-app)) 'known-caller)
                (setf (?other-funs an-app) *other-funs*)
                (mapc #'sfc-topform (rest (?arg-list an-app))))
              (prog1
                  (sfc-form (?form an-app) 'known-caller)
                (setf (?other-funs an-app) *other-funs*)
                (mapc #'sfc-topform (?arg-list an-app))))))
  (setq *other-funs* T)
  nil)

(defmethod sfc-form ((a-var-ref var-ref) &optional known-caller)
  (declare (ignore known-caller))
  (setq *other-funs* T)
  nil)

(defmethod sfc-form ((a-named-const named-const) &optional known-caller)
  (sfc-form (?value a-named-const) known-caller))


;; Symbole, die einen konstanten Wert enthalten, werden bereits in Pass 2 durch
;; diesen ersetzt.
(defmethod sfc-form ((a-sym sym) &optional known-caller)
  (declare (ignore known-caller))
  (setq *other-funs* T)
  nil)

(defmethod sfc-form ((a-structured-literal structured-literal)
                     &optional known-caller)
  (declare (ignore known-caller))
  (sfc-form (?value a-structured-literal)))

(defmethod sfc-form ((an-array array) &optional known-caller)
  (declare (ignore known-caller))
  (let* ((total-size (array-total-size an-array))
         (flat-array (make-array total-size
                                 :displaced-to an-array
                                 :element-type (array-element-type an-array))))
    (dotimes (index total-size)
      (sfc-form (aref flat-array index)))))

(defmethod sfc-form ((a-cons cons) &optional known-caller)
  (declare (ignore known-caller))
  (sfc-form (car a-cons))
  (sfc-form (cdr a-cons)))

(defmethod sfc-form ((a-literal-instance literal-instance)
                     &optional known-caller)
  (declare (ignore known-caller))
  (mapc #'sfc-form (?value-list a-literal-instance)))

(defmethod sfc-form ((a-defined-fun defined-fun) &optional known-caller)
  (cond ((eq known-caller 'known-caller) (pushnew *current-context*
                                                  (?called-by a-defined-fun)))
        ((not known-caller) (setf (?unknown-caller a-defined-fun) T))
        (T nil))
  (list a-defined-fun))

(defmethod sfc-form ((a-fun fun) &optional known-caller)
  (unless known-caller
    (setf (?unknown-caller a-fun) T))
  (list a-fun))

(defmethod sfc-form ((a-setq-form setq-form) &optional known-caller)
  (declare (ignore known-caller))
  (sfc-form (?form a-setq-form)))

(defmethod sfc-form ((a-progn-form progn-form) &optional known-caller)
  (mapc #'(lambda (a-form) (sfc-form a-form 'no-caller))
        (butlast (?form-list a-progn-form)))
  (sfc-form (first (last (?form-list a-progn-form))) known-caller))

(defmethod sfc-form ((an-if-form if-form) &optional known-caller)
  (sfc-form (?pred an-if-form) 'no-caller)
  (nconc
   (sfc-form (?then an-if-form) known-caller)
   (sfc-form (?else an-if-form) known-caller)))

(defmethod sfc-form ((a-switch-form switch-form) &optional known-caller)
  (nconc
   (mapcan #'(lambda (a-form) (sfc-form a-form known-caller))
           (?case-list a-switch-form))
   (sfc-form (?otherwise a-switch-form) known-caller)))

(defmethod sfc-form ((a-labeled-form labeled-form) &optional known-caller)
  (sfc-form (?form a-labeled-form) known-caller))

(defmethod sfc-form ((a-let*-form let*-form) &optional known-caller)
  (mapc #'sfc-topform (?init-list a-let*-form))
  (sfc-form (?body a-let*-form) known-caller))

(defmethod sfc-form ((a-labels-form labels-form) &optional known-caller)
  (sfc-fun-list (?fun-list a-labels-form))
  (sfc-form (?body a-labels-form) known-caller))

(defmethod sfc-form ((a-let/cc-form let/cc-form) &optional known-caller)
  (setf (?unknown-caller (?cont a-let/cc-form)) nil)
  (sfc-form (?body a-let/cc-form) known-caller))

(defmethod sfc-form ((a-cont cont) &optional known-caller)
  (unless known-caller
    (setf (?unknown-caller a-cont) T))
  (list a-cont))

(defmethod sfc-form ((a-tagbody-form tagbody-form) &optional known-caller)
  (let ((tagged-forms (?tagged-form-list a-tagbody-form)))
    (if tagged-forms
        (progn
          (sfc-form (?first-form a-tagbody-form) 'no-caller)
          (mapc #'(lambda (a-form) (sfc-form (?form a-form) 'no-caller))
                (butlast tagged-forms))
          (sfc-form (?form (first (last tagged-forms))) known-caller))
        (sfc-form (?first-form a-tagbody-form) known-caller))))

(defmethod sfc-form ((a-tagged-form tagged-form) &optional known-caller)
  (declare (ignore known-caller))
  nil)

(defmethod sfc-form ((a-mv-lambda mv-lambda) &optional known-caller)
  (sfc-topform (?arg a-mv-lambda))
  (sfc-form (?body a-mv-lambda) known-caller))

;;------------------------------------------------------------------------------
(provide "appfuns")
