;;; -*- Mode: LISP; Syntax: Common-lisp; Package: YY; Base: 10 -*-
;;; edit-text.lisp
;;;
;;;  Copyright (C) 1989,1990,1991 Aoyama Gakuin University
;;;
;;;		All Rights Reserved
;;;
;;; This software is developed for the YY project of Aoyama Gakuin University.
;;; Permission to use, copy, modify, and distribute this software
;;; and its documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notices appear in all copies and that
;;; both that copyright notice and this permission notice appear in 
;;; supporting documentation, and that the name of Aoyama Gakuin
;;; not be used in advertising or publicity pertaining to distribution of
;;; the software without specific, written prior permission.
;;;
;;; This software is made available AS IS, and Aoyama Gakuin makes no
;;; warranty about the software, its performance or its conformity to
;;; any specification. 
;;;
;;; To make a contact: Send E-mail to ida@csrl.aoyama.ac.jp for overall
;;; issues. To ask specific questions, send to the individual authors at
;;; csrl.aoyama.ac.jp. To request a mailing list, send E-mail to 
;;; yyonx-request@csrl.aoyama.ac.jp.
;;;
;;; Authors:
;;;   version 1.4 92/04/21 by t.kosaka

(in-package :yy)

;;; $@M?$($i$l$?J8;z$,4A;z$+$I$&$+%A%'%C%/$9$k(J
(defun kanji-charp (character)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (if character
	  (if (>= (char-code character) #xA1)
		  T
		nil)
	nil))

;;; $@J8;zI}$G:BI8JQ498e$N(Jdx,dy$@$rJV$9(J
(defun return-dx-dy (object char)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-edit-slots
   (font) object
   (let* ((width (if (kanji-charp char)
					  (font-kanji-width font)
					(character-width font char)))
		   (dx (* (fourth (sixth object)) width))
		   (dy (* (fifth (sixth object)) width)))
	 (values (truncate (+ dx 0.555555)) (truncate (+ dy 0.555555))))))

;;; $@M?$($i$l$k%j!<%8%g%s$O!":8>e6y$N:BI87O$G$"$k!#(J
;;; read $@$H(Jread-line$@$N0Y$N%j!<%8%g%s$N3HBg(J
;;; w-region $@%o!<%k%I$N%j!<%8%g%s(J
;;; region $@BP>]$H$J$k%j!<%8%g%s(J
;;; temp-region$@3HBg$9$k%j!<%8%g%s(J($@2?$i$+$N3HBg$,$"$C$?$H$-$NLa$jCM(J)
;;; $@3HBg$,9T$J$o$l$F$$$J$1$l$P!"(JNIL$@$,$+$($k(J
;;; $@3HBg$5$l$k;~$O!"(Jworld$@$N9b$5!"I}$NH>J,$,2C$($i$l$k(J
(defun region-union-expand-text (w-region region temp-region)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-region-slots
   ((w-left left) (w-bottom bottom) (w-right right) (w-top top)
	(w-width width) (w-height height)) w-region
   (with-region-slots 
	((o-left left) (o-bottom bottom)
	 (o-top top) (o-right right) (o-width width) (o-height height)) region
	(let ((set-left w-left) (set-bottom w-bottom) 
		  (set-right w-right)
		  (set-top w-top)
		  (expand-width (round (/ w-width 2)))
		  (expand-height (round (/ w-height 2))))

	  (when (< w-width (+ o-left o-width)) 
			;; $@?7$7$$%j!<%8%g%s$,1&$K$O$_$G$F$$$k(J
			(setf set-right (+ w-left w-width expand-width))
;			(format t "migi : ~a  ~a  Set: ~a ~%" w-width (+ o-left o-width)
;					set-right)
	  )
						  
	  (when (> w-left o-left)
;			(format t "hidari : ~a  ~a ~%" w-left o-left)
		  ;; $@?7$7$$%j!<%8%g%s$,:8$K$O$_$G$F$$$k(J
		  (setf set-left (- w-left 
							(max expand-width
								 (abs (- w-left o-left))))))

	  (when (< w-height (+ o-bottom o-height))
;			(format t "shita : ~a  ~a ~%" w-height (+ o-bottom o-height))
			;; $@?7$7$$%j!<%8%g%s$,2<$K$O$_$G$F$$$k(J
			(setf set-top
				  (+ w-bottom w-width expand-height)))
	  
	  (when (> w-bottom o-bottom)
;			(format t "ue : ~a  ~a ~%" w-bottom o-bottom)
			;; $@?7$7$$%j!<%8%g%s$,>e$K$O$_$G$F$$$k(J
			(setf set-bottom
				  (- w-bottom 
					 (max expand-height
						  (abs (- w-bottom o-bottom))))))
	  
	  (if (or (/= set-right w-right)
			  (/= set-bottom w-bottom)
			  (/= set-top w-top)
			  (/= set-left w-left))
		  (progn
			(with-region-slots
			 ((tl left) (tr right) (tb bottom) (tt top)) temp-region
			 (setf tl set-left tt set-top
				   tr set-right tb set-bottom))
;			(format t "Original left ~a bottom ~a width ~a height~a ~% ~a ~%" 
;					w-left w-bottom w-width w-height temp-region)
			temp-region)
		nil)
	  ))))


;;; $@J8;zNs$NIA2h%W%m%H%3%kH/@8(J
;;; $@2#=q$-MQ(J
(defmethod draw-text-with-matrix-read ((matrix graphic-transform-matrix-non)
									   x y string object)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((fno (font-no (edit-font object)))
		(color (edit-color object))
		(stream (edit-stream object)))
	(with-slots
	 ((start-x world-x-start) (start-y world-y-start) 
	  (tno world-territory-no)) stream
	  (setf x (+ x start-x)
			y (+ y start-y))
	  (yy-protocol-31 tno x y (avialble-operation color)
					  (color-no color) fno string))))


;;; $@J8;zNs$NIA2h%W%m%H%3%kH/@8(J
(defmethod draw-text-with-matrix-read ((matrix graphic-transform-matrix-exec)
									   x y string object)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
	(let ((fno (font-no (edit-font object)))
		  (color (edit-color object))
		  (xtime (round (matrix-x-time matrix)))
		  (ytime (round (matrix-y-time matrix)))
		  (theta 
		   #+:YY2.0
		   (round (* (matrix-theta matrix) 64))
		   #-:YY2.0
		   (if (not (zerop (matrix-theta matrix)))
			   (round (*  (matrix-theta matrix)
								 -64))
			 0))
		  (stream (edit-stream object)))
	  (with-slots
	   ((start-x world-x-start) (start-y world-y-start) 
		(tno world-territory-no)) stream
		(setf x (+ x start-x)
			  y (+ y start-y))
		(yy-protocol-44 tno x y (avialble-operation color)
						(color-no color) fno xtime ytime theta string))))

;;; $@2~9T;~$KNN0h$N@_Dj$r9T$J$&J8;zNsF~NOMQ(J
;;; $@J8;zNs$N=PNO(J
(defun line-feed-op-read (start-x start-y end-x end-y string object)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))

  ;; $@:GBg:G>.$NNN0h@_Dj(J
  (let ((font-height (font-kanji-height (edit-font object)))
		(nx (translate-xy-edit start-x object))
		(ny (translate-xy-edit start-y object))
		(nnx (translate-xy-edit end-x object))
		(nny (translate-xy-edit end-y object)))
	(with-region-slots
	 (left bottom top right) (edit-region1 object)
	 (setf left (min nx nnx)
		   bottom (- (min ny nny) font-height)
		   top (+ (max ny nny) font-height)
		   right (max nx nnx)))
	
	(when (region-union-expand-text 
		   (edit-stream object)
		   (edit-region1 object) (edit-region2 object))
		  (setf (world-region (edit-stream object)) (edit-region2 object)))
	(draw-text-with-matrix-read (edit-matrix object) nx ny string object)
	))


;;; $@2~9T;~$KNN0h$N@_Dj$r9T$J$&J8;zNsF~NOMQ(J
;;; $@J8;zNs$N=PNO(J
(defun line-feed-op-read-tate (start-x start-y end-x end-y string object)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))

  ;; $@:GBg:G>.$NNN0h@_Dj(J
  (let ((font-height (font-kanji-height (edit-font object)))
		(nx start-x) (ny (translate-xy-edit start-y object))
		(nnx end-x) (nny (translate-xy-edit end-y object))
		(fno (font-no (edit-font object)))
		(color (edit-color object))
		(stream (edit-stream object)))
	(with-region-slots
	 (left bottom top right) (edit-region1 object)
	 (setf left (min left nx nnx)
		   bottom (- (min bottom ny nny) font-height)
		   top (+ (max top ny nny) font-height)
		   right (max right nx nnx)))
	
	(when (region-union-expand-text 
		   stream
		   (edit-region1 object) (edit-region2 object))
		  (setf (world-region stream) (edit-region2 object)))

	(with-slots
	 ((start-x world-x-start) (start-y world-y-start) 
	  (tno world-territory-no)) stream
	  (yy-protocol-43 tno (+ nx start-x)  (+ ny start-y) 
					  (avialble-operation color)
					  (color-no color) fno string))
	))

;;; read-line,read$@MQ$NJ8;zNs$N=PNO(J
(defun drawing-text-read (x y string object &optional (end-mode nil))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (if (eq (edit-direction object) :vertical)
	  (drawing-text-read-tate string x y object end-mode)
	(drawing-text-read-yoko string x y object end-mode)))

(defun read-kaigyou-shori (x y line-feed left-m matrix)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((theta (matrix-theta matrix))
		(return-x 0) (return-y 0)
		(orignal-y 0))
	(setf (matrix-theta matrix) (* theta -1))
	(setf orignal-y (second (multiple-value-list
							 (transform-by-matrix-xy-int x y matrix))))
	(setf (matrix-theta matrix) theta)
	(multiple-value-setq
	 (return-x return-y)
	 (transform-by-matrix-xy-int left-m (+ orignal-y line-feed)
								 matrix))
	(values return-x return-y)))
	 
;;; $@J8;z%j%9%H=*N;$^$GJ8;zNs$r=PNO$9$k(J
;;; $@%o!<%k%I$O!"<+F0E*$K9-$,$k(J
;;; $@La$jCM$O!"%+!<%=%k0LCV$G$"$k!#(J($@2#=q$-MQ(J)
;;; $@2#=q$-J8;zNs$NI=<((J read-lline read $@MQ(J
;;; x y$@$O!"%+!<%=%k0LCV(J
(defun drawing-text-read-yoko (string x y object end-mode)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let* ((start 0) (end 0) (endx x) (out-put nil)
		 (font-width 0)	 (truncate-flg nil)
		 (real-stx x) (real-sty y)
		 (real-endx x) (real-endy y)
		 (return-x 0) (return-y 0)
		 (end-mode-flg t)
		 (next-step-x 0) (next-step-y 0)
		 (ch #\a))

	(with-edit-slots
	 ((right-m right-margin) (left-m left-margin)
	  line-feed font truncate-width) object

	  ;; $@4A;z$NI}$r5a$a$k!#(J
	  (setf font-width (font-kanji-width font))

	  ;; $@J8;z$r7W;;(J
	  (do ((char-list string (cdr char-list)))
		  ((null char-list))

		 ;; $@=PNO@)8f%U%i%0(JOFF
		 (setf out-put nil)

		 ;; $@0lJ8;z%+%&%s%H%"%C%W(J
		 (incf end)
		 ;;  $@0lJ8;z<h$j=P$9(J
		 (setf ch (take-out-car-moji char-list))

		;; $@%H%i%s%1%$%H=hM}(J
		 (if (and truncate-flg (/= 10 (char-code ch))
				 (/= 13 (char-code ch)))
			(incf start)
		  (setf truncate-flg nil))

		(multiple-value-setq  ;; $@J8;zI}$G<!$NJ8;z$N%9%F%C%W$r5a$a$k(J
		 (next-step-x next-step-y)
		 (return-dx-dy object ch))

		(cond 		;; $@J8;z$NH=CG(J
		 ((kanji-charp ch) 		 ;; $@4A;z$N>l9g(J
		  (incf endx font-width)  (incf real-endx next-step-x)
		  (incf real-endy next-step-y)
		  (when 
		   (>= endx right-m)  ;; $@%^!<%8%s$r1[$($?(J
		   (if (null truncate-width)
			   (progn
				 (decf real-endx next-step-x) (decf real-endy next-step-y)
				 (line-feed-op-read real-stx real-sty real-endx real-endy
									(subseq (coerce string 	'simple-string)
											start (- end 1))
									object)
				 (multiple-value-setq
				  (real-stx real-sty)
					   (read-kaigyou-shori 
						real-endx real-endy 
						line-feed left-m (car (sixth object))))

				 (push ch char-list)

				 (setf endx left-m real-endx real-stx real-endy real-sty
					   start (- end 1) end start
					   out-put T)
				 )
			 (setf truncate-flg T)))

		  (when (and end-mode end-mode-flg)
				(setf return-x real-endx
					  return-y real-endy
					  end-mode-flg nil))
			 
		  #+(or LUCID (and EXCL (not ICS)))
  		    (setf char-list (cdr char-list)
				  end (1+ end))
			)

		 ((or (= (char-code ch) 10)  ;;; $@2~9T=hM}(J
			  (= (char-code ch) 13))

		  (line-feed-op-read real-stx real-sty real-endx real-endy
							 (subseq (coerce string  'simple-string)
									 start (- end 1))
							 object)
		  (multiple-value-setq
           (real-stx real-sty)
		   (read-kaigyou-shori real-endx real-endy 
							   line-feed left-m (car (sixth object))))
		  (setf endx left-m real-endx real-stx real-endy real-sty
				start end end start out-put t)
		  (if (and end-mode end-mode-flg)
			  (setf return-x real-endx
					return-y real-endy
					end-mode-flg nil))
		  )
		 (t ;; $@1Q;z$N>l9g(J
		  (incf endx (character-width font ch))
		  (incf real-endx next-step-x) (incf real-endy next-step-y)
 		  (when (>= endx right-m)                ;; $@%^!<%8%s$r1[$($?(J
				(decf endx (character-width font ch))
				(decf real-endx next-step-x)
				(decf real-endy next-step-y)
				(if (null truncate-width)
					(progn 
					  (line-feed-op-read real-stx real-sty real-endx real-endy
								  (subseq (coerce string  'simple-string)
										  start (- end 1)) object)
					  (multiple-value-setq
					   (real-stx real-sty)
					   (read-kaigyou-shori 
						real-endx real-endy 
						line-feed left-m (car (sixth object))))
					  (push ch char-list)
					  (setf endx left-m real-endx real-stx real-endy real-sty
							start (- end 1) end start out-put t)
					  )
				  (setf truncate-flg T))
				)
		  (if (and end-mode end-mode-flg)
			  (setf return-x real-endx
					return-y real-endy
					end-mode-flg nil))
		  )
		 ))
		
	 ;; $@=PNO%A%'%C%/(J 
	 (unless out-put
			 (line-feed-op-read real-stx real-sty real-endx real-endy
								(subseq (coerce string 'simple-string)
										start end) object))
	 (if end-mode
		 (values return-x return-y)
	   (values real-endx real-endy))
	 )))

;;; $@J8;z%j%9%H=*N;$^$GJ8;zNs$r=PNO$9$k(J
;;; $@%o!<%k%I$O!"<+F0E*$K9-$,$k(J
;;; $@La$jCM$O!"%+!<%=%k0LCV$G$"$k!#(J($@=D=q$-(J)
;;; $@2#=q$-J8;zNs$NI=<((J read-lline read $@MQ(J
;;; x y$@$O!"%+!<%=%k0LCV(J
(defun drawing-text-read-tate (string x y object end-mode)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let* ((start 0) (end 0) (nowx x) (nowy y) 
		 (endx x) (endy y) (out-put nil)
		 (font-height 0) (truncate-flg nil)
		 (return-x 0) (return-y 0)
		 (end-mode-flg t)
		 (ch #\a))

	(with-edit-slots
	 ((top-m top-margin) (bottom-m bottom-margin)
	  line-feed font truncate-height) object

	  ;; $@4A;z$NI}$r5a$a$k!#(J
	  (setf font-height (font-kanji-height font))

	  ;; $@J8;z$r7W;;(J
	  (do ((char-list string (cdr char-list)))
		  ((null char-list))

		 ;; $@=PNO@)8f%U%i%0(JOFF
		 (setf out-put nil)

		 ;; $@0lJ8;z%+%&%s%H%"%C%W(J
		 (incf end)
		 ;;  $@0lJ8;z<h$j=P$9(J
		 (setf ch (take-out-car-moji char-list))

		;; $@%H%i%s%1%$%H=hM}(J
		(if (and truncate-flg (/= 10 (char-code ch))
				 (/= 13 (char-code ch)))
			(incf start)
		  (setf truncate-flg nil))

;		(format t "endx: ~a  endy ~a ~%" endx endy)

		(cond 		;; $@J8;z$NH=CG(J
		 ((kanji-charp ch) 		 ;; $@4A;z$N>l9g(J
		  (incf endy font-height)  
		  (when 
		   (> endy bottom-m)  ;; $@%^!<%8%s$r1[$($?(J
		   (if (null truncate-height)
			   (progn
				 (decf endy font-height) 
				 (line-feed-op-read-tate nowx nowy endx endy
									(subseq (coerce string 	'simple-string)
											start (- end 1))
									object)
				 (push ch char-list)
				 (setf endx (- endx line-feed) endy top-m
					   nowx endx nowy endy start (- end 1) end start
					   out-put T)
				 )
			 (setf truncate-flg T)))

		  (when (and end-mode end-mode-flg)
				(setf return-x endx
					  return-y endy
					  end-mode-flg nil))
			 
		  #+(or LUCID (and EXCL (not ICS)))
  		    (setf char-list (cdr char-list)
				  end (1+ end))
			)

		 ((or (= (char-code ch) 10)  ;;; $@2~9T=hM}(J
			  (= (char-code ch) 13))

		  (line-feed-op-read-tate nowx nowy endx endy
							 (subseq (coerce string  'simple-string)
									 start (- end 1))
							 object)
		  (setf endx (- nowx line-feed) endy top-m
				nowx endx nowy endy
				start end end start out-put t)
		  (if (and end-mode end-mode-flg)
			  (setf return-x endx
					return-y endy
					end-mode-flg nil))
		  )
		 (t ;; $@1Q;z$N>l9g(J
		  (incf endy (character-width font ch))
 		  (when (> endy bottom-m)                ;; $@%^!<%8%s$r1[$($?(J
				(decf endy (character-width font ch))
				(if (null truncate-height)
					(progn 
					  (line-feed-op-read-tate 
					   nowx nowy endx endy
					   (subseq (coerce string  'simple-string)
							   start (- end 1)) object)
					  (push ch char-list)
					  (setf endx (- endx line-feed) endy top-m
							nowx endx nowy endy
							start (- end 1) end start out-put t)
					  )
				  (setf truncate-flg T))
				)
		  (if (and end-mode end-mode-flg)
			  (setf return-x endx
					return-y endy
					end-mode-flg nil))
		  )
		 ))
		
	 ;; $@=PNO%A%'%C%/(J 
	 (unless out-put
			 (line-feed-op-read-tate nowx nowy endx endy
								(subseq (coerce string 'simple-string)
										start end) object))
;	 (format t "Last endx: ~a  endy ~a ~%" endx endy)
	 (if end-mode
		 (values return-x return-y)
	   (values endx endy))
	 )))


;;; read-line,read$@MQ$N%+!<%=%k0LCV$r5a$a$k(J
(defun cursor-position-by-index (x y index string object)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (if (eq (edit-direction object) :vertical)
	  (cursor-position-by-index-tate string index x y object)
	(cursor-position-by-index-yoko string index x y object)))

;;; index$@$G;X<($5$l$k0LCV$^$G$N%+!<%=%k0LCV$r5a$a$k(J
;;; x y$@$O!"=i4|%+!<%=%k0LCV(J
(defun cursor-position-by-index-yoko (string index x y object)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let* ((endx x) (stx x) (sty y) (enx x) (eny y)
		 (start 0) (end 0)
		 (font-width (font-kanji-width (edit-font object)))
		 (truncate-flg nil)
		 (step-x 0) (step-y 0)
		 (ch #\a))

	(with-edit-slots
	 ((right-m right-margin) (left-m left-margin)
	  line-feed font truncate-width) object

	  ;; $@J8;z$r7W;;(J
	  (do ((char-list string (cdr char-list)))
		  ((or (null char-list)
			   (= index end)))

		 ;; $@0lJ8;z%+%&%s%H%"%C%W(J
		 (incf end)
		 ;;  $@0lJ8;z<h$j=P$9(J
		 (setf ch (take-out-car-moji char-list))

		(multiple-value-setq  ;; $@J8;zI}$G<!$NJ8;z$N%9%F%C%W$r5a$a$k(J
		 (step-x step-y)
		 (return-dx-dy object ch))

		;; $@%H%i%s%1%$%H=hM}(J
		(if (and truncate-flg (/= 10 (char-code ch))
				 (/= 13 (char-code ch)))
			(incf start)
		  (setf truncate-flg nil))

		(cond 		;; $@J8;z$NH=CG(J
		 ((kanji-charp ch) 		 ;; $@4A;z$N>l9g(J
		  (incf endx font-width) (incf enx step-x) (incf eny step-y)
		  (when (>= endx right-m)  ;; $@%^!<%8%s$r1[$($?(J
				(decf endx font-width) (decf enx step-x) (decf eny step-y)
				(if (null truncate-width)
					(progn
					  (multiple-value-setq
					   (stx sty)
					   (read-kaigyou-shori 
						enx eny line-feed left-m (car (sixth object))))
					  (push ch char-list)
					  (setf endx left-m  
							start (- end 1) end start))
				  (setf truncate-flg T))
				(setf enx stx eny sty)
		  )
		  #+(or LUCID (and EXCL (not ICS)))
  		    (setf char-list (cdr char-list)
				  end (1+ end))
			)
		 ((or (= (char-code ch) 10)  ;;; $@2~9T=hM}(J
			  (= (char-code ch) 13))
		  (multiple-value-setq
		   (stx sty)
		   (read-kaigyou-shori 
			enx eny line-feed left-m (car (sixth object))))
		  (setf endx left-m start end end start
				enx stx eny sty)
		  )
		 (t ;; $@1Q;z$N>l9g(J
		  (incf endx (character-width font ch))
		  (incf enx step-x) (incf eny step-y)
 		  (when (>= endx right-m)                ;; $@%^!<%8%s$r1[$($?(J
				(decf endx (character-width font ch))
				(decf enx step-x) (decf eny step-y)
				(if (null truncate-width)
					(progn
					  (multiple-value-setq
					   (stx sty)
					   (read-kaigyou-shori 
						enx eny 
						line-feed left-m (car (sixth object))))
					  (push ch char-list)
					  (setf endx left-m  start (- end 1) end start))
				  (setf truncate-flg T))
				(setf enx stx eny sty)
				)
		  )
		 )
	  ))
	  (values enx eny)
  ))

;;; index$@$G;X<($5$l$k0LCV$^$G$N%+!<%=%k0LCV$r5a$a$k(J
;;; x y$@$O!"=i4|%+!<%=%k0LCV(J
(defun cursor-position-by-index-tate (string index x y object)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let* ((endx x) (endy y) 
		 (start 0) (end 0)
		 (font-height (font-kanji-height (edit-font object)))
		 (truncate-flg nil)
		 (ch #\a))

	(with-edit-slots
	 ((top-m top-margin) (bottom-m bottom-margin)
	  line-feed font truncate-height) object

	  ;; $@J8;z$r7W;;(J
	  (do ((char-list string (cdr char-list)))
		  ((or (null char-list)
			   (= index end)))

		 ;; $@0lJ8;z%+%&%s%H%"%C%W(J
		 (incf end)
		 ;;  $@0lJ8;z<h$j=P$9(J
		 (setf ch (take-out-car-moji char-list))

		;; $@%H%i%s%1%$%H=hM}(J
		(if (and truncate-flg (/= 10 (char-code ch))
				 (/= 13 (char-code ch)))
			(incf start)
		  (setf truncate-flg nil))

		(cond 		;; $@J8;z$NH=CG(J
		 ((kanji-charp ch) 		 ;; $@4A;z$N>l9g(J
		  (incf endy font-height) 
		  (when (> endx bottom-m)  ;; $@%^!<%8%s$r1[$($?(J
				(decf endy font-height) 
				(if (null truncate-height)
					(progn
					  (push ch char-list)
					  (setf endx (- endx line-feed)  endy top-m
							start (- end 1) end start))
				  (setf truncate-flg T))
		  )
		  #+(or LUCID (and EXCL (not ICS)))
  		    (setf char-list (cdr char-list)
				  end (1+ end))
			)
		 ((or (= (char-code ch) 10)  ;;; $@2~9T=hM}(J
			  (= (char-code ch) 13))
		  (setf endx (- endx line-feed) endy top-m
				start end end start)
		  )
		 (t ;; $@1Q;z$N>l9g(J
		  (incf endy (character-width font ch))
 		  (when (> endx bottom-m)                ;; $@%^!<%8%s$r1[$($?(J
				(decf endy (character-width font ch))
				(if (null truncate-height)
					(progn
					  (push ch char-list)
					  (setf endx (- endx line-feed) endy top-m
							start (- end 1) end start))
				  (setf truncate-flg T))
				)
		  )
		 )
	  ))
	  (values endx endy)
  ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  $@J8;zNsJT=8(J $@4X?t74(J   ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; $@%G%k%_%?!<$NF~NO;~$NI=<((J
(defun end-comming-disply (object)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  ;; $@A0$NI=<($r>C$9(J
  (drawing-text-read (edit-start-x object) (edit-start-y object)
					 (edit-string-list object) object)

  (drawing-text (edit-stream object)
				(coerce (edit-string-list object) 'simple-string))
  )

;;; $@#1J8;zF~NO(J $@:G8eHx$KDI2C(J
;;; data $@$O!":GBg#2J8;z(J
(defun  add-end-display (object data)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((newx 0) (newy 0))

	(incf (edit-index object))
	(if (= (length data) 2)
		(progn 
		  (incf (edit-index object))
		  (setf (edit-string-list object)
				(append (edit-string-list object) 
					   (list (char data 0) (char data 1))))
		  (multiple-value-setq
		   (newx newy) (drawing-text-read (edit-position-x object) 
										  (edit-position-y object)
										  (coerce data 'list) object)))
	  (progn 
		  (setf (edit-string-list object)
				(append (edit-string-list object) 
						(list (char data 0))))
		  (multiple-value-setq
		   (newx newy) (drawing-text-read (edit-position-x object) 
										  (edit-position-y object)
										  (coerce data 'list) object))
		))
;	(format t "newx: ~a newy: ~a ~%" newx newy)
	(setf
	 (edit-position-x object) newx
	 (edit-position-y object) newy))
  )


;;; $@#1J8;zF~NO(J $@ESCf0LCV$KDI2C(J
(defun add-continue-display (object data)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  ;; $@%+!<%=%k0J9_$r>C5n(J
  (drawing-text-read  (edit-position-x object) 
					  (edit-position-y object)
					  (nthcdr (edit-index object)
							  (edit-string-list object))
					  object)
  ;;$@J8;z$rI=<((J
  (let ((new-x 0) (new-y 0))
	(if (= (length data) 2)
		(progn
		  (if (> (edit-index object) 0) 
			  (progn
			   (push (char data 1) (cdr (nthcdr (- (edit-index object) 1)
												(edit-string-list object))))
			   (push (char data 0) (cdr (nthcdr (- (edit-index object) 1)
												(edit-string-list object))))
			   )
			(progn
			  (push (char data 1) (edit-string-list object))
			  (push (char data 0) (edit-string-list object))))

		  (multiple-value-setq
		   (new-x new-y)
		   (drawing-text-read  (edit-position-x object) 
							   (edit-position-y object)
							   (nthcdr (edit-index object)
									   (edit-string-list object))
							   object t))
		  (incf (edit-index object) 2)
		  (setf 
		   (edit-position-x object) new-x
		   (edit-position-y object) new-y))
	  (progn
		(if (> (edit-index object) 0)
			(push (char data 0) (cdr (nthcdr (- (edit-index object) 1)
											 (edit-string-list object))))
		  (push (char data 0) (edit-string-list object)))

		(multiple-value-setq
		 (new-x new-y)
		 (drawing-text-read  (edit-position-x object) 
							 (edit-position-y object)
							 (nthcdr (edit-index object)
									 (edit-string-list object))
							 object t))
		(incf (edit-index object))
		(setf 
		 (edit-position-x object) new-x
		 (edit-position-y object) new-y))
	  ))
  )
	
;;; index$@HVL\$NJ8;z$rJ8;zNs%j%9%H$+$i<h$j=|$/(J
(defun take-out-moji (moji-list index)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((item #\a))
	(if (zerop index)
		(setf item (pop moji-list))
	  (setf item (pop (cdr (nthcdr (- index 1) moji-list)))))

	(if (kanji-charp item)
		(if (zerop index)
			(setf item (pop moji-list))
		  (setf item (pop (cdr (nthcdr (- index 1) moji-list)))))
	  )
  moji-list))
  

;;; $@#1J8;z>C5n(J
(defun delete-text (object)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))

    (when
	 (edit-string-list object)
	 ;; $@%+!<%=%k0J9_$r>C5n(J
	 (drawing-text-read  (edit-position-x object)
						 (edit-position-y object)
						 (nthcdr (edit-index object)
								 (edit-string-list object))
						 object)
	 
	 ;; $@0lJ8;z<h$j=|$/(J
	 (setf (edit-string-list object)
		   (take-out-moji (edit-string-list object)
						  (edit-index object)))

	 (if (edit-string-list object)
		 ;; $@%+!<%=%k0J9_$r:FI=<((J
		 (drawing-text-read  (edit-position-x object)
							 (edit-position-y object)
							 (nthcdr (edit-index object)
									 (edit-string-list object))
							 object t))
	 nil
	 ))

;;; $@A0$NJ8;z$r>C5n(J
(defun back-space-text (object)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
    (unless 
	 (zerop (edit-index object))
	 (before-cursor-text object)
	 (delete-text object))
	)

;;; $@%+!<%=%k$rA0$K$9$k(J
(defun before-cursor-text (object)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (unless 
   (zerop (edit-index object))
   (decf (edit-index object))
   (if (kanji-charp (nth (edit-index object) (edit-string-list object)))
	   (decf (edit-index object)))
  
   (multiple-value-bind
	(nx ny) (cursor-position-by-index
			 (edit-start-x object) (edit-start-y object)
			 (edit-index object) (edit-string-list object)  object)
	(setf 
	 (edit-position-x object) nx
	 (edit-position-y object) ny)
	))
  nil
  )

;;; $@%+!<%=%k$r8e$m$K$9$k(J
(defun next-cursor-text (object)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (when 
   (< (edit-index object) (length (edit-string-list object)))

   (if (kanji-charp (nth (edit-index object) (edit-string-list object)))
	   (incf (edit-index object)))

   (incf (edit-index object))
  
   (multiple-value-bind
	(nx ny) (cursor-position-by-index
			 (edit-start-x object) (edit-start-y object) 
			 (edit-index object) (edit-string-list object) object)
	(setf 
	 (edit-position-x object) nx
	 (edit-position-y object) ny)
	))
  nil
  )

;;; $@2>L>4A;zJQ49%&%#%s%I%&$NI=<((J
(defun kana-kanji (object)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0))
		   (special *henkan-window*))
  (let ((stream (edit-stream object))
		(ret-string ""))
	(activate *henkan-window*)
	(select-window *henkan-window*)
	(setf ret-string (read-line *henkan-window*))
	(deactivate *henkan-window*)
	(setf ret-string (reverse (coerce ret-string 'list)))
	(if (or (= (char-code (car ret-string)) 10)
			(= (char-code (car ret-string)) 13))
		(pop ret-string))
	(with-slots 
	 (input-string)  stream
	 (dolist (item ret-string)
			 (push item input-string))
	 (select-window stream))))


;;; $@F~NO%b!<%I$N(J
;;; $@%F%-%9%HF~NO%+!<%=%k$NI=<(!?HsI=<((J
(defun display-text-cursor (object)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((xx (edit-position-x object))
		(yy (translate-xy-edit (edit-position-y object) object))
		(width 0) (height 0) (font (edit-font object))
		(ch (nth (edit-index object) (edit-string-list object)))
		(color (edit-color object))
		(stream (edit-stream object)))

	 (when (null ch)
		   (setf ch #\a))
	 (if (eq (edit-direction object) :vertical) ;; $@=D=q$-(J
		 (if (kanji-charp ch)
			 (setf yy (- yy (font-kanji-height font))
			  width (font-kanji-width font)
				   height (font-kanji-height font))
		   (setf yy (- yy (character-base-line font ch))
				 width (font-kanji-width font)
				 height (character-width font ch)))
	   (if (kanji-charp ch)
		   (setf yy (- yy (font-kanji-base-line font))
				 width (font-kanji-width font)
				 height (font-kanji-height font))
		 (setf xx (+ xx 1)
			   yy (- yy	(character-base-line font ch))
			   width (character-width font ch)
			   height (font-kanji-height font)))
	   )
	 (with-slots 
	  ((tno world-territory-no) 
	   world-x-start world-y-start) stream
	   (yy-protocol-28 tno (+ xx world-x-start) 
					   (+ yy world-y-start) width height 
					   (avialble-operation color)
					   (color-no color) 0))
	 ))

;;; read-line $@MQF~NOJT=8=hM}(J
(defun edit-read-line-internal (object stream)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((code nil) (kanji "  "))
	(loop
	 ;; 1$@J8;z<h$j=P$9(J
	 (with-slots 
	  (input-string)  stream
	  (if (= (length input-string) 0)
		  (return nil))
	  (setf code (pop input-string)))
	 ;; $@%F%-%9%H%+!<%=%k$r>C5n(J
	 (display-text-cursor object)
	 ;; $@J8;z%3%s%H%m!<%k(J
	 (if (assoc (char-code code) (input-mask-table stream))
		 ;; $@4X?t<B9T(J
		 (progn
		   (funcall (cdr
					 (assoc (char-code code) 
							(input-mask-table stream))) object)
		   (display-text-cursor object)
		   #-CMU
		   nil
		   #+CMU
		   (return nil)
		   )
	   ;; $@F~NO=hM}(J
	   (case 
		(char-code code)
		((10 13) ;; $@%G%k%_%?!<$,Mh$?!#(J
		 (end-comming-disply object)
		 (return T))
		(T 				 ;; $@F~NOJ8;z(J
		 (if (> (char-code code) #xA1)
			 (progn  ;; $@4A;z$,Mh$?(J
			   (with-slots
				(input-string) stream
				(setf (char kanji 0) code
					  (char kanji 1) (pop input-string)))
			   
			   (if (= (length (edit-string-list object)) 
					  (edit-index object))
				   (add-end-display object kanji)
				 (add-continue-display object kanji)))
		   ;; $@%"%9%-!<J8;z(J
		   (if (= (length (edit-string-list object))
				  (edit-index object))
			   (add-end-display object  (string code))
			 (add-continue-display object  (string code)))
		   )
		 (display-text-cursor object)
		 #+CMU
		 nil
		 #+CMU
		 (return nil)
		 )
		)))
	))
  
;;; $@F~NO$NDd;_%A%'%C%/(J
(defun end-of-read-stream (stream)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots 
   (end-of-read) stream
   end-of-read))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; YY Edit read line T.kosaka    ;;;
;;; This is viewport mode         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; added unwind-protect by Yohta on 5.Nov.90
(defmethod yy-edit-read-line ((stream viewport-window-stream))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0))
		   (special *event-loop-process* *tuuchi-stop* *accept-tuuchi-stop*))

  (with-slots
   ((pos cursor-position) translate-coordinate 
	transform-by-matrix 
	work-region1 work-region2
	left-margin right-margin top-margin bottom-margin font
	line-feed output-direction truncate-width 
	truncate-height world-height) stream

	(unless right-margin
			(setf right-margin MOST-POSITIVE-FIXNUM
				  left-margin 0))
	(let ((xx 0) (yy 0) (ox (position-x pos))
		  (oy (position-y pos)))
	  (multiple-value-setq
	   (xx yy) (transform-by-matrix-xy (position-x pos) (position-y pos)
									   transform-by-matrix :return :multiple))
	  (let* ((right-m (if right-margin  right-margin
						MOST-POSITIVE-FIXNUM))
			 (left-m (if left-margin left-margin 0))
			 (top-m (if top-margin	top-margin 0))
			 (bottom-m (if bottom-margin  bottom-margin
						 MOST-POSITIVE-FIXNUM))
			 (object
			  (list nil xx yy output-direction
					(list line-feed font left-m right-m top-m bottom-m 
						  truncate-width truncate-height)
					(list transform-by-matrix translate-coordinate 
						  world-height)
					(graphic-color stream)
					work-region1 work-region2
					xx yy 0 stream xx yy)))

		(initialize-object-matrix transform-by-matrix object)
		
		;; $@IA2h%j!<%8%g%s$N=i4|@_Dj(J
		(with-region-slots
		 (left bottom top right) work-region1
		 (setf left MOST-POSITIVE-FIXNUM bottom MOST-POSITIVE-FIXNUM
			   right most-negative-fixnum top most-negative-fixnum))

		;;$@%+!<%=%k$rI=<((J
		(display-text-cursor object)

		(setf (end-of-read stream) nil)

		;; $@F~NO$NFbIt=hM}(J
		(let ((proc-list (list #'edit-read-line-internal object stream)))
		  (push proc-list
				(read-event-proc stream))

		  (wait-process #'end-of-read-stream stream)

		  ;;; $@<h$j=|$/(J
		  (setf 
		   (read-event-proc stream)
		   (delete proc-list (read-event-proc stream))
		   (position-x pos) ox
		   (position-y pos) oy)
		  
		  (coerce (edit-string-list object) 'simple-string)
		)))))

(defmethod return-coordinate-and-direction ((stream page-window-stream))
  (let* ((co (stream-translate-coordinate stream))
		 (co-s (if (eq (class-name (class-of co)) 
					 'translate-coordinate-left-bottom)
				 1
			   0))
		 (di (if (eq (stream-output-direction stream)
					 :horizontal)
				 0
			   1)))
	(values co-s di)))
		 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; YY Edit read line T.kosaka    ;;;
;;; This is Page mode             ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; added unwind-protect by yohta on 4.Nov.90
(defmethod yy-edit-read-line ((stream page-window-stream))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0))
		   (special *event-loop-process* *accept-tuuchi-stop* *tuuchi-stop*))
  ;; added unwind-protect by yohta on 4.Nov.90
  (let ((terminate-string "  ")
		(interrupt-string " ")
		(ret-string ""))
	;; $@=*N;%-!<$N@_Dj(J
	(setf (char terminate-string 1) (code-char 10)
		  (char terminate-string 0)  (code-char 13))
		
	(setf (char interrupt-string 0) (code-char 3))

	(with-slots
	 ((tno world-territory-no)
	  font line-feed column line) stream
	  (multiple-value-bind
	   (co di) (return-coordinate-and-direction stream)
	   (yy-protocol-80 tno (font-no font) line-feed
					   di co column line)

	   ;; $@%W%m%H%3%k$r@8@.(J
	   (yy-protocol-82 tno column line
					   1 "" terminate-string interrupt-string)))
		
	;; $@F~NO$5$l$k$^$GBT$D!#(J
	(wait-process #'input-ok stream)

    ;(yy-protocol-83  (world-territory-no stream))

	(setf ret-string
		  (coerce (stream-input-string-internal stream) 'string))

	;; $@F~NOJ8;zNs$r#0$K$9$k(J
	(setf (slot-value stream 'input-string) nil)
	ret-string))

;;; edit-read $@$NFbIt=hM}(J
(defun edit-read-internal (object stream)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((code nil) (kanji "  ")
		(cr-flg nil))
	(loop

	 (with-slots 
	  (input-string) stream
	  (if (= (length input-string) 0)
		  (return nil))
	  ;; 1$@J8;z<h$j=P$9(J
	  (setf code (pop input-string)))

	 ;; $@%F%-%9%H%+!<%=%k$r>C5n(J
	 (display-text-cursor object)
				 
	 ;; $@J8;z$rJQ99(J
	 (setf code (real-line-feed-char code))

	 ;; $@J8;z%3%s%H%m!<%k(J
	 (if (assoc (char-code code) (input-mask-table stream))
		 ;; $@4X?t<B9T(J
		 (progn
		   (funcall (cdr(assoc (char-code code) 
							   (input-mask-table stream))) object)
		   (display-text-cursor object)
		   #-CMU
		   nil
		   #+CMU
		   (return nil)
		   )
	   ;; $@F~NO=hM}(J
	   (progn
		 (cond
		  ((> (char-code code) #xA1)
		   ;; $@4A;z$,Mh$?(J
		   (with-slots 
			(input-string) stream
			(setf (char kanji 0) code
				  (char kanji 1) (pop input-string)))
					  
		   (if (= (length (edit-string-list object)) 
				  (edit-index object))
			   (add-end-display object kanji)
			 (add-continue-display object kanji)))
		  ;; $@%"%9%-!<J8;z(J
		  (t
		   (when (member (char-code code)
						 (list 10 13 (char-code #\linefeed)
							   (char-code #\space)))
				 (setf cr-flg T))

		   (if (= (length (edit-string-list object))
				  (edit-index object))
			   (add-end-display object  (string code))
			 (add-continue-display object  (string code))))
		  )
		 ;; $@=*N;%A%'%C%/(J
		 (if (read-end-check 
			  (make-real-read-string (edit-string-list object))
			  cr-flg)
			 (progn
			   (end-comming-disply object)
			   (return t))
		   (progn 
			 (display-text-cursor object)
			 #-CMU
			 nil
			 #+CMU
			 (return nil)
			 ))
		 ))))
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; YY Edit read  T.kosaka    ;;;
;;;        ViewPort MODE      ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defmethod yy-edit-read ((stream viewport-window-stream))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots
   ((pos cursor-position) translate-coordinate 
	transform-by-matrix 
	work-region1 work-region2
	left-margin right-margin top-margin bottom-margin font
	line-feed output-direction truncate-width 
	truncate-height world-height) stream

	(unless right-margin
			(setf right-margin MOST-POSITIVE-FIXNUM
				  left-margin 0))
	(let ((xx 0) (yy 0) (ret-val nil)
		  (ox (position-x pos)) (oy (position-y pos)))
	  
	  (multiple-value-setq
	   (xx yy) (transform-by-matrix-xy (position-x pos) (position-y pos)
									   transform-by-matrix :return :multiple))
	  (let* ((right-m (if right-margin  right-margin MOST-POSITIVE-FIXNUM))
			 (left-m (if left-margin left-margin  0))
			 (top-m (if top-margin	top-margin  0))
			 (bottom-m (if bottom-margin  bottom-margin MOST-POSITIVE-FIXNUM))
			 (object
			  (list nil xx yy output-direction
					(list line-feed font left-m right-m top-m bottom-m 
						  truncate-width truncate-height)
					(list transform-by-matrix translate-coordinate 
						  world-height)
					(graphic-color stream)
					work-region1 work-region2
					xx yy 0 stream xx yy)))

		(initialize-object-matrix transform-by-matrix object)

		;; $@IA2h%j!<%8%g%s$N=i4|@_Dj(J
		(with-region-slots
		 (left bottom top right) work-region1
		 (setf left MOST-POSITIVE-FIXNUM bottom MOST-POSITIVE-FIXNUM
			   right most-negative-fixnum top most-negative-fixnum))

		;;$@%+!<%=%k$rI=<((J
		(display-text-cursor object)
		(setf (end-of-read stream) nil)

		;; $@F~NO$NFbIt=hM}(J
		(let ((proc-list (list #'edit-read-internal object stream)))
		  (push proc-list
				(read-event-proc stream))
		  (wait-process #'end-of-read-stream stream)
		  ;;; $@<h$j=|$/(J
		  (setf 
		   (read-event-proc stream)
		   (delete proc-list (read-event-proc stream))))
		
		(setf (position-x pos) ox
			  (position-y pos) oy
			  ret-val (coerce (edit-string-list object) 'simple-string))
		(read-from-string ret-val)
		)
	  )))
  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; YY Edit read line T.kosaka    ;;;
;;; This is Page mode             ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defmethod yy-edit-read ((stream page-window-stream))
  (let ((ret-val1 nil))
    ;; added unwind-protect by yohta on 5.Nov.90
    (unwind-protect
	(let ((terminate-string "  ")
	      (interrupt-string " ")
	      (semi-strminate-string "    ")
	      (ret-string nil) 
	      (event-list nil))

           ;;; $@=*N;%-!<$N@_Dj(J
	  (setf (char terminate-string 0)
	    (code-char 10)
	    (char terminate-string 1)
	    (code-char 13))

           ;;; $@3d$j9~$_$-!<$N@_Dj(J
	  (setf (char interrupt-string 0)
	    (code-char 3))

           ;;; $@ESCfDd;_%-!<$N@_Dj(J
	  (setf (char semi-strminate-string 1)
	    #\)
	    (char semi-strminate-string 2) #\"
	    (char semi-strminate-string 3) (code-char 13))

          ;;; $@%W%m%H%3%k$r@8@.(J
	  (yy-protocol-82 (world-territory-no stream)
			  (page-column stream)
			  (page-line stream)
			  1	  semi-strminate-string
			  ""  interrupt-string)

           ;;; $@J8;zNs$r>C5n(J
	  (setf (slot-value stream 'input-string) nil)

	  (loop 
  	     ;;; $@F~NO$,Mh$k$^$GBT$D(J
	    (wait-process 'input-ok stream)
	    (setf event-list (stream-input-string-internal stream))
             ;;; $@F~NOJ8;zNs$r#0$K$9$k(J
	    (setf (slot-value stream 'input-string) nil)
             ;;; $@J8;zNs$r<h$j=P$9(J
	    (setf ret-string 
	      (change-to-real-line-feed 
		   (make-real-read-string event-list)))

		;; $@=*N;%A%'%C%/(J
		(if (read-end-check ret-string T)
			(return))
	    )

	  (setf ret-val1 (read-from-string ret-string))

	  ;; $@F~NOJ8;zNs$r#0$K$9$k(J
	  (setf (slot-value stream 'input-string) nil)

	  (yy-protocol-83 (world-territory-no stream))

      ;; $@9T$H(Jy-position $@$r@_Dj(J
;	  (setf ret (yy-protocol-84 (world-territory-no stream)
	;			    (graphic-operation stream)
		;		    (color-no (graphic-color stream))
			;	    (font-no (stream-font stream))
				;    ""))
;	  (set-y-position-for-page-maode stream (nth 4 ret) (nth 5 ret))
	  )
	)
	ret-val1))


;;; end of file

