;;; -*- Mode: LISP; Package: YY-GEO; Syntax: Common-Lisp; Base: 10 -*-
;;;
(in-package :yy-geo)

(defmacro m11! (matrix) `(aref ,matrix 0))
(defmacro m12! (matrix) `(aref ,matrix 1))
(defmacro m13! (matrix) `(aref ,matrix 2))
(defmacro m14! (matrix) `(aref ,matrix 3))

(defmacro m21! (matrix) `(aref ,matrix 4))
(defmacro m22! (matrix) `(aref ,matrix 5))
(defmacro m23! (matrix) `(aref ,matrix 6))
(defmacro m24! (matrix) `(aref ,matrix 7))

(defmacro m31! (matrix) `(aref ,matrix 8))
(defmacro m32! (matrix) `(aref ,matrix 9))
(defmacro m33! (matrix) `(aref ,matrix 10))
(defmacro m34! (matrix) `(aref ,matrix 11))

(defmacro m41! (matrix) `(aref ,matrix 12))
(defmacro m42! (matrix) `(aref ,matrix 13))
(defmacro m43! (matrix) `(aref ,matrix 14))
(defmacro m44! (matrix) `(aref ,matrix 15))

(defmacro get-vector-x (vector) `(car ,vector))
(defmacro get-vector-y (vector) `(cadr ,vector))
(defmacro get-vector-z (vector) `(caddr ,vector))

#+Symbolics
(defun show-matrix (matrix)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (loop for i below 4
	do
    (format t "~%")
    (loop for j below 4
	  do
      (let ((a (+ j (* i 4))))
	(format t "~10d:~2d " (aref matrix a) a)))))
;;;
(defun make-view-matrix ()
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (make-array
    16 :initial-contents
    (list  0.9701   -.0756  -.0858   -.0858
	   0.0000   1.2850  -.0858   -.0858
	   -.2425   -.3024  -.3432   -.3432
	   0.00      .00    0.00     1.0))) ;;; Default View Matrix 

(defun make-4x4-matrix ()
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (make-array 16 :initial-contents
	      '(
		1.0 0.0 0.0 0.0
		0.0 1.0 0.0 0.0
		0.0 0.0 1.0 0.0
		0.0 0.0 0.0 1.0)))

(defun reset-matrix (matrix)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (setf
    (m11! matrix) 1.0 (m12! matrix) 0.0 (m13! matrix) 0.0 (m14! matrix) 0.0
    (m21! matrix) 0.0 (m22! matrix) 1.0 (m23! matrix) 0.0 (m24! matrix) 0.0
    (m31! matrix) 0.0 (m32! matrix) 0.0 (m33! matrix) 1.0 (m34! matrix) 0.0
    (m41! matrix) 0.0 (m42! matrix) 0.0 (m43! matrix) 0.0 (m44! matrix) 1.0))

(defun normal-vector-by-vertex (v1 v2 v3)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots ((x1 x)(y1 y)(z1 z)) v1
    (with-slots ((x2 x)(y2 y)(z2 z)) v2
      (with-slots ((x3 x)(y3 y)(z3 z)) v3
	(let ((vx (- x2 x1))
	      (vy (- y2 y1))
	      (vz (- z2 z1))
	      (wx (- x2 x3))
	      (wy (- y2 y3))
	      (wz (- z2 z3)))
	  (let ((x (- (* vy wz) (* vz wy)))
		(y (- (* vz wx) (* vx wz)))
		(z (- (* vx wy) (* vy wx)))
		d)
	    (setf d (sqrt (+ (* x x) (* y y) (* z z))))
	    (if (not (zerop d))
		(list (/ x d) (/ y d) (/ z d)))))))))

(defun calculate-matrix (matrix vector)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((x (get-vector-x vector))
	(y (get-vector-y vector))
	(z (get-vector-z vector)))
    (calculate-matrix-xyz matrix x y z)))

(defun calculate-matrix-vertex (matrix vertex)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (x y z) vertex
    (multiple-value-list
      (calculate-matrix-xyz matrix x y z))))

(defun calculate-view-position (matrix vector view-factor)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (multiple-value-bind (xx yy zz hh)
      (calculate-matrix matrix vector)
    (if (zerop hh)
	(values (round xx) (round yy) hh)
	(let ((factor (/ view-factor hh)))
	  (values
	    (round (* factor xx))
	    (round (* -1 factor yy))
	    (* factor zz))))))

(defun move-matrix-xyz (matrix x y z)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (setf (m41! matrix) x
	(m42! matrix) y
	(m43! matrix) z)
  matrix)

(defun calculate-matrix-xyz (matrix x y z)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (values
    (+ (* x (m11! matrix)) (* y (m21! matrix)) (* z (m31! matrix)) (m41! matrix))
    (+ (* x (m12! matrix)) (* y (m22! matrix)) (* z (m32! matrix)) (m42! matrix))
    (+ (* x (m13! matrix)) (* y (m23! matrix)) (* z (m33! matrix)) (m43! matrix))
    (+ (* x (m14! matrix)) (* y (m24! matrix)) (* z (m34! matrix)) (m44! matrix))))

(defun multiple-matrix (matrix1 matrix2 &optional (matrix3 (make-4x4-matrix)))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (setf (m11! matrix3) (+ (* (m11! matrix1) (m11! matrix2))
			  (* (m12! matrix1) (m21! matrix2))
			  (* (m13! matrix1) (m31! matrix2))
			  (* (m14! matrix1) (m41! matrix2)))
	(m21! matrix3) (+ (* (m11! matrix1) (m12! matrix2))
			  (* (m12! matrix1) (m22! matrix2))
			  (* (m13! matrix1) (m32! matrix2))
			  (* (m14! matrix1) (m42! matrix2)))
	(m31! matrix3) (+ (* (m11! matrix1) (m13! matrix2))
			  (* (m12! matrix1) (m23! matrix2))
			  (* (m13! matrix1) (m33! matrix2))
			  (* (m14! matrix1) (m43! matrix2)))
	(m41! matrix3) (+ (* (m11! matrix1) (m14! matrix2))
			  (* (m12! matrix1) (m24! matrix2))
			  (* (m13! matrix1) (m34! matrix2))
			  (* (m14! matrix1) (m44! matrix2)))
	(m12! matrix3) (+ (* (m21! matrix1) (m11! matrix2))
			  (* (m22! matrix1) (m21! matrix2))
			  (* (m23! matrix1) (m31! matrix2))
			  (* (m24! matrix1) (m41! matrix2)))
	(m22! matrix3) (+ (* (m21! matrix1) (m12! matrix2))
			  (* (m22! matrix1) (m22! matrix2))
			  (* (m23! matrix1) (m32! matrix2))
			  (* (m24! matrix1) (m42! matrix2)))
	(m32! matrix3) (+ (* (m21! matrix1) (m13! matrix2))
			  (* (m22! matrix1) (m23! matrix2))
			  (* (m23! matrix1) (m33! matrix2))
			  (* (m24! matrix1) (m43! matrix2)))
	(m42! matrix3) (+ (* (m21! matrix1) (m14! matrix2))
			  (* (m22! matrix1) (m24! matrix2))
			  (* (m23! matrix1) (m34! matrix2))
			  (* (m24! matrix1) (m44! matrix2)))
	(m13! matrix3) (+ (* (m31! matrix1) (m11! matrix2))
			  (* (m32! matrix1) (m21! matrix2))
			  (* (m33! matrix1) (m31! matrix2))
			  (* (m34! matrix1) (m41! matrix2)))
	(m23! matrix3) (+ (* (m31! matrix1) (m12! matrix2))
			  (* (m32! matrix1) (m22! matrix2))
			  (* (m33! matrix1) (m32! matrix2))
			  (* (m34! matrix1) (m42! matrix2)))
	(m33! matrix3) (+ (* (m31! matrix1) (m13! matrix2))
			  (* (m32! matrix1) (m23! matrix2))
			  (* (m33! matrix1) (m33! matrix2))
			  (* (m34! matrix1) (m43! matrix2)))
	(m43! matrix3) (+ (* (m31! matrix1) (m14! matrix2))
			  (* (m32! matrix1) (m24! matrix2))
			  (* (m33! matrix1) (m34! matrix2))
			  (* (m34! matrix1) (m44! matrix2)))
	(m14! matrix3) (+ (* (m41! matrix1) (m11! matrix2))
			  (* (m42! matrix1) (m21! matrix2))
			  (* (m43! matrix1) (m31! matrix2))
			  (* (m44! matrix1) (m41! matrix2)))
	(m24! matrix3) (+ (* (m41! matrix1) (m12! matrix2))
			  (* (m42! matrix1) (m22! matrix2))
			  (* (m43! matrix1) (m32! matrix2))
			  (* (m44! matrix1) (m42! matrix2)))
	(m34! matrix3) (+ (* (m41! matrix1) (m13! matrix2))
			  (* (m42! matrix1) (m23! matrix2))
			  (* (m43! matrix1) (m33! matrix2))
			  (* (m44! matrix1) (m43! matrix2)))
	(m44! matrix3) (+ (* (m41! matrix1) (m14! matrix2))
			  (* (m42! matrix1) (m24! matrix2))
			  (* (m43! matrix1) (m34! matrix2))
			  (* (m44! matrix1) (m44! matrix2))))
  matrix3)


(defvar *move-aim-point-matrix*  (make-4x4-matrix))
(defvar *work-view-matrix*       (make-4x4-matrix))
(defvar *change-view-matrix*     (make-4x4-matrix))
(defvar *change-vanishig-matrix* (make-4x4-matrix))

(defun change-view-matrix (view-matrix view-point aim-point &optional vanishig)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((xv (get-vector-x view-point))
	(yv (get-vector-y view-point))
	(zv (get-vector-z view-point))
	(xf (get-vector-x aim-point))
	(yf (get-vector-y aim-point))
	(zf (get-vector-z aim-point))
	(p (or (get-vector-x vanishig) 0.0))
	(q (or (get-vector-y vanishig) 0.0))
	(r (or (get-vector-z vanishig) 0.0)))
    (reset-matrix *move-aim-point-matrix*)
    (reset-matrix *change-vanishig-matrix*)
    (setf
      (m41! *move-aim-point-matrix*) (- xf)
      (m42! *move-aim-point-matrix*) (- yf)
      (m43! *move-aim-point-matrix*) (- zf)
      (m14! *change-vanishig-matrix*) p
      (m24! *change-vanishig-matrix*) q
      (m34! *change-vanishig-matrix*) r)
    (let ((dx (- xv xf))
	  (dy (- yv yf))
	  (dz (- zv zf)))
      (let ((r  (sqrt  (+ (* dx dx) (* dy dy) (* dz dz))))
	    (r1 (sqrt  (+ (* dx dx) (* dy dy)))))
	(reset-matrix *change-view-matrix*)
	(unless (zerop r)
	  (if (zerop r1)
	      (setf (m33! *change-view-matrix*) -1.0
		    (m34! *change-view-matrix*) 0.0
		    )
	      (let ((sin-t (/ (- yv yf) r1))
		    (cos-t (/ (- xv xf) r1))
		    (sin-p (/ (- zv zf) r))
		    (cos-p (/ r1 r)))
		(setf
		  (m11! *change-view-matrix*) (* sin-t -1.0)
		  (m12! *change-view-matrix*) (* cos-t sin-p -1.0)
		  (m13! *change-view-matrix*) (* cos-t cos-p)
		  (m14! *change-view-matrix*) 0.0
		  (m21! *change-view-matrix*) cos-t
		  (m22! *change-view-matrix*) (* sin-t sin-p -1.0)
		  (m23! *change-view-matrix*) (* sin-t cos-p)
		  (m24! *change-view-matrix*) 0.0
		  (m31! *change-view-matrix*) 0.0
		  (m32! *change-view-matrix*) cos-p
		  (m33! *change-view-matrix*) sin-p
		  (m34! *change-view-matrix*) 0.0
		  (m41! *change-view-matrix*) 0.0
		  (m42! *change-view-matrix*) 0.0
		  (m43! *change-view-matrix*) 0.0
		  (m44! *change-view-matrix*) 1.0))
	      ))))
    (multiple-matrix *move-aim-point-matrix*
		     *change-view-matrix*
		     *work-view-matrix*)
    (multiple-matrix *work-view-matrix*
		     *change-vanishig-matrix*
		     view-matrix)
    )
  view-matrix)

(defun change (view-point aim-point alph beta)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((xv (get-vector-x view-point))
	(yv (get-vector-y view-point))
	(zv (get-vector-z view-point))
	(xf (get-vector-x aim-point))
	(yf (get-vector-y aim-point))
	(zf (get-vector-z aim-point)))
    (let ((dx (- xv xf))
	  (dy (- yv yf))
	  (dz (- zv zf)))
      (let ((r  (sqrt  (+ (* dx dx) (* dy dy) (* dz dz))))
	    (r1 (sqrt  (+ (* dx dx) (* dy dy)))))
	(let ((a (atan dy dx))
	      (b (atan dz r1)))
	  (setf alph (+ a alph)
		beta (+ b beta)
		r1 (* r (abs (cos beta)))
		)
	  (let ((xx (* r1 (cos alph)))
		(yy (* r1 (sin alph)))
		(zz (* r (sin beta))))
	    (list (+ xf xx) (+ yf yy) (+ zf zz))))))))
