;;; -*- Mode: Lisp; Syntax: ANSI-Common-Lisp; Package: CLIM-INTERNALS; Base: 10; Lowercase: Yes -*-

(in-package :clim-internals)


;; The half-height of a tick mark.
(defparameter *tick-mark-half-height* 3)

;; Canned line styles.
(defparameter *time-scale-line-style* (make-line-style :thickness 2))
(defparameter *time-scale-tick-style* (make-line-style :thickness 1))

;; A canned *fully merged* text style.
(defparameter *time-scale-label-style* (make-text-style :fix :roman :small))

;; This defines DRAW-RULER and DRAW-RULER*, and does some of the other
;; necessary DEFGENERIC-type stuff.
;; Note: this example only draws *horizontal* rulers.  Handling the
;; other cases is left as an exercise for the reader.
(define-graphics-generic draw-ruler (x1 y1 x2 y2 nticks &key labels)
  :positions-to-transform (x1 y1 x2 y2)
  :drawing-options nil)

;; This defines the hooks into output recording.
(define-graphics-recording draw-ruler (ink line-style clipping-region)
  :bounding-rectangle
    (multiple-value-bind (label-width label-height)
	(if labels
	    (text-size stream (first labels) :text-style *time-scale-label-style*)
	    (values 0 0))
      (declare (ignore label-width))
      (values (coordinate (- (min x1 x2) 1))
	      (coordinate (- (min y1 y2) *tick-mark-half-height* 1))
	      (coordinate (+ (max x1 x2) 1))
	      (coordinate (+ (max y1 y2) *tick-mark-half-height* label-height 1))))
  ;; No :HIGHLIGHTING-TEST, just use the bounding rectangle.
  ;; No :HIGHLIGHTING-FUNCTION, just use a rectangle to do it 
  )

;; This actually draws the ruler, tick marks, labels, etc.
(defmethod medium-draw-ruler* ((medium basic-medium) x1 y1 x2 y2 nticks labels)
  (check-type labels (or null sequence))
  (when labels
    (assert (= (length labels) nticks) ()
	    "The number of labels must be the same as the number of tick marks"))
  (letf-globally (((medium-line-style medium) *time-scale-line-style*)
		  ((medium-text-style medium) *time-scale-label-style*))
    (medium-draw-line* medium x1 y1 x2 y2)
    (let ((interval (/ (- x2 x1) (1- nticks)))
	  (tick-offset 0)
	  (tmhh *tick-mark-half-height*))
      (dotimes (n nticks)
	(let ((x (floor (+ x1 tick-offset))))
	  (setf (medium-line-style medium) *time-scale-tick-style*)
	  (medium-draw-line* medium x (- y1 tmhh) x (+ y1 tmhh))
	  (when labels
	    (let ((label (elt labels n)))
	      (medium-draw-text* medium label x (+ y1 tmhh 1)
				 0 (length label) :center :top
				 x (+ y1 tmhh 1) nil))))
	(incf tick-offset interval)))))
