(in-package :halwed)

;;; **********************************************************************
;;; Copyright (c) 92, 93 Hallvard Traetteberg. All rights reserved.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted and may be copied as long as 
;;; no fees or compensation are charged for use, copying, or accessing
;;; this software and all copies of this software include this copyright
;;; notice.  Suggestions, comments and bug reports are welcome.  Please 
;;; address email to: Hallvard.Tretteberg@si.sintef.no
;;; **********************************************************************

(defstruct (wedlin (:include edlin))
  (row         0)
  (num-rows   nil)
  (layout     nil)
  (moved-from nil)
  (draw-from  nil)
  )

(defun standard-wrap-fun (char) (not (alphanumericp char)))

(defclass halwed (haled)
  (
   (win            :initarg :window :initform nil :accessor wed-win)

   (top	           :initarg :top                  :accessor wed-top)
   (bottom         :initarg :bottom               :accessor wed-bottom)
   (num-rows                                      :accessor wed-num-rows)

   (left-margin    :initarg :left-margin          :accessor wed-left-margin)
   (right-margin   :initarg :right-margin         :accessor wed-right-margin)

   (wrap-fun       :initarg :wrap-fun  :initform #'standard-wrap-fun :accessor wed-wrap-fun)
   (cont-char      :initarg :cont-char :initform #\\ :accessor wed-cont-char)
   (top-line       :initform   0 :accessor wed-top-line)
   (top-row        :initform   0 :accessor wed-top-row)
   (last-top-row   :initform   0 :accessor wed-last-top-row)
   (max-row        :initform   0 :accessor wed-max-row)
   (drawn-markers  :initform nil :accessor wed-drawn-markers)
   )
  )

(defmethod edlin-constructor ((wed halwed)) (values #'make-wedlin))

(defun line-height (wed)
  (clim:stream-line-height (wed-win wed)))

(defun char-width (wed &optional (char #\x))
  (clim:stream-character-width (wed-win wed) char))

(defun string-width (wed string start end)
  (clim:stream-string-width (wed-win wed) string :start start :end end))

(defun line-length (wed &optional (char #\x))
  (truncate (- (wed-right-margin wed) (wed-left-margin wed))
            (char-width wed char)))

(defun ensure-wed-fills-window (wed &key (left-margin nil) (right-margin nil)
                                         (top nil) (bottom nil))
  (macrolet ((setf-slot (slot op start &optional (default 0) (xtra 0))
               `(when (or ,slot (not (slot-boundp wed ',slot)))
                  (setf (slot-value wed ',slot)
                        (,op ,start (if (integerp ,slot) ,slot ,default) ,xtra))))
             )
    (when (wed-win wed)
      (multiple-value-bind (left-x top-y right-x bottom-y)
                           (clim:window-inside-edges (wed-win wed))
        (setf-slot left-margin  + left-x  (char-width wed))
        (setf-slot right-margin - right-x  0 (char-width wed (or (wed-cont-char wed) #\Space)))
        (setf-slot top          + top-y)
        (setf-slot bottom       + bottom-y)
        )
      (setf (wed-num-rows wed) (truncate (- (wed-bottom wed) (wed-top wed))
                                         (line-height wed))
            (wed-max-row wed) (wed-num-rows wed))
      )))

(defmethod initialize-instance :after ((wed halwed) &rest initargs)
  (declare (ignore initargs))
  (ensure-wed-fills-window wed)
  )

(defun pos-past-width (wed string width start end)
  (setf end (or end (length string)))
  (labels
    ((bin-search (left right)
       (let* ((new-try (truncate (+ left right) 2))
              (new-width (string-width wed string start new-try)))
         (if (<= new-width width)
           (if (<= (- width new-width)
                   (char-width wed (aref string new-try)))
             (values new-try)
             (bin-search new-try right))
           (bin-search left new-try))
         )))
    (if (<= (string-width wed string start end) width)
      (values nil)
      (let ((win (wed-win wed)))
        (if (clim:text-style-fixed-width-p (clim:medium-text-style win) win)
          (+ start (truncate width (char-width wed)))
          (bin-search start end))))
    ))

(defun row-y-position (wed row)
  (+ (wed-top wed)
     (* (- row (wed-top-row wed)) (line-height wed))))

(defun y-row-position (wed y)
  (+ (truncate (- y (wed-top wed)) (line-height wed))
     (wed-top-row wed)))

(defun layout-wedlin (wed wedlin width wrap-fun &optional (force-p nil))
  (let* ((string (wedlin-str wedlin)) (end (length string))
         (layout (and (not force-p) (wedlin-layout wedlin)))
         (modified-at (if force-p 0 (wedlin-modified-at wedlin)))
         (wrap 0) (row 0) (result nil) (draw-col nil))
    (unless modified-at
      (return-from layout-wedlin (values layout nil)))
    (labels
      ((wrapping-position (start)
         (let ((margin (pos-past-width wed string width start end)))
           (unless margin
             (return-from wrapping-position (values end nil)))
           (let ((wrap (position-if wrap-fun string :from-end t :start start :end margin)))
             (values (if wrap (1+ wrap) margin) margin)))))
      (loop
        (if layout
          (progn
            (if (or (not (cdar layout)) (< modified-at (cdar layout))
                    (= modified-at end (cdar layout)))
              (multiple-value-bind (new-wrap margin)
                                   (wrapping-position wrap)
                (setf draw-col (min (caar layout) new-wrap modified-at))
                (setf layout nil)
                (push (cons new-wrap margin) result))
              (push (car layout) result))
            (pop layout))
          (multiple-value-bind (new-wrap margin) (wrapping-position wrap)
            (unless draw-col
              (setf draw-col (min new-wrap modified-at)))
            (push (cons new-wrap margin) result))
          )
        (setf wrap (caar result))
        (when (or (not wrap) (= wrap end))
	  (return-from layout-wedlin (values (nreverse result) draw-col)))
        (incf row))
      (values))))

(defun string-layout-calc (wed op arg wedlin)
  (let ((left (wed-left-margin wed))
        (string (wedlin-str wedlin)) (layout (wedlin-layout wedlin))
        (wrap 0) (row-num 0))
    (dolist (row-layout layout)
      (ecase op
        (:col ; arg = (x . row) or row
         (when (= row-num (if (consp arg) (cdr arg) arg))
           (return (values (if (consp arg)
                             (or (pos-past-width wed string (- (car arg) left) wrap (car row-layout))
                                 (car row-layout))
                             wrap)
                     row-num))))
        (:pos ; arg = col
         (when (or (not (cdr row-layout)) (<= arg (car row-layout)))
           (return (values (+ left (string-width wed string wrap
                                                 (min arg (car row-layout))))
                     row-num))))
        (:row ; arg = col
         (when (or (not (cdr row-layout)) (< arg (car row-layout)))
           (return (values row-num row-num))))
        )
      (setf wrap (car row-layout))
      (incf row-num))
    ))

(defun layout-wed (wed &optional (force-p nil) (wedlins (ed-lines wed)))
  (unless (wed-win wed)
    (return-from layout-wed (values)))
  (let ((left (wed-left-margin  wed)) (right (wed-right-margin wed))
        (wrap-fun (wed-wrap-fun wed))
        (row 0))
    (labels
      ((layout-wed-aux (wedlin)
         (setf (wedlin-moved-from wedlin)
               (unless (= (wedlin-row wedlin) row)
                 (wedlin-row wedlin)))
         (setf (wedlin-row wedlin) row)
         (multiple-value-bind (new-layout draw-col)
             (layout-wedlin wed wedlin (- right left) wrap-fun force-p)
           (setf (wedlin-layout wedlin) new-layout
                 (wedlin-num-rows wedlin) (length new-layout)
                 (wedlin-draw-from wedlin) draw-col
                 (wedlin-modified-at wedlin) nil)
           (incf row (wedlin-num-rows wedlin)))
         ))
      (map 'nil #'layout-wed-aux wedlins)
      (setf (ed-modified-lines wed) nil)
      (values))))

(defmethod (setf wed-win) :after (value wed)
  (declare (ignore value))
  (ensure-wed-fills-window wed :left-margin t :right-margin t :top t :bottom t)
  (layout-wed wed t)
  )

(defun line-row-rows-away (wed line col rows)
  (let* ((wedlin (ed-line wed line))
         (row (string-layout-calc wed :row col wedlin))
         (accum-rows (- (if (minusp rows) 0 (wedlin-num-rows wedlin)) row))
         (dline (signum rows)))
    (loop
      (if (minusp rows)
        (when (<= accum-rows rows)
          (return (values line (- rows accum-rows))))
        (when (> accum-rows rows)
          (return (values line (- (wedlin-num-rows wedlin) (- accum-rows rows)))))
        )
      (when (or (minusp (+ line dline)) (>= (+ line dline) (length (ed-lines wed))))
        (return (values line nil)))
      (incf line dline)
      (setf wedlin (ed-line wed line))
      (incf accum-rows (* dline (wedlin-num-rows wedlin)))
      )))

(defun update-wed-layout (wed &optional (force-p nil))
  (unless (wed-win wed)
    (return-from update-wed-layout (values)))
  (layout-wed wed force-p)
  (let* ((point (ed-point wed))
         (line (marker-line point))
         (wedlin (ed-line wed line))
         (col (min (marker-col point) (length (edlin-str wedlin))))
         (row (+ (wedlin-row wedlin) (string-layout-calc wed :row col wedlin)))
         )
    (when (or (< row (wed-top-row wed))
              (>= row (+ (wed-top-row wed) (wed-num-rows wed))))
      (multiple-value-bind (top-line top-row)
          (line-row-rows-away wed line col (truncate (wed-num-rows wed) -2))
        (unless top-row
          (setf top-row 0))
        (let ((top-wedlin (ed-line wed top-line)))
          (setf (wed-last-top-row wed) (wed-top-row wed)
                (wed-top-row wed) (+ (wedlin-row top-wedlin) top-row))
          (setf (wed-top-line wed) top-line))))
    (values)))

(defun clear-row-from-x (wed row x &optional (num-rows nil))
  (clim:draw-rectangle* (wed-win wed) x (row-y-position wed row)
                        (+ (wed-right-margin wed) 10)
                        (row-y-position wed (+ row (or num-rows 1)))
                        :ink clim:+background+))

(defun draw-string-layout (wed wedlin row col)
  (let* ((win (wed-win wed))
         (left (wed-left-margin wed)) (right (wed-right-margin wed))
         (top-row (wed-top-row wed))
         (max-row (+ top-row (wed-num-rows wed)))
         (string (wedlin-str wedlin))
         (wrap 0))
    (dolist (row-layout (wedlin-layout wedlin))
      (when (>= row max-row)
        (return (values)))
      (when (and (>= row top-row)
                 (or (not (cdr row-layout)) (<= col (car row-layout))))
        (let ((x (+ left (string-width wed string wrap col))))
          (clear-row-from-x wed row x)
          (clim:draw-text* win string x (row-y-position wed row)
                           :start col :end (car row-layout)
                           :align-y :top)
          (when (and (cdr row-layout) (wed-cont-char wed))
            (clim:draw-text* win (string (wed-cont-char wed))
                             right (row-y-position wed row)
                             :align-y :top))
          )
        (setf col (car row-layout)))
      (setf wrap (car row-layout))
      (incf row))
    (values)))

(defun draw-marker (wed marker &key (align-y :bottom))
  (let ((wedlin (ed-line wed (marker-line marker))))
    (multiple-value-bind (caret-x caret-row)
                         (string-layout-calc wed :pos (marker-col marker) wedlin)
      (let* ((y1 (row-y-position wed (+ (wedlin-row wedlin)
                                        (if (eq align-y :top) caret-row (1+ caret-row)))))
             (y2 (if (eq align-y :top) (+ y1 2) (- y1 2))))
        (clim:draw-line* (wed-win wed) (- caret-x 2) y1 caret-x y2)
        (clim:draw-line* (wed-win wed) (+ caret-x 2) y1 caret-x y2)
        (values (cons caret-x y1))))))

(defun draw-markers (wed)
  (setf (wed-drawn-markers wed)
        (list (when (ed-point wed)
                (draw-marker wed (ed-point wed)))
              (when (ed-mark wed)
                (draw-marker wed (ed-mark wed) :align-y :top)))))

(defun erase-drawn-marker (wed marker-pos &key (align-y :bottom))
  (let* ((medium (wed-win wed))
         (caret-x (car marker-pos))
         (y1 (cdr marker-pos))
         (y2 (if (eq align-y :top) (+ y1 2) (- y1 2))))
    (clim:with-drawing-options (medium :ink clim:+background+)
      (clim:draw-line* medium (- caret-x 2) y1 caret-x y2)
      (clim:draw-line* medium (+ caret-x 2) y1 caret-x y2))
    ))

(defun erase-drawn-markers (wed)
  (let ((markers-pos (wed-drawn-markers wed)))
    (when (car markers-pos)
      (erase-drawn-marker wed (car markers-pos)))
    (mapc #'(lambda (marker-pos)
              (when marker-pos
                (erase-drawn-marker wed marker-pos :align-y :top)))
          (cdr markers-pos))
    (setf (wed-drawn-markers wed) nil)
    ))

(defun refresh-wed (wed &optional (force-p nil))
  (unless (and (wed-win wed) (clim:window-visibility (wed-win wed)))
    (return-from refresh-wed (values)))
  (erase-drawn-markers wed)
  (let* ((top-row  (wed-top-row      wed))
         (last-top (wed-last-top-row wed))
         (num-rows (wed-num-rows wed))
         (line (wed-top-line wed))
         (wedlin (ed-line wed line))
         (row (wedlin-row wedlin)))
    (loop
      (if (or force-p
              (/= (- row top-row)
                  (- (or (wedlin-moved-from wedlin) (wedlin-row wedlin)) last-top)))
        (draw-string-layout wed wedlin row 0)
        (when (wedlin-draw-from wedlin)
          (draw-string-layout wed wedlin row (wedlin-draw-from wedlin)))
        )
      (setf (wedlin-row wedlin) row)
      (setf ; (wedlin-moved-from wedlin) nil
            (wedlin-draw-from  wedlin) nil)
      (incf row (wedlin-num-rows wedlin))
      (incf line)
      (when (or (>= row (+ top-row num-rows)) (>= line (length (ed-lines wed))))
        (return (values)))
      (setf wedlin (ed-line wed line))
      )
    (setf (wed-last-top-row wed) top-row)
    (let ((rel-row (- row top-row)))
      (when (and (< rel-row (wed-max-row wed)) (< rel-row num-rows))
        (clear-row-from-x wed row (wed-left-margin wed) (- num-rows rel-row)))
      (setf (wed-max-row wed) rel-row))
    (draw-markers wed)
    (values)))

(defun position-marker (wed x y)
  (let* ((y-row (y-row-position wed y))
         (line (position-if #'(lambda (wedlin)
                                (> (wedlin-row wedlin) y-row))
                            (ed-lines wed)))
         (real-line (cond ((null line) (1- (length (ed-lines wed))))
                          ((> line 0) (1- line))
                          (t 0))))
    (let* ((wedlin (ed-line wed real-line))
           (col (string-layout-calc wed :col (cons x (min (1- (wedlin-num-rows wedlin))
                                                          (- y-row (wedlin-row wedlin))))
                                    wedlin)))
      (make-marker real-line col))
    ))

(defun position-set-marker (wed x y marker-num &optional (markers (ed-markers wed)))
  (erase-drawn-markers wed)
  (setf (markers-marker markers marker-num) (position-marker wed x y))
  (draw-markers wed)
  )

(defun wed-command-case (wed command)
  (case command
    (refresh (refresh-wed wed t))
    (layout  (wed::ensure-wed-fills-window wed :left-margin t :right-margin t :top t :bottom t)
             (wed:update-wed-layout wed t)
             (wed:refresh-wed wed t))
    (t (ed:ed-command wed command)))
  )

(defun wed-command (wed command)
  (wed-command-case wed command))