(in-package :clim-user)

;;; Simple application with one main "bulletin board" pane, in
;;; which inferior panes are created.

(define-application-frame bboard-test ()
  ((inferiors :initform nil :accessor bboard-inferiors))
  (:panes ((bboard :application :scroll-bars nil)
	   (interactor :interactor)))
  (:layout ((main (:column 1
			   (bboard :rest)
			   (interactor 1/5))))))

;;; Utility to read a button press event, ignoring other input.
(defun get-button-press (stream)
  (loop
      (setq event (read-gesture :stream stream))
    (if (typep event 'pointer-button-press-event)
	(return (values (event-window event) 
			(pointer-event-x event)
			(pointer-event-y event)))
	(beep stream))))

;;; Of course, in real life you would write a blank-area translator
;;; that called this command with the x and y coordinates of the
;;; click that invoked it and create the window there.
(define-bboard-test-command (com-create-pane :name t)
    ()
  (let* ((frame *application-frame*)
	 (bboard (get-frame-pane frame 'bboard)))
    ;; Get an x/y location to create the inferior window.
    (multiple-value-bind (win x y) (get-button-press bboard)
      (declare (ignore win))
      ;; Make the pane there.
      (let ((pane (open-window-stream :parent bboard :left x :top y 
				      :width 100 :height 100
				      :input-buffer (stream-input-buffer bboard))))
	(push pane (bboard-inferiors frame)) ; remember it
	(window-expose pane)		; see it
	(force-output bboard)))))

;;; Read the comment on com-create-pane.
;;; Click on a pane, then on a new place for it.
;;; Has at least one bug: if you click on an inferior (rather than
;;; the bboard pane itself) for the new position the pane will
;;; be moved to the wrong place.
(define-bboard-test-command (com-move-pane :name t)
    ()
  (let* ((frame *application-frame*)
	 (bboard (get-frame-pane frame 'bboard)))
    ;; Click on a pane.
    (let ((pane (get-button-press bboard)))
      ;; Make sure it is one of the known inferiors
      (if (find pane (bboard-inferiors frame))
	  ;; Get a new place to put it.
	  (multiple-value-bind (win x y) (get-button-press bboard)
	    (declare (ignore win))
	    ;; Move it.  I can't believe that this is unexported.
	    ;; Can use (setf (bounding-rectangle-min-x pane) x)
	    ;; and repeat for y...
	    (clim::bounding-rectangle-set-position* pane x y))
	  (beep bboard)))))

(define-bboard-test-command (com-delete-pane :name t)
    ()
  (let* ((frame *application-frame*)
	 (bboard (get-frame-pane frame 'bboard)))
    ;; Click on a pane.
    (let ((pane (get-button-press bboard)))
      ;; Make sure it is one of the known inferiors
      (if (find pane (bboard-inferiors frame))
	  (progn
	    (setf (bboard-inferiors frame)
		  (delete pane (bboard-inferiors frame)))
	    (setf (window-visibility pane) nil)
	    (force-output bboard))
	  (beep)))))

;;; This command lays out all the panes in a diagonal
(define-bboard-test-command (com-align-panes :name t)
    ()
  (let* ((frame *application-frame*)
	 (bboard (get-frame-pane frame 'bboard))
	 (x 0) (y 0) (increment 20))
    ;; Traverse the panes from oldest to newest
    (dolist (pane (reverse (bboard-inferiors frame)))
      ;; Move each to new location
      (clim::bounding-rectangle-set-position* pane x y)
      (incf x increment) (incf y increment))
    (force-output bboard)))

;;; Create the frame and run it.  Try the "Create Pane" and "Move Pane"
;;; commands.
(defun test-bboard (root)
  (let ((frame (make-application-frame 'bboard-test
				       :parent root
				       :width 500 :height 500)))
    (run-frame-top-level frame)
    frame))
