;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: PS; Base:10 -*-

 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; ;                                                                         ; ;
; ;                           postscript.lisp                               ; ;
; ;                                                                         ; ;
 ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;  Programmer:  Erik Eilerts

#|*****************************************************************************
 |
 |  Copyright 1991 University of Texas at Austin
 |
 |  Permission to use, copy, modify, distribute, and sell this software and its
 |  documentation for any purpose is hereby granted without fee, provided that
 |  the above copyright notice appear in all copies and that both that
 |  copyright notice and this permission notice appear in supporting
 |  documentation, and that the name of UT-Austin not be used in advertising
 |  or publicity pertaining to distribution of the software without specific
 |  specific, written prior permission.  UT-Austin makes no representations 
 |  about the suitability of this software for any purpose.  It is provided 
 |  "as is" without express or implied warranty.
 |
 |  UT-AUSTIN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 |  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL 
 |  UT-AUSTIN BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 |  OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, 
 |  WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, 
 |  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS 
 |  SOFTWARE.
 |
 |*****************************************************************************
 |#

(in-package 'ps)

;; -------------------------------------------------------------------
;;                      Variables to CHANGE
;; -------------------------------------------------------------------


;; POSTSCRIPT-AFM-DIRECTORY - points to the directory that has the
;; Abobe Font Metrics (AFM) files, ie Courier.afm, Courier-Bold.afm, etc.
;; It's /lusr/lib/ps on our systems, but I have no idea what it might be
;; on your system.  If you do:  
;;      man AFM    or    man afm
;; you should be able to figure out if you have it and where it is.

(defvar *postscript-afm-directory* "/lusr/lib/ps/")      ;; <<<== CHANGE


;; OUTPUT-PATH - default file and directory to store the drawing to

(defvar *default-output-path* "/v/sage/v0/brewery/tree-display.ps")


;; OUTPUT-DEVICE-ORIGINAL-BOUNDARIES - the boundaries of your laser printer.
;; Each laser printer has a certain (left, top, right, bottom) set of
;; coordinates past which the postscript drawing gets clipped.  I have computed
;; the coordinates for an Apple Laser Writer to be (22 784 592 8).

(defvar *output-device-original-boundaries* '(22 784 592 8))

;; -------------------------------------------------------------------
;;                           Default Variables
;; -------------------------------------------------------------------

;; DEFAULT-POSTSCRIPT-FONT - font used for text strings if no font parameter
;;                          is passed to the draw-text function

(defvar *default-postscript-font* (clim:make-text-style :serif :roman :normal))


;; DESTINATION - the destination of the postscript drawing.
;;   :laser-printing      -  it will be sent to a laser printer
;;   :document-inclusion  -  it will be included in a document, such as LaTeX

(defvar *default-destination* :laser-printing)


;; ORIENTATION - the orientation of the drawing when it's printed
;;   :portrait   -  no change to coordinates
;;   :landscape  -  all coordinates are rotated by 90 degrees

(defvar *default-orientation* :portrait)


;; NUMBER-OF-PAGES - whether to dispaly the drawing across multiple pages
;;                  or not.
;;   :one       -  the drawing will be scaled so that it all fits on one page
;;   :multiple  -  the drawing will be divided up with each part being
;;                drawn on a separate page (this option only works when
;;                the destination is :laser-printing)

(defvar *default-number-of-pages* :one)


;; DISPLAY-DATE  -  display the current date on the plot
;;   t   -  displays the date
;;   nil -  the date's not displayed

(defvar *default-display-date* t)


;; PLOT-LABEL  -  A text string to display at the bottom or top of the drawing
;;                When the date is also displayed, the string will appear to
;;                the right of the date.
;;   nil       -  no string to draw
;;   "string"  -  the string to draw

(defvar *default-plot-label* nil)


;; LABEL-LOCATION  -  location on the page to display the date and plot label
;;   :top     -  the date/label string starts in the upper left corner
;;   :bottom  -  the date/label string starts in the bottom left corner

(defvar *default-label-location* :bottom)


;; LABEL-SCALE  -  the scale at which to draw the date/label string
;;   > 1 = enlarged
;;   1.0 = normal size
;;   < 1 = shrunk

(defvar *default-label-scale* 0.7)


;; MULTI-PAGE-SCALE  -  the scale of the drawings when number-of-pages is
;;                     :multiple.
;;   > 1 = enlarged
;;   1.0 = normal size
;;   < 1 = shrunk

(defvar *default-multi-page-scale* 0.7)


;; -------------------------------------------------------------------
;;                           Global variables
;; -------------------------------------------------------------------

(defvar *postscripter* nil)            ;; the default postscript handler

(defvar *use-postscript-sizing* nil)   ;; whether to use postscript or
                                       ;; CLIM sizing on text strings

(defparameter *black* 0)               ;; 0 = BLACK in Postscript
(defparameter *white* 1)               ;; 1 = WHITE in Postscript

;; -------------------------------------------------------------------
;;                           Postscript Handler
;; -------------------------------------------------------------------

(defclass postscript-handler ()
  ((output-path :accessor output-path :initform *default-output-path*)
   (destination :accessor destination :initform *default-destination*)
   (orientation :accessor orientation :initform *default-orientation*)
   (number-of-pages :accessor number-of-pages
		    :initform *default-number-of-pages*)
   (display-date :accessor display-date :initform *default-display-date*)
   (plot-label :accessor plot-label :initform *default-plot-label*)
   (label-location :accessor label-location :initform *default-label-location*)
   (label-scale :accessor label-scale :initform *default-label-scale*)
   (multi-page-scale :accessor multi-page-scale
		     :initform *default-multi-page-scale*)

   (temp-output-path :accessor temp-output-path :initform nil)
   (fonts-in-use :accessor fonts-in-use :initform nil)
   (font-index :accessor font-index :initform 0)
   (font-list :accessor font-list :initform nil)
   (current-font  :accessor current-font)
   (current-window :accessor current-window :initform nil)
   (linewidth :accessor linewidth :initform 1)
   (grayval :accessor grayval :initform 0)              ;; 0 = black
   (current-color :accessor current-color :initform 0)  ;; 0 = black
   (current-line-dashes :accessor current-line-dashes :initform nil)
   (postscript-output-file :accessor postscript-output-file :initform nil)
   (compute-boundingbox :accessor compute-boundingbox :initform t)
   (clip-region-active :accessor clip-region-active :initform nil)
   (bounding-box :accessor bounding-box :initform nil)
   (output-device-boundaries :accessor output-device-boundaries 
                :initform (copy-list *output-device-original-boundaries*))
   (output-device-original-boundaries 
                :accessor output-device-original-boundaries 
                :initform *output-device-original-boundaries*)
  ))


(defmethod initialize-instance :after ((postscripter postscript-handler) &key)
  (with-slots (font-list) postscripter
    (dolist (font-metric *postscript-font-metrics*)
      (let* ((font-name (first font-metric))
	     (font-height (second font-metric))
	     (font-ascent (third font-metric))
	     (font-descent (fourth font-metric))
	     (char-widths (cddddr font-metric))
	     (font-array (make-array 256 :initial-contents char-widths)))

	(push (make-instance 'postscript-font
			     :font-height font-height
			     :font-ascent font-ascent
			     :font-descent font-descent
			     :font-name font-name
			     :font-array font-array) font-list)))

    (setf font-list (nreverse font-list))

    (initialize-postscript-variables postscripter)))


(defmethod initialize-postscript-variables ((postscripter postscript-handler))
  (with-slots (current-font fonts-in-use clip-region-active
	       output-device-boundaries compute-boundingbox bounding-box)
              postscripter
    (setf current-font nil)
    (setf fonts-in-use nil)
    (setf clip-region-active nil)
    (setf output-device-boundaries
	  (copy-list (output-device-original-boundaries postscripter)))
    (setf compute-boundingbox t)
    ;; use list here, not quote.  Otherwise, the destructive
    ;; (setf (first bounding-box)  xx) will forever change the quoted
    ;; value (which is supposed to be constant).
    (setf bounding-box (list 999999 999999 -999999 -999999))))

	

(export '(*use-postscript-sizing* *postscript-afm-directory*
	  *default-postscript-font* *postscripter* *postscript-font-metrics*
	  with-postscript-sizing postscript-handler
	  setup-output-configuration open-stream close-stream
	  output-path current-window destination orientation multi-page-scale
	  number-of-pages display-date plot-label label-location label-scale
	  text-style-ascent text-style-descent text-style-height text-size
	  set-bounding-box compute-bounding-box next-page
	  start-region end-region comment
	  translate scale
	  set-clipping-box set-super-clipping-box end-clipping-box
	  draw-text draw-line draw-vector draw-circle draw-arc
	  draw-point draw-rectangle draw-segment draw-lines
	  draw-polygon draw-ring draw-icon)
	'ps)


;; -------------------------------------------------------------------
;;                        Helping Functions
;; -------------------------------------------------------------------

#+symbolics
(defun get-author () 
  (scl::send (scl::send si::*user* :primary-name)))

#+lucid 
(defun get-author ()
  (string-upcase (environment-variable "USER")))

#-(or symbolics lucid)  ;don't know how to get the user's name
(defun get-author () "J. Random User")


(defun string-char-search (char-set string)
  (setq char-set (mapcar #'string char-set))
  (do ((i 0 (1+ i)))
      ((> i (1- (length string)))
       nil)
    (dolist (char char-set)
      (when (string= string (string char) :start1 i :end1 (1+ i))
        (return-from string-char-search i)))))

(defun find-substring-after-string (string1 string2)
  (let ((string1-len (length string1))
	(string2-len (length string2))
	(is-a-subseq nil))
    (if (> string1-len string2-len)
        nil
        (do ((i 0 (1+ i)))
            ((or (> i (- string2-len string1-len)) is-a-subseq)
	     is-a-subseq)
          (when (string= string1 string2 :start2 i :end2 (+ i string1-len))
	     (return-from find-substring-after-string 
			  (subseq string2 (+ i string1-len))))))))


(defun find-substring-before-char (char string)
  (let ((char-index (string-char-search (list char) string)))
    (when char-index
      (subseq string 0 char-index))))

(defun find-substring-before-string (string1 string2)
  (let ((string1-len (length string1))
	(string2-len (length string2))
	(is-a-subseq nil))
    (if (> string1-len string2-len)
        nil
        (do ((i 0 (1+ i)))
            ((or (> i (- string2-len string1-len)) is-a-subseq)
	     is-a-subseq)
          (when (string= string1 string2 :start2 i :end2 (+ i string1-len))
	     (return-from find-substring-before-string 
		  (subseq string2 0 (- (+ i string1-len) string1-len))))))))

(defun blank-string? (string)
  (if (not (stringp string))
      T
      (do ((i 0 (1+ i)))
          ((>= i (length string))
          T)
        (when (not (char= (char string i) #\space))
          (return-from blank-string? NIL)))))


;; -------------------------------------------------------------------
;;                           Postscript Font
;; -------------------------------------------------------------------

(defclass postscript-font ()
  ((font-height :initarg :font-height :accessor font-height)
   (font-ascent :initarg :font-ascent :accessor font-ascent)
   (font-descent :initarg :font-descent :accessor font-descent)
   (font-name :initarg :font-name :accessor font-name)
   (font-array :initarg :font-array :accessor font-array)))

      
(defvar *postscript-font-metrics* '(
(:|Courier| 1085 795 290 NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600
 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600
 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600
 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600
 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600
 600 600 600 600 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 600 600 600 600 600 600 600 600 600 600 600 600 600 NIL NIL NIL 600 600 600
 600 NIL 600 600 600 600 600 600 600 NIL NIL 600 NIL 600 600 600 600 600 600
 600 600 NIL 600 600 NIL 600 600 600 600 NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL 600 NIL NIL NIL NIL 600 600 NIL 600 NIL
 NIL NIL NIL NIL NIL NIL NIL NIL 600 NIL NIL 600 600 NIL 600 NIL NIL NIL NIL)
(:|Courier-Bold| 1205 855 350 NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600
 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600
 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600
 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600
 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600
 600 600 600 600 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 600 600 600 600 600 600 600 600 600 600 600 600 600 NIL NIL NIL 600 600 600
 600 NIL 600 600 600 600 600 600 600 NIL NIL 600 NIL 600 600 600 600 600 600
 600 600 NIL 600 600 NIL 600 600 600 600 NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL 600 NIL NIL NIL NIL 600 600 NIL 600 NIL
 NIL NIL NIL NIL NIL NIL NIL NIL 600 NIL NIL 600 600 NIL 600 NIL NIL NIL NIL)
(:|Courier-Oblique| 1085 795 290 NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600
 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600
 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600
 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600
 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600
 600 600 600 600 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 600 600 600 600 600 600 600 600 600 600 600 600 600 NIL NIL NIL 600 600 600
 600 NIL 600 600 600 600 600 600 600 NIL NIL 600 NIL 600 600 600 600 600 600
 600 600 NIL 600 600 NIL 600 600 600 600 NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL 600 NIL NIL NIL NIL 600 600 NIL 600 NIL
 NIL NIL NIL NIL NIL NIL NIL NIL 600 NIL NIL 600 600 NIL 600 NIL NIL NIL NIL)
(:|Courier-BoldOblique| 1205 855 350 NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600
 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600
 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600
 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600
 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600 600
 600 600 600 600 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 600 600 600 600 600 600 600 600 600 600 600 600 600 NIL NIL NIL 600 600 600
 600 NIL 600 600 600 600 600 600 600 NIL NIL 600 NIL 600 600 600 600 600 600
 600 600 NIL 600 600 NIL 600 600 600 600 NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL 600 NIL NIL NIL NIL 600 600 NIL 600 NIL
 NIL NIL NIL NIL NIL NIL NIL NIL 600 NIL NIL 600 600 NIL 600 NIL NIL NIL NIL)
(:|Helvetica| 1156 939 217 NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL 278 278 355 556 556 889 667 222 333 333 389 584 278 333 278
 278 556 556 556 556 556 556 556 556 556 556 278 278 584 584 584 556 1015 667
 667 722 722 667 611 778 722 278 500 667 556 833 722 778 667 778 722 667 611
 722 667 944 667 667 611 278 278 278 469 556 222 556 556 500 556 556 278 556
 556 222 222 500 222 833 556 556 556 556 333 500 278 556 500 722 500 500 500
 334 260 334 584 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 333 556 556 167 556 556 556 556 191 333 556 333 333 500 500 NIL 556 556 556
 278 NIL 537 350 222 333 333 556 1000 1000 NIL 611 NIL 333 333 333 333 333 333
 333 333 NIL 333 333 NIL 333 333 333 1000 NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL 1000 NIL 370 NIL NIL NIL NIL 556 778 1000 365 NIL
 NIL NIL NIL NIL 889 NIL NIL NIL 278 NIL NIL 222 611 944 611 NIL NIL NIL NIL)
(:|Helvetica-Bold| 1157 936 221 NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL 278 333 474 556 556 889 722 278 333 333 389 584 278 333 278
 278 556 556 556 556 556 556 556 556 556 556 333 333 584 584 584 611 975 722
 722 722 722 667 611 778 722 278 556 722 611 833 722 778 667 778 722 667 611
 722 667 944 667 667 611 333 278 333 584 556 278 556 611 556 611 556 333 611
 611 278 278 556 278 889 611 611 611 611 389 556 333 611 556 778 556 556 500
 389 280 389 584 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 333 556 556 167 556 556 556 556 238 500 556 333 333 611 611 NIL 556 556 556
 278 NIL 556 350 278 500 500 556 1000 1000 NIL 611 NIL 333 333 333 333 333 333
 333 333 NIL 333 333 NIL 333 333 333 1000 NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL 1000 NIL 370 NIL NIL NIL NIL 611 778 1000 365 NIL
 NIL NIL NIL NIL 889 NIL NIL NIL 278 NIL NIL 278 611 944 611 NIL NIL NIL NIL)
(:|Helvetica-Oblique| 1156 939 217 NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL 278 278 355 556 556 889 667 222 333 333 389 584 278 333 278
 278 556 556 556 556 556 556 556 556 556 556 278 278 584 584 584 556 1015 667
 667 722 722 667 611 778 722 278 500 667 556 833 722 778 667 778 722 667 611
 722 667 944 667 667 611 278 278 278 469 556 222 556 556 500 556 556 278 556
 556 222 222 500 222 833 556 556 556 556 333 500 278 556 500 722 500 500 500
 334 260 334 584 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 333 556 556 167 556 556 556 556 191 333 556 333 333 500 500 NIL 556 556 556
 278 NIL 537 350 222 333 333 556 1000 1000 NIL 611 NIL 333 333 333 333 333 333
 333 333 NIL 333 333 NIL 333 333 333 1000 NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL 1000 NIL 370 NIL NIL NIL NIL 556 778 1000 365 NIL
 NIL NIL NIL NIL 889 NIL NIL NIL 278 NIL NIL 222 611 944 611 NIL NIL NIL NIL)
(:|Helvetica-BoldOblique| 1157 936 221 NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL 278 333 474 556 556 889 722 278 333 333 389 584 278 333 278
 278 556 556 556 556 556 556 556 556 556 556 333 333 584 584 584 611 975 722
 722 722 722 667 611 778 722 278 556 722 611 833 722 778 667 778 722 667 611
 722 667 944 667 667 611 333 278 333 584 556 278 556 611 556 611 556 333 611
 611 278 278 556 278 889 611 611 611 611 389 556 333 611 556 778 556 556 500
 389 280 389 584 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 333 556 556 167 556 556 556 556 238 500 556 333 333 611 611 NIL 556 556 556
 278 NIL 556 350 278 500 500 556 1000 1000 NIL 611 NIL 333 333 333 333 333 333
 333 333 NIL 333 333 NIL 333 333 333 1000 NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL 1000 NIL 370 NIL NIL NIL NIL 611 778 1000 365 NIL
 NIL NIL NIL NIL 889 NIL NIL NIL 278 NIL NIL 278 611 944 611 NIL NIL NIL NIL)
(:|Times-Roman| 1156 904 252 NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL 250 333 408 500 500 833 778 333 333 333 500 564 250 333 250
 278 500 500 500 500 500 500 500 500 500 500 278 278 564 564 564 444 921 722
 667 667 722 611 556 722 722 333 389 722 611 889 722 722 556 722 667 556 611
 722 722 944 722 722 611 333 278 333 469 500 333 444 500 444 500 444 333 500
 500 278 278 500 278 778 500 500 500 500 333 389 278 500 500 722 500 500 444
 480 200 480 541 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 333 500 500 167 500 500 500 500 180 444 500 333 333 556 556 NIL 500 500 500
 250 NIL 453 350 333 444 444 500 1000 1000 NIL 444 NIL 333 333 333 333 333 333
 333 333 NIL 333 333 NIL 333 333 333 1000 NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL 889 NIL 276 NIL NIL NIL NIL 611 722 889 310 NIL
 NIL NIL NIL NIL 667 NIL NIL NIL 278 NIL NIL 278 500 722 500 NIL NIL NIL NIL)
(:|Times-Bold| 1221 965 256 NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL 250 333 555 500 500 1000 833 333 333 333 500 570 250 333 250
 278 500 500 500 500 500 500 500 500 500 500 333 333 570 570 570 500 930 722
 667 722 722 667 611 778 778 389 500 778 667 944 722 778 611 778 722 556 667
 722 722 1000 722 722 667 333 278 333 581 500 333 500 556 444 556 444 333 500
 556 278 333 556 278 833 556 500 556 556 444 389 333 556 500 722 500 500 444
 394 220 394 520 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 333 500 500 167 500 500 500 500 278 500 500 333 333 556 556 NIL 500 500 500
 250 NIL 540 350 333 500 500 500 1000 1000 NIL 500 NIL 333 333 333 333 333 333
 333 333 NIL 333 333 NIL 333 333 333 1000 NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL 1000 NIL 300 NIL NIL NIL NIL 667 778 1000 330 NIL
 NIL NIL NIL NIL 722 NIL NIL NIL 278 NIL NIL 278 500 722 556 NIL NIL NIL NIL)
(:|Times-Italic| 1182 930 252 NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL 250 333 420 500 500 833 778 333 333 333 500 675 250 333 250
 278 500 500 500 500 500 500 500 500 500 500 333 333 675 675 675 500 920 611
 611 667 722 611 611 722 722 333 444 667 556 833 667 722 611 722 611 500 556
 722 611 833 611 556 556 389 278 389 422 500 333 500 500 444 500 444 278 500
 500 278 278 444 278 722 500 500 500 500 389 389 278 500 444 667 444 444 389
 400 275 400 541 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 389 500 500 167 500 500 500 500 214 556 500 333 333 500 500 NIL 500 500 500
 250 NIL 523 350 333 556 556 500 889 1000 NIL 500 NIL 333 333 333 333 333 333
 333 333 NIL 333 333 NIL 333 333 333 889 NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL 889 NIL 276 NIL NIL NIL NIL 556 722 944 310 NIL
 NIL NIL NIL NIL 667 NIL NIL NIL 278 NIL NIL 278 500 667 500 NIL NIL NIL NIL)
(:|Times-BoldItalic| 1223 973 250 NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL 250 389 555 500 500 833 778 333 333 333 500 570 250 333 250
 278 500 500 500 500 500 500 500 500 500 500 333 333 570 570 570 500 832 667
 667 667 722 667 667 722 778 389 500 667 611 889 722 722 611 722 667 556 611
 722 667 889 667 611 611 333 278 333 570 500 333 500 500 444 500 444 333 500
 556 278 278 500 278 778 556 500 500 500 389 389 278 556 444 667 500 444 389
 348 220 348 570 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 389 500 500 167 500 500 500 500 278 500 500 333 333 556 556 NIL 500 500 500
 250 NIL 500 350 333 500 500 500 1000 1000 NIL 500 NIL 333 333 333 333 333 333
 333 333 NIL 333 333 NIL 333 333 333 1000 NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL 944 NIL 266 NIL NIL NIL NIL 611 722 944 300 NIL
 NIL NIL NIL NIL 722 NIL NIL NIL 278 NIL NIL 278 500 722 500 NIL NIL NIL NIL)
(:|Symbol| 1303 1010 293 NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL 250 333 713 500 549 833 778 439 333 333 500 549 250 549 250
 278 500 500 500 500 500 500 500 500 500 500 278 278 549 549 549 444 549 696
 660 710 612 652 763 603 765 351 631 724 686 918 739 750 768 741 580 592 632
 690 439 768 645 795 650 333 863 333 658 500 500 631 549 549 494 439 521 411
 603 329 603 549 549 576 521 549 549 521 549 603 439 576 713 686 493 686 494
 480 200 480 549 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL NIL
 620 247 549 167 713 500 753 753 753 753 1042 987 603 987 603 400 549 411 549
 549 713 494 460 549 549 549 549 1000 603 1000 658 823 686 795 987 768 768 823
 768 768 713 713 713 713 713 713 713 768 713 790 790 890 823 549 250 713 603
 603 1042 987 603 987 603 494 329 790 790 786 713 384 384 384 384 384 384 494
 494 494 494 790 329 274 686 686 686 384 384 384 384 384 384 494 494 494 NIL)
))


;; After everything has been defined, make an instance of the ps handler.

(setf *postscripter* (make-instance 'postscript-handler))

