;;; -*- Package: CLIM-USER; Syntax: Common-Lisp; Base: 10 -*-
;;; Modifie par Vincent Keunen le 93/06/03 a 11:06 sur un Macintosh IIfx
;;;  Le nom du fichier est Documents:Lisp:clim:library:clim-1:physical-page-utilities.lisp
;;;  Modifications de cette version : -
;;; MI2:>keunen>clim>examples>physical-page-utilities.lisp.24,
;;;   Edited by Keunen running on NRBMI1 on 16-Feb-93 17:38:41


#|
This code includes various utilities to generate reports on standard sheets of paper
(presently, only A4 page sizes have been tested - hint).  It is VERY limited; actually
it does what I need to do for my project!...  Nevertheless, feel free to use it.

The basic part of the program is a preview mechanism: you preview a "physical page" (ie
a real paper sheet) on the screen, then you have the option to print it.  This lets you
experiment with clim to get to the desired result without wasting too much paper.  This is
particularly useful because CLIM does not match very well screen and paper output, as you
will soon see.  Actually, what you will see will not be exactly what you will get 
(wywswnbewywg 8-O ).  I tried to adapt a number of things so that printing is closer to
the screen output, but...  One thing I want to do is to try and use the postscript language
utility (PS) from Erik Eilerts.  

You can start by evaluating (start-up-physical-page-frame).  Then a number of
commands appear in a new framework:

"Page Borders"      : displays the borders of the real paper sheet
"Print Area"        : displays the Apple LW printable area (found by trial & error)
"Grid"              : displays a grid with coordinates (easier to guess where to draw!)
"Content"           : displays the actual contents of the page (draw-page-content)
"Print"             : generates a ps file (prints it under Genera)
"Clear"             : clears the whole display area
"Clear Print Area"  : clears only the print area (needs work)
"Exit"              : you'll have to guess this one

To test your own output, modify the function "draw-page-content".

|#


(in-package :clim-user)

(export '(*beginning-y-position* 
	   *current-y-position* draw-page-content draw-title
	   modified-stream-set-cursor-position* start-up-physical-page-frame
	   *the-physical-page-frame-main-window* with-modified-stream-cursor-position
	   draw-title))

(defvar *beginning-y-position* 40)

(defvar *current-y-position* *beginning-y-position*
  "This variable controls the y position for titles.  It is incremented by
   draw-banner-title, draw-simple-title,...")


;;;;----------------------------------------------------------------
;;;;Page sizes definition

(defclass page-descriptor ()
    ((page-kind :accessor page-kind :initform :A4 :initarg :page-kind)
     (physical-page-heigth :accessor physical-page-heigth
			   :initarg :physical-page-heigth)
     (physical-page-width :accessor physical-page-width
			  :initarg :physical-page-width)
     (print-area-heigth :accessor print-area-heigth
			:initarg :print-area-heigth)
     (print-area-width :accessor print-area-width
		       :initarg :print-area-width)))

(defvar *A4-page-descriptor*
	(make-instance 'page-descriptor
		       :page-kind :A4
		       :physical-page-heigth 700
		       :physical-page-width 500
		       :print-area-heigth 686
		       :print-area-width 469))
#+unimplemented
(defvar *US-letter-page-descriptor*
	(make-instance 'page-descriptor
		       :page-kind :US-letter
		       :physical-page-heigth ???
		       :physical-page-width ???
		       :print-area-heigth ???
		       :print-area-width ???))
#+unimplemented
(defvar *Legal-letter-page-descriptor*
	(make-instance 'page-descriptor
		       :page-kind :Legal-letter
		       :physical-page-heigth ???
		       :physical-page-width ???
		       :print-area-heigth ???
		       :print-area-width ???))
#+unimplemented
(defvar *B5-page-descriptor*
	(make-instance 'page-descriptor
		       :page-kind :B5
		       :physical-page-heigth ???
		       :physical-page-width ???
		       :print-area-heigth ???
		       :print-area-width ???))


(defvar *current-page-descriptor* nil)
(defvar *ps-output-file* nil)

;;;choose the page you want to use here ---------------------------
(setf *current-page-descriptor* *A4-page-descriptor*)
(setf *ps-output-file* (make-pathname :name "physical-page"
				      :type "ps"))



;;;;----------------------------------------------------------------
;;;; Frame definition

(defvar *the-physical-page-frame* nil)
(defvar *the-physical-page-frame-main-window* nil)
(defvar *space-to-the-left-of-page* 100)
(defvar *space-above-page* 40)

(defvar *ps-stream/screen-stream-x-delta* -10)          ;-19)
(defvar *ps-stream/screen-stream-y-delta* -64)


(define-application-frame physical-page-frame ()
    ((one-state-variable :initform nil
			 :accessor one-state-variable)
     (page-borders-displayed :initform nil
			     :accessor page-borders-displayed?)
     (grid-displayed :initform nil :accessor grid-displayed?)
     (print-area-displayed :initform nil :accessor print-area-displayed?))
  (:command-table (physical-page-frame
		    :inherit-from
		    (user-command-table accept-values-pane)))
  (:command-definer t)
  (:panes ((display :application)
	   (menu :command-menu)
	   (interactor :interactor)))
  (:layout (#-MCL
            (high
	      (:row 1
	       (:column :rest
		(menu :rest))
	       (:column 8/10
		(display :rest))))
            #+MCL
            (high
	      (:row 1
	            (:column 1
		             (display :rest))))
	    (standard
	      (:column 1
	       (display :rest)
	       (menu :compute)
	       (interactor 1/5))))))


;;;;----------------------------------------------------------------
;;;; Starting up the frame

#+Genera
(define-genera-application physical-page-frame :select-key #\4)


(defun start-up-physical-page-frame ()
  (run-frame-top-level
    (make-application-frame
      'physical-page-frame
      :parent
      ;;thanks to Oliver Christ <oli@adler.ims.uni-stuttgart.de> for the code below
      #+Lucid (open-root-window
		:clx
		:host (lcl:environment-variable "DISPLAY"))
      #+Allegro-v4.1 (open-root-window
		       :clx
		       :host (system:getenv "DISPLAY"))
      #+MCL          (open-root-window :mcl)
      #+Genera       (open-root-window :sheet)
      ;;Please contact keunen@nrb.be if you modify this source code.
      #-(or Lucid Allegro-v4.1 MCL Genera)
      (warning "Unknown CLIM/LISP combination.  Please modify the
start-up-physical-page-frame function to your needs."))))


;;;;----------------------------------------------------------------
;;;; For easy access

(defmethod run-frame-top-level :before
	   ((the-frame physical-page-frame))
  (setf *the-physical-page-frame* the-frame)
  (setf *the-physical-page-frame-main-window*
	(get-frame-pane *the-physical-page-frame* 'display)))


;;;;----------------------------------------------------------------
;;;; Commands


(define-physical-page-frame-command (com-borders :menu "Page Borders") ()
  (display-physical-page-borders))

(define-physical-page-frame-command (com-show-print-area :menu "Print Area") ()
  (with-translation
    (*the-physical-page-frame-main-window* *space-to-the-left-of-page* *space-above-page*)
    (display-print-area *the-physical-page-frame-main-window*)))

(define-physical-page-frame-command (com-grid :menu "Grid") ()
  (with-translation (*the-physical-page-frame-main-window*
		      *space-to-the-left-of-page*
		      *space-above-page*)
    (draw-grid *the-physical-page-frame-main-window*)))

(define-physical-page-frame-command (com-display-content :menu "Content") ()
  (with-translation
    (*the-physical-page-frame-main-window* *space-to-the-left-of-page* *space-above-page*)
    (draw-page-content)))



(define-physical-page-frame-command (com-print :menu "Print") ()
  #-mcl (clim-utils::make-process
         #'(lambda ()
             (print-page-content 
              :print-grid? (grid-displayed?
			     ;;*application-frame* ;;does not work under genera ???
			     *the-physical-page-frame*)))        ; this works
         :name "Print page content")
  #+mcl (progn
          (print-page-content :print-grid? (grid-displayed? *application-frame*))))


(define-physical-page-frame-command (com-clear :menu "Clear") ()
  (window-clear *the-physical-page-frame-main-window*)
  (setf (page-borders-displayed? *application-frame*) nil)
  (setf (grid-displayed? *application-frame*) nil)
  (setf (print-area-displayed? *application-frame*) nil))

(define-physical-page-frame-command (com-clear-print-area :menu "Clear Print Area") ()
  (with-translation
    (*the-physical-page-frame-main-window* *space-to-the-left-of-page* *space-above-page*)
    (clear-print-area *the-physical-page-frame-main-window*)
    (when (print-area-displayed? *application-frame*)
      (setf (print-area-displayed? *application-frame*) nil)
      (display-print-area))
    (when (grid-displayed? *application-frame*)
      (setf (grid-displayed? *application-frame*) nil)
      (draw-grid *the-physical-page-frame-main-window*))))

(define-physical-page-frame-command (com-exit-physical-page-frame 
                                  :menu #-mcl "Exit" #+mcl "Quit") ()
  (frame-exit *application-frame*))


;;;;----------------------------------------------------------------
;;; Printing utilities

(defclass postscript-view ()
  ())

(defvar +postscript-view+
  (make-instance 'postscript-view))


;;;----------------------------------------------------------------
;;; Page borders display

;;;This is part of the programmers' interface
(defun display-physical-page-borders ()
  (with-translation (*the-physical-page-frame-main-window* -15 -7)
    ;;1/2 delta ps-page/screen-page
    (draw-rectangle* *the-physical-page-frame-main-window*
		     *space-to-the-left-of-page*
		     *space-above-page*
		     (+ *space-to-the-left-of-page*
			(physical-page-width *current-page-descriptor*))
		     (+ *space-above-page*
			(physical-page-heigth *current-page-descriptor*))
		     :filled nil :line-thickness 3
		     :ink (if (page-borders-displayed? *application-frame*)
			      +background+ +foreground+)))
  (setf (page-borders-displayed? *application-frame*)
	(not (page-borders-displayed? *application-frame*))))




;;;----------------------------------------------------------------
;;; Grid display

;;;This is part of the programmers' interface
(defun draw-grid (stream &optional (vertical-spacing 50)
		  (horizontal-spacing 50))
  (let ((the-color
	  (if (grid-displayed? *application-frame*)
	      +background+ +foreground+)))
    (draw-vertical-grid stream :spacing horizontal-spacing :color the-color)
    (draw-horizontal-grid stream :spacing vertical-spacing :color the-color)
    (setf (grid-displayed? *application-frame*)
	  (not (grid-displayed? *application-frame*)))))

(defun print-grid (stream &optional (vertical-spacing 50)
		   (horizontal-spacing 50))
  (let ((the-color +foreground+))
    (draw-vertical-grid stream :spacing horizontal-spacing :color the-color
			:label-position :inside)
    (draw-horizontal-grid stream :spacing vertical-spacing :color the-color
			  :label-position :inside)))

(defun draw-horizontal-grid (stream &key (spacing 50) (label-size :small)
			     (color +foreground+) (label-position :outside)
                             (last-line nil))
  (let ((label-placement (if (eq label-position :outside)
			     20
			     -20)))
    (loop for i from 1
	  for position = (* i spacing)
	  until (>= position
		    (print-area-heigth *current-page-descriptor*))
	  do
      (draw-text* stream (format nil "~a" position)
		  (- label-placement) position :align-x :right :align-y :center
		  :text-size label-size
		  :ink color)
      (draw-line* stream
		  0 position (1- (print-area-width *current-page-descriptor*))
		  position
		  :line-dashes '(3 2)
		  :ink color)
      (draw-text* stream (format nil "~a" position)
		  (+ (print-area-width *current-page-descriptor*) label-placement)
		  position :align-y :center
		  :text-size label-size
		  :ink color)))
  (when last-line
    (draw-line* stream
		0 (print-area-heigth *current-page-descriptor*)
                (1- (print-area-width *current-page-descriptor*))
                (print-area-heigth *current-page-descriptor*)
		:line-dashes '(3 2)
		:ink color)))

(defun draw-vertical-grid (stream &key (spacing 50) (label-size :small)
			   (color +foreground+) (label-position :outside)
                           (last-line nil))
  (let ((label-placement (if (eq label-position :outside)
			     20
			     -20)))
    (loop for i from 1
	  for position = (* i spacing)
	  until (> position (print-area-width *current-page-descriptor*))
	  do
      (draw-text* stream (format nil "~a" position)
		  position (- label-placement) :align-x :center
		  :text-size label-size
		  :ink color)
      (draw-line* stream
		  position 0 position (1- (print-area-heigth *current-page-descriptor*))
		  :line-dashes '(3 2)
		  :ink color)
      (draw-text* stream (format nil "~a" position)
		  position (+ (print-area-heigth *current-page-descriptor*) label-placement)
		  :align-x :center
		  :align-y :top
		  :text-size label-size
		  :ink color)))
  (when last-line
    (draw-line* stream
		(print-area-width *current-page-descriptor*)
		0
		(print-area-width *current-page-descriptor*)
		(1- (print-area-heigth *current-page-descriptor*))
		:line-dashes '(3 2)
		:ink color)))

;;;----------------------------------------------------------------
;;; Print area display

;;;This is part of the programmers' interface
(defun display-print-area (&optional (stream *the-physical-page-frame-main-window*))
  (draw-rectangle* stream 0 0
		   (print-area-width *current-page-descriptor*)
		   (print-area-heigth *current-page-descriptor*)
		   :filled nil
		   :ink (if (print-area-displayed? *application-frame*)
			      +background+ +foreground+))
  (setf (print-area-displayed? *application-frame*)
	(not (print-area-displayed? *application-frame*))))

(defun clear-print-area (&optional (stream *the-physical-page-frame-main-window*))
  (draw-rectangle* stream 0 0
		   (print-area-width *current-page-descriptor*)
		   (print-area-heigth *current-page-descriptor*)
		   :filled t
		   :ink +background+))

;;;----------------------------------------------------------------
;;;; Titles display


(defun draw-banner-title (stream y-position text &key (size :large)
			         (time-stamp :no) (ink +white+))
  ;;Unfortunately, this function contains stream specific code to
  ;; account for clim's not wysiwyg between screen and postscript output.
  (let ((banner-height (case size
	                 (:large 40)
	                 (:small 20)))
	(text-size (typecase stream
	             (clim::postscript-stream
	              (case size
		        (:large :huge)
		        (:small :large)))
	             (t (case size
		          (:large :very-large)
		          (:small :normal)))))
	(time-stamp-size
         (typecase stream
           (clim::postscript-stream :normal)
           (t :small))))
    (draw-rectangle* stream 0 (- y-position (/ banner-height 2))
		     (print-area-width *current-page-descriptor*)
		     (+ y-position (/ banner-height 2))
		     :filled t)
    (draw-text* stream text 10 y-position
		:align-x :left :align-y :center
		:text-size text-size :text-family :sans-serif :text-face :bold
		:ink ink)
    (when (eq time-stamp :yes)
      (draw-text* stream "your time function"  ;(nanesse::current-iso-date-and-time)
		  (- (print-area-width *current-page-descriptor*) 0)
		  y-position
		  :align-x :right :align-y :center
		  :text-size time-stamp-size :text-family :sans-serif :text-face :bold
		  :ink ink))
    (setf *current-y-position* (+ *current-y-position* 40))))


(defun draw-simple-title (stream text y-position)
  (draw-text* stream text 10 y-position
              :align-x :left :align-y :center
              :text-size :normal :text-family :sans-serif :text-face :bold
              :ink clim:+black+)
  (setf *current-y-position* (+ *current-y-position* 20)))


;;; This function "summarizes" all other ones.  Please use this one only for titles.
(defun draw-title (text stream &key (level 1))
  (case level
    (1 (draw-banner-title stream *current-y-position* text))
    (2 (draw-banner-title stream *current-y-position* text :size :small))
    (otherwise (draw-simple-title stream text *current-y-position*))))


;;;----------------------------------------------------------------
;;;; Various utilities


(defun modified-stream-set-cursor-position* (stream x y)
  (let ((x-increment
	  (typecase stream
	    (clim::postscript-stream *ps-stream/screen-stream-x-delta*)
	    (t *space-to-the-left-of-page*)))
	(y-increment
	  (typecase stream
	    (clim::postscript-stream *ps-stream/screen-stream-y-delta*)
	    (t *space-above-page*))))
    (stream-set-cursor-position* stream
				 (+ x-increment x)
				 (+ y-increment y))))

#|
;;;This nice macro is from Philip L. Stubblefield, Rockwell International.
(defmacro with-stream-cursor-position ((stream absolute-x absolute-y) &body body)
  "Sets the cursor position to the given absolute coordinates on STREAM,
   executes the body, and then restores the old cursor position."
  (check-type stream symbol)
  (let ((old-x (gensym "OLD-X-"))
	(old-y (gensym "OLD-Y-")))
    `(multiple-value-bind (,old-x ,old-y)
	 (stream-cursor-position* ,stream)
       (unwind-protect
	   (progn
	     (stream-set-cursor-position* ,stream ,absolute-x ,absolute-y)
	     ,@body)
	 (stream-set-cursor-position* ,stream ,old-x ,old-y)))))
|#


(defmacro with-modified-stream-cursor-position ((stream absolute-x absolute-y) &body body)
  "Sets the cursor position to the given absolute coordinates on STREAM,
   executes the body, and then restores the old cursor position."
  (check-type stream symbol)
  (let ((old-x (gensym "OLD-X-"))
	(old-y (gensym "OLD-Y-")))
    `(multiple-value-bind (,old-x ,old-y)
	 (stream-cursor-position* ,stream)
       (unwind-protect
	   (progn
	     (modified-stream-set-cursor-position* ,stream ,absolute-x ,absolute-y)
	     ,@body)
	 (stream-set-cursor-position* ,stream ,old-x ,old-y)))))


;;;----------------------------------------------------------------
;;; Content display (modify this to suit your needs)

(defun draw-page-content (&optional (stream *the-physical-page-frame-main-window*))
  (setf *current-y-position* *beginning-y-position*)
  (with-drawing-options (stream :ink +black+)
    (draw-title "Example Report" stream :level 1)
    (draw-title "Here is a table" stream :level 2)
    (draw-title "Look at it now!" stream :level 3)
    (draw-table-example stream)
    (draw-title "Wasn't it nice?" stream :level 3)))


(defun draw-table-example (stream)
  (with-modified-stream-cursor-position (stream 20 *current-y-position*)
    (formatting-table (stream
                       :equalize-column-widths t)
      (formatting-row (stream)
        (formatting-cell (stream)
          (format stream "Hello"))
        (formatting-cell (stream)
          (format stream "Hello 2 2 2"))
        (formatting-cell (stream)
          (format stream "Hello 333")))
      (formatting-row (stream)
        (formatting-cell (stream)
          (format stream "Hello line 2"))
        (formatting-cell (stream)
          (format stream "Hello line 2 2 2 2"))
        (formatting-cell (stream)
          (format stream "Hello line 2 333")))))
  (setf *current-y-position* (+ *current-y-position* 50)))     ; 100 = table height



;;;this is part of the programmers' interface
(defun print-page-content (&key (print-grid? nil))
  (with-open-file (the-file *ps-output-file* 
			    :direction :output
			    :if-exists :new-version)
    (with-output-to-postscript-stream
      (stream the-file)
      (with-translation (stream *ps-stream/screen-stream-x-delta*
				*ps-stream/screen-stream-y-delta*)
	(when print-grid? (print-grid stream))
	(draw-page-content stream))))
  #+Genera
  (cp:execute-command "hardcopy file" *ps-output-file*)
  #+mcl
  (format t "To download a postscript file under MCL, use the code in postcript-file-downloading from the clim repository."))
