;;;--------------------------------------------------------------------------------------------------------------------------------
;;        This presentation type called, my-pathname, was developed in Clim1.1 in Franz's ACL.
;;        AUTHOR: Curt Eggemeyer --> curt@eraserhead.jpl.nasa.gov
;;        This is the best pathname completion facility that I could come up with. This provides a very robust completion
;;        capability on both canonicalized file types (pathnames that end with a .extension on them) and regular files. This
;;        presentation type will perform automatic merging of files with the same name but different extensions. In addition,
;;        the user may at any time during the entering of a pathname perform partial completions via hitting the spacebar or
;;        use of the describe facility (throws up a menu of potential partial pathnames)! There are several little problems
;;        encounter using this presentation type:
;;        1) The tab key option for partial pathname completion doesn't work (you have to use the spacebar)! This also implies
;;           that for a command with multiple arguments that uses this presentation type as an argument, it must be the
;;           very last arg! (el-sucko)
;;        2) The menu of partial pathnames thrown up by the describe method, isn't mouse movable, so you should position your
;;           pointer before you hit the good ole <control-?>. Anybody know how to get around this problem!
;;        3) This type doesn't automagically dump a successful accept of my-pathname directory component into *default-pathname-default*
;;           for the next my-pathname accept operation. You must do that following a successful accept by a  ---v
;;                    (setq *default-pathname-default* (make-pathname :directory (pathname-directory whatever-your-pathname-was)))
;;        4) The describe mechanism seems to garble some of the interaction dialog (IS this a CLIM1.1 problem!)
;;
;;--------------------------------------------------------------------------------------------------------------------------------
;;          USE:
;;          (accept 'my-pathname) .. ;expects a pathname for reading from with an extension on it, IF *Pathname-Canonical-Type* is non-nil!
;;          (accept '(my-pathname t) .. ;expects a pathname for writing to (It supplies the extension automatically, IF
;;                                                                                         *Pathname-Canonical-Type* is non-nil!)
;;                    ie: You typed /dir/subdir/filename<cr> and *Pathname-Canonical-Type* is 'fred then you get
;;                                        #p"/dir/subdir/filename.fred" (extensions are always lowercase!)
;;          In a command argument form:
;;          (define-command (com-get-path :name t) ((path 'my-pathname :default (Complete-File-Type 'lisp))) ...) ;Reads in .lisp files!
;;          (define-command (com-write-path :name t) ((path '(my-pathname t) :default (Complete-File-Type 'lisp))) ...)
;;					;Writes out .lisp files!

;;        Anybody out there have any improvements on this, I would like to know about it!
;;            ENJOY!

(eval-when (compile) (proclaim '(optimize (speed 3) (safety 0) (debug 0) (space 1))))

(defvar *Pathname-Canonical-Type* 'lisp "Filetype extension specifier either a symbol or nil")
(defvar *Partial-Pathname* nil "Used exclusively for tracking pathname progress between describe and accept methods")
(defvar *Pathname-Init-Start* nil "Used exclusively for accept buffer position after command prompt")

(defmacro Without-File-Errors (&body FORMS)
  (let ((tag (gensym)))
    `(block ,tag
      (handler-bind ((file-error #'(lambda (c) (return-from ,tag (values nil c)))))
	,@FORMS))))
	   
(defun Flexible-Name (PATHNAME)
  (cond ((Without-File-Errors (truename PATHNAME)))  ;don't let truename barf on you!
	((namestring PATHNAME))))

(defun Complete-Partial-Pathname-Input (string)
  (let* ((wild-path (without-file-errors
		     (directory (make-pathname :name (pathname-name (concatenate 'string string "*"))
					       :directory (pathname-directory string) :type :wild))))
	 (directory-part (directory-namestring (cond ((car wild-path))
						     ((without-file-errors (truename (directory-namestring string))))
						     (string))))
	 (completed-string (complete-from-possibilities (namestring (merge-pathnames directory-part string))
							(mapcar 'flexible-name wild-path) nil :name-key #'namestring
							:value-key 'namestring)))
    (cond ((string= string completed-string) (cons completed-string (mapcar 'namestring wild-path)))
	  ((cdr wild-path) (cons completed-string (mapcar 'namestring wild-path)))
	  ((not (without-file-errors (directory (concatenate 'string completed-string "/"))))
	   (cons completed-string (mapcar 'namestring wild-path)))
	  (t (Complete-Partial-Pathname-Input (concatenate 'string completed-string "/"))))))

(defun Complete-File-Type (NEW-TYPE)
  "(NEW-TYPE)
Arguments NEW-TYPE  File canonical type your looking for (file extension -> ~/a-file.extension)
Returns   A pathname that can be used as the default for my-pathname type given this file extension type
          If a complete file pathname cannot be found then return the present directory path for the user to work from
Side-Effects   This puppy will set the global *Pathname-Canonical-Type* appropriately to be used by the other routines
"
  (declare (special *Pathname-Canonical-Type* *default-pathname-defaults*) (type symbol NEW-TYPE) (type string name can-strg) (type t pos))
  (setf *Pathname-Canonical-Type* NEW-TYPE)
  (let ((name (namestring *default-pathname-defaults*)) pos
	(can-strg (cond ((null *Pathname-Canonical-Type*) "")
			(t (string-downcase (concatenate 'string "." (string-downcase *Pathname-Canonical-Type*)))))))
    (cond ((setq pos (position #\. name :from-end t))
	   (setq name (concatenate 'string (subseq name 0 (min (length name) pos)) can-strg)))
	  ((setq name (concatenate 'string name can-strg))))
    (when (null (probe-file name))
      (when (setq pos (position #\/ name :from-end t))
	(setq name (subseq name 0 (1+ pos)))))
    (cond ((null (pathname-name name)) (make-pathname :name name))
	  (t (make-pathname :name (pathname-name name) :directory (pathname-directory (make-pathname :name name)))))))

(define-presentation-type My-Pathname (&optional (Save-P nil)) ;save-p is a flag (t - pathname to write to, nil - pathname to read from)
  :inherit-from 'Pathname :history t :description "Optionally Canonical-Type pathnames")
(define-presentation-method presentation-typep (object (type My-Pathname))
  (or Save-P (probe-file (namestring object))))
(define-presentation-method presentation-subtypep ((type1 My-Pathname) type2)
  (let ((check-1 (with-presentation-type-parameters (My-Pathname type1) Save-P))
	(check-2 (with-presentation-type-parameters (My-Pathname type2) Save-P)))
    (values (eq check-1 check-2) t)))

(define-presentation-method describe-presentation-type ((type my-pathname) stream plural-count)
  (declare (special *Your-Application* *Partial-Pathname* *Pathname-Canonical-Type* *Pathname-Init-Start*))
  (let ((plural-p (or (eql plural-count t) (and (integerp plural-count) (> plural-count 1)))) menu-choice)
    (when (null plural-p) (write-string "a " stream))
    (write-string "pathname" stream)
    (when plural-p (write-char #\s stream))
    (when *Pathname-Canonical-Type* (format stream " of ~a canonical type" *Pathname-Canonical-Type*))
    (write-string (if Save-P " for writing to" " for reading from") stream)
    (setq menu-choice (menu-choose (sort (if Save-P (Complete-Partial-Pathname-Input *Partial-Pathname*)
					   (cdr (Complete-Partial-Pathname-Input *Partial-Pathname*))) #'string-lessp)
				   :associated-window stream :n-columns 2 :default-style '(:fix :roman :small)
				   :label "Potential Pathnames" :pointer-documentation (get-frame-pane *Your-Application* 'Documentation)))
    (when menu-choice
      (when (without-file-errors (directory (concatenate 'string menu-choice "/")))
	(setq menu-choice (concatenate 'string menu-choice "/")))
      (unless (clim::rescanning-p stream) (replace-input stream menu-choice :buffer-start *Pathname-Init-Start*))
      (setq *Partial-Pathname* menu-choice))))

(define-presentation-method accept ((type my-pathname) (stream stream) (view textual-view) &key default)
  (declare (special *default-pathname-defaults* *pathname-canonical-type* *Pathname-Init-Start* *Partial-Pathname*)
	   (type list a-list) (type t matches) (type string can-strg path old-path old-strg) (type integer buf-start)
	   (type character char))
  (let (old-path success-p a-list matches char buf-start (old-strg (namestring *default-pathname-defaults*))
	(can-strg (cond ((null *pathname-canonical-type*) "")
			(t (concatenate 'string "." (string-downcase *pathname-canonical-type*)))))
	(*Pathname-Init-Start* (clim::input-position stream)))
    (setq *Partial-Pathname* "")
    (loop finally (return nil) do                    ;BIG HARRY LOOP of reading in user input and munging as we go!
	  (setq char (read-gesture :stream stream) buf-start (clim::input-position stream))
	  (cond ((and (= buf-start (1+ *Pathname-Init-Start*)) (char-equal #\~ char))    ;like to show the user where is homedir is!
		 (setq buf-start (1- buf-start) *Partial-Pathname* (namestring (user-homedir-pathname)))
		 (replace-input stream *Partial-Pathname* :buffer-start buf-start))
		((member char '(#\newline #\return :eof))   ; :eof was needed to prevent accept-from-string barfing!
		 (unread-gesture char :stream stream)       ; To get mucho complete facility to work, this code is awful
		 (setq old-path *Partial-Pathname*)
		 (cond (Save-P
			(setq *Partial-Pathname* (namestring (merge-pathnames *Partial-Pathname*
									      (subseq old-strg 0 (position #\. old-strg :from-end t))))
			      a-list (list *Partial-Pathname* *Partial-Pathname*)))
		       ((null (position #\/ (namestring *Partial-Pathname*) :test #'char-equal :from-end t))
			(cond ((string-equal (namestring *Partial-Pathname*) "") (setq old-strg (namestring default)))
			      ((decf buf-start (length *Partial-Pathname*))
			       (setq *Partial-Pathname* (concatenate 'string (namestring default) *Partial-Pathname*) old-path "")))
			(setq *Partial-Pathname* (namestring (merge-pathnames *Partial-Pathname*
									      (subseq old-strg 0 (position #\. old-strg :from-end t))))
			      a-list (Complete-Partial-Pathname-Input *Partial-Pathname*) *Partial-Pathname* (car a-list)))
		       (t (setq a-list (Complete-Partial-Pathname-Input *Partial-Pathname*) *Partial-Pathname* (car a-list))))
		 (unless (clim::rescanning-p stream)
		   (replace-input stream (subseq *Partial-Pathname* (length old-path)) :buffer-start buf-start))
		 (cond ((and Save-P
			     (cond ((null *Pathname-Canonical-Type*)
				    (return-from accept
				      (values (make-pathname :name (concatenate 'string (pathname-name *Partial-Pathname*) can-strg)
							     :directory (pathname-directory *Partial-Pathname*)) 'my-pathname)))
				   ((search can-strg *Partial-Pathname* :test #'string=)
				    (return-from accept
				      (values (make-pathname :name (concatenate 'string (pathname-name *Partial-Pathname*) can-strg)
							     :directory (pathname-directory *Partial-Pathname*)) 'my-pathname)))
				   ((setq *Partial-Pathname* (concatenate 'string *Partial-Pathname* can-strg))
				    (write-string can-strg stream)
				    (return-from accept
				      (values (make-pathname :name (concatenate 'string (pathname-name *Partial-Pathname*) can-strg)
							     :directory (pathname-directory *Partial-Pathname*)) 'my-pathname))))))
		       ((cddr a-list)
			(cond ((and (setq success-p (search can-strg *Partial-Pathname* :test #'char-equal :from-end t))
				    *Pathname-Canonical-Type*)
				  (return-from accept
				    (values (make-pathname :name (concatenate 'string (pathname-name *Partial-Pathname*) can-strg)
							   :directory (pathname-directory *Partial-Pathname*)) 'my-pathname)))
			      ((and *Pathname-Canonical-Type* (char-equal #\. (aref *Partial-Pathname* (1- (length *Partial-Pathname*))))
				    (loop for i in (cdr a-list) do
					  (when (search can-strg i :test #'char-equal :from-end t)
					    (return (setq *Partial-Pathname* i)))))
			       (unless (clim::rescanning-p stream)
				 (replace-input stream (subseq *Partial-Pathname* (length old-path)) :buffer-start buf-start))
			       (return-from accept
				 (values (make-pathname :name (concatenate 'string (pathname-name *Partial-Pathname*) can-strg)
							:directory (pathname-directory *Partial-Pathname*)) 'my-pathname)))
			      ((char-equal (aref *Partial-Pathname* (1- (length *Partial-Pathname*))) #\/)
			       (simple-parse-error "A directory is not a file pathname"))
			      ((and (position #\. *Partial-Pathname* :from-end t) (not success-p))
			       (simple-parse-error "Wrong canonical type completion"))
			      ((probe-file *Partial-Pathname*)
			       (return-from accept
				 (values (make-pathname :name (concatenate 'string (pathname-name *Partial-Pathname*) can-strg)
							:directory (pathname-directory *Partial-Pathname*)) 'my-pathname)))
			      (t (simple-parse-error "Invalid or incomplete pathname"))))
		       ((null (cdr a-list))
			(simple-parse-error "Can't complete present pathname input~@[ with .~a extension~]" *Pathname-Canonical-Type*))
		       ((probe-file *Partial-Pathname*)
			(return-from accept
			  (values (make-pathname :name (concatenate 'string (pathname-name *Partial-Pathname*) can-strg)
						 :directory (pathname-directory *Partial-Pathname*)) 'my-pathname)))
		       (t (simple-parse-error "Invalid pathname~@[ with .~a extension~]" *Pathname-Canonical-Type*))))
		((eq char #\control-meta-y)                    ;Let's yank in the default path!
		 (setq buf-start (1- buf-start))
		 (when (string-equal *Partial-Pathname* "")
		   (setq *Partial-Pathname* (namestring default))
		   (unless (clim::rescanning-p stream)
		     (replace-input stream (subseq *Partial-Pathname* (length old-path)) :buffer-start buf-start))))
		((or (char-equal char #\space) (char-equal char #\tab))           ;WHY doesn't tab work, so I don't need spacebar!
		 (cond ((or (setq matches (string-equal "" *Partial-Pathname*))
			    (and (not matches) (char-equal #\/ (aref *Partial-Pathname* 0))))
			(setq old-path *Partial-Pathname*)
			(cond (matches (setq *Partial-Pathname* (namestring default)))
			      (t (setq a-list (Complete-Partial-Pathname-Input *Partial-Pathname*) *Partial-Pathname* (car a-list))))
			(setq buf-start (1- buf-start))
			(when (and *Pathname-Canonical-Type* (char-equal #\. (aref *Partial-Pathname* (1- (length *Partial-Pathname*)))))
			  (loop for i in (cdr a-list) do
				(when (search can-strg i :test #'char-equal :from-end t)
				  (setq success-p t)
				  (return (setq *Partial-Pathname* i)))))
			(unless (clim::rescanning-p stream)
			  (replace-input stream (subseq *Partial-Pathname* (length old-path)) :buffer-start buf-start)))
		       (t (simple-parse-error "~a is not a valid pathname (no directory component)!" *Partial-Pathname*))))
		(t (setq *Partial-Pathname* (concatenate 'string *Partial-Pathname* (string char))))))))


