;;;; Code for viewing directory contents in any list box.

;; This code is somewhat redundant with the "directory" code
;; in windows\dlgfile.lsp, but not cleanly enough for me to
;; have merged them in short order.  But we may want to
;; at some point. --- cheetham 3/94

;;;; Changelog
;;;; <1> cheetham 4/94
;;;;     use with=delayed-redraw when listing a new directory in the
;;;;     find-in-file directory browser list, so that it won't redraw
;;;;     a second time to add the [.] (current dir) entry to the list

(in-package :pc)

;; ??? This still needs to be put into the kernel I guess.
(export '(win::ddl_readwrite win::ddl_readonly win::ddl_hidden
          win::ddl_system win::ddl_directory win::ddl_archive
          win::ddl_postmsgs win::ddl_drives win::ddl_exclusive)
   :win)

(defun makelong (lo hi)
   (logior (ash hi 16) lo))

(defconstant win::ddl_readwrite #x0000)
(defconstant win::ddl_readonly #x0001)
(defconstant win::ddl_hidden #x0002)
(defconstant win::ddl_system #x0004)
(defconstant win::ddl_directory #x0010)
(defconstant win::ddl_archive #x0020)
(defconstant win::ddl_postmsgs #x2000)
(defconstant win::ddl_drives #x4000)
(defconstant win::ddl_exclusive #x8000)

;; (export 'cg::subdirectories-to-list-box :cg)
(defun cg::subdirectories-to-list-box
    (list-box directory-string
       &key static-text (drive-p t)(current-directory-p t))
   (cg::files-to-list-box
    list-box directory-string
    :static-text static-text :read-write-p nil
    :directory-p t :drive-p drive-p
    :current-directory-p current-directory-p))

;; (export 'cg::files-to-list-box :cg)
(defun cg::files-to-list-box
    (list-box directory-string
       &key static-text
       (read-write-p t)(read-only-p nil)
       (hidden-p nil)(system-p nil)(archive-p nil)
       (directory-p nil)(drive-p nil)
       (current-directory-p nil))
   
   ;; Windows requires the pathname to end with a wildcard spec,
   ;; so assume that the passed-in file does not end in a
   ;; no-wildcard filename.
   (unless (or (find #\? directory-string)
               (find #\* directory-string))
      (setq directory-string (concatenate 'string directory-string
                                (if (eql (elt directory-string
                                            (1- (length directory-string)))
                                       #\\)
                                   "*.*"
                                   "\\*.*"))))
   
   ;; Tell Windows to put the directory contents into the list-box,
   ;; and the directory namestring into the static-text.
   (with-delayed-redraw (list-box) ;; <1>
      (let ((static-handle (and static-text
                                (window-handle
                                   (dialog-item-window static-text))))
            (string #.(make-string 128)))
         (DlgDirList
            (window-handle (dialog-item-dialog list-box))
            (substring directory-string 0 (strlen directory-string))
            (GetDlgCtrlId (window-handle (dialog-item-window list-box)))
            (if static-handle
               (GetDlgCtrlId static-handle)
               0)
            (#+aclnt identity
             #-aclnt loword ;; hack to get an unsigned word (?)
             (makelong 
                (logior
                   (if read-only-p DDL_READONLY 0)
                   (if hidden-p DDL_HIDDEN 0)
                   (if system-p DDL_SYSTEM 0)
                   (if archive-p DDL_ARCHIVE 0)
                   (if directory-p DDL_DIRECTORY 0)
                   (if drive-p DDL_DRIVES 0)
                   ;; this one caused a problem due to being the
                   ;; high bit of the word.
                   ;; acl used pro::be here (unavailable),
                   ;; so we did the loword/makelong hack above instead
                   (if read-write-p 0 DDL_EXCLUSIVE)
                   )
                0)))
         
         ;; Ask Windows what the directory string is that it put into
         ;; the static-text, and update the lisp value accordingly.
         (when static-handle
            (sendmessage-with-pointer static-handle WM_GETTEXT
             128 string :static :static)
            (setf (slot-value static-text 'cg::value)
                  (substring string 0 (strlen string))))
         
         (let ((*delay-side-effects* t))
            ;; Always print the directory contents in the standard way.
            (setf (cg::dialog-item-key list-box) 'identity))
         
         ;; Make the list-box remember which static-text is associated
         ;; with it for this directory functionality.
         (setf (getf (dialog-item-plist list-box) :directory-static-text)
               static-text)
         
         ;; Remember what directory the list-box is relative to.
         (setf (getf (dialog-item-plist list-box) :directory-string)
               directory-string)
         
         ;; Do this particularly so that a set-value-fn will fire.
         ;; This needs to go after the DlgDirList but before
         ;; the list-box-update-range.
         (set-dialog-item-value list-box nil)
         
         ;; Update the lisp range and value of the list-box
         ;; to match what Windows put into it.
         (list-box-update-range list-box)
         (when current-directory-p
            (list-widget-add-item list-box "[.]" 0))
         )))

(defun list-box-update-range (list-box)
   ;; Useful for making the lisp range and value match the Windows values
   ;; after Windows has updated the range instead of us
   ;; doing it programmatically.
   ;; Specifically after using the DlgDirList message to place the
   ;; contents of a diretory into a list-box.
   (let* ((window (dialog-item-window list-box))
          (handle (window-handle window))
          (range nil)
          (string (make-string 128))
          (index (SendMessage handle LB_GETCURSEL 0 0 :static)))
      (dotimes (j (SendMessage handle LB_GETCOUNT 0 0 :static))
         (sendmessage-with-pointer handle LB_GETTEXT j string
          :static t)
         (setq range (nconc range
                        (list (substring string 0 (strlen string))))))
      (setf (slot-value list-box 'cg::range) range)
      (setf (slot-value list-box 'cg::value)
            (and (not (minusp index))
                 (nth index range)))
      range))

;; The user may want to write a similar double-click-fn that
;; puts files in a second list-box, for example.
;; We should provide the source to this one as an example.
;; (export 'cg::subdirectory-list-box-double-click-fn :cg)
(defun cg::subdirectory-list-box-double-click-fn (dialog list-box)
   (let ((pathname-string (cg::pathname-string-from-directory-list-box
                           list-box))
         (directory-string (getf (dialog-item-plist list-box)
                              :directory-string))
         (static-text (getf (dialog-item-plist list-box)
                         :directory-static-text)))
      (unless directory-string
         (error "You tried to use subdirectory-list-box-double-click-fn ~
                without first calling subdirectories-to-list-box ~
                or files-to-list-box."))
      (when (and pathname-string
                 ;; Make sure it's a directory, not a filename
                 (eql #\\ (elt pathname-string
                             (1- (length pathname-string))))
                 ;; Do nothing if clicking on same directory
                 ;; such as when [.] is clicked
                 ;; or [..] when we're at a top-level directory
                 (not (string= pathname-string
                         (subseq directory-string 0
                            (1+ (position #\\ directory-string
                                   :from-end t))))))
         (subdirectories-to-list-box list-box pathname-string 
          :static-text static-text)))
   t)

;; (export 'cg::pathname-string-from-directory-list-box :cg)
(defun cg::pathname-string-from-directory-list-box
    (list-box &optional do-not-set-new-drive)
   (let* ((file (dialog-item-value list-box))
          (directory (getf (dialog-item-plist list-box)
                        :directory-string))
          (old-current-directory (dos-current-directory *local-host*))
          index)
      (unless directory
         (error "You called pathname-string-from-directory-list-box~
without first calling subdirectories-to-list-box ~
or files-to-list-box."))
      (setq directory (format nil "~(~a~)"
                         (subseq directory 0
                            (position #\\ directory :from-end t))))
      (cond ((null file) nil) ; no subdirectory or file selected
            ((string= "[.]" file) ; current directory
             (concatenate 'string directory "\\"))
            ((string= "[..]" file) ; the parent dir
             (if (setq index (position #\\ directory :from-end t))
                (subseq directory 0 (1+ index))
                (concatenate 'string directory "\\")))
            ((and (eql #\[ (elt file 0))
                  (eql #\- (elt file 1))) ; a drive letter
             ;; Must temporarily change the current DOS drive so
             ;; we can return the current directory on that drive.
             (cond (do-not-set-new-drive
                      (format nil "~a:\\" (elt file 2)))
                   (t
                      (set-dos-current-directory
                       (format nil "~a:" (elt file 2)))
                      (prog1
                         (namestring (dos-current-directory *local-host*))
                         (set-dos-current-directory old-current-directory)))))
            ((eql #\[ (elt file 0)) ; a subdirectory
             (format nil 
                ;; Insert a backslash after the parent directory
                ;; unless it's a toplevel directory, in which case
                ;; the slash is already there.
                ;; Also append a backslash after the subdiretory
                ;; to indicate that the returnded value is a
                ;; directory rather than a file.
                "~a\\~a\\"
                #+no
                (if (eql (elt directory (1- (length directory)))
                       #\\)
                   "~a~a\\"
                   "~a\\~a\\")
                directory 
                ;; remove the surrounding [...]
                (subseq file 1 (1- (length file)))))
            (t ; a (non-directory) file
               (format nil
                  "~a\\~a"
                  #+no
                  (if (eql (elt directory (1- (length directory)))
                         #\\)
                     "~a~a"
                     "~a\\~a")
                  directory file)))))
