;-*- mode:lisp; package:boxer ;base: 8; fonts:cptfont -*-

;;; Macro Definitions and Variable Declarations for the BOXER File system
;;;
;;; (C) Copyright 1984 Massachusetts Institute of Technology
;;;
;;; Permission to use, copy, modify, distribute, and sell this software
;;; and its documentation for any purpose is hereby granted without fee,
;;; provided that the above copyright notice appear in all copies and that
;;; both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of M.I.T. not be used in
;;; advertising or publicity pertaining to distribution of the software
;;; without specific, written prior permission.  M.I.T. makes no
;;; representations about the suitability of this software for any
;;; purpose.  It is provided "as is" without express or implied warranty.
;;;
;;;
;;;                          +-Data--+
;;; This file is part of the | BOXER | system.
;;;                          +-------+
;;;

;*********************************************************************************************
;*                              TOP  LEVEL  DEFINITIONS                                      *
;*********************************************************************************************

;;;Pathname Construction and manipulation...

(FS:DEFINE-CANONICAL-TYPE :BOX "Box"	;default type for SAVE/READ
	      (:TOPS-20 "Box")
	      (:VMS "Box")
	      (:ITS "Box"))

(defprop :box 16. :binary-file-byte-size)


;;initializations...

(DEFVAR *BOXER-PATHNAME-DEFAULT* (TELL (FS:DEFAULT-PATHNAME) :NEW-CANONICAL-TYPE ':BOX)
  "Default pathname for saving boxer files")

(DEFVAR *INIT-FILE-SPECIFIER* (FS:MERGE-PATHNAMES "boxer.init" *BOXER-PATHNAME-DEFAULT*)
  "The default name of the initial Boxer world load. ")

(DEFVAR *STICKY-FILE-DEFAULTING?* T
  "A switch to make the default filename the last pathname that was used. ")

(SETQ *FILE-PORT-HASH-TABLE* (MAKE-HASH-TABLE))

(DEFVAR *ROW-CHAS-POINTER-ADJUST* NIL
  "A flag which the newly constructed row checks to see if it should forward pointers
   to its chas. A Kludge until I write the fasdumper. ")

(DEFVAR *FASDUMP?* T)				;use the fasdumper or not ?

(DEFVAR *BIT-ARRAYS-ARE-ROW-MAJOR-ORDERED?* #+LMITI NIL #-LMITI T)

;;; BINARY file format...
;;; Commands are in the form of 16. bit numbers (apparently the max size for file streams)
;;; The top four bits in a command make up a limited number of immediate op-codes in which
;;; the next 12. bits make up an immediate argument to the first op-code
;;; the four bit box command code escapes to more specific box commands and 
;;; another four bit sequence escapes to general commands in the next word

;*********************************************************************************************
;*                                    DEFINITIONS                                            *
;*********************************************************************************************


;;; Opcode definitions
(DEFCONST %%BIN-OP-HIGH 1404)
(DEFCONST %%BIN-OP-LOW 0014)

(DEFCONST %%BIN-OP-IM-ARG-SIZE (^ 2 12.))
(DEFCONST %%BIN-OP-ARG-SIZE (^ 2 16.))

;;; Currently supported version number
(DEFCONST *VERSION-NUMBER* 3)

;;; Dumping variables

(DEFVAR *BIN-DUMP-TABLE*)
(DEFVAR *BIN-DUMP-INDEX*)
(DEFVAR *BIN-DUMP-PACKAGE*)
(DEFVAR *OUTERMOST-DUMPING-BOX* NIL
  "The top level box which is being dumped. ")
(DEFVAR *RESTORE-TURTLE-STATE* NIL
  "Determines if the state of turtle boxes should be saved. ")

(DEFRESOURCE DUMP-HASH-TABLE ()
  :CONSTRUCTOR (MAKE-INSTANCE 'SI:EQ-HASH-TABLE)
  :INITIAL-COPIES 0)

(DEFMACRO MAKE-BIN-OP-DISPATCH-TABLE ()
  `(MAKE-ARRAY 100))

(DEFMACRO BIN-OP-DISPATCH (TABLE NUMBER)
  `(AREF ,TABLE ,NUMBER))

(DEFMACRO STORE-BIN-OP-DISPATCH (VALUE TABLE NUMBER)
  `(ASET ,VALUE ,TABLE ,NUMBER))

(DEFPROP BIN-OP-DISPATCH
	 ((BIN-OP-DISPATCH TABLE NUMBER) . (STORE-BIN-OP-DISPATCH SI:VAL TABLE NUMBER))
	 SETF)

;; so we can get the commands from their number format and vice versa
(DEFVAR *BIN-OP-COMMAND-NAME-TABLE* (MAKE-BIN-OP-DISPATCH-TABLE))

(DEFMACRO DEFINE-BIN-OP (NAME VALUE INDEX)
  `(PROGN 'COMPILE
     (DEFCONST ,NAME ,VALUE)
     (SETF (BIN-OP-DISPATCH *BIN-OP-COMMAND-NAME-TABLE* ,INDEX) ',NAME)))

(DEFUN DECODE-BIN-OP (BIN-OP-NUMBER)
  (AREF *BIN-OP-COMMAND-NAME-TABLE* BIN-OP-NUMBER))



;;; immediate commands.  The meaning of the 20 bit arg is specified in the comment
(DEFMACRO DEFINE-IMMEDIATE-BIN-OP (NAME VALUE)
  `(DEFINE-BIN-OP ,NAME ,VALUE ,VALUE))

(DEFINE-IMMEDIATE-BIN-OP BIN-OP-NUMBER-IMMEDIATE 0)	;<number>
(DEFINE-IMMEDIATE-BIN-OP BIN-OP-TABLE-FETCH-IMMEDIATE 1)	;<table address>
(DEFINE-IMMEDIATE-BIN-OP BIN-OP-CHA-IMMEDIATE 2)	;<character number>
(DEFINE-IMMEDIATE-BIN-OP BIN-OP-BOX-IMMEDIATE 3)	;<box type>
(DEFINE-IMMEDIATE-BIN-OP BIN-OP-STRING-IMMEDIATE 4)     ;<string length>
(DEFINE-IMMEDIATE-BIN-OP BIN-OP-LIST-IMMEDIATE 5)	;<list length>
(DEFINE-IMMEDIATE-BIN-OP BIN-OP-ARRAY 6)	        ;number of options
(DEFINE-IMMEDIATE-BIN-OP BIN-OP-ROW-IMMEDIATE 7)	;number of chas
(DEFINE-IMMEDIATE-BIN-OP BIN-OP-NAME-AND-INPUT-ROW-IMMEDIATE 10)	;number of chas
(DEFINE-IMMEDIATE-BIN-OP BIN-OP-NAME-ROW-IMMEDIATE 11)
(DEFINE-IMMEDIATE-BIN-OP BIN-OP-COMMAND-IMMEDIATE 17)	;<command>

;;; specific box commands
(DEFMACRO DEFINE-BOX-BIN-OP (NAME VALUE)
  `(DEFINE-BIN-OP ,NAME ,(DPB BIN-OP-BOX-IMMEDIATE %%BIN-OP-HIGH VALUE) ,VALUE))

(DEFINE-BOX-BIN-OP BIN-OP-DOIT-BOX 20)
(DEFINE-BOX-BIN-OP BIN-OP-DATA-BOX 21)
(DEFINE-BOX-BIN-OP BIN-OP-PORT-BOX 22)
(DEFINE-BOX-BIN-OP BIN-OP-GRAPHICS-BOX 23)
(DEFINE-BOX-BIN-OP BIN-OP-TURTLE-BOX 24)	;without turtle state
(DEFINE-BOX-BIN-OP BIN-OP-TURTLE-BOX* 25)	;with turtle state, including bit array
(DEFINE-BOX-BIN-OP BIN-OP-LL-BOX 26)
(define-box-bin-op bin-op-graphics-data-box 31)
(define-box-bin-op bin-op-sprite-box 32)
;; for compatibility with pre version 4.0 files
(DEFINE-BOX-BIN-OP BIN-OP-LL-BOX-PRESCENCE-MARKER 27)

;;; Other commands
(DEFMACRO DEFINE-COMMAND-BIN-OP (NAME VALUE)
  `(DEFINE-BIN-OP ,NAME ,(DPB BIN-OP-COMMAND-IMMEDIATE %%BIN-OP-HIGH VALUE) ,VALUE))

(DEFINE-COMMAND-BIN-OP BIN-OP-TABLE-FETCH 35)
(DEFINE-COMMAND-BIN-OP BIN-OP-END-OF-BOX 36)
(DEFINE-COMMAND-BIN-OP BIN-OP-STRING 37)
(DEFINE-COMMAND-BIN-OP BIN-OP-SYMBOL 40)
(DEFINE-COMMAND-BIN-OP BIN-OP-PACKAGE-SYMBOL 41)

(DEFINE-COMMAND-BIN-OP BIN-OP-POSITIVE-FIXNUM 42)
(DEFINE-COMMAND-BIN-OP BIN-OP-NEGATIVE-FIXNUM 43)
(DEFINE-COMMAND-BIN-OP BIN-OP-POSITIVE-FLOAT 44)
(DEFINE-COMMAND-BIN-OP BIN-OP-NEGATIVE-FLOAT 45)

(DEFINE-COMMAND-BIN-OP BIN-OP-ROW 46)
(DEFINE-COMMAND-BIN-OP BIN-OP-LIST 47)

(DEFINE-COMMAND-BIN-OP BIN-OP-INITIALIZE-AND-RETURN-ARRAY 50)
(DEFINE-COMMAND-BIN-OP BIN-OP-INITIALIZE-AND-RETURN-NUMERIC-ARRAY 51)

(DEFINE-COMMAND-BIN-OP BIN-OP-FORMAT-VERSION 52)
(DEFINE-COMMAND-BIN-OP BIN-OP-EOF 53)

(DEFINE-COMMAND-BIN-OP BIN-OP-FILE-PROPERTY-LIST 54)

(DEFINE-COMMAND-BIN-OP BIN-OP-TABLE-STORE 55)

(DEFINE-COMMAND-BIN-OP BIN-OP-SIMPLE-CONS 56)
(DEFINE-COMMAND-BIN-OP BIN-OP-NAME-AND-INPUT-ROW 57)
(DEFINE-COMMAND-BIN-OP BIN-OP-NAME-ROW 60)

;;graphics stuff
(DEFINE-COMMAND-BIN-OP BIN-OP-GRAPHICS-SHEET 61)
(DEFINE-COMMAND-BIN-OP BIN-OP-GRAPHICS-OBJECT 62)
(define-command-bin-op bin-op-turtle 63)


(DEFMACRO WRITING-BIN-FILE ((BOX STREAM FILE) &BODY BODY)
  `(WITH-OPEN-FILE (,STREAM ,FILE ':DIRECTION ':OUTPUT ':CHARACTERS NIL)
     (USING-RESOURCE (*BIN-DUMP-TABLE* DUMP-HASH-TABLE)
       (START-BIN-FILE ,STREAM)
       (LET ((*BIN-DUMP-INDEX* 0)
	     (*BIN-DUMP-PACKAGE* PACKAGE)
	     (*OUTERMOST-DUMPING-BOX* ,BOX))
	 ,@BODY))
     (END-BIN-FILE ,STREAM)))

;*********************************************************************************************
;*                                LOADING   DEFINITIONS                                      *
;*********************************************************************************************

;;; Loading variables
(DEFRESOURCE BIN-LOAD-TABLE ()
  :CONSTRUCTOR (MAKE-ARRAY 1000))

(DEFVAR *NO-VALUE-MARKER* (NCONS 'NO-VALUE))
(DEFVAR *BIN-NEXT-COMMAND-FUNCTION*)

(DEFVAR *BIN-LOAD-TABLE*)
(DEFVAR *BIN-LOAD-INDEX*)
(DEFVAR *LOAD-PACKAGE*)
(DEFVAR *FILE-BIN-VERSION*)
(DEFVAR *ROW-MAJOR-ORDER?* T
  "Specifies how bit-arrays were dumped out. The default is T due to existence of many
   old files which were dumped out in zippy lisp")

(DEFVAR *BIN-OP-LOAD-COMMAND-TABLE* (MAKE-BIN-OP-DISPATCH-TABLE))

(DEFVAR *SUPPORTED-OBSOLETE-VERSIONS* '(1. 2.))

(DEFMACRO BIN-NEXT-COMMAND (&REST ARGS)
  `(FUNCALL *BIN-NEXT-COMMAND-FUNCTION* . ,ARGS))

(DEFMACRO LOADING-BIN-FILE ((STREAM NEXT-COMMAND-FUNCTION SKIP-READING-PROPERTY-LIST)
			    &BODY BODY)
  `(LET* ((*BIN-NEXT-COMMAND-FUNCTION* ,NEXT-COMMAND-FUNCTION)
	  (*BIN-LOAD-INDEX* 0)
	  (*FILE-BIN-VERSION* 0)
	  (*ROW-MAJOR-ORDER?* *ROW-MAJOR-ORDER?*))
       (USING-RESOURCE (*BIN-LOAD-TABLE* BIN-LOAD-TABLE)
	 (BIN-LOAD-START ,STREAM ,SKIP-READING-PROPERTY-LIST)
	   (PROGN . ,BODY))))

;;;Load command definitions...
;;;There are three types of commands

(DEFMACRO DEFINE-BIN-COMMAND-OP (OP-NAME DEFINING-FUNCTION TABLE FUNCTION-PREFIX ARGLIST
				 &BODY DEFINITION)
  (LET ((FUNCTION-NAME (LET (#-3600 (DEFAULT-CONS-AREA WORKING-STORAGE-AREA))
			 (INTERN (STRING-APPEND FUNCTION-PREFIX OP-NAME)))))
    `(PROGN 'COMPILE
       (SETF (BIN-OP-DISPATCH ,TABLE (LDB %%BIN-OP-LOW ,OP-NAME)) ',FUNCTION-NAME)
       (RECORD-SOURCE-FILE-NAME ',OP-NAME ',DEFINING-FUNCTION)
       (LOCAL-DECLARE ((SYS:FUNCTION-PARENT ,OP-NAME ,DEFINING-FUNCTION))
	 (DEFUN ,FUNCTION-NAME ,ARGLIST . ,DEFINITION)))))

;;; A command that may return a value, but does not store it in the table
(DEFMACRO DEFINE-LOAD-COMMAND (OP-NAME ARGLIST &BODY BODY)
  `(DEFINE-BIN-COMMAND-OP ,OP-NAME DEFINE-LOAD-COMMAND
			  *BIN-OP-LOAD-COMMAND-TABLE* "LOAD-" ,ARGLIST
     . ,BODY))

;;; A command that does not return a value at all
(DEFMACRO DEFINE-LOAD-COMMAND-FOR-EFFECT (OP-NAME ARGLIST &BODY BODY)
  `(DEFINE-BIN-COMMAND-OP ,OP-NAME DEFINE-LOAD-COMMAND-FOR-EFFECT
			  *BIN-OP-LOAD-COMMAND-TABLE* "LOAD-" ,ARGLIST
     ,@BODY
     *NO-VALUE-MARKER*))

;;; A command that returns a value stored in the next slot in the table
(DEFMACRO DEFINE-LOAD-COMMAND-FOR-VALUE (OP-NAME ARGLIST &BODY BODY)
  `(DEFINE-BIN-COMMAND-OP ,OP-NAME DEFINE-LOAD-COMMAND-FOR-VALUE
			  *BIN-OP-LOAD-COMMAND-TABLE* "LOAD-" ,ARGLIST
     (ENTER-BIN-LOAD-TABLE (PROGN . ,BODY))))

(DEFMACRO ENTER-BIN-LOAD-TABLE (VALUE)
  `(LET ((.INDEX. *BIN-LOAD-INDEX*))
     (INCF *BIN-LOAD-INDEX*)
     (ENTER-BIN-LOAD-TABLE-INTERNAL ,VALUE .INDEX.)))

;;; Loading Loading stuff common to all boxes
(DEFMACRO LOAD-VANILLA-BOX ((STREAM) &BODY BODY)
  `(LET* ((NAME (BIN-NEXT-VALUE ,STREAM))
	  (DISPLAY-LIST (BIN-NEXT-VALUE ,STREAM))
	  ;; these next three lines are for compatibility with the turtle box version of BOXER
	  (INITIAL-ENVIRONMENT (BIN-NEXT-VALUE ,STREAM))
	  (TURTLE-BINDING-PAIR (ASSQ '%TURTLE INITIAL-ENVIRONMENT))
	  (ENVIRONMENT (IF (NOT-NULL TURTLE-BINDING-PAIR)
			   (PUSH (CONS *EXPORTING-BOX-MARKER* (CDR TURTLE-BINDING-PAIR))
				 INITIAL-ENVIRONMENT)
			   INITIAL-ENVIRONMENT))
	  ;; leave this here for non local-library files (< version 4.0)
	  ;; I'm changing this cause UNIX file streams are losing on :TYIPEEK
	  (local-library (progn (if (not (= (send ,stream :tyi)
					    BIN-OP-LL-BOX-PRESCENCE-MARKER))
				    (cl:error "There should be a local library marker here"))
				(bin-next-value ,stream)))
;	  (LOCAL-LIBRARY (WHEN (= (SEND STREAM :TYIPEEK) BIN-OP-LL-BOX-PRESCENCE-MARKER)
;			   (SEND STREAM :TYI)
;			   ;; a local library HAS been dumped so return it or else NIL
;			   ;; REMOVE this SOON !!!!
;			   (BIN-NEXT-VALUE ,STREAM)))
	  )
     (PROGN . ,BODY)))


;;; Rel 4.5 lossage in not having a KEYWORD package.  We will dump names with colon prefixes
;;; into the KEYWORD package and on loading (in rel 4.5) put them back into the USER package
;;; so that files will be rel 5.0 compatible with this crock for rel 4.5

#+rel4
(package-declare keyword global 100)

#+rel4
(defvar pkg-keyword-package (pkg-find-package 'keyword))
