;; -*- Mode:LISP; Package:(BOXER GLOBAL 1000); Base: 8.; Fonts: CPTFONT,CPTFONTI -*-

;; (C) Copyright 1983-1985 Massachusetts Institute of Technology
;;
;; Permission to use, copy, modify, distribute, and sell this software
;; and its documentation for any purpose is hereby granted without fee,
;; provided that the above copyright notice appear in all copies and that
;; both that copyright notice and this permission notice appear in
;; supporting documentation, and that the name of M.I.T. not be used in
;; advertising or publicity pertaining to distribution of the software
;; without specific, written prior permission.  M.I.T. makes no
;; representations about the suitability of this software for any
;; purpose.  It is provided "as is" without express or implied warranty.
;;

;;;this file contains random code having to do with screen structure
;;;such as allocation/deallocation code, mouse tracking stuff and box
;;;border functions

;;;All of the high level redisplay code is in the file REDISP and
;;;the low level code for accessing and patching up screen structure is
;;;to be found in the file LODISP

;;;; LOW-LEVEL SCREEN-OBJ allocation/deallocation code.

;;; Use our own resource allocation/deallocation scheme here because the
;;; Lispm's DEALLOCATE-RESOURCE is so slow that it significantly slows
;;; down the whole redisplay code.

(DEFUN SETUP-REDISPLAY ()
  (SETQ FREE-SCREEN-ROWS NIL
	FREE-SCREEN-BOXS NIL
	FREE-GRAPHICS-SCREEN-BOXS NIL)
  (DOTIMES (I INITIAL-NO-OF-FREE-SCREEN-ROWS)
    (PUSH (MAKE-INSTANCE 'SCREEN-ROW) FREE-SCREEN-ROWS))
  (DOTIMES (I INITIAL-NO-OF-FREE-SCREEN-BOXS)
    (PUSH (MAKE-INSTANCE 'SCREEN-BOX) FREE-SCREEN-BOXS))
  (DOTIMES (I INITIAL-NO-OF-FREE-GRAPHICS-SCREEN-BOXS)
    (PUSH (MAKE-INSTANCE 'GRAPHICS-SCREEN-BOX) FREE-GRAPHICS-SCREEN-BOXS)))

(DEFMETHOD (SCREEN-BOX :RE-INIT) (NEW-ACTUAL-OBJ)
  (SETQ ACTUAL-OBJ NEW-ACTUAL-OBJ
	WID 0
	HEI 0
	X-GOT-CLIPPED? NIL
	Y-GOT-CLIPPED? NIL
	TICK -1
	NEEDS-REDISPLAY-PASS-2? NIL
	FORCE-REDISPLAY-INFS? NIL
	SCREEN-ROWS NIL
	SCREEN-ROW NIL
	BPS NIL))

(DEFMETHOD (SCREEN-ROW :RE-INIT) (NEW-ACTUAL-OBJ)
  (SETQ ACTUAL-OBJ NEW-ACTUAL-OBJ
	WID 0
	HEI 0
	X-GOT-CLIPPED? NIL
	Y-GOT-CLIPPED? NIL
	TICK -1
	NEEDS-REDISPLAY-PASS-2? NIL
	FORCE-REDISPLAY-INFS? #+3600 T #-3600 NIL))

 (DEFMETHOD (SCREEN-ROW :AFTER :RE-INIT) (IGNORE)
  (SETQ SCREEN-CHAS NIL
	SCREEN-BOX NIL))

(DEFMETHOD (SCREEN-BOX :AFTER :RE-INIT) (NEW-ACTUAL-BOX)
  (TELL SELF :SET-BOX-TYPE (TELL NEW-ACTUAL-BOX :TYPE)))




;;;; HIGH-LEVEL SCREEN-OBJ allocation/deallocation code.

;;; This code is responsible for allocating screen-objs to represent actual
;;; objs. This code isn't terribly complicated, but it is basic to the rest
;;; of the display code, so it is probably a good idea to understand how it
;;; works. So, listen carefully... This code is based on the following basic
;;; assumptions:
;;;  
;;;   No actual object can be displayed more than once at any
;;;    "level". For example, the same box cannot be displayed
;;;    right next to itself. On the other hand a port to a box
;;;    can be displayed right next to the box since the lispm
;;;    port object is neq to the lispm box object.
;;;
;;;   That whenever moving of actual objs is implemented (this
;;;;   includes boxing and unboxing operations) redisplay clues
;;;    which tell what happened will be added and this code will
;;;    be updated to take these clues into account.
;;;
;;; Given these assumptions, and given that:
;;;    ACTUAL-OBJ
;;;        is an actual obj to be displayed (a screen-obj is
;;;        needed in order to display it)
;;;    SUPERIOR-SCREEN-BOX
;;;        is the screen-box in which the actual obj is going
;;;        to be displayed
;;;    SCREEN-OBJ
;;;        is the screen-obj which represents the actual obj
;;;        when it is displayed in that particular superior
;;;        screen-box
;;; Then:
;;;
;;;     (ACTUAL-OBJ , SUPERIOR-SCREEN-OBJ)  SCREEN-OBJ
;;;
;;; The :ALLOCATE-SCREEN-OBJ-FOR-USE-IN method uses this mapping to allocate
;;; screen-objs to represent actual objs. Calling this method is the only
;;; correct way to get screen-objs which represent actual objs.

(DEFMETHOD (ACTUAL-OBJ-MIXIN :ALLOCATE-SCREEN-OBJ-FOR-USE-IN) (USE-IN-SCREEN-BOX)
  (LET ((EXISTING-SCREEN-OBJ (ASSQ USE-IN-SCREEN-BOX SCREEN-OBJS)))
    (IF (NOT-NULL EXISTING-SCREEN-OBJ)
	(CDR EXISTING-SCREEN-OBJ)
	(LET ((NEW-SCREEN-OBJ (ALLOCATE-SCREEN-OBJ-INTERNAL SELF)))
	  (PUSH (CONS USE-IN-SCREEN-BOX NEW-SCREEN-OBJ) SCREEN-OBJS)
	  (WHEN (SCREEN-BOX? NEW-SCREEN-OBJ)
	    (TELL NEW-SCREEN-OBJ :SET-SUPERIOR-SCREEN-BOX USE-IN-SCREEN-BOX)
	    (TELL NEW-SCREEN-OBJ :SET-NAME
		  (TELL-CHECK-NIL (TELL SELF :NAME-ROW) :TEXT-STRING)))
	  NEW-SCREEN-OBJ))))

(DEFUN ALLOCATE-SCREEN-SHEET-FOR-USE-IN (GRAPHICS-SHEET USE-IN-SCREEN-BOX)
  (LET* ((SCREEN-OBJS (GRAPHICS-SHEET-SCREEN-OBJS GRAPHICS-SHEET))
	 (EXISTING-SCREEN-OBJ (ASSQ USE-IN-SCREEN-BOX SCREEN-OBJS)))
    (IF (NOT-NULL EXISTING-SCREEN-OBJ)
	(CDR EXISTING-SCREEN-OBJ)
	(LET ((NEW-SCREEN-OBJ (ALLOCATE-SCREEN-OBJ-INTERNAL GRAPHICS-SHEET)))
	  (SETF (GRAPHICS-SHEET-SCREEN-OBJS GRAPHICS-SHEET)
		(PUSH (CONS USE-IN-SCREEN-BOX NEW-SCREEN-OBJ) SCREEN-OBJS))
	  NEW-SCREEN-OBJ))))

(DEFUN SCREEN-STRUCTURE-ACTUAL-SUPERIOR-BOX (SCREEN-BOX)
  (LET ((SUPERIOR-SCREEN-BOX (TELL SCREEN-BOX :SUPERIOR-SCREEN-BOX)))
    (WHEN (SCREEN-BOX? SUPERIOR-SCREEN-BOX)
      (TELL SUPERIOR-SCREEN-BOX :ACTUAL-OBJ))))

(defmethod (actual-obj-mixin :allocate-outermost-screen-box-for-use-in)
	   (window &OPTIONAL (SCREEN-BOX (BP-SCREEN-BOX *POINT*)))
  (let ((actual-superior-box (SCREEN-STRUCTURE-ACTUAL-SUPERIOR-BOX SCREEN-BOX)))
    (tell self					
	  :allocate-screen-obj-for-use-in
	  (if actual-superior-box
	      (tell actual-superior-box
		    :allocate-outermost-screen-box-for-use-in window
		    (TELL SCREEN-BOX :SUPERIOR-SCREEN-BOX))
	      window))))

(DEFMETHOD (ACTUAL-OBJ-MIXIN :SCREEN-OBJS) ()
  (MAPCAR #'CDR SCREEN-OBJS))

;;; Whenever any section of code is done with a screen-obj which they got by
;;; calling :allocate-screen-obj-for-use-in they should deallocate that screen-
;;; obj by sending it a deallocate-self message. If there are no more users
;;; of that screen-obj, it will be returned to the pool of free screen-objs
;;; of that type.

(DEFWHOPPER (SCREEN-OBJ :DEALLOCATE-SELF) ()
  (WHEN (NULL (TELL SELF :SUPERIOR))
    (CONTINUE-WHOPPER)))

(DEFMETHOD (SCREEN-ROW :DEALLOCATE-SELF) ()
  (TELL SELF :DEALLOCATE-INFERIORS)
  (TELL ACTUAL-OBJ :DELETE-SCREEN-OBJ SELF)
  (DEALLOCATE-SCREEN-OBJ-INTERNAL SELF))

(DEFMETHOD (SCREEN-BOX :DEALLOCATE-SELF) ()
  (TELL SELF :DEALLOCATE-INFERIORS))

(DEFMETHOD (GRAPHICS-SCREEN-BOX :DEALLOCATE-SELF) ()
  ;; shadow out the message here since we are not running resources on GRAPHICS-SCREEN-SHEET's
  NIL)
						
(DEFMETHOD (SCREEN-BOX :DEALLOCATE-INFERIORS) ()
  (LET ((INFERIORS (TELL SELF :INFERIORS)))
    (WHEN (#+SYMBOLICS LISTP #-SYMBOLICS CONSP INFERIORS)
      (TELL SELF :KILL-SCREEN-OBJ (CAR INFERIORS))
      (DOLIST (INFERIOR INFERIORS)
	(TELL INFERIOR :DEALLOCATE-SELF)))))

(DEFMETHOD (SCREEN-ROW :DEALLOCATE-INFERIORS) ()
  (LET ((INFERIOR-BOXES (EXTRACT-SCREEN-BOXES (TELL SELF :INFERIORS))))
    (WHEN (NOT-NULL INFERIOR-BOXES)
      (TELL SELF :KILL-SCREEN-OBJ (CAR INFERIOR-BOXES))
      (DOLIST (INFERIOR INFERIOR-BOXES)
	(TELL INFERIOR :DEALLOCATE-SELF)))))

(DEFMETHOD (ACTUAL-OBJ-MIXIN :DELETE-SCREEN-OBJ) (SCREEN-OBJ)
  (SETQ SCREEN-OBJS (DELETE (RASSQ SCREEN-OBJ SCREEN-OBJS) SCREEN-OBJS)))

(DEFUN QUEUE-SCREEN-OBJ-FOR-DEALLOCATION (SCREEN-OBJ)
  (LOCAL-DECLARE ((SPECIAL SCREEN-OBJS-DEALLOCATION-QUEUE))
    (SPLICE-ITEM-ONTO-LIST SCREEN-OBJS-DEALLOCATION-QUEUE SCREEN-OBJ)))

(DEFUN QUEUE-SCREEN-OBJS-FOR-DEALLOCATION (SCREEN-OBJS)
  (LOCAL-DECLARE ((SPECIAL SCREEN-OBJS-DEALLOCATION-QUEUE))
    (SPLICE-LIST-ONTO-LIST SCREEN-OBJS-DEALLOCATION-QUEUE SCREEN-OBJS)))



(DEFUN SCREEN-OBJ-OFFSETS (SCREEN-OBJ)
  (VALUES (SCREEN-OBJ-X-OFFSET SCREEN-OBJ)
	  (SCREEN-OBJ-Y-OFFSET SCREEN-OBJ)))

(DEFUN SET-SCREEN-OBJ-OFFSETS (SCREEN-OBJ NEW-X-OFFSET NEW-Y-OFFSET)
  (SETF (SCREEN-OBJ-X-OFFSET SCREEN-OBJ) NEW-X-OFFSET)
  (SETF (SCREEN-OBJ-Y-OFFSET SCREEN-OBJ) NEW-Y-OFFSET))

(DEFUN SCREEN-OBJ-SIZE (SCREEN-OBJ)
  (VALUES (SCREEN-OBJ-WID SCREEN-OBJ)
	  (SCREEN-OBJ-HEI SCREEN-OBJ)))

(DEFUN SCREEN-OBJS-SIZE (SCREEN-OBJS &AUX (WID 0) (HEI 0))
  (COND ((SCREEN-CHA? (CAR SCREEN-OBJS))
	 (DOLIST (SCREEN-CHA SCREEN-OBJS)
	   (SETQ WID (+ WID (SCREEN-OBJ-WID SCREEN-CHA))
		 HEI (MAX HEI (SCREEN-OBJ-HEI SCREEN-CHA)))))
	(T
	 (DOLIST (SCREEN-ROW SCREEN-OBJS)
	   (SETQ WID (MAX WID (SCREEN-OBJ-WID SCREEN-ROW))
		 HEI (+ HEI (SCREEN-OBJ-HEI SCREEN-ROW))))))
  (VALUES WID HEI))

(DEFUN SCREEN-BOXES-AND-WHITESPACE-SIZE (SCREEN-BOXES &AUX(WID 0) (HEI 0))
  (LET ((FIRST-BOX (CAR SCREEN-BOXES))
	(LAST-BOX (CAR (LAST SCREEN-BOXES))))
    (SETQ WID (- (+ (SCREEN-OBJ-X-OFFSET LAST-BOX) (SCREEN-OBJ-WID LAST-BOX))
		 (SCREEN-OBJ-X-OFFSET FIRST-BOX)))
    (DOLIST (SCREEN-BOX SCREEN-BOXES)
      (SETQ HEI (MAX (SCREEN-OBJ-HEI SCREEN-BOX) HEI)))
    (VALUES WID HEI)))

(DEFUN SCREEN-OBJS-WID (SCREEN-OBJS)
  (MULTIPLE-VALUE-BIND (WID NIL) (SCREEN-OBJS-SIZE SCREEN-OBJS) WID))

(DEFUN SCREEN-OBJS-HEI (SCREEN-OBJS)
  (MULTIPLE-VALUE-BIND (NIL HEI) (SCREEN-OBJS-SIZE SCREEN-OBJS) HEI))

(DEFUN SCREEN-OBJS-NEXT-SCREEN-OBJ-DELTA-OFFSETS-WHEN-ERASED (SCREEN-OBJS)
  (IF (SCREEN-CHA? (CAR SCREEN-OBJS))
      (VALUES (SCREEN-OBJS-WID SCREEN-OBJS) 0)
      (VALUES 0 (SCREEN-OBJS-HEI SCREEN-OBJS))))


(DEFUN MAP-OVER-SCREEN-OBJ (SCREEN-OBJ FN)
  (FUNCALL FN SCREEN-OBJ)
  (MAP-OVER-SCREEN-OBJS (TELL SCREEN-OBJ :INFERIORS) FN))

(DEFUN MAP-OVER-SCREEN-OBJS (LIST-OF-SCREEN-OBJS FN)
  (DOLIST (SCREEN-OBJ LIST-OF-SCREEN-OBJS)
    (MAP-OVER-SCREEN-OBJ SCREEN-OBJ FN)))


(DEFUN SCREEN-OBJ-ZERO-SIZE (SCREEN-OBJ)
  (SETF (SCREEN-OBJ-WID SCREEN-OBJ) 0)
  (SETF (SCREEN-OBJ-HEI SCREEN-OBJ) 0))

(DEFUN ERASE-SCREEN-CHA (SCREEN-CHA X-OFFSET Y-OFFSET)
  (IF (NOT-NULL SCREEN-CHA)
    (LET ((WID (CHA-WIDTH SCREEN-CHA))
	  (HEI (CHA-HEI (FONT-NO SCREEN-CHA))))
      (DRAW-RECTANGLE TV:ALU-ANDCA WID HEI X-OFFSET Y-OFFSET))
    (FERROR "null screen-cha for some reason")))

(DEFUN ERASE-SCREEN-BOX (SCREEN-BOX X-OFFSET Y-OFFSET)
  (MULTIPLE-VALUE-BIND (WID HEI)
      (SCREEN-OBJ-SIZE SCREEN-BOX)
    (DRAW-RECTANGLE TV:ALU-ANDCA WID HEI X-OFFSET Y-OFFSET))
  (SCREEN-OBJ-ZERO-SIZE SCREEN-BOX)
  (TELL SCREEN-BOX :SET-NEEDS-REDISPLAY-PASS-2? T)
  (TELL SCREEN-BOX :SET-FORCE-REDISPLAY-INFS? T))


(DEFUN SCREEN-OBJECT-WIDTH (SCREEN-OBJECT)
  (when screen-object
    (IF (SCREEN-CHA? SCREEN-OBJECT)
	(CHA-WIDTH SCREEN-OBJECT)
	(SCREEN-OBJ-WID SCREEN-OBJECT))))

(DEFUN SCREEN-OBJECT-NEW-WIDTH (SCREEN-OBJECT)
  (when screen-object
    (IF (SCREEN-CHA? SCREEN-OBJECT)
	(CHA-WIDTH SCREEN-OBJECT)
	(SCREEN-OBJ-NEW-WID SCREEN-OBJECT))))

(DEFUN-METHOD ERASE-CHAS-TO-EOL SCREEN-ROW (CHA-NO STARTING-X-OFFSET STARTING-Y-OFFSET)
  (LET ((CHAS (GATHER-SCREEN-CHAS CHA-NO (LENGTH SCREEN-CHAS)))
	(CURRENT-X-OFFSET STARTING-X-OFFSET)
	(CURRENT-Y-OFFSET STARTING-Y-OFFSET))
    (DO* ((CHAS-LEFT CHAS (CDR CHAS-LEFT))
	  (CHA-TO-ERASE (CAR CHAS-LEFT) (CAR CHAS-LEFT)))
	 ((NULL CHA-TO-ERASE))
      (WHEN (SCREEN-CHA? CHA-TO-ERASE)
	(ERASE-SCREEN-CHA CHA-TO-ERASE CURRENT-X-OFFSET CURRENT-Y-OFFSET))
      (SETQ CURRENT-X-OFFSET (+ CURRENT-X-OFFSET (SCREEN-OBJECT-WIDTH CHA-TO-ERASE))))))

(DEFUN ERASE-SCREEN-CHAS (CHAS STARTING-X-OFFSET STARTING-Y-OFFSET)
  (LET ((CURRENT-X-OFFSET STARTING-X-OFFSET)
	(CURRENT-Y-OFFSET STARTING-Y-OFFSET))
    (DO* ((CHAS-LEFT CHAS (CDR CHAS-LEFT))
	  (CHA-TO-ERASE (CAR CHAS-LEFT) (CAR CHAS-LEFT))
	  (x-incrementer (SCREEN-OBJECT-WIDTH CHA-TO-ERASE)
			 (SCREEN-OBJECT-WIDTH CHA-TO-ERASE)))
	 ((NULL CHA-TO-ERASE))
      (IF (SCREEN-CHA? CHA-TO-ERASE)
	  (ERASE-SCREEN-CHA CHA-TO-ERASE CURRENT-X-OFFSET CURRENT-Y-OFFSET)
	  (ERASE-SCREEN-BOX CHA-TO-ERASE CURRENT-X-OFFSET CURRENT-Y-OFFSET))
      (SETQ CURRENT-X-OFFSET (+ CURRENT-X-OFFSET x-incrementer)))))
    
(DEFUN ERASE-SCREEN-OBJ (SCREEN-OBJ)
  (WHEN (NOT-NULL SCREEN-OBJ)
    (CHECK-SCREEN-OBJ-ARG SCREEN-OBJ)
    (MULTIPLE-VALUE-BIND (WID HEI)
	(SCREEN-OBJ-SIZE SCREEN-OBJ)
      (MULTIPLE-VALUE-BIND (X-OFFSET Y-OFFSET)
	  (SCREEN-OBJ-OFFSETS SCREEN-OBJ)
	(DRAW-RECTANGLE TV:ALU-ANDCA WID HEI X-OFFSET Y-OFFSET)
	(SCREEN-OBJ-ZERO-SIZE SCREEN-OBJ)
	(TELL SCREEN-OBJ :SET-NEEDS-REDISPLAY-PASS-2? T)
	(TELL SCREEN-OBJ :SET-FORCE-REDISPLAY-INFS? T)))))

(DEFUN ERASE-SCREEN-OBJS (SCREEN-OBJS)
  (WHEN (NOT-NULL SCREEN-OBJS)
    (CHECK-SCREEN-OBJ-ARG (FIRST SCREEN-OBJS))
    (MULTIPLE-VALUE-BIND (WID HEI)
	(SCREEN-OBJS-SIZE SCREEN-OBJS)
      (MULTIPLE-VALUE-BIND (X-OFFSET Y-OFFSET)
	  (SCREEN-OBJ-OFFSETS (CAR SCREEN-OBJS))
	(DRAW-RECTANGLE TV:ALU-ANDCA WID HEI X-OFFSET Y-OFFSET)
	(DOLIST (SCREEN-OBJ SCREEN-OBJS)
	  (SCREEN-OBJ-ZERO-SIZE SCREEN-OBJ)
	  (TELL SCREEN-OBJ :SET-NEEDS-REDISPLAY-PASS-2? T)
	  (TELL SCREEN-OBJ :SET-FORCE-REDISPLAY-INFS? T))))))
	  
(DEFUN MOVE-SCREEN-BOXES (SCREEN-BOXES DELTA-X DELTA-Y)
  (WHEN (NOT-NULL SCREEN-BOXES)
    (CHECK-SCREEN-BOX-ARG (FIRST SCREEN-BOXES))
    (MULTIPLE-VALUE-BIND (WID HEI)
	(SCREEN-BOXES-AND-WHITESPACE-SIZE SCREEN-BOXES)
      (MULTIPLE-VALUE-BIND (X-OFFSET Y-OFFSET)
	  (SCREEN-OBJ-OFFSETS (CAR SCREEN-BOXES))
	(BITBLT-MOVE-REGION WID HEI X-OFFSET Y-OFFSET DELTA-X DELTA-Y)
	(DOLIST (SCREEN-BOX SCREEN-BOXES)
	  (INCF (SCREEN-OBJ-X-OFFSET SCREEN-BOX) DELTA-X)
	  (INCF (SCREEN-OBJ-Y-OFFSET SCREEN-BOX) DELTA-Y))))))

(DEFUN MOVE-SCREEN-OBJ (SCREEN-OBJ DELTA-X DELTA-Y)
  (WHEN (NOT-NULL SCREEN-OBJ)
    (CHECK-SCREEN-OBJ-ARG SCREEN-OBJ)
    (MULTIPLE-VALUE-BIND (WID HEI)
	(SCREEN-OBJ-SIZE SCREEN-OBJ)
      (MULTIPLE-VALUE-BIND (X-OFFSET Y-OFFSET)
	  (SCREEN-OBJ-OFFSETS SCREEN-OBJ)
	(BITBLT-MOVE-REGION WID HEI X-OFFSET Y-OFFSET DELTA-X DELTA-Y)
	(INCF (SCREEN-OBJ-X-OFFSET SCREEN-OBJ) DELTA-X)
	(INCF (SCREEN-OBJ-Y-OFFSET SCREEN-OBJ) DELTA-Y)))))

(DEFUN MOVE-SCREEN-OBJS (SCREEN-OBJS DELTA-X DELTA-Y)
  (WHEN (NOT-NULL SCREEN-OBJS)
    (CHECK-SCREEN-OBJ-ARG (FIRST SCREEN-OBJS))
    (MULTIPLE-VALUE-BIND (WID HEI)
	(SCREEN-OBJS-SIZE SCREEN-OBJS)
      (MULTIPLE-VALUE-BIND (X-OFFSET Y-OFFSET)
	  (SCREEN-OBJ-OFFSETS (CAR SCREEN-OBJS))
	(BITBLT-MOVE-REGION WID HEI X-OFFSET Y-OFFSET DELTA-X DELTA-Y)
	(DOLIST (SCREEN-OBJ SCREEN-OBJS)
	  (INCF (SCREEN-OBJ-X-OFFSET SCREEN-OBJ) DELTA-X)
	  (INCF (SCREEN-OBJ-Y-OFFSET SCREEN-OBJ) DELTA-Y))))))

(DEFUN MOVE-GRAPHICS-SHEET (GRAPHICS-SCREEN-SHEET DELTA-X DELTA-Y)
  (WHEN (NOT-NULL GRAPHICS-SCREEN-SHEET)
    (CHECK-GRAPHICS-SCREEN-SHEET-ARG GRAPHICS-SCREEN-SHEET)
    (LET* ((GRAPHICS-SHEET (GRAPHICS-SCREEN-SHEET-ACTUAL-OBJ GRAPHICS-SCREEN-SHEET))
	   (WID (GRAPHICS-SHEET-DRAW-WID GRAPHICS-SHEET))
	   (HEI (GRAPHICS-SHEET-DRAW-HEI GRAPHICS-SHEET))
	   (X-OFFSET (GRAPHICS-SCREEN-SHEET-X-OFFSET GRAPHICS-SCREEN-SHEET))
	   (Y-OFFSET (GRAPHICS-SCREEN-SHEET-Y-OFFSET GRAPHICS-SCREEN-SHEET)))
      (BITBLT-MOVE-REGION WID HEI X-OFFSET Y-OFFSET DELTA-X DELTA-Y)
      (INCF (GRAPHICS-SCREEN-SHEET-X-OFFSET GRAPHICS-SCREEN-SHEET) DELTA-X)
      (INCF (GRAPHICS-SCREEN-SHEET-Y-OFFSET GRAPHICS-SCREEN-SHEET) DELTA-Y))))

(DEFUN MOVE-INFERIOR-SCREEN-OBJS (INFERIORS DELTA-X DELTA-Y)
  (COND ((NULL INFERIORS))
	((GRAPHICS-SCREEN-SHEET? INFERIORS)
	 (MOVE-GRAPHICS-SHEET INFERIORS DELTA-X DELTA-Y))
	((AND (LISTP INFERIORS) (SCREEN-OBJ? (CAR INFERIORS)))
	 (MOVE-SCREEN-OBJS INFERIORS DELTA-X DELTA-Y))
	((SCREEN-OBJ? INFERIORS)
	 (MOVE-SCREEN-OBJ INFERIORS DELTA-X DELTA-Y))
       (T
	(FERROR "Don't know how to move inferior screen object(s), ~S" INFERIORS))))
	 
(DEFUN GRAY-SIZE-AND-OFFSETS (SCREEN-BOX)
  (MULTIPLE-VALUE-BIND (OUTER-WID OUTER-HEI)
      (SCREEN-BOX-BORDERS-FN ':MINIMUM-SIZE SCREEN-BOX)
    (MULTIPLE-VALUE-BIND (IL IT IR IB)
	(SCREEN-BOX-BORDERS-FN ':BORDER-WIDS SCREEN-BOX)
      (VALUES (- OUTER-WID IL IR) (- OUTER-HEI IT IB) IL IT))))

(DEFUN MOVE-GRAY-REGION (SCREEN-BOX DELTA-X DELTA-Y)
  (MULTIPLE-VALUE-BIND (GRAY-WID GRAY-HEI GRAY-X GRAY-Y)
      (GRAY-SIZE-AND-OFFSETS SCREEN-BOX)
    (BITBLT-MOVE-REGION GRAY-WID GRAY-HEI GRAY-X GRAY-Y DELTA-X DELTA-Y)))


;:SHRUNK   USE *SHRUNK-BOX-WID* AND *SHRUNK-BOX-HEI*
;:NORMAL  IF ACTUAL-BOX HAS FIXED-SIZE USE IT OTHERWISE USE OTHER CONSTRAINT
;:OUTERMOST USE OUTERMOST-SIZE

;; Note that with name tabs on the sides of boxes we have to make sure that the fixed size
;; refers to the part of the box with actual contents in it rather than the size of the entire
;; box label included

(DEFGET-METHOD (BOX :DISPLAY-STYLE-LIST) DISPLAY-STYLE-LIST)
(DEFSET-METHOD (BOX :SET-DISPLAY-STYLE-LIST) DISPLAY-STYLE-LIST)

(DEFGET-METHOD (SCREEN-BOX :DISPLAY-STYLE-LIST) DISPLAY-STYLE-LIST)
(DEFSET-METHOD (SCREEN-BOX :SET-DISPLAY-STYLE-LIST) DISPLAY-STYLE-LIST)

(DEFMETHOD (BOX :DISPLAY-STYLE) ()
  (CAR DISPLAY-STYLE-LIST))

;;; 1IMPORTANT.  The numbers returned by the various0 FIXED-SIZE1 methods refer to the size that
0;;; 1the0 INFERIORS1 want to be and NOT the size of the entire box since the size of the0 NAME
;;; 1can change

0(DEFMETHOD (BOX :FIXED-SIZE) ()
  (LET ((DISPLAY-STYLE (TELL SELF :DISPLAY-STYLE)))
    (SELECTQ DISPLAY-STYLE
      (:SHRUNK     (VALUES *SHRUNK-BOX-WID* *SHRUNK-BOX-HEI*))
      (:NORMAL     (TELL SELF :FIXED-SIZE-1))
      (OTHERWISE   (TELL SELF :FIXED-SIZE-1)))))

(DEFMETHOD (BOX :FIXED-SIZE?) ()
  (OR (EQ (CAR DISPLAY-STYLE-LIST) ':FIXED)
      (NUMBERP (CADR DISPLAY-STYLE-LIST))
      (NUMBERP (CADDR DISPLAY-STYLE-LIST))))

(DEFMETHOD (BOX :FIXED-SIZE-1) ()
  (VALUES (CADR DISPLAY-STYLE-LIST) (CADDR DISPLAY-STYLE-LIST)))

(DEFMETHOD (BOX :SET-DISPLAY-STYLE) (NEW-VALUE)
  (RPLACA DISPLAY-STYLE-LIST NEW-VALUE)
  (DOLIST (SCREEN-BOX (TELL SELF :SCREEN-OBJS))
    (TELL SCREEN-BOX :SET-FORCE-REDISPLAY-INFS? T)))

(DEFMETHOD (BOX :SET-FIXED-SIZE) (NEW-FIXED-WID NEW-FIXED-HEI)
  (RPLACA (CDR  DISPLAY-STYLE-LIST) NEW-FIXED-WID)  
  (RPLACA (CDDR DISPLAY-STYLE-LIST) NEW-FIXED-HEI))

(DEFMETHOD (BOX :AFTER :SET-FIXED-SIZE) (NEW-FIXED-WID NEW-FIXED-HEI)
  1;; A crock to get characters that were clipped to be redisplayed.
0  (UNLESS (AND NEW-FIXED-WID NEW-FIXED-HEI)
    (DOLIST (SBOX (TELL SELF :SCREEN-OBJS))
      (TELL SBOX :SET-FORCE-REDISPLAY-INFS?))))

(DEFMETHOD (SCREEN-BOX :DISPLAY-STYLE) ()
  (LET ((ACTUAL-OBJ-DISPLAY-STYLE-LIST (TELL ACTUAL-OBJ :DISPLAY-STYLE-LIST)))
    (OR (CAR DISPLAY-STYLE-LIST) (CAR ACTUAL-OBJ-DISPLAY-STYLE-LIST))))

(DEFMETHOD (SCREEN-BOX :FIXED-SIZE) ()
  (LET ((DISPLAY-STYLE (TELL SELF :DISPLAY-STYLE)))
    (SELECTQ DISPLAY-STYLE
      (:SHRUNK     (VALUES *SHRUNK-BOX-WID* *SHRUNK-BOX-HEI*))
      (:NORMAL     (TELL SELF :FIXED-SIZE-1))
      (OTHERWISE   (TELL SELF :FIXED-SIZE-1)))))

(DEFMETHOD (SCREEN-BOX :FIXED-SIZE-1) ()
  (MULTIPLE-VALUE-BIND (ACTUAL-OBJ-FIXED-WID ACTUAL-OBJ-FIXED-HEI)
      (TELL ACTUAL-OBJ :FIXED-SIZE-1)
    (VALUES (OR (CADR  DISPLAY-STYLE-LIST) ACTUAL-OBJ-FIXED-WID)
	    (OR (CADDR DISPLAY-STYLE-LIST) ACTUAL-OBJ-FIXED-HEI))))

(DEFMETHOD (SCREEN-BOX :SET-DISPLAY-STYLE) (NEW-VALUE)
  (RPLACA DISPLAY-STYLE-LIST NEW-VALUE)
  (TELL SELF :SET-FORCE-REDISPLAY-INFS? T))

(DEFMETHOD (SCREEN-BOX :SET-FIXED-SIZE) (NEW-FIXED-WID NEW-FIXED-HEI)
  (RPLACA (CDR  DISPLAY-STYLE-LIST) NEW-FIXED-WID)  
  (RPLACA (CDDR DISPLAY-STYLE-LIST) NEW-FIXED-HEI))

(DEFMETHOD (BOX :SHRINK) ()
  (TELL SELF :SET-DISPLAY-STYLE ':SHRUNK)
  (TELL SELF :MODIFIED))

(DEFMETHOD (BOX :UNSHRINK) ()
  (TELL SELF :SET-DISPLAY-STYLE ':NORMAL)
  (TELL SELF :MODIFIED))


(DEFMETHOD (SCREEN-BOX :SHRINK) ()
  (TELL ACTUAL-OBJ :SHRINK))

(DEFMETHOD (SCREEN-BOX :UNSHRINK) ()
  (TELL ACTUAL-OBJ :UNSHRINK))

(DEFMETHOD (SCREEN-ROW :LENGTH) ()
  (LENGTH SCREEN-CHAS))



;;;stuff for BOXTOPS

(DEFMETHOD (SCREEN-BOX :NAME-AND-INPUTS-ONLY) ()
  ;; add code here for displaying the inputs rather than greystuff
  (TELL SELF :GRAY-BODY))

(DEFMETHOD (SCREEN-ROW :UPDATE-SIZE-FOR-NAMING-ROW) (MAX-WID IGNORE)
  (WHEN (TELL SELF :NEEDS-REDISPLAY-PASS-1?)
    ;; We can't use the same :REDISPLAY-PASS-1 that normal screen rows use because
    ;; it erases out of synch characters which makes it REAL hard to properly erase
    ;; the name by using Xoring
    (SETQ SCREEN-CHAS (TELL ACTUAL-OBJ :CHAS))
    ;; We can cheat here because we are guaranteed that the name row will ONLY contain
    ;; characters and because a SCREEN-CHA = ACTUAL-CHA
    (LOOP FOR SCREEN-CHA IN SCREEN-CHAS
	  FOR FONT = (FONT-NO SCREEN-CHA)
	  FOR CLIPPED-P = T
	  SUM (CHA-WID FONT (CHA-CODE SCREEN-CHA)) INTO WIDTH
	  MAXIMIZE (CHA-HEI FONT) INTO HEIGHT
	  WHILE (< WIDTH MAX-WID)
	  DO (SETQ CLIPPED-P NIL)
	  FINALLY
	  (SETQ NEW-WID WIDTH
		NEW-HEI HEIGHT
		NEW-X-GOT-CLIPPED? CLIPPED-P))
    (TELL SELF :GOT-REDISPLAYED)))



;;; Things having to do with a window's outermost screen box.

(DEFUN OUTERMOST-BOX (&OPTIONAL (WINDOW *BOXER-PANE*))
  (SCREEN-OBJ-ACTUAL-OBJ (OUTERMOST-SCREEN-BOX WINDOW)))

(DEFUN OUTERMOST-SCREEN-BOX (&OPTIONAL (WINDOW *BOXER-PANE*))
  (TELL WINDOW :OUTERMOST-SCREEN-BOX))

(DEFMETHOD (ACTUAL-OBJ-MIXIN :DISPLAYED-SCREEN-OBJS) (&OPTIONAL (WINDOW *BOXER-PANE*))
  (LET ((ALL-SCREEN-OBJS (TELL SELF :SCREEN-OBJS))
	(OUTERMOST-SCREEN-BOX (OUTERMOST-SCREEN-BOX WINDOW)))
    (WITH-COLLECTION
      (DOLIST (SCREEN-OBJ ALL-SCREEN-OBJS)
	(IF (TELL SCREEN-OBJ :SUPERIOR? OUTERMOST-SCREEN-BOX)
	    (COLLECT SCREEN-OBJ))))))

(DEFMETHOD (SCREEN-OBJ :SUPERIOR?) (SCREEN-OBJ)
  "Is the Arg a superior of the instance ?"
  (LET ((SUPERIOR (TELL SELF :SUPERIOR)))
    (OR (EQ SCREEN-OBJ SELF)
	(EQ SCREEN-OBJ SUPERIOR)
	(AND (SCREEN-OBJ? SUPERIOR)
	     (TELL SUPERIOR :SUPERIOR? SCREEN-OBJ)))))

;;; Stuff for zooming in and out of boxes

(DEFUN GET-PREVIOUS-OUTERMOST-BOX-VALUES ()
  (LET ((PREVIOUS-OUTERMOST-SCREEN-BOX (POP *OUTERMOST-SCREEN-BOX-STACK*)))
    (IF (NULL PREVIOUS-OUTERMOST-SCREEN-BOX)
	(VALUES *INITIAL-BOX* (CAR (TELL *INITIAL-BOX* :SCREEN-OBJS)))
	(VALUES (TELL PREVIOUS-OUTERMOST-SCREEN-BOX :ACTUAL-OBJ)
		PREVIOUS-OUTERMOST-SCREEN-BOX))))

(DEFSUBST BOX-BORDER-ZOOM-IN (NEW-SCREEN-BOX WINDOW)
  (DRAWING-ON-WINDOW (WINDOW)
    (WHEN (TELL NEW-SCREEN-BOX :VISIBLE?) 
      (MULTIPLE-VALUE-BIND (NEW-SCREEN-BOX-WID NEW-SCREEN-BOX-HEI)
	  (SCREEN-OBJ-SIZE NEW-SCREEN-BOX)
	(MULTIPLE-VALUE-BIND (NEW-SCREEN-BOX-X NEW-SCREEN-BOX-Y)
	    (TELL NEW-SCREEN-BOX :POSITION)
	  (MULTIPLE-VALUE-BIND (OUTERMOST-SCREEN-BOX-WID OUTERMOST-SCREEN-BOX-HEI)
	      (OUTERMOST-SCREEN-BOX-SIZE)
	    (MULTIPLE-VALUE-BIND (OUTERMOST-SCREEN-BOX-X OUTERMOST-SCREEN-BOX-Y)
		(OUTERMOST-SCREEN-BOX-POSITION)	
	      (BOX-BORDERS-FN ':ZOOM (TELL (TELL NEW-SCREEN-BOX :ACTUAL-OBJ) :TYPE)
			      NEW-SCREEN-BOX
			      OUTERMOST-SCREEN-BOX-WID OUTERMOST-SCREEN-BOX-HEI
			      NEW-SCREEN-BOX-WID NEW-SCREEN-BOX-HEI
			      OUTERMOST-SCREEN-BOX-X OUTERMOST-SCREEN-BOX-Y
			      NEW-SCREEN-BOX-X NEW-SCREEN-BOX-Y
			      20.))))))))

(DEFSUBST BOX-BORDER-ZOOM-OUT (OLD-SCREEN-BOX WINDOW)
  (DRAWING-ON-WINDOW (WINDOW)
    (WHEN (TELL OLD-SCREEN-BOX :VISIBLE?) 
      (MULTIPLE-VALUE-BIND (OLD-SCREEN-BOX-WID OLD-SCREEN-BOX-HEI)
	  (SCREEN-OBJ-SIZE OLD-SCREEN-BOX)
	(MULTIPLE-VALUE-BIND (OLD-SCREEN-BOX-X OLD-SCREEN-BOX-Y)
	    (TELL OLD-SCREEN-BOX :POSITION)
	  (MULTIPLE-VALUE-BIND (OUTERMOST-SCREEN-BOX-WID OUTERMOST-SCREEN-BOX-HEI)
	      (OUTERMOST-SCREEN-BOX-SIZE)
	    (MULTIPLE-VALUE-BIND (OUTERMOST-SCREEN-BOX-X OUTERMOST-SCREEN-BOX-Y)
		(OUTERMOST-SCREEN-BOX-POSITION)	
	      (BOX-BORDERS-FN ':ZOOM (TELL (TELL OLD-SCREEN-BOX :ACTUAL-OBJ) :TYPE)
			      OLD-SCREEN-BOX
			      OLD-SCREEN-BOX-WID OLD-SCREEN-BOX-HEI
			      OUTERMOST-SCREEN-BOX-WID OUTERMOST-SCREEN-BOX-HEI
			      OLD-SCREEN-BOX-X OLD-SCREEN-BOX-Y
			      OUTERMOST-SCREEN-BOX-X OUTERMOST-SCREEN-BOX-Y
			      16.))))))))

(DEFUN SET-OUTERMOST-BOX (NEW-OUTERMOST-BOX &OPTIONAL (NEW-OUTERMOST-SCREEN-BOX
							(CAR (TELL-CHECK-NIL
							       NEW-OUTERMOST-BOX
							       :DISPLAYED-SCREEN-OBJS)))
			  (WINDOW *BOXER-PANE*))
  (LET ((OLD-OUTERMOST-SCREEN-BOX *OUTERMOST-SCREEN-BOX*))
    (IF (OR (GRAPHICS-BOX? NEW-OUTERMOST-BOX)
	    (AND (PORT-BOX? NEW-OUTERMOST-BOX)
		 (GRAPHICS-BOX? (TELL NEW-OUTERMOST-BOX :PORTS))))
	(BEEP)
	(WHEN (NAME-ROW? (POINT-ROW)) (MOVE-POINT (BOX-FIRST-BP-VALUES NEW-OUTERMOST-BOX)))
	(REDRAW-STATUS-LINE (TELL NEW-OUTERMOST-BOX :NAME))
	(BOX-BORDER-ZOOM-OUT NEW-OUTERMOST-SCREEN-BOX WINDOW)
	(SET-OUTERMOST-SCREEN-BOX
	  (tell new-outermost-box :allocate-outermost-screen-box-for-use-in window
		NEW-OUTERMOST-SCREEN-BOX)
	  WINDOW)
	(BOX-BORDER-ZOOM-IN OLD-OUTERMOST-SCREEN-BOX WINDOW))))

;;;these should go somewhere else eventually...
(DEFMETHOD (SCREEN-OBJ :VISIBLE?)()
  (MEMQ SELF (TELL (TELL SELF :ACTUAL-OBJ) :DISPLAYED-SCREEN-OBJS)))

(DEFUN SET-OUTERMOST-SCREEN-BOX (NEW-OUTERMOST-SCREEN-BOX &OPTIONAL (WINDOW *BOXER-PANE*))
  (WITHOUT-INTERRUPTS				;keep the mouse process from looking at 
    (REDISPLAYING-WINDOW (WINDOW)		;the screen when it is in a munged state
      (UNLESS (EQ NEW-OUTERMOST-SCREEN-BOX *OUTERMOST-SCREEN-BOX*)
	(DECONFIGURE-SCREEN-BOX-TO-BE-OUTERMOST-BOX *OUTERMOST-SCREEN-BOX* WINDOW)
	(CONFIGURE-SCREEN-BOX-TO-BE-OUTERMOST-BOX NEW-OUTERMOST-SCREEN-BOX WINDOW)
	(ERASE-SCREEN-OBJ *OUTERMOST-SCREEN-BOX*) 
	(SETQ *OUTERMOST-SCREEN-BOX* NEW-OUTERMOST-SCREEN-BOX)))
    (SETQ *OUTERMOST-SCREEN-BOX* (OUTERMOST-SCREEN-BOX))	;why is this neccessary ?
    (LET ((*COMPLETE-REDISPLAY-IN-PROGRESS?* T)
	  (OLD-SCREEN-ROW (TELL-CHECK-NIL NEW-OUTERMOST-SCREEN-BOX :SCREEN-ROW)))
      (WHEN (SCREEN-ROW? OLD-SCREEN-ROW)
	;; we need to break up the screen-structure
	(TELL OLD-SCREEN-ROW :KILL-SCREEN-CHAS-FROM 0)
	(TELL (TELL OLD-SCREEN-ROW :SUPERIOR) :DEALLOCATE-SELF))
      (REDISPLAY-WINDOW WINDOW))))

(DEFUN CONFIGURE-SCREEN-BOX-TO-BE-OUTERMOST-BOX (SCREEN-BOX &OPTIONAL (WINDOW *BOXER-PANE*))
  (MULTIPLE-VALUE-BIND (MAX-WID MAX-HEI)
      (OUTERMOST-SCREEN-BOX-SIZE WINDOW)
    (MULTIPLE-VALUE-BIND (X-OFFSET Y-OFFSET)
	(OUTERMOST-SCREEN-BOX-POSITION WINDOW)
      (TELL SCREEN-BOX :SET-DISPLAY-STYLE ':NORMAL)
      (TELL SCREEN-BOX :SET-FIXED-SIZE MAX-WID MAX-HEI)      
      (TELL SCREEN-BOX :SET-OFFSETS X-OFFSET Y-OFFSET))))

(DEFUN DECONFIGURE-SCREEN-BOX-TO-BE-OUTERMOST-BOX (SCREEN-BOX &OPTIONAL IGNORE)
  (TELL SCREEN-BOX :SET-DISPLAY-STYLE NIL)
  (TELL SCREEN-BOX :SET-FIXED-SIZE NIL NIL)
  (TELL SCREEN-BOX :SET-OFFSETS 0 0))


;;;; Interaction with redisplayable-window-mixin.

(DEFMETHOD (REDISPLAYABLE-WINDOW-MIXIN :BEFORE :INIT) (&REST IGNORE)
  (UNLESS (MEMQ SELF *REDISPLAYABLE-WINDOWS*)
	  (PUSH SELF *REDISPLAYABLE-WINDOWS*)))

(DEFMETHOD (REDISPLAYABLE-WINDOW-MIXIN :AFTER :KILL) (&REST IGNORE)
  (SETQ *REDISPLAYABLE-WINDOWS* (DELETE SELF *REDISPLAYABLE-WINDOWS*)))

(DEFMETHOD (REDISPLAYABLE-WINDOW-MIXIN :OUTERMOST-SCREEN-BOX) ()
  OUTERMOST-SCREEN-BOX)

(DEFMETHOD (REDISPLAYABLE-WINDOW-MIXIN :SET-OUTERMOST-SCREEN-BOX) (NEW-VALUE)
  (SETQ OUTERMOST-SCREEN-BOX NEW-VALUE))



(DEFUN REDISPLAY-CLUE (TYPE &REST ARGS)
  (LET ((HANDLER (GET TYPE ':REDISPLAY-CLUE)))
    (IF (NOT-NULL HANDLER)
	(LEXPR-FUNCALL HANDLER TYPE ARGS)
	(FERROR "~S is an unknown type of redisplay-clue." TYPE))))

(DEFUN (:PROPERTY :CLEAR-SCREEN :REDISPLAY-CLUE) (&REST IGNORE)
  (PUSH '(:CLEAR-SCREEN) *REDISPLAY-CLUES*))





(DEFUN OUTERMOST-SCREEN-BOX? (SCREEN-OBJ)
  (AND (SCREEN-BOX? SCREEN-OBJ)
       (EQ SCREEN-OBJ (OUTERMOST-SCREEN-BOX))))

(DEFMETHOD (SCREEN-OBJ :POSITION) (&AUX TEMP)
  (MULTIPLE-VALUE-BIND (SUPERIOR-X-OFF SUPERIOR-Y-OFF)
      (COND ((OUTERMOST-SCREEN-BOX? SELF)
	     (VALUES 0 0))
	    (T
	     (SETQ TEMP (TELL SELF :SUPERIOR))
	     (TELL TEMP :POSITION)))
    (VALUES (+ SUPERIOR-X-OFF X-OFFSET)
	    (+ SUPERIOR-Y-OFF Y-OFFSET))))

(DEFMETHOD (SCREEN-CHA :NEXT-SCREEN-CHA-POSITION) ()
  (MULTIPLE-VALUE-BIND (X Y)
      (TELL SELF :POSITION)
    (VALUES (+ X WID) Y)))

(DEFMETHOD (SCREEN-ROW :FIRST-SCREEN-CHA-POSITION) ()
  (TELL SELF :POSITION))

(DEFMETHOD (SCREEN-BOX :FIRST-SCREEN-CHA-POSITION) ()
  (MULTIPLE-VALUE-BIND (X Y)
      (TELL SELF :POSITION)
    (MULTIPLE-VALUE-BIND (IL IT NIL NIL)
	(SCREEN-BOX-BORDERS-FN ':BORDER-WIDS SELF)
      (VALUES (+ X IL) (+ Y IT)))))


;;;box border functions...
;leave these out for now until name tags are finished
;(DEFFLAVOR BOX-BORDERS-BLINKER
;	((BOX-TYPE ':DOIT)
;	 (WINDOW-X 0)
;	 (WINDOW-Y 0)
;	 (WID 0)
;	 (HEI 0))
;	;; There is no sense giving the the mouse
;	;; fast tracking blinker mixin. Since it
;	;; changes size all the time, and it will
;	;; often be bigger than 32. by 32.
;	(TV:MOUSE-BLINKER-MIXIN TV:BLINKER))
;
;(DEFMETHOD (BOX-BORDERS-BLINKER :BEFORE :INIT) (&REST IGNORE)
;  (MULTIPLE-VALUE-BIND (X-OFF Y-OFF)
;      (TV:SHEET-CALCULATE-OFFSETS *BOXER-PANE* TV:MAIN-SCREEN)
;    (TELL SELF :SET-OFFSETS X-OFF Y-OFF)))
;
;(DEFMETHOD (BOX-BORDERS-BLINKER :SET-BOX-TYPE) (NEW-VALUE)
;  (OR (EQ NEW-VALUE BOX-TYPE)
;      (TV:PREPARE-SHEET (TV:SHEET)
;	(TV:OPEN-BLINKER SELF)
;	(SETQ BOX-TYPE NEW-VALUE))))
;
;(DEFMETHOD (BOX-BORDERS-BLINKER :SET-WINDOW-X) (NEW-VALUE)
;  (OR (EQ NEW-VALUE WINDOW-X)
;      (TV:PREPARE-SHEET (TV:SHEET)
;	(TV:OPEN-BLINKER SELF)
;	(SETQ WINDOW-X NEW-VALUE))))
;
;(DEFMETHOD (BOX-BORDERS-BLINKER :SET-WINDOW-Y) (NEW-VALUE)
;  (OR (EQ NEW-VALUE WINDOW-Y)
;      (TV:PREPARE-SHEET (TV:SHEET)
;	(TV:OPEN-BLINKER SELF)
;	(SETQ WINDOW-Y NEW-VALUE))))
;
;(DEFMETHOD (BOX-BORDERS-BLINKER :SET-CURSORPOS) (X Y)
;  (MULTIPLE-VALUE-BIND (MIN-WID MIN-HEI)
;      (BOX-BORDERS-FN ':MINIMUM-SIZE BOX-TYPE)
;    (LET ((NEW-WID (MAX MIN-WID (- X WINDOW-X)))
;	  (NEW-HEI (MAX MIN-HEI (- Y WINDOW-Y))))
;      (OR (AND (EQ WID NEW-WID) (EQ HEI NEW-HEI))
;	  (TV:PREPARE-SHEET (TV:SHEET)
;	    (TV:OPEN-BLINKER SELF)
;	    (SETQ WID NEW-WID
;		  HEI NEW-HEI))))))
;
;(DEFMETHOD (BOX-BORDERS-BLINKER :READ-CURSORPOS) ()
;  (VALUES WINDOW-X WINDOW-Y))
;
;(DEFMETHOD (BOX-BORDERS-BLINKER :SET-WID) (NEW-VALUE)
;  (OR (EQ NEW-VALUE WID)
;      (TV:PREPARE-SHEET (TV:SHEET)
;	(TV:OPEN-BLINKER SELF)
;	(SETQ WID NEW-VALUE))))
;
;(DEFMETHOD (BOX-BORDERS-BLINKER :SET-HEI) (NEW-VALUE)
;  (OR (EQ HEI NEW-VALUE)
;      (TV:PREPARE-SHEET (TV:SHEET)
;	(TV:OPEN-BLINKER SELF)
;	(SETQ HEI NEW-VALUE))))
;
;(DEFMETHOD (BOX-BORDERS-BLINKER :SIZE) ()
;  (VALUES WID HEI))
;
;(DEFMETHOD (BOX-BORDERS-BLINKER :BLINK) ()
;  (DRAWING-ON-WINDOW-WITHOUT-PREPARE-SHEET (TV:SHEET)
;    (BOX-BORDERS-FN ':DRAW BOX-TYPE WID HEI WINDOW-X WINDOW-Y)))
;
;(DEFUN ADJUST-BOX-SIZE-WITH-MOUSE (WINDOW)
;  (USING-BOX-BORDERS-BLINKER (BL)
;    (MULTIPLE-VALUE-BIND (WINDOW-X WINDOW-Y)
;	(MOUSE-POSITION-IN-WINDOW-COORDINATES WINDOW)
;      (LET* ((SCREEN-BOX (FIND-SCREEN-BOX-AT-POSITION WINDOW-X WINDOW-Y WINDOW))
;	     (ACTUAL-BOX (SCREEN-OBJ-ACTUAL-OBJ SCREEN-BOX))
;	     (BOX-TYPE (TELL ACTUAL-BOX :TYPE))
;	     (WID (SCREEN-OBJ-WID SCREEN-BOX))
;	     (HEI (SCREEN-OBJ-HEI SCREEN-BOX)))
;	(MULTIPLE-VALUE-BIND (WINDOW-X WINDOW-Y)
;	    (SCREEN-OBJ-POSITION SCREEN-BOX)
;	  (TELL BL :SET-WINDOW-X WINDOW-X)
;	  (TELL BL :SET-WINDOW-Y WINDOW-Y)
;	  (MULTIPLE-VALUE-BIND (X-OFFSET Y-OFFSET)
;	      (TV:SHEET-CALCULATE-OFFSETS WINDOW TV:MOUSE-SHEET)
;	    (MULTIPLE-VALUE-BIND (IL IT NIL NIL)
;		(TELL WINDOW :MARGINS)
;	      (TV:MOUSE-WARP (+ X-OFFSET IL WINDOW-X WID)
;			     (+ Y-OFFSET IT WINDOW-Y HEI))))
;	  (DRAWING-ON-WINDOW (*BOXER-PANE*)
;	    (BOX-BORDERS-FN ':DRAW BOX-TYPE WID HEI WINDOW-X WINDOW-Y))
;	  (TELL BL :SET-VISIBILITY ':ON)
;	  (TELL BL :TRACK-MOUSE)
;	  (PROCESS-WAIT "Adjust Size" #'ADJUST-BOX-SIZE-WITH-MOUSE-SLEEP-FN)
;	  (MULTIPLE-VALUE-BIND (NEW-WID NEW-HEI)
;	      (TELL BL :SIZE)
;	    (TELL ACTUAL-BOX :SET-FIXED-SIZE NEW-WID NEW-HEI))
;	  (USE-CURSOR-BLINKER)
;	  (DRAWING-ON-WINDOW (*BOXER-PANE*)
;	    (BOX-BORDERS-FN ':DRAW BOX-TYPE WID HEI WINDOW-X WINDOW-Y))
;	  (FORCE-REDISPLAY)
;	  (SETQ MOUSE-MOVES-HANDLER 'MOUSE-IS-STOPPED-HANDLER
;		TV:MOUSE-RECONSIDER T))))))
;
;(DEFUN ADJUST-BOX-SIZE-WITH-MOUSE-SLEEP-FN ()
;  (ZEROP TV:MOUSE-LAST-BUTTONS))



(DEFUN OUTERMOST-SCREEN-BOX-SIZE (&OPTIONAL (WINDOW *BOXER-PANE*))
  (MULTIPLE-VALUE-BIND (WINDOW-INNER-WID WINDOW-INNER-HEI)
      (TELL WINDOW :INSIDE-SIZE)
    (VALUES (- WINDOW-INNER-WID (* 2 *SPACE-AROUND-OUTERMOST-SCREEN-BOX*))
	    (- WINDOW-INNER-HEI (* 2 *SPACE-AROUND-OUTERMOST-SCREEN-BOX*)))))

(DEFUN OUTERMOST-SCREEN-BOX-POSITION (&OPTIONAL IGNORE)
  (VALUES *SPACE-AROUND-OUTERMOST-SCREEN-BOX*
	  *SPACE-AROUND-OUTERMOST-SCREEN-BOX*))


;;;;Operations Particular to SCREEN-BPs.

(DEFUN CURRENT-SCREEN-ROW (ACTUAL-ROW &OPTIONAL (SCREEN-BOX (BP-SCREEN-BOX *POINT*)))
  (LET ((SCREEN-ROWS (TELL ACTUAL-ROW :DISPLAYED-SCREEN-OBJS)))
    (DOLIST (SCREEN-ROW SCREEN-ROWS)
      (WHEN (EQ (TELL SCREEN-ROW :SUPERIOR) SCREEN-BOX)
	(RETURN SCREEN-ROW)))))

(DEFUN BP-POSITIONS (BP)
  (CHECK-BP-ARG BP)
  (LET ((BOX (BP-BOX BP))
	(ROW (BP-ROW BP)))
    (COND ((NULL BOX) NIL)
	  ((NAME-ROW? ROW)
	   (SCREEN-BOX-NAME-ROW-BP-POSITION (BP-SCREEN-BOX *POINT*) ROW))
	  ((EQ ':SHRUNK (TELL (BP-SCREEN-BOX *POINT*) :DISPLAY-STYLE))
	   (SCREEN-BOX-FIRST-BP-POSITION (BP-SCREEN-BOX *POINT*)))
	  ((NULL (CURRENT-SCREEN-ROW ROW))
	   (SCREEN-BOX-LAST-BP-POSITION (BP-SCREEN-BOX *POINT*)))
	  (T
	   (ROW-POINT-POSITION (CURRENT-SCREEN-ROW ROW))))))

(DEFUN SCREEN-BOX-FIRST-BP-POSITION (SCREEN-BOX)
  (MULTIPLE-VALUE-BIND (X Y)
      (TELL SCREEN-BOX :POSITION)
    (MULTIPLE-VALUE-BIND (IL IT IGNORE IGNORE)
	(SCREEN-BOX-BORDERS-FN ':BORDER-WIDS SCREEN-BOX)
      (CONS (+ X IL) (+ Y IT)))))

(DEFUN SCREEN-BOX-LAST-BP-POSITION (SCREEN-BOX)
  (MULTIPLE-VALUE-BIND (X Y)
      (TELL SCREEN-BOX :POSITION)
    (CONS (+ X (TELL SCREEN-BOX :WID))
	  (- (+ Y (TELL SCREEN-BOX :HEI)) *MINIMUM-CURSOR-HEIGHT*))))

(DEFUN SCREEN-BOX-NAME-ROW-BP-POSITION  (SCREEN-BOX NAME-ROW)
  (LET ((CHA-NO (BP-CHA-NO *POINT*)))
    (MULTIPLE-VALUE-BIND (X Y)
	(TELL SCREEN-BOX :POSITION)
      (MULTIPLE-VALUE-BIND (TAB-X TAB-Y)
	  (SCREEN-BOX-BORDERS-FN ':TAB-OFFSETS SCREEN-BOX)
	(LOOP FOR CHA IN (TELL NAME-ROW :CHAS)
	      FOR INDEX = 0 THEN (1+ INDEX)
	      UNTIL (= INDEX CHA-NO)
	      SUM (CHA-WID (FONT-NO CHA) (CHA-CODE CHA)) INTO LAST-X
	      FINALLY (RETURN (CONS (+ X TAB-X LAST-X) (+ Y TAB-Y))))))))

(DEFUN ROW-POINT-POSITION (SCREEN-ROW)
  (LET* ((ROW (TELL SCREEN-ROW :ACTUAL-OBJ))
	 (LENGTH-IN-CHAS (TELL ROW :LENGTH-IN-CHAS))
	 (CHA-NO (BP-CHA-NO *POINT*)))
    (COND ((NULL (BP-SCREEN-BOX *POINT*))
	   (FERROR NIL "Lost the current Screen Box"))
	  ((>= CHA-NO LENGTH-IN-CHAS)
	   (END-OF-ROW-POINT-LOCATION SCREEN-ROW)) 
	  (T (INSIDE-OF-ROW-POINT-LOCATION SCREEN-ROW CHA-NO)))))

(DEFUN END-OF-ROW-POINT-LOCATION (SCREEN-ROW)
  (MULTIPLE-VALUE-BIND (SCREEN-ROW-X SCREEN-ROW-Y)
      (TELL SCREEN-ROW :POSITION)
    (CONS (+ SCREEN-ROW-X (SCREEN-OBJ-WID SCREEN-ROW)) SCREEN-ROW-Y)))

(DEFUN INSIDE-OF-ROW-POINT-LOCATION (SCREEN-ROW CHA-NO)
  (MULTIPLE-VALUE-BIND (SCREEN-ROW-X SCREEN-ROW-Y)
      (TELL SCREEN-ROW :POSITION)
    (CONS (+ SCREEN-ROW-X (X-COORDINATE-OF-CHA-NO SCREEN-ROW CHA-NO)) SCREEN-ROW-Y)))

(DEFUN X-COORDINATE-OF-CHA-NO (ROW CHA-NO &AUX(X-COORD 0))
  (DO* ((INDEX 0 (+ INDEX 1))
	(CHA (TELL ROW :SCREEN-CHA-AT-CHA-NO INDEX) (TELL ROW :SCREEN-CHA-AT-CHA-NO INDEX)))
       ((OR (NULL CHA)(= INDEX CHA-NO)) X-COORD)
    (SETQ X-COORD (+ X-COORD (SCREEN-OBJECT-WIDTH CHA)))))



(DEFUN FIND-SCREEN-BP-AT-POSITION (X Y &OPTIONAL (WINDOW *BOXER-PANE*))
  (REDISPLAYING-WINDOW (WINDOW)
    (MULTIPLE-VALUE-BIND (OUTERMOST-SCREEN-BOX-X-OFFSET OUTERMOST-SCREEN-BOX-Y-OFFSET)
	(OUTERMOST-SCREEN-BOX-POSITION WINDOW)
      (TELL (OUTERMOST-SCREEN-BOX WINDOW) :FIND-SCREEN-BP-AT-OFFSET
					  (- X OUTERMOST-SCREEN-BOX-X-OFFSET)
					  (- Y OUTERMOST-SCREEN-BOX-Y-OFFSET)))))

(DEFMETHOD (SCREEN-CHA :FIND-SCREEN-BP-AT-OFFSET) (X-OFF Y-OFF)
  Y-OFF						;prevent bound but never used warnings
  (IF (> X-OFF (// WID 2))
      (TELL SELF :NEXT-SCREEN-BP)
      (TELL SELF :SCREEN-BP)))

(DEFMETHOD (SCREEN-ROW :FIND-SCREEN-BP-AT-OFFSET) (X-OFF Y-OFF &AUX TEMP)
  (DO* ((ITER-SCREEN-CHAS SCREEN-CHAS (CDR ITER-SCREEN-CHAS))
	(SCREEN-CHA (CAR ITER-SCREEN-CHAS) (CAR ITER-SCREEN-CHAS)))
       ((NULL ITER-SCREEN-CHAS)
	;; We have gone through all this screen row's screen chas
	;; without finding a screen cha at the specified offset.
	;; Just return this screen-row's last screen bp.
	(TELL SELF :LAST-SCREEN-BP))
    (LET ((SCREEN-CHA-X-OFFSET (SCREEN-OBJ-X-OFFSET SCREEN-CHA))
	  (SCREEN-CHA-Y-OFFSET (SCREEN-OBJ-Y-OFFSET SCREEN-CHA))
	  (SCREEN-CHA-WID (SCREEN-OBJ-WID SCREEN-CHA))
	  (SCREEN-CHA-HEI (SCREEN-OBJ-HEI SCREEN-CHA)))
      (COND ((AND (> (+ SCREEN-CHA-X-OFFSET SCREEN-CHA-WID) X-OFF)
		  (< SCREEN-CHA-HEI Y-OFF)
		  (> Y-OFF (// (+ HEI SCREEN-CHA-HEI) 2))
		  (NOT-NULL (SETQ TEMP (TELL SELF :NEXT-SCREEN-ROW))))
	     ;; This screen cha is at the right x-off, but it is so
	     ;; short that the specified offset is actually closer
	     ;; to somehing in the next screen row. So ask the next
	     ;; screen row to find that something. [Note the next
	     ;; screen row won't screw us by passing the buck back
	     ;; cause the rule says you can only pass the buck down].
	     (RETURN
	       (TELL TEMP :FIND-SCREEN-BP-AT-OFFSET
			  (- X-OFF (- (SCREEN-OBJ-X-OFFSET TEMP) X-OFFSET))
			  (- Y-OFF (- (SCREEN-OBJ-Y-OFFSET TEMP) Y-OFFSET)))))
	    ((> (+ SCREEN-CHA-X-OFFSET SCREEN-CHA-WID) X-OFF)
	     ;; This screen cha is at the right x-off, and it is
	     ;; tall enough to catch the y-off too.
	     (RETURN
	       (TELL SCREEN-CHA :FIND-SCREEN-BP-AT-OFFSET
				(- X-OFF SCREEN-CHA-X-OFFSET)
				(- Y-OFF SCREEN-CHA-Y-OFFSET))))))))

(DEFMETHOD (SCREEN-BOX :FIND-SCREEN-BP-AT-OFFSET) (X-OFF Y-OFF)
  (DO* ((ITER-SCREEN-ROWS SCREEN-ROWS (CDR ITER-SCREEN-ROWS))
	(ITER-SCREEN-ROW (CAR ITER-SCREEN-ROWS) (CAR ITER-SCREEN-ROWS)))
       ((NULL ITER-SCREEN-ROWS)
	;; We have gone through all this screen box's screen rows
	;; without finding a screen row at the specified offset.
	;; Just return this screen-box's last screen-bp
	(TELL SELF :LAST-SCREEN-BP))
    (LET ((ITER-SCREEN-ROW-X-OFFSET (SCREEN-OBJ-X-OFFSET ITER-SCREEN-ROW))
	  (ITER-SCREEN-ROW-Y-OFFSET (SCREEN-OBJ-Y-OFFSET ITER-SCREEN-ROW))
	  (ITER-SCREEN-ROW-HEI (SCREEN-OBJ-HEI ITER-SCREEN-ROW)))
      (COND ((AND (> (+ ITER-SCREEN-ROW-Y-OFFSET ITER-SCREEN-ROW-HEI) Y-OFF)
		  (SCREEN-ROW? ITER-SCREEN-ROW))
	     (RETURN
	       (TELL ITER-SCREEN-ROW :FIND-SCREEN-BP-AT-OFFSET
				     (- X-OFF ITER-SCREEN-ROW-X-OFFSET)
				     (- Y-OFF ITER-SCREEN-ROW-Y-OFFSET))))))))

(DEFMETHOD (SCREEN-CHA :SCREEN-BP) ()
  (LET ((BP (MAKE-BP 'FIXED)))
    (MOVE-BP BP (CHA-BP-VALUES ACTUAL-OBJ))
    BP))

(DEFMETHOD (SCREEN-CHA :NEXT-SCREEN-BP) ()
  (LET ((BP (MAKE-BP 'FIXED)))
    (MOVE-BP BP (CHA-NEXT-BP-VALUES ACTUAL-OBJ))
    BP))

(DEFMETHOD (SCREEN-ROW :FIRST-SCREEN-BP) ()
  (LET ((BP (MAKE-BP 'FIXED)))
    (MOVE-BP BP (ROW-FIRST-BP-VALUES ACTUAL-OBJ))
    BP))

(DEFMETHOD (SCREEN-ROW :LAST-SCREEN-BP) ()
  (LET ((BP (MAKE-BP 'FIXED)))
    (MOVE-BP BP (ROW-LAST-BP-VALUES ACTUAL-OBJ))
    BP))

(DEFMETHOD (SCREEN-BOX :FIRST-SCREEN-BP) ()
  (LET ((BP (MAKE-BP 'FIXED)))
    (MOVE-BP BP (BOX-FIRST-BP-VALUES ACTUAL-OBJ))
    BP))

(DEFMETHOD (SCREEN-BOX :LAST-SCREEN-BP) ()
  (LET ((BP (MAKE-BP 'FIXED)))
    (MOVE-BP BP (BOX-LAST-BP-VALUES ACTUAL-OBJ))
    BP))

(DEFUN SCREEN-BOX ()
  (TELL *POINT* :SCREEN-BOX))

(DEFUN SCREEN-ROW ()
  (TELL (TELL *POINT* :ROW) :ALLOCATE-SCREEN-OBJ-FOR-USE-IN (SCREEN-BOX)))

(DEFUN INF-CURRENT-SCREEN-BOX (BOX)		;returns the screen obj of box which is within
  (CAR (MEM #'(LAMBDA (SUPERIOR-BOX BOX) (EQ SUPERIOR-BOX (TELL BOX :SUPERIOR-SCREEN-BOX)))
	    (BP-SCREEN-BOX *POINT*)
	    (TELL BOX :DISPLAYED-SCREEN-OBJS))))



;;;; BOX-BORDERS-FN

(DEFUN DECLARE-BOX-BORDERS-FN-PARAMETERS (BOX-TYPE TYPE-LABEL-STRING
					  &OPTIONAL (TYPE-LABEL-FONT-NO 1)
						    (TYPE-LABEL-INDENTATION 5)
						    (BORDER-WID 1)
						    (BORDER-SPA 1)
						    (NAME-BORDER-SPA 1)
						    (NAME-BORDER-WID 1)
						    (NAME-HIGHLIGHT T))
  (BOX-BORDERS-FN-SET-TYPE-LABEL-STRING BOX-TYPE TYPE-LABEL-STRING)
  (BOX-BORDERS-FN-SET-TYPE-LABEL-FONT-NO BOX-TYPE TYPE-LABEL-FONT-NO)
  (BOX-BORDERS-FN-SET-TYPE-LABEL-INDENTATION BOX-TYPE TYPE-LABEL-INDENTATION)
  (BOX-BORDERS-FN-SET-BORDER-WID BOX-TYPE BORDER-WID)
  (BOX-BORDERS-FN-SET-BORDER-SPA BOX-TYPE BORDER-SPA)
  (BOX-BORDERS-FN-SET-NAME-BORDER-WID BOX-TYPE NAME-BORDER-WID)
  (BOX-BORDERS-FN-SET-NAME-BORDER-SPA BOX-TYPE NAME-BORDER-SPA)
  (BOX-BORDERS-FN-SET-NAME-HIGHLIGHT  BOX-TYPE NAME-HIGHLIGHT))

(DECLARE-BOX-BORDERS-FN-PARAMETERS :DOIT-BOX "")
(DECLARE-BOX-BORDERS-FN-PARAMETERS :DATA-BOX "Data")
(DECLARE-BOX-BORDERS-FN-PARAMETERS :LL-BOX "Local Library")
(DECLARE-BOX-BORDERS-FN-PARAMETERS :PORT-BOX "Port")
(DECLARE-BOX-BORDERS-FN-PARAMETERS :GRAPHICS-BOX "Graphics")
(DECLARE-BOX-BORDERS-FN-PARAMETERS :GRAPHICS-DATA-BOX "Graphics Data")
(DECLARE-BOX-BORDERS-FN-PARAMETERS :SPRITE-BOX "Sprite")
(DECLARE-BOX-BORDERS-FN-PARAMETERS :INPUT-BOX "Input")

(DEFSELECT (BOX-BORDERS-FN)
  (:MINIMUM-SIZE . BOX-BORDERS-FN-MINIMUM-SIZE)
  (:BORDER-WIDS . BOX-BORDERS-FN-BORDER-WIDS)
  (:DRAW . BOX-BORDERS-FN-DRAW)
  (:CHANGE-SIZE . BOX-BORDERS-FN-CHANGE-SIZE)
  (:CHANGE-SIZE-PASS-1 . BOX-BORDERS-FN-CHANGE-SIZE-PASS-1)
  (:CHANGE-SIZE-PASS-2 . BOX-BORDERS-FN-CHANGE-SIZE-PASS-2)
  (:CHANGE-NAME-PASS-1 . BOX-BORDERS-FN-CHANGE-NAME-PASS-1)
  (:CHANGE-NAME-PASS-2 . BOX-BORDERS-FN-CHANGE-NAME-PASS-2)
  (:ZOOM . BOX-BORDERS-FN-ZOOM)
  (:TAB-SIZE . BOX-BORDERS-FN-NAME-TAB-SIZE)
  (:TAB-SPACE . BOX-BORDERS-FN-NAME-TAB-SPACE)
  (:TAB-OFFSETS . BOX-BORDERS-FN-NAME-TAB-OFFSETS))

(DEFUN SCREEN-BOX-BORDERS-FN (OP SCREEN-BOX &REST ARGS)
  (LEXPR-FUNCALL 'BOX-BORDERS-FN
		  OP (TELL (SCREEN-OBJ-ACTUAL-OBJ SCREEN-BOX) :TYPE) SCREEN-BOX ARGS))

(DEFUN BOX-BORDERS-FN-NAME-TAB-SIZE (IGNORE BOX-TYPE SCREEN-BOX)
    (LET ((OUTER-WID 0) (OUTER-HEI 0) (X 0) (Y 0))
      (BOX-BORDERS-FN-BIND-INTERESTING-VALUES
	(IF (NULL SHOW-NAME-ROW)
	    ;; there is no name row so it isn't going to have a size
	    (VALUES 0 0)
	    ;; otherwise the size will be...
	    (BOX-BORDERS-FN-BIND-NAMED-BOX-PARAMETERS (NIL)
	      (VALUES NAME-TAB-WID NAME-TAB-HEI))))))

(DEFUN BOX-BORDERS-FN-NAME-TAB-SPACE (IGNORE BOX-TYPE SCREEN-BOX)
    (LET ((OUTER-WID 0) (OUTER-HEI 0) (X 0) (Y 0))
      (BOX-BORDERS-FN-BIND-INTERESTING-VALUES
	(IF (NULL SHOW-NAME-ROW)
	    ;; there is no name row so it isn't going to have a size
	    (VALUES 0 0)
	    ;; otherwise the size will be...
	    (BOX-BORDERS-FN-BIND-NAMED-BOX-PARAMETERS (NIL)
	      (VALUES (+ NAME-TAB-WID BORDER-SPA)
		      (+ NAME-TAB-HEI BORDER-SPA
			 (// (MAX 0 (- TYPE-LABEL-HEI BORDER-WID)) 2))))))))

(DEFUN BOX-BORDERS-FN-NAME-TAB-OFFSETS (IGNORE BOX-TYPE IGNORE &REST IGNORE)
  (BOX-BORDERS-FN-BIND-CONSTANT-VALUES
    ;; prevent bound but never used errors
    TYPE-LABEL-WID TYPE-LABEL-INDENTATION
    (VALUES (+ BORDER-SPA NAME-BORDER-WID NAME-BORDER-SPA)
	    (+ BORDER-SPA (// (MAX 0 (- TYPE-LABEL-HEI BORDER-WID)) 2)
	       NAME-BORDER-WID NAME-BORDER-SPA))))

(DEFUN BOX-BORDERS-FN-MINIMUM-SIZE (IGNORE BOX-TYPE SCREEN-BOX)
  (LET ((OUTER-WID 0) (OUTER-HEI 0) (X 0) (Y 0))
    (BOX-BORDERS-FN-BIND-INTERESTING-VALUES
      (IF (NULL SHOW-NAME-ROW)
	  ;; There isn't a name row so we compute the box border parameters like we used to
	  (BOX-BORDERS-FN-BIND-UNNAMED-BOX-PARAMETERS 
	    (IF (EQ BOX-TYPE ':PORT-BOX)
		(VALUES (MAX *MINIMUM-BOX-WID*
			     (+ (* 2 *PORT-BOX-BORDER-GAP*)
				(* 2 BORDER-SPA)
				(* 4 BORDER-WID)
				(* 2 TYPE-LABEL-INDENTATION)
				TYPE-LABEL-WID))
			(MAX *MINIMUM-BOX-HEI*
			     (+ (* 2 *PORT-BOX-BORDER-GAP*)
				(* 2 BORDER-SPA)
				(* 4 BORDER-WID)
				(// (MAX 0 (- TYPE-LABEL-HEI BORDER-WID)) 2))))
		(VALUES (MAX *MINIMUM-BOX-WID*
			     (+ (* 2 BORDER-SPA)
				(* 2 BORDER-WID)
				(* 2 TYPE-LABEL-INDENTATION)
				TYPE-LABEL-WID))
			(MAX *MINIMUM-BOX-HEI*
			     (+ (* 2 BORDER-WID)
				(* 2 BORDER-SPA)
				(// (MAX 0 (- TYPE-LABEL-HEI BORDER-WID)) 2))))))
	  ;; Otherwise, we have to deal with the name row's size
	  (BOX-BORDERS-FN-BIND-NAMED-BOX-PARAMETERS (NIL)
	    (IF (EQ BOX-TYPE ':PORT-BOX)
		(VALUES (MAX (+ NAME-TAB-WID *MINIMUM-BOX-WID*)
			     (+ NAME-TAB-WID
				(* 2 *PORT-BOX-BORDER-GAP*)
				(* 2 BORDER-SPA)
				(* 4 BORDER-WID)
				(* 2 TYPE-LABEL-INDENTATION)
				TYPE-LABEL-WID))
			(MAX NAME-TAB-HEI
			     *MINIMUM-BOX-HEI*
			     (+ (* 2 *PORT-BOX-BORDER-GAP*)
				(* 2 BORDER-SPA)
				(* 4 BORDER-WID)
				(// (MAX 0 (- TYPE-LABEL-HEI BORDER-WID)) 2))
			     (+ (* 2 BORDER-SPA)
				NAME-TAB-HEI
				(// (MAX 0 (- TYPE-LABEL-HEI BORDER-WID)) 2))))
		(VALUES (MAX (+ NAME-TAB-WID *MINIMUM-BOX-WID*)
			     (+ NAME-TAB-WID
				(* 2 BORDER-SPA)
				(* 2 BORDER-WID)
				(* 2 TYPE-LABEL-INDENTATION)
				TYPE-LABEL-WID))
			(MAX NAME-TAB-HEI
			     *MINIMUM-BOX-HEI*
			     (+ (* 2 BORDER-WID)
				(* 2 BORDER-SPA)
				(// (MAX 0 (- TYPE-LABEL-HEI BORDER-WID)) 2))
			     (+ (* 2 BORDER-SPA)
				NAME-TAB-HEI
				(// (MAX 0 (- TYPE-LABEL-HEI BORDER-WID)) 2))))))))))

(DEFUN BOX-BORDERS-FN-BORDER-WIDS (IGNORE BOX-TYPE SCREEN-BOX &OPTIONAL (OLD-P NIL))
  (LET ((OUTER-WID 0) (OUTER-HEI 0) (X 0) (Y 0))
    (BOX-BORDERS-FN-BIND-INTERESTING-VALUES
      (IF (NULL SHOW-NAME-ROW)
	  ;; There isn't a name row so we compute the box border parameters like we used to
	  (BOX-BORDERS-FN-BIND-UNNAMED-BOX-PARAMETERS
	    (IF (EQ BOX-TYPE ':PORT-BOX)
		(VALUES  (+ (* 2 *PORT-BOX-BORDER-GAP*) (* 2 BORDER-WID))
			 (+ (* 2 *PORT-BOX-BORDER-GAP*)
			    (MAX BORDER-WID TYPE-LABEL-HEI) BORDER-WID)
			 (+ (* 2 *PORT-BOX-BORDER-GAP*)
			    (* 2 BORDER-WID))
			 (+ (* 2 *PORT-BOX-BORDER-GAP*)
			    (* 2 BORDER-WID)))
		(VALUES  (+ (* 2 BORDER-SPA) BORDER-WID)
			 (+ (* 2 BORDER-SPA) (MAX BORDER-WID TYPE-LABEL-HEI))
			 (+ (* 2 BORDER-SPA) BORDER-WID)
			 (+ (* 2 BORDER-SPA) BORDER-WID))))
	  ;; Otherwise, we have to deal with the name row's size
	  (BOX-BORDERS-FN-BIND-NAMED-BOX-PARAMETERS (OLD-P)
	    (IF (EQ BOX-TYPE ':PORT-BOX)
		(VALUES (+ (* 2 *PORT-BOX-BORDER-GAP*) (* 2 BORDER-WID) NAME-TAB-WID)
			(+ (* 2 *PORT-BOX-BORDER-GAP*)
			   (MAX BORDER-WID TYPE-LABEL-HEI) BORDER-WID)
			(+ (* 2 *PORT-BOX-BORDER-GAP*) (* 2 BORDER-WID))
			(+ (* 2 *PORT-BOX-BORDER-GAP*) (* 2 BORDER-WID)))
		(VALUES (+ (* 2 BORDER-SPA) BORDER-WID NAME-TAB-WID)
			(+ (* 2 BORDER-SPA) (MAX BORDER-WID TYPE-LABEL-HEI))
			(+ (* 2 BORDER-SPA) BORDER-WID)
			(+ (* 2 BORDER-SPA) BORDER-WID))))))))

(DEFUN BOX-BORDERS-FN-DRAW (IGNORE BOX-TYPE SCREEN-BOX OUTER-WID OUTER-HEI X Y
			    &OPTIONAL (OLD-P NIL) (NO-NAME-P NIL) (NO-TAB-P NIL))
  (OR (ZEROP OUTER-WID)
      (ZEROP OUTER-HEI)
      (BOX-BORDERS-FN-BIND-INTERESTING-VALUES
	(IF (AND (OR (NULL SHOW-NAME-ROW) NO-NAME-P)
		 (NOT (AND OLD-P (TELL SCREEN-BOX :NAME))))
	    ;; There isn't a name row so we draw the box borders like we used to
	    (BOX-BORDERS-FN-BIND-UNNAMED-BOX-PARAMETERS
	      (WITH-CLIPPING-INSIDE (X Y OUTER-WID OUTER-HEI)
		(LET ((*DRAW-CLIPPED-CHAS?* NIL))
		  (DRAW-BOX-BORDERS))))
	    ;; Looks like thers IS a name row so we have to do some extra work
	    (BOX-BORDERS-FN-BIND-NAMED-BOX-PARAMETERS (OLD-P)
	      (WITH-CLIPPING-INSIDE (X Y OUTER-WID OUTER-HEI)
		(UNLESS NO-TAB-P
		  (DRAW-SCREEN-ROW-FOR-NAMING)
		  (DRAW-NAME-BORDERS))
		(LET ((*DRAW-CLIPPED-CHAS?* NIL))
		  (DRAW-BOX-BORDERS))))))))

(DEFUN BOX-BORDERS-FN-CHANGE-SIZE (IGNORE BOX-TYPE SCREEN-BOX
				   OLD-WID OLD-HEI NEW-WID NEW-HEI X Y)
  (BOX-BORDERS-FN ':DRAW BOX-TYPE SCREEN-BOX NEW-WID NEW-HEI X Y)
  (BOX-BORDERS-FN ':DRAW BOX-TYPE SCREEN-BOX OLD-WID OLD-HEI X Y T))

(DEFUN BOX-BORDERS-FN-CHANGE-SIZE-PASS-1 (IGNORE BOX-TYPE SCREEN-BOX
					  OLD-WID OLD-HEI IGNORE IGNORE X Y)
  (BOX-BORDERS-FN ':DRAW BOX-TYPE SCREEN-BOX OLD-WID OLD-HEI X Y T NIL T))

(DEFUN BOX-BORDERS-FN-CHANGE-SIZE-PASS-2 (IGNORE BOX-TYPE SCREEN-BOX
					  IGNORE IGNORE NEW-WID NEW-HEI X Y)
  (BOX-BORDERS-FN ':DRAW BOX-TYPE SCREEN-BOX NEW-WID NEW-HEI X Y NIL NIL T))

(DEFUN BOX-BORDERS-FN-CHANGE-NAME-PASS-1 (IGNORE BOX-TYPE SCREEN-BOX
					  OLD-WID OLD-HEI IGNORE IGNORE X Y)
  (IF (NULL (TELL SCREEN-BOX :NAME))
      (BOX-BORDERS-FN ':DRAW BOX-TYPE SCREEN-BOX OLD-WID OLD-HEI X Y T T)
      (BOX-BORDERS-FN ':DRAW BOX-TYPE SCREEN-BOX OLD-WID OLD-HEI X Y T)))

(DEFUN BOX-BORDERS-FN-CHANGE-NAME-PASS-2 (IGNORE BOX-TYPE SCREEN-BOX
					  IGNORE IGNORE NEW-WID NEW-HEI X Y)
  (IF (NULL (TELL (TELL SCREEN-BOX :ACTUAL-OBJ) :NAME-ROW))
      (BOX-BORDERS-FN ':DRAW BOX-TYPE SCREEN-BOX NEW-WID NEW-HEI X Y NIL T)
      (BOX-BORDERS-FN ':DRAW BOX-TYPE SCREEN-BOX NEW-WID NEW-HEI X Y)))

(DEFUN BOX-BORDERS-FN-ZOOM (IGNORE BOX-TYPE SCREEN-BOX
			    START-WID START-HEI END-WID END-HEI
			    START-X START-Y END-X END-Y STEPS)
  (HACKS:WITH-REAL-TIME
    (MULTIPLE-VALUE-BIND (MIN-WID MIN-HEI)
	(BOX-BORDERS-FN ':MINIMUM-SIZE BOX-TYPE SCREEN-BOX)
      (MAXIMIZE START-WID MIN-WID)
      (MAXIMIZE START-HEI MIN-HEI)
      (MAXIMIZE END-WID MIN-WID)
      (MAXIMIZE END-HEI MIN-HEI)
      (LET* ((WID-INCREMENT (// (- END-WID START-WID) STEPS))
	     (HEI-INCREMENT (// (- END-HEI START-HEI) STEPS))
	     (X-INCREMENT (// (- END-X START-X) STEPS))
	     (Y-INCREMENT (// (- END-Y START-Y) STEPS)))
	(DO ((I 0 (+ I 1))
	     (WID START-WID (+ WID WID-INCREMENT))
	     (HEI START-HEI (+ HEI HEI-INCREMENT))
	     (X START-X (+ X X-INCREMENT))
	     (Y START-Y (+ Y Y-INCREMENT)))
	    ((>= I STEPS))
	  (BOX-BORDERS-FN ':DRAW BOX-TYPE SCREEN-BOX WID HEI X Y NIL NIL T)
	  (PROCESS-SLEEP *BOX-ZOOM-WAITING-TIME* "ZooM")
	  (BOX-BORDERS-FN ':DRAW BOX-TYPE SCREEN-BOX WID HEI X Y NIL NIL T))))))



;;; circular structure support

(DEFUN PORT-HAS-BEEN-DISPLAYED-ENOUGH? (PORT)
  (LET ((ENTRY (CDR (ASSQ PORT PORT-REDISPLAY-HISTORY))))
    (AND ENTRY ( ENTRY *PORT-REDISPLAY-DEPTH*))))

(DEFUN UPDATE-PORT-REDISPLAY-HISTORY (PORT)
  (LET ((ENTRY (ASSQ PORT PORT-REDISPLAY-HISTORY)))
    (IF (NULL ENTRY) (APPEND PORT-REDISPLAY-HISTORY (NCONS (CONS PORT 1)))
	(LET ((NEW-HISTORY (COPYLIST PORT-REDISPLAY-HISTORY)))
	  (SETF (CDR (ASSQ PORT NEW-HISTORY)) (1+ (CDR ENTRY)))
	  NEW-HISTORY))))

;;; some styles...

(DEFINE-BOX-ELLIPSIS-STYLE BOX-ELLIPSIS-SOLID-LINES)

(DEFUN (:PROPERTY BOX-ELLIPSIS-SOLID-LINES DRAW-SELF) (X-COORD Y-COORD)
  (LOOP FOR X FROM X-COORD TO (+ X-COORD (// *BOX-ELLIPSIS-WID* 2.))
	      BY (+ *BOX-ELLIPSIS-THICKNESS* *BOX-ELLIPSIS-SPACING*)
	FOR Y FROM Y-COORD TO (+ Y-COORD (// *BOX-ELLIPSIS-WID* 2.))
	      BY (+ *BOX-ELLIPSIS-THICKNESS* *BOX-ELLIPSIS-SPACING*)
	FOR WID FROM (- *BOX-ELLIPSIS-WID* (* 2 *BOX-ELLIPSIS-THICKNESS*)) DOWNTO 0
		BY (* 2 (+ *BOX-ELLIPSIS-THICKNESS* *BOX-ELLIPSIS-SPACING*))
	FOR HEI FROM *BOX-ELLIPSIS-HEI* DOWNTO 0
		BY (* 2 (+ *BOX-ELLIPSIS-THICKNESS* *BOX-ELLIPSIS-SPACING*))
	DO (DRAW-RECTANGLE TV:ALU-XOR *BOX-ELLIPSIS-THICKNESS* HEI X Y)
	   (DRAW-RECTANGLE TV:ALU-XOR WID *BOX-ELLIPSIS-THICKNESS*
			   (+ X *BOX-ELLIPSIS-THICKNESS*) Y)
	   (DRAW-RECTANGLE TV:ALU-XOR *BOX-ELLIPSIS-THICKNESS* HEI
			   (+ X WID *BOX-ELLIPSIS-THICKNESS*) Y)
	   (DRAW-RECTANGLE TV:ALU-XOR WID *BOX-ELLIPSIS-THICKNESS*
			   (+ X *BOX-ELLIPSIS-THICKNESS*)
			   (+ Y HEI (- *BOX-ELLIPSIS-THICKNESS*)))))

(DEFINE-BOX-ELLIPSIS-STYLE BOX-ELLIPSIS-CORNER-DOTS)

(DEFUN (:PROPERTY BOX-ELLIPSIS-CORNER-DOTS DRAW-SELF) (X-COORD Y-COORD)
  (LOOP FOR X FROM X-COORD TO (+ X-COORD (// *BOX-ELLIPSIS-WID* 2.))
	      BY (+ *BOX-ELLIPSIS-THICKNESS* *BOX-ELLIPSIS-SPACING*)
	FOR Y FROM Y-COORD TO (+ Y-COORD (// *BOX-ELLIPSIS-WID* 2.))
	      BY (+ *BOX-ELLIPSIS-THICKNESS* *BOX-ELLIPSIS-SPACING*)
	FOR WID FROM (- *BOX-ELLIPSIS-WID* (* 2 *BOX-ELLIPSIS-THICKNESS*)) DOWNTO 0
		BY (* 2 (+ *BOX-ELLIPSIS-THICKNESS* *BOX-ELLIPSIS-SPACING*))
	FOR HEI FROM *BOX-ELLIPSIS-HEI* DOWNTO 0
		BY (* 2 (+ *BOX-ELLIPSIS-THICKNESS* *BOX-ELLIPSIS-SPACING*))
	DO (DRAW-RECTANGLE TV:ALU-XOR *BOX-ELLIPSIS-THICKNESS* *BOX-ELLIPSIS-THICKNESS* X Y)
	   (DRAW-RECTANGLE TV:ALU-XOR *BOX-ELLIPSIS-THICKNESS* *BOX-ELLIPSIS-THICKNESS*
			   (+ X WID *BOX-ELLIPSIS-THICKNESS*) Y)
	   (DRAW-RECTANGLE TV:ALU-XOR *BOX-ELLIPSIS-THICKNESS* *BOX-ELLIPSIS-THICKNESS*
			   X (+ Y HEI (- *BOX-ELLIPSIS-THICKNESS*)))
	   (DRAW-RECTANGLE TV:ALU-XOR *BOX-ELLIPSIS-THICKNESS* *BOX-ELLIPSIS-THICKNESS*
			   (+ X WID *BOX-ELLIPSIS-THICKNESS*)
			   (+ Y HEI (- *BOX-ELLIPSIS-THICKNESS*)))))


;;; Region marking stuff OBSOLETE !!!!!!
;;; These1 will only0 work for regions that are1 single0 screen rows

;;; updated versions. keep around for old code to use
(DEFUN MARK-ROW (ROW)
  (LET ((START-BP (MAKE-BP ':FIXED))
	(STOP-BP (MAKE-BP ':FIXED)))
    (SET-BP-ROW START-BP ROW)
    (SET-BP-CHA-NO START-BP 0)
    (SET-BP-ROW STOP-BP ROW)
    (SET-BP-CHA-NO STOP-BP (TELL ROW :LENGTH-IN-CHAS))
    (let ((region (MAKE-EDITOR-REGION START-BP STOP-BP)))
      (TELL REGION :TURN-ON)
      (PUSH REGION REGION-LIST))))

(DEFUN UNMARK-ROW (ROW)
  ;; first find the region
  (let ((region (mem #'(lambda (x y) (eq x (car(tell y :get-rows-from-bps))))
		     row region-list)))
    (when (not-null region)
      (flush-region (car region)))))

;(DEFVAR REGIONS NIL)
;
;(DEFVAR *CURRENT-SCREEN-REGION* NIL
;  "The screen structure corresponding to the current region. ")
;
;(DEFFLAVOR REGION
;	((UID NIL))
;	(TV:RECTANGULAR-BLINKER FLAVOR-HACKING-MIXIN)
;  (:OUTSIDE-ACCESSIBLE-INSTANCE-VARIABLES UID))
;
;(DEFTYPE-CHECKING-MACROS REGION "A Boxer Editor Region Blinker")
;
;(DEFMETHOD (REGION :BLINK) ()
;  (DRAWING-ON-WINDOW-WITHOUT-PREPARE-SHEET (TV:SHEET)
;    (DRAW-RECTANGLE TV:ALU-XOR TV:WIDTH TV:HEIGHT TV:X-POS TV:Y-POS)))
;
;(DEFUN MAKE-REGION (UID WINDOW)
;  (LET ((NEW-REGION (TV:MAKE-BLINKER WINDOW 'REGION ':VISIBILITY NIL ':FOLLOW-P NIL)))
;    (SETF (REGION-UID NEW-REGION) UID)
;    (PUSH NEW-REGION REGIONS)
;    NEW-REGION))
;
;(DEFUN MARK-REGION (WINDOW UID VISIBILITY &OPTIONAL WID HEI X Y)
;  (LET ((REGION (OR (CAR (MEM #'(LAMBDA (UID REG) (EQ UID (REGION-UID REG))) UID REGIONS))
;		    (MAKE-REGION UID WINDOW))))
;    (ALTERING-REGION (REGION)
;      (IF WID (SETF (REGION-WID REGION) WID))
;      (IF HEI (SETF (REGION-HEI REGION) HEI))
;      (IF X   (SETF (REGION-X REGION) X))
;      (IF Y   (SETF (REGION-Y REGION) Y))
;      (SETF (REGION-VISIBILITY REGION) VISIBILITY))))

;(DEFUN MARK-CURRENT-REGION (ROW)
;  (COND ((NULL ROW)
;	 (UNMARK-SCREEN-ROW *CURRENT-SCREEN-REGION*)
;	 (SETQ *CURRENT-SCREEN-REGION* NIL))
;	(T
;	 (LET ((SCREEN-ROW (CURRENT-SCREEN-ROW ROW)))
;	   (MULTIPLE-VALUE-BIND (X Y)
;	       (TELL SCREEN-ROW :POSITION)
;	     (LET ((WID (SCREEN-OBJ-WID SCREEN-ROW))
;		   (HEI (SCREEN-OBJ-HEI SCREEN-ROW)))
;	       (MARK-REGION *BOXER-PANE* SCREEN-ROW T WID HEI X Y)))
;	   (SETQ *CURRENT-SCREEN-REGION* SCREEN-ROW)))))
;
;(DEFUN MARK-SCREEN-ROW (SCREEN-ROW)
;  (CHECK-SCREEN-ROW-ARG SCREEN-ROW)
;  (MULTIPLE-VALUE-BIND (X Y)
;      (TELL SCREEN-ROW :POSITION)
;    (LET ((WID (SCREEN-OBJ-WID SCREEN-ROW))
;	  (HEI (SCREEN-OBJ-HEI SCREEN-ROW)))
;      (MARK-REGION *BOXER-PANE* SCREEN-ROW T WID HEI X Y))))
;
;(DEFUN FIND-ROW-BLINKER (SCREEN-ROW LIST-OF-REGIONS)
;  (DOLIST (REGION LIST-OF-REGIONS)
;    (WHEN (REGION? REGION)
;      (WHEN (EQ SCREEN-ROW (REGION-UID REGION))
;	(RETURN REGION)))))
;        
;(DEFUN REMOVE-ROW-BLINKER (SCREEN-ROW LIST-OF-REGIONS)
;  (DELQ (FIND-ROW-BLINKER SCREEN-ROW LIST-OF-REGIONS)
;       LIST-OF-REGIONS))
;
;(DEFUN UNMARK-SCREEN-ROW (SCREEN-ROW)
;  (tell (find-row-blinker screen-row regions) :set-visibility nil)
;  (SETF REGIONS (REMOVE-ROW-BLINKER SCREEN-ROW REGIONS))
;  (SETF (TV:SHEET-BLINKER-LIST *BOXER-PANE*)
;	(REMOVE-ROW-BLINKER SCREEN-ROW
;			    (TV:SHEET-BLINKER-LIST *BOXER-PANE*))))
;
;(DEFMETHOD (REGION :UPDATE) ()
;  (IF (TELL UID :VISIBLE?)
;      (UPDATE-ROW-BLINKER SELF UID)
;      ;(SHRINK-ROW-BLINKER SELF)
;      (UNMARK-SCREEN-ROW UID)))
;
;(DEFUN UPDATE-ROW-BLINKER (OLD-REGION SCREEN-ROW)
;  (ALTERING-REGION (OLD-REGION)
;    (MULTIPLE-VALUE-BIND (X Y)
;	(TELL SCREEN-ROW :POSITION)
;      (LET ((WID (SCREEN-OBJ-WID SCREEN-ROW))
;	    (HEI (SCREEN-OBJ-HEI SCREEN-ROW)))
;	(SETF (REGION-WID OLD-REGION) WID)
;	(SETF (REGION-HEI OLD-REGION) HEI)
;	(SETF (REGION-X OLD-REGION) X)
;	(SETF (REGION-Y OLD-REGION) Y)))))
;
;(DEFUN SHRINK-ROW-BLINKER (REGION)
;  (ALTERING-REGION (REGION)
;    (SETF (REGION-WID REGION) 0)
;    (SETF (REGION-HEI REGION) 0)
;    (SETF (REGION-VISIBILITY REGION) NIL)))

;(DEFUN UPDATE-ACTUAL-ROW-BLINKER (ROW)
;  (UPDATE-ROW-BLINKER (CAR (TELL ROW :DISPLAYED-SCREEN-OBJS))))
;
;(DEFUN UPDATE-REGION (REGION)
;  (CHECK-REGION-ARG REGION)
;  (TELL REGION :UPDATE))
;
;(DEFUN UNMARK-ALL-ROWS ()			;this does not unmark the *current-region*
;  (DOLIST (REGION REGIONS)
;    (UNLESS (EQ (REGION-UID REGION) *CURRENT-SCREEN-REGION*)
;      (UNMARK-SCREEN-ROW (REGION-UID REGION)))))
