;; -*- Mode:LISP; Package:(BOXER GLOBAL 1000); Base:8. ; Fonts:cptfont, cptfontb -*-

#|
            Copyright 1985 Massachusetts Institute of Technology

 Permission to use, copy, modify, distribute, and sell this software
 and its documentation for any purpose is hereby granted without fee,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of M.I.T. not be used in
 advertising or publicity pertaining to distribution of the software
 without specific, written prior permission.  M.I.T. makes no
 representations about the suitability of this software for any
 purpose.  It is provided "as is" without express or implied warranty.


                                         +-Data--+
                This file is part of the | BOXER | system
                                         +-------+


 This file contains top level definitions for BOXER Editor Commands 


|#

;;;; The basics

(DEFBOXER-COMMAND COM-ABORT ()
  "aborts any editing in progress.  flushes
numeric arguments and removes the current
region. "
  ;; if there is a region, get rid of it
  (LET ((REGION-TO-FLUSH (OR *REGION-BEING-DEFINED* (GET-CURRENT-REGION))))
    (UNLESS (NULL REGION-TO-FLUSH)
      (FLUSH-REGION REGION-TO-FLUSH)))
  (BOXER-EDITOR-ERROR "Editor Top Level")
  (WITH-MULTIPLE-EXECUTION			;this is here so that numeric args are flushed
    (*THROW 'BOXER-EDITOR-TOP-LEVEL NIL)))

(DEFBOXER-COMMAND COM-INCREMENT-NUMERIC-ARG ()
  "specifies part of the next command's numeric argument. "
  (IF *EDITOR-NUMERIC-ARGUMENT*
      (SET-EDITOR-NUMERIC-ARG (+ (* 10. *EDITOR-NUMERIC-ARGUMENT*)
					 (NUMBER-CODE BU:*KEY-CODE-BEING-HANDLED*)))
      (SET-EDITOR-NUMERIC-ARG (NUMBER-CODE BU:*KEY-CODE-BEING-HANDLED*))))

;This uses only the global value of bu:*key-code-being-handled*.
;You can't bind it from boxer.
(DEFBOXER-COMMAND COM-SELF-INSERT ()
  "inserts the last character typed.
with a numeric argument (n), inserts
the character n times. "
  (WITH-MULTIPLE-EXECUTION
    (INSERT-CHA *POINT*			   
		(MAKE-CHA BU:*KEY-CODE-BEING-HANDLED*)
		':MOVING)))

(DEFBOXER-COMMAND COM-QUOTE-SELF-INSERT ()
  "inserts any keyboard character.
with a numeric argument, inserts that
many copies of the character. "
  (LET ((BU:*KEY-CODE-BEING-HANDLED* (TELL TERMINAL-IO :TYI)))
    (COM-SELF-INSERT)))

(DEFBOXER-COMMAND COM-SPACE ()
  "inserts a space.  with a numeric
argument (n), inserts n spaces. "
  (WITH-MULTIPLE-EXECUTION
    (INSERT-CHA *POINT*			   
		(MAKE-CHA BU:*KEY-CODE-BEING-HANDLED*)
		':MOVING)))

(DEFBOXER-COMMAND COM-RETURN ()
  "inserts a new line into the buffer
at the cursor location.  with a numeric
argument (n), inserts n new lines. When 
in the name portion of a box, enters the
box itself. "
    (COND ((NAME-ROW? (POINT-ROW))
	   (COM-EXIT-BOX)
	   (COM-BACKWARD-CHA)
	   (COM-ENTER-BOX))
	  (T 
	   (WITH-MULTIPLE-EXECUTION
	     (INSERT-ROW *POINT* (MAKE-INITIALIZED-ROW) ':MOVING))))
    (SETQ *COLUMN* 0))

(DEFBOXER-COMMAND COM-OPEN-LINE ()
  "inserts a blank line after the cursor.
with a numeric arg (n), inserts n blank lines. "
  (WITH-MULTIPLE-EXECUTION
    (INSERT-ROW *POINT* (MAKE-INITIALIZED-ROW) ':MOVING)
    (MOVE-POINT (BP-BACKWARD-CHA-VALUES *POINT*))))



;;;; Single Character Commands

(DEFBOXER-COMMAND COM-RUBOUT ()
  "Rubs out one character.  with numeric
argument (n), rubs out n characters. "
  (WITH-MULTIPLE-EXECUTION
    (LET ((DELETED-CHA (RUBOUT-CHA *POINT* ':MOVING)))
      (kill-buffer-push deleted-cha ':BACKWARD)
      (SETQ *COLUMN* (BP-CHA-NO *POINT*)))))

(DEFBOXER-COMMAND COM-DELETE ()
  "deletes one character.  with numeric
argument (n), delete n characters. "
  (WITH-MULTIPLE-EXECUTION
    (LET ((OLD-ROW (BP-ROW *POINT*))
	  (OLD-CHA-NO (BP-CHA-NO *POINT*)))
      (MOVE-POINT (BP-FORWARD-CHA-VALUES *POINT*))
      (IF (OR (NEQ OLD-ROW (BP-ROW *POINT*))
	      (NEQ OLD-CHA-NO (BP-CHA-NO *POINT*)))
	  (kill-buffer-push 
	    (RUBOUT-CHA *POINT* ':MOVING)
	    ':forward)))))

(DEFBOXER-COMMAND COM-FORWARD-CHA ()
  "moves forward one character.  with
numeric argument (n), moves forward
n characters. "
  (WITH-MULTIPLE-EXECUTION
    (MOVE-POINT (BP-FORWARD-CHA-VALUES *POINT*))))

(DEFBOXER-COMMAND COM-BACKWARD-CHA ()
  "moves backward one character.  with
numeric argument (n), moves backward
n characters. "
  (WITH-MULTIPLE-EXECUTION
    (MOVE-POINT (BP-BACKWARD-CHA-VALUES *POINT*))))



;;;; Cursor Movement

(DEFBOXER-COMMAND COM-BEGINNING-OF-ROW ()
  "moves to the beginning of the row. "
   (MOVE-POINT (ROW-FIRST-BP-VALUES (BP-ROW *POINT*))))

(DEFBOXER-COMMAND COM-END-OF-ROW ()
  "moves to the end of the row. "
  (MOVE-POINT (ROW-LAST-BP-VALUES (BP-ROW *POINT*))))

(DEFBOXER-COMMAND COM-BEGINNING-OF-BOX ()
  "moves to the beginning of the box. "
  (MOVE-POINT (BOX-FIRST-BP-VALUES (BOX-POINT-IS-IN)))
  (dolist (screen-row (tell (box-point-is-in) :screen-objs))
    (tell screen-row :set-scroll-to-actual-row (tell (box-point-is-in) :first-inferior-row))))

(DEFBOXER-COMMAND COM-END-OF-BOX ()
  "moves to the end of the box. "
  (MOVE-POINT (BOX-LAST-BP-VALUES (BOX-POINT-IS-IN)))
  (ENSURE-ROW-IS-DISPLAYED (POINT-ROW) (POINT-SCREEN-BOX)))

(DEFBOXER-COMMAND COM-PREVIOUS-ROW ()
  "moves up vertically to the previous
row.  With numeric argument (n), moves
up n rows.  Tries to stay as close as
possible to the original column. "
  (WITH-MULTIPLE-EXECUTION
    (LET* ((ROW (BP-ROW *POINT*))
	   (PREVIOUS-ROW (TELL-CHECK-NIL ROW :PREVIOUS-ROW))
	   (PREVIOUS-ROW-LENGTH-IN-CHAS
	     (TELL-CHECK-NIL PREVIOUS-ROW :LENGTH-IN-CHAS))
	   (CHA-NO (BP-CHA-NO *POINT*))
	   (CURRENT-ROW-LENGTH-IN-CHAS (TELL ROW :LENGTH-IN-CHAS)))
      (COND ((NULL PREVIOUS-ROW))
	    ((< CHA-NO CURRENT-ROW-LENGTH-IN-CHAS)
	     (SETQ *COLUMN* CHA-NO)
	     (MOVE-POINT-1 PREVIOUS-ROW
			   (MIN PREVIOUS-ROW-LENGTH-IN-CHAS *COLUMN*)))
	    ((< *COLUMN* CHA-NO)
	     (SETQ *COLUMN* CHA-NO)
	     (MOVE-POINT-1 PREVIOUS-ROW
			   (MIN PREVIOUS-ROW-LENGTH-IN-CHAS *COLUMN*)))
	    (T
	     (MOVE-POINT-1 PREVIOUS-ROW
			   (MIN PREVIOUS-ROW-LENGTH-IN-CHAS *COLUMN*))))
      (ENSURE-ROW-IS-DISPLAYED (POINT-ROW) (POINT-SCREEN-BOX)))))


;; this one goes to the name row if it's there
(DEFBOXER-COMMAND COM-PREVIOUS-ROW-OR-NAME ()
  "moves up vertically to the previous
row.  With numeric argument (n), moves
up n rows.  Tries to stay as close as
possible to the original column. "
  (WITH-MULTIPLE-EXECUTION
    (LET* ((ROW (BP-ROW *POINT*))
	   (PREVIOUS-ROW (TELL-CHECK-NIL ROW :PREVIOUS-ROW))
	   (PREVIOUS-ROW-LENGTH-IN-CHAS
	     (TELL-CHECK-NIL PREVIOUS-ROW :LENGTH-IN-CHAS))
	   (CHA-NO (BP-CHA-NO *POINT*))
	   (CURRENT-ROW-LENGTH-IN-CHAS (TELL ROW :LENGTH-IN-CHAS)))
      (COND ((NULL PREVIOUS-ROW)
	     (COM-NAME-BOX))
	    ((< CHA-NO CURRENT-ROW-LENGTH-IN-CHAS)
	     (SETQ *COLUMN* CHA-NO)
	     (MOVE-POINT-1 PREVIOUS-ROW
			   (MIN PREVIOUS-ROW-LENGTH-IN-CHAS *COLUMN*)))
	    ((< *COLUMN* CHA-NO)
	     (SETQ *COLUMN* CHA-NO)
	     (MOVE-POINT-1 PREVIOUS-ROW
			   (MIN PREVIOUS-ROW-LENGTH-IN-CHAS *COLUMN*)))
	    (T
	     (MOVE-POINT-1 PREVIOUS-ROW
			   (MIN PREVIOUS-ROW-LENGTH-IN-CHAS *COLUMN*))))
      (ENSURE-ROW-IS-DISPLAYED (POINT-ROW) (POINT-SCREEN-BOX)))))
  


(DEFBOXER-COMMAND COM-NEXT-ROW ()
    "moves up vertically down the next
row.  With numeric argument (n), moves
down n rows.  Tries to stay as close as
possible to the original column. "
  (WITH-MULTIPLE-EXECUTION
    (LET* ((ROW (BP-ROW *POINT*))
	   (NEXT-ROW (TELL-CHECK-NIL ROW :NEXT-ROW))
	   (NEXT-ROW-LENGTH-IN-CHAS (TELL-CHECK-NIL NEXT-ROW :LENGTH-IN-CHAS))
	   (CHA-NO (BP-CHA-NO *POINT*))
	   (CURRENT-ROW-LENGTH-IN-CHAS (TELL ROW :LENGTH-IN-CHAS)))
      (COND ((NULL NEXT-ROW) 
	     (COM-END-OF-ROW)
	     (COM-RETURN))
	    ((< CHA-NO CURRENT-ROW-LENGTH-IN-CHAS)
	     (SETQ *COLUMN* CHA-NO)
	     (MOVE-POINT-1 NEXT-ROW (MIN NEXT-ROW-LENGTH-IN-CHAS *COLUMN*))
	     (ENSURE-ROW-IS-DISPLAYED (POINT-ROW) (POINT-SCREEN-BOX) 1))
	    ((< *COLUMN* CHA-NO)
	     (SETQ *COLUMN* CHA-NO)
	     (MOVE-POINT-1 NEXT-ROW (MIN NEXT-ROW-LENGTH-IN-CHAS *COLUMN*))
	     (ENSURE-ROW-IS-DISPLAYED (POINT-ROW) (POINT-SCREEN-BOX) 1))
	    (T
	     (MOVE-POINT-1 NEXT-ROW (MIN NEXT-ROW-LENGTH-IN-CHAS *COLUMN*))
	     (ENSURE-ROW-IS-DISPLAYED (POINT-ROW) (POINT-SCREEN-BOX) 1))))))



;;; Generalized movement
;;; Move the POINT to another (possibly non-local) location specified by BP
;;; This function performs all of the neccessary zooms, expands and scrolls
;;; so that the user has some idea of where he is going

(DEFUN MOVE-TO-BP (BP &OPTIONAL (MOVING-BP *POINT*))
  (IF (TELL (BP-BOX BP) :SUPERIOR? (BP-BOX MOVING-BP))
      (DOWNWARD-MOVE-TO-BP BP MOVING-BP)
      ;; looks like we are going to have to go up before we can go down
      (UPWARD-MOVE-TO-COMMON-BOX BP MOVING-BP)
      (DOWNWARD-MOVE-TO-BP BP MOVING-BP)))

;;; Move upward until we reach a place where BP is in some inferior of (POINT-BOX)
;;; We have to march up the screen structure rather than the actual structure because
;;; we might be inside of a port

(DEFUN UPWARD-MOVE-TO-COMMON-BOX (BP &OPTIONAL (MOVING-BP *POINT*))
  (LET  ((BOX (BP-BOX MOVING-BP)))
    (COND ((TELL (BP-BOX BP) :SUPERIOR? BOX))	;we have arrived
	  (T (UNLESS (EQ BOX *INITIAL-BOX*)
	       (TELL BOX :EXIT (TELL (BP-SCREEN-BOX MOVING-BP) :SUPERIOR-SCREEN-BOX)
		     (TELL BOX :SUPERIOR-BOX) T))
	     (UPWARD-MOVE-TO-COMMON-BOX BP MOVING-BP)))))

;;; The destination is in some inferior of the current box
(DEFUN DOWNWARD-MOVE-TO-BP (BP &OPTIONAL (MOVING-BP *POINT*))
  (LET* ((ROW (BP-ROW BP))
	 (OLD-ROW (BP-ROW MOVING-BP))
	 (SCREEN-ROW (VISIBLE-SCREEN-OBJ-OF-INFERIOR-ACTUAL-OBJ ROW
								(BP-SCREEN-BOX MOVING-BP))))
    (COND ((NOT-NULL SCREEN-ROW)
	   ;; the destination is visible already
	   (MOVE-BP MOVING-BP (BP-VALUES BP))
	   (SET-BP-SCREEN-BOX MOVING-BP (TELL SCREEN-ROW :SCREEN-BOX)))
	  ((MEMQ ROW (TELL (BP-BOX MOVING-BP) :ROWS))
	   ;; the destination is in the current box but is scrolled out of sight
	   (MOVE-BP MOVING-BP (BP-VALUES BP))
	   (ENSURE-ROW-IS-DISPLAYED (BP-ROW MOVING-BP) (BP-SCREEN-BOX MOVING-BP)
				    (IF (ROW-> ROW OLD-ROW) 1 -1)))
	  (T
	   (LET* ((PATH (FIND-PATH-FROM-SUPERIOR-TO-INFERIOR (BP-BOX MOVING-BP)
							     (BP-BOX BP)))
		  (NEW-BOX (LOWEST-VISIBLE-BOX (BP-SCREEN-BOX MOVING-BP) PATH)))
	     (COND ((NULL PATH)
		    (EDITOR-BARF "The BP, ~A, is not in an inferior of ~A" BP
				 (BP-BOX MOVING-BP)))
		   ((NULL NEW-BOX)
		    ;; the downward chain of boxes is not visible probably because
		    ;; we are scrolled to the wrong place in the current screen box
		    ;; so we scroll to the correct row, then try again
		    (MOVE-BP MOVING-BP (BOX-SELF-BP-VALUES (CAR PATH)))
		    (ENSURE-ROW-IS-DISPLAYED (BP-ROW MOVING-BP) (BP-SCREEN-BOX MOVING-BP)
					     (IF (ROW-> (TELL (CAR PATH) :SUPERIOR-ROW)
							OLD-ROW)
						 1 -1))
		    (DOWNWARD-MOVE-TO-BP BP MOVING-BP))
		   (T
		    ;; move to lowest visible box, zoom, then try again
		    (MOVE-BP MOVING-BP (BOX-FIRST-BP-VALUES NEW-BOX))
		    (SET-BP-SCREEN-BOX MOVING-BP
		      (VISIBLE-SCREEN-OBJ-OF-INFERIOR-ACTUAL-OBJ NEW-BOX
							         (BP-SCREEN-BOX MOVING-BP)))
		      (WHEN (OR (GRAPHICS-BOX? NEW-BOX)
				(AND (PORT-BOX? NEW-BOX)
				     (GRAPHICS-BOX? (TELL NEW-BOX :PORTS))))
			;; The chain is in an inferior of a GRAPHICS/GRAPHICS-DATA-BOX
			;; which is currently in GRAPHICS mode so we have to toggle it
			;; before we can zoom it up
			(TELL NEW-BOX :TOGGLE-TYPE))
		      ;; now we can zoom
		      (PUSH *OUTERMOST-SCREEN-BOX* *OUTERMOST-SCREEN-BOX-STACK*)
		      (LET ((*BOX-ZOOM-WAITING-TIME* (* *BOX-ZOOM-WAITING-TIME* 2)))
			;; slow things down a bit
			(SET-OUTERMOST-BOX (BP-BOX MOVING-BP) (BP-SCREEN-BOX MOVING-BP)))
		      ;; then try again
		      (DOWNWARD-MOVE-TO-BP BP MOVING-BP))))))))

(DEFBOXER-COMMAND COM-MOVE-TO-BP (BP)
  "Moves the cursor to the place specified by BP"
  (MOVE-TO-BP BP))

;;;; More movement.  This is primarily for moving to/from port targets although it
;;; may turn out to be more useful as it has a better idea of what is happening than the
;;; functions above

;;; This allows you to specify a path of boxes from the top of the world.  This is needed
;;; for moving to points that are within a port.  Note that we can't use screen structure 
;;; because it may not exist for arbitrary points in the hierarchy

;;; Paths are organized from the top to the bottom

;; this is like a BP except that instead of a screen box, it relies on a absolute path from
;; the top of the editor hierarchy.  The reason for this is that screen structure can be
;; reclaimed.  This is the ONLY reliable way of maintaining a position in the boxer hierarchy
;; independent of the location of the *point*

(defvar *port-zooming-bread-crumbs* nil
  "as we zoom throught ports to their targets, we leave a trail of where we;ve been.")

(defvar *port-zooming-pause-time* .5)
(defvar *port-zooming-slowdown-factor* 2)

(defstruct (absolute-boxer-pointer :named 
				   (:conc-name abp-)
				   (:predicate abp?)
				   (:constructor %make-abp (row cha-no path)))
  (row nil)
  (cha-no 0)
  (path nil))

(DEFUN GET-PATH (BP &optional real-structure?)
  (IF (or real-structure? (NOT (SCREEN-BOX? (BP-SCREEN-BOX BP))))
      ;; either we march up the editor object hierarchy or else
      (nreverse 
	(WITH-COLLECTION
	  (DO ((BOX (BP-BOX BP) (TELL BOX :SUPERIOR-BOX)))
	      ((NOT (BOX? BOX)))
	    (COLLECT BOX))))
      ;; we walk up the screen hierarchy
      (nreverse
	(WITH-COLLECTION
	  (DO ((SBOX (BP-SCREEN-BOX BP) (TELL SBOX :SUPERIOR-SCREEN-BOX)))
	      ((NOT (SCREEN-BOX? SBOX)))
	    (COLLECT (TELL SBOX :ACTUAL-OBJ)))))))

(defun make-abp-from-bp (bp)
  (%make-abp (bp-row bp) (bp-cha-no bp) (get-path bp)))

(defun abp= (abp1 abp2)
  (and (abp? abp1)
       (abp? abp2)
       (equal (abp-path abp1) (abp-path abp2))
       (eq (abp-row abp1) (abp-row abp2))
       (= (abp-cha-no abp1) (abp-cha-no abp2))))

(defun move-point-along-path (row cha-no path)
  (let ((*box-zoom-waiting-time* (* *box-zoom-waiting-time* *port-zooming-slowdown-factor*)))
    ;; first move up to a common superior box
    (do ((box (box-screen-point-is-in) (box-screen-point-is-in)))
	((or (memq box path)
	     (eq box *initial-box*)))
      (tell box :exit (tell (point-screen-box) :superior-screen-box)
	    (tell box :superior-box) t))
    ;; now walk down the remainder of the path
    (dolist (box (cdr (memq (box-screen-point-is-in) path)))
      (let ((old-row (point-row))
	    (old-screen-box (point-screen-box)))
	;; move to the next Box in the path
	(move-point (box-self-bp-values box))
	(set-point-screen-box old-screen-box) 
	;; and make sure that where we moved to is visible
	(ensure-row-is-displayed (point-row) old-screen-box
				 (if (row-> (point-row) old-row) 1 -1))
	(com-enter-box)
	;; if we have entered a shrunken box, then we should expand it
	(when (eq ':shrunk (tell (point-box) :display-style))
	  (com-expand-box)
	  (redisplay))
	(when (or (tell (point-screen-box) :x-got-clipped?)
		    (tell (point-screen-box) :y-got-clipped?))
	    ;; if the box is clipped, then expand it
	    (com-expand-box))))
    ;; we are no in the lowest box and all we have to do is to go to the row
    (move-point-1 row cha-no (point-screen-box))))

(defun move-to-port-target (port)
  (when (port-box? port)
    (let ((pos (make-abp-from-bp *point*)))
      (unless (abp= (car  *port-zooming-bread-crumbs*) pos)
	(push pos *port-zooming-bread-crumbs*))
      (move-point-along-path (point-row) (point-cha-no) (get-path *point* t)))))

(defboxer-command com-move-to-port-target ()
  "Move to the target of the port"
  (if (port-box? (box-screen-point-is-in))
      (move-to-port-target (box-screen-point-is-in))
      (beep)))

(defboxer-command com-follow-bread-crumbs ()
  "Move to saved location(s)"
  (let ((pos (pop *port-zooming-bread-crumbs*)))
    (when (abp? pos)
      (move-point-along-path (abp-row pos) (abp-cha-no pos) (abp-path pos)))))

(defboxer-function ctrl-meta-space-key com-move-to-port-target)

(defboxer-function ctrl-meta-r-key com-follow-bread-crumbs)



;;;; Word Commands

;;; primitives for word operations

(DEFUN BP-OVER-VALUES (BP DIRECTION DELIMITER-CHAS)
  (LET ((NOT-FIRST-CHA? NIL))
    (MAP-OVER-CHAS-IN-LINE (BP DIRECTION)
      (LET ((DELIMITER-CHA? (MEMQ (CHA-CODE CHA) DELIMITER-CHAS)))
	(COND ((AND (NULL CHA)
		    (NULL NEXT-OR-PREVIOUS-ROW)) ;end/beginning of the box
	       (RETURN (VALUES ROW CHA-NO)))
	      ((AND (NULL CHA) NOT-FIRST-CHA?)   ;end/beginning of the line
	       (RETURN (VALUES ROW CHA-NO)))
	      ((AND NOT-FIRST-CHA? DELIMITER-CHA?)          ;end of the word
	       (RETURN (VALUES ROW CHA-NO)))
	      ((NOT DELIMITER-CHA?)                         ;beginning of word
	       (SETQ NOT-FIRST-CHA? T)))))))

(DEFUN BP-FORWARD-WORD-VALUES (BP)
  (BP-OVER-VALUES BP 1 *WORD-DELIMITERS*))

(DEFUN BP-BACKWARD-WORD-VALUES (BP)
  (BP-OVER-VALUES BP -1 *WORD-DELIMITERS*))

(DEFBOXER-COMMAND COM-FORWARD-WORD ()
"moves forward one word. with numeric
argument (n), moves forward n words. "
  (WITH-MULTIPLE-EXECUTION
    (MOVE-POINT (BP-FORWARD-WORD-VALUES *POINT*))))

(DEFBOXER-COMMAND COM-BACKWARD-WORD ()
"moves backward one word. with numeric
argument (n), moves backward n words. "
  (WITH-MULTIPLE-EXECUTION
    (MOVE-POINT (BP-BACKWARD-WORD-VALUES *POINT*))))



(DEFUN RUBOUT-OVER-VALUES (BP DIRECTION DELIMITER-CHAS)
  (LET ((NOT-FIRST-CHA? NIL))
    (MAP-OVER-CHAS-IN-LINE (BP DIRECTION)
      (LET ((DELIMITER-CHA? (MEMQ (CHA-CODE CHA) DELIMITER-CHAS))
	    (FORCE-BP-TYPE ':MOVING))
	(COND ((AND (NULL CHA)(NULL NEXT-OR-PREVIOUS-ROW));end/beginning of the box
	       (RETURN (VALUES ROW CHA-NO)))
	      ((AND NOT-FIRST-CHA? (NULL CHA))           ;end/beginning of the line
	       (RETURN (VALUES ROW CHA-NO)))
	      ((AND NOT-FIRST-CHA? DELIMITER-CHA?)           ;end of the word
	       (RETURN (VALUES ROW CHA-NO)))
	      ((NOT DELIMITER-CHA?)                          ;beginning of word
	       (SETQ NOT-FIRST-CHA? T)
	       (ACTION-AT-BP-INTERNAL
		 (increment-key-tick) ;crock
		 (kill-buffer-push (tell row :cha-at-cha-no (1- cha-no)) ':backward)
		 (TELL ROW :DELETE-CHA-AT-CHA-NO (1- CHA-NO))))
	      (T                                        ;delimiter chas before word
	       (ACTION-AT-BP-INTERNAL
		 (increment-key-tick) ;crock
		 (kill-buffer-push (tell row :cha-at-cha-no (1- cha-no)) ':backward)
		 (TELL ROW :DELETE-CHA-AT-CHA-NO (1- CHA-NO)))))))))

(DEFUN DELETE-OVER-VALUES (BP DELIMITER-CHAS)
  (DO* ((ROW (BP-ROW BP) ROW)
	(NEXT-ROW (TELL-CHECK-NIL ROW :NEXT-ROW)
		  (TELL-CHECK-NIL ROW :NEXT-ROW))
	(CHA-NO (BP-CHA-NO BP)
		(BP-CHA-NO BP))
	(CHA (TELL ROW :CHA-AT-CHA-NO CHA-NO)
	     (TELL ROW :CHA-AT-CHA-NO CHA-NO))
	(NOT-FIRST-CHA?))
       (NIL)
    (COND ((AND (NULL NOT-FIRST-CHA?)
		(NULL CHA)
		(NOT-NULL NEXT-ROW))
	   (SETQ ROW NEXT-ROW
		 CHA-NO 0))
	  (T (LET ((DELIMITER-CHA? (MEMQ (CHA-CODE CHA) DELIMITER-CHAS))
		   (FORCE-BP-TYPE ':MOVING))
	       (COND ((AND (NULL CHA) (NULL NEXT-ROW))   ;end/beginning of the box
		      (RETURN (VALUES ROW CHA-NO)))
		     ((AND NOT-FIRST-CHA? (NULL CHA))    ;end/beginning of the line
		      (RETURN (VALUES ROW CHA-NO)))
		     ((AND NOT-FIRST-CHA? DELIMITER-CHA?)  ;end of the word
		      (RETURN (VALUES ROW CHA-NO)))
		     ((NOT DELIMITER-CHA?)                 ;beginning of word
		      (SETQ NOT-FIRST-CHA? T)
		      (ACTION-AT-BP-INTERNAL
			(kill-buffer-push (tell row :cha-at-cha-no cha-no) ':forward)
			(TELL ROW :DELETE-CHA-AT-CHA-NO CHA-NO )))
		     (T                                ;delimiter chas before word
		      (ACTION-AT-BP-INTERNAL
			(kill-buffer-push (tell row :cha-at-cha-no cha-no) ':forward)
			(TELL ROW :DELETE-CHA-AT-CHA-NO CHA-NO)))))))))



(DEFUN RUBOUT-WORD (BP)
  (RUBOUT-OVER-VALUES BP -1 *WORD-DELIMITERS*))

(DEFBOXER-COMMAND COM-RUBOUT-WORD ()
  "kills backward one word.  with numeric
argument (n), kills backward n words. "
  (WITH-MULTIPLE-EXECUTION
    (MOVE-POINT (RUBOUT-WORD *POINT*))))

(DEFUN DELETE-WORD (BP)
  (DELETE-OVER-VALUES BP  *WORD-DELIMITERS*))

(DEFBOXER-COMMAND COM-DELETE-WORD ()
  "kills forward one word.  with numeric
argument (n), kills forward n words. "
  (WITH-MULTIPLE-EXECUTION
    (MOVE-POINT (DELETE-WORD *POINT*))))




;;;; Fonts

(DEFUN CHANGE-CHAS-OVER-VALUES (BP DIRECTION DELIMITER-CHAS FCN &REST ARGS)
  (LET ((NOT-FIRST-CHA? NIL))
    (MAP-OVER-CHAS-IN-LINE (BP DIRECTION)
      (LET ((DELIMITER-CHA? (MEMQ (CHA-CODE CHA) DELIMITER-CHAS)))
	(COND ((AND (NULL CHA)(NULL NEXT-OR-PREVIOUS-ROW))	;end/beginning of the box
	       (RETURN (VALUES ROW CHA-NO)))
	      ((AND NOT-FIRST-CHA? (NULL CHA))	;end/beginning of the line
	       (RETURN (VALUES ROW CHA-NO)))
	      ((AND NOT-FIRST-CHA? DELIMITER-CHA?)	;end of the word
	       (RETURN (VALUES ROW CHA-NO)))
	      ((NOT DELIMITER-CHA?)		;beginning of word
	       (SETQ NOT-FIRST-CHA? T)
	       (TELL ROW :CHANGE-CHA-AT-CHA-NO CHA-NO
		     (LEXPR-FUNCALL FCN (TELL ROW :CHA-AT-CHA-NO CHA-NO) ARGS)))
	      (T				;delimiter chas before word
	       (TELL ROW :CHANGE-CHA-AT-CHA-NO CHA-NO
		     (LEXPR-FUNCALL FCN (TELL ROW :CHA-AT-CHA-NO CHA-NO) ARGS))))))))

(DEFUN BP-CHANGE-FONT-FORWARD-WORD-VALUES (BP &OPTIONAL (NEW-FONT-NO 0))
  (CHANGE-CHAS-OVER-VALUES BP 1 *WORD-DELIMITERS* #'SET-FONT-NO NEW-FONT-NO))

;;; These use a losing interface.  We need a better way to input the desired font
(DEFBOXER-COMMAND COM-CHANGE-FONT-WORD (&OPTIONAL (NEW-FONT-NO
						    (OR *EDITOR-NUMERIC-ARGUMENT* 0)))
  "Changes the font of the next word to be whatever the current numeric arg is. "
  (RESET-EDITOR-NUMERIC-ARG)
  (MOVE-POINT (BP-CHANGE-FONT-FORWARD-WORD-VALUES *POINT* NEW-FONT-NO)))

(DEFBOXER-COMMAND COM-CHANGE-FONT-CHA (&OPTIONAL (NEW-FONT-NO
						   (OR *EDITOR-NUMERIC-ARGUMENT* 0)))
  "Changes the font of the next character to be whatever the current numeric arg is. "
  (RESET-EDITOR-NUMERIC-ARG)
  (TELL (POINT-ROW) :CHANGE-CHA-AT-CHA-NO
	            (POINT-CHA-NO)
		    (SET-FONT-NO (TELL(POINT-ROW) :CHA-AT-CHA-NO (POINT-CHA-NO)) NEW-FONT-NO))
  (MOVE-POINT (BP-FORWARD-CHA-VALUES *POINT*)))



;;; These are o.k. for release since you don't have to worry about input for them

(DEFBOXER-COMMAND COM-BOLDFACE-FONT-WORD ()
  "Changes the next word to be in boldface. "
  (WITH-MULTIPLE-EXECUTION
    (MOVE-POINT (BP-CHANGE-FONT-FORWARD-WORD-VALUES *POINT* *BOLDFACE-FONT-NO*))))

(DEFBOXER-COMMAND COM-BOLDFACE-FONT-CHA ()
  "Change the next character to be in boldface. "
  (WITH-MULTIPLE-EXECUTION
    (TELL (POINT-ROW) :CHANGE-CHA-AT-CHA-NO
	  (POINT-CHA-NO)
	  (SET-FONT-NO (TELL(POINT-ROW) :CHA-AT-CHA-NO (POINT-CHA-NO))
		       *BOLDFACE-FONT-NO*))
    (MOVE-POINT (BP-FORWARD-CHA-VALUES *POINT*))))

(DEFBOXER-COMMAND COM-ITALICS-FONT-WORD ()
  "Changes the next word to be in italics. "
  (WITH-MULTIPLE-EXECUTION
    (MOVE-POINT (BP-CHANGE-FONT-FORWARD-WORD-VALUES *POINT* *ITALICS-FONT-NO*))))

(DEFBOXER-COMMAND COM-ITALICS-FONT-CHA ()
  "Change the next character to be in italics. "
  (WITH-MULTIPLE-EXECUTION
    (TELL (POINT-ROW) :CHANGE-CHA-AT-CHA-NO
	  (POINT-CHA-NO)
	  (SET-FONT-NO (TELL(POINT-ROW) :CHA-AT-CHA-NO (POINT-CHA-NO))
		       *ITALICS-FONT-NO*))
    (MOVE-POINT (BP-FORWARD-CHA-VALUES *POINT*))))



;;;; Capitalization

(DEFUN BP-UPPERCASE-FORWARD-WORD-VALUES (BP)
  (CHANGE-CHAS-OVER-VALUES BP 1 *WORD-DELIMITERS* #'CHAR-UPCASE))

(DEFUN BP-LOWERCASE-FORWARD-WORD-VALUES (BP)
  (CHANGE-CHAS-OVER-VALUES BP 1 *WORD-DELIMITERS* #'CHAR-DOWNCASE))

(DEFBOXER-COMMAND COM-UPPERCASE-WORD ()
  "Uppercases one or more words forward. "
  (WITH-MULTIPLE-EXECUTION
    (MOVE-POINT (BP-UPPERCASE-FORWARD-WORD-VALUES *POINT*))))

(DEFBOXER-COMMAND COM-LOWERCASE-WORD ()
  "Changes one or more words forward to be in lowercase. "
  (WITH-MULTIPLE-EXECUTION
    (MOVE-POINT (BP-LOWERCASE-FORWARD-WORD-VALUES *POINT*))))




;;;; Scrolling

(DEFBOXER-COMMAND COM-SCROLL-DN-ONE-SCREEN-BOX ()
  "displays the next box of text. "
  (LET* ((SCREEN-BOXS (TELL-CHECK-NIL (BOX-POINT-IS-IN) :DISPLAYED-SCREEN-OBJS)))
    (DOLIST (SCREEN-BOX SCREEN-BOXS)
      (TELL SCREEN-BOX :SCROLL-DN-ONE-SCREEN-BOX))
    (LET ((NEW-FIRST-ROW
	    (SCREEN-OBJ-ACTUAL-OBJ (CAR (TELL (CAR SCREEN-BOXS) :SCREEN-ROWS)))))
      (MOVE-POINT-1 NEW-FIRST-ROW (MIN (TELL NEW-FIRST-ROW :LENGTH-IN-CHAS)
				       (BP-CHA-NO *POINT*))))))

(DEFBOXER-COMMAND COM-SCROLL-UP-ONE-SCREEN-BOX ()
  "displays the previous box of text. "
  (LET ((SCREEN-BOXS (TELL-CHECK-NIL (BOX-POINT-IS-IN) :DISPLAYED-SCREEN-OBJS)))
    (DOLIST (SCREEN-BOX SCREEN-BOXS)
      (TELL SCREEN-BOX :SCROLL-UP-ONE-SCREEN-BOX))
    (LET ((NEW-FIRST-ROW
	    (SCREEN-OBJ-ACTUAL-OBJ (CAR (TELL (CAR SCREEN-BOXS) :SCREEN-ROWS)))))
      (MOVE-POINT-1 NEW-FIRST-ROW (MIN (TELL NEW-FIRST-ROW :LENGTH-IN-CHAS)
					    (BP-CHA-NO *POINT*))))))



;;;; Killing Stuff

(DEFBOXER-COMMAND COM-KILL-TO-END-OF-ROW ()
  "kills forward to the end of the line. "
  (LET* ((ROW (BP-ROW *POINT*))
	 (NEXT-ROW (TELL ROW :NEXT-ROW))
	 (ROW-LENGTH-IN-CHAS (TELL ROW :LENGTH-IN-CHAS))
	 (CHA-NO (BP-CHA-NO *POINT*)))
    (COND ((< CHA-NO ROW-LENGTH-IN-CHAS)
	   (kill-buffer-push (DELETE-CHAS-TO-END-OF-ROW *POINT* ':FIXED) ':FORWARD))
	  ((NULL NEXT-ROW))
	  (T (kill-buffer-push ':newline ':forward)
	     (TELL (BP-BOX *POINT*) :DELETE-ROW NEXT-ROW)
	   (INSERT-ROW-CHAS *POINT* NEXT-ROW ':FIXED)))))

(DEFBOXER-COMMAND COM-YANK ()
  "inserts the last piece of text that was killed. "
  (let ((item (kill-buffer-top)))
    (setf (kill-buffer-top) (copy-thing item))
    (insert-list-of-things item)))

(DEFBOXER-COMMAND COM-YANK-NO-COPY ()
  "Inserts the last piece of text that was killed and
removes it from the kill buffer.  No copy is made."
  (RESET-EDITOR-NUMERIC-ARG)
  (LET ((ITEM (KILL-BUFFER-TOP)))
    (SETF (KILL-BUFFER-TOP) NIL)
    (INSERT-LIST-OF-THINGS ITEM))
  (COM-ROTATE-KILL-BUFFER))

(defun insert-list-of-things (things)
  (if (listp things)
      (dolist (thing things)
	(insert-thing thing))
      (insert-thing things)))

;;;from coms
(defun insert-thing (thing)
  (cond ((null thing))
	((or (box? thing) (cha? thing)) (insert-cha *point* thing ':moving))
	((row? thing) (if (zerop (tell thing :length-in-chas))
			  (insert-row *point* thing ':moving)
			  (INSERT-ROW-CHAS *POINT* thing ':MOVING)))
	((EDITOR-REGION? THING)
	 (YANK-REGION *POINT* THING)
	 (UNLESS *HIGHLIGHT-YANKED-REGION*
	   (TELL THING :TURN-OFF))
	 (SETQ *CURRENT-EDITOR-REGION* THING))
	((eq thing ':newline) (insert-row *point* (make-initialized-row) ':moving))
	((listp thing) (insert-list-of-things thing))
	(t (ferror "Unusual object found in boxer kill buffer"))))



(defun kill-buffer-push (item direction)
  (if (= *number-of-non-kill-commands-executed* 1)
      (if (eq direction *kill-buffer-last-direction*)
	  (cond ((eq direction ':forward)
		 (ensure-list item)
		 (ensure-list (car *kill-buffer*))
		 (setf (car *kill-buffer*)
		       (nconc (car *kill-buffer*) item)))
		((eq direction ':backward)
		 (ENSURE-LIST (car *kill-buffer*))
		 (setf (car *kill-buffer*)
		       (CONS item (car *kill-buffer*)))))
	  (push item *kill-buffer*))
      (push item *kill-buffer*))
  (if (> (length *kill-buffer*) 8.) (setf (nthcdr 8. *kill-buffer*) nil))
  (setq *kill-buffer-last-direction* direction)
  (setq *number-of-non-kill-commands-executed* 0)
  *kill-buffer*)

;for control-meta-y, sort of.
(DEFBOXER-COMMAND COM-ROTATE-KILL-BUFFER ()
  "rotates the kill buffer. "
  (setq *kill-buffer* (nconc (cdr *kill-buffer*) (ncons (car *kill-buffer*)))))

;this function copys things if they're boxer structures that have uniqueness.
;This should probably return a PRE-BOX but I can't figure out how they work.
(DEFUN COPY-THING (BOXER-THING)
  (COND ((BOX? BOXER-THING) (COPY-TOP-LEVEL-BOX BOXER-THING))
	((ROW? BOXER-THING) (COPY-ROW BOXER-THING))
	((CHA? BOXER-THING) BOXER-THING)
	((EDITOR-REGION? BOXER-THING) (TELL BOXER-THING :COPY))
	((listp boxer-thing) (mapcar #'copy-thing boxer-thing))
	(T BOXER-THING)))				;aw, who cares?

;This is called by the function which handles keystrokes every time it executes a command.  If, when we execute
;a killing/saving command (i.e., call kill-buffer-push) the count is not 1, then the last command wasn't a kill
;and we should make a new entry into the kill buffer.
(DEFUN INCREMENT-KEY-TICK ()
  (INCF *NUMBER-OF-NON-KILL-COMMANDS-EXECUTED*))

;I don't know what to do abotu writing a new one that works.
;This is the old, efficient version which would delete your stuff with no hope of getting it back.
;(DEFUN COM-KILL-TO-END-OF-BOX ()
;      (DELETE-CHAS-TO-END-OF-ROW *POINT* ':FIXED)
;      (DELETE-ROWS-TO-END-OF-BOX *POINT* ':moving))



;;; Lispm interface
(DEFBOXER-COMMAND COM-YANK-FROM-LISP ()
  "Yanks text from the Lisp Machine's Kill Ring. "
  (ZWEI:WITH-EDITOR-STREAM (S :INTERVAL (SEND ZWEI:*KILL-HISTORY* :YANK) :START :BEGINNING)
     (LOOP FOR CHA = (SEND S :TYI)
	   UNTIL (NULL CHA)
	   DO (IF (CHAR= CHA #\CR)
		  (INSERT-ROW *POINT* (MAKE-INITIALIZED-ROW))
		  (INSERT-CHA *POINT* CHA)))))



;;;; Search

;;; Note: This is depending upon the fact that chas in LispM strings are 
;;; the same as chas in Boxer Rows

;;; Iterates through the characters in the row in the direction specified
;;; (positive = left-to-right) and returns either NIL if the character is not found
;;; or, the CHA-NO of the found character, or, if BOX-FIRST? is non-NIL, and a box appears
;;; before the character, then the box is returned (useful for depth first string searches)
(DEFUN FIND-CHA (CHARACTER ROW DIRECTION START-CHA-NO BOX-FIRST?)
  (LOOP WITH CHAS = (TELL ROW :CHAS)
	WITH ROW-LENGTH = (LENGTH CHAS)
	FOR CHA-NO = START-CHA-NO THEN (+ CHA-NO DIRECTION)
	UNTIL (OR (MINUSP CHA-NO) ( CHA-NO ROW-LENGTH))
	FOR CHA = (NTH CHA-NO CHAS)
	WHEN (AND BOX-FIRST? (BOX? CHA))
	  RETURN CHA
	WHEN (AND (NOT (BOX? CHA))
		  (OR (CHAR= CHA CHARACTER)
		      (AND (NULL *CASE-AFFECTS-STRING-SEARCH*)
			   (CHAR= (CHAR-UPCASE CHA) (CHAR-UPCASE CHARACTER)))))
					    
	  RETURN CHA-NO))

;;; Loops through in the characters in string and in row (starting at CHA-NO) until 
;;; either a mismatch occurs and NIL is returned or else the string runs out in which 
;;; case the CHA-NO of where the string ran out is returned
(DEFUN STRING-MATCH? (STRING ROW STARTING-CHA-NO)
  (LOOP FOR INDEX FROM 0 TO (1- (STRING-LENGTH STRING))
	FOR CHA-NO = (+ INDEX STARTING-CHA-NO)
	FOR SCHA = (AREF STRING INDEX)
	FOR RCHA = (TELL ROW :CHA-AT-CHA-NO CHA-NO)
	WHEN (NULL RCHA)
	  RETURN NIL
	UNLESS (AND (NOT (BOX? RCHA))
		    (OR (CHAR= RCHA SCHA)
			(AND (NULL *CASE-AFFECTS-STRING-SEARCH*)
			     (CHAR= (CHAR-UPCASE RCHA) (CHAR-UPCASE SCHA)))))
	  RETURN NIL
	FINALLY (RETURN CHA-NO)))

;;; Iterates through the characters in the row in the direction specified
;;; (positive = left-to-right) and returns either NIL if the string is not found
;;; or, the CHA-NO of the end of the string, or, if BOX-FIRST? is non-NIL, and a box appears
;;; before the string, then the box is returned (useful for depth first string searches)
(DEFUN STRING-IN-ROW? (STRING ROW &OPTIONAL (DIRECTION 1) BOX-FIRST? START-CHA-NO)
  (LET* ((STARTING-CHA-NO (IF (NUMBERP START-CHA-NO) START-CHA-NO
			      (IF (PLUSP DIRECTION) 0 (1- (TELL ROW :LENGTH-IN-CHAS)))))
	 (CHA-NO (FIND-CHA (AREF STRING 0) ROW DIRECTION STARTING-CHA-NO BOX-FIRST?)))
    (COND ((BOX? CHA-NO) CHA-NO)
	  ((NULL CHA-NO) NIL)
	  ((NUMBERP CHA-NO)
	   (LET ((END-CHA-NO (STRING-MATCH? STRING ROW CHA-NO)))
	     (IF (NULL END-CHA-NO)
		 (STRING-IN-ROW? STRING ROW DIRECTION BOX-FIRST? (+ CHA-NO DIRECTION))
		 END-CHA-NO)))
	  (T (EDITOR-BARF "Bad value for CHA-NO. ")))))



(DEFUN BOX-ROWS-FOR-SEARCH (BOX DIRECTION)
  (IF (PLUSP DIRECTION)
      (TELL BOX :ROWS)
      (REVERSE (TELL BOX :ROWS))))

(DEFUN GET-ROWS-FOR-SEARCH (BP DIRECTION)
  (MEMQ (BP-ROW BP) (BOX-ROWS-FOR-SEARCH (BP-BOX BP) DIRECTION)))

(DEFUN GET-BOXES-FOR-SEARCH (BP DIRECTION)
  (IF (PLUSP DIRECTION)
      (SUBSET #'BOX? (NTHCDR (BP-CHA-NO BP) (TELL (BP-ROW BP) :CHAS)))
      (REVERSE (SUBSET #'BOX? (FIRSTN (BP-CHA-NO BP) (TELL (BP-ROW BP) :CHAS))))))

(DEFUN FLAT-SEARCH (STRING &OPTIONAL (DIRECTION 1) (BP *POINT*))
  (LET* ((ROWS (GET-ROWS-FOR-SEARCH BP DIRECTION))
	 (FIRST-ROW-CHA-NO
	   (STRING-IN-ROW? STRING (CAR ROWS) DIRECTION NIL (BP-CHA-NO BP))))
    (IF (NOT-NULL FIRST-ROW-CHA-NO)
	(MAKE-INITIALIZED-BP ':MOVING (CAR ROWS) (1+ FIRST-ROW-CHA-NO))
	(LOOP FOR ROW IN (CDR ROWS)
	      FOR VAL = (STRING-IN-ROW? STRING ROW DIRECTION)
	      WHEN (NUMBERP VAL)
		RETURN (MAKE-INITIALIZED-BP ':MOVING ROW (1+ VAL))))))



(DEFUN DEEP-SEARCH-ROW (STRING ROW DIRECTION)
  (LET ((CHA-NO (STRING-IN-ROW? STRING ROW DIRECTION T)))
    (COND ((NUMBERP CHA-NO) (MAKE-INITIALIZED-BP ':MOVING ROW (1+ CHA-NO)))
	  (T (DOLIST (BOX (IF (PLUSP DIRECTION) (TELL ROW :BOXES-IN-ROW)
			      (REVERSE (TELL ROW :BOXES-IN-ROW))))
	       (LET ((VAL (DEEP-SEARCH-BOX STRING
						 (BOX-ROWS-FOR-SEARCH BOX DIRECTION)
						 DIRECTION)))
		 (WHEN (BP? VAL)
		   (RETURN VAL))))))))

(DEFUN DEEP-SEARCH-BOX (STRING ROWS DIRECTION)
  (DOLIST (ROW ROWS)
    (LET ((VAL (DEEP-SEARCH-ROW STRING ROW DIRECTION)))
      (WHEN (BP? VAL)
	(RETURN VAL)))))

(DEFUN DEEP-SEARCH (STRING &OPTIONAL (DIRECTION 1) (BP *POINT*))
  (LET* ((ROWS (GET-ROWS-FOR-SEARCH BP DIRECTION))
	 (FIRST-ROW-CHA-NO
	   (STRING-IN-ROW? STRING (CAR ROWS) DIRECTION T (BP-CHA-NO BP))))
    (COND ((NUMBERP FIRST-ROW-CHA-NO)
	   (MAKE-INITIALIZED-BP ':MOVING (CAR ROWS) (1+ FIRST-ROW-CHA-NO)))
	  ((DOLIST (BOX (GET-BOXES-FOR-SEARCH BP DIRECTION))
	     (LET ((VAL (DEEP-SEARCH-BOX STRING
					 (BOX-ROWS-FOR-SEARCH BOX DIRECTION)
					 DIRECTION)))
	       (WHEN (BP? VAL)
		 (RETURN VAL)))))
	  (T
	   (DOLIST (ROW (CDR ROWS))
	     (LET ((VAL (DEEP-SEARCH-ROW STRING ROW DIRECTION)))
	       (WHEN (BP? VAL)
		 (RETURN VAL))))))))



;;; These don't hack CR's so we'll trunctate after reading the input so we don't get errors
;;; farther down....

(DEFBOXER-COMMAND COM-FORWARD-FLAT-SEARCH ()
    "Moves the cursor forward to 
the location of a specified string 
No CR's are allowed. The search is 
a breadth first one."
  (LET* ((STRING (STRINGIFY (GET-FIRST-ROW
			      (GET-BOXER-INPUT "String to Search Forward For:") T)))
	 (NEW-BP (FLAT-SEARCH STRING)))
    (COND ((NULL NEW-BP) (BEEP))
	  ((BP? NEW-BP)
	   (MOVE-POINT (BP-VALUES NEW-BP))
	   ;; clean up
	   (TELL (BP-ROW NEW-BP) :DELETE-BP NEW-BP)
	   (ENSURE-ROW-IS-DISPLAYED (POINT-ROW) (POINT-SCREEN-BOX) 1))
	  (T (BOXER-EDITOR-ERROR "Search lossage")))))

(DEFBOXER-COMMAND COM-BACKWARD-FLAT-SEARCH ()
    "Moves the cursor backward to 
the location of a specified string 
No CR's are allowed. The search is 
a breadth first one."
  (LET* ((STRING (STRINGIFY (GET-FIRST-ROW
			      (GET-BOXER-INPUT "String to Search Backward For:") T)))
	 (NEW-BP (FLAT-SEARCH STRING -1)))
    (COND ((NULL NEW-BP) (BEEP))
	  ((BP? NEW-BP)
	   (MOVE-POINT (BP-VALUES NEW-BP))
	   ;; clean up
	   (TELL (BP-ROW NEW-BP) :DELETE-BP NEW-BP)
	   (ENSURE-ROW-IS-DISPLAYED (POINT-ROW) (POINT-SCREEN-BOX) -1))
	  (T (BOXER-EDITOR-ERROR "Search lossage")))))

(DEFBOXER-COMMAND COM-FORWARD-DEEP-SEARCH ()
    "Moves the cursor forward to 
the location of a specified string 
No CR's are allowed. The search is 
a depth first one."
  (LET* ((STRING (STRINGIFY (GET-FIRST-ROW
			      (GET-BOXER-INPUT "String to Deep Search Forward For:") T)))
	 (NEW-BP (DEEP-SEARCH STRING)))
    (COND ((NULL NEW-BP) (BEEP))
	  ((BP? NEW-BP)
	   (MOVE-TO-BP NEW-BP)
	   ;; clean up
	   (TELL (BP-ROW NEW-BP) :DELETE-BP NEW-BP))
	  (T (BOXER-EDITOR-ERROR "Search lossage")))))

(DEFBOXER-COMMAND COM-BACKWARD-DEEP-SEARCH ()
    "Moves the cursor backward to 
the location of a specified string 
No CR's are allowed. The search is 
a depth first one."
  (LET* ((STRING (STRINGIFY (GET-FIRST-ROW
			      (GET-BOXER-INPUT "String to Deep Search Backward For:") T)))
	 (NEW-BP (DEEP-SEARCH STRING -1)))
    (COND ((NULL NEW-BP) (BEEP))
	  ((BP? NEW-BP)
	   (MOVE-TO-BP NEW-BP)
	   ;; clean up
	   (TELL (BP-ROW NEW-BP) :DELETE-BP NEW-BP))
	  (T (BOXER-EDITOR-ERROR "Search lossage")))))


;;;; Random useful things

(DEFBOXER-COMMAND COM-NAME-BOX ()
  "edits the name of the box the cursor is
in. places cursor in the name row of the box,
creating one if one does not exist. "
  (IF (OR (EQ (POINT-BOX) *INITIAL-BOX*) (EQ *OUTERMOST-SCREEN-BOX* (SCREEN-BOX-POINT-IS-IN)))
      (BOXER-EDITOR-ERROR  "You cannot name the outermost box")
      (LET* ((BOX-TO-NAME (BOX-SCREEN-POINT-IS-IN))
	     (DESTINATION-SCREEN-BOX (SCREEN-BOX-POINT-IS-IN)))
	(UNLESS (ROW? (TELL BOX-TO-NAME :NAME-ROW))
	  (TELL BOX-TO-NAME :MAKE-NAME-ROW))
	(MOVE-POINT-1 (TELL BOX-TO-NAME :NAME-ROW) 0 DESTINATION-SCREEN-BOX)
	(TELL BOX-TO-NAME :MODIFIED))))

(DEFBOXER-COMMAND COM-FORCE-REDISPLAY ()
  "clears and then redisplays the screen. "
  (FORCE-REDISPLAY))
 
(DEFBOXER-COMMAND COM-BREAK ()
  "enters a LISP breakpoint. "
 (UNWIND-PROTECT
    (BREAK "Boxer")
    (FORCE-REDISPLAY)))

(DEFBOXER-COMMAND COM-BUG ()
  "sends a bug report about BOXER. "
  (BUG-BOXER))



;;;; Box Commands

(DEFBOXER-COMMAND COM-MAKE-BOX ()
"makes a DOIT box at the cursor location."
  (IF (NAME-ROW? (POINT-ROW))
      (BOXER-EDITOR-ERROR "You cannot make boxes on a name row. ")
      (LET ((BOX (MAKE-INITIALIZED-BOX)))
	(INSERT-CHA *POINT* BOX ':FIXED)
	(REDISPLAY))))

(DEFBOXER-COMMAND COM-TOGGLE-BOX-TYPE ()
  "toggles the type of the box that the 
cursor is in.  Data  Doit or Graphics 
Graphics-Data.  Ports toggle their targets. "
  (TELL (BOX-POINT-IS-IN) :TOGGLE-TYPE))

(DEFBOXER-COMMAND COM-MAKE-DATA-BOX ()
  "makes a DATA box at the cursor location.
BEEPs if the cursor is on a NAME row. "
  (IF (NAME-ROW? (POINT-ROW))
      (BOXER-EDITOR-ERROR "You cannot make boxes on a name row. ")
      (LET ((BOX (MAKE-INITIALIZED-BOX)))
	(TELL BOX :SET-TYPE 'DATA-BOX)
	(INSERT-CHA *POINT* BOX ':FIXED)
	(REDISPLAY))))

(DEFBOXER-COMMAND COM-ENTER-BOX (&OPTIONAL (BOX (BOX-POINT-IS-NEAR))
		                (SCREEN-BOX (SCREEN-BOX-POINT-IS-NEAR)))
  "enters the nearest box.  prefers the
trailing box to the leading one. "
  (WHEN (BOX? BOX)
    (when (eq ':shrunk (tell box :display-style))
      (tell box :unshrink)
      (tell box :modified))
    (MOVE-POINT (BOX-FIRST-VISIBLE-BP-VALUES BOX SCREEN-BOX))
    (SET-POINT-SCREEN-BOX SCREEN-BOX)
    (TELL BOX :ENTER)))

(DEFBOXER-COMMAND COM-MAKE-AND-ENTER-BOX ()
  "Makes a DOIT box where the cursor
is and places the cursor inside. "
  (COM-MAKE-BOX)
  (COM-ENTER-BOX))

(DEFBOXER-COMMAND COM-MAKE-AND-ENTER-DATA-BOX ()
  "Makes a Data box where the cursor
is and places the cursor inside. "
  (COM-MAKE-DATA-BOX)
  (COM-ENTER-BOX))

(DEFBOXER-COMMAND COM-MAKE-AND-NAME-BOX ()
  "Makes a named DOIT box where the cursor
is and places the cursor inside the name. "		;
  (COM-MAKE-BOX)
  (COM-ENTER-BOX)
  (COM-NAME-BOX))

(DEFBOXER-COMMAND COM-MAKE-AND-NAME-DATA-BOX ()
  "Makes a Named Data box where the cursor
is and places the cursor inside the name. "
  (COM-MAKE-DATA-BOX)
  (COM-ENTER-BOX)
  (COM-NAME-BOX))

(DEFBOXER-COMMAND COM-EXIT-BOX ()
  "exits the box the cursor is in.
cursor is placed directly AFTER the
exited box.  If the box is fullscreen,
then it is shrunken first. "
  (LET ((BOX (BOX-SCREEN-POINT-IS-IN)))
    (UNLESS (EQ BOX *INITIAL-BOX*)
      (TELL BOX :EXIT (tell (SCREEN-BOX-POINT-IS-IN) :superior-screen-box)
	    (tell box :superior-box) t))))



;;;; Shrinking and Expanding

(DEFBOXER-COMMAND COM-COLLAPSE-BOX (&OPTIONAL (BOX (BOX-SCREEN-POINT-IS-IN)))
  "shrinks the box the cursor is in. "
  (LET ((BOX-DISPLAY-STYLE (TELL BOX :DISPLAY-STYLE)))
    (COND ((EQ BOX *INITIAL-BOX*))
	  ((AND (EQ BOX (OUTERMOST-BOX))
		(NOT(NULL (TELL (OUTERMOST-BOX) :GET-SHRINK-PROOF?))))
	   NIL)
	  ((EQ BOX (OUTERMOST-BOX))
	   (MULTIPLE-VALUE-BIND (NEW-OUTERMOST-BOX NEW-OUTERMOST-SCREEN-BOX)
	       (GET-PREVIOUS-OUTERMOST-BOX-VALUES)
	     (SET-OUTERMOST-BOX NEW-OUTERMOST-BOX NEW-OUTERMOST-SCREEN-BOX)))
	  ((EQ BOX-DISPLAY-STYLE ':NORMAL)
	   (TELL BOX :SHRINK)
	   (WHEN (EQ BOX (BOX-SCREEN-POINT-IS-IN))
	     (COM-EXIT-BOX))))))


(DEFBOXER-COMMAND COM-SHRINK-BOX (&OPTIONAL (BOX (BOX-SCREEN-POINT-IS-IN)))
  "makes the box the cursor is in Tiny and
then exits. "
  (LET ((BOX-DISPLAY-STYLE (TELL BOX :DISPLAY-STYLE)))
    (COND ((OR ;(EQ BOX-DISPLAY-STYLE ':SHRUNK)
	        (EQ BOX *INITIAL-BOX*)
	        (NULL (TELL BOX :SUPERIOR-BOX))))

	  ((AND (EQ BOX (OUTERMOST-BOX))
		(NOT (NULL (TELL (OUTERMOST-BOX) :GET-SHRINK-PROOF?))))
	   NIL)

	  ((EQ BOX (OUTERMOST-BOX))
	   (TELL BOX :SHRINK)
	   (MULTIPLE-VALUE-BIND (NEW-OUTERMOST-BOX NEW-OUTERMOST-SCREEN-BOX)
	       (GET-PREVIOUS-OUTERMOST-BOX-VALUES)
	     (SET-OUTERMOST-BOX NEW-OUTERMOST-BOX NEW-OUTERMOST-SCREEN-BOX))
	   (WHEN (EQ BOX (BOX-SCREEN-POINT-IS-IN))
	     (COM-EXIT-BOX)))
	  ((EQ BOX-DISPLAY-STYLE ':NORMAL)
	   (TELL BOX :SHRINK)
	   (WHEN (EQ BOX (BOX-SCREEN-POINT-IS-IN))
	     (COM-EXIT-BOX))))))



(DEFBOXER-COMMAND COM-EXPAND-BOX (&OPTIONAL (BOX (BOX-SCREEN-POINT-IS-IN))
		                 (SCREEN-BOX (SCREEN-BOX-POINT-IS-IN)))
  "expands the box the cursor is in. "
  (LET ((BOX-DISPLAY-STYLE (TELL BOX :DISPLAY-STYLE)))
    (COND ((OR (EQ BOX (OUTERMOST-BOX))
	       (EQ BOX *INITIAL-BOX*)))
	  ((EQ BOX-DISPLAY-STYLE ':NORMAL)
	   ;;store away the old outermost screen box
	   (PUSH *OUTERMOST-SCREEN-BOX* *OUTERMOST-SCREEN-BOX-STACK*)	   
	   (SET-OUTERMOST-BOX BOX SCREEN-BOX)
	   (SET-POINT-SCREEN-BOX SCREEN-BOX))
	  (T
	   (TELL BOX :UNSHRINK)
	   (SET-POINT-SCREEN-BOX SCREEN-BOX)))))

(DEFBOXER-COMMAND COM-MAKE-SHRINK-PROOF-SCREEN ()
  "makes the outermost box shrink proof. "
  (TELL (OUTERMOST-BOX) :SET-SHRINK-PROOF? T))

(DEFBOXER-COMMAND COM-UNSHRINK-PROOF-SCREEN ()
  "allows the outermost box to be shrunken. "
  (TELL (OUTERMOST-BOX) :SET-SHRINK-PROOF? NIL))

(DEFBOXER-COMMAND COM-SET-OUTERMOST-BOX (&OPTIONAL (BOX (BOX-SCREEN-POINT-IS-IN))
			                (SCREEN-BOX (SCREEN-BOX-POINT-IS-IN)))
  "makes the box the cursor is in the
outermost box unless the box is either
a Graphics-box or a port to one. "
  (UNLESS (or (GRAPHICS-BOX? BOX) (eq *outermost-screen-box* screen-box)
	      (AND (PORT-BOX? BOX) (GRAPHICS-BOX? (TELL BOX :PORTS))))
    ;;store away the old outermost screen box
    (PUSH *OUTERMOST-SCREEN-BOX* *OUTERMOST-SCREEN-BOX-STACK*)
    (SET-OUTERMOST-BOX BOX SCREEN-BOX)))



;;;; Regions

(DEFBOXER-COMMAND COM-DEFINE-REGION ()
  "defines a region between the current
location of the cursor and the cursor. "
  (LET ((LOCAL-REGION (GET-LOCAL-REGION)))
    (COND ((NOT-NULL LOCAL-REGION)		;there already IS a region in the current box
	   (SETQ *REGION-BEING-DEFINED* LOCAL-REGION)
	   ;; we have to decide which BP of the region to replace with *POINT*
	   (IF (BP-< *POINT* (TELL LOCAL-REGION :START-BP))
	       (TELL LOCAL-REGION :SET-START-BP *POINT*)
	       (TELL LOCAL-REGION :SET-STOP-BP  *POINT*)))
	  (T					;There is No current region so we make one
	   (SETQ *REGION-BEING-DEFINED*
		 (MAKE-EDITOR-REGION (MAKE-INITIALIZED-BP :FIXED (POINT-ROW) (POINT-CHA-NO))))
	   (TELL *REGION-BEING-DEFINED* :TURN-ON)
	   (PUSH *REGION-BEING-DEFINED* REGION-LIST)))))

(DEFBOXER-COMMAND COM-INSTALL-REGION ()
  "installs the current region"
  (UNLESS (NULL *REGION-BEING-DEFINED*)
    (LET ((OLD-START-BP (TELL *REGION-BEING-DEFINED* :START-BP))
	  (OLD-STOP-BP  (TELL *REGION-BEING-DEFINED* :STOP-BP)))
    (MULTIPLE-VALUE-BIND (NEW-START-BP NEW-STOP-BP)	;make sure the BP's are at the
	(ORDER-BPS OLD-START-BP OLD-STOP-BP)
      (TELL *REGION-BEING-DEFINED* :SET-START-BP NEW-START-BP)
      (TELL *REGION-BEING-DEFINED* :SET-STOP-BP  NEW-STOP-BP)
      (INSTALL-REGION *REGION-BEING-DEFINED*)
      (UNLESS (OR (EQ *POINT* OLD-START-BP) (EQ *MOUSE-BP* OLD-START-BP))
	(TELL (BP-ROW OLD-START-BP) :DELETE-BP OLD-START-BP))
      (UNLESS (OR (EQ *POINT* OLD-STOP-BP) (EQ *MOUSE-BP* OLD-STOP-BP))
	(TELL (BP-ROW OLD-STOP-BP) :DELETE-BP OLD-STOP-BP))))))

(DEFBOXER-COMMAND COM-FLUSH-REGION ()
  "gets rid of the current region--if it exists. "
  (LET ((REGION-TO-FLUSH (OR *REGION-BEING-DEFINED* (GET-CURRENT-REGION))))
    (UNLESS (NULL REGION-TO-FLUSH)
      (FLUSH-REGION REGION-TO-FLUSH))))



(DEFBOXER-COMMAND COM-KILL-REGION ()
  "kills all the characters in the current region. "
  (LET ((REGION-TO-KILL (OR *REGION-BEING-DEFINED* (GET-CURRENT-REGION))))
    (IF (NULL REGION-TO-KILL)
	(BOXER-EDITOR-ERROR "There is no region that I can find. ")
	(KILL-REGION REGION-TO-KILL)
	(KILL-BUFFER-PUSH REGION-TO-KILL ':FORWARD)
	(FLUSH-REGION REGION-TO-KILL))))

;;; this is really boxify at *point* for now
(DEFBOXER-COMMAND COM-BOXIFY-REGION ()
  "puts all of the characters in the current
region into a box. "
  (LET* ((REGION-TO-BOX (OR *REGION-BEING-DEFINED* (GET-CURRENT-REGION))))
    (UNLESS (NULL REGION-TO-BOX)
      (KILL-REGION REGION-TO-BOX)
      (COM-MAKE-BOX)
      (COM-ENTER-BOX)
      (YANK-REGION *POINT* REGION-TO-BOX)
      (FLUSH-REGION REGION-TO-BOX)
      (SETQ REGION-TO-BOX NIL))))

(DEFBOXER-COMMAND COM-MARK-ROW ()
  "marks the current row to be 
the current region. "
  (IF (NAME-ROW? (POINT-ROW))
      (COM-EXIT-BOX)
      (LET ((START-BP (MAKE-INITIALIZED-BP :FIXED (POINT-ROW) 0))
	    (STOP-BP  (MAKE-INITIALIZED-BP :FIXED
					   (POINT-ROW)
					   (TELL (POINT-ROW) :LENGTH-IN-CHAS))))
	(SETQ *REGION-BEING-DEFINED* (MAKE-EDITOR-REGION START-BP STOP-BP))
	(TELL *REGION-BEING-DEFINED* :TURN-ON)
	(PUSH *REGION-BEING-DEFINED* REGION-LIST))))

(DEFBOXER-COMMAND COM-UNMARK-REGION ()
  "unmarks the current region. "
  (LET ((REGION-TO-UNMARK (OR *REGION-BEING-DEFINED* (GET-CURRENT-REGION))))
    (UNLESS (NULL REGION-TO-UNMARK)
      (FLUSH-REGION REGION-TO-UNMARK))))



;;;; Program Execution

(DEFBOXER-COMMAND COM-DOIT ()
  "calls the evaluator on the
current region.  If there is no
current region, marks the current
row instead. "
  (LET ((REGION (OR *REGION-BEING-DEFINED* (GET-CURRENT-REGION))))
    (COND ((NOT-NULL REGION)  
	   (DOIT-INTERNAL))	
	  (T
	   (COM-MARK-ROW)))))

(DEFBOXER-COMMAND com-doit-now-give-lispm-errors ()
  "calls the evaluator without using the
BOXER error handler. "
  (let ((*boxer-error-handler-p* nil))
    (com-doit-now)))

(DEFBOXER-COMMAND COM-DOIT-NOW ()
  "calls the evaluator on the
current region.  If there is no
current region, evaluates the
current row instead. "
  (LET ((REGION (OR *REGION-BEING-DEFINED* (GET-CURRENT-REGION))))
    (IF (NULL REGION) (COM-MARK-ROW))
    (DOIT-INTERNAL)))

(defun doit-internal ()
  (LET ((REGION-TO-DO (OR *REGION-BEING-DEFINED* (GET-CURRENT-REGION))))
    (UNLESS (NULL REGION-TO-DO)
      (UNWIND-PROTECT
	(DOIT-PRINT-RETURNED-VALUE
	  (EVAL-REGION-CATCHING-ERRORS REGION-TO-DO))
	(FLUSH-REGION REGION-TO-DO)))))

;;; Look for a | returned-value comment.  If there is one, delete the text
;;; to the right of it, and the | too, if the value is not to be printed.
;;; If there is no | and the value is to be printed, then make one.
;;; If the value is to be printed, print it.
(DEFUN DOIT-PRINT-RETURNED-VALUE (RETURNED-VALUE)
  (SETQ RETURNED-VALUE
	(COND ((MEMQ RETURNED-VALUE *RETURNED-VALUES-NOT-TO-PRINT*)
	       NIL)
	      (T (IF (BOX? RETURNED-VALUE)
		     (COPY-TOP-LEVEL-BOX RETURNED-VALUE)
		   (FORMAT NIL "~A" RETURNED-VALUE)))))
  (LET* ((BP (MAKE-BP ':MOVING))
	 (ROW (BP-ROW *POINT*))
	 (ROW-CHAS (TELL ROW :CHAS))
	 (EXISTING-VERTICAL-BAR
	   (CAR (MEM #'(LAMBDA (CODE CHA)
			 (EQ CODE (CHA-CODE CHA))) #/| ROW-CHAS))))
    (COND ((NOT-NULL EXISTING-VERTICAL-BAR)
	   (LET ((EXISTING-VERTICAL-BAR-CHA-NO
		   (TELL ROW :CHA-CHA-NO EXISTING-VERTICAL-BAR)))
	     (DOLIST (BP (TELL ROW :BPS)) 
	       (SETF (BP-CHA-NO BP)
		     (MIN EXISTING-VERTICAL-BAR-CHA-NO
			  (BP-CHA-NO BP))))
	     (MOVE-BP-1 BP ROW (+ EXISTING-VERTICAL-BAR-CHA-NO
				  (IF (NOT (NULL RETURNED-VALUE)) 1 0)))
	     (DELETE-CHAS-TO-END-OF-ROW BP ':FIXED))
	   (MOVE-BP *POINT* (ROW-LAST-BP-VALUES ROW)))
	  ((NOT (NULL RETURNED-VALUE))
	   (MOVE-BP BP (ROW-LAST-BP-VALUES ROW))
	   (INSERT-ROW-CHAS BP (MAKE-ROW '("   |")) ':MOVING)))
    (WHEN (NOT (NULL RETURNED-VALUE))
      (INSERT-ROW-CHAS BP (MAKE-ROW `(,RETURNED-VALUE))))))


(DEFBOXER-COMMAND COM-EDIT-LOCAL-LIBRARY ()
  "edits the curretn box's local library. "
  (LET ((LL (TELL (POINT-BOX) :LOCAL-LIBRARY)))
    (INSERT-CHA *POINT* LL ':FIXED)
    ;(REDISPLAY)				  ;make ll-box screen structure
    (COM-ENTER-BOX)))

;; this will lose in the prescence of labels !!!!
(DEFBOXER-COMMAND COM-PROMPT ()
  "inserts the argument names of the function
by the cursor. "
  (LET ((FUN (FUNCTION-AT-POINT)))
    (cond ((or (doit-box? fun)
	       (and (symbolp fun) (get fun 'arglist))
	       (and (symbolp fun) (boxer-boundp fun)
		    (boxer-function? (boxer-symeval fun))))
	   (insert-arglist fun))
	  (T (BOXER-EDITOR-ERROR "Can't find a function near the cursor. ")))))

(defun insert-arglist (fun)
  (MOVE-POINT-1 (POINT-ROW) (FIND-SYMBOL-END-NO *POINT*) (POINT-SCREEN-BOX))
  (insert-row-chas *POINT*
		   (make-row (mapcar #'(lambda (u)
					 (string-append " " u ":"))
				     (GET-ARG-NAMES-FROM-ARGLIST (BOXER-ARGLIST FUN))))
		   ':MOVING))



(DEFBOXER-COMMAND COM-GOTO-TOP-LEVEL ()
  "moves to the top of the WORLD box. "
  (MOVE-POINT (BOX-FIRST-BP-VALUES *INITIAL-BOX*))
  (SET-POINT-SCREEN-BOX (CAR (TELL *INITIAL-BOX* :SCREEN-OBJS)))
  (SETQ *OUTERMOST-SCREEN-BOX-STACK* NIL)
  (SET-OUTERMOST-BOX *INITIAL-BOX* (CAR (TELL *INITIAL-BOX* :SCREEN-OBJS))))



(DEFBOXER-COMMAND COM-FIX-BOX-SIZE ()
  "fixes the size of the box to be the
current height and width. "
  (MULTIPLE-VALUE-BIND (CURRENT-WID CURRENT-HEI)
      (SCREEN-OBJ-SIZE (SCREEN-BOX-POINT-IS-IN))
    (MULTIPLE-VALUE-BIND (L-WID T-WID R-WID B-WID)
	(WITH-FONT-MAP-BOUND (*BOXER-PANE*)
	  (SCREEN-BOX-BORDERS-FN ':BORDER-WIDS (SCREEN-BOX-POINT-IS-IN)))
      (TELL (BOX-SCREEN-POINT-IS-IN) :SET-FIXED-SIZE
	    (- CURRENT-WID L-WID R-WID) (- CURRENT-HEI T-WID B-WID)))))

(DEFBOXER-COMMAND COM-UNFIX-BOX-SIZE ()
  "unfixes the size of the box.  "
  (TELL (BOX-SCREEN-POINT-IS-IN) :SET-FIXED-SIZE NIL NIL)
  (TELL (BOX-SCREEN-POINT-IS-IN) :MODIFIED NIL))



;;;; Ports

(DEFUN CHECK-FOR-SUPERIOR (BOX1 BOX2)
  (COND ((NULL BOX1) NIL)
	((EQ BOX1 BOX2) T)
	(T (CHECK-FOR-SUPERIOR (TELL BOX1 :SUPERIOR-BOX) BOX2))))

(DEFMETHOD (BOX :SUPERIOR?) (ANOTHER-BOX)
  "is the arg a superior of the box ?"
  (CHECK-FOR-SUPERIOR SELF ANOTHER-BOX))

(DEFUN PORT-TO-INTERNAL (BOX)
  (LET ((NEW-PORT (MAKE-INITIALIZED-BOX :TYPE 'PORT-BOX)))
    (TELL NEW-PORT :SET-PORT-TO-BOX BOX)
    NEW-PORT))

(DEFBOXER-COMMAND COM-MAKE-PORT ()
  "specifies the current box as the target
of a port. "
  (SETQ *COM-MAKE-PORT-CURRENT-PORT* (PORT-TO-INTERNAL (POINT-BOX))))

(DEFBOXER-COMMAND COM-PLACE-PORT ()
  "inserts a port to the (previously)
specified target. "
  (WHEN (PORT-BOX? *COM-MAKE-PORT-CURRENT-PORT*)
;    (COND ((TELL (POINT-BOX) :SUPERIOR?
;		 (TELL *COM-MAKE-PORT-CURRENT-PORT* :PORTS))
;	   (FERROR "You are trying to port to a superior of the present box"))
;	  (T
	   (INSERT-CHA *POINT* *COM-MAKE-PORT-CURRENT-PORT*)
	   (SETQ *COM-MAKE-PORT-CURRENT-PORT* NIL)))



;;; graphics boxes

(DEFBOXER-COMMAND COM-TOGGLE-INTO-GRAPHICS-BOX ()
  "toggles the current box into a graphics box. "
  (UNLESS (OUTERMOST-SCREEN-BOX? (SCREEN-BOX-POINT-IS-IN))
    (COM-FIX-BOX-SIZE)
    (TELL (BOX-SCREEN-POINT-IS-IN) :SET-FLAVOR 'GRAPHICS-BOX)
    (COM-EXIT-BOX)))

(DEFBOXER-COMMAND COM-MAKE-GRAPHICS-BOX ()
  "inserts a graphics box at the cursor location. "
  (INSERT-CHA *POINT* (MAKE-GRAPHICS-BOX)))

(DEFBOXER-COMMAND com-make-graphics-data-box ()
    "inserts a graphics-data-box at the cursor location. "
  (insert-cha *point* (make-graphics-data-box)))

(DEFBOXER-COMMAND com-make-sprite-box ()
    "inserts a sprite-box at the cursor location. "
  (insert-cha *point* (make-sprite-box)))



;;;; DOCUMENTATION

;;; This should use resources or something....
(DEFUN COPY-HELP-BOX (HELP-BOX)
  (LET ((COPY (TELL HELP-BOX :COPY)))
    (WHEN (TELL HELP-BOX :EXIT-TRIGGER-ENABLED?)
      (TELL COPY :SET-EXIT-TRIGGER (TELL HELP-BOX :EXIT-TRIGGER))
      (TELL COPY :ENABLE-EXIT-TRIGGER))
    COPY))

(DEFBOXER-COMMAND COM-HELP ()
  "Displays Information About Commands.  It prompts
for a character which specifies the type of help. 
Currently valid characters are:

A  Displays commands whose names contain a given
   substring.  
C  Displays the Documentation for a Command."
  (LET ((HELP-BOX (COPY-HELP-BOX *TOP-LEVEL-HELP-BOX*)))
    (UNWIND-PROTECT
	(PROGN
	  (INSERT-CHA *POINT* HELP-BOX ':FIXED)
	  (COM-ENTER-BOX)
	  (REDISPLAY)
	  (COM-END-OF-BOX)
	  (REDISPLAY)
	  (LOOP FOR INPUT = (TELL TERMINAL-IO :ANY-TYI) THEN (TELL TERMINAL-IO :ANY-TYI)
		WHEN (MEMBER INPUT '(#\A #\a))
		  DO (COM-APROPOS-HELP)
		     (REDISPLAY)
		WHEN (MEMBER INPUT '(#\c #\C))
		  DO (COM-COMMAND-HELP)
		     (REDISPLAY)
		UNTIL (MEMBER INPUT '(#/) #\} #/c-})))
	  (COM-EXIT-BOX)
	  (WHEN (MEMQ HELP-BOX (TELL (TELL HELP-BOX :SUPERIOR-ROW) :CHAS))
	    (TELL (TELL HELP-BOX :SUPERIOR-ROW) :DELETE-CHA HELP-BOX))))))

(DEFBOXER-COMMAND COM-COMMAND-HELP ()
  "displays documentation for a given command"
  (LET ((HELP-BOX (COPY-HELP-BOX *COMMAND-DOCUMENTATION-HELP-BOX*)))
    (UNWIND-PROTECT
	(PROGN
	  (INSERT-CHA *POINT* HELP-BOX ':FIXED)
	  (COM-ENTER-BOX)
	  (REDISPLAY)
	  (COM-END-OF-BOX)
	  (REDISPLAY)
	  (LET ((KEY-TO-DOCUMENT (LOOKUP-KEY-NAME (TELL TERMINAL-IO :TYI))))
	    (COND ((AND (BOXER-FDEFINED? KEY-TO-DOCUMENT)
			(FUNCTIONP (BOXER-SYMEVAL KEY-TO-DOCUMENT))
			(STRINGP (DOCUMENTATION (BOXER-SYMEVAL KEY-TO-DOCUMENT))))
		   ;; it is a standard editor command
		   (INSERT-CHA *POINT*
			       (MAKE-BOX-FROM-STRING
				 (STRING-APPEND
				   (FORMAT NIL "The ~A ~%" KEY-TO-DOCUMENT)
				   (GET (BOXER-SYMEVAL KEY-TO-DOCUMENT)
					'EDITOR-DOCUMENTATION)))))
		  ((AND (BOXER-FDEFINED? KEY-TO-DOCUMENT)
			(FUNCTIONP (BOXER-SYMEVAL KEY-TO-DOCUMENT)))
		   (INSERT-CHA *POINT*
			       (MAKE-BOX-FROM-STRING
				 (STRING-APPEND
				   (FORMAT NIL "The ~A is~%" KEY-TO-DOCUMENT)
				   "Undocumented"))))
		  (T
		   (INSERT-CHA *POINT*
			       (MAKE-BOX-FROM-STRING
				 (STRING-APPEND
				   (FORMAT NIL "The ~A is~%" KEY-TO-DOCUMENT)
				   "Undefined"))))))
	  (TELL HELP-BOX :APPEND-ROW
		(MAKE-ROW '("Type any character to make this box go away")))
	  (REDISPLAY)
	  (TELL TERMINAL-IO :ANY-TYI))
      (COM-EXIT-BOX)
      (WHEN (MEMQ HELP-BOX (TELL (TELL HELP-BOX :SUPERIOR-ROW) :CHAS))
	(TELL (TELL HELP-BOX :SUPERIOR-ROW) :DELETE-CHA HELP-BOX)))))

(DEFUN GET-APPROPRIATE-COMMANDS (STRING)
  (LOOP FOR COM IN *BOXER-EDITOR-COMMANDS*
	WHEN (STRING-SEARCH STRING COM)
	  COLLECT COM))

(EVAL-WHEN (LOAD)
  (TELL *APROPOS-DOCUMENTATION-HELP-BOX* :SET-EXIT-TRIGGER
	#'(LAMBDA () (THROW 'MINI-COMMAND-LOOP NIL)))
  (TELL *APROPOS-DOCUMENTATION-HELP-BOX* :ENABLE-EXIT-TRIGGER)
  )

(DEFUN STRING-FOR-AVAILABLE-KEYS (COM)
  (LET* ((KEYS (GET-KEYS-FOR-COMMAND COM))
	 (AVAILABLE-KEYS (LOOP FOR KEY IN KEYS
			       WHEN (EQ COM (BOXER-SYMEVAL KEY)) COLLECT KEY)))
    (COND ((NULL KEYS) "is not currently installed on any key. ")
	  ((NULL AVAILABLE-KEYS) "is not available in this box. ")
	  (T (LOOP WITH S = (FORMAT NIL "Invoked by~%")
		   FOR KEY IN KEYS
		   WHEN (EQ COM (BOXER-SYMEVAL KEY))
		   DO (SETQ S (STRING-APPEND S (FORMAT NIL "  the ~A~%" KEY)))
		   FINALLY
		     (RETURN S))))))

(DEFBOXER-COMMAND COM-APROPOS-HELP ()
  "Displays all the comands whose names
contain a given substring. "
  (LET ((HELP-BOX (COPY-HELP-BOX *APROPOS-DOCUMENTATION-HELP-BOX*)))
    (UNWIND-PROTECT
	(*CATCH 'EXIT-FROM-HELP-BOX
	  (INSERT-CHA *POINT* HELP-BOX ':FIXED)
	  (COM-ENTER-BOX)
	  (REDISPLAY)
	  (COM-END-OF-BOX)
	  (LET* ((APROPOS-STRING (STRING (GET-FIRST-ELEMENT
					   (GET-BOXER-INPUT "Type a String.  then exit"))))
		 (COMS (GET-APPROPRIATE-COMMANDS APROPOS-STRING)))
	    (INSERT-ROW *POINT*
			(MAKE-ROW
			  (NCONS (FORMAT NIL
					 "Commands with ~A in their name" APROPOS-STRING))))
	    (LOOP FOR COM IN COMS
		  FOR BOX = (MAKE-BOX-FROM-STRING
			      (FORMAT NIL "~A~%~A~%~A"
				      COM
				      (GET COM 'EDITOR-DOCUMENTATION)
				      (STRING-FOR-AVAILABLE-KEYS COM)))
		  UNLESS (EQ COM (CAR COMS))
		    ;; shrink all the boxes except the first one
		    DO (TELL BOX :SHRINK)
		  DO (INSERT-ROW *POINT* (MAKE-ROW (NCONS BOX)))))
	  (TELL HELP-BOX :APPEND-ROW
		(MAKE-ROW '("Exit this box and it will go away")))
	  (REDISPLAY)
	  (MINI-BOXER-COMMAND-LOOP))
      (WHEN (TELL (POINT-BOX) :SUPERIOR? HELP-BOX)
	(LET ((NEW-SCREEN-BOX
		(BP-COMPUTE-NEW-SCREEN-BOX-OUT (POINT-BOX)
					       HELP-BOX
					       (POINT-SCREEN-BOX))))
	  (MOVE-POINT (BOX-SELF-BP-VALUES HELP-BOX))
	  (SET-POINT-SCREEN-BOX (TELL NEW-SCREEN-BOX :SUPERIOR-SCREEN-BOX))))
      (WHEN (MEMQ HELP-BOX (TELL (TELL HELP-BOX :SUPERIOR-ROW) :CHAS))
	(TELL (TELL HELP-BOX :SUPERIOR-ROW) :DELETE-CHA HELP-BOX)))))



;;;; Niceties

(DEFBOXER-COMMAND COM-HARDCOPY-MENU ()
  "Pop up the top level hardcopy menu"
  (LN03:TOP-LEVEL-HARDCOPY-MENU))

(DEFBOXER-COMMAND COM-HARDCOPY-SCREEN ()
  "Hardcopy a portion of the screen.  The default values hardcopy the entire window"
  (LN03:HARDCOPY-SCREEN))

(DEFBOXER-COMMAND COM-SET-HARDCOPY-OPTIONS ()
  "Change the Hardcopy Options"
  (LN03:SET-HARDCOPY-OPTIONS))

(DEFBOXER-COMMAND COM-BOOT-MACHINE ()
  "Boot the Lisp Machine"
  (WHEN (YES-OR-NO-P "Do You REALLY want to boot the machine ? ")
    (LOGOUT)
    (SI:HALT (FORMAT NIL "boot~%"))))

;;; Install the COMS we just defined
(DEFBOXER-FUNCTION CTRL-META-H-KEY COM-HARDCOPY-MENU)
(DEFBOXER-FUNCTION CTRL-META-S-KEY COM-HARDCOPY-SCREEN)
(DEFBOXER-FUNCTION CTRL-META-O-KEY COM-SET-HARDCOPY-OPTIONS)
(DEFBOXER-FUNCTION CTRL-META-*-KEY COM-BOOT-MACHINE)



;;;; MOUSE-CLICKS

(DEFUN COM-MOUSE-COLLAPSE-BOX (WINDOW X Y)
  ;; Note that this is designed to be called in the Boxer process,
  ;; not in the Mouse Process -- This is important!!!
  (WITH-MOUSE-BP-BOUND (X Y WINDOW)
    (LET ((NEW-BOX (BP-BOX MOUSE-BP))
	  (NEW-ROW (BP-ROW MOUSE-BP))
	  (NEW-CHA-NO (BP-CHA-NO MOUSE-BP)))
      (WHEN (AND (NOT-NULL NEW-ROW) (BOX? NEW-BOX))
	(SEND-EXIT-MESSAGES NEW-BOX MOUSE-SCREEN-BOX)
	(MOVE-POINT-1 NEW-ROW NEW-CHA-NO MOUSE-SCREEN-BOX)
	(COM-COLLAPSE-BOX)))))

(DEFUN COM-MOUSE-EXPAND-BOX (WINDOW X Y)
  ;; Note that this is designed to be called in the Boxer process,
  ;; not in the Mouse Process -- This is important!!!
  (WITH-MOUSE-BP-BOUND (X Y WINDOW)
    (LET ((NEW-BOX (BP-BOX MOUSE-BP))
	  (NEW-ROW (BP-ROW MOUSE-BP))
	  (NEW-CHA-NO (BP-CHA-NO MOUSE-BP)))
      (WHEN (AND (NOT-NULL NEW-ROW) (BOX? NEW-BOX))
	(SEND-EXIT-MESSAGES NEW-BOX MOUSE-SCREEN-BOX)
	(MOVE-POINT-1 NEW-ROW NEW-CHA-NO MOUSE-SCREEN-BOX)
	(TELL NEW-BOX :ENTER)
	(COM-EXPAND-BOX)))))

(DEFUN COM-MOUSE-SHRINK-BOX (WINDOW X Y)
  ;; Note that this is designed to be called in the Boxer process,
  ;; not in the Mouse Process -- This is important!!!
  (WITH-MOUSE-BP-BOUND (X Y WINDOW)
    (LET ((NEW-BOX (BP-BOX MOUSE-BP))
	  (NEW-ROW (BP-ROW MOUSE-BP))
	  (NEW-CHA-NO (BP-CHA-NO MOUSE-BP)))
      (WHEN (AND (NOT-NULL NEW-ROW) (BOX? NEW-BOX))
	(SEND-EXIT-MESSAGES NEW-BOX MOUSE-SCREEN-BOX)
	(MOVE-POINT-1 NEW-ROW NEW-CHA-NO MOUSE-SCREEN-BOX)
	(COM-SHRINK-BOX)))))

(DEFUN COM-MOUSE-SET-OUTERMOST-BOX (WINDOW X Y)
  ;; Note that this is designed to be called in the Boxer process,
  ;; not in the Mouse Process -- This is important!!!
  (WITH-MOUSE-BP-BOUND (X Y WINDOW)
    (LET ((OLD-ACTUAL-BOX (POINT-BOX))
	  (NEW-BOX (BP-BOX MOUSE-BP))
	  (NEW-ROW (BP-ROW MOUSE-BP))
	  (NEW-CHA-NO (BP-CHA-NO MOUSE-BP)))
      (WHEN (AND (NOT-NULL NEW-ROW) (BOX? NEW-BOX))
	(SEND-EXIT-MESSAGES NEW-BOX MOUSE-SCREEN-BOX
			    (eq (tell old-actual-box :superior-box) NEW-BOX))
	(MOVE-POINT-1 NEW-ROW NEW-CHA-NO MOUSE-SCREEN-BOX)
	(IF (GRAPHICS-BOX? NEW-BOX)
	    (COM-EXPAND-BOX)
	    (COM-SET-OUTERMOST-BOX))
	(TELL NEW-BOX :ENTER)))))

(DEFUN COM-MOUSE-MOVE-POINT (WINDOW X Y)
  ;; Note that this is designed to be called in the Boxer process,
  ;; not in the Mouse Process -- This is important!!!
  (WITH-MOUSE-BP-BOUND (X Y WINDOW)
    (LET ((OLD-ACTUAL-BOX (POINT-BOX))
	  (NEW-BOX (BP-BOX MOUSE-BP))
	  (NEW-ROW (BP-ROW MOUSE-BP))
	  (NEW-CHA-NO (BP-CHA-NO MOUSE-BP)))
      (WHEN (AND (NOT-NULL NEW-ROW) (NOT-NULL NEW-CHA-NO) (NOT-NULL NEW-BOX))
	(SEND-EXIT-MESSAGES NEW-BOX MOUSE-SCREEN-BOX
			    (eq (tell old-actual-box :superior-box) NEW-BOX))
	(when (eq ':shrunk (tell new-box :display-style))
	  (tell new-box :unshrink)
	  (tell new-box :modified))
	(MOVE-POINT-1 NEW-ROW NEW-CHA-NO MOUSE-SCREEN-BOX))
      (TELL NEW-BOX :ENTER)))
  (REDISPLAY-CURSOR))

(DEFUN COM-MOUSE-DEFINE-REGION (WINDOW X Y)
  (WITH-MOUSE-BP-BOUND (X Y WINDOW)
    MOUSE-SCREEN-BOX ;the variable was bound but never used....    
    (LET ((LOCAL-REGION (OR *REGION-BEING-DEFINED* (GET-CURRENT-REGION))))
      (COND ((NOT-NULL LOCAL-REGION)		;there already IS a region in the current box
	     (SETQ *REGION-BEING-DEFINED*   LOCAL-REGION
		   *FOLLOWING-MOUSE-REGION* LOCAL-REGION)
	     ;; we have to decide which BP of the region to replace with *POINT*
	     (IF (BP-< MOUSE-BP (TELL LOCAL-REGION :START-BP))
		 (TELL LOCAL-REGION :SET-START-BP *MOUSE-BP*)
		 (TELL LOCAL-REGION :SET-STOP-BP  *MOUSE-BP*)))
	    (T
	     ;; There is No current region so we make one
	     ;; between the *POINT* which is moved to where the mouse is and
	     ;; wherever it is that we let go of the mouse
	     (MOVE-POINT (BP-VALUES  MOUSE-BP))
	     (REDISPLAY-CURSOR)
	     (SETQ *REGION-BEING-DEFINED*
		   (MAKE-EDITOR-REGION *POINT* *MOUSE-BP*)
		   *FOLLOWING-MOUSE-REGION* *REGION-BEING-DEFINED*)
	     (TELL *REGION-BEING-DEFINED* :TURN-ON)
	     (PUSH *REGION-BEING-DEFINED* REGION-LIST))))))

(DEFUN COM-MOUSE-RELEASE-REGION (WINDOW X Y)
  "Releases the mouse from the region being created. "
  (WITH-MOUSE-BP-BOUND (X Y WINDOW)
    MOUSE-SCREEN-BOX				;bound but never used...
    (UNLESS (NULL *REGION-BEING-DEFINED*)
      (COND ((EQ *MOUSE-BP* (TELL *REGION-BEING-DEFINED* :START-BP))
	     (LET ((NEW-BP (MAKE-BP ':FIXED)))
	       (MOVE-BP NEW-BP (BP-VALUES *MOUSE-BP*))
	       (TELL *REGION-BEING-DEFINED* :SET-START-BP NEW-BP)))
	    ((EQ *MOUSE-BP* (TELL *REGION-BEING-DEFINED* :STOP-BP))
	     (LET ((NEW-BP (MAKE-BP ':FIXED)))
	       (MOVE-BP NEW-BP (BP-VALUES *MOUSE-BP*))
	       (TELL *REGION-BEING-DEFINED* :SET-STOP-BP NEW-BP)))))))

;;; If you think you want to use this, then you are probably wrong
;;; look at COM-MOUSE-RELEASE-REGION instead
(DEFUN COM-MOUSE-INSTALL-REGION (WINDOW X Y)
  WINDOW X Y ;the variables were bound, but never...
  (UNLESS (NULL *REGION-BEING-DEFINED*)
    (LET ((OLD-START-BP (TELL *REGION-BEING-DEFINED* :START-BP))
	  (OLD-STOP-BP  (TELL *REGION-BEING-DEFINED* :STOP-BP)))
      (MULTIPLE-VALUE-BIND (NEW-START-BP NEW-STOP-BP)	;make sure the BP's are at the
	  (ORDER-BPS OLD-START-BP OLD-STOP-BP)
	(TELL *REGION-BEING-DEFINED* :SET-START-BP NEW-START-BP)
	(TELL *REGION-BEING-DEFINED* :SET-STOP-BP  NEW-STOP-BP)
	(INSTALL-REGION *REGION-BEING-DEFINED*)
	(UNLESS (OR (EQ *POINT* OLD-START-BP) (EQ *MOUSE-BP* OLD-START-BP))
	  (TELL (BP-ROW OLD-START-BP) :DELETE-BP OLD-START-BP))
	(UNLESS (OR (EQ *POINT* OLD-STOP-BP) (EQ *MOUSE-BP* OLD-STOP-BP))
	  (TELL (BP-ROW OLD-STOP-BP) :DELETE-BP OLD-STOP-BP))))))

(DEFUN COM-MOUSE-GRAB-SPRITE (SPRITE)
  (TELL SPRITE :USURP-MOUSE-WAITING-FOR-BUTTON-RAISE))

;;; sprite commands
(defun com-sprite-right-click (turtle)
  (let* ((sprite-box (tell turtle :sprite-box))
	 (binding (tell sprite-box
		       :lookup-static-variable-in-box-only
		       'bu:r-click)))
    (unless (null binding)
      (boxer-telling (CDR BINDING) sprite-box))))

(defun com-sprite-middle-click (turtle)
  (let* ((sprite-box (tell turtle :sprite-box))
	 (binding (tell sprite-box
		       :lookup-static-variable-in-box-only
		       'bu:m-click)))
    (unless (null binding)
      (boxer-telling (CDR BINDING) sprite-box))))

(defun com-sprite-left-click (turtle)
  (let* ((sprite-box (tell turtle :sprite-box))
	 (binding (tell sprite-box
		       :lookup-static-variable-in-box-only
		       'bu:l-click)))
    (unless (null binding)
      (boxer-telling (CDR BINDING) sprite-box))))

;;; These are used to direct sprite commands to the appropriate place 
(defun graphics-box-near (box)
  (cond ((or (graphics-box? box) (graphics-data-box? box))
	 box)
	((eq *initial-box* box) nil)
	(t (graphics-box-near (tell box :superior-box)))))

(defun sprite-box-near (box)
  (cond ((sprite-box? box)
	 box)
	((eq *initial-box* box) nil)
	(t (sprite-box-near (tell box :superior-box)))))


;;;; More COMS

;;;; commands for transparent boxes

(defboxer-command com-export-box-names ()
  "Exports all of the variables in the current 
box into the surrounding box"
  (if (port-box? (box-screen-point-is-in))
      (boxer-editor-error "cant export names from a port")
      (tell (box-screen-point-is-in) :export-all-variables)))

(defboxer-command com-embargo-box-names ()
  "Turns of exporting of Box names"
  (let ((box (box-screen-point-is-in)))
    (when (and (not (port-box? box))
	       (not (null (tell box :get-exports))))
      (tell box :set-exports nil)
      ;; this ought to just remove the export marker rather than removing everything
      ;; and then putting the name back but I'm feeling lazy
      (tell (tell box :superior-box) :remove-all-static-bindings box)
      (let ((name (tell box :name-row)))
	(when (name-row? name)
	  (tell name :update-bindings T))))))

(defboxer-function ctrl-meta-circle-key com-export-box-names)
(defboxer-function ctrl-meta-e-key      com-embargo-box-names)

(defboxer-function transparent-box? ((port-to box))
  (let ((target (get-port-target box)))
    (boxer-boolean
      (and (box? target)
	   (not (null (tell target :get-exports)))
	   (memq target (tell (tell target :superior-box) :get-exporting-boxes))))))
