/*
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.

$Id: names.c,v 1.10 1994/02/15 02:29:32 janssen Exp $
*/

#include <stdio.h>

#include <iluptype.h>
#include <string.h>

#define HYPHEN		'-'
#define UNDERSCORE	'_'
#define EOS		'\0'

extern string name_lang_name(Name, string);
extern string name_base_name(Name);

#define ISUPPER(a)	(((a)<='Z')&&((a)>='A'))
#define ISLOWER(a)	(((a)<='z')&&((a)>='z'))
#define TOLOWER(a)	((ISLOWER(a))?(a):((a) | 0x20))

enum syntype { synonym_Method, synonym_Type, synonym_Exception, synonym_Constant, synonym_Interface, synonym_Invalid };

struct synonym {
  enum syntype type;
  string realname;
  string classname;
};

extern list new_list();
list Synonyms = NULL;

extern string interface_name(Interface);

boolean ReadSynonyms (string filename)
{
  enum syntype type;
  FILE *f;
  int l;
  char buf[2000];
  string new;
  char *blankpos;
  string blank;
  struct synonym *s;
  static char MethodC[] = "method";
  static char ExceptionC[] = "exception";
  static char TypeC[] = "type";
  static char InterfaceC[] = "interface";
  static char ConstantC[] = "constant";

  if ((f = fopen(filename, "r")) == NULL)
    return (FALSE);
  if (Synonyms == NULL)
    Synonyms = new_list();
  else
    list_clear (Synonyms, FALSE);
  while (fgets(buf, sizeof(buf), f) != NULL)
    {
      buf[(l = strlen(buf)) - 1] = '\0';
      if (l == 1)	/* blank line */
	continue;
      if (buf[0] == '#')	/* comment */
	continue;
      type = synonym_Invalid;
      blankpos = strchr(buf, ' ');
      if (blankpos != NULL)
	{
	  *blankpos = '\0';
	  if (strstr(MethodC, buf) == MethodC)
	    type = synonym_Method;
	  else if (strstr(TypeC, buf) == TypeC)
	    type = synonym_Type;
	  else if (strstr(ConstantC, buf) == ConstantC)
	    type = synonym_Constant;
	  else if (strstr(ExceptionC, buf) == ExceptionC)
	    type = synonym_Exception;
	  else if (strstr(InterfaceC, buf) == InterfaceC)
	    type = synonym_Interface;
	}
      if (type == synonym_Invalid
	  || (blank = strchr(blankpos + 1, ' ')) == NULL)
	{
	  fprintf (stderr, "Warning:  Synonyms file %s contains bad line <%s>.  Ignoring it.\n", filename, buf);
	  continue;
	}
      new = (string ) Malloc(l - (blankpos + 1 - buf));
      strncpy (new, blankpos + 1, l - (blankpos + 1 - buf));
      blank = new + (blank - (blankpos + 1));
      *blank = '\0';
      s = (struct synonym *) Malloc(sizeof(struct synonym));
      s->type = type;
      s->realname = new;
      s->classname = blank + 1;
      list_insert (Synonyms, s);
    }
  fclose (f);
  return (TRUE);
}

struct synfind {
  enum syntype type;
  string name;
};

static boolean MatchSynonym (struct synonym *s, struct synfind *name)
{
  if (s->type == name->type && strcmp(s->realname, name->name) == 0)
    return (TRUE);
  else
    return (FALSE);
}

static string FindSynonym (enum syntype type, char *interfacename, char *thingname, char *methodname)
{
  char buf[2000];
  struct synfind s;
  struct synonym *syn;

  if (Synonyms == NULL)
    return (NULL);
  sprintf (buf, (type == synonym_Interface) ? "%s"
	   : ((type == synonym_Method) ? "%s.%s.%s" : "%s.%s"),
	   interfacename, thingname, methodname);
  s.type = type;
  s.name = buf;
  syn = (struct synonym *) list_find (Synonyms, (boolean (*) (void *, void *)) MatchSynonym, &s);
  if (syn == NULL)
    return (NULL);
  else
    {
      return (syn->classname);
    }
}

static void do_M3_fixups (string buf)
{
  char *p;

  /* replace hyphen with underscore */

  for (p = buf;  *p != '\0';  p++)
    if (*p == HYPHEN)
      *p = UNDERSCORE;
}

string m3_interface_name (Interface i)
{
  string n;

  if ((n = name_lang_name(i->name, "m3")) == NULL)
    {
      char buf[500];
      if ((n = FindSynonym (synonym_Interface, name_base_name(i->name), NULL, NULL)) == NULL)
	{
	  sprintf (buf, "%s", name_base_name(i->name));
	  do_M3_fixups (buf);
	  n = buf;
	}
      name_set_lang_name(i->name, "m3", n);
      n = name_lang_name(i->name, "m3");
    }
  return (n);
}

static string find_builtin_name (Type t, boolean full)
{
  static struct n {
    char *iluname;
    char *m3name;
    char *m3interface;
  } names[] = {
    { "integer", "INTEGER", NULL },
    { "cardinal", "CARDINAL", NULL },
    { "shortinteger", "ShortInteger", "Ilu" },
    { "shortcardinal", "ShortCardinal", "Ilu" },
    { "longinteger", "LongInteger", "Ilu" },
    { "longcardinal", "LongCardinal", "Ilu" },
    { "byte", "Byte", "Ilu" },
    { "boolean", "BOOLEAN", NULL },
    { "real", "Real", "Ilu" },
    { "shortreal", "ShortReal", "Ilu" },
    { "longreal", "LongReal", "Ilu" },
    { "character", "Character", "Ilu" },
    { "shortcharacter", "CHAR", NULL },
    { "longcharacter", "LongCharacter", "Ilu" }};
  char *n;
  int i;

  if ((n = name_lang_name(t->name, full ? "m3-full" : "m3-simple")) == NULL)
    {
      char buf2[500];

      for (i = 0;  i < (sizeof(names)/sizeof(struct n));  i += 1)
	{
	  if (strcmp (name_base_name(t->name), names[i].iluname) == 0)
	    break;
	}
      if (i >= (sizeof(names)/sizeof(struct n)))
	return (NULL);

      if (full && (names[i].m3interface != NULL))
	sprintf (buf2, "%s.%s", names[i].m3interface, names[i].m3name);
      else
	strcpy (buf2, names[i].m3name);
      do_M3_fixups (buf2);
      n = buf2;
      name_set_lang_name (t->name, full ? "m3-full" : "m3-simple", n);
      n = name_lang_name(t->name, full ? "m3-full" : "m3-simple");
    }
  return (n);
}

string m3_type_name (Type t, boolean full)
{
  string n;

  if (t->builtIn)
    return (find_builtin_name(t, full));

  if ((n = name_lang_name(t->name, full ? "m3-full" : "m3-simple")) == NULL)
    {
      char buf2[500];

      if ((n = FindSynonym
	   (synonym_Type,
	    ((t->importInterfaceName == NULL) ? interface_name(t->interface) : t->importInterfaceName),
	    type_name(t), NULL)) == NULL)
	{
	  if (full)
	    sprintf (buf2, "%s.%s",
		     t->builtIn ? "Ilu" :
		     ((t->importInterfaceName == NULL) ? m3_interface_name(t->interface) : t->importInterfaceName),
		     name_base_name(t->name));
	  else
	    strcpy (buf2, name_base_name(t->name));
	  do_M3_fixups (buf2);
	  n = buf2;
	}
      name_set_lang_name (t->name, full ? "m3-full" : "m3-simple", n);
      n = name_lang_name(t->name, full ? "m3-full" : "m3-simple");
    }
  return (n);
}

string m3_argument_name (Argument a)
{
  string n;

  if ((n = name_lang_name(a->name, "m3")) == NULL)
    {
      char buf[500];
      sprintf (buf, "%s", name_base_name(a->name));
      do_M3_fixups (buf);
      name_set_lang_name (a->name, "m3", buf);
      n = name_lang_name(a->name, "m3");
    }
  return (n);
}

static boolean find_reserved_procedure_name (char *name)
{
  return (FALSE);
}

string m3_procedure_name (Procedure p, boolean full)
{
  string n;

  if ((n = name_lang_name(p->name, full ? "m3-full" : "m3-simple")) == NULL)
    {
      char buf1[500], buf2[500];
      if ((n = FindSynonym (synonym_Method, (p->object->importInterfaceName == NULL) ? interface_name(p->object->interface) : p->object->importInterfaceName, type_name(p->object), name_base_name(p->name))) == NULL)
	{
	  strcpy (buf1, name_base_name(p->name));
	  do_M3_fixups (buf1);
	  if (!full)
	    strcpy (buf2, buf1);
	  else
	    sprintf (buf2, "%s.%s",
		     (strcmp(buf1, "echo-self") == 0) ? "ilu" : m3_interface_name(p->object->interface),
		     buf1);
	  n = buf2;
	}
      if (find_reserved_procedure_name(n))
	{
	  sprintf (buf1, "ilu%s", n);
	  fprintf (stderr, "Warning:  ILU method \"%s:%s\" will be called \"%s\" in the Modula-3 interface, as simple \"%s\" conflicts with a built-in Modula-3 keyword of the same name.\n",
		   type_name(p->object), n, buf2, n);
	  n = buf1;
	}
      name_set_lang_name (p->name, full ? "m3-full" : "m3-simple", n);
      n = name_lang_name(p->name, full ? "m3-full" : "m3-simple");
    }
  return (n);
}

string m3_exception_name (Exception e, boolean full)
{
  string n;

  if ((n = name_lang_name(e->name, full ? "m3-full" : "m3-simple")) == NULL)
    {
      char buf[500];
      if ((n = FindSynonym (synonym_Exception, (e->importInterfaceName == NULL) ? interface_name(e->interface) : e->importInterfaceName, name_base_name(e->name), NULL)) == NULL)
	{
	  if (full)
	    sprintf (buf, "%s.%s",
		     e->importInterfaceName == NULL ? m3_interface_name(e->interface) : e->importInterfaceName,
		     name_base_name(e->name));
	  else
	    strcpy (buf, name_base_name(e->name));
	  do_M3_fixups (buf);
	  n = buf;
	}
      name_set_lang_name (e->name, full ? "m3-full" : "m3-simple", n);
      n = name_lang_name(e->name, full ? "m3-full" : "m3-simple");
    }
  return (n);
}

string m3_constant_name (Constant c, boolean full)
{
  string n;

  if ((n = name_lang_name(c->name, full ? "m3-full" : "m3-simple")) == NULL)
    {
      char buf[500];
      if ((n = FindSynonym (synonym_Constant, (c->importInterfaceName == NULL) ? interface_name(c->interface) : c->importInterfaceName, name_base_name(c->name), NULL)) == NULL)
	{
	  if (full)
	    sprintf (buf, "%s.%s",
		     c->importInterfaceName == NULL ? m3_interface_name(c->interface) : c->importInterfaceName,
		     name_base_name(c->name));
	  else
	    strcpy (buf, name_base_name(c->name));
	  do_M3_fixups (buf);
	  n = buf;
	}
      name_set_lang_name (c->name, full ? "m3-full" : "m3-simple", n);
      n = name_lang_name(c->name, full ? "m3-full" : "m3-simple");
    }
  return (n);
}

string m3_string (string s)
{
  char buf[1000];

  if (s == NULL)
    return (NULL);

  strcpy (buf, s);
  do_M3_fixups (buf);
  return (ilu_strdup(buf));
}

static void do_C_fixups (string buf)
{
  char *p;

  /* replace hyphen with underscore */

  for (p = buf;  *p != '\0';  p++)
    if (*p == HYPHEN)
      *p = UNDERSCORE;
}

string c_interface_name (Interface i)
{
  string n;

  if ((n = name_lang_name(i->name, "C")) == NULL)
    {
      char buf[500];
      sprintf (buf, "%s", name_base_name(i->name));
      do_C_fixups (buf);
      name_set_lang_name (i->name, "C", buf);
      n = name_lang_name(i->name, "C");
    }
  return (n);
}

string c_type_name (Type t)
{
  string n;

  if ((n = name_lang_name(t->name, "C")) == NULL)
    {
      char buf[500];
      sprintf (buf, "%s%s%s",
	       t->builtIn ? "ilu" : c_interface_name(t->interface),
	       t->builtIn ? "_" : "_T_",
	       name_base_name(t->name));
      do_C_fixups (buf);
      name_set_lang_name (t->name, "C", buf);
      n = name_lang_name(t->name, "C");
    }
  return (n);
}
