/*
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/

/*
$Id: code.c,v 1.37 1994/04/06 01:40:49 severson Exp $
*/

#ifdef MACOS
#pragma segment cplcode
#endif

#include "cplusplus.h"

/***********************************************************************\
*************************************************************************
**
**		Global class code
**
*************************************************************************
\***********************************************************************/

/*=====================================================================*\
**=====================================================================**
**              Exceptions
**=====================================================================**
\*=====================================================================*/

static void generate_exception_table (Interface interface, Context context)
{
  Exception e;
  int i;
  boolean addComma;

  if (list_size(interface->exceptions) > 0)
    {
      fprintf (context->file, "static struct %s_Exceptions_s e = {", cplusplus_interface_name(interface));
      for (i = 0, addComma = FALSE;  i < list_size(interface->exceptions);  i++)
	{
	  e = (Exception) list_ref (interface->exceptions, i);
	  if (e->interface == interface && e->importInterfaceName == NULL)
	    {
	      fprintf (context->file, "%s\n\t(ilu_Exception) \"%s\"", addComma ? "," : "", cplusplus_simple_name(e->name));
	      addComma = TRUE;
	    }
	}
      fprintf (context->file, "};\n\n");

      fprintf (context->file, "struct %s_Exceptions_s * %s_G::Exceptions()\n{\n",
	       cplusplus_interface_name(interface), cplusplus_interface_name(interface));
      fprintf (context->file, "\treturn(&e);\n}\n\n");
    }
}

static void CatchException (Exception e, Context context)
{
  character buf[1000];

  if (type_basic_type(e->type) != void_Type)
    {
      fprintf (context->file, "\t\telse if (_val->returnCode == %s)\n\t\t\t{\n\t\t\t", cplusplus_exception_name(e));
      if (TypeIsNonObjectStruct(e->type) || type_basic_type(e->type) == array_Type)
	fprintf (context->file, "\t_val->values.%s_Value = (%s) malloc(sizeof(%s));\n\t\t\t",
		 cplusplus_exception_name(e), cplusplus_return_type(e->type), cplusplus_type_name(e->type));
      sprintf (buf, "_val->values.%s_Value", cplusplus_exception_name(e));
      UnpackValue (context, e->type, e->def, buf,
		   (TypeIsNonObjectStruct(e->type) || type_basic_type(e->type) == array_Type),
		   FALSE, FALSE);
      fprintf (context->file, "\t\t\t}\n");
    }
}

static void generate_catch_exception (Context context)
{
  fprintf (context->file, "static void %sCatchException (ilu_Call _call, %sStatus *_val, ilu_Cardinal _ecode)\n{\n",
	   cplusplus_interface_name(context->interface), cplusplus_interface_name(context->interface));
  fprintf (context->file, "\tif (_ecode > _call->ca_method->me_exceptionCount || _ecode == 0) {\n");
  fprintf (context->file, "\t\t_val->returnCode = ilu::ProtocolError;\n");
  fprintf (context->file, "\t\t_val->values.anyvalue = (ilu_Cardinal) ilu_ProtocolException_Unknown;\n\t}\n");
  fprintf (context->file, "\telse {\n\t\t_val->returnCode = ilu::ExceptionOfMethod(_call->ca_method, _ecode);\n");
  fprintf (context->file, "\t\tif (_val->returnCode == NULL)\n\t\t\treturn;\n\n");
  list_enumerate (context->interface->exceptions, (iluparser_EnumProc) CatchException, context);
  fprintf (context->file, "\n\t}\n\treturn;\n}\n\n");
}

static void SendExceptionValue (Exception e, Context context)
{
  char buf[1000];

  if (e->type == NULL)
    return;

  fprintf (context->file, "\telse if (stat->returnCode == %s)\n\t\t{\n\t\t", cplusplus_exception_name(e));
  sprintf (buf, "stat->values.%s_Value", cplusplus_exception_name(e));
  EncodeValue (e->type, buf, FALSE, context);
  fprintf (context->file, "\t\t}\n");
}

static void SizeException (Exception e, Context context)
{
  char buf[1000];

  if (e->type == NULL)
    return;

  fprintf (context->file, "\telse if (stat->returnCode == %s)\n\t\t{", cplusplus_exception_name(e));
  fprintf (context->file, "\targSize = (0");
  sprintf (buf, "stat->values.%s_Value", cplusplus_exception_name(e));
  SizeType (e->type, buf, FALSE, context);
  fprintf (context->file, "); }\n");
}

void generate_send_exception (Context context)
{
  fprintf (context->file, "#include <stdarg.h>\n\n");
  fprintf (context->file, "int %s_G::SendException (ilu_Call _call, %sStatus *stat)\n",
	   cplusplus_interface_name(context->interface),
	   cplusplus_interface_name(context->interface));
  fprintf (context->file, "{\n\tilu_Cardinal argSize = 0;\n");
  fprintf (context->file, "\tilu_Cardinal eIndex = 0, i, limit;  ilu_Method m;\n");
  fprintf (context->file, "\tm = _call->ca_method;\n");
  fprintf (context->file, "\tfor (i = 1, limit = m->me_exceptionCount;  i <= limit;  i += 1)\n");
  fprintf (context->file, "\t\tif (ilu::ExceptionOfMethod(m, i) == stat->returnCode) eIndex = i;\n");
  fprintf (context->file, "\tif (stat->returnCode == NULL)\n\t\t/* do nothing */;\n");
  list_enumerate (context->interface->exceptions, (iluparser_EnumProc) SizeException, context);
  fprintf (context->file, "\tilu::BeginException (_call, eIndex, argSize);\n");
  fprintf (context->file, "\tif (stat->returnCode == NULL)\n\t\t;\n");
  list_enumerate (context->interface->exceptions, (iluparser_EnumProc) SendExceptionValue, context);
  fprintf (context->file, "\tilu::FinishException (_call);\n\treturn(0);\n}\n\n");
}

static void SetExceptionValue (Exception e, Context context)
{
  char *vatype;
  char *cast;

  if (e->type == NULL)
    return;
  fprintf (context->file, "\telse if (stat->returnCode == %s)\n\t\t{\n", cplusplus_exception_name(e));
  if (type_basic_type(e->type) == shortreal_Type)
    {
      vatype = "double";
      cast = "(float) ";
    }
  else
    {
      cast = "";
      vatype = cplusplus_return_type(e->type);
    }
  fprintf (context->file, "\t\t\tstat->values.%s_Value = %sva_arg(ap, %s);\n\t\t}\n",
	   cplusplus_exception_name(e), cast, vatype);
}     

static void generate_signal_exception (Context context)
{
  fprintf (context->file, "#include <stdarg.h>\n\n");
  fprintf (context->file, "void %s_G::RaiseException (%sStatus *stat, ilu_Exception exception...)\n",
	   cplusplus_interface_name(context->interface),
	   cplusplus_interface_name(context->interface));
  fprintf (context->file, "{\n\tva_list ap; va_start(ap, exception);\n");
  fprintf (context->file, "\tstat->returnCode = exception;\n");
  fprintf (context->file, "\tif (exception == NULL)\n\t\t;\n");
  list_enumerate (context->interface->exceptions, (iluparser_EnumProc) SetExceptionValue, context);
  fprintf (context->file, "\treturn;\n}\n\n");
}

static void generate_exception_procs (Context context)
{
  generate_catch_exception (context);
  generate_signal_exception (context);
  generate_send_exception (context);
}

/*=====================================================================*\
**=====================================================================**
**              IO for non-object types
**=====================================================================**
\*=====================================================================*/

static unsigned long DimCount;

static void DimHeader (long int d, Context context)
{
  fprintf (context->file, "\t{ register int _i%lu;\n\tfor (_i%lu = 0;  _i%lu < %lu;  _i%lu += 1)\n",
	   DimCount, DimCount, DimCount, d, DimCount);
  DimCount += 1;
}

static void DimRef (long int d, char *buf)
{
  sprintf (buf + strlen(buf), "[_i%lu]", DimCount);
  DimCount += 1;
}

static void DimFooter (long int d, Context context)
{
  fprintf (context->file, "\t}\n");
}

struct double_s {
  Context c;
  Type t;
  unsigned int id;
};

static void OutputRecordArg (Argument arg, Context context)
{
  char buf[1000];

  sprintf (buf, "(%s_val->%s)", TypeIsNonObjectStruct(arg->type) ? "&" : "", cplusplus_argument_name(arg));
  EncodeValue (arg->type, buf, FALSE, context);
}

static void OutputUnionType (Argument a, struct double_s *s)
{
  Type type = a->type;
  char buffer[1000];

  sprintf (buffer, "%s_val->value.%s", TypeIsNonObjectStruct(type) ? "&" : "", cplusplus_simple_name(type->name));
  fprintf (s->c->file, "\t\tcase %s_%s:\n\t\t", cplusplus_type_name(s->t), cplusplus_simple_name(type->name));
  EncodeValue (type, buffer, FALSE, s->c);
  fprintf (s->c->file, "\t\t\tbreak;\n");
}

static void generate_output_code (Type type, enum PrimitiveTypes t, Context context)
{
  TypeDescription d = type_description(type);

  fprintf (context->file, "void %s_G::Output_%s (ilu_Call _call, %s _val)\n{\n",
	   cplusplus_interface_name(context->interface), cplusplus_simple_name(type->name),
	   cplusplus_parameter_type(type));

  if (t == array_Type)
    {
      char buf[1000];
	  
      fprintf (context->file, "\tilu::OutputArray (_call);\n");
      DimCount = 0;
      list_enumerate (d->structuredDes.array.dimensions, (iluparser_EnumProc) DimHeader, context);
      sprintf (buf, "_val");
      DimCount = 0;
      list_enumerate (d->structuredDes.array.dimensions, (iluparser_EnumProc) DimRef, buf);
      EncodeValue (d->structuredDes.array.type, buf, d->structuredDes.array.optional, context);
      DimCount = 0;
      list_enumerate (d->structuredDes.array.dimensions, (iluparser_EnumProc) DimFooter, context);
      fprintf (context->file, "\tilu::EndArray (_call);\n");
    }
  else if (t == sequence_Type)
    {
      char buf[200];
      Type st = d->structuredDes.sequence.type;

      if (type_basic_type(st) == byte_Type)
	{
	  fprintf (context->file, "\tilu::OutputBytes (_call, _val->Array(), _val->Length(), %lu);\n",
		   d->structuredDes.sequence.limit);
	}
      else
	{
	  fprintf (context->file, "\tilu::OutputSequence (_call, _val->Length(), %lu);\n\t{\n", d->structuredDes.sequence.limit);
	  fprintf (context->file, "\t%s *data = _val->Array();\n\tilu_Cardinal i;\n\n",
		   (type_basic_type(st) == object_Type) ? cplusplus_parameter_type(st) : cplusplus_type_name(st));
	  fprintf (context->file, "\tfor(i = 0;  i < _val->Length();  i++)\n");
	  sprintf (buf, "%sdata[i]", TypeIsNonObjectStruct(st) ? "&" : "");
	  EncodeValue (d->structuredDes.sequence.type, buf, FALSE, context);
	  fprintf (context->file, "\t};\n\tilu::EndSequence(_call);\n");
	}
    }
  else if (t == union_Type)
    {
      struct double_s s;

      s.c = context;
      s.t = type;
      fprintf (context->file, "\tilu::OutputUnion (_call, _val->descriminator);\n");
      fprintf (context->file, "\tswitch (_val->descriminator) {\n");
      list_enumerate (d->structuredDes.uniond.types, (iluparser_EnumProc) OutputUnionType, &s);
      fprintf (context->file, "\t};\n\tilu::EndUnion(_call);\n");
    }
  else if (t == record_Type)
    {
      fprintf (context->file, "\tilu::OutputRecord (_call);\n");
      list_enumerate (d->structuredDes.record, (iluparser_EnumProc) OutputRecordArg, context);
      fprintf (context->file, "\tilu::EndRecord (_call);\n");
    }
  /* else if (t == object_Type) done elsewhere */

  fprintf (context->file, "}\n\n");
}

static void InputRecordArg (Argument arg, Context context)
{
  char buf[1000];

  sprintf (buf, "_val->%s", cplusplus_argument_name(arg));
  UnpackValue (context, arg->type, 0, buf, FALSE, FALSE, FALSE);
}

static void InputUnionType (Argument a, struct double_s *s)
{
  char buffer[1000];
  fprintf (s->c->file, "\t\tcase %u:\n\t\t", s->id++);

  sprintf (buffer, "_val->value.%s", cplusplus_simple_name(a->type->name));
  UnpackValue (s->c, a->type, 0, buffer, FALSE, FALSE, FALSE);

  fprintf (s->c->file, "\t\t\t_val->descriminator = %s_%s;\n", cplusplus_type_name(s->t), cplusplus_simple_name(a->type->name));
  fprintf (s->c->file, "\t\t\tbreak;\n");
}

static void generate_input_code (Type type, enum PrimitiveTypes t, Context context)
{
  char *ret = (char *) cplusplus_return_type(type);
  TypeDescription d = type_description(type);
  char *name = cplusplus_type_name(type);

  context->class = type;
  fprintf (context->file, "%s %s_G::Input_%s (ilu_Call _call, %s _ref)\n{\n\t%s _val;\n\n",
	   ret, cplusplus_interface_name(context->interface), cplusplus_simple_name(type->name), ret, ret);
  if (t == array_Type)
    {
      char buf[1000];

      fprintf (context->file, "\tilu::InputArray (_call);\n");
      fprintf (context->file, "\tif (_ref != NULL) _val = _ref; else _val = (%s) malloc(sizeof(%s));\n",
	       ret, name);
      DimCount = 0;
      list_enumerate (d->structuredDes.array.dimensions, (iluparser_EnumProc) DimHeader, context);
      sprintf (buf, "(*_val)");
      DimCount = 0;
      list_enumerate (d->structuredDes.array.dimensions, (iluparser_EnumProc) DimRef, buf);
      UnpackValue (context, d->structuredDes.array.type, type->def, buf, FALSE, d->structuredDes.array.optional, FALSE);
      DimCount = 0;
      list_enumerate (d->structuredDes.array.dimensions, (iluparser_EnumProc) DimFooter, context);
      fprintf (context->file, "\tilu::EndArray(_call);\n");
    }
  else if (t == sequence_Type)
    {
      Type et = d->structuredDes.sequence.type;
      string tname = (string) cplusplus_return_type(et);

      if (type_basic_type(et) == byte_Type)
	{
	  fprintf (context->file, "\tilu_Cardinal _count;\n\tilu_Byte *_bytes;\n\n");
	  fprintf (context->file, "\t_bytes = ilu::InputBytes (_call, NULL, &_count, %lu);\n",
		   d->structuredDes.sequence.limit);
	  fprintf (context->file, "\t_val = _%s_sequence::Create (_count, _bytes);\n", name);
	}
      else
	{
	  fprintf (context->file, "\tilu_Cardinal _count, _index;\n");
	  fprintf (context->file, "\t%s _tmp;\n\n", tname);
	  fprintf (context->file, "\tilu::InputSequence (_call, &_count, %lu);\n",
		   d->structuredDes.sequence.limit);
	  fprintf (context->file, "\tif (_ref != NULL) _val = _ref; else _val = new _%s_sequence;\n", name);
	  fprintf (context->file, "\tfor (_index = 0;  _index < _count;  _index++)\n\t\t{\n\t");
	  UnpackValue (context, d->structuredDes.sequence.type, type->def, "_tmp", TypeIsNonObjectStruct(type), FALSE, TRUE);
	  fprintf (context->file, "\t\t_val->Append (_tmp);\n\t};\n");
	  fprintf (context->file, "\tilu::EndSequence(_call);\n");
	}
    }
  else if (t == union_Type)
    {
      struct double_s s;

      fprintf (context->file, "\tilu_ShortCardinal descriminator;\n");
      fprintf (context->file, "\tilu::InputUnion (_call, &descriminator);\n");
      fprintf (context->file, "\tif (_ref != NULL) _val = _ref; else _val = (%s) malloc(sizeof(%s));\n",
	       ret, name);
      fprintf (context->file, "\tswitch (descriminator) {\n");
      s.c = context;
      s.t = type;
      s.id = 0;
      list_enumerate (d->structuredDes.uniond.types, (iluparser_EnumProc) InputUnionType, &s);
      fprintf (context->file, "\t\tdefault:\n\t\t\tbreak;\n\t\t};\n");
      fprintf (context->file, "\tilu::EndUnion(_call);\n");
    }
  else if (t == record_Type)
    {
      fprintf (context->file, "\tilu::InputRecord(_call);\n");
      fprintf (context->file, "\tif (_ref != NULL) _val = _ref; else _val = (%s) malloc(sizeof(%s));\n", ret, name);
      list_enumerate (d->structuredDes.record, (iluparser_EnumProc) InputRecordArg, context);
      fprintf (context->file, "\tilu::EndRecord(_call);\n");
    }
  /* else if (t == object_Type) done elsewhere */

  fprintf (context->file, "\treturn (_val);\n}\n\n");
}

static void SizeOfUnion (Argument a, struct double_s *s)
{
  char buffer[1000];

  fprintf (s->c->file, "\t\t  case %s_%s:\n", cplusplus_type_name(s->t), cplusplus_simple_name(a->type->name));
  sprintf (buffer, "%s_val->value.%s", TypeIsNonObjectStruct(a->type) ? "&" : "", cplusplus_simple_name(a->type->name));
  fprintf (s->c->file, "\t\t\t_size = (_size");
  SizeType (a->type, buffer, FALSE, s->c);
  fprintf (s->c->file, ");\n\t\t\tbreak;\n\n");
}

static void SizeRecordArg (Argument arg, Context context)
{
  char buf[1000];

  sprintf (buf, "(%s_val->%s)", TypeIsNonObjectStruct(arg->type) ? "&" : "", cplusplus_argument_name(arg));
  SizeType (arg->type, buf, FALSE, context);
}

static void generate_sizeof_code (Type type, enum PrimitiveTypes t, Context context)
{
  TypeDescription d = type_description(type);

  fprintf (context->file, "ilu_Cardinal %s_G::SizeOf_%s (ilu_Call _call, %s _val)\n{\n",
	   cplusplus_interface_name(context->interface),
	   cplusplus_simple_name(type->name), cplusplus_parameter_type(type));
  if (t == array_Type)
    {
      char buf[1000];

      fprintf (context->file, "\tilu_Cardinal _size = 0;\n\n");
      fprintf (context->file, "\t_size = ilu::SizeOfArray (_call);\n");
      DimCount = 0;
      list_enumerate (d->structuredDes.array.dimensions, (iluparser_EnumProc) DimHeader, context);
      sprintf (buf, "_val");
      DimCount = 0;
      list_enumerate (d->structuredDes.array.dimensions, (iluparser_EnumProc) DimRef, buf);
      fprintf (context->file, "\t\t_size = (_size ");
      SizeType (d->structuredDes.array.type, buf, d->structuredDes.array.optional, context);
      fprintf (context->file, ");\n");
      DimCount = 0;
      list_enumerate (d->structuredDes.array.dimensions, (iluparser_EnumProc) DimFooter, context);
      fprintf (context->file, "\tilu::EndArray(_call);\n\treturn (_size);\n");
    }
  else if (t == sequence_Type)
    {
      char buf[1000];
      Type st = d->structuredDes.sequence.type;

      if (type_basic_type(st) == byte_Type)
	{
	  fprintf (context->file, "\treturn(ilu::SizeOfBytes (_call, _val->Array(), _val->Length(), %lu));\n",
		   d->structuredDes.sequence.limit);
	}
      else
	{
	  fprintf (context->file, "\tilu_Cardinal _size = 0;\n\n");
	  fprintf (context->file, "\tregister %s *data = _val->Array();\n\n",
		   (type_basic_type(st) == object_Type) ? cplusplus_parameter_type(st) : cplusplus_type_name(st));
	  fprintf (context->file, "\tregister ilu_Cardinal i = _val->Length();\n\n");
	  fprintf (context->file, "\t_size = ilu::SizeOfSequence (_call, i, %lu);\n",
		   d->structuredDes.sequence.limit);
	  fprintf (context->file, "\tif (_size > 0) {\n");
	  fprintf (context->file, "\t\tfor (i = 0;  i < _val->Length();  i++)\n\t\t_size = (_size ");
	  sprintf (buf, "%sdata[i]", TypeIsNonObjectStruct(st) ? "&" : "");
	  SizeType (d->structuredDes.sequence.type, buf, d->structuredDes.sequence.optional, context);
	  fprintf (context->file, ");\n\t};\n\tilu::EndSequence(_call);\n\treturn (_size);\n");
	}
    }
  else if (t == union_Type)
    {
      struct double_s s;

      s.c = context;
      s.t = type;
      fprintf (context->file, "\tilu_Cardinal _size = 0;\n\n");
      fprintf (context->file, "\t_size = ilu::SizeOfUnion (_call, _val->descriminator);\n");
      fprintf (context->file, "\tif (_size > 0) {\n");
      fprintf (context->file, "\t\tswitch (_val->descriminator) {\n");
      list_enumerate (d->structuredDes.uniond.types, (iluparser_EnumProc) SizeOfUnion, &s);
      fprintf (context->file, "\t\t};\n\t};\n\tilu::EndUnion(_call);\n\treturn (_size);\n");
    }
  else if (t == record_Type)
    {
      fprintf (context->file, "\tilu_Cardinal _size = 0;\n\n");
      fprintf (context->file, "\t_size = ilu::SizeOfRecord (_call);\n");
      fprintf (context->file, "\t_size = (_size ");
      list_enumerate (d->structuredDes.record, (iluparser_EnumProc) SizeRecordArg, context);
      fprintf (context->file, ");\n\tilu::EndRecord(_call);\n\treturn (_size);\n");
    }
  /* else if (t == object_Type) done elsewhere */
  fprintf (context->file, "}\n\n");
}

static void generate_type_io_code (Type type, Context context)
{
  enum PrimitiveTypes t = type_basic_type(type);

  if ((TypeIsNonObjectStruct(type)
       OR (TypeIsArray(type)
	   AND (((type_basic_type(type_description(type)->structuredDes.array.type) != byte_Type)
		 AND (type_basic_type(type_description(type)->structuredDes.array.type) != shortcharacter_Type))
		OR (list_size(type_description(type)->structuredDes.array.dimensions) > 1)))
       OR (t == sequence_Type
	   AND ((type_basic_type(type_description(type)->structuredDes.array.type) != shortcharacter_Type)
		OR (list_size(type_description(type)->structuredDes.array.dimensions) > 1))))
      AND (type->importInterfaceName == NULL))
    {
      generate_output_code (type, t, context);
      generate_input_code (type, t, context);
      generate_sizeof_code (type, t, context);
    }
}

static void generate_ios_code (Interface interface, Context context)
{
  list_enumerate (interface->types, (iluparser_EnumProc) generate_type_io_code, context);
}

static void generate_global_code (Interface interface, Context context)
{
  generate_exception_table (interface, context);
  generate_exception_procs (context);
  generate_ios_code (interface, context);
}

static void RegisterClass (Type class, Context context)
{
  if (!class->builtIn && class->importInterfaceName == NULL)
    {
      fprintf (context->file, "\tiluObject::RegisterClass (%s::ILUClassRecord, Create_%s);\n",
	       cplusplus_type_name(class), cplusplus_type_name(class));
      fprintf (context->file, "\tSetupExceptionVectors_%s();\n", cplusplus_type_name(class));
    }
}

static void generate_registration_code (Interface interface, Context context)
{
  char *interface_name = (char *) cplusplus_interface_name(interface);

  fprintf (context->file, "/* the following is all done to achieve load-time module initialization.\n");
  fprintf (context->file, "   We declare a private class which only has one instance, statically declared.\n");
  fprintf (context->file, "   We use the constructor of the class to do all the initializations we need\n");
  fprintf (context->file, "   for the module, trusting that the single static instance of the class will\n");
  fprintf (context->file, "   be initialized before the user code is given control. */\n\n");  
  
  fprintf (context->file, "class _%s_RegistrationClass {\n\n public:\n\n", interface_name);
  fprintf (context->file, "  _%s_RegistrationClass();\n};\n\n", interface_name);

  fprintf (context->file, "static class _%s_RegistrationClass _%s_RegistrationInstance;\n\n",
	   interface_name, interface_name);

  fprintf (context->file, "_%s_RegistrationClass::_%s_RegistrationClass()\n{\n",
	   interface_name, interface_name);
  list_enumerate (interface->classes, (iluparser_EnumProc) RegisterClass, context);
  fprintf (context->file, "};\n\n");
}

static void generate_sequence_code (Type seq, Context context)
{
  char *tn, *spn, *stn, *st, *srt;
  Type sequenceType;

  if (type_basic_type(seq) != sequence_Type)
    return;

  sequenceType = type_description(seq)->structuredDes.sequence.type;

  if (type_basic_type(sequenceType) == shortcharacter_Type)
    {
      /* no declaration, already handled in typedef */
    }

  else {
    
    tn = cplusplus_type_name(seq);
    stn = cplusplus_simple_name(seq->name);
    st = (type_basic_type(sequenceType) == object_Type) ? cplusplus_parameter_type(sequenceType) : cplusplus_type_name(sequenceType);
    srt = cplusplus_return_type(sequenceType);
    spn = cplusplus_parameter_type(sequenceType);

    fprintf (context->file, "_%s_sequence::_%s_sequence ()\n{\n", tn, tn);
    fprintf (context->file, "  _maximum = 0;\n  _length = 0;\n  _buffer = NULL;\n}\n\n");

    fprintf (context->file, "_%s_sequence::~_%s_sequence ()\n{\n", tn, tn);
    fprintf (context->file, "  if (_buffer != NULL) free((char *) _buffer);\n};\n\n");

    fprintf (context->file, "%s _%s_sequence::Create (ilu_Cardinal initial_size, %s *initial_data)\n{\n",
	     tn, tn, st);
    fprintf (context->file, "  %s s = new _%s_sequence;\n", tn, tn);
    fprintf (context->file, "  if (initial_data != NULL) {\n    s->_buffer = initial_data;\n");
    fprintf (context->file, "    s->_length = initial_size;\n    s->_maximum = initial_size;\n  }\n");
    fprintf (context->file, "  else if (initial_size > 0)\n    s->_buffer = (%s *) malloc((unsigned int)(sizeof(%s) * (s->_maximum = initial_size)));\n", st, st);
    fprintf (context->file, "  return s;\n}\n\n");

    fprintf (context->file, "void _%s_sequence::Clear (ilu_Boolean free_contents)\n{\n", tn);
    fprintf (context->file, "  if (_buffer != NULL && free_contents) {\n");
    fprintf (context->file, "    free((char *) _buffer);\n");
    fprintf (context->file, "    _maximum = 0;\n");
    fprintf (context->file, "    _buffer = (%s *) 0;\n", st);
    fprintf (context->file, "  }\n");
    fprintf (context->file, "  _length = 0;\n");
    fprintf (context->file, "}\n\n");
	
    fprintf (context->file, "void _%s_sequence::Append (%s item)\n{\n", tn, spn);
    fprintf (context->file, "  if (_buffer == NULL) {\n");
    fprintf (context->file, "    _buffer = (%s *) malloc ((unsigned int)(sizeof(%s) * (_maximum = 20)));\n  }\n", st, st);
    fprintf (context->file, "  else if (_maximum <= _length) {\n");
    fprintf (context->file, "    _maximum *= 2;\n");
    fprintf (context->file, "    _buffer = (%s *) realloc ((char *) _buffer, (unsigned int)(sizeof(%s) * _maximum));\n", st, st);
    fprintf (context->file, "  };\n");
    fprintf (context->file, "  _buffer[_length] = %sitem;\n", TypeIsNonObjectStruct(sequenceType) ? "*" : "");
    fprintf (context->file, "  _length += 1;\n");
    fprintf (context->file, "}\n\n");
	
    fprintf (context->file, "%s _%s_sequence::RemoveHead ()\n{\n", srt, tn);
    fprintf (context->file, "  static %s k;\n", st);
    fprintf (context->file, "  if (_length > 0) {\n");
    fprintf (context->file, "    k = _buffer[0];\n");
    fprintf (context->file, "    if (_length > 1)\n");
    fprintf (context->file, "      memcpy (&_buffer[1], &_buffer[0], (int)((_length - 1) * sizeof(%s)));\n", st);
    fprintf (context->file, "    _length -= 1;\n");
    fprintf (context->file, "    return (%sk);\n", TypeIsNonObjectStruct(sequenceType) ? "&" : "");
    fprintf (context->file, "  }\n");
    fprintf (context->file, "  else\n    return ((%s) 0);\n", srt);
    fprintf (context->file, "}\n\n");

    fprintf (context->file, "%s _%s_sequence::RemoveTail ()\n{\n", srt, tn);
    fprintf (context->file, "  static %s k;\n", st);
    fprintf (context->file, "  if (_length > 0) {\n");
    fprintf (context->file, "    k = _buffer[_length-1];\n");
    fprintf (context->file, "    _length -= 1;\n");
    fprintf (context->file, "    return (%sk);\n", TypeIsNonObjectStruct(sequenceType) ? "&" : "");
    fprintf (context->file, "  }\n");
    fprintf (context->file, "  else\n    return ((%s) 0);\n", srt);
    fprintf (context->file, "}\n\n");

    fprintf (context->file, "ilu_Cardinal _%s_sequence::RemoveAll (ilu_Boolean (*matchproc)(%s, void *), void *arg)\n{\n", tn, spn);
    fprintf (context->file, "  ilu_Cardinal i, oldLength = _length;\n");
    fprintf (context->file, "  for (i = 0;  i < _length;  i += 1)\n");
    fprintf (context->file, "    if ((*matchproc)(%s_buffer[i], arg)) {\n", TypeIsNonObjectStruct(sequenceType) ? "&" : "");
    fprintf (context->file, "      memcpy (&_buffer[i + 1], &_buffer[i], (int)((_length - i - 1) * sizeof(%s)));\n", st);
    fprintf (context->file, "      _length -= 1;\n");
    fprintf (context->file, "    };\n");
    fprintf (context->file, "  return (oldLength - _length);\n");
    fprintf (context->file, "}\n\n");

    fprintf (context->file, "%s _%s_sequence::Find (ilu_Boolean (*matchproc)(%s, void *), void *arg)\n{\n", srt, tn, spn);
    fprintf (context->file, "  register int i;  static %s k;\n", st);
    fprintf (context->file, "  for (i = 0;  i < _length;  i += 1)\n");
    fprintf (context->file, "    if ((*matchproc)(%s_buffer[i], arg))\n", TypeIsNonObjectStruct(sequenceType) ? "&" : "");
    fprintf (context->file, "      return (%s_buffer[i]);\n", TypeIsNonObjectStruct(sequenceType) ? "&" : "");
    fprintf (context->file, "  return ((%s) 0);\n", srt);
    fprintf (context->file, "}\n\n");

    fprintf (context->file, "void _%s_sequence::Enumerate (void (*enumproc)(%s, void *), void *arg)\n{\n", tn, spn);
    fprintf (context->file, "  register int i;\n");
    fprintf (context->file, "  for (i = 0;  i < _length;  i += 1)\n");
    fprintf (context->file, "    (*enumproc)(%s_buffer[i], arg);\n", TypeIsNonObjectStruct(sequenceType) ? "&" : "");
    fprintf (context->file, "}\n\n");

    fprintf (context->file, "ilu_Cardinal _%s_sequence::Length ()\n{\n", tn);
    fprintf (context->file, "  return _length;\n");
    fprintf (context->file, "}\n\n");

    fprintf (context->file, "%s * _%s_sequence::Array ()\n{\n", st, tn);
    fprintf (context->file, "  return _buffer;\n");
    fprintf (context->file, "}\n\n");

    fprintf (context->file, "%s _%s_sequence::Nth (ilu_Cardinal index)\n{\n", srt, tn);
    fprintf (context->file, "  if (index < _length)\n");
    fprintf (context->file, "    return (%s_buffer[index]);\n", TypeIsNonObjectStruct(sequenceType) ? "&" : "");
    fprintf (context->file, "  else\n");
    fprintf (context->file, "    return ((%s) 0);\n", srt);
    fprintf (context->file, "}\n\n");
    }
}

#define IsHexDigit(x)	((((x)>='0')&&((x)<='9'))||(((x)>='a')&&((x)<='f'))||(((x)>='A')&&((x)<='F')))
#define HexValue(x)	(((x)<='9')?((x)-'0'):(((x)<='F')?((x)-'A'):(((x)<='f')?((x)-'a'):0)))

static void ConvertStringForm (char *buf, string str)
{
  register char *p, *q;

  for (p = str, q = buf;  *p != '\0';)
    {
      if (*p == '#')
	{
	  ++p;
	  if (*p == 'n')
	    *q++ = '\\', *q++ = 'n', p++;
	  else if (*p == 'r')
	    *q++ = '\\', *q++ = 'r', p++;
	  else if (IsHexDigit(*p))
	    {
	      unsigned int c = 0;
	      unsigned int i;

	      while (IsHexDigit(*p) && (i = c * 16 + HexValue(*p)) < 256)
		{
		  c = i;
		  ++p;
		}
	      sprintf (q, "\\%o", c);
	      q += strlen(q);
	    }
	  else if (*p == '\\' || *p == '"')
	    *q++ = '\\', *q++ = *p++;
	  else
	    *q++ = *p++;
	}
      else if (*p == '\\' || *p == '"')
	*q++ = '\\', *q++ = *p++;
      else
	*q++ = *p++;
    }
  *q = '\0';
}

#define CONSTANT_NAME(c)		(cplusplus_constant_name(c))

static void InitializeConstant (Constant c, Context context)
{
  enum PrimitiveTypes t = type_basic_type(c->type);

  if (c->interface != context->interface)
    return;

  if ((t == cardinal_Type || t == shortcardinal_Type || t == byte_Type)
      && (c->value->type == integer_Type))
    {
      fprintf (context->file, "%s %s = %lu;\n",
	       (t == cardinal_Type) ? "ilu_Cardinal" : ((t == byte_Type) ? "ilu_Byte" : "ilu_ShortCardinal"),
	       CONSTANT_NAME(c),
	       c->value->val.i.value);
    }
  else if ((t == integer_Type || t == shortinteger_Type)
      && (c->value->type == integer_Type))
    {
      fprintf (context->file, "%s %s = %s%lu;\n",
	       (t == integer_Type) ? "ilu_Integer" : "ilu_ShortInteger",
	       CONSTANT_NAME(c),
	       (c->value->val.i.sign < 0) ? "-" : "",
	       c->value->val.i.value);
    }
  else if (t == shortcharacter_Type
	   && (c->value->type == integer_Type))
    {
      fprintf (context->file, "ilu_ShortCharacter %s = %s%lu;\n",
	       CONSTANT_NAME(c),
	       (c->value->val.i.sign < 0) ? "-" : "",
	       c->value->val.i.value);
    }
  else if (t == shortcharacter_Type
	   && (c->value->type == shortcharacter_Type))
    {
      fprintf (context->file, "ilu_ShortCharacter %s = \\%03o;\n",
	       CONSTANT_NAME(c), c->value->val.s[0]);
    }
  else if ((t == real_Type || t == shortreal_Type)
      && (c->value->type == real_Type))
    {
      fprintf (context->file, "%s %s = %s%s.%se%ld;\n",
	       (t == real_Type) ? "ilu_Real" : "ilu_ShortReal",
	       CONSTANT_NAME(c),
	       (c->value->val.r.sign < 0) ? "-" : "",
	       c->value->val.r.value, (c->value->val.r.fraction == NULL) ? "0" : c->value->val.r.fraction,
	       c->value->val.r.exponent);
    }
  else if ((t == real_Type || t == shortreal_Type)
      && (c->value->type == integer_Type))
    {
      fprintf (context->file, "%s %s = %s%lu.0;\n",
	       (t == real_Type) ? "ilu_Real" : "ilu_ShortReal",
	       CONSTANT_NAME(c),
	       (c->value->val.i.sign < 0) ? "-" : "",
	       c->value->val.i.value);
    }
  else if (t == boolean_Type)
    {
      fprintf (context->file, "ilu_Boolean %s = %s;\n",
	       CONSTANT_NAME(c),
	       (c->value->val.b) ? "ilu_TRUE" : "ilu_FALSE");
    }
  else if (t == sequence_Type
	   && (type_basic_type(type_description(c->type)->structuredDes.sequence.type) == shortcharacter_Type)
	   && c->value->type == shortcharacter_Type)
    {
      char buf[1000];
      
      ConvertStringForm (buf, c->value->val.s);
      fprintf (context->file, "ilu_CString %s = \"%s\";\n",
	       CONSTANT_NAME(c),
	       buf);
    }
  else
    fprintf (stderr, "Invalid constant, %s, encountered.\n", name_base_name(c->name));
}

/***********************************************************************\
*************************************************************************
**
**		Global class
**
*************************************************************************
\***********************************************************************/

extern void generate_class_code(Type type, Context context);
extern void GenerateNecessaryIncludes(Context context);

void generate_code (Interface interface, FILE *file)
{
  struct context_s context;

  context.file = file;
  context.interface = interface;

  fprintf (file, "#include <ilu.H>\n");
  fprintf (file, "#include <%s.H>\n\n",
	   cplusplus_interface_name(interface));

  generate_global_code (interface, &context);

  list_enumerate (interface->types, (iluparser_EnumProc) generate_sequence_code, &context);
  list_enumerate (interface->classes, (iluparser_EnumProc) generate_class_code, &context);
  list_enumerate (interface->constants, (iluparser_EnumProc) InitializeConstant, &context);

  generate_registration_code (interface, &context);
}
