(* IluKernel.i3 *)
(* Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights
   Reserved.

   Unlimited use, reproduction, and distribution of this software is
   permitted.  Any copy of this software must include both the above
   copyright notice of Xerox Corporation and this paragraph.  Any
   distribution of this software must comply with all applicable United
   States export control laws.  This software is made available AS IS,
   and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
   INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
   MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND
   NOTWITHSTANDING ANY OTHER PROVISION CONTAINED HEREIN, ANY LIABILITY
   FOR DAMAGES RESULTING FROM THE SOFTWARE OR ITS USE IS EXPRESSLY
   DISCLAIMED, WHETHER ARISING IN CONTRACT, TORT (INCLUDING NEGLIGENCE)
   OR STRICT LIABILITY, EVEN IF XEROX CORPORATION IS ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGES. *)
(* Edited by Mike Spreitzer March 15, 1994 3:45 pm PST *)

(* The C interface to the ILU kernel. *)

INTERFACE IluKernel;
IMPORT Ctypes;

TYPE Byte = [0 .. 255];
TYPE Bool = Ctypes.int;
TYPE ShortInt = Ctypes.short;
TYPE Integer = Ctypes.long;
TYPE LongInt = ARRAY [0 .. 1] OF Integer;
TYPE ShortCard = Ctypes.unsigned_short;
TYPE Cardinal = Ctypes.unsigned_long;
TYPE LongCard = ARRAY [0 .. 1] OF Cardinal;
TYPE ShortReal = REAL;
TYPE Real = LONGREAL;
TYPE LongReal = EXTENDED;
TYPE ShortChar = Ctypes.unsigned_char;
TYPE Character = Ctypes.unsigned_short;
TYPE EnumOrd = Ctypes.unsigned_short;
TYPE Bytes = Ctypes.void_star;
TYPE C_String = UNTRACED REF BITS 8 FOR ['\000' .. '\377'];
TYPE WString = UNTRACED REF BITS 16 FOR Character;

TYPE
  ObjectType =
    UNTRACED REF
      RECORD (* M3 expression of datatype defined in kernel/ilu.h *)
        name       : C_String; (* ILU name of class *)
        brand      : C_String; (* brand on class *)
        unique_id  : C_String; (* unique id for type graph of class *)
        singleton  : Bool;
        collectible: Bool;
        authentication: C_String; (* non-NULL with name of
                                     authentication type, if any *)
        methods: Method;
        (* Addr of first elt in the array of methods; array excludes
           Ping, id 0. *)
        method_count    : Cardinal;
        superclass_count: Cardinal;
        superclass_ids: UNTRACED REF C_String;
        (* address of vector of strings *)
        superclasses: UNTRACED REF ObjectType;
        (* address of vector of pointers to superclasses *)
        shown: Bool;
      END;
  (* An object type defined in ISL.  This data structure is called
     "ilu_Class" in iluExports.h. *)

TYPE Method = UNTRACED REF Method_Rec;
TYPE
  Method_Rec =
    RECORD (* M3 expression of datatype defined in kernel/ilu.h *)
      name        : C_String;
      id          : Cardinal;
      cacheable   : Bool;
      asynchronous: Bool;
      exceptionVector: UNTRACED REF Exception; (* Addr of first elt in
                                                  the array of possible
                                                  exceptions *)
      exceptionCount: Cardinal;
      stubproc: Ctypes.void_star := NIL;
      (* A StubProc is stored here.  The truth, some C code, declares
         this as a function pointer; by declaring this field a pointer,
         we get the the M3 compiler to verify that an M3 PROCEDURE ...
         RAISES {} fits into a pointer. *)
    END;

TYPE
  StubProc = PROCEDURE (call: Call) RAISES {};
  (* LOOPHOLE(call.ls_data, REFANY) is a ConnectionWorker. *)

TYPE Exception = C_String;

TYPE
  Call = UNTRACED REF RECORD
                        serialNumber: Cardinal;
                        server      : Server;
                        introType   : ObjectType;
                        method      : Method;
                        conn        : Connection;
                        ls_data     : ADDRESS;
                      END;

TYPE Server = UNTRACED BRANDED REF RECORD END;
TYPE Object = UNTRACED BRANDED REF RECORD END;
TYPE Port = UNTRACED BRANDED REF RECORD END;
TYPE Connection = UNTRACED BRANDED REF RECORD END;

TYPE
  ObjectTable =
    UNTRACED REF
      RECORD
        object_of_ih: PROCEDURE (self: ObjectTable; ih: C_String):
                        Object;
        ih_of_object: PROCEDURE (self: ObjectTable; obj: Object):
                        C_String;
        set_assoc: PROCEDURE
                     (self: ObjectTable; ih: C_String; obj: Object);
        remove_assoc: PROCEDURE
                        (self: ObjectTable; ih: C_String; obj: Object);
        has_objs : PROCEDURE (self: ObjectTable): Bool;
        free_self: PROCEDURE (self: ObjectTable);
        rock     : ADDRESS;
      END;

TYPE
  ProtocolException =
    {Success, NoSuchClassAtServer, ClassVersionMismatch,
     NoSuchMethodOnClass, GarbageArguments, Unknown, LostConnection,
     RequestRejected, RequestTimeout};

CONST
  PacketType_Request = 0;
  PacketType_Reply   = 1;

  InterpretRequest_Failed    = 1;
  InterpretRequest_Succeeded = 2;
  InterpretRequest_Handled   = 3;

  ProtocolExceptionNames = ARRAY [1 .. 8] OF
                             TEXT{
                             "NoSuchClassAtServer",
                             "ClassVersionMismatch",
                             "NoSuchMethodOnClass", "GarbageArguments",
                             "Unknown", "LostConnection",
                             "RequestRejected", "RequestTimeout"};

<*EXTERNAL*> PROCEDURE ilu_SetDebugLevel (bits: Ctypes.unsigned_long);

<*EXTERNAL*> PROCEDURE ilu_SetDebugLevelViaString (spec: C_String);

(* ================ Client side routines ================ *)

<*EXTERNAL*> PROCEDURE ilu_ObjectOfSBH (sbh, type_id: C_String;
                                        static_type : ObjectType ):
  Object;

<*EXTERNAL*> PROCEDURE ilu_ClassOfObject (object: Object): ObjectType;

<*EXTERNAL*> PROCEDURE ilu_ServerOfObject (object: Object): Server;

<*EXTERNAL*> PROCEDURE ilu_BeginCall (server: Server): Call;

<*EXTERNAL*> PROCEDURE ilu_BeginRequest (call      : Call;
                                         intro_type: ObjectType;
                                         method    : Method;
                                         argSize   : Cardinal    ):
  Bool;

<*EXTERNAL*> PROCEDURE ilu_FinishRequest (call: Call): Bool;

<*EXTERNAL*>
  PROCEDURE ilu_GetReply (call: Call; VAR errorStatus: Cardinal):
  INTEGER;

<*EXTERNAL*>
  PROCEDURE ilu_ExceptionOfMethod (method: Method; index: Cardinal):
  Exception;

<*EXTERNAL*> PROCEDURE ilu_FinishCall (call: Call);


(* ================ Server side routines ================ *)

<*EXTERNAL*>
  PROCEDURE ilu_CreateTrueServer (id: C_String; objtab: ObjectTable):
  Server;

<*EXTERNAL*> PROCEDURE ilu_InventID (): C_String;

<*EXTERNAL*> PROCEDURE ilu_CreateHashObjectTable (): ObjectTable;

<*EXTERNAL*> PROCEDURE ilu_SetDefaultServer (s: Server);

<*EXTERNAL*> PROCEDURE ilu_GetDefaultServer (): Server;

<*EXTERNAL*>
  PROCEDURE ilu_CreatePort (server                     : Server;
                            protocolInfo, transportInfo: C_String ):
  Port;

<*EXTERNAL*> PROCEDURE ilu_SetServerDefaultPort (s: Server; p: Port);

<*EXTERNAL*>
  PROCEDURE ilu_FindOrCreateTrueObject (ih    : C_String;
                                        server: Server;
                                        ot    : ObjectType;
                                        lspo  : ADDRESS     ): Object;

<*EXTERNAL*> PROCEDURE ilu_IhOfObject (o: Object): C_String;

<*EXTERNAL*>
  PROCEDURE ilu_FileDescriptorOfMooringOfPort (    port  : Port;
                                               VAR closed: Bool  ):
  Integer;

<*EXTERNAL*>
  PROCEDURE ilu_HandleNewConnection (port: Port; VAR closed: Bool):
  Connection;

<*EXTERNAL*> PROCEDURE ilu_ClosePort (port: Port);

<*EXTERNAL*>
  PROCEDURE ilu_BlockingWaitForInputOnConnection (conn : Connection;
                                                  limit: FineTimePtr ):
  Bool;

<*EXTERNAL*> PROCEDURE ilu_ThreadPerRequest (conn: Connection): Bool;

<*EXTERNAL*> PROCEDURE ilu_ReceiveRequest (    conn     : Connection;
                                           VAR call     : Call;
                                           VAR introType: ObjectType;
                                           VAR meth     : Method;
                                           VAR serialNumber: Cardinal

  ): ReceiveRequestStatus;

TYPE
  ReceiveRequestStatus =
    {conn_closed, (* done with this connection *)
     no_request, (* there wasn't really any input waiting *)
     read_failed, (* err sys may give more info *)
     not_request, (* wrong type message received *)
     interp_failed, (* protocol's interp proc failed *)
     builtin_meth, (* built-in; already handled *)
     good_request (* decent message received *)
    };

CONST
  ReceiveRequestStatusName = ARRAY OF
                               TEXT{
                               "conn_closed", "no_request",
                               "read_failed", "not_request",
                               "interp_failed", "builtin_meth",
                               "good_request"};


<*EXTERNAL*>
  PROCEDURE ilu_GetServerSingleton (s: Server; introType: ObjectType):
  Object;

<*EXTERNAL*> PROCEDURE ilu_RequestRead (call: Call);

<*EXTERNAL*> PROCEDURE ilu_BeginReply (call      : Call;
                                       exceptions: Bool;
                                       argSize   : Cardinal ): Bool;

<*EXTERNAL*> PROCEDURE ilu_FinishReply (call: Call): Bool;

<*EXTERNAL*>
  PROCEDURE ilu_BeginException (call: Call; evalue, argSize: Cardinal):
  Bool;

<*EXTERNAL*> PROCEDURE ilu_FinishException (call: Call): Bool;

<*EXTERNAL*> PROCEDURE ilu_NoReply (call: Call);

<*EXTERNAL*> PROCEDURE ilu_CloseConnection (conn: Connection);


(* ================ I/O parameters ================ *)

TYPE
  FineTimePtr = UNTRACED REF FineTime;
  FineTime = RECORD
               s: Integer;
               t: Cardinal
             END;

<*EXTERNAL*> PROCEDURE ilu_shim_FineTime_Now (VAR t: FineTime);

<*EXTERNAL*> VAR ilu_FineTimeRate: Cardinal;

<*EXTERNAL*> PROCEDURE ilu_shim_FineTime_Add (VAR a, b, c: FineTime);
<*EXTERNAL*> PROCEDURE ilu_shim_FineTime_Sub (VAR a, b, c: FineTime);
<*EXTERNAL*> PROCEDURE ilu_shim_FineTime_Cmp (VAR a, b: FineTime):
  Integer;

<*EXTERNAL*> PROCEDURE ilu_rescale (n, dfrom, dto: Cardinal): Cardinal;

TYPE
  MainLoop =
    RECORD
      run : PROCEDURE (stop: Ctypes.IntStar);
      exit: PROCEDURE (stop: Ctypes.IntStar);
      register_input: PROCEDURE
                        (fd: Ctypes.Int;
                         proc: PROCEDURE (fd: Ctypes.Int; rock: ADDRESS);
                         rock: ADDRESS): Bool;
      unregister_input: PROCEDURE (fd: Ctypes.Int): Bool;
      register_output: PROCEDURE
                         (fd: Ctypes.Int;
                          proc: PROCEDURE
                                  (fd: Ctypes.Int; rock: ADDRESS);
                          rock: ADDRESS): Bool;
      unregister_output: PROCEDURE (fd: Ctypes.Int): Bool;
      create_alarm     : PROCEDURE (): ADDRESS;
      set_alarm: PROCEDURE
                   (alarm: ADDRESS;
                    t    : FineTime;
                    proc : PROCEDURE (rock: ADDRESS);
                    rock : ADDRESS                    );
      unset_alarm: PROCEDURE (alarm: ADDRESS);
    END;

<*EXTERNAL*> PROCEDURE ilu_SetMainLoop (ml: UNTRACED REF MainLoop);

TYPE
  WaitTech =
    RECORD
      read_wait: PROCEDURE
                   (fd: INTEGER; VAR sure: Bool; limit: FineTimePtr)
                   RAISES {};
      write_wait: PROCEDURE
                    (fd: INTEGER; VAR sure: Bool; limit: FineTimePtr)
                    RAISES {};
    END;

<*EXTERNAL*> PROCEDURE ilu_SetWaitTech (wt: UNTRACED REF WaitTech);

(* ================ Other routines ================ *)

<*EXTERNAL*>
  PROCEDURE ilu_ParseStringBindingHandle (sbh: C_String;
                                          VAR instance_id, contact_info: C_String):
  Bool;

<*EXTERNAL*> PROCEDURE ilu_ParseOID (    instance_id  : C_String;
                                     VAR oh, server_id: C_String  ):
  Bool;

<*EXTERNAL*> PROCEDURE ilu_SBHOfObject (ko: Object): C_String;

<*EXTERNAL*> PROCEDURE ilu_MstidOfObject (ko: Object): C_String;

<*EXTERNAL*> PROCEDURE ilu_IDOfServer (ks: Server): C_String;

<*EXTERNAL*> PROCEDURE ilu_TrueServerP (ks: Server): Bool;


(* ================ Locking routines ================ *)

TYPE
  LockTech = RECORD
               mcreate : PROCEDURE (d1, d2: C_String): ADDRESS;
               acquire : PROCEDURE (m: ADDRESS);
               hold    : PROCEDURE (m: ADDRESS);
               release : PROCEDURE (m: ADDRESS);
               mdestroy: PROCEDURE (m: ADDRESS);

               ccreate : PROCEDURE (d1, d2: C_String): ADDRESS;
               notify  : PROCEDURE (c: ADDRESS);
               cdestroy: PROCEDURE (c: ADDRESS);
               wait    : PROCEDURE (c: ADDRESS; m: ADDRESS);
             END;

<*EXTERNAL*> PROCEDURE ilu_RegisterLockTech (lt: UNTRACED REF LockTech);


<*EXTERNAL*> PROCEDURE ilu_EnterServer (server: Server; ot: ObjectType);

<*EXTERNAL*> PROCEDURE ilu_ExitServer (server: Server; ot: ObjectType);


(* ================ [Un]Marshalling routines ================ *)


<*EXTERNAL*> PROCEDURE ilu_EndSequence (call: Call);

<*EXTERNAL*> PROCEDURE ilu_EndUnion (call: Call);

<*EXTERNAL*> PROCEDURE ilu_EndArray (call: Call);

<*EXTERNAL*> PROCEDURE ilu_EndRecord (call: Call);


<*EXTERNAL*> PROCEDURE ilu_OutputShortInteger (call: Call; i: ShortInt);

<*EXTERNAL*> PROCEDURE ilu_OutputInteger (call: Call; i: Integer);

<*EXTERNAL*> PROCEDURE ilu_OutputLongInteger (call: Call; i: LongInt);

<*EXTERNAL*>
  PROCEDURE ilu_OutputShortCardinal (call: Call; i: ShortCard);

<*EXTERNAL*> PROCEDURE ilu_OutputCardinal (call: Call; i: Cardinal);

<*EXTERNAL*> PROCEDURE ilu_OutputLongCardinal (call: Call; i: LongCard);

<*EXTERNAL*> PROCEDURE ilu_shim_OutputShortReal (call: Call; f: Real);
  (* Declared to take a C "double" because M3 is compiled to "classic"
     C.  This shim proc calls the ANSI C kernel proc that takes a C
     "float". *)

<*EXTERNAL*> PROCEDURE ilu_OutputReal (call: Call; d: Real);

<*EXTERNAL*> PROCEDURE ilu_OutputLongReal (call: Call; f: LongReal);

<*EXTERNAL*> PROCEDURE ilu_OutputCharacter (call: Call; i: Character);

<*EXTERNAL*> PROCEDURE ilu_OutputEnum (call: Call; i: EnumOrd);

<*EXTERNAL*> PROCEDURE ilu_OutputByte (call: Call; b: Byte);

<*EXTERNAL*> PROCEDURE ilu_OutputBoolean (call: Call; b: Bool);

<*EXTERNAL*>
  PROCEDURE ilu_OutputOptional (call: Call; optionalStatus: Bool);

<*EXTERNAL*> PROCEDURE ilu_OutputSequence (call      : Call;
                                           len, limit: Cardinal;
                                           optional, provided: Bool);

<*EXTERNAL*>
  PROCEDURE ilu_OutputSequenceMark (call: Call; extent: Cardinal);

<*EXTERNAL*> PROCEDURE ilu_OutputUnion (call              : Call;
                                        discriminator     : EnumOrd;
                                        optional, provided: Bool     );

<*EXTERNAL*>
  PROCEDURE ilu_OutputArray (call: Call; optional, provided: Bool);

<*EXTERNAL*>
  PROCEDURE ilu_OutputRecord (call: Call; optional, provided: Bool);


<*EXTERNAL*> PROCEDURE ilu_OutputString (call      : Call;
                                         s         : C_String;
                                         len, limit: Cardinal;
                                         optional  : Bool      );

<*EXTERNAL*> PROCEDURE ilu_OutputStringVec (call    : Call;
                                            s       : C_String;
                                            len     : Cardinal;
                                            optional: Bool      );

<*EXTERNAL*> PROCEDURE ilu_OutputWString (call      : Call;
                                          s         : WString;
                                          len, limit: Cardinal;
                                          buf : UNTRACED REF C_String;
                                          blen: UNTRACED REF Cardinal;
                                          optional: Bool);

<*EXTERNAL*>
  PROCEDURE ilu_OutputWStringVec (call    : Call;
                                  s       : WString;
                                  len     : Cardinal;
                                  buf     : UNTRACED REF C_String;
                                  blen    : UNTRACED REF Cardinal;
                                  optional: Bool                   );

<*EXTERNAL*> PROCEDURE ilu_OutputBytes (call      : Call;
                                        o         : Bytes;
                                        len, limit: Cardinal;
                                        optional  : Bool      );

<*EXTERNAL*> PROCEDURE ilu_OutputOpaque (call    : Call;
                                         o       : Bytes;
                                         len     : Cardinal;
                                         optional: Bool      );

<*EXTERNAL*>
  PROCEDURE ilu_OutputObjectID (call           : Call;
                                obj            : Object;
                                descriminator_p: Bool;
                                static_type    : ObjectType );


<*EXTERNAL*>
  PROCEDURE ilu_InputShortInteger (call: Call; VAR i: ShortInt);

<*EXTERNAL*> PROCEDURE ilu_InputInteger (call: Call; VAR i: Integer);

<*EXTERNAL*>
  PROCEDURE ilu_InputLongInteger (call: Call; VAR i: LongInt);

<*EXTERNAL*>
  PROCEDURE ilu_InputShortCardinal (call: Call; VAR i: ShortCard);

<*EXTERNAL*> PROCEDURE ilu_InputCardinal (call: Call; VAR i: Cardinal);

<*EXTERNAL*>
  PROCEDURE ilu_InputLongCardinal (call: Call; VAR i: LongCard);

<*EXTERNAL*>
  PROCEDURE ilu_InputShortReal (call: Call; VAR f: ShortReal);

<*EXTERNAL*> PROCEDURE ilu_InputReal (call: Call; VAR d: Real);

<*EXTERNAL*> PROCEDURE ilu_InputLongReal (call: Call; VAR f: LongReal);

<*EXTERNAL*> PROCEDURE ilu_InputEnum (call: Call; VAR i: ShortCard);

<*EXTERNAL*>
  PROCEDURE ilu_InputCharacter (call: Call; VAR i: Character);

<*EXTERNAL*> PROCEDURE ilu_InputByte (call: Call; VAR b: Byte);

<*EXTERNAL*> PROCEDURE ilu_InputBoolean (call: Call; VAR b: Bool);

<*EXTERNAL*>
  PROCEDURE ilu_InputOptional (call: Call; VAR optionalStatus: Bool);

<*EXTERNAL*> PROCEDURE ilu_InputSequence (    call    : Call;
                                          VAR len     : Cardinal;
                                              limit   : Cardinal;
                                              optional: Bool;
                                          VAR provided: Bool      );

<*EXTERNAL*>
  PROCEDURE ilu_InputSequenceMark (call: Call; extent: Cardinal);

<*EXTERNAL*> PROCEDURE ilu_InputUnion (    call         : Call;
                                       VAR discriminator: ShortCard;
                                           optional     : Bool;
                                       VAR provided     : Bool       );

<*EXTERNAL*> PROCEDURE ilu_InputArray (    call    : Call;
                                           optional: Bool;
                                       VAR provided: Bool  );

<*EXTERNAL*> PROCEDURE ilu_InputRecord (    call    : Call;
                                            optional: Bool;
                                        VAR provided: Bool  );

<*EXTERNAL*> PROCEDURE ilu_InputString (    call    : Call;
                                        VAR s       : C_String;
                                        VAR len     : Cardinal;
                                            limit   : Cardinal;
                                            optional: Bool      );
  (* Sets len to be the length of the fetched string.  Caller may pass
     non-null s, in which case the string will be stored there (with no
     length checking!); otherwise, callee will allocate just enough
     space (and return ownership to caller). *)

<*EXTERNAL*> PROCEDURE ilu_InputStringVec (    call    : Call;
                                           VAR s       : C_String;
                                               len     : Cardinal;
                                               optional: Bool      );

<*EXTERNAL*> PROCEDURE ilu_InputWString (    call    : Call;
                                         VAR s       : WString;
                                         VAR len     : Cardinal;
                                             limit   : Cardinal;
                                             optional: Bool      );

<*EXTERNAL*> PROCEDURE ilu_InputWStringVec (    call    : Call;
                                            VAR s       : WString;
                                                len     : Cardinal;
                                                optional: Bool      );

<*EXTERNAL*> PROCEDURE ilu_InputBytes (    call    : Call;
                                       VAR o       : Bytes;
                                       VAR len     : Cardinal;
                                           limit   : Cardinal;
                                           optional: Bool      );

<*EXTERNAL*> PROCEDURE ilu_InputOpaque (    call    : Call;
                                        VAR o       : Bytes;
                                            len     : Cardinal;
                                            optional: Bool      );

<*EXTERNAL*> PROCEDURE ilu_InputObjectID (    call           : Call;
                                          VAR o              : Object;
                                              discriminator_p: Bool;
                                          static_type: ObjectType);

<*EXTERNAL*> PROCEDURE ilu_RegisterClass (c: ObjectType);

<*EXTERNAL*> PROCEDURE ilu_GetGcCallbackClass (): ObjectType;

<*EXTERNAL*> PROCEDURE ilu_FindClassFromID (id: C_String): ObjectType;

<*EXTERNAL*>
  PROCEDURE ilu_RegisterLanguageSpecificObject (ko: Object; lso: ADDRESS);

<*EXTERNAL*> PROCEDURE ilu_GetLanguageSpecificObject (ko: Object):
  ADDRESS;


<*EXTERNAL*> PROCEDURE ilu_SizeOfShortInteger (call: Call; i: ShortInt):
  Cardinal;

<*EXTERNAL*> PROCEDURE ilu_SizeOfInteger (call: Call; i: Integer):
  Cardinal;

<*EXTERNAL*> PROCEDURE ilu_SizeOfLongInteger (call: Call; i: LongInt):
  Cardinal;

<*EXTERNAL*>
  PROCEDURE ilu_SizeOfShortCardinal (call: Call; i: ShortCard):
  Cardinal;

<*EXTERNAL*> PROCEDURE ilu_SizeOfCardinal (call: Call; i: Cardinal):
  Cardinal;

<*EXTERNAL*> PROCEDURE ilu_SizeOfLongCardinal (call: Call; i: LongCard):
  Cardinal;

<*EXTERNAL*> PROCEDURE ilu_shim_SizeOfShortReal (call: Call; d: Real):
  Cardinal;

<*EXTERNAL*> PROCEDURE ilu_SizeOfReal (call: Call; d: Real): Cardinal;

<*EXTERNAL*> PROCEDURE ilu_SizeOfLongReal (call: Call; d: LongReal):
  Cardinal;

<*EXTERNAL*> PROCEDURE ilu_SizeOfEnum (call: Call; i: ShortCard):
  Cardinal;

<*EXTERNAL*> PROCEDURE ilu_SizeOfCharacter (call: Call; i: Character):
  Cardinal;

<*EXTERNAL*> PROCEDURE ilu_SizeOfByte (call: Call; i: Byte): Cardinal;

<*EXTERNAL*> PROCEDURE ilu_SizeOfBoolean (call: Call; i: Bool):
  Cardinal;

<*EXTERNAL*>
  PROCEDURE ilu_SizeOfOptional (call: Call; optionalStatus: Bool):
  Cardinal;

<*EXTERNAL*> PROCEDURE ilu_SizeOfSequence (call      : Call;
                                           len, limit: Cardinal;
                                           optional, provided: Bool):
  Cardinal;

<*EXTERNAL*>
  PROCEDURE ilu_SizeOfUnion (call              : Call;
                             descriminator     : ShortCard;
                             optional, provided: Bool       ): Cardinal;

<*EXTERNAL*>
  PROCEDURE ilu_SizeOfArray (call: Call; optional, provided: Bool):
  Cardinal;

<*EXTERNAL*>
  PROCEDURE ilu_SizeOfRecord (call: Call; optional, provided: Bool):
  Cardinal;

<*EXTERNAL*> PROCEDURE ilu_SizeOfString (call    : Call;
                                         i       : C_String;
                                         l, limit: Cardinal;
                                         optional: Bool      ):
  Cardinal;

<*EXTERNAL*> PROCEDURE ilu_SizeOfStringVec (call    : Call;
                                            i       : C_String;
                                            l       : Cardinal;
                                            optional: Bool      ):
  Cardinal;

<*EXTERNAL*> PROCEDURE ilu_SizeOfWString (call    : Call;
                                          i       : WString;
                                          l, limit: Cardinal;
                                          buf : UNTRACED REF C_String;
                                          blen: UNTRACED REF Cardinal;
                                          optional: Bool): Cardinal;

<*EXTERNAL*>
  PROCEDURE ilu_SizeOfWStringVec (call    : Call;
                                  i       : WString;
                                  l       : Cardinal;
                                  buf     : UNTRACED REF C_String;
                                  blen    : UNTRACED REF Cardinal;
                                  optional: Bool                   ):
  Cardinal;

<*EXTERNAL*> PROCEDURE ilu_SizeOfBytes (call    : Call;
                                        o       : Bytes;
                                        l, limit: Cardinal;
                                        optional: Bool      ): Cardinal;

<*EXTERNAL*> PROCEDURE ilu_SizeOfOpaque (call    : Call;
                                         o       : Bytes;
                                         l       : Cardinal;
                                         optional: Bool      ):
  Cardinal;

<*EXTERNAL*>
  PROCEDURE ilu_SizeOfObjectID (call           : Call;
                                obj            : Object;
                                discriminator_p: Bool;
                                static_type    : ObjectType ): Cardinal;


TYPE ObjectNoter = PROCEDURE (obj: Object; vi: Ctypes.int);

<*EXTERNAL*> PROCEDURE ilu_SetNoter (n: ObjectNoter);

<*EXTERNAL*> PROCEDURE ilu_VeryInterested (obj: Object): Bool;

<*EXTERNAL*> PROCEDURE ilu_SetGcClient (interest: Object);

END IluKernel.
