;;; -*- Mode:Lisp; Package: ILU; Syntax:COMMON-LISP; Base:10 -*-
#|
Copyright (c) 1991, 1992, 1993, 1994 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.

$Id: ilu-marshalling.lisp,v 1.2 1994/04/30 07:50:30 janssen Exp $
|#

(cl:in-package :ilu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; General locking rules (see src/runtime/kernel/locking.txt):
;;
;;   Size-computing routines:  L1, L2, Main unconstrained
;;   Marshalling and un-marshalling routines:
;;                    L2 >= {call's connection's callmu, iomu}
;;                    L1, Main unconstrained
;;
;;   (Objects are special.  See end of file.)
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Primitive output functions
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun null-write (call n optional-p)
  (declare (ignore call n optional-p))
  ;; do nothing
  )

(defmacro define-primitive-writer (writer type c-fn)
  `(progn
     (define-c-function ,(intern c-fn) "primitive writer fn" ,c-fn
       (:ilu-call ,type) :fixnum :inline t)
     (defmacro ,writer (call object optional-p)
       `(let ((call ,call)		; Have to bind all these so that we can
	      (object ,object)		; ignore one & preserve order of eval.
	      (optional-p ,optional-p))
	  (declare (ignore optional-p))
	  (,',(intern c-fn) call object)))))

(define-primitive-writer integer-write :integer "ilu_OutputInteger")

(define-primitive-writer short-integer-write :fixnum "ilu_OutputShortInteger")

(define-primitive-writer cardinal-write :cardinal "ilu_OutputCardinal")

(define-primitive-writer short-cardinal-write
    :fixnum "ilu_OutputShortCardinal")

(define-c-function ilu_long-cardinal-write
    "Write a 32-bit unsigned value"
  "ilu_OutputLongCardinal"
  (:ilu-call :long-cardinal) :fixnum :inline t)

(defmacro long-cardinal-write (call value optional-p)
  (declare (ignore optional-p))
  `(let ((call ,call)
	 (val ,value))
     (let ((valarray (make-array 2 :element-type '(unsigned-byte 32)
				 :initial-contents
				 `(,@(logand val #xFFFFFFFF) ,@(logand (ash val -32) 0xFFFFFFFF)))))
       (ilu_long-cardinal-write call valarray))))

(define-c-function ilu_long-integer-write
    "Write a 32 bit signed value"
  "ilu_OutputLongInteger"
  (:ilu-call :long-integer) :fixnum :inline t)

(defmacro long-integer-write (call value optional-p)
  (declare (ignore optional-p))
  `(let ((call ,call)
	 (val ,value))
     (let ((valarray (make-array 2 :element-type '(signed-byte 32)
				 :initial-contents
				 `(,@(logand val #xFFFFFFFF) ,@(logand (ash val -32) 0xFFFFFFFF)))))
       (ilu_long-integer-write call valarray))))

(define-primitive-writer enumeration-entry-write
    :fixnum "ilu_OutputEnum")

(define-primitive-writer real-write :double-float "ilu_OutputReal")

(define-primitive-writer long-real-write :double-float "ilu_OutputLongReal")

#+lisp-understands-ansi-c-parameter-passing
(define-primitive-writer short-real-write :single-float "ilu_OutputShortReal")

#-lisp-understands-ansi-c-parameter-passing
(progn
  (define-c-function ilulisp_short-real-write "Output a 32-bit real number"
    "ilulisp_KandROutputShortReal" (:ilu-call :single-float) :void :inline t)
  (defmacro short-real-write (call value optional-p)
    (declare (ignore optional-p))
    `(ilulisp_short-real-write ,call ,value)))

(define-primitive-writer byte-write :fixnum "ilu_OutputByte")

(define-c-function ilu_boolean-write
    "Write a boolean value"
  "ilu_OutputBoolean"
  (:ilu-call :fixnum) :fixnum :inline t)

(defmacro boolean-write (call c optional-p)
  `(let ((call ,call)
	 (c ,c)
	 (optional-p ,optional-p))
     (declare (ignore optional-p))
     (ilu_boolean-write call (if c 1 0))))

(define-c-function ilu_character-write 
    "Write a character (16 bits)"
    "ilu_OutputCharacter"
  (:ilu-call :short-cardinal) :fixnum :inline t)
(defmacro character-write (call c optional-p)
  (declare (ignore optional-p))
  `(ilu_character-write ,call (char-code ,c)))

(defmacro short-character-write (call c optional-p)
  (declare (ignore optional-p))
  `(byte-write ,call (char-code ,c) optional-p))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Primitive input functions
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro define-primitive-reader (reader type c-fn)
  `(progn
     (define-c-function ,(intern c-fn) "primitive reader fn" ,c-fn (:ilu-call) ,type :inline t)
     (defmacro ,reader (call optional-p)
       `(let ((call ,call)		; Have to bind both so that we can
	      (optional-p ,optional-p))	; ignore one & preserve order of eval.
	  (declare (ignore optional-p))
	  (,',(intern c-fn) call)))))

(defun null-read (call optional-p)
  (declare (ignore call optional-p))
  nil)

(defmacro character-read (call optional-p)
  `(construct-unicode-character-from-code
    (short-cardinal-read ,call ,optional-p)))

(defmacro short-character-read (call optional-p)
  `(construct-iso-latin-1-character-from-code (byte-read ,call ,optional-p)))


(define-primitive-reader integer-read :integer "ilulisp_InputInteger")

(define-c-function ilulisp_cardinal-read
    "Read a 32-bit value"
  "ilulisp_InputCardinal"
  (:ilu-call) :cardinal :inline t)

(defun cardinal-read (call optional-p)	; hack around franz unsigned bug
  (declare (ignore optional-p))
  (let ((cardinal (ilulisp_cardinal-read call)))
    (if (minusp cardinal)
	(+ cardinal (expt 2 32))
      cardinal)))

(define-c-function ilu_long-cardinal-read
    "Read a 32 bit unsigned value"
    "ilu_InputLongCardinal"
  (:ilu-call :long-cardinal) :fixnum :inline t)

(defun long-cardinal-read (call optional-p)
  (declare (ignore optional-p))
  (let ((val (make-array 2 :element-type '(unsigned-byte 32))))
    (when (ilu_long-cardinal-read call val)
      (+ (aref val 0) (ash (aref val 1) 32)))))    

(define-c-function ilu_long-integer-read
    "Read a 64 bit integer"
    "ilu_InputLongInteger"
  (:ilu-call :long-integer) :fixnum :inline t)

(defun long-integer-read (call optional-p)
  (declare (ignore optional-p))
  (let ((val (make-array 2 :element-type '(signed-byte 32))))
    (when (ilu_long-integer-read call val)
      (+ (aref val 0) (ash (aref val 1) 32)))))    

(define-c-function ilu_boolean-read
    "Read a boolean value"
    "ilulisp_InputBoolean"
  (:ilu-call) :fixnum :inline t)

(defmacro boolean-read (call optional-p)
  (declare (ignore optional-p))
  `(not (zerop (ilu_boolean-read ,call))))

(define-primitive-reader short-integer-read
    :fixnum "ilulisp_InputShortInteger")

(define-primitive-reader short-cardinal-read
    :fixnum "ilulisp_InputShortCardinal")

(define-primitive-reader enumeration-entry-read
    :fixnum "ilulisp_InputEnumCode")

(define-primitive-reader real-read :double-float "ilulisp_InputReal")

(define-primitive-reader long-real-read :double-float "ilulisp_InputLongReal")

#+lisp-understands-ansi-c-parameter-passing
(define-primitive-reader short-real-read
    :single-float "ilulisp_ANSIInputShortReal")

#-lisp-understands-ansi-c-parameter-passing
(define-primitive-reader short-real-read
    :single-float "ilulisp_KandRInputShortReal")

(define-primitive-reader byte-read :fixnum "ilulisp_InputByte")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Primitive size functions
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro define-primitive-sizer (sizer type c-fn)
  `(progn
     (define-c-function ,(intern c-fn) "primitive sizer fn" ,c-fn
       (:ilu-call ,type) :fixnum)
     (defmacro ,sizer (call object optional-p)
       `(let ((call ,call)		; Have to bind all these so that we can
	      (object ,object)		; ignore one & preserve order of eval.
	      (optional-p ,optional-p))
	  (declare (ignore optional-p))
	  (,',(intern c-fn) call object)))))

(defun null-size (call n optional-p)
  (declare (ignore call n optional-p))
  0)

(define-primitive-sizer integer-size :integer "ilu_SizeOfInteger")

(define-primitive-sizer short-integer-size :short-integer "ilu_SizeOfShortInteger")

(define-primitive-sizer cardinal-size :cardinal "ilu_SizeOfCardinal")

(define-primitive-sizer short-cardinal-size :short-cardinal "ilu_SizeOfShortCardinal")

(define-primitive-sizer enumeration-entry-size :fixnum "ilu_SizeOfEnum")

(define-primitive-sizer real-size :double-float "ilu_SizeOfReal")

(define-primitive-sizer long-real-size :double-float "ilu_SizeOfLongReal")

#+lisp-understands-ansi-c-parameter-passing
(define-primitive-sizer short-real-size :single-float "ilu_SizeOfShortReal")

#-lisp-understands-ansi-c-parameter-passing
(progn
  (define-c-function ilulisp_short-real-size "SizeOf a 32-bit real number"
    "ilulisp_KandRSizeOfShortReal" (:ilu-call :single-float) :fixnum :inline t)
  (defmacro short-real-size (call value optional-p)
    (declare (ignore optional-p))
    `(ilulisp_short-real-size ,call ,value)))

(define-primitive-sizer byte-size :fixnum "ilu_SizeOfByte")

(define-c-function ilu_boolean-size
    "Take the size of a boolean"
    "ilu_SizeOfBoolean"
  (:ilu-call :fixnum) :fixnum :inline t)

(defmacro boolean-size (call b optional-p)
  `(let ((call ,call)
	 (b ,b)
	 (optional-p ,optional-p))
     (declare (ignore optional-p))
     (ilu_boolean-size call (if b 1 0))))

(define-c-function ilu_long-integer-size
    "Take the size of a 64 bit int"
    "ilu_SizeOfLongInteger"
  (:ilu-call :long-integer) :fixnum :inline t)

(defmacro long-integer-size (call value optional-p)
  (declare (ignore optional-p))
  `(let ((call ,call)
	 (val ,value))
     (let ((valarray (make-array 2 :element-type '(signed-byte 32)
				 :initial-contents
				 `(,@(logand val #xFFFFFFFF) ,@(logand (ash val -32) 0xFFFFFFFF)))))
       (ilu_long-integer-size call valarray))))

(defmacro long-cardinal-size (call value optional-p)
  (declare (ignore optional-p))
  `(let ((call ,call)
	 (val ,value))
     (let ((valarray (make-array 2 :element-type '(signed-byte 32)
				 :initial-contents
				 `(,@(logand val #xFFFFFFFF) ,@(logand (ash val -32) 0xFFFFFFFF)))))
       (ilu_long-cardinal-size call valarray))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; string (short character sequence) functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-c-function ilu_output-string
    "Output a string"
    "ilulisp_OutputString"
  (:ilu-call :string :cardinal :cardinal :fixnum :fixnum) :fixnum :inline t)

(defun string-write (call s limit opt)
  (ilu_output-string call (if (and opt (null s)) "" s)
		     (if s (length s) 0) limit (if opt 1 0) (if s 1 0)))

(define-c-function ilulisp_input-string
    "Input a string"
    "ilulisp_InputString"
  (:ilu-call :cardinal :fixnum) :pointer :inline t)

(defun string-read (call limit opt)
  (let* ((c-string (ilulisp_input-string call limit (if opt 1 0)))
	 (lisp-string (char*-to-string c-string)))
    (free c-string)
    lisp-string))

(define-c-function ilu_size-of-string
    "Take the size of a string"
    "ilu_SizeOfString"
  (:ilu-call :string :cardinal :cardinal :fixnum) :fixnum)

(defmacro string-size (call s limit opt)
  `(let ((call ,call)
	 (opt ,opt)
	 (s ,s))
     (declare (simple-string s))
     (ilu_size-of-string call s (length s) ,limit (if opt 1 0))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; short character vector functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-c-function ilu_output-string-vec
    "Output a vector of short character"
    "ilulisp_OutputStringVec"
  (:ilu-call :string :cardinal :fixnum :fixnum) :fixnum :inline t)

(defmacro char-vector-write (call s len opt)
  `(ilu_output-string-vec ,call (or ,s "") (if ,s ,len 0) ,opt (if ,s 1 0)))

(define-c-function ilulisp_input-char-vector
    "Input a vector of short character"
    "ilulisp_InputCharacterVector"
  (:ilu-call :cardinal :fixnum) :pointer :inline t)

(defun char-vector-read (call length opt)
  (let ((c-string (ilulisp_input-char-vector call length (if opt 1 0))))
    (when (not (= c-string 0))
      (let ((string (make-string length)))
	(bytencopy string c-string length)
	(free c-string)
	string))))

(define-c-function ilu_char-vector-size
    "Size a vector of short character"
    "ilulisp_SizeOfStringVec"
  (:ilu-call :string :cardinal :fixnum :fixnum) :fixnum)

(defmacro char-vector-size (call s opt)
  `(let ((call ,call)
	 (opt ,opt)
	 (s ,s))
     (declare (simple-string s))
     (ilu_char-vector-size call (if s s "")
			   (if s (length s) 0)
			   (if opt 1 0)
			   (if s 1 0))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; character sequence functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-c-function ilu_output-wide-string
    "Output UNICODE string"
  "ilulisp_OutputWString" (:ilu-call :wide-string :cardinal :cardinal :fixnum :fixnum) :fixnum :inline t)

(defun unicode-string-write (call s limit opt)
  (if (null s)
      (ilu_output-wide-string call "" 0 limit (if opt 1 0) 0)
    (let ((v (map '(simple-array (unsigned-byte 16)) #'char-code s)))
      (ilu_output-wide-string call v (length v) limit (if opt 1 0) 1))))

(define-c-function ilulisp_input-wide-string
    "Input unicode string"
  "ilulisp_InputWString" (:ilu-call :cardinal :fixnum) :pointer :inline t)

(defun unicode-string-read (call limit opt)
  (let* ((c-string (ilulisp_input-wide-string call limit (if opt 1 0)))
	 (lisp-string (char*-to-string c-string)))
    (free c-string)
    lisp-string))

(define-c-function ilu_sizeof-wide-string
    "Size UNICODE string"
    "ilulisp_SizeOfWString"
  (:ilu-call :wide-string :cardinal :cardinal :fixnum :fixnum) :fixnum)

(defun unicode-string-size (call s limit opt)
  (if (null s)
      (ilu_sizeof-wide-string call "" 0 limit (if opt 1 0) 0)
    (let ((v (map '(simple-array (unsigned-byte 16)) #'char-code s)))
      (ilu_sizeof-wide-string call v (length v) limit (if opt 1 0) 1))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; character vector functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-c-function ilu_output-wide-char-vector
    "Output unicode vector"
    "ilulisp_OutputWStringVec"
  (:ilu-call :wide-string :cardinal :fixnum :fixnum) :fixnum :inline t)

(defun unicode-vector-write (call s opt)
  (if (null s)
      (ilu_output-wide-char-vector call "" 0 (if opt 1 0) 0)
    (let ((v (map '(simple-array (unsigned-byte 16)) #'char-code s)))
      (ilu_output-wide-char-vector call v (length v) (if opt 1 0) 1))))

(define-c-function ilulisp_input-wide-char-vector
    "Input Unicode vector"
    "ilulisp_InputWStringVec"
  (:ilu-call :cardinal :fixnum) :pointer :inline t)

(defun unicode-vector-read (call length opt)
  (let ((c-string (ilulisp_input-wide-char-vector call length (if opt 1 0))))
    (when (not (= c-string 0))
      (let ((string (make-string length)))
	(bytencopy string c-string length)
	(free c-string)
	string))))

(define-c-function ilu_sizeof-wide-char-vector
    "Size unicode vector"
    "ilulisp_SizeOfWStringVec"
  (:ilu-call :wide-string :fixnum :fixnum :fixnum) :fixnum)

(defun unicode-vector-size (call s opt)
  (if (null s)
      (ilu_sizeof-wide-char-vector call "" 0 (if opt 1 0) 0)
    (let ((v (map '(simple-array (unsigned-byte 16)) #'char-code s)))
      (ilu_sizeof-wide-char-vector call v (length v) (if opt 1 0) 1))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; byte sequence functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-c-function ilu_output-bytes
    "Output byte sequence"
    "ilulisp_OutputByteSequence"
  (:ilu-call :byte-sequence :cardinal :cardinal :fixnum :fixnum) :fixnum :inline t)

(defmacro byte-sequence-write (call v limit opt)
  `(let ((call ,call)
	 (opt ,opt)
	 (v ,v))
     (declare (type (simple-array (unsigned-byte 8) (*)) v))
     (ilu_output-bytes call (or v "") (if v (length v) 0) ,limit (if opt 1 0)
		       (if v 1 0))))

(define-c-function ilulisp_input-byte-sequence "" "ilulisp_InputByteSequence"
  (:ilu-call :cardinal-pointer :cardinal :fixnum) :pointer :inline t)

(defun byte-sequence-read (call limit opt)
  (let* ((s (make-array 1 :element-type '(unsigned-byte 32)))
	 (p (ilulisp_input-byte-sequence call s limit (if opt 1 0))))
    (when (not (= p 0))
      (let ((seq (make-array (aref s 0) :element-type '(unsigned-byte 8))))
	(declare (dynamic-extent s))
	(bytencopy seq p (aref s 0))
	(free p)
	seq))))

(define-c-function ilu_size-of-bytes "" "ilulisp_SizeOfByteSequence"
  (:ilu-call :byte-sequence :cardinal :cardinal :fixnum :fixnum) :fixnum)

(defmacro byte-sequence-size (call v limit opt)
  `(let ((call ,call)
	 (opt ,opt)
	 (v ,v))
     (declare (type (simple-array (unsigned-byte 8) (*)) v))
     (ilu_size-of-bytes call (if v v "")
			(if v (length v) 0)
			,limit
			(if opt 1 0)
			(if v 1 0))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; byte vector functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-c-function ilu_byte-vector-write "" "ilulisp_OutputByteVector"
  (:ilu-call :byte-sequence :cardinal) :fixnum :inline t)

(defmacro byte-vector-write (call v)
  `(let ((call ,call)
	 (v ,v))
     (declare (type (simple-array (unsigned-byte 8) (*)) v))
     (ilu_byte-vector-write call (or v "") (if v (length v) 0))))

(define-c-function ilu_byte-vector-read "" "ilulisp_InputByteVector"
  (:ilu-call :cardinal) :pointer :inline t)

(defun byte-vector-read (call len)
  (let ((v (ilu_byte-vector-read call len)))
    (when (not (= v 0))
      (let ((seq (make-array len :element-type '(unsigned-byte 8))))
	(declare (dynamic-extent s))
	(bytencopy seq v len)
	(free v)
	seq))))

(define-c-function ilu_byte-vector-size "" "ilulisp_SizeOfByteVector"
  (:ilu-call :byte-sequence :cardinal) :fixnum)

(defmacro byte-vector-size (call v)
  `(let ((call ,call)
	 (v ,v))
     (declare (type (simple-array (unsigned-byte 8) (*)) v))
     (ilu_byte-vector-size call (if v v "") (if v (length v) 0))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; sequence functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-c-function ilu_sequence-write "" "ilu_OutputSequence"
  (:ilu-call :cardinal :cardinal :fixnum :fixnum) :fixnum :inline t)

(defmacro sequence-write (call value limit opt)
  (let ((val (gensym)))
    `(let ((,val ,value))
       (ilu_sequence-write ,call (length ,val) ,limit (if ,opt 1 0) (if ,val 1 0)))))

(define-c-function ilulisp_input-sequence "" "ilulisp_InputSequence"
  (:ilu-call :cardinal :fixnum) :integer :inline t)

(defmacro sequence-read (call limit opt)
  (let ((i (gensym)))
    `(let ((,i (ilulisp_input-sequence ,call ,limit (if ,opt 1 0))))
       (when (>= ,i 0)
	 ,i))))

(define-c-function ilu_sequence-size "" "ilu_SizeOfSequence"
  (:ilu-call :cardinal :cardinal :fixnum :fixnum) :fixnum)

(defmacro sequence-size (call value limit opt)
  (let ((val (gensym)))
    `(let ((,val ,value))
       (ilu_sequence-size ,call (length ,val) ,limit (if ,opt 1 0) (if ,val 1 0)))))

(define-c-function sequence-end "" "ilu_EndSequence"
  (:ilu-call) :fixnum :inline t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; union functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-c-function ilu_union-write "" "ilu_OutputUnion"
  (:ilu-call :short-cardinal :fixnum :fixnum) :fixnum :inline t)

(defmacro union-write (call index opt)
  (let ((i (gensym)))
    `(let ((,i ,index))
       (ilu_union-write ,call (or ,i 0) (if ,opt 1 0) (if ,i 1 0)))))

(define-c-function ilulisp_input-union "" "ilulisp_InputUnion"
  (:ilu-call :fixnum) :fixnum :inline t)

(defmacro union-read (call opt)
  (let ((i (gensym)))
    `(let ((,i (ilulisp_input-union ,call (if ,opt 1 0))))
       (when (>= ,i 0)
	 ,i))))

(define-c-function ilu_union-size "" "ilu_SizeOfUnion"
  (:ilu-call :short-cardinal :fixnum :fixnum) :fixnum)

(defmacro union-size (call index opt)
  (let ((i (gensym)))
    `(let ((,i ,index))
       (ilu_union-size ,call (or ,i 0) (if ,opt 1 0) (if ,i 1 0)))))

(define-c-function union-end "" "ilu_EndUnion"
  (:ilu-call) :fixnum :inline t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; optional functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-c-function ilu_optional-write "" "ilu_OutputOptional"
  (:ilu-call :fixnum) :fixnum :inline t)

(defmacro optional-write (call opt)
  `(ilu_optional-write ,call (if ,opt 1 0)))

(define-c-function ilulisp_input-optional "" "ilulisp_InputOptional"
  (:ilu-call) :fixnum :inline t)

(defmacro optional-read (call)
  `(not (zerop (ilulisp_input-optional ,call))))

(define-c-function ilu_optional-size "" "ilu_SizeOfOptional"
  (:ilu-call :fixnum) :fixnum)

(defmacro optional-size (call index)
  `(ilu_optional-size ,call (if ,index 1 0)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; record functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-c-function ilu_record-write "" "ilu_OutputRecord"
  (:ilu-call :fixnum :fixnum) :fixnum :inline t)

(defmacro record-write (call record opt)
  `(ilu_record-write ,call (if ,opt 1 0) (if ,record 1 0)))

(define-c-function ilu_record-read "" "ilulisp_InputRecord"
  (:ilu-call :fixnum) :cardinal :inline t)

(defmacro record-read (call optional-p)
  `(= (ilu_record-read ,call (if ,optional-p 1 0)) 1))

(define-c-function ilu_record-size "" "ilu_SizeOfRecord"
  (:ilu-call :fixnum :fixnum) :fixnum)

(defmacro record-size (call record opt)
  `(ilu_record-size ,call (if ,opt 1 0) (if ,record 1 0)))

(define-c-function record-end "" "ilu_EndRecord"
  (:ilu-call) :fixnum :inline t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; array functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-c-function ilu_array-write "" "ilu_OutputArray"
  (:ilu-call :fixnum :fixnum) :fixnum :inline t)

(defmacro array-write (call array opt)
  `(ilu_array-write ,call (if ,opt 1 0) (if ,array 1 0)))

(define-c-function ilu_array-read "" "ilulisp_InputArray"
  (:ilu-call :fixnum) :cardinal :inline t)

(defmacro array-read (call optional-p)
  `(= (ilu_array-read ,call (if ,optional-p 1 0)) 1))

(define-c-function ilu_array-size "" "ilu_SizeOfArray"
  (:ilu-call :fixnum :fixnum) :fixnum)

(defmacro array-size (call array opt)
  `(ilu_array-size ,call (if ,opt 1 0) (if ,array 1 0)))

(define-c-function array-end "" "ilu_EndArray"
  (:ilu-call) :fixnum :inline t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; object I/O functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;  before: L1 = {},
;;  after:  *o!=NULL => Inside(*o's server, static_type);
;;  after:  *o==NULL => L1 = {};
;;  forall conn: (L2 >= {conn.iomu}) => (L2 >= {conn.callmu});
;;  L2 >= {call's connection's callmu, iomu};
;;  Main otherwise unconstrained
(define-c-function ilulisp_input-object-id "" "ilulisp_InputObjectID"
  (:ilu-call :ilu-object-pointer :pointer :fixnum) :ilu-object :inline t)

;;  before: Inside(s, cl);
;;  after:				   L1 disjoint {cmu, s};
;;  after: cl collectible	        => L1  not >=  {gcmu};
;;  after: cl collectible & s surrogate => Main Invariant holds;
;;  where s = h's server and cl = h's type.
;;  (We don't really need to hold cmu for surrogate or non-collectible
;;   objects, but this is convenient because ilu_Enter/ExitServer can
;;   be used.)
(define-c-function object-id-write "" "ilu_OutputObjectID"
  (:ilu-call :ilu-object :fixnum :pointer) :fixnum :inline t)

;; L1 >= {obj's server}
(define-c-function ilu_object-id-size "" "ilu_SizeOfObjectID"
  (:ilu-call :ilu-object :fixnum :pointer) :fixnum :inline t)
