;;; -*- Mode:Lisp; Package: ILU; Syntax:COMMON-LISP; Base:10 -*-
#|
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.

$Id: ilu-macros.lisp,v 1.37 1994/04/28 02:49:56 janssen Exp $
|#

(cl:in-package :ilu)

;;;; first, some generic code generators for operations on ILU types

(defmacro remove-nils (&rest body)
  `(remove nil ,@body :test #'eq))

(defun make-primitive-name (format-string type-name)
  (intern (funcall #'format nil format-string type-name)
	  (find-package "ILU")))

(defun make-constructed-name (format-string type-name)
  (let* ((package (symbol-package type-name))
	 (name
	  (intern (funcall #'format nil format-string type-name) package)))
    (unless (eq (symbol-package name) package)
      (warn "~S and ~S are not in the same package." type-name name))
    name))


(defun make-read-form (call type)
  (destructuring-bind (meta-type name &optional optional-p) type
    (ecase meta-type
      (:primitive
       `(,(make-primitive-name "~A-READ" name) ,call ,optional-p))
      (:constructed
       `(,(make-constructed-name "~A-read" name) ,call ,optional-p))
      (:object
       `(object-read (find-class ',name) ,call ',name)))))

(defun make-write-form (call type obj)
  (destructuring-bind (meta-type name &optional optional-p) type
    (ecase meta-type
      (:primitive
       `(,(make-primitive-name "~A-WRITE" name) ,call ,obj ,optional-p))
      (:constructed
       `(,(make-constructed-name "~A-write" name) ,call ,obj ,optional-p))
      (:object
       `(object-write ,call ,obj ',name)))))

(defun make-size-form (call type obj)
  (destructuring-bind (meta-type name &optional optional-p) type
    (ecase meta-type
      (:primitive
       `(,(make-primitive-name "~A-SIZE" name) ,call ,obj ,optional-p))
      (:constructed
       `(,(make-constructed-name "~A-size" name) ,call ,obj ,optional-p))
      (:object
       `(object-size ,call ,obj ',name)))))



(defvar *primitive-types*
    ;;<ILU name>     <Lisp name>              <prototype form>
    '((:null          null                    nil)
      (:character     character               #\space)
      (:short-character character             #\space)
      (:cardinal      (unsigned-byte 32)      0)
      (:short-cardinal(unsigned-byte 16)      0)
      (:long-cardinal (unsigned-byte 64)      0)
      (:integer       (signed-byte 32)        0)
      (:short-integer (signed-byte 16)        0)
      (:long-integer  (signed-byte 64)	      0)
      (:boolean	      (or nil t)	      't)
      (:float         double-float            0.0D0)
      (:short-float   single-float            0.0)
      (:long-float    double-float	      0.0D0)
      (:byte          (unsigned-byte 8)       0)))
                 
(defun type-prototype-form (type)
  (destructuring-bind (meta-type name &optional optional-p) type
    (if optional-p
	nil
      (case meta-type
	(:primitive
	 (third (assoc name *primitive-types*)))
	(:constructed
	 `(get ',name 'prototype))
	(:object
	 `(make-instance ',name))))))

(defun type-lisp-type (type)
  (destructuring-bind (meta-type name &optional optional-p) type
    (let ((lisp-type (case meta-type
		       (:primitive (second (assoc name *primitive-types*)))
		       ((:constructed :object) (second type)))))
      (if optional-p `(or null ,lisp-type) lisp-type))))


;;;; DEFINE-TYPE: emits a DEFTYPE

(defmacro define-primitive-type (name type)
  `(deftype ,name () ',(type-lisp-type type)))

;;;; constructed types

;;;; DEFINE-RECORD: emits a DEFSTRUCT and appropriate read, write & size code

(defun make-name (format-string &rest format-args)
  (let ((name (apply #'format nil format-string format-args)))
    (shadow (make-symbol name))		; shadow before intern at macroexpand
    (intern name)))			; time to avoid any conflicts

(defun export-names (names)
  `(eval-when (compile eval load)	; shadow before intern at load time too
     (shadow ',(mapcar #'(lambda (s) (make-symbol (symbol-name s))) names))
     (export ',names)))

(defmacro define-record-type (name &rest slots)
  (let* ((name-string (symbol-name name))
	 (slot-names (mapcar #'first slots))
	 (slot-types (mapcar #'second slots))
	 (accessors
	  (mapcar #'(lambda (name) (make-name "~A-~A" name-string name))
		  slot-names))
	 (constructor (make-name "MAKE-~A" name-string))
	 (predicate (make-name "~A-P" name-string)))
    `(progn
       ,(export-names (list* constructor predicate accessors))
       (defstruct ,name
	 ,@(mapcar
	    #'(lambda (name type)
		`(,name ,(type-prototype-form type)
			:type ,(type-lisp-type type)))
	    slot-names slot-types))
       (setf (get ',name 'prototype) (,constructor)))))

(defmacro define-record (name &rest slots)
  (let* ((name-string (symbol-name name))
	 (slot-names (mapcar #'first slots))
	 (slot-keys (mapcar
		     #'(lambda (name)
			 (intern (symbol-name name) (find-package :keyword)))
		     slot-names))
	 (slot-types (mapcar #'second slots))
	 (accessors
	  (mapcar #'(lambda (name) (make-name "~A-~A" name-string name))
		  slot-names))
	 (constructor (make-name "MAKE-~A" name-string))
	 (read-fn (intern (format nil "~A-read" name-string)))
	 (write-fn (intern (format nil "~A-write" name-string)))
	 (size-fn (intern (format nil "~A-size" name-string))))
    `(progn
       (defun ,read-fn (call optional-p)
	 (prog1 (when (record-read call optional-p)
		  (,constructor
		   ,@(mapcan #'(lambda (key type)
			  `(,key ,(make-read-form 'call type)))
			     slot-keys slot-types)))
	   (record-end call)))
       (defun ,write-fn (call record optional-p)
	 (record-write call record optional-p)
	 (unless (and optional-p (null record))
	   ,@(mapcar
	      #'(lambda (type accessor)
		  (make-write-form 'call type `(,accessor record)))
	      slot-types accessors))
	 (record-end call)
	 record)
       (defun ,size-fn (call record optional-p)
	 (prog1 (+ (record-size call record optional-p)
		   (if (and optional-p (null record))
		       0
		     (+ ,@(mapcar
			   #'(lambda (type accessor)
			       (make-size-form 'call type `(,accessor record)))
			   slot-types accessors))))
	   (record-end call))))))



;;;; DEFINE-ENUMERATION: enumerations are symbols, mapped to numbers on wire

(defmacro define-enumeration-type (name &rest specs)
  (let* ((name-string (symbol-name name))
	 (names (mapcar #'first specs))
	 (values (mapcar #'second specs))
	 (constant (intern (format nil "+~A-ENUMERATION+" name-string))))
    `(progn
       (defconstant ,constant ',(mapcar #'cons values names))
       (deftype ,name () '(member ,@names))
       (setf (get ',name 'prototype) ',(first names)))))

(defmacro define-enumeration (name &rest specs)
  (declare (ignore specs))
  (let* ((name-string (symbol-name name))
	 (constant (intern (format nil "+~A-ENUMERATION+" name-string)))
	 (read-fn (intern (format nil "~A-read" name-string)))
	 (write-fn (intern (format nil "~A-write" name-string)))
	 (size-fn (intern (format nil "~A-size" name-string))))
    `(progn
       (defun ,read-fn (call optional-p)
	 (let ((key (enumeration-entry-read call optional-p)))
	   (when key
	     (cdr (assoc key ,constant :test #'=)))))
       (defun ,write-fn (call symbol optional-p)
	 (enumeration-entry-write
	  call
	  (dolist (entry ,constant
		    (error "~A not an valid ~A value" symbol ',name))
	    (when (eq (cdr entry) symbol) (return (car entry))))
	  optional-p)
	 symbol)
       (defun ,size-fn (call symbol optional-p)
	 (enumeration-entry-size
	  call
	  (dolist (entry ,constant
		    (error "~A not an valid ~A value" symbol ',name))
	    (when (eq (cdr entry) symbol) (return (car entry))))
	  optional-p)))))




;;;; DEFINE-SEQUENCE: sequences are represented by lists

(defmacro define-sequence-type (name type limit)
  (declare (ignore limit))
  `(progn
     ,@(cond 
	((and (eq (first type) :primitive)
	      (eq (second type) :short-character))
	 `((deftype ,name () 'simple-string)
	   (setf (get ',name 'prototype) "")))
	((and (eq (first type) :primitive)
	      (eq (second type) :character))
	 `((deftype ,name () 'simple-string)
	   (setf (get ',name 'prototype) "")))
	((and (eq (first type) :primitive) (eq (second type) :byte))
	 `((deftype ,name () '(simple-array (unsigned-byte 8) (*)))
	   (setf (get ',name 'prototype)
	     (make-array 0 :element-type '(unsigned-byte 8)))))
	(t `((deftype ,name () 'list)
	     (setf (get ',name 'prototype) '()))))))

(defmacro define-sequence (name type limit)
  (let* ((name-string (symbol-name name))
	 (read-fn (intern (format nil "~A-read" name-string)))
	 (write-fn (intern (format nil "~A-write" name-string)))
	 (size-fn (intern (format nil "~A-size" name-string))))
    `(progn
       ,@(cond 
	  ((and (eq (first type) :primitive)
		(eq (second type) :short-character))
	   `((defun ,read-fn (call optional-p)
	       (string-read call ,limit optional-p))
	     (defun ,write-fn (call s optional-p)
	       (string-write call s ,limit optional-p))
	     (defun ,size-fn (call s optional-p)
	       (string-size call s ,limit optional-p))))
	  ((and (eq (first type) :primitive)
		(eq (second type) :character))
	   `((defun ,read-fn (call optional-p)
	       (unicode-string-read call ,limit optional-p))
	     (defun ,write-fn (call s optional-p)
	       (unicode-string-write call s ,limit optional-p))
	     (defun ,size-fn (call s optional-p)
	       (unicode-string-size call s ,limit optional-p))))
	  ((and (eq (first type) :primitive) (eq (second type) :byte))
	   `((defun ,read-fn (call optional-p)
	       (byte-sequence-read call ,limit optional-p))
	     (defun ,write-fn (call v optional-p)
	       (byte-sequence-write call v ,limit optional-p))
	     (defun ,size-fn (call v optional-p)
	       (byte-sequence-size call v ,limit optional-p))))
	  (t `((defun ,read-fn (call optional-p)
		 (let ((value '())
		       (count (sequence-read call ,limit optional-p)))
		   (prog1 (when count
			    (dotimes (i count (nreverse value))
			      (push ,(make-read-form 'call type) value)))
		     (sequence-end call))))
	       (defun ,write-fn (call list optional-p)
		 (sequence-write call list ,limit optional-p)
		 (dolist (entry list)
		   ,(make-write-form 'call type 'entry))
		 (sequence-end call)
		 list)
	       (defun ,size-fn (call list optional-p)
		 (let ((size (sequence-size call list ,limit optional-p)))
		   (prog1 (dolist (entry list size)
			    (incf size ,(make-size-form 'call type 'entry)))
		     (sequence-end call))))))))))

;;;; DEFINE-ARRAY: vectors of characters are represented as strings, and
;;;; vectors of bytes are represented as arrays.  Other arrays are arrays.

(defmacro define-array-type (name type &rest dimensions)
  `(progn
     ,@(cond 
	((and (eq (first type) :primitive)
	      (eq (second type) :short-character)
	      (= (length dimensions) 1))
	 `((deftype ,name () '(simple-string ,(car dimensions)))
	   (setf (get ',name 'prototype) (make-string ,(car dimensions)))))
	((and (eq (first type) :primitive)
	      (eq (second type) :character)
	      (= (length dimensions) 1))
	 `((deftype ,name () '(simple-string ,(car dimensions)))
	   (setf (get ',name 'prototype) (make-string ,(car dimensions)))))
	((and (eq (first type) :primitive)
	      (eq (second type) :byte)
	      (= (length dimensions) 1))
	 `((deftype ,name ()
	     '(simple-array (unsigned-byte 8) (,(car dimensions))))
	   (setf (get ',name 'prototype)
	     (make-array ,(car dimensions)
			 :element-type '(unsigned-byte 8)))))
	(t
	 `((deftype ,name ()
	     '(simple-array ,(type-lisp-type type) ,dimensions))
	   (setf (get ',name 'prototype)
	     (make-array ',dimensions
			 :element-type ',(type-lisp-type type))))))))

(defmacro define-array (name type &rest dimensions)
  (let* ((name-string (symbol-name name))
	 (read-fn (intern (format nil "~A-read" name-string)))
	 (write-fn (intern (format nil "~A-write" name-string)))
	 (size-fn (intern (format nil "~A-size" name-string))))
    `(progn
       ,@(cond 
	  ((and (eq (first type) :primitive)
		(eq (second type) :short-character)
		(= (length dimensions) 1))
	   `((defun ,read-fn (call optional-p)
	       (char-vector-read call ,(car dimensions) optional-p))
	     (defun ,write-fn (call s optional-p)
	       (char-vector-write call s ,(car dimensions) optional-p))
	     (defun ,size-fn (call s optional-p)
	       (char-vector-size call s ,(car dimensions) optional-p))))
	  ((and (eq (first type) :primitive)
		(eq (second type) :character)
		(= (length dimensions) 1))
	   `((defun ,read-fn (call optional-p)
	       (unicode-char-vector-read
		 call ,(car dimensions) optional-p))
	     (defun ,write-fn (call s optional-p)
	       (unicode-char-vector-write
		 call s ,(car dimensions) optional-p))
	     (defun ,size-fn (call s optional-p)
	       (unicode-char-vector-size
		 call s ,(car dimensions) optional-p))))
	  ((and (eq (first type) :primitive)
		(eq (second type) :byte)
		(= (length dimensions) 1))
	   `((defun ,read-fn (call optional-p)
	       (declare (ignore optional-p))
	       (byte-vector-read call ,(car dimensions)))
	     (defun ,write-fn (call v optional-p)
	       (declare (ignore optional-p))
	       (byte-vector-write call v))
	     (defun ,size-fn (call v optional-p)
	       (declare (ignore optional-p))
	       (byte-vector-size call v))))
	  ((and (eq (first type) :primitive)
		(member (second type) '(:short-character :byte :character)))
	   `((defun ,read-fn (call optional-p)
	       (when (array-read call optional-p)
		 (let ((value
			(make-array ',dimensions
				    :element-type ',(type-lisp-type type))))
		   ,(labels ((reader (vars dims)
			       (if (null (cdr dims))
				   (ecase (second type)
				     (:short-character
				      `(ilu-lisp_char-vector-read
					call value (+ ,@(cdr vars)) ,(car dims)
					optional-p))
				     (:byte
				      `(ilu-lisp_byte-vector-read
					call value (+ ,@(cdr vars)) ,(car dims)
					optional-p))
				     (:character
				      `(ilu-lisp_unicode-char-vector-read
					call value (+ ,@(cdr vars)) ,(car dims)
					optional-p)))
				 `(dotimes (,(cadar vars) ,(car dims))
				    ,(reader
				      (cons (list '* (gensym) (cadr dims))
					    vars)
				      (cdr dims))))))
		      (reader (list (list '* (gensym) (car dimensions)))
			      dimensions))
		   (array-end call)
		   value)))
	     (defun ,write-fn (call array optional-p)
	       (array-write call array optional-p)
	       ,(labels ((writer (vars dims)
			   (if (null (cdr dims))
			       (ecase (second type)
				 (:short-character
				  `(ilu-lisp_char-vector-write
				    call array (+ ,@(cdr vars)) ,(car dims)
				    optional-p))
				 (:byte
				  `(ilu-lisp_byte-vector-write
				    call array (+ ,@(cdr vars)) ,(car dims)
				    optional-p))
				 (:character
				  `(ilu-lisp_unicode-char-vector-write
				    call array (+ ,@(cdr vars)) ,(car dims)
				    optional-p)))
			     `(dotimes (,(cadar vars) ,(car dims))
				,(writer
				  (cons (list '* (gensym) (cadr dims)) vars)
				  (cdr dims))))))
		  (writer (list (list '* (gensym) (car dimensions)))
			  dimensions))
	       (array-end call)
	       array)
	     (defun ,size-fn (call optional-p)
	       (let ((size (array-size call array optional-p)))
		 ,(labels ((sizer (vars dims)
			     (if (null (cdr dims))
				 `(incf size
					,(ecase (second type)
					   (:short-character
					    `(ilu-lisp_char-vector-size
					      call array (+ ,@(cdr vars))
					      ,(car dims) optional-p))
					   (:byte
					    `(ilu-lisp_byte-vector-size
					      call array (+ ,@(cdr vars))
					      ,(car dims) optional-p))
					   (:character
					    `(ilu-lisp_unicode-char-vector-size
					      call array (+ ,@(cdr vars))
					      ,(car dims) optional-p))))
			       `(dotimes (,(cadar vars) ,(car dims))
				  ,(sizer
				    (cons (list '* (gensym) (cadr dims)) vars)
				    (cdr dims))))))
		    (sizer (list (list '* (gensym) (car dimensions)))
			   dimensions))
		 (array-end call)
		 size))))
	  (t
	   `((defun ,read-fn (call optional-p)
	       (prog1 (when (array-read call optional-p)
			(let ((value
			       (make-array
				',dimensions
				:element-type ',(type-lisp-type type))))
			  ,(labels
			       ((reader (vars dims)
				  `(dotimes (,(car vars) ,(car dims))
				     ,(if (null (cdr dims))
					  `(setf (aref value ,@(reverse vars))
					     ,(make-read-form 'call type))
					(reader (cons (gensym) vars)
						(cdr dims))))))
			     (reader (list (gensym)) dimensions))
			  value))
		 (array-end call)))
	     (defun ,write-fn (call array optional-p)
	       (array-write call array optional-p)
	       (unless (and optional-p (null array))
		 ,(labels ((writer (vars dims)
			     `(dotimes (,(car vars) ,(car dims))
				,(if (null (cdr dims))
				     (make-write-form
				      'call type `(aref array
							,@(reverse vars)))
				   (writer (cons (gensym) vars)
					   (cdr dims))))))
		    (writer (list (gensym)) dimensions)))
	       (array-end call)
	       array)
	     (defun ,size-fn (call array optional-p)
	       (let ((size (array-size call array optional-p)))
		 (unless (and optional-p (null array))
		   ,(labels ((sizer (vars dims)
			       `(dotimes (,(car vars) ,(car dims))
				  ,(if (null (cdr dims))
				       `(incf
					 size
					 ,(make-size-form
					   'call type
					   `(aref array ,@(reverse vars))))
				     (sizer (cons (gensym) vars)
					    (cdr dims))))))
		      (sizer (list (gensym)) dimensions)))
		 (array-end call)
		 size))))))))



;;;; DEFINE-UNION: use lisp's run-time typing to distinguish union types

(defmacro do-clauses ((index item list) &body body)
  (let ((tail (gensym)) (result (gensym)))
    `(do ((,result '() (let ((,item (car ,tail))) (cons ,@body ,result)))
	  (,index 0 (1+ ,index))
	  (,tail ,list (cdr ,tail)))
	 ((null ,tail) (nreverse ,result)))))

(defmacro define-union-type (name &rest types)
  `(progn
     (deftype ,name () '(or ,@(mapcar #'type-lisp-type types)))
     (setf (get ',name 'prototype) ,(type-prototype-form (first types)))))

(defmacro define-union (name &rest types)
  (let* ((name-string (symbol-name name))
	 (read-fn (intern (format nil "~A-read" name-string)))
	 (write-fn (intern (format nil "~A-write" name-string)))
	 (size-fn (intern (format nil "~A-size" name-string))))
    `(progn
       (defun ,read-fn (call optional-p)
	 (prog1 (ecase (union-read call optional-p)
		  ((nil) nil)
		  ,@(do-clauses (i type types)
		      `(,i ,(make-read-form 'call type))))
	   (union-end call)))
       (defun ,write-fn (call value optional-p)
	 (if (and optional-p (null value))
	     (union-write call nil optional-p)
	   (etypecase value
	     ,@(do-clauses (i type types)
		 `(,(type-lisp-type type)
		   (union-write call ,i optional-p)
		   ,(make-write-form 'call type 'value)))))
	 (union-end call)
	 value)
       (defun ,size-fn (call value optional-p)
	 (prog1 (if (and optional-p (null value))
		    (union-size call nil optional-p)
		  (etypecase value
		    ,@(do-clauses (i type types)
			`(,(type-lisp-type type)
			  (+ (union-size call ,i optional-p)
			     ,(make-size-form 'call type 'value))))))
	   (union-end call))))))



;;;; DEFINE-OPTIONAL

(defmacro define-optional-type (name type)
  `(progn
     (deftype ,name () '(or nil ,(type-lisp-type type)))
     (setf (get ',name 'prototype) ,(type-prototype-form type))))

(defmacro define-optional (name type)
  (let* ((name-string (symbol-name name))
	 (read-fn (intern (format nil "~A-read" name-string)))
	 (write-fn (intern (format nil "~A-write" name-string)))
	 (size-fn (intern (format nil "~A-size" name-string))))
    `(progn
       (defun ,read-fn (call optional-p)
	 (declare (ignore optional-p))
	 (when (optional-read call)
	   ,(make-read-form 'call type)))
       (defun ,write-fn (call value optional-p)
	 (declare (ignore optional-p))
	 (optional-write call (not (null value)))
	 (when value
	   ,(make-write-form 'call type 'value))
	 value)
       (defun ,size-fn (call value optional-p)
	 (declare (ignore optional-p))
	 (if value
	     (+ (optional-size call t)
		,(make-size-form 'call type 'value))
	   (optional-size call nil))))))


(defstruct (method-spec (:type list) (:conc-name ms-))
  name id functional-p asynchronous-p return-type args exceptions)

(defmacro define-class-type (name supers singleton-p authentication methods)
  (declare (ignore authentication))
  `(defclass ,name
       ,(or supers
	 (if singleton-p '(ilu-singleton-object) '(ilu-object)))
       ,(mapcan #'(lambda (method)
		    (when (and (ms-functional-p method)
			       (ms-return-type method)
			       (null (ms-args method)))
		      `((,(ms-name method)))))
		methods)))

(defmacro define-class (name c-class-set-fn c-class-get-fn c-class-name-fn
			authentication)
  (declare (ignore authentication))
  `(progn

     (define-c-function ,(intern c-class-set-fn) "" ,c-class-set-fn
       (:ilu-object) :void)
     (define-c-function ,(intern c-class-get-fn) "" ,c-class-get-fn
       () :pointer)
     (define-c-function ,(intern c-class-name-fn) "" ,c-class-name-fn
       () :pointer)

     (defmethod set-c-class (ilu-object (obj ,name))
       (,(intern c-class-set-fn) ilu-object))
     (defmethod ilu-class-record ((classname (eql ',name)))
       (,(intern c-class-get-fn)))
     (defmethod ilu-class-name ((classname (eql ',name)))
       (,(intern c-class-name-fn)))

     (initialize-ilu-class ',name)))



;;;; DEFINE-EXCEPTION: expands to DEFINE-CONDITION plus

(defmacro define-exception-type (name value-type)
  `(define-condition ,name (rpc-exception)
     (,@(if value-type
	    `((exception-value :type ,(type-lisp-type value-type)))))))

(defmacro define-exception (name value-type)
  (when value-type
    `(progn
       (defmethod exception-value-read (call (self ,name))
	 ,(make-read-form 'call value-type))
       (defmethod exception-value-size (call (self ,name))
	 ,(make-size-form 'call value-type `(exception-value self)))
       (defmethod exception-value-write (call (self ,name))
	 ,(make-write-form
	   'call value-type `(exception-value self))))))


;;;; DEFINE-METHOD: expands into a defmethod

(defmacro define-method (name id singleton-p functional-p asynchronous-p
			 args exceptions return-type)
  (destructuring-bind ((self self-type) &rest other-args) args
    (declare (ignore self))
    (let ((other-arg-names (mapcar #'first other-args))
	  (other-arg-types (mapcar #'second other-args))
	  (other-arg-dirs (mapcar #'third other-args))
	  (has-return-values (or return-type
				 (dolist (dir (mapcar #'third other-args) nil)
				   (when (or (eq dir :inout) (eq dir :out))
				     (return t)))))
	  (self (gensym))
	  (class-record (gensym))
	  (call (gensym)))
      `(defmethod ,name ((,self ,(second self-type))
			 ,@(remove-nils
			    (mapcar #'(lambda (dir name)
					(unless (eq dir :out) name))
				    other-arg-dirs other-arg-names)))
	 ,(when (and functional-p return-type (null other-args))
	    `(when (slot-boundp ,self ',name)
	       (return-from ,name (apply #'values (slot-value ,self ',name)))))
	 (let ((,call (begin-call ,self))
	       (,class-record (ilu-class-record ',(second self-type))))
	   (begin-request
	    ,call ,class-record (rpc-method ,self ,class-record ,id)
	    (+ ,@(remove-nils
		  (mapcar #'(lambda (dir type name)
			      (if (or (eq dir :in) (eq dir :inout))
				  (make-size-form call type name)))
			  other-arg-dirs other-arg-types other-arg-names))
	       ,(if singleton-p 0 `(object-id-size ,call ,self 1 0 nil))))
	   ,@(unless singleton-p
	       `((object-id-write ,call (kernel-obj ,self) 1 0)))
	   ,@(remove-nils
	      (mapcar #'(lambda (dir type name) (if (or (eq dir :in) (eq dir :inout))
						    (make-write-form call type name)))
		      other-arg-dirs other-arg-types other-arg-names))
	   (finish-request ,call)
	   ,(if asynchronous-p
		`(progn (finish-call ,call)
			(values nil t))
	      `(multiple-value-bind (status index)
		   (wait-for-reply ,call)
		 (let (,@(if has-return-values
			   `((return-value
			      (if (and
				   (= status +protocol-exception-success+)
				   (= index 0))
				  (list
				   ,@(remove-nils
				      (cons
				       (when return-type
					 (make-read-form call return-type))
				       (remove-nils
					(mapcar #'(lambda (dir type)
						    (when (or (eq dir :inout)
							      (eq dir :out))
						      (make-read-form call type)))
						other-arg-dirs other-arg-types))))
				   t)))))
		       ,@(when exceptions
			   `((exception
			      (when (and
				     (= status +protocol-exception-success+)
				     (/= index 0))
				(make-instance (nth (1- index) ',exceptions)
				  :call ,call))))))
		   (finish-call ,call)
		   (if (= status +protocol-exception-success+)
		       (if (= index 0)
			   ,(if has-return-values
				(if (and functional-p
					 return-type
					 (null other-args))
				    `(apply #'values (setf (slot-value ,self ',name)
						       return-value))
				  `(apply #'values return-value))
			      `(values nil t))
			 ,(and exceptions `(error exception)))
		     (error (make-instance 'protocol-error
			      :exception-value status)))))))))))

;;; Moved DEFINE-SERVER-CLASS and DEFINE-METHOD-SERVER-STUB to ilu-server.lisp

