/*
Copyright (c) 1991, 1992, 1993, 1994 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.

$Id: ilu-lisp-skin.c,v 1.26 1994/05/06 00:09:34 janssen Exp $
*/

#include <string.h>
#include <stdlib.h>

#include <ilu.h>

#define OPTCHECK(a)	((opt && !provided)?((ilu_cardinal)0):(a))

void ilulisp_bytencopy (unsigned char *to, unsigned char *from, unsigned int count)
{
  if (to != NULL && from != NULL && count > 0)
    memmove (to, from, count);
}

ilu_string ilulisp_Strdup (ilu_string str)
{
  ilu_string p;

  if (str == NULL)
    return (NULL);
  else if ((p = malloc(strlen(str) + 1)) == NULL)
    return (NULL);
  else
    {
      strcpy(p, str);
      return(p);
    }
}

long ilulisp_InputInteger (ilu_Call call)
{
  long i;
  ilu_InputInteger (call, &i);
  return (i);
}

int ilulisp_InputBoolean (ilu_Call call)
{
  ilu_boolean i;
  ilu_InputBoolean (call, &i);
  return (i);
}

unsigned long ilulisp_InputCardinal (ilu_Call call)
{
  unsigned long i;
  ilu_InputCardinal (call, &i);
  return (i);
}

short ilulisp_InputShortInteger (ilu_Call call)
{
  short i;
  ilu_InputShortInteger (call, &i);
  return (i);
}

unsigned short ilulisp_InputShortCardinal (ilu_Call call)
{
  unsigned short i;
  ilu_InputShortCardinal (call, &i);
  return (i);
}

unsigned short ilulisp_InputEnumCode (ilu_Call call)
{
  unsigned short i;
  ilu_InputEnum (call, &i);
  return (i);
}

unsigned int ilulisp_InputOptional (ilu_Call call)
{
  ilu_boolean b;
  ilu_InputOptional (call, &b);
  return (b ? 1 : 0);
}

double ilulisp_InputReal (ilu_Call call)
{
  double f;
  ilu_InputReal (call, &f);
  return (f);
}

double ilulisp_InputLongReal (ilu_Call call)
{
  double f;
  ilu_InputLongReal (call, &f);
  return (f);
}

/* for use if lisp understands ANSI parameter passing */
float ilulisp_ANSIInputShortReal (ilu_Call call)
{
  float f;
  ilu_InputShortReal (call, &f);
  return (f);
}

/* the next 3 for use if lisp doesn't understand ANSI parameter passing */
double ilulisp_KandRInputShortReal (ilu_Call call)
{
  float f;
  ilu_InputShortReal (call, &f);
  return ((double) f);
}

void ilulisp_KandROutputShortReal (ilu_Call call, double r)
{
  float f = r;
  ilu_OutputShortReal (call, f);
}

int ilulisp_KandRSizeOfShortReal (ilu_Call call, double r)
{
  float f = r;
  return (ilu_SizeOfShortReal (call, f));
}

unsigned char ilulisp_InputByte (ilu_Call call)
{
  unsigned char b;
  ilu_InputByte (call, &b);
  return (b);
}

/* short character sequence functions *****************************************/

char * ilulisp_InputString (ilu_Call call, ilu_cardinal limit, ilu_boolean opt)
{
  char *p = NULL;
  unsigned long i;

  ilu_InputString (call, &p, &i, limit, opt);
  return (p);
}

void ilulisp_OutputString (ilu_Call call, ilu_string string, ilu_cardinal len, ilu_cardinal limit, ilu_boolean opt, ilu_boolean provided)
{
  ilu_OutputString (call, OPTCHECK(string), OPTCHECK(len), limit, opt);
}

ilu_cardinal ilulisp_SizeOfString (ilu_Call call, ilu_string string, ilu_cardinal len, ilu_cardinal limit, ilu_boolean opt, ilu_boolean provided)
{
  return (ilu_SizeOfString (call, OPTCHECK(string), OPTCHECK(len), limit, opt));
}

/* short character vector functions *****************************************/

char *ilulisp_InputCharacterVector (ilu_Call call, long unsigned int len, ilu_boolean opt)
{
  char *p;

  ilu_InputStringVec (call, &p, len, opt);
  return (p);
}

void ilulisp_OutputStringVec (ilu_Call call, ilu_string string, ilu_cardinal len, ilu_boolean opt, ilu_boolean provided)
{
  ilu_OutputStringVec (call, OPTCHECK(string), OPTCHECK(len), opt);
}

ilu_cardinal ilulisp_SizeOfStringVec (ilu_Call call, ilu_string string, ilu_cardinal len, ilu_boolean opt, ilu_boolean provided)
{
  return (ilu_SizeOfStringVec (call, OPTCHECK(string), OPTCHECK(len), opt));
}

/* byte vector functions *****************************************/

unsigned char *ilulisp_InputByteVector (ilu_Call call, long unsigned int len)
{
  unsigned char *p;

  ilu_InputOpaque (call, &p, len, ilu_FALSE);
  return (p);
}

void ilulisp_OutputByteVector (ilu_Call call, unsigned char *v, long unsigned int len)
{
  ilu_OutputOpaque (call, v, len, ilu_FALSE);
}

ilu_cardinal ilulisp_SizeOfByteVector (ilu_Call call, unsigned char *v, long unsigned int len)
{
  return(ilu_SizeOfOpaque (call, v, len, ilu_FALSE));
}

/* byte sequence functions *****************************************/

unsigned char *ilulisp_InputByteSequence (ilu_Call call, long unsigned int *i, long unsigned int limit, ilu_boolean opt)
{
  ilu_bytes p;
  ilu_InputBytes (call, &p, i, limit, opt);
  return (p);
}

void ilulisp_OutputByteSequence (ilu_Call call, unsigned char *v, long unsigned int len, long unsigned int limit, ilu_boolean opt, ilu_boolean provided)
{
  ilu_OutputBytes (call, OPTCHECK(v), OPTCHECK(len), limit, opt);
}

ilu_cardinal ilulisp_SizeOfByteSequence (ilu_Call call, unsigned char *v, long unsigned int len, long unsigned int limit, ilu_boolean opt, ilu_boolean provided)
{
  return(ilu_SizeOfBytes (call, OPTCHECK(v), OPTCHECK(len), limit, opt));
}

/* wide character sequence functions *****************************************/

ilu_string ilulisp_InputWString (ilu_Call call, ilu_cardinal *len, ilu_cardinal limit, ilu_boolean opt)
{
  ilu_wstring s = NULL;
  ilu_string s2 = NULL;
  ilu_InputWString (call, &s, len, limit, opt);
  if (s != NULL)
    {
      register int i;
      s2 = malloc(*len + 1);
      for(i = 0;  i < *len;  i++)
	s2[i] = (s[i] < 128 && s[i] > 0) ? s[i] : 127;
      s2[*len] = 0;
      free(s);
    }
  return(s2);
}

void ilulisp_OutputWString (ilu_Call call, ilu_wstring string, ilu_cardinal len, ilu_cardinal limit, ilu_boolean opt, ilu_boolean provided)
{
  ilu_OutputWString (call, OPTCHECK(string), OPTCHECK(len), limit, NULL, NULL, opt);
}

ilu_cardinal ilulisp_SizeOfWString (ilu_Call call, ilu_wstring string, ilu_cardinal len, ilu_cardinal limit, ilu_boolean opt, ilu_boolean provided)
{
  return(ilu_SizeOfWString (call, OPTCHECK(string), OPTCHECK(len), limit, NULL, NULL, opt));
}

/* wide character vector functions *****************************************/

ilu_string ilulisp_InputWStringVec (ilu_Call call, ilu_cardinal len, ilu_boolean opt)
{
  ilu_wstring s = NULL;
  ilu_string s2 = NULL;
  ilu_InputWStringVec (call, &s, len, opt);
  if (s != NULL)
    {
      register int i;
      s2 = malloc(len + 1);
      for(i = 0;  i < len;  i++)
	s2[i] = (s[i] < 128 && s[i] > 0) ? s[i] : 127;
      s2[len] = 0;
      free(s);
    }
  return(s2);
}

void ilulisp_OutputWStringVec (ilu_Call call, ilu_wstring string, ilu_cardinal len, ilu_boolean opt, ilu_boolean provided)
{
  ilu_OutputWStringVec (call, OPTCHECK(string), OPTCHECK(len), NULL, NULL, opt);
}

ilu_cardinal ilulisp_SizeOfWStringVec (ilu_Call call, ilu_wstring string, ilu_cardinal len, ilu_boolean opt, ilu_boolean provided)
{
  return(ilu_SizeOfWStringVec (call, OPTCHECK(string), OPTCHECK(len), NULL, NULL, opt));
}

/* structured type functions *****************************************/

ilu_integer ilulisp_InputSequence (ilu_Call call, ilu_boolean opt, ilu_cardinal limit)
{
  ilu_cardinal l;
  ilu_boolean provided = ilu_TRUE;

  ilu_InputSequence (call, &l, limit, opt, &provided);
  return (provided ? ((ilu_integer) l) : -1);
}

int ilulisp_InputUnion (ilu_Call call, ilu_boolean opt)
{
  ilu_shortcardinal l;
  ilu_boolean provided = ilu_TRUE;

  ilu_InputUnion (call, &l, opt, &provided);
  return (provided ? ((int) l) : -1);
}

int ilulisp_InputArray (ilu_Call call, ilu_boolean opt)
{
  ilu_boolean provided = ilu_TRUE;

  ilu_InputArray (call, opt, &provided);
  return (provided ? 1 : 0);
}

int ilulisp_InputRecord (ilu_Call call, ilu_boolean opt)
{
  ilu_boolean provided = ilu_TRUE;

  ilu_InputRecord (call, opt, &provided);
  return (provided ? 1 : 0);
}

/* Input object ID functions *****************************************/

ilu_Object ilulisp_InputObjectID (ilu_Call call, ilu_refany *classrec, ilu_Class pclass, int disc)
{
  ilu_Object p;

  ilu_InputObjectID (call, &p, disc, pclass);
  if (p == NULL)
    {
      if (disc)
	fprintf (stderr, "******** Couldn't input discriminator of type %s (%s)\n",
		 pclass->cl_name, pclass->cl_unique_id);
      else
	fprintf (stderr, "******** Couldn't input object with pclass %s (%s)\n",
		 pclass->cl_name, pclass->cl_unique_id);
    }
  else
    *classrec = p->ob_class;
  return (p);
}

ilu_string ilulisp_IDOfClass (ilu_Class c)
{
  return (c->cl_unique_id);
}

ilu_Connection ilulisp_HandleNewConnection (ilu_Port p)
{
  ilu_boolean closed = ilu_FALSE;
  ilu_Connection c;

  c = ilu_HandleNewConnection (p, &closed);
  if (!closed)
    return c;
  else
    return NULL;
}

ilu_cardinal ilulisp_IDOfMethod (ilu_Method m)
{
  return (m->me_id);
}

ilu_Method ilulisp_MethodOfCall (ilu_Call c)
{
  return (c->ca_method);
}

int ilulisp_FileDescriptorOfMooringOfPort (ilu_Port p)
{
  ilu_boolean closed = ilu_FALSE;
  int fd;

  fd = ilu_FileDescriptorOfMooringOfPort (p, &closed);
  if (closed)
    return (-1);
  else
    return (fd);  
}

int ilulisp_ReceiveRequest (ilu_Connection conn, ilu_Call *call, ilu_Class *intro_type, char **method_name)
{
  ilu_ReceiveRequestStatus stat;
  ilu_Call call2;
  ilu_Class class2;
  ilu_cardinal sn2;
  ilu_Method method2;
  
  stat = ilu_ReceiveRequest (conn, &call2, &class2, &method2, &sn2);
  if (stat == ilu_conn_closed)
    {
      ilu_CloseConnection (conn);
      return (-1);
    }
  else if (stat == ilu_read_failed || stat == ilu_not_request || stat == ilu_interp_failed)
    {
      return (-2);
    }
  else if (stat == ilu_no_request || stat == ilu_builtin_meth)
    {
      return (0);
    }
  else
    {
      *call = call2;
      *intro_type = class2;
      *method_name = method2->me_name;
      return (sn2);
    }
}

/* make sure all modules from libilu.a are included by referencing some routines */

void ______ilulisp_NeverCalled(void)
{
  ilu_MXAClear(NULL, NULL);		/* alarmux.c */
  ilu_BeginCall(NULL);			/* call.c */
  ilu_ClassOfObject(NULL);		/* object.c */
  ilu_IDOfMethod(NULL);			/* method.c */
  ilu_ClosePort (NULL);			/* port.c */
  _ilu_hash_PairsInTable();		/* hash.c */
  ilu_CloseConnection(NULL);		/* connect.c */
  ilu_SetDebugLevel(0);			/* debug.c */
  ilu_RegisterLockTech(NULL);		/* locks.c */
  ilu_FindTransportTypeFromInfo(NULL);	/* transprt.c */
  ilu_FindProtocolTypeFromInfo(NULL);	/* protocol.c */
  ilu_CreateTrueServer(NULL, NULL);	/* server.c */
  ilu_VeryInterested(NULL);		/* gc.c */
  _ilu_vector_new (0);			/* vector.c */
  ilu_SetMainLoop (NULL);		/* mainloop.c */
  ilu_FindClassFromID(NULL);		/* type.c */
  ilu_PublishObject(NULL);		/* simpbind.c */
  _ilu_sunrpc_Protocol();		/* sunrpc.c */
  _ilu_tcp_TransportClass();		/* tcp.c */
  _ilu_udp_TransportClass();		/* udp.c */
  _ilu_mem_TransportClass();		/* inmem.c */
}
