/*
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* $Id: port.c,v 1.26 1994/04/16 02:13:01 janssen Exp $ */

#define _POSIX_SOURCE

#include <string.h>
#include <ilu.h>
#include "iluntrnl.h"

#include "port.h"
#include "mooring.h"
#include "protocol.h"
#include "transprt.h"
#include "object.h"
#include "server.h"

/*L2, Main unconstrained*/
/*L1 < port's server*/

int ilu_FileDescriptorOfMooringOfPort (ilu_Port port, ilu_boolean *closed)
{
  ilu_integer fd;
  ilu_Server s = port_server(port);
  _ilu_AcquireMutex(server_lock(s));
  fd = mooring_file_descriptor(port_mooring(port));
  *closed = port->po_closed;
  _ilu_ReleaseMutex(server_lock(s));
  return(fd);
}

/*Main Invariant holds*/

ilu_Connection ilu_HandleNewConnection (ilu_Port port,
					ilu_boolean *closed)
{
  ilu_Transport t = NULL;
  ilu_Connection c = NULL;
  ilu_Server s = port->po_server;
  ilu_integer dfd = port->po_mooring->mo_transportClass->tc_cFD;
  _ilu_AcquireMutex(server_lock(s));
  *closed = port->po_closed;
  _ilu_ReleaseMutex(server_lock(s));
  if (*closed)
      return (NULL);
  _ilu_AcquireMutex(ilu_cmu);
  if (ilu_fdbudget < ilu_fdsused + ilu_fdsidle + dfd) {
      _ilu_ReduceFdsTo(ilu_fdbudget - dfd);
      if ((ilu_fdbudget < ilu_fdsused + ilu_fdsidle + dfd) && (dfd>0)) {
          DEBUG(CONNECTION_DEBUG,
		(stderr,
		 "HandleNewConnection: FD budget exhausted.\n"));
	  _ilu_ReleaseMutex(ilu_cmu);
	  return NULL;
	}
    }
  if ((t = mooring_accept(port_mooring(port))) != NULL) {
      ilu_string tinfo = transport_form_handle(t);
      _ilu_AcquireMutex(server_lock(s));
      if (port->po_closed) {
          *closed = TRUE;
          transport_close(t);
	  free(tinfo);
          return (NULL);
        }
      c = _ilu_CreateConnection(t, tinfo, port_protocol(port),
				port->po_pinfo, port, port_server(port));
      _ilu_ReleaseMutex(server_lock(s));
      free(tinfo);
    }
  _ilu_ReleaseMutex(ilu_cmu);
  return (c);
}

int call_cache_size = 5;

ilu_Port ilu_CreatePort (ilu_Server s,
			 ilu_string protocolinfo,
			 ilu_string mooringinfo)
{
  ilu_TransportClass tc;
  ilu_Protocol p;
  char buf[1000];
  ilu_Port new = NULL;

  if (protocolinfo == NULL || mooringinfo == NULL)
    return (NULL);

#ifndef MACOS
  if (sscanf(mooringinfo, "%[^_]_", buf) != 1)
      return (NULL);
#else
  /* AC: changed to a by-hand parse because of bug in Mac. sscanf()... */
  if ((tmp=strchr(mooringinfo,'_'))==NULL)
        return NULL;
  strncpy(buf,mooringinfo,tmp-mooringinfo);
  buf[tmp-mooringinfo]='\0';
#endif /* MACOS */

  if ((tc = _ilu_GetTransportClassByName (buf)) != NULL
      && (p = _ilu_GetProtocolFromInfo (protocolinfo)) != NULL)
    {
      ilu_integer dfd = tc->tc_pFD;
      _ilu_AcquireMutex(ilu_cmu);
      _ilu_ReduceFdsTo(ilu_fdbudget - dfd);
      if ((ilu_fdbudget < ilu_fdsused + ilu_fdsidle + dfd) && (dfd > 0)) {
          fprintf(stderr, "%s %s %s because over FD budget",
		  "ilu_CreatePort:  can't create port",
		  protocolinfo, mooringinfo);
          _ilu_ReleaseMutex(ilu_cmu);
          return NULL;
        }
      ilu_fdsused += dfd;
      _ilu_ReleaseMutex(ilu_cmu);
      new = (ilu_Port) malloc(sizeof(struct _ilu_Port_s));
      new->po_server = s;
      new->po_pinfo = _ilu_Strdup(protocolinfo);
      new->po_protocol = p;
      new->po_mooring = (*(tc->tc_create_mooring))
		     ((*(tc->tc_interpret_info))(mooringinfo));
      new->po_tinfo = mooring_form_handle(new->po_mooring);
      new->po_closed = FALSE;
      new->po_next = NULL;
      new->po_connHead.next = new->po_connHead.prev = NULL;

      _ilu_AcquireMutex(server_lock(s));
        { /* Under server lock: */
          new->po_next = server_ports(s);
          server_ports(s) = new;
          if (server_default_port(s) == NULL)
            server_default_port(s) = new;
          if (tc->tc_timesout) {
              static const ilu_CachedCall c0 = {"\0", 0, NULL, NULL,
                                                NULL, 0};
              int i;
              new->po_call_cache = (ilu_CachedCall *)
                                malloc(call_cache_size
                                       * sizeof(ilu_CachedCall));
              _ilu_Assert(new->po_call_cache != NULL,
                          "CreatePort: malloc call cache failed");
              new->po_call_cache_size = call_cache_size;
              for (i=0; i < new->po_call_cache_size; i++)
                new->po_call_cache[i] = c0;
            }
          else {
            new->po_call_cache = NULL;
            new->po_call_cache_size = 0;
          }
        new->po_call_cache_finger = 0;
        }
      _ilu_ReleaseMutex(server_lock(s));
    }
  return (new);
}

/*L1_sup < cmu*/
/*L2, Main unconstrained*/

void ilu_ClosePort(ilu_Port port)
{
  ilu_Server s = port->po_server;
  _ilu_AcquireMutex(ilu_cmu);
  _ilu_AcquireMutex(server_lock(s));
  _ilu_ClosePort(port);
  _ilu_ReleaseMutex(server_lock(s));
  _ilu_ReleaseMutex(ilu_cmu);
}

void ilu_DestroyPort(ilu_Port port)
{
  ilu_Server s = port->po_server;
  int i;
  _ilu_AcquireMutex(ilu_cmu);
  _ilu_AcquireMutex(server_lock(s));
  _ilu_ClosePort(port);
  _ilu_Assert(port_connections(port) == NULL,
              "DestroyPort: some connections still exist");
  if (port->po_call_cache != NULL)
      for (i = 0; i < port->po_call_cache_size; i++)
          if (port->po_call_cache[i].cc_reply != NULL)
              free(port->po_call_cache[i].cc_reply);
  free(port->po_tinfo);
  free(port->po_pinfo);
  free(port);
  _ilu_ReleaseMutex(server_lock(s));
  _ilu_ReleaseMutex(ilu_cmu);
}


/*L1 >= {port's server}*/
void _ilu_ClearPortFromServer(ilu_Port port, ilu_Server s)
{
  ilu_Port *pp;
  for (pp = &server_ports(s); (*pp) != NULL; pp = &port_next(*pp) ) {
      if ((*pp) == port) {
          *pp = port_next(port);
          return;
        }
    }
  _ilu_Assert(0, "ClosePort: not found in server");
}

/*L1_sup = port's server; L1 >= {cmu}*/
void _ilu_ClosePort(ilu_Port port)
{
  ilu_Server s = port->po_server;
  ilu_integer dfd = port->po_mooring->mo_transportClass->tc_pFD;
  if (port_closed(port))
      return;
  port->po_closed = TRUE;
  port->po_mooring->mo_transportClass->tc_close_mooring(port->po_mooring);
  ilu_fdsused -= dfd;
  if (s->sr_default_port == port) {
      s->sr_default_port = s->sr_ports;
      while (s->sr_default_port != NULL && port_closed(s->sr_default_port))
          s->sr_default_port = s->sr_default_port->po_next;
    }
  if (port_connections(port) == NULL)
      _ilu_ClearPortFromServer(port, s);
  return;
}
