/*
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* $Id: inmemory.c,v 1.14 1994/02/05 23:10:42 janssen Exp $ */
/* Last tweaked by Mike Spreitzer December 21, 1993 3:04 pm PST */

#define _POSIX_SOURCE

#include <unistd.h>

#include "ilu.h"
#include "iluInternals.h"

#include "transport.h"
#include "mooring.h"

#include "bufferlist.h"

struct mem_parms {
  ilu_boolean (*contact_proc)(ilu_Mooring, ilu_Transport);
  ilu_refany mooring;
  bufferList buffer;
  ilu_Transport peer;
};

struct tlist {
  struct tlist *next;
  ilu_Transport transport;
};

ilu_TransportClass _ilu_mem_TransportClass(void);

static ilu_boolean ContactProc (ilu_Mooring self, ilu_Transport new)
{
  ilu_Transport p = (ilu_Transport) malloc(sizeof(struct _ilu_Transport_s));
  struct tlist *t;
  struct mem_parms *data;

  p->tr_class = _ilu_mem_TransportClass();
  p->tr_fd = -1;
  
  data = (struct mem_parms *) malloc(sizeof(struct mem_parms));
  data->contact_proc = ContactProc;
  data->buffer = (bufferList) malloc(sizeof(struct bufferlist_s));
  data->buffer->head = NULL;
  data->buffer->tail = NULL;
  data->buffer->size = 0;
  data->peer = new;
  data->mooring = (ilu_refany) self;
  p->tr_data = (ilu_refany) data;
  ((struct mem_parms *)(new->tr_data))->peer = p;
  ((struct mem_parms *)(new->tr_data))->buffer = (bufferList) malloc(sizeof(struct bufferlist_s));
  ((struct mem_parms *)(new->tr_data))->buffer->head = NULL;
  ((struct mem_parms *)(new->tr_data))->buffer->tail = NULL;
  ((struct mem_parms *)(new->tr_data))->buffer->size = 0;

  /* add new transport to list of pending transports */

  t = (struct tlist *) malloc(sizeof(struct tlist));
  t->next = ((struct tlist *)(self->mo_data));
  t->transport = p;
  self->mo_data = (ilu_private) t;

  return (TRUE);
}

static ilu_boolean Connect (ilu_Transport self)
{
  struct mem_parms *p;
  ilu_boolean status = FALSE;

  if (self != NULL)
    {
      p = (struct mem_parms *) self->tr_data;
      status = (p->contact_proc)(p->mooring, self);
    }
  return (status);
}

static void Close (ilu_Transport self)
{
  struct mem_parms *p;
  bufferElement b, old;

  if (self != NULL)
    {
      p = (struct mem_parms *) (self->tr_data);
      p->peer = NULL;
      for (b = p->buffer->head;  b != NULL; )
	{
	  free(b->packet);
	  old = b;
	  b = b->next;
	  free (old);
	  p->buffer->size -= 1;
	}
      free(p->buffer);
      free(p);

      self->tr_fd = -1;
      self->tr_class = NULL;
      self->tr_data = NULL;
    }
}

static ilu_integer NDataPresent (ilu_Transport self)
{
  ilu_boolean dp = FALSE;

  if (self != NULL)
    {
      dp = (((struct mem_parms *) self->tr_data)->buffer->size > 0);
    }
  return (dp ? 1 : 0);
}

static ilu_boolean WaitForInput (ilu_Transport self, ilu_boolean *sure,
				 ilu_FineTime *limit)
{
  if (self != NULL)
    {
      while (((struct mem_parms *)(self->tr_data))->buffer->size <= 0) {
	  if (limit != NULL
	      && ilu_FineTime_Cmp(ilu_FineTime_Now(), *limit) > 0) {
	      *sure = FALSE;
	      return TRUE;
	    }
	  sleep(1);
	}
      *sure = TRUE;
      return (TRUE);
    }
  else
    return (FALSE);
}

static ilu_boolean ReadMessage (ilu_Transport self, ilu_bytes *buffer, ilu_cardinal *bufferSize)
{
  ilu_boolean status = FALSE;

  if (self != NULL)
    {
      if (((struct mem_parms *) self->tr_data)->buffer->size >= 1)
	{
	  struct mem_parms *p = (struct mem_parms *) self->tr_data;
	  bufferList b = p->buffer;

	  if (b->size >= 1 && b->head != NULL)
	    {
	      *buffer = b->head->packet;
	      *bufferSize = b->head->size;
	      b->head = b->head->next;
	      if (b->head == NULL)
		b->tail = NULL;
	      b->size -= 1;
	      status = (TRUE);
	    }
	}
    }
  return (status);
}

static ilu_boolean SendMessage (ilu_Transport self, ilu_bytes buffer, ilu_cardinal bufferSize)
{
  struct mem_parms *p, *peer;
  bufferElement b;
  ilu_boolean status = FALSE;

  if (self != NULL)
    {
      p = (struct mem_parms *) (self->tr_data);
      if (p->peer != NULL)
	{
	  peer = ((struct mem_parms *) (p->peer->tr_data));

	  b = (bufferElement) malloc(sizeof(struct bufferlist_element_s));
	  b->next = NULL;
	  b->packet = buffer;
	  b->size = bufferSize;
	  b->EOM = TRUE;
	  if (peer->buffer->tail == NULL)
	    {
	      peer->buffer->tail = b;
	      peer->buffer->head = b;
	      peer->buffer->size = 1;
	    }
	  else
	    {
	      peer->buffer->tail->next = b;
	      peer->buffer->size += 1;
	    }
	  status = TRUE;
	}
    }
  return (status);
}

static ilu_boolean FlushOutput (ilu_Transport self)
{
  /* needs nothing */
  return (TRUE);
}

static ilu_refany InterpretInfo (ilu_string info)
{
  struct mem_parms *p = (struct mem_parms *)
			malloc(sizeof(struct mem_parms));

  p->buffer = (bufferList) malloc(sizeof(struct bufferlist_s));
  p->peer = NULL;

  if (sscanf (info, "memory_%u_%u", (unsigned long *) &p->contact_proc,
	      (unsigned long *) &p->mooring) == 2)
    return ((ilu_refany) p);
  else
    {
      free(p);
      return (NULL);
    }
}

static ilu_string FormInfo (ilu_refany parameters)
{
  char buf[100];
  sprintf (buf, "memory_%u_%u",
	   (unsigned long) ((struct mem_parms *)parameters)->contact_proc,
	   (unsigned long) ((struct mem_parms *)parameters)->mooring);
  return ((ilu_string) _ilu_Strdup(buf));
}

static ilu_Transport AcceptConnection (ilu_Mooring self)
{
  struct tlist *t;
  ilu_Transport tr = NULL;

  if (self != NULL)
    {
      if ((t = (struct tlist *)(self->mo_data)) != NULL)
	{
	  tr = t->transport;
	  self->mo_data = (ilu_private) t->next;
	  t->next = NULL;
	  t->transport = NULL;
	  free(t);
	}
    }
  return (tr);
}

static ilu_Mooring CreateMooring (struct mem_parms *parms)
{
  ilu_Mooring self;

  _ilu_AutoSetDebugLevel();

  self = (ilu_Mooring) malloc (sizeof(struct _ilu_Mooring_s));

  self->mo_fd = -1;
  self->mo_transportClass = _ilu_mem_TransportClass();
  self->mo_accept_connection = AcceptConnection;
  self->mo_data = NULL;

  return (self);
}

ilu_TransportClass _ilu_mem_TransportClass (void)
{
  static ilu_TransportClass m = NULL;
  if (m == NULL)
    {
      m = (ilu_TransportClass) malloc(sizeof(struct _ilu_TransportClass_s));
      m->tc_type = ilu_TransportType_Memory;
      m->tc_interpret_info = InterpretInfo;
      m->tc_form_info = FormInfo;
      m->tc_connect = Connect;
      m->tc_close = Close;
      m->tc_n_data_present = NDataPresent;
      m->tc_wait_for_input = WaitForInput;
      m->tc_read_message = ReadMessage;
      m->tc_send_message = SendMessage;
      m->tc_flush_output = FlushOutput;
      m->tc_create_mooring = (ilu_Mooring (*)(ilu_private)) CreateMooring;
    }
  return (m);
}
