/*
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* $Id: iluxport.h,v 1.85 1994/04/28 02:43:10 janssen Exp $ */
/* $Locker:  $ */

#ifndef _ILU_EXPORTS_
#define _ILU_EXPORTS_

#if (defined ( __STDC__ ) || defined ( __cplusplus ))
/* So I can write ansi prototypes and still use Sun cc */
#define	_FN_DECL(type_and_name, args) type_and_name args
#else
#define _FN_DECL(type_and_name, args) type_and_name()
#endif


/* Unless specified otherwise, a result type of ilu_boolean
 * indicates TRUE is returned on normal completion, and
 * FALSE indicates something went wrong.
 */


/* ================ Locking ================ */
/*
We use mutual exclusion.  There are mutexes at two levels of
abstraction, discussed with L1 and L2.

At L2, there is a call mutex per connection and an I/O mutex per
connection.  The call and I/O mutexes are held during I/O of a request
or reply message on the connection; the I/O mutex is also held while
closing a connection.  For a connection using a concurrent protocol,
the call mutex is released while waiting for a reply; for a connection
using a non-concurrent protocol, the call mutex is held from the start
of the request to the end of the reply.  When starting a call over a
non-concurrent protocol, one of the server's available connections is
used, or a new one is created and used if all the existing ones have
their call mutexes held.  When starting a type or GC callback over a
concurrent protocol, the secondary connection is used if the primary
one's call mutex is held.  When starting an ordinary call over a
concurrent protocol, we block until the primary connection's call
mutex is available.  conn.callmu < conn.iomu.

At L1, there are a several mutexes:
smu:	global mutex for the server table;
otmu:	global mutex for object type data structures;
cmu:	global LRU list of connections
prmu:	global mutex for protocol registry
trmu:	global mutex for transport registry
gcmu:	global mutex for GC data structures
timu:	global mutex for alarm implementation.
server:	one mutex per server.

There can be no deadlocks that involve only L1 mutexes because threads
acquire L1 mutexes in an order consistent with a fixed partial order.
Here is the fixed partial order:

cmu < server
smu < server
server < prmu
server < trmu
gcmu < server
gcmu < timu
cmu < smu		(for something or other)
gcmu < cmu		(for gc.c:gcaInvoke)
cmu < timu		(for call.c:GR{Set,Cancel})
prmu < otmu		(for Protocol->interpret_request)

There is a common locking invariant, called the "Main Invariant":

  L1 = {}, and
  forall conn: (L2 >= {conn.iomu}) => (L2 >= {conn.callmu})

This is exactly what's guaranteed to hold while an application's
service routines are called.  This could be among the things
guaranteed to hold while a stub is marshalling or unmarshalling (and
will be when we no longer buffer entire messages).  This is exactly
what's guaranteed to hold while waiting for I/O on a File Descriptor
to be enabled.  This is among the things guaranteed to hold while
doing I/O on a File Descriptor.

There is another common requirement, which is with respect to some
server s and object type cl, is called Inside(s, cl), and is this:

  ~GC(cl)	      => L1 >= {      cmu, s};
   GC(cl) &&  true(s) => L1 >= {gcmu, cmu, s};
   GC(cl) && ~true(s) => L1  = {      cmu, s};
   GC(cl) && ~true(s) => forall conn: (L2 >= {conn.iomu})
					 => (L2 >= {conn.callmu}
  Main unconstrained.

Note that this invariant has the property that if C2 is a subtype of
C1, then Inside(s, C1) => Inside(s, C2).  This property is used in
situations where we know some static type of an object, but not
necessarily the most specific type (yet).

There is only one kind of mutex held while waiting for a reply: a
connection's call mutex.

Theorem: There can be no deadlocks involving L2 mutexes.  Proof: There
are two cases when a thread blocks trying to acquire a call mutex: (A)
when making an ordinary call (not a type or GC callback) on a server
with a concurrent connection, and (B) when making a type or GC
callback on a server with a concurrent connection.  In case A, the
call mutex in question is held only during the processing of a message
(not between request and reply messages), which will eventually finish
without trying to acquire any mutex held by the thread blocked on
acquiring the call mutex.  At worst the message processing may involve
a type or GC callback, but these will acquire a different call mutex
(and maybe a server's tabmu, which won't be held by the thread trying
acquire the call mutex in question), and the server can implement the
calls using only tightly held locks.  Case B is like case A, but the
message processing in progress will not itself initiate nested type or
GC callbacks.  Similarly, a thread blocks trying to acquire an I/O
mutex while another thread completes some I/O task that won't try to
acquire any mutexes held by the first thread.

We document locking requirements with comments about the symbols "L1"
and "L2", which stand for the sets of mutexes held by a thread;
"L1_sup" stands for the maximum member of L1 (but we suppose that
"L1_sup < XXX" is satisfied by a thread holding an empty set of L1
mutexes); "L2_sup" is not meaningful because we don't impose a partial
order on L2 mutexes.  For data, the comments say what mutexes must be
held to access the datum.  For procedures, the comments say what
mutexes must be held to call the procedure, and, if the procedure
changes the set of held mutex, how.

We have three sorts of locking comments: those about L1, those about
L2, and those about whether the Main Invariant applies.  Locking
comments come in blocks.  There are two kinds of blocks of locking
comments: a "sticky" block is followed by a blank line; a "one-shot"
is not.  A locking comment is also called "sticky" or "one-shot",
depending on the kind of the comment block in which the comment is
contained.  A one-shot comment applies only to the immediately
following item.  A sticky comment of a certain sort applies to all
items between it and the next sticky comment of the same sort, except
those items to which a one-shot comment of the same sort applies.
Another exception is this: where the Main Invariant holds, we needn't
explicitly override comments of the L1 sort.


Sadly, we need condition variables to get high-performance
multithreaded operation.  A thread can wait on a condition variable.
Another thread can "notify" that condition variable.  This causes all
threads currently waiting on the condition variable to return from the
wait operation.  To prevent timing splinters, decisions about waiting
and notifying should be made inside a mutex.  This means the mutex
must be released while waiting on a condition variable, and there must
be no possibilty of a thread switch between the release of the mutex
and the start of the wait; the wait operation takes the mutex as an
argument, because in a pre-emptive threads environment (eg, PCR) the
release and the wait must be an atomic thread operation.

Some runtimes (eg, a single-threaded one) cannot support condition
variables; these runtimes supply NULL for all the condition variable
operations.
*/

/*L1, L2, Main unconstrained*/

_FN_DECL(void ilu_RegisterLockTech, ( ilu_LockTech *lt ) );
/* A language-specific runtime calls this procedure at most once.
 * The procs are like their iluntrnl.h counterparts.
 * A single-threaded runtime can give NULL as the argument.
 * A single-threaded runtime can give NULL condition var procs;
 * a multi-threaded runtime *must* provide non-NULL ones.
 */

/*before: 				       L1_sup < cmu;
  before: cl collectible		    => L1_sup < gcmu;
  before: cl collectible & server surrogate => Main Invariant holds;
  after:  Inside(server, cl)*/
_FN_DECL(void ilu_EnterServer, (ilu_Server server, ilu_Class cl));
/* Needed by LS runtime to call GetLanguageSpecificObject or
 * RegisterLanguageSpecificObject */

/*before: Inside(server, cl);
  after:				      L1 disjoint {cmu, server};
  after: cl collectible			   => L1 disjoint {gcmu};
  after: cl collectible & server surrogate => Main Invariant holds*/
_FN_DECL(void ilu_ExitServer, (ilu_Server server, ilu_Class cl));


/* ================ Deleting Objects ================ */
/*
A kernel object (ilu_Object) is accessible only inside its server's mutex.  That mutex's invariant includes this: either (1) the kernel object and its language-specific object (LSO) point to each other, or (2) neither the kernel object nor the LSO points to the other.  Sadly, this means application-specific code for introducing and finalizing an object must run inside the server's mutex.

The kernel may be "interested" in a kernel object for one of a few reasons:
* It is a collectible true object with remote surrogates extant or still possible (the timeout hasn't expired).
* It is a true object on which one of the built-in methods is working.
* It is a collectible surrogate and the kernel is notifying the true server of the positive existance of the surrogate.
* It is a collectible surrogate and the kernel is notifying the true server that the surrogate is being deleted.
* It has an associated LSO.

We say the kernel is "very interested" in a kernel object if it is interested for any but the last two of the above reasons.  The kernel keeps the language runtime appraised of whether the kernel is currently very interested in each object.

The language runtime may, at any time, choose to disassociate the kernel object and the LSO.  A language with a garbage collector might choose to keep the object in a collector-visible global data structure during those times when the kernel is very interested in the object, and choose to disassociate the KO and LSO when the LSO is finalized.  A language with or without a garbage collector might choose to give the application the opportunity to explicitly disassociate the KO and LSO.

When the language runtime finds itself holding an LSO with no associated KO, it can choose either to declare the LSO "broken", or try to find or create a KO for it.  In the latter case, a disassociated true LSO will need to be holding the object's OID (ie, server plus server-relative ID); a disassociated surrogate LSO will need to be holding the full SBH of the object.  When the language runtime finds itself holding a KO with no associated LSO it may try to find or create an LSO, or --- if the object is true --- complain that the object is "closed".

As long as the kernel is interested, the object stays in the kernel server's hash table of objects.  When the kernel is not interested in the object, the kernel will un-table, destroy, and free the object.

The LS runtime, stubs, and application --- as well as the kernel --- promise to not hold onto a kernel object (directly --- indirect through an LSO or server is OK, because of the server's invariant) while outside its server's mutex.  Pay close attention to the locking dance done during marshalling and unmarshalling.  If an LS runtime promises to never disassociate a certain KO and LSO, that runtime may hold onto that KO outside its server's mutex.  This could be done for the true GC callback object exported by a client of collectible objects.  Similarly, where the kernel has indicated its "interest" in an object, the kernel may hold onto that object outside its server's mutex.
*/


/* ==================================== from debug.c */

/*L1, L2, Main unconstrained*/

_FN_DECL(void ilu_SetDebugLevel, (unsigned long bits));
_FN_DECL(void ilu_SetDebugLevelViaString, (char *spec));


/* ================ Time ================ */
/*L1, L2, Main unconstrained*/

extern ilu_cardinal ilu_FineTimeRate;

ilu_FineTime ilu_FineTime_Now(void);

ilu_FineTime ilu_FineTime_Add(ilu_FineTime a, ilu_FineTime b);

ilu_FineTime ilu_FineTime_Sub(ilu_FineTime a, ilu_FineTime b);

ilu_FineTime ilu_FineTime_Mul(ilu_FineTime a, float b);

ilu_integer ilu_FineTime_Cmp(ilu_FineTime a, ilu_FineTime b);
/* sgn(result) == sgn(a-b) */

#define ilu_FineTime_Eq(a, b) (((a).ft_s==(b).ft_s) && ((a).ft_t==(b).ft_t))

ilu_cardinal ilu_rescale(ilu_cardinal n, ilu_cardinal dfrom,
					 ilu_cardinal dto);
/* Returns floor(X(dto)*n/X(dfrom)), where
   X(c) = (double) (one more than the biggest ilu_cardinal) if c==0,
   X(c) = (double) c					    if c!=0.
   Caller guarantees 0 <= n < X(dfrom).*/

ilu_FineTime ilu_FineTime_FromDouble(double seconds);


/* ================ FD & Connection Management ================ */
/* Because we may open multiple connections to a server, we need some policy for when to close them.  That policy is this: the application gives the ILU kernel a "File Descriptor Budget".  The ILU kernel promises to use no more than this many File Descriptors at once.  [How much sense does this make for the Macintosh?  Other non-UNIX-like OSes?]  Off the top of this budget we take FDs needed for serving (one per listening socket and one per accept).  The remainder is allocated to outgoing connections (over transports that use FDs --- ie, not inmemory).  When we want to consume a new FD, and there's no room left in the budget, we go looking for an idle outgoing connection (one with no outstanding calls) to close.  All idle outgoing connections are kept in a doubly-linked list, ordered by when the connection went idle (most recently at the front). */

/*Main Invariant holds*/
_FN_DECL(ilu_cardinal ilu_SetFDBudget, (ilu_cardinal n));
/* Sets the FD budget to n, if possible.  This is impossible when n is smaller than the previous budget, and the kernel can't close enough idle outgoing connections to reach n; in this case the kernel sets the budget as low as it can.  In all cases the new budget is returned. */


/* ================ Retracted routines ================ */
/*

_FN_DECL(ilu_Object ilu_CreateObject, ( ilu_string ih,
	ilu_Server server, ilu_Class cl,
	ilu_refany languageSpecificObject, ilu_string sbh ));
 * Migrate away from this.  It has ill-defined locking. *

_FN_DECL(ilu_Connection ilu_ConnectionOfCall, (ilu_Call call));
 * The connection of the call is needed by the following procs. *

_FN_DECL(ilu_private ilu_CreateLock, ( ilu_refany lockedData ));

_FN_DECL(ilu_boolean ilu_AcquireReadLock, ( ilu_private lock ));

_FN_DECL(ilu_boolean ilu_ReleaseReadLock, ( ilu_private lock ));

_FN_DECL(ilu_boolean ilu_AcquireWriteLock, ( ilu_private lock ));

_FN_DECL(ilu_boolean ilu_ReleaseWriteLock, ( ilu_private lock ));

_FN_DECL(void ilu_DestroyLock, ( ilu_private lock ));

_FN_DECL(ilu_Method ilu_MethodOfCall, (ilu_Call call));

_FN_DECL(ilu_Object ilu_ObjectOfCall, (ilu_Call call));

_FN_DECL(ilu_cardinal ilu_SerialNumberOfCall, (ilu_Call call));

_FN_DECL(void ilu_DefaultObjectPort, ( ilu_Object obj,
				       ilu_Port port ));
 * If the given object doesn't yet have a SBH, it's given one that
 * uses the given port for contact info.  Of course, the given port
 * must be one through which the (necessarily true) object's server
 * is being exported. *

_FN_DECL(ilu_boolean ilu_DataPresent, ( ilu_Connection conn ));
 * Is there something to read here?  If not, but
 * ilu_BlockingWaitForInputOnConnection has been called since the
 * last read, ilu_DropConnection should be called. * 

_FN_DECL(ilu_bytes ilu_ReadPacket, ( ilu_Connection conn,
	ilu_cardinal *packetLength,
	ilu_PacketType *packetType,
	ilu_cardinal *packetSN ));
 * Read a call or reply.  Returns the entire protocol-level
 * packet, including *type and *serialNumber.
 * 

_FN_DECL(ilu_Call ilu_BuildCall, ( ilu_cardinal serialNumber,
	ilu_Connection conn, ilu_Object object, ilu_Method method ));
 * Upon receipt of a call packet, the server invokes ilu_BuildCall
 * with NULL object and method. * 

_FN_DECL(ilu_boolean ilu_InterpretPipeRequest, (ilu_Call call,
	ilu_bytes packet, ilu_cardinal len));

_FN_DECL(enum ilu_InterpretRequestStatus ilu_InterpretRequest,
	 (ilu_Call call, ilu_bytes packet, ilu_cardinal len));
 * Server calls this after ilu_BuildCall to finish filling
 * in the fields of the call.  Afterward, if o =
 * ilu_GetLanguageSpecificObject(call->object) isn't NULL,
 * the server invokes the stub, like this:
 * (*call->ca_method->me_stubproc)(call, (void *)o)
 * The stub completes the handling of the call.
 * The stub begins by un-marshalling the arguments.
 * The server then executes the call.
 * The server then sends the reply (more on that later).
 * The server is then done with the request.
 * 

_FN_DECL(void ilu_SweepGCCallbacks, ( void ));
 * Called periodically by server LS runtime to do periodic GC tasks. * 

*/



/* ================ Client side routines ================ */

/*Main Invariant holds; L1, L2 otherwise unconstrained*/

/*before: L1 = {};
  after:  result!=NULL => Inside(result's server, static_type);
  after:  result==NULL => L1 = {};
  forall conn: (L2 >= {conn.iomu}) => (L2 >= {conn.callmu});
  Main otherwise unconstrained*/
_FN_DECL(ilu_Object ilu_ObjectOfSBH, (ilu_string sbh,
				      ilu_string mstid,
				      ilu_Class static_type));
/* Importing an object begins with calling this procedure,
 * which returns the kernel representation of an object
 * (which may be true or a surrogate).
 * mstid is the unique_id of the true object; in general, it's
 * a subtype of static_type; mstid may be NULL, which means we don't know
 * (and will never find out!) any type more specific than the type
 * statically associated with "this position".
 * Storage of sbh and mstid is owned by the caller.
 * Neither sbh nor static_type may be NULL.
 * If result!=NULL && ilu_GetLanguageSpecificObject(result)==NULL,
 * the caller will invoke ilu_RegisterLanguageSpecificObject
 * on the result before unlocking the server. */

/*L1 >= {obj's server}; L2, Main unconstrained*/
_FN_DECL(ilu_refany ilu_GetLanguageSpecificObject, (ilu_Object obj));
/* Returns the language-specific object, if any, associated with
 * the given kernel object. */

/*Inside(obj's server, obj's type)*/
_FN_DECL(void ilu_RegisterLanguageSpecificObject,
	 (ilu_Object obj, ilu_refany lso));
/* Makes a link from the given kernel object to the given
   language-specific object; removes such a link if given lso==NULL.
   This clues the kernel into whether the application is using the object.
   If lso==NULL, this may provoke the kernel to destroy and free the
   kernel object; see "Deleting Objects" above.
   L1 mutexes are exited and re-entered inside this procedure!
 */

/*after: L2 >= {call's conn's callmu, iomu} iff result non-NULL*/
_FN_DECL(ilu_Call ilu_BeginCall, (ilu_Server server));
/* Client calls this to initiate a call.  Will return NULL
 * if a connection cannot be allocated.
 */

/*L2 >= {call's conn's callmu, iomu}*/
/*L1, Main unconstrained*/

/*L1_sup < prmu*/
_FN_DECL(ilu_boolean ilu_BeginRequest, (ilu_Call call,
	ilu_Class intro_type, ilu_Method method,
	ilu_cardinal argSize));
/* Client calls this to introduce the arguments.  intro_type is
 * the object type that introduced the method, and method is the rep'n
 * in that object type.  The size includes that of the discriminator,
 * if any.  Next, the arguments are mashalled, using the marshalling
 * routines introduced later. */

/*Main Invariant holds*/

_FN_DECL(ilu_boolean ilu_FinishRequest, (ilu_Call call));
/* End bracket for arguments.  If the call is of an ASYNCHRONOUS
 * method, the client proceeds to ilu_FinishCall; otherwise, the
 * client continues with ilu_GetReply. */

_FN_DECL(ilu_ProtocolException ilu_GetReply, (ilu_Call call,
	ilu_cardinal *errorStatus));
/* The client calls this to wait for the reply message
 * and begin processing it.  The result is success or a
 * protocol-level error.  If a protocol error is being reported,
 * the client stub next calls ilu_FinishCall, and then passes the
 * error to client code. This procedure also decodes whether the
 * marshalled results are normal results or an exception parameter;
 * `errorStatus` gets 0 if success is being reported, otherwise the
 * exception index.  The client next calls the appropriate
 * unmarshalling routine, then ilu_FinishCall, and finally returns
 * to the client code. */

/*L2    >=    {call's conn's callmu, iomu} before,
 *L2 disjoint {call's conn's callmu, iomu} after */
_FN_DECL(void ilu_FinishCall, (ilu_Call call));
/* Client calls this to clean up after a call:
 * after results/exceptions have been unmarshalled.
 * This procedure frees the call data structure. */

/*L1, L2, Main unconstrained*/

_FN_DECL(ilu_Exception ilu_ExceptionOfMethod, ( ilu_Method method,
						ilu_cardinal index));
/* This maps the exception index into a method-independent
 * (indeed, even object-type-independent) representation for
 * the exception.  This is useful for writing one exception
 * unmarshalling routine to share among all the methods of
 * an object type.  The index is 1+ the subscript into the
 * method's exceptionVector. */

/*L1 < cmu*/

_FN_DECL(void ilu_BankServer, ( ilu_Server s ));
/* Begin shutting down the given server.  Henceforth no more objects may be added to the server (for a surrogate server, this means unmarshalling currently unknown surrogates will fail).  Closes all the server's ports (if it's a true server), and each open connection as soon as its I/O mutex is not held.  An application or LS runtime can free damn near all a server's resources by calling this procedure and then unlinking every kernel and LS object in the server (the enumeration proc being wary for the changes in the objtab (including a call on ot_free_self!) that are consequences of that unlinking). */


/* ================ Object Type Registry ================ */

/*L1_sup < otmu*/
/*L2, Main unconstrained*/

_FN_DECL(void ilu_RegisterClass, ( ilu_Class c ));
    /* Must be called before any objects of this type are
     * unmarshalled. */

_FN_DECL(ilu_Class ilu_GetGcCallbackClass, (void));
/* This also registers the class, if necessary. */

_FN_DECL(ilu_Class ilu_FindClassFromID, ( char *unique_id ));
_FN_DECL(ilu_Class ilu_FindClassFromName, ( char *classname ));
    /* Ways to look up registered object types. */

_FN_DECL(ilu_boolean ilu_IsSubObjectType, ( ilu_Class a, ilu_Class b ));
/* Returns TRUE iff a is a subtype of b
   (including the degenerate case of a=b). */


/* ================ Server side ================ */

typedef struct ilu_ObjectTable_struct ilu_ObjectTable_s, *ilu_ObjectTable;

/*L1_sup < smu*/
/*L2, Main unconstrained*/

_FN_DECL(ilu_Server ilu_CreateTrueServer, ( ilu_string id,
					    ilu_ObjectTable objtab));
/* A server module starts by declaring its existence, with a call on ilu_CreateTrueServer.  id may not be NULL.  If a non-NULL objtab is given, the kernel will call its ot_object_of_ih when unmarshalling a reference to an object not currently in the server's hash table of objects; otherwise, only tabled objects may be unmarshalled.  Ownership of the arguments is associated with the result. */

_FN_DECL(ilu_string ilu_InventID, (void));
/* Generates a string that's unique over space and time.  A server with nothing better to use might call this to get an ID.  */

struct ilu_ObjectTable_struct {
  /* Fields are readonly.  Before and after calls:
     L1 >= {server}; L2, Main unconstrained.
   */
  
  /*L1 >= {server}; L1 >= {gcmu} if result is true and collectible*/
  _FN_DECL(ilu_Object (*ot_object_of_ih), (ilu_ObjectTable self,
					   ilu_string ih));
  /* Returns the object associated with the given instance handle,
     or NULL if no such object.  Caller owns ih.  The object
     returned is obtained by calling ilu_FindOrCreateTrueObject. */
  
  _FN_DECL(void (*ot_free_self), (ilu_ObjectTable self));
	/* The server using this object table is being closed,
	   ot_object_of_ih will not be called again.
	   Release appropriate resources and free(this struct*). */
  
  ilu_private ot_rock;
};
/* An object table gives the application the ability to create true objects upon presentation of an instance handle.  The object table is (ultimately) implemented by the application, and passed to the kernel through ilu_CreateTrueServer.  For those applications that don't need this, NULL can be passed. */

/*L1, L2, Main unconstrained*/

_FN_DECL(void ilu_SetDefaultServer, (ilu_Server s));
/* A server module may optionally declare itself to be the "default server" of its program; this has an effect on the behavior of ilu_FindOrCreateTrueObject (see below). */

_FN_DECL(ilu_Server ilu_GetDefaultServer, ( void ));
/* Returns the "default serer", or NULL if there is none. */

/*Main Invariant holds*/
_FN_DECL(ilu_Port ilu_CreatePort, ( ilu_Server s,
				    ilu_string protocolInfo,
				    ilu_string transportInfo ));
/* A server then creates a Port on which to listen for connection
 * requests.  The protocolInfo may be a prefix of a real
 * protocolInfo string; it must at least identify the protocol.
 * The transportInfo must syntactically be a full transportInfo;
 * it may have fields that mean "unspecified".
 * Caller owns the string arguments.
 */

/*L1_sup < s*/
_FN_DECL(void ilu_SetServerDefaultPort, ( ilu_Server s, ilu_Port p ));
/* If more than one port is created for a server, this operation specifies which of those ports is used to create contact info for objects in that server.  A no-op if the port is closed. */

/*L1 >= {the object's server};
  L1 >= {gcmu} if cl collectible*/
_FN_DECL(ilu_Object ilu_FindOrCreateTrueObject, ( ilu_string ih,
	ilu_Server server, ilu_Class cl,
	ilu_refany languageSpecificObject ));
/* This procedure is used for creating true objects.  It's called with non-NULL ih, server (maybe), cl, and languageSpecificObject.  If a "default server" has been declared, the caller may actually pass NULL for server, and ilu_FindOrCreateTrueObject will use the default server.  The LS runtime lets the application choose ih, and/or provides a default way of choosing ih.  Storage for ih is owned by the object.  If the kernel object already exists, its object type must be exactly cl.
*/

/*L2, Main unconstrained*/
/*L1 < port's server*/

_FN_DECL(int ilu_FileDescriptorOfMooringOfPort, ( ilu_Port port,
						  ilu_boolean *closed ));
/* The server then waits for connection requests to show up on the port.
 * A multi-threaded runtime does this by forking a thread per port;
 * a single-threaded one calls ilu_RegisterInputSource.  This proc
 * sets *closed; result is meaningful only if *closed is false. */

/*L1_sup < cmu*/

/*Main Invariant holds*/
_FN_DECL(ilu_Connection ilu_HandleNewConnection, ( ilu_Port port,
						   ilu_boolean *closed ));
/* When input shows up on the FD for a port, the server calls this
 * procedure to create a connection.  This proc returns a new
 * "incoming" connection (the other kind, "outgoing", don't appear
 * in the interface to the kernel) to the port's server.  This proc
 * sets *closed; result is meaningful only if *closed is false.
 * The result will be NULL if opening the connection now would
 * exceed the kernel's FD budget.
 * If the port is closed, the server may eventually call
 * ilu_DestroyPort. */

/*L1_sup < conn's server*/

_FN_DECL(int ilu_FileDescriptorOfConnection, ( ilu_Connection conn ));
/* Should probably go away. */

/*Main Invariant holds*/
_FN_DECL(ilu_boolean ilu_BlockingWaitForInputOnConnection,
	 ( ilu_Connection conn, ilu_FineTime *limit ));
/* The server then waits for input to show up on that connection.
 * Again, a multi-threaded runtime forks a thread that calls
 * this procedure (passing NULL for limit means +infinity),
 * a single-threaded runtime uses ilu_RegisterInputSource.
 * ilu_BlockingWaitForInputOnConnection returns FALSE if there was an
 * error, in which case ilu_CloseConnection should then be called.
 * After input allegedly shows up, the server begins processing the
 * request by calling ilu_ReceiveRequest. */

typedef enum {ilu_conn_closed,	/* this connection's closed or broken */
	      ilu_no_request,	/* wasn't really any input waiting */
	      ilu_read_failed,	/* err sys may give more info */
	      ilu_not_request,	/* wrong type message received */
	      ilu_interp_failed,/* protocol's interp proc failed */
	      ilu_builtin_meth,	/* built-in; already handled */
	      ilu_good_request	/* decent message received */
	     } ilu_ReceiveRequestStatus;

/*Main Invariant holds*/
/*before: L2 disjoint {conn's callmu, iomu},
 *after:  L2     >=   {conn's callmu, iomu} if result==ilu_good_request,
 *after:  L2 disjoint {conn's callmu, iomu} if result!=ilu_good_request */
 
_FN_DECL(ilu_ReceiveRequestStatus ilu_ReceiveRequest,
	 ( ilu_Connection conn,	ilu_Call *call, ilu_Class *intro_type,
	   ilu_Method *meth, ilu_cardinal *sn ));
/* A server calls this to start processing a request.  If the result is ilu_good_request, ilu_RequestRead and then one of ilu_BeginReply, ilu_BeginException, or ilu_NoReply must later be called.  When result is ilu_good_request, this procedure stores through call, intro_type, and meth; these arguments are otherwise unused.  This procedure stores through sn when result is ilu_good_request, and in certain other circumstances.  When result is ilu_conn_closed, the server should eventually call ilu_CloseConnection (the connection might be broken but not yet closed).  If result is ilu_good_request then the server invokes the stub, like this:
	(*meth->me_stubproc)(call).
The language-specific runtime can use the "private" field of the ilu_Call to pass other information to the stub.
*/

_FN_DECL(ilu_ReceiveRequestStatus ilu_ReceivePipeRequest,
	 ( ilu_Connection conn,	ilu_Call *call, ilu_Class *intro_type,
	   ilu_Method *meth,	ilu_cardinal *sn ));
/* A server calls this to start processing a pipe request.  If the result is ilu_good_request, ilu_RequestRead and then one of ilu_BeginReply, ilu_BeginException, or ilu_NoReply must later be called.  When result is ilu_good_request, this procedure stores through call, intro_type, and meth; these arguments are otherwise unused.  This procedure stores through sn when result is ilu_good_request, and in certain other circumstances.  When result is ilu_conn_closed, the server should eventually call ilu_CloseConnection. */

/*L1, L2, Main unconstrained*/
_FN_DECL(ilu_boolean ilu_ThreadPerRequest, ( ilu_Connection conn ));
/* After calling ilu_ReceiveRequest, and before invoking the stub,
 * a multithreaded runtime consults this procedure to decide whether
 * to fork a thread to process this request. */

/*before: L1 = {},
  after:  result!=NULL => Inside(s, intro_type);
  after:  result==NULL => L1 = {};
  forall conn: (L2 >= {conn.iomu}) => (L2 >= {conn.callmu});
  Main otherwise unconstrained*/
ilu_Object ilu_GetServerSingleton(ilu_Server s, ilu_Class intro_type);
/* The stub unmarshalls the arguments, beginning with the discriminator.
   If call->ca_intro_type is a singleton, ilu_GetServerSingleton is called
   to get the discriminator; otherwise, the discriminator is unmarshalled
   by a call on ilu_InputObjectID.
 */

/*L1_sup < cmu*/
/*Main unconstrained*/

/*before: L2     >= {call's conn's callmu, iomu},
 *after:  L2 not >= {call's conn's         iomu},
 *after:  L2     >= {call's conn's callmu} iff protocol not concurrent*/
_FN_DECL(void ilu_RequestRead, (ilu_Call call));
/* A server stub calls this after unmarshalling the arguments,
 * before executing the procedure. */


/*before: L2 not >=   {call's conn's iomu},
	  L2     >=   {call's conn's callmu} iff protocol not concurrent;
   after: L2     >=   {call's conn's callmu, iomu} if result is true,
	  L2 disjoint {call's conn's callmu, iomu} if result is false. */

_FN_DECL(ilu_boolean ilu_BeginReply, (ilu_Call call,
	ilu_boolean exceptions, ilu_cardinal argSize));
/* Server stub calls this to introduce successful results.
 * `exceptions` indicates whether this call might raise
 * exceptions.  If result is TRUE, the results are marshalled;
 * if result is FALSE, done with request. */

_FN_DECL(ilu_boolean ilu_BeginException, (ilu_Call call,
	ilu_cardinal evalue, ilu_cardinal argSize));
/* If the call should raise an exception instead of return some
 * results, the server stub calls this (instead of ilu_BeginReply)
 * to introduce the exception&parameter.  For protocol exceptions, evalue
 * is 0 and argSize is the ilu_ProtocolException.  For programmer-defined
 * exceptions, evalue is 1 + the subscript into the method's
 * exceptionVector and argSize is the marshalled size of the
 * exeption parameter.  If result is TRUE, then the exception parameter
 * is marshalled next; otherwise, done with request. */

/*Main Invariant holds*/
/*L2    >=    {call's conn's callmu, iomu} before,
 *L2 disjoint {call's conn's callmu, iomu} after*/

_FN_DECL(ilu_boolean ilu_FinishReply, (ilu_Call call));
/* End bracket for success results.  Done with request. */

_FN_DECL(ilu_boolean ilu_FinishException, (ilu_Call call));
/* End bracket for exception result.  Done with request. */

/*before: L2 not >=   {call's conn's iomu},
	  L2     >=   {call's conn's callmu} iff protocol not concurrent;
  after:  L2 disjoint {call's conn's callmu, iomu}*/
_FN_DECL(void ilu_NoReply, (ilu_Call call));
/* The server stub calls this for asynchronous methods.
 * Done with request. */

/*L2 = {}*/
/*L1_sup < cmu*/

_FN_DECL(void ilu_CloseConnection, ( ilu_Connection conn ));
/* A true server was serving on conn, and calls this to cease serving.
 * This procedure can *ONLY* be called on "incoming" connections.
 */

_FN_DECL(void ilu_DestroyConnection, ( ilu_Connection conn ));
/* This is called on an incoming connection after ilu_CloseConnection
 * has been called and when the true server is not processing (and will
 * never again process) any calls from this connection.
 * This procedure frees conn.
 *
 * !!! MISTAKES WILL CAUSE MEMORY SMASHES !!!
 *
 * This proc can be called correctly because incoming connections
 * appear only in an ilu_Server or an incoming ilu_Call or in the args
 * of certain kernel interface procedures.  The connection can be
 * reached from an ilu_Server only while holding the server lock.
 * An ilu_Call appears only in kernel interface procs that the true server
 * is guaranteeing never again to call with an ilu_Call from conn,
 * and in an ilu_Pipe --- whose locking isn't designed yet.
 * The true server is guaranteeing to never again make any kernel calls
 * that explicitly mention this connection.
 *
 */

/*L2, Main unconstrained*/
_FN_DECL(void ilu_ClosePort, (ilu_Port port));
/* A true server can call this to cease exporting itself through
 * the given port.  If the port was the server's default port,
 * some other port (if there are any) is chosen to be the default.
 * Since the contact info in an SBH mentions only one port, woe
 * unto a client that uses an SBH for a closed port.
 */

_FN_DECL(void ilu_DestroyPort, ( ilu_Port port ));
/* This is called on a port after ilu_ClosePort
 * has been called and when the true server is not processing (and will
 * never again process) any connections to this port.
 * This procedure frees port.
 *
 * !!! MISTAKES WILL CAUSE MEMORY SMASHES !!!
 *
 * This proc can be called correctly because a port appears only in
 * an ilu_Connection, an ilu_Server, or a call on a kernel interface
 * procedure.  The server is guaranteeing there are no open connections
 * from this port, and never again to call a kernel interface proc with
 * this port; the server lock is held while accessing ports of the server.
 *
 */

/* ================ Concurrent I/O routines ================ */

/*Main Invariant holds; L1, L2 otherwise unconstrained*/
_FN_DECL(void ilu_RunMainLoop, (int *stop) );
/* A single-threaded runtime calls this to animate all true servers, handle the alarm, and do any other registered input processing.  A multi-threaded runtime never calls this procedure, instead forking threads to animate servers and pass the time.  This procedure may be invoked recursively, so long as no two cuncurrent invocations are given the address for "stop".  This procedure processes input and time until ilu_ExitMainLoop is invoked on stop. */

/*L1, L2, Main unconstrained; synch provided by single-threadedness*/

_FN_DECL(void ilu_ExitMainLoop, (int *stop) );
/* This causes the current invocation of ilu_RunMainLoop on stop to return once it's done with the input handler or alarm it's currently executing. */

_FN_DECL(ilu_boolean ilu_RegisterInputSource, (int fd,
	/*Main Invariant holds; L1, L2 otherwise unconstrained*/
	void (*proc)(int fd, ilu_private rock),
	ilu_private rock) );
/* A single-threaded runtime calls this procedure to declare how to handle input on a given FD.  It returns FALSE if it can't do its job due to some resource limitation.  ilu_UnregisterInputSource must be called on this FD before ilu_RegisterInputSource can be called on it again.  */

_FN_DECL(ilu_boolean ilu_UnregisterInputSource, (int fd));
/* A single-threaded runtime calls this procedure to cease handling input on an FD.  It returns FALSE if input on the FD wasn't being handled. */

_FN_DECL(ilu_boolean ilu_RegisterOutputSource, (int fd,
	/*Main Invariant holds; L1, L2 otherwise unconstrained*/
	void (*proc)(int fd, ilu_private rock),
	ilu_private rock) );
/* A single-threaded runtime calls this procedure to queue output for an FD.  It returns FALSE if it can't do its job due to some resource limitation.  ilu_UnregisterOutputSource must be called on this FD before ilu_RegisterOutputSource can be called on it again.  */

_FN_DECL(ilu_boolean ilu_UnregisterOutputSource, (int fd));
/* A single-threaded runtime calls this procedure to cease queueing output on an FD.  It returns FALSE if output wasn't being queued on the FD. */

/*L1_sup < timu; L2, Main unconstrained*/

_FN_DECL(ilu_refany ilu_CreateAlarm, (void));
/* Available in both single-threaded and multi-threaded environments.
   Creates a (re)settable alarm. */

_FN_DECL(void ilu_SetAlarm, (ilu_refany alarm, ilu_FineTime t,
			     /*for invoking: Main Invariant holds*/
			     void (*proc)(ilu_private rock),
			     ilu_private rock));
/* An alarm has a
 * trigger time and a closure.  The closure is invoked once, as soon after
 * the trigger time as the runtime is able.  ilu_SetAlarm overwrites the
 * previous setting of the alarm.
 */

_FN_DECL(void ilu_UnsetAlarm, (ilu_refany alarm));
/* Effectively sets the trigger time to infinity. */

typedef struct {
	/* These fields are readonly*/
	
	/*Main Invariant holds; L1, L2 otherwise unconstrained*/
	_FN_DECL(void (*ml_run), (int *stop) );
	
	/*L1, L2, Main unconstrained*/
	
	_FN_DECL(void (*ml_exit), (int *stop) );
	_FN_DECL(ilu_boolean (*ml_register_input), (
		int fd,
		/*Main Invariant holds;
		  L2 otherwise unconstrained*/
		void (*proc)(int fd, ilu_private rock),
		ilu_private rock ) );
	_FN_DECL(ilu_boolean (*ml_unregister_input), (int fd));
	_FN_DECL(ilu_boolean (*ml_register_output), (
		int fd,
		/*Main Invariant holds;
		  L2 otherwise unconstrained*/
		void (*proc)(int fd, ilu_private rock),
		ilu_private rock ) );
	_FN_DECL(ilu_boolean (*ml_unregister_output), (int fd));
	
	/*L1_sup < timu*/
	
	_FN_DECL(ilu_refany (*ml_create_alarm), (void));
	_FN_DECL(void (*ml_set_alarm), (ilu_refany alarm, ilu_FineTime t,
				     /*Main Invariant holds;
				       L2 otherwise unconstrained*/
				     void (*proc)(ilu_private rock),
				     ilu_private rock));
	_FN_DECL(void (*ml_unset_alarm), (ilu_refany alarm));
} ilu_MainLoop;

/*L1, L2, Main unconstrained; synch provided by single-threadedness*/

_FN_DECL(void ilu_SetMainLoop, (ilu_MainLoop *ml) );
/* A single-threaded runtime, or an application running thereon, can call this procedure to supply a non-standard implementation of the main loop (eg, the main loop of another toolkit).  A multi-threaded runtime calls this to supply the implementation of alarms; the other procedure slots are NULL.  This procedure should be called before any calls on ilu_RunMainLoop, ilu_ExitMainLoop, ilu_RegisterInputSource, ilu_UnregisterInputSource, ilu_RegisterOutputSource, ilu_UnregisterOutputSource, ilu_CreateAlarm, ilu_SetAlarm, or ilu_UnsetAlarm; when we get our error system we can report violations.  The storage for the argument is never freed.  The kernel makes few enough calls on create_alarm that each could fork a new thread (in a multi-threaded runtime); the application's demands are not constrained. */

typedef struct {
	/* These fields are readonly*/
	/*For calling: Main Invariant holds, L2 otherwise unconstrained*/

	_FN_DECL(void (* wt_read_wait), (int fd, ilu_boolean *sure,
					 ilu_FineTime *limit));
	_FN_DECL(void (* wt_write_wait), (int fd, ilu_boolean *sure,
					  ilu_FineTime *limit));

} ilu_WaitTech;
/* These two procedures return ASAP after any of the following three conditions becomes true:

(1) the appropriate kind of I/O can be done on the given file descriptor without blocking,

(2) when an exceptional condition exists on the FD, or

(3) when *limit (+infinity if limit==NULL) is exceeded.

These procedures set *sure.  When *sure is set true, one of the first two conditions held; when *sure is set false, the third condition held (and still does, of course).  This data structure is only for use in multi-threaded programs, and these procedures block only the calling thread. */

/*L1, L2, Main unconstrained; called only from start code*/

_FN_DECL(void ilu_SetWaitTech, (ilu_WaitTech *wt));
/* A multi-threaded runtime (even one using a kernel threads package, because the default implementation calls ilu_RunMainLoop) calls this procedure to supply the means to block a thread until it can do I/O on a given file descriptor. */


/* ================ Alarm Multiplexing ================ */
/* These data structures and procedures are useful for multiplexing multiple alarms on top of one alarm.  The client constructs procedures with the proper alarming signatures by calling the procedures below to do most of the work.  The data structures and procedures below operate within some mutex provided by the client; call that mutex mxamu. */

typedef struct _ilu_Alarmette_s ilu_Alarmette_s, *ilu_Alarmette;

/*L2, Main unconstrained*/

struct _ilu_Alarmette_s {
  /*L1 >= {mxamu}*/
  
  ilu_Alarmette al_next, al_prev;	/* list links */
  ilu_boolean  al_set;			/* in queue? */
  ilu_FineTime al_trigger;		/* when to execute */
};
/* A data structure common to all alarms multiplexed into other alarms.
   When set==TRUE, next and prev are non-NULL; when set==FALSE,
   they are NULL.  Initialize set to FALSE and next and prev to NULL. */

typedef struct {
  /*L1 >= {mxamu} for access and invocations*/
  
  ilu_Alarmette ar_head;	/* of queue of alarmettes yet to trigger */
  /*for calling: L1_sup = mxamu, & other things true of ilu_MXAProc(..)*/
  void (*ar_invoke)(ilu_Alarmette a);    /* invoke one now */
  void (*ar_set   )(ilu_FineTime t);   /* schedule a call on ilu_MXAProc */
  void (*ar_cancel)(void);	      /* cancel that */
} ilu_AlarmRep;
/* Data and procedures provided by the client.
   head->next and head->prev are initialized to head.
   urset and urcancel manipulate the one alarm into which Alarmettes
   are multiplexed by the procedures below. */

/*L1 >= {mxamu}*/

void ilu_MXASet(ilu_AlarmRep *ar, ilu_Alarmette a, ilu_FineTime t);
/* Schedule (*ar->invoke)(a) to happen ASAP after t;
   this replaces any other scheduled invocation of a. */

void ilu_MXAClear(ilu_AlarmRep *ar, ilu_Alarmette a);
/* Cancel the scheduled invocation of a, if any. */

/*L1_sup = mxamu*/
void ilu_MXAProc(ilu_FineTime u, ilu_AlarmRep *ar);
/* It's now u; make all the (*ar->invoke)() calls that are scheduled
   to have happened by now, but haven't. */


/* ================ (Un)Marshalling routines ================ */

/*L2 >= {call's connection's callmu, iomu}*/
/*L1, Main unconstrained*/
/*(When we cease buffering entire messages: Main Invariant holds)*/

/* End brackets for [un]marshalling and size routines;
 * call these after the contents introduced by a call on
 * (Output|Input|SizeOf)(Sequence|Union|Array|Record).
 */
_FN_DECL(void ilu_EndSequence, (ilu_Call call));
_FN_DECL(void ilu_EndUnion, (ilu_Call call));
_FN_DECL(void ilu_EndArray, (ilu_Call call));
_FN_DECL(void ilu_EndRecord, (ilu_Call call));

/* Marshalling routines */
_FN_DECL(void ilu_OutputShortInteger, (ilu_Call call,
				       ilu_shortinteger i));
_FN_DECL(void ilu_OutputInteger, (ilu_Call call, ilu_integer i));
_FN_DECL(void ilu_OutputLongInteger, (ilu_Call call,
				       ilu_longinteger i));
_FN_DECL(void ilu_OutputShortCardinal, (ilu_Call call,
					ilu_shortcardinal i));
_FN_DECL(void ilu_OutputCardinal, (ilu_Call call, ilu_cardinal i));
_FN_DECL(void ilu_OutputLongCardinal, (ilu_Call call,
					ilu_longcardinal i));
_FN_DECL(void ilu_OutputShortReal, (ilu_Call call, float f));
_FN_DECL(void ilu_OutputReal, (ilu_Call call, double d));
_FN_DECL(void ilu_OutputLongReal, (ilu_Call call, ilu_longreal f));
_FN_DECL(void ilu_OutputEnum, (ilu_Call call,
			       ilu_shortcardinal i));
_FN_DECL(void ilu_OutputCharacter, (ilu_Call call,
				    ilu_character i));
_FN_DECL(void ilu_OutputByte, (ilu_Call call, ilu_byte b));
_FN_DECL(void ilu_OutputBoolean, (ilu_Call call, ilu_boolean b));
_FN_DECL(void ilu_OutputOptional, (ilu_Call call,
				   ilu_boolean optionalStatus));

/* Output an OPTIONAL X or an X; `optional` indicates
 * which.  If `optional`, `provided` indicates the value is not
 * NULL. */
_FN_DECL(void ilu_OutputSequence, (ilu_Call call,
	ilu_cardinal len, ilu_cardinal limit, ilu_boolean optional,
	ilu_boolean provided));
_FN_DECL(void ilu_OutputSequenceMark, (ilu_Call call,
	ilu_cardinal extent));
    /* Call this every 2^16-1 elements.  ??? What's extent ??? */
_FN_DECL(void ilu_OutputUnion, (ilu_Call call,
	ilu_shortcardinal discriminator, ilu_boolean optional,
	ilu_boolean provided));
_FN_DECL(void ilu_OutputArray, (ilu_Call call, ilu_cardinal length,
	ilu_boolean optional, ilu_boolean provided));
_FN_DECL(void ilu_OutputRecord, (ilu_Call call,
	ilu_boolean optional, ilu_boolean provided));

/* Output an OPTIONAL X or an X; `optional` indicates
 * which.  If `optional`, the value may be NULL. */
_FN_DECL(void ilu_OutputString, (ilu_Call call, ilu_string s,
	ilu_cardinal len, ilu_cardinal limit,
	ilu_boolean optional));
    /* Variable-length array of short character. */

_FN_DECL(void ilu_OutputStringVec, (ilu_Call call, ilu_string s,
	ilu_cardinal len, ilu_boolean optional));
    /* Fixed-length array of short character. */

_FN_DECL(void ilu_OutputWString, (ilu_Call call, ilu_wstring s,
	ilu_cardinal len, ilu_cardinal limit, ilu_string *buf,
	ilu_cardinal *blen, ilu_boolean optional));
    /* Variable-length array of character.  If buf!=NULL and
       blen!=NULL, the UTF-2 encoding is stored there. */

_FN_DECL(void ilu_OutputWStringVec, (ilu_Call call, ilu_wstring s,
	ilu_cardinal len, ilu_string *buf, ilu_cardinal *blen,
	ilu_boolean optional));
    /* Fixed-length array of character. */

_FN_DECL(void ilu_OutputBytes, (ilu_Call call, ilu_bytes o,
	ilu_cardinal len, ilu_cardinal limit,
	ilu_boolean optional));
    /* Variable-length array of byte. */

_FN_DECL(void ilu_OutputOpaque, (ilu_Call call, ilu_opaque o,
	ilu_cardinal len, ilu_boolean optional));
    /* Fixed-length array of byte. */

/*before: Inside(s, cl);
  after:				 L1 disjoint {cmu, s};
  after: cl collectible		      => L1  not >=  {gcmu};
  after: cl collectible & s surrogate => Main Invariant holds;
  where s = h's server and cl = h's type.
  (We don't really need to hold cmu for surrogate or non-collectible
   objects, but this is convenient because ilu_Enter/ExitServer can
   be used.)*/
_FN_DECL(void ilu_OutputObjectID, (ilu_Call call, ilu_Object obj,
	ilu_boolean discriminator_p, ilu_Class static_type));
/* Output a object; `discriminator_p` iff in discriminator position. */

/* Un-marshalling routines */
/*L2 >= {call's connection's callmu, iomu}*/
/*L1, Main unconstrained*/

_FN_DECL(void ilu_InputShortInteger, (ilu_Call call,
				      ilu_shortinteger *i));
_FN_DECL(void ilu_InputInteger, (ilu_Call call, ilu_integer *i));
_FN_DECL(void ilu_InputLongInteger, (ilu_Call call,
				      ilu_longinteger *i));
_FN_DECL(void ilu_InputShortCardinal, (ilu_Call call,
				       ilu_shortcardinal *i));
_FN_DECL(void ilu_InputCardinal, (ilu_Call call, ilu_cardinal *i));
_FN_DECL(void ilu_InputLongCardinal, (ilu_Call call,
				       ilu_longcardinal *i));
_FN_DECL(void ilu_InputShortReal, (ilu_Call call, float *f));
_FN_DECL(void ilu_InputReal, (ilu_Call call, double *d));
_FN_DECL(void ilu_InputLongReal, (ilu_Call call, ilu_longreal *f));
_FN_DECL(void ilu_InputEnum, (ilu_Call call,
			      ilu_shortcardinal *i));
_FN_DECL(void ilu_InputCharacter, (ilu_Call call,
				   ilu_character *i));
_FN_DECL(void ilu_InputByte, (ilu_Call call, ilu_byte *b));
_FN_DECL(void ilu_InputBoolean, (ilu_Call call, ilu_boolean *b));
_FN_DECL(void ilu_InputOptional, (ilu_Call call,
				  ilu_boolean *optionalStatus));
_FN_DECL(void ilu_InputSequence, (ilu_Call call,
	ilu_cardinal *len, ilu_cardinal limit,
	ilu_boolean optional, ilu_boolean *provided));
_FN_DECL(void ilu_InputSequenceMark, (ilu_Call call,
				      ilu_cardinal extent));
_FN_DECL(void ilu_InputUnion, (ilu_Call call,
	ilu_shortcardinal *discriminator, ilu_boolean optional,
	ilu_boolean *provided));
_FN_DECL(void ilu_InputArray, (ilu_Call call,
	ilu_boolean optional, ilu_boolean *provided));
_FN_DECL(void ilu_InputRecord, (ilu_Call call,
	ilu_boolean optional, ilu_boolean *provided));

_FN_DECL(void ilu_InputString, (ilu_Call call, ilu_string *s,
	ilu_cardinal *len, ilu_cardinal limit,
	ilu_boolean optional));
/* Sets *len to be the length of the fetched string.  Caller may
 * pass non-null *s, in which case the string will be stored there
 * (with no length checking!); otherwise, callee will allocate
 * just enough space (and return ownership to caller). */

_FN_DECL(void ilu_InputStringVec, (ilu_Call call, ilu_string *s,
	ilu_cardinal len, ilu_boolean optional));
_FN_DECL(void ilu_InputWString, (ilu_Call call, ilu_wstring *s,
	ilu_cardinal *len, ilu_cardinal limit,
	ilu_boolean optional));
_FN_DECL(void ilu_InputWStringVec, (ilu_Call call, ilu_wstring *s,
	ilu_cardinal len, ilu_boolean optional));
_FN_DECL(void ilu_InputOpaque, (ilu_Call call, ilu_opaque *o,
	ilu_cardinal len, ilu_boolean optional));
_FN_DECL(void ilu_InputBytes, (ilu_Call call, ilu_bytes *o,
	ilu_cardinal *len, ilu_cardinal limit,
	ilu_boolean optional));

/*before: L1 = {},
  after:  *o!=NULL => Inside(*o's server, static_type);
  after:  *o==NULL => L1 = {};
  forall conn: (L2 >= {conn.iomu}) => (L2 >= {conn.callmu});
  L2 >= {call's connection's callmu, iomu};
  Main otherwise unconstrained*/
_FN_DECL(void ilu_InputObjectID, (ilu_Call call, ilu_Object *o,
	ilu_boolean discriminator_p, ilu_Class static_type));
/* static_type is not NULL.
   Afterward, if *o!=NULL && ilu_GetLanguageSpecificObject(*o)==NULL,
   the caller will invoke ilu_RegisterLanguageSpecificObject
   on *o before unlocking the server. */


/* Size-computing routines */
/*L1, L2, Main unconstrained*/

_FN_DECL(ilu_cardinal ilu_SizeOfShortInteger, (ilu_Call call,
					    ilu_shortinteger i));
_FN_DECL(ilu_cardinal ilu_SizeOfInteger, (ilu_Call call,
					  ilu_integer i));
_FN_DECL(ilu_cardinal ilu_SizeOfLongInteger, (ilu_Call call,
					    ilu_longinteger i));
_FN_DECL(ilu_cardinal ilu_SizeOfShortCardinal, (ilu_Call call,
					    ilu_shortcardinal i));
_FN_DECL(ilu_cardinal ilu_SizeOfCardinal, (ilu_Call call,
					   ilu_cardinal i));
_FN_DECL(ilu_cardinal ilu_SizeOfLongCardinal, (ilu_Call call,
					    ilu_longcardinal i));
_FN_DECL(ilu_cardinal ilu_SizeOfShortReal, (ilu_Call call,
					    float d));
_FN_DECL(ilu_cardinal ilu_SizeOfReal, (ilu_Call call, double d));
_FN_DECL(ilu_cardinal ilu_SizeOfLongReal, (ilu_Call call,
					    ilu_longreal d));
_FN_DECL(ilu_cardinal ilu_SizeOfEnum, (ilu_Call call,
				       ilu_shortcardinal i));
_FN_DECL(ilu_cardinal ilu_SizeOfCharacter, (ilu_Call call,
					    ilu_character i));
_FN_DECL(ilu_cardinal ilu_SizeOfByte, (ilu_Call call, ilu_byte i));
_FN_DECL(ilu_cardinal ilu_SizeOfBoolean, (ilu_Call call, ilu_boolean i));
_FN_DECL(ilu_cardinal ilu_SizeOfOptional, (ilu_Call call,
				ilu_boolean optionalStatus));
_FN_DECL(ilu_cardinal ilu_SizeOfSequence, (ilu_Call call,
	ilu_cardinal len, ilu_cardinal limit, ilu_boolean optional,
	ilu_boolean provided));
_FN_DECL(ilu_cardinal ilu_SizeOfUnion, (ilu_Call call,
	ilu_shortcardinal discriminator, ilu_boolean optional,
	ilu_boolean provided));
_FN_DECL(ilu_cardinal ilu_SizeOfArray, (ilu_Call call,ilu_cardinal length,
	ilu_boolean optional, ilu_boolean provided));
_FN_DECL(ilu_cardinal ilu_SizeOfRecord, (ilu_Call call,
	ilu_boolean optional, ilu_boolean provided));
_FN_DECL(ilu_cardinal ilu_SizeOfString, (ilu_Call call,
	ilu_string i, ilu_cardinal l, ilu_cardinal limit,
	ilu_boolean optional));
_FN_DECL(ilu_cardinal ilu_SizeOfStringVec, (ilu_Call call,
	ilu_string i, ilu_cardinal l, ilu_boolean optional));
_FN_DECL(ilu_cardinal ilu_SizeOfWString, (ilu_Call call,
	ilu_wstring i, ilu_cardinal l, ilu_cardinal limit,
	ilu_string *buf, ilu_cardinal *blen,
	ilu_boolean optional));
_FN_DECL(ilu_cardinal ilu_SizeOfWStringVec, (ilu_Call call,
	ilu_wstring i, ilu_cardinal l, ilu_string *buf,
	ilu_cardinal *blen, ilu_boolean optional));

_FN_DECL(ilu_cardinal ilu_SizeOfOpaque, (ilu_Call call,
	ilu_opaque o, ilu_cardinal l, ilu_boolean optional));

_FN_DECL(ilu_cardinal ilu_SizeOfBytes, (ilu_Call call, ilu_bytes o,
	ilu_cardinal l, ilu_cardinal limit, ilu_boolean optional));

/*L1 >= {obj's server}*/
_FN_DECL(ilu_cardinal ilu_SizeOfObjectID, (ilu_Call call,
	ilu_Object obj, ilu_boolean discriminator_p,
	ilu_Class static_type));


/* ================ Simple Binding ================ */

/*before: Inside(s, cl)
  after:				 L1 disjoint {cmu, s};
  after: cl collectible		      => L1  not >=  {gcmu};
  after: cl collectible & s surrogate => Main Invariant holds;
  where s = obj's server and cl = obj's type.
  (We don't really need to hold cmu for surrogate or non-collectible
   objects, but this is convenient because ilu_Enter/ExitServer can
   be used.)*/
_FN_DECL(/* OPTIONAL */ char * ilu_PublishObject, (ilu_Object obj));
/* Publishes the SBH of the object in the local object domain.  Returns an "ownership proof", a string which must be supplied to withdraw the object */

/*before: Inside(s, cl)
  after:				 L1 disjoint {cmu, s};
  after: cl collectible		      => L1  not >=  {gcmu};
  after: cl collectible & s surrogate => Main Invariant holds;
  where s = obj's server and cl = obj's type.
  (We don't really need to hold cmu for surrogate or non-collectible
   objects, but this is convenient because ilu_Enter/ExitServer can
   be used.)*/
_FN_DECL(ilu_boolean ilu_WithdrawObject, (ilu_Object obj, /* PASS */ char * ownership_proof));
/* Withdraws the object "obj", if "ownership_proof" is that returned when the object was registered. */

/*before: L1 = {};
  after:  result!=NULL => Inside(result's server, pclass);
  after:  result==NULL => L1 = {};
  forall conn: (L2 >= {conn.iomu}) => (L2 >= {conn.callmu});
  Main otherwise unconstrained */
_FN_DECL(/* OPTIONAL */ ilu_Object ilu_LookupObject, (char *oid, ilu_Class pclass));
/* Attempts to find an object with OID "oid" in the local object domain.  "pclass" is the putative class of the object. */

/* ================ Other routines ================ */

/*L1, L2, Main unconstrained*/

_FN_DECL(ilu_boolean ilu_ParseStringBindingHandle,
	(ilu_string sbh, ilu_string *oid,
	 ilu_string *serverhandle));
    /* If oid#NULL, stores into *oid a new string;
     * same for serverhandle. */

_FN_DECL(ilu_boolean ilu_ParseOID, ( ilu_string oid,
	ilu_string *instance_handle, ilu_string *server_id));
    /* Parse an object-id, like ParseStringBindingHandle. */

/*Main Invariant holds; L2 otherwise unconstrained*/
_FN_DECL(ilu_boolean ilu_PingObject, (ilu_Object o));
/* Returns ilu_TRUE if the true object exists, and the process serving it can be contacted; ilu_FALSE otherwise. */

/*L1 >= {obj's server}; L1_sup < prmu*/
_FN_DECL(ilu_string ilu_SBHOfObject, ( ilu_Object obj ));
/* Returns the string owned by the object.  Allocates a new one, if necessary.  May return NULL if the object's server isn't exported through any port; may return an invalid one if the cached SBH references a closed port. */

_FN_DECL(ilu_string ilu_MstidOfObject, ( ilu_Object obj ));
/* Returns the ID of the most specific type of the given object.  May return NULL (why?).  Storage for result owned by the object.  Caller should thus hold obj's server in order to know the object won't be freed upon return. */

_FN_DECL(ilu_string ilu_GetILUVersion, (void));

_FN_DECL(void ilu_SetCommunicationsErrorHandler,
	(void (*handlerProc)(ilu_Call call, ilu_integer err,
	 ilu_cardinal io_dir) ));

_FN_DECL(ilu_cardinal ilu_IDOfMethod, (ilu_Method method));

_FN_DECL(ilu_cardinal ilu_ExceptionCountOfMethod, (ilu_Method method));

_FN_DECL(ilu_string ilu_NameOfMethod, (ilu_Method method));

_FN_DECL(void ilu_SetMethodStubProc, (ilu_Method method, void (*proc)(ilu_Call)));

_FN_DECL(ilu_refany ilu_GetMethodStubProc, (ilu_Method method));

_FN_DECL(ilu_Method ilu_FindMethodByID, ( ilu_Class intro_type,
					  ilu_cardinal ID ));

_FN_DECL(ilu_string ilu_IDOfServer, ( ilu_Server s ));

_FN_DECL(ilu_boolean ilu_TrueServerP, ( ilu_Server s ));


/*L1, L2, Main unconstrained*/
/*(But be careful about holding directly onto an ilu_Object)*/

_FN_DECL(ilu_Class ilu_ClassOfObject, ( ilu_Object obj ));

_FN_DECL(ilu_boolean ilu_TrueInstanceP, ( ilu_Object obj ));

_FN_DECL(ilu_Server ilu_ServerOfObject, ( ilu_Object obj ));

_FN_DECL(ilu_string ilu_IhOfObject, ( ilu_Object o ));
/* Return the ih of the given object; result is owned by the object.
   This could be useful when mapping a disassociated true KO
   to a LSO (ih is what's needed to consult the objtab). */


/*============ from pipe.c =========================*/

/*L1, L2 not designed yet*/

_FN_DECL(ilu_string ilu_Pipe_Type, ( ilu_Pipe pipe ));

_FN_DECL(ilu_boolean ilu_Pipe_OtherEndGone, ( ilu_Pipe pipe ));

_FN_DECL(ilu_boolean ilu_Pipe_WillBlock, ( ilu_Pipe pipe ));

_FN_DECL(void ilu_Pipe_Destroy, ( ilu_Pipe pipe ));

_FN_DECL(ilu_Pipe ilu_Pipe_Create, ( ilu_string contact_info,
				     ilu_string iluTypeName,
				     ilu_boolean sink_p ));

_FN_DECL(ilu_boolean ilu_Pipe_Open, ( ilu_Pipe pipe ));

_FN_DECL(ilu_Call ilu_Pipe_Call, ( ilu_Pipe pipe ));

/* ==================================== from gc.c */

/* ======== Server Side ======== */

/*L2, Main unconstrained*/

/*Inside(obj's server, obj's type)*/
typedef void (*ilu_ObjectNoter)(ilu_Object obj,
				int vi);
/* The LS runtime provides this procedure.  The kernel calls this procedure when the kernel becomes, or ceases being, very interested in an object.  The LS runtime uses this opportunity to keep a LS object around as long as there are surrogates.  This procedure should not call anything [eg, RegLSO(obj, NULL)] that might free the object. */

/*L1_sup < cmu*/
_FN_DECL(void ilu_SetNoter, (ilu_ObjectNoter n));
/* The LS runtime calls this at most once, before any objects are created.
 */

/*L1 >= {obj's server};
  obj true && collectible => L1 >= {gcmu}*/
_FN_DECL(ilu_boolean ilu_VeryInterested, (ilu_Object obj));
/* Tests whether the kernel is very interested in this object. */

/* ======== Client Side ======== */

/*L1, L2, Main unconstrained*/

_FN_DECL(void ilu_SetGcClient, (ilu_Object interest));
/* A client of GCed objects calls this --- once, before
   RegisterLanguageSpecificObject on any GCed object. */

#endif /* _ILU_EXPORTS_ */

