;;; -*- Package: USER; Syntax: Common-Lisp; Base: 10 -*-

;;; The code below provides a simple mechanism to catch all errors.  Three
functions
;;; are provided: without-errors, with-errors-to-file and
with-errors-to-file-and-printer.
;;; See their documentation below.  (Note: the backtrace and print features will
;;; probably need to be added for your environment.  They only work in MCL
(and the
;;; printing in MCL requires the "postscript-file-downloading.lisp"
extension from
;;; the clim and the mcl archives)).


;;; Keunen Vincent                  Network Research Belgium
;;; R&D, Software Engineer          Parc Industriel des Hauts-Sarts
;;; tel: +32 41 407282              2e Avenue, 65
;;; fax: +32 41 481170              BE-4040 Herstal, Belgium
;;; internet: keunen@montefiore.ulg.ac.be

(in-package :user)

#+genera
(in-package :future-common-lisp-user)

(export '(without-errors with-errors-to-file
with-errors-to-file-and-printer iso-date-and-time))

(defmacro without-errors ((&key (stream *error-output*)
                                (backtrace nil))
                          &body body)
  "The code wrapped by this macro will never error.  Instead, the error message
will be written to *error-output* or to the stream provided explicitly.
When an error occurs, two values are returned: NIL and the error."
  (let ((tag (gensym)))
    `(block ,tag
       (handler-bind
	 ((error
	    #'(lambda (condition)
		(format ,stream
			"~%~a -> Unexpected error detected and ignored: ~&   ~a"
			(iso-date-and-time) condition)
                (when ,backtrace
                  (let ((*debug-io* ,stream))
                    #+mcl (print-call-history)
                    #-mcl (format t "Don't know how to access backtrace in
this environment.")))
		(return-from ,tag (values nil condition)))))
	 ,@body))))


(defmacro with-errors-to-file ((&key (file (make-pathname :name "error-output"
							  :type "text")))
			       &body body)
  "The code wrapped by this macro will never error.  Instead, the error message
will be written to a file. When an error occurs, two values are returned:
NIL and the error."
  `(with-open-file (stream ,file
			   :direction :output
			   :if-exists :append
			   :if-does-not-exist :create)
     (without-errors (:stream stream)
       ,@body)))


(defmacro with-errors-to-file-and-printer
          ((&key (file (make-pathname :name "error-output"
                                      :type "text"))
                 (backtrace nil)
                 (print nil))
           &body body)
  "The code wrapped by this macro will never error.  Instead, the error message
will be written to a file. When an error occurs, two values are returned:
NIL and the error."
  `(let ((temp-file
          (make-pathname :name "one-error-output"
                         :type "temp")))
     (prog1
       (with-open-file (stream1 ,file
			        :direction :output
			        :if-exists :append
			        :if-does-not-exist :create)
         (with-open-file (stream2 temp-file
			          :direction :output
			          :if-exists :overwrite
			          :if-does-not-exist :create)
           (let ((stream (make-broadcast-stream stream1 stream2)))
             (without-errors (:stream stream :backtrace ,backtrace)
               ,@body))))
       (without-errors ()
         (when ,print
           #+mcl (ccl:print-ps-file-via-ae temp-file)
           #-mcl (format t "Don't know how to print in this environment."))
         (delete-file temp-file)))))

;;; To get the full date formatting, see the "date-formatter" file from
;;; mkant (see the Lisp FAQ file /public/think/lisp/faq.text on ftp.think.com
;;; for more info)
(defun iso-date-and-time ()
  (multiple-value-bind (sec min hour day month year dow) (get-decoded-time)
    (declare (ignore dow))
    (format nil "~A/~2,'0D/~2,'0D ~2,'0d:~2,'0d:~2,'0d"
	    (mod year 100) month day hour min sec)))

#|

(without-errors (:stream *standard-output*)
  (format t "Hello, you know.")
  (error "This is a simple error."))

(with-errors-to-file ()
  (format t "Hello, you know.")
  (error "This is a simple error."))

(defun bogus-function ()
  "This function breaks unless some dummy created a file named hdggffsdhjg!!"
  (with-open-file (stream "hdggffsdhjg" :direction :input)
    (read stream)))
(progn
  (format t "~&Hello, you know.")
  (with-errors-to-file ()
    (bogus-function))
  (format t "~&Hello, you know."))

|#

;;; *EOF*
