/*  /usr/mnr/pp/bfs/bfs.c - Generic best-first-search.  mnr 7/4/88.

    What it is:
	bfs does a best-first search on a generic search space.  It
	produces a path with least maximum-cost-state.  If all
	transitions have positive cost, this is an optimal path to
	the destination.

    Interface:
	bfs(num_states,num_trans,num_sources,sources,source_costs,
	    is_dest,gen_trans,inv_trans,
	    &path_source,&path_length,&path,&path_cost)
      where
        num_states   = number of states in the state space
	num_trans    = max number of transitions available between states.
		       Note: transitions must be one-to-one.
	num_sources  = number of source states
	sources      = array of (int) source states
	source_costs = array of (short int) source costs
	is_dest      = (*is_dest)(state) == 1 iff state is a destination
	gen_trans is a transition function provided by the caller, which
	is called like this:
	    (*gen_trans)(source,newstates,newcosts)
		         int    int *     short int *
	It must fill in num_trans new states (some of which may be -1, which
	indicates the transition is not allowed for that state), and for each
	new state other than -1, the cost of the transition.
	Costs are short ints, -32767 to 32766. The total cost to the
	destination must not exceed these bounds.
        The caller must also provide
	    (*inv_trans)(&prevstate,transition,state)
        which finds 'prevstate' such that 'transition' on it gives 'state'.

    Returned values:
	The function fills in these variables:
	    path_length = transitions in best path.
			  -1 means that the destination was not found.
	    path = a pointer to an array of path_length transition numbers
	    path_cost = the sum of the costs of the transitions, a short int.

    Data structures:
	cost  = array of xmax*ymax short ints, == best cost from source.
	        -- cost of 32767 means not yet calculated.
	cost1 = array of xmax*ymax*ways_in short ints, == cost from each
		direction.  32767 means not calculated.
	frontier = expandable heap of pointers into the cost1 array,
		   corresponding to states to be expanded (possibly entered
		   multiply; the least-cost entry will be expanded)
	expanded = array of chars, non-zero == expanded already

    Algorithm:
	On each iteration, the min-cost item in the frontier is retrieved.
	If its state has been expanded, it is thrown away [ this occurs when
	a state is entered more than once into the frontier].  If not,
	the state is expanded, by calling gen_trans.  Each resulting state
	(if not -1, and the new cost is less than the best cost so far) is
	entered into the frontier.
	The search terminates when the heap is empty, or when a destination
	state is removed from the frontier.

    Requires:
        heap.c, heap.h - heap package

*/

#include "heap.h"
/* #include "/usr/cs/include/libc.h" */
/* #include "usegc.h"*/

numcmp(a,b)  /* comparison function for heap package */
char *a,*b;
{
	return (int)(*(short int *)a-*(short int *)b);
}

bfs(num_states,num_trans,num_sources,sources,source_costs,
    is_dest,gen_trans,inv_trans,
    path_source_p,path_length_p,path_p,path_cost_p)
   int num_states,num_trans,num_sources,*sources,(*is_dest)(),
       (*gen_trans)(),(*inv_trans)(),*path_source_p,*path_length_p,**path_p;
   short int *source_costs,*path_cost_p;
{
    heap *frontier;
    short int *cost,*cost1,*minptr,*newptr,newcost,*newcosts;
    int i,j,min,new,*newstates;
    char *expanded;

/* Initialization */

    frontier=hp_make(numcmp,1000); /* expandable heap */
    cost=(short int *)malloc(num_states * sizeof(short int));
    cost1=(short int *)malloc(num_states * num_trans * sizeof(short int));
    expanded=(char *)malloc(num_states * sizeof(char));
    newstates=(int *)malloc(num_trans * sizeof(int));
    newcosts=(short int *)malloc(num_trans * sizeof(short int));
    for (i=0;i<num_states;++i) {
	cost[i] = 32767;
	expanded[i] = 0;
	for (j=0;j<num_trans;++j)
	    *(cost1+j*num_states+i)=32767;
    }

    for (i=0;i<num_sources;++i) {
	cost[sources[i]] = *(cost1+0+sources[i]) = source_costs[i];
	hp_enter(frontier,cost1+0+sources[i]);
    }

/* Inner loop: expand min. cost node until you find the destination */

    while(minptr=(short int *)hp_delmin(frontier)) {
	min=(minptr-cost1)%num_states;
	if (expanded[min]) continue;
	expanded[min]=1;
	if ((*is_dest)(min)) goto found_dest;
	gen_trans(min,newstates,newcosts);
	for (i=0;i<num_trans;++i) {       /* enter all next states  */
	    new=newstates[i];
	    newcost=newcosts[i];
	    if (new>-1) {
		newcost+=cost[min];
		if (newcost<cost[new] && !expanded[new]) {
		    *(newptr=cost1+i*num_states+new)=cost[new]=newcost;
		    hp_enter(frontier,newptr);
		}
	    }
	}
      }

/* Dropped out of loop: dest not found. */
    *path_length_p= -1;
    free(cost); free(cost1); free(expanded); 
    free(newstates); free(newcosts); 
    free(frontier->hp_heap);
    free(frontier);
    return;

/* Found the destination */
  found_dest:
    retrace_path(num_states,num_trans,inv_trans,num_sources,sources,min,
    		 cost,cost1,path_source_p,path_length_p,path_p,path_cost_p);
    free(cost); free(cost1); free(expanded); 
    free(newstates); free(newcosts); 
    free(frontier->hp_heap);
    free(frontier);
}


/* Retrace the path using inv_trans.  Do it once to find the length; then
   create the 'path' array and fill in the transitions on the second pass. */

retrace_path(num_states,num_trans,inv_trans,num_sources,sources,dest,
	     cost,cost1,path_source_p,path_length_p,path_p,path_cost_p)
    int num_states,num_trans,(*inv_trans)(),num_sources,*sources,
        dest,*path_source_p,*path_length_p,**path_p;
    short int *cost,*cost1,*path_cost_p;
{
    int pos,i,len;

    pos=dest;
    len=0;
    while(1) {
	for (i=0;i<num_sources;++i)
	    if (pos==sources[i]) goto found_source;
	for (i=0;i<num_trans;++i)
	    if (*(cost1+i*num_states+pos)==cost[pos]) break;
	(*inv_trans)(&pos,i,pos);
	++len;
    }
  found_source:
    *path_source_p=pos;
    *path_length_p=len;
    (*path_p)=(int *)malloc(len*sizeof(int));
    pos=dest;
    while(pos!=*path_source_p) {
	for (i=0;i<num_trans;++i)
	    if (*(cost1+i*num_states+pos)==cost[pos]) break;
	(*path_p)[--len]=i;
	(*inv_trans)(&pos,i,pos);
    }
    *path_cost_p=cost[dest];
}
