/* 
 * Copyright (C) 1993 Mark Boyns (boyns@sdsu.edu)
 *
 * This file is part of rplay.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "conf.h"
#include "version.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <string.h>
#include "rplay.h"

typedef struct
{
	char	*name;
	int	min_args;
	int	max_args;
	char	*usage;
#ifdef __STDC__
	void	(*func)(int argc, char **argv);
#else
	void	(*func)();
#endif
} COMMAND;

#ifdef __STDC__
void	command_open(int argc, char **argv);
void	command_close(int argc, char **argv);
void	command_play(int argc, char **argv);
void	command_find(int argc, char **argv);
void	command_help(int argc, char **argv);
void	command_list(int argc, char **argv);
void	command_quit(int argc, char **argv);
void	command_put(int argc, char **argv);
void	command_get(int argc, char **argv);
void	command_access(int argc, char **argv);
void	command_unknown(int argc, char **argv);
void	argv_to_command(char **argv);
int	connected();
void	done(int exit_value);
void	usage();
#else
void	command_open(/* int argc, char **argv */);
void	command_close(/* int argc, char **argv */);
void	command_play(/* int argc, char **argv */);
void	command_find(/* int argc, char **argv */);
void	command_help(/* int argc, char **argv */);
void	command_list(/* int argc, char **argv */);
void	command_quit(/* int argc, char **argv */);
void	command_put(/* int argc, char **argv */);
void	command_get(/* int argc, char **argv */);
void	command_access(/* int argc, char **argv */);
void	command_unknown(/* int argc, char **argv */);
void	argv_to_command(/* char **argv */);
int	connected();
void	done(/* int exit_value */);
void	usage();
#endif

COMMAND	commands[] =
{
	"access",	0,	0,	"",				command_access,
	"close",	0,	0,	"",				command_close,
	"continue",	1,	-1,	"[-PNnv] sound ...",		command_play,
	"find",		1,	1,	"sound",			command_find,
	"get",		1,	2,	"sound [filename]",		command_get,
	"help",		0,	1,	"[command]",		       	command_help,
	"list",		0,	1,	"[connections|hosts|servers|spool|sounds]",		command_list,
	"open",		1,	2,	"hostname [port]",		command_open,
	"pause",	1,	-1,	"[-PNnv] sound ...",		command_play,
	"play",		1,	-1,	"[-PNnv] sound ...",		command_play,
	"put",		1,	1,	"sound",			command_put,
	"quit",		0,	0,	"",				command_quit,
	"stop",		1,	-1,	"[-PNnv] sound ...",		command_play,
};

#define NCOMMANDS	(sizeof(commands)/sizeof(COMMAND))
#undef MIN
#define MIN(a,b)	((a)>(b)?(b):(a))

int	rptp_fd = -1;
char	rptp_buf[4096];
char	command[RPTP_MAX_LINE];
char	response[RPTP_MAX_LINE];

extern int	optind;
extern char	*optarg;

#ifdef __STDC__
main(int argc, char **argv)
#else
main(argc, argv)
int	argc;
char	**argv;
#endif
{
	char	buf[256];
	int	i;
	int	port = RPTP_PORT;
	int	c;
	char	*av[RPTP_MAX_ARGS], *p, *host = NULL;
	int	ac, first;

	while ((c = getopt(argc, argv, "h:p:v")) != -1)
	{
		switch (c)
		{
		case 'h':
			host = optarg;
			break;

		case 'p':
			port = atoi(optarg);
			break;
		
		case 'v':
			printf("%s\n", rplay_version);
			done(0);

		case '?':
			usage();
			done(0);
		}
	}

	if (optind != argc)
	{
		usage();
		done(1);
	}

	if (host == NULL)
	{
		host = rplay_default_host();
	}

	rptp_fd = rptp_open(host, port, response, sizeof(response));
	if (rptp_fd < 0)
	{
		rptp_perror("open");
	}
	else
	{
		printf("Connected to %s port %d.\n", host, port);
		printf("%s\n", response+1);
	}

	for (;;)
	{
		printf("rptp> ");
		fflush(stdout);
		if (fgets(buf, sizeof(buf), stdin) == NULL)
		{
			done(0);
		}
		first = 1;
		ac = 0;
		while (p = strtok(first ? buf : NULL, " \t\r\n"))
		{
			av[ac++] = p;
			first = 0;
		}
		av[ac] = NULL;

		if (av[0] == NULL)
		{
			continue;
		}

		for (i = 0; i < NCOMMANDS; i++)
		{
			if (strcasecmp(commands[i].name, av[0]) == 0)
			{
				if (commands[i].min_args >= 0 && ac-1 < commands[i].min_args
					|| commands[i].max_args >= 0 && ac-1 > commands[i].max_args)
				{
					printf("Usage: %s %s\n", commands[i].name, commands[i].usage);
				}
				else
				{
					(*commands[i].func)(ac, av);
				}
				break;
			}
		}
		if (i == NCOMMANDS)
		{
			command_unknown(ac, av);
		}
	}
}

#ifdef __STDC__
void	argv_to_command(char **argv)
#else
void	argv_to_command(argv)
char	**argv;
#endif
{
	command[0] = '\0';
	for (; *argv; argv++)
	{
		strcat(command, *argv);
		if (*(argv+1))
		{
			strcat(command, " ");
		}
	}
}

int	connected()
{
	if (rptp_fd != -1)
	{
		return 1;
	}
	else
	{
		printf("You're not connected, use open first.\n");
		return 0;
	}
}

#ifdef __STDC__
void	command_open(int argc, char **argv)
#else
void	command_open(argc, argv)
int	argc;
char	**argv;
#endif
{
	int	port;

	if (rptp_fd != -1)
	{
		printf("You're already connected, use close first.\n");
	}
	else
	{
		port = argc == 3 ? atoi(argv[2]) : RPTP_PORT;
		rptp_fd = rptp_open(argv[1], port, response, sizeof(response));
		if (rptp_fd < 0)
		{
			rptp_perror("open");
		}
		else
		{
			printf("Connected to %s port %d.\n", argv[1], port);
			printf("%s\n", response+1);
		}
	}
}

#ifdef __STDC__
void	command_close(int argc, char **argv)
#else
void	command_close(argc, argv)
int	argc;
char	**argv;
#endif
{
	if (connected())
	{
		rptp_close(rptp_fd);
		rptp_fd = -1;
		printf("Connection closed.\n");
	}
}

#ifdef __STDC__
void	command_play(int argc, char **argv)
#else
void	command_play(argc, argv)
int	argc;
char	**argv;
#endif
{
	argv_to_command(argv);

	if (!connected())
	{
		return;
	}

	switch (rptp_command(rptp_fd, command, response, sizeof(response)))
	{
	case -1:
		rptp_perror(argv[0]);
		command_close(argc, argv);
		return;

	case 1:
		printf("%s\n", response+1);
		return;

	case 0:
		break;
	}
}

#ifdef __STDC__
void	command_find(int argc, char **argv)
#else
void	command_find(argc, argv)
int	argc;
char	**argv;
#endif
{
	argv_to_command(argv);

	switch (rptp_command(rptp_fd, command, response, sizeof(response)))
	{
	case -1:
		rptp_perror(argv[0]);
		command_close(argc, argv);
		return;

	case 1:
		printf("%s\n", response+1);
		return;

	case 0:
		break;
	}

	printf("%s\n", response+1);
}

#ifdef __STDC__
void	command_access(int argc, char **argv)
#else
void	command_access(argc, argv)
int	argc;
char	**argv;
#endif
{
	argv_to_command(argv);
	switch (rptp_command(rptp_fd, command, response, sizeof(response)))
	{
	case -1:
		rptp_perror(argv[0]);
		command_close(argc, argv);
		return;
		
	case 1:
		printf("%s\n", response+1);
		return;
		
	case 0:
		break;
	}
	
	printf("%s\n", response+1);
}

#ifdef __STDC__
void	command_help(int argc, char **argv)
#else
void	command_help(argc, argv)
int	argc;
char	**argv;
#endif
{
	int	i;
	
	if (argc == 2)
	{
		for (i = 0; i < NCOMMANDS; i++)
		{
			if (strcasecmp(commands[i].name, argv[1]) == 0)
			{
				printf("Usage: %s %s\n", commands[i].name, commands[i].usage);
				return;
			}
		}
		command_unknown(argc, argv);
	}
	else
	{
		printf("access   Display remote access permissions.\n");
		printf("close    Close the current server connection.\n");
		printf("continue Continue paused sounds.\n");
		printf("find     Search for a sound.\n");
		printf("get      Retrieve a sound.\n");
		printf("help     Display help information.\n");
		printf("list     Display various server information.\n");
		printf("open     Connect to a server.\n");
		printf("pause    Pause sounds that are playing.\n");
		printf("play     Play sounds\n");
		printf("put      Send a sound.\n");
		printf("quit     Terminate the rptp session.\n");
		printf("stop     Stop sounds that are playing.\n");
	}
}

#ifdef __STDC__
void	command_list(int argc, char **argv)
#else
void	command_list(argc, argv)
int	argc;
char	**argv;
#endif
{
	int	n;

	if (!connected())
	{
		return;
	}

	argv_to_command(argv);

	switch (rptp_command(rptp_fd, command, response, sizeof(response)))
	{
	case -1:
		rptp_perror(argv[0]);
		command_close(argc, argv);
		return;

	case 1:
		printf("%s\n", response+1);
		return;

	case 0:
		break;
	}

	for (;;)
	{
		n = rptp_getline(rptp_fd, rptp_buf, sizeof(rptp_buf));
		if (n < 0)
		{
			rptp_perror("list");
			command_close(argc, argv);
			break;
		}
		if (strcmp(rptp_buf, ".") == 0) 
		{
			break;
		}
		printf("%s\n", rptp_buf);
	}
}

#ifdef __STDC__
void	command_quit(int argc, char **argv)
#else
void	command_quit(argc, argv)
int	argc;
char	**argv;
#endif
{
	if (rptp_fd != -1)
	{
		rptp_close(rptp_fd);
		printf("Connection closed.\n");
	}
	done(0);
}

#ifdef __STDC__
void	command_put(int argc, char **argv)
#else
void	command_put(argc, argv)
int	argc;
char	**argv;
#endif
{
	FILE		*fp;
	int		size, n, nwritten;
	struct stat	st;
	char		buf[32];

	if (!connected())
	{
		return;
	}

	if (stat(argv[1], &st) < 0)
	{
		perror(argv[1]);
		return;
	}
	fp = fopen(argv[1], "r");
	if (fp == NULL)
	{
		perror(argv[1]);
		return;
	}
	size = st.st_size;

	sprintf(buf, "%d", size);

	argv[2] = buf;
	argv[3] = NULL;

	argv_to_command(argv);

	switch (rptp_command(rptp_fd, command, response, sizeof(response)))
	{
	case -1:
		rptp_perror(argv[0]);
		command_close(argc, argv);
		return;

	case 1:
		printf("%s\n", response+1);
		return;

	case 0:
		break;
	}

	while (size > 0)
	{
		n = fread(rptp_buf, 1, sizeof(rptp_buf), fp);
		nwritten = rptp_write(rptp_fd, rptp_buf, n);
		if (nwritten != n)
		{
			rptp_perror("put");
			command_close(argc, argv);
			break;
		}
		size -= n;
	}

	fclose(fp);
}

#ifdef __STDC__
void	command_get(int argc, char **argv)
#else
void	command_get(argc, argv)
int	argc;
char	**argv;
#endif
{
	FILE	*fp;
	char	*filename;
	char	*p;
	int	size, n, nread;
	
	if (!connected())
	{
		return;
	}

	if (argc == 3)
	{
		filename = argv[2];
		argv[2] = NULL;
	}
	else
	{
		filename = argv[1];
	}

	argv_to_command(argv);

	switch (rptp_command(rptp_fd, command, response, sizeof(response)))
	{
	case -1:
		rptp_perror(argv[0]);
		command_close(argc, argv);
		return;

	case 1:
		printf("%s\n", response+1);
		return;

	case 0:
		break;
	}

	fp = fopen(filename, "w");
	if (fp == NULL)
	{
		perror(filename);
		return;
	}

	p = strtok(response+1, " ");
	size = atoi(strtok(NULL, "\r\n"));
	
	while (size > 0)
	{
		n = MIN(sizeof(rptp_buf), size);
		nread = rptp_read(rptp_fd, rptp_buf, n);
		if (nread != n)
		{
			rptp_perror("get");
			break;
		}
		fwrite(rptp_buf, 1, n, fp);
		size -= n;
	}

	fclose(fp);
}

#ifdef __STDC__
void	command_unknown(int argc, char **argv)
#else
void	command_unknown(argc, argv)
int	argc;
char	**argv;
#endif
{
	printf("Unknown command '%s'.\n", argv[0]);
}

void	usage()
{
	printf("\n%s\n\n", rplay_version);
	printf("usage: rptp [options]\n");
	printf("\t-h host   Specify the RPTP host, default = %s.\n",
		rplay_default_host());
	printf("\t-p port   Use port instead of the default RPTP port, default = %d.\n", RPTP_PORT);
	printf("\t-v        Display rplay version information.\n");
	printf("\n");
}

#ifdef __STDC__
void	done(int exit_value)
#else
void	done(exit_value)
int	exit_value;
#endif
{
	exit(exit_value);
}
